! PGBIN -- Histogram of binned data

subroutine PGBIN( nbin, x, data, center )

   integer          :: nbin
   double precision :: x(*), data(*)
   logical          :: center

   ! Plot a histogram of NBIN values with X(1..NBIN) values along
   ! the ordinate, and DATA(1...NBIN) along the abscissa. Bin width is
   ! spacing between X values.
   !
   ! Arguments:
   !  NBIN   (input)  : number of values.
   !  X      (input)  : abscissae of bins.
   !  DATA   (input)  : data values of bins.
   !  CENTER (input)  : if TRUE, the X values denote the center of the
   !                    bin; if FALSE, the X values denote the lower
   !                    edge (in X) of the bin.
   !--
   ! 19-Aug-1992 - Change argument check (TJP).
   ! 29-Feb-2020 - Use now double precision instead of single precision [EC].
   !  6-Apr-2020 - Calling syntax to GRVCT0 updated [EC].
   !-----------------------------------------------------------------------

   integer :: ibin
   double precision :: tx(4), ty(4)

   !------ end of declarations -- execution starts hereafter  ------

   ! Check arguments.
   if( nbin < 2 ) return

   call pgbbuf()

   ! Draw Histogram. Centered and uncentered bins are treated separately.
   if( center ) then
      ! set up initial point.
      tx(2) = (3.*x(1) - x(2))/2.
      ty(2) = data(1)
      tx(3) = (x(1) + x(2))/2.
      ty(3) = ty(2)
      call grvct0( 2, 2, tx(2), ty(2) )
      ! Draw initial horizontal line
      ! Now loop over bins
      do ibin = 2, nbin-1
         tx(1) = tx(3)
         tx(2) = tx(1)
         tx(3) = ( x(ibin) + x(ibin+1) ) / 2.
         ty(1) = ty(3)
         ty(2) = data(ibin)
         ty(3) = ty(2)
         call grvct0( 2, 3, tx, ty )
      end do
      ! Now draw last segment.
      tx(1) = tx(3)
      tx(2) = tx(1)
      tx(3) = (3.*x(nbin) - x(nbin-1) )/2.
      ty(1) = ty(3)
      ty(2) = data(nbin)
      ty(3) = ty(2)
      call grvct0( 2, 3, tx, ty )

   else ! Uncentered bins
      ! Set up first line.
      tx(2) = x(1)
      ty(2) = data(1)
      tx(3) = x(2)
      ty(3) = ty(2)
      call grvct0( 2, 2, tx(2), ty(2) )
      do ibin = 2, nbin
         tx(1) = tx(3)
         tx(2) = tx(1)
         if( ibin == nbin ) then
            tx(3) = 2.*x(nbin) - x(nbin-1)
         else
            tx(3) = x(ibin+1)
         end if
         ty(1) = ty(3)
         ! Get height for last segment.
         ty(2) = data(ibin)
         ty(3) = ty(2)
         call grvct0( 2, 3, tx, ty )
      end do
   end if

   call pgebuf()

end subroutine pgbin
