! PGBAND -- Read cursor position, with anchor

integer function PGBAND( cursor, mode, posn, xref, yref, x, y, ch,      &
                         rect_inside_axes, axis_scale_x, axis_scale_y,  &
                         control_key_down, read_cursor_valid )

   integer,          intent(in) :: cursor, mode, posn
   double precision, intent(in) :: xref, yref
   double precision             :: x, y
   character(len=*)             :: ch
   logical,          intent(in) :: rect_inside_axes
   integer, optional :: axis_scale_x, axis_scale_y
   integer, optional :: control_key_down, read_cursor_valid
   !------ API end ------

   ! Read the cursor position and a character typed by the user.
   ! The position is returned in world coordinates.  PGBAND positions
   ! the cursor at the position specified (if POSN=1), allows the user to
   ! move the cursor using the mouse or arrow keys or whatever is available
   ! on the device. When he has positioned the cursor, the user types a
   ! single character on the keyboard; PGBAND then returns this
   ! character and the new cursor position (in world coordinates).
   !
   ! Some interactive devices offer a selection of cursor types,
   ! implemented as thin lines that move with the cursor, but without
   ! erasing underlying graphics. Of these types, some extend between
   ! a stationary anchor-point at XREF,YREF, and the position of the
   ! cursor, while others simply follow the cursor without changing shape
   ! or size. The cursor type is specified with one of the following MODE
   ! values. Cursor types that are not supported by a given device, are
   ! treated as MODE=0.
   !
   ! -- CURSOR (added by E. Canot -- 10 juin 2006): specify the cursor shape.
   !      (see the numbering in the 'mod_win_db' module)
   !
   ! -- If MODE=0, the anchor point is ignored and the routine behaves
   !    like PGCURS.
   ! -- If MODE=1, a straight line is drawn joining the anchor point
   !    and the cursor position.
   ! -- If MODE=2, a hollow rectangle is extended as the cursor is moved,
   !    with one vertex at the anchor point and the opposite vertex at the
   !    current cursor position; the edges of the rectangle are horizontal
   !    and vertical.
   ! -- If MODE=3, two horizontal lines are extended across the width of
   !    the display, one drawn through the anchor point and the other
   !    through the moving cursor position. This could be used to select
   !    a Y-axis range when one end of the range is known.
   ! -- If MODE=4, two vertical lines are extended over the height of
   !    the display, one drawn through the anchor point and the other
   !    through the moving cursor position. This could be used to select
   !    an X-axis range when one end of the range is known.
   ! -- If MODE=5, a horizontal line is extended through the cursor
   !    position over the width of the display. This could be used to select
   !    a Y-axis value such as the start of a Y-axis range. The anchor point
   !    is ignored.
   ! -- If MODE=6, a vertical line is extended through the cursor
   !    position over the height of the display. This could be used to
   !    select an X-axis value such as the start of an X-axis range. The
   !    anchor point is ignored.
   ! -- If MODE=7, a cross-hair, centered on the cursor, is extended over
   !    the width and height of the display. The anchor point is ignored.
   ! -- If MODE=8, a rectangle, centered on the cursor, of given width and
   !    height (passed in XREF and YREF) is displayed. The anchor point is
   !    ignored.
   ! -- If MODE=101, a user-defined cursor is displayed (usually the true
   !    cursor is fully transparent). Used for magnetic grid implementation.
   !
   ! -- CUSTOM_CURSOR: 0 by default (no change w.r.t. to the common use).
   !                   1 to set a band to MF_USER1_CURSOR, while the true
   !                     cursor is fully transparent.
   !                   2 to set a band to MF_USER2_CURSOR, while the true
   !                     cursor is fully transparent.
   !
   ! Returns:
   !  PGBAND          : 1 if the call was successful; 0 if the device
   !                    has no cursor or some other error occurs.
   ! Arguments:
   !  MODE   (input)  : display mode (0, 1, ..., 8: see above).
   !  POSN   (input)  : if POSN=1, PGBAND attempts to place the cursor
   !                    at point (X,Y); if POSN=0, it leaves the cursor
   !                    at its current position. (On some devices this
   !                    request may be ignored.)
   !  XREF   (input)  : the world x-coordinate of the anchor point.
   !  YREF   (input)  : the world y-coordinate of the anchor point.
   !  X      (in/out) : the world x-coordinate of the cursor.
   !  Y      (in/out) : the world y-coordinate of the cursor.
   !  CH     (output) : the character typed by the user; if the device has
   !                    no cursor or if some other error occurs, the value
   !                    CHAR(0) [ASCII NUL character] is returned.
   !  RECT_INSIDE_AXES: if TRUE, the rectangle drawn is constrained to be
   !                    located inside the axes (only for MODE=8).
   !
   ! Note: The cursor coordinates (X,Y) may be changed by PGBAND even if
   ! the device has no cursor or if the user does not move the cursor.
   ! Under these circumstances, the position returned in (X,Y) is that of
   ! the pixel nearest to the requested position.
   !--
   !  7-Sep-1994 - New routine [TJP].
   ! 10-Jun-2006 - New arg added (first): CURSOR.
   ! 30-Aug-2006 - Check arg CURSOR.
   !  9-Aug-2009 - Added MODE=8 and 3 args added: XMAX, YMAX, RECT_INSIDE_AXES.
   ! 21-Apr-2016 - Fixed OFFSET (depends on the screen resolution);
   !               changed way of processing RECT_INSIDE_AXES;
   !               removed args XMAX, YMAX (unused now).
   ! 29-Feb-2020 - Use now double precision instead of single precision.
   ! 13-Apr-2021 - When mode=0, initialize iref, jref to zero to avoid a
   !               warning from valgrind.
   !  9-May-2021 - Update call to grcurs().
   !  9-Jul-2022 - Added MODE=101.
   !-----------------------------------------------------------------------

   integer :: i, j, iref, jref
   integer :: imin, imax, jmin, jmax, inside
   double precision :: imax_real, jmax_real

   !------ end of declarations -- execution starts hereafter  ------

   if( cursor < 0 .or. MF_MAX_CURSOR_INDEX < cursor )                   &
             call grwarn('Invalid CURSOR argument in PGBAND')
   if( mode < 0 .or. 8 < mode .and. mode /= 100 .and. mode /= 101 )                       &
             call grwarn('Invalid MODE argument in PGBAND')
   if( posn < 0 .or. 1 < posn )                                         &
             call grwarn('Invalid POSN argument in PGBAND')

   i = nint(pgxorg(pgid) + x*pgxscl(pgid))
   j = nint(pgyorg(pgid) + y*pgyscl(pgid))
   if( mode == 8 ) then
      ! In this mode, IREF and JREF are not coordinates, but length
      ! taking abs, because axes may be inverted...
      iref = abs( nint(xref*pgxscl(pgid)) )
      jref = abs( nint(yref*pgyscl(pgid)) )

      if( rect_inside_axes ) then
         inside = 1
      else
         inside = 0
      end if
   else if( mode == 0 ) then
      ! The following values will not be referenced. They are initialized
      ! only to avoid a complain from valgrind...
      iref = 0
      jref = 0
   else
      iref = nint(pgxorg(pgid) + xref*pgxscl(pgid))
      jref = nint(pgyorg(pgid) + yref*pgyscl(pgid))
   end if

   if( present(axis_scale_x) ) then
      pgband = grcurs( cursor, i, j, iref, jref, mode, posn, ch, inside,   &
                       pgxorg(pgid), pgyorg(pgid), pgxscl(pgid), pgyscl(pgid), &
                       axis_scale_x, axis_scale_y,                         &
                       control_key_down=control_key_down,                  &
                       read_cursor_valid=read_cursor_valid )
   else
      pgband = grcurs( cursor, i, j, iref, jref, mode, posn, ch, inside,   &
                       pgxorg(pgid), pgyorg(pgid), pgxscl(pgid), pgyscl(pgid), &
                       control_key_down=control_key_down,                  &
                       read_cursor_valid=read_cursor_valid )
   end if

   x = (i - pgxorg(pgid))/pgxscl(pgid)
   y = (j - pgyorg(pgid))/pgyscl(pgid)

end function
