! GRXHLS -- convert RGB color to HLS color

subroutine GRXHLS( r, g, b, h, l, s )

   double precision, intent(in)  :: r, g, b
   double precision, intent(out) :: h, l, s
   !------ API end ------

   ! GRPCKG: Convert a color specified in the RGB color model to one in
   ! the HLS model.  This is a support routine: no graphics I/O occurs.
   ! The inverse transformation is accomplished with routine GRXRGB.
   ! Reference: SIGGRAPH Status Report of the Graphic Standards Planning
   ! Committee, Computer Graphics, Vol.13, No.3, Association for
   ! Computing Machinery, New York, NY, 1979.
   !
   ! Arguments:
   !
   ! R,G,B (real, input): red, green, blue color coordinates, each in the
   !       range 0.0 to 1.0. Input outside this range causes HLS = (0,1,0)
   !       [white] to be returned.
   ! H,L,S (real, output): hue (0 to 360), lightness (0 to 1.0), and
   !       saturation (0 to 1.0).
   !--
   !  2-Jul-1984 - New routine [TJP].
   ! 29-Sep-1994 - Force H to be in range 0-360 [Remko Scharroo; TJP].
   ! 29-Feb-2020 - Use now double precision instead of single precision [EC].
   !-----------------------------------------------------------------------

   double precision :: ma, mi, rr, gg, bb, d

   !------ end of declarations -- execution starts hereafter  ------

   h = 0.0
   l = 1.0
   s = 0.0
   ma = max(r,g,b)
   mi = min(r,g,b)
   if (ma.gt.1.0 .or. mi.lt.0.0) return
   rr = (ma-r)
   gg = (ma-g)
   bb = (ma-b)

   ! Lightness
   l = 0.5*(ma+mi)

   ! Achromatic case (R=G=B)
   if (ma.eq.mi) then
      s = 0.0
      h = 0.0

      ! Chromatic case
   else
      ! Saturation
      d = ma-mi
      if (l.le.0.5) then
         s = d/(ma+mi)
      else
         s = d/(2.0-ma-mi)
      end if
      ! Hue
      if (r.eq.ma) then
         ! yellow to magenta
         h = (2.0*d+bb-gg)
      else if (g.eq.ma) then
         h = (4.0*d+rr-bb)
      else
         ! B == MA
         h = (6.0*d+gg-rr)
      end if
      h = mod(h*60.0/d,360.0)
      if (h.lt.0.0) h = h+360.0
   end if

end subroutine
