! GRSLS -- set line style

subroutine GRSLS( linestyle )

   integer, intent(in) :: linestyle
   !------ API end ------

   ! GRPCKG: Set the line style for subsequent plotting on the current  device.
   ! The different line styles are always generated in hardware.
   ! Four different line styles are available, with the following codes:
   !    1 (full line), 2 (dashed), 3 (dot-dash-dot-dash), 4 (dotted),
   !                   5 (inverted dashed).
   ! The default is 1 (normal full line).
   !
   ! Argument:
   !
   ! linestyle (input, integer): the line-style code for subsequent plotting
   !                             on the current device (in range 1-4).
   !--
   !  9-Feb-1983 - [TJP].
   !  3-Jun-1984 - Add GMFILE device [TJP].
   !  5-Aug-1986 - Add GREXEC support [AFT].
   ! 21-Feb-1987 - If needed, calls begin picture [AFT].
   ! 19-Jan-1987 - Fix bug in GREXEC call [TJP].
   ! 16-May-1989 - Fix bug for hardware line dash [TJP].
   !  1-Sep-1994 - Do not call driver to get size and capabilities [TJP].
   !
   !  2-Aug-2018 - Conditional quick return added.
   ! 14-Sep-2018 - Removed dash pattern number 5 (not used).
   ! 29-Feb-2020 - Use now double precision instead of single precision.
   !  2-Dec-2021 - Setup for software dashing removed.
   !  4-Dec-2021 - Fix a bug: LAST_LINEWIDTH must be checked, as actual
   !               dash setting (for linestyle/=1) depends on linewidth!
   ! 22-Mar-2023 - Added a now code: 5, for inverted dashes (as a results,
   !               such a dashed line may be superposed with a standard
   !               dashed line).
   ! 23-Mar-2023 - Add a check of LAST_LINESTYLE_IS_VALID even for X11.
   !-----------------------------------------------------------------------

   integer :: linestyle0, ibuf(1), lchr
   double precision :: rbuf(1), tmp
   character(len=10) :: chr

   character(len=4) :: type
   logical :: inter

   !------ end of declarations -- execution starts hereafter  ------

   if( grcide < 1 ) then
      call grwarn('GRSLS - no graphics device is active.')
      return
   end if

   linestyle0 = linestyle
   if( linestyle0 < 1 .or. 5 < linestyle0 ) then
      call grwarn('GRSLS - invalid line-style requested.')
      linestyle0 = 1
   end if

   ! Inquire if hardware dash is not available.
   if( grgcap(grcide)(3:3) == 'N' ) then
      call grwarn('GRSLS - software dash is no longer available.')
      return
   end if

   call grqtyp( type, inter )
   ! Ignore the request if the line style is unchanged.
   if( type == 'EPS' .or. type == "PDF" ) then
      ! For EPS and PDF, be sure that we are in the save Graphic State.
      ! So, define LAST_LINESTYLE_IS_VALID = .true. at each change of
      ! style but cancel this boolean if there is a 'grestore' command.
      if( LAST_LINESTYLE_IS_VALID .and. linestyle0 == 1 ) then
         if( linestyle0 == grstyl(grcide) ) return
      end if
   else
      if( LAST_LINESTYLE_IS_VALID .and. linestyle0 == 1 ) then
         if( linestyle0 == grstyl(grcide) ) return
      else
         if( linestyle0*grwidt(grcide) == grdpat(grcide) ) return
      end if
   end if

   if( grpltd(grcide) ) then
      ibuf(1) = linestyle0
      call grexec( grgtyp, SET_LINE_STYLE, rbuf, ibuf, chr, lchr )
   end if

   ! Save the current line style.
   grstyl(grcide) = linestyle0

   ! If dashing, save a pseudo pattern, because the dash pattern depends
   ! also on the line width.
   if( linestyle0 == 1 ) then
      grdpat(grcide) = 0.
   else
      grdpat(grcide) = linestyle0*grwidt(grcide)
   end if

   if( linestyle0 == 1 ) then
      LAST_LINESTYLE_IS_VALID = .true.
   end if

end subroutine
