! GRPARS -- parse device specification string

integer function GRPARS( spec, dev, type, append )

   character(len=*), intent(in)  :: spec
   character(len=*), intent(out) :: dev
   integer,          intent(out) :: type
   logical,          intent(out) :: append
   !------ API end ------

   ! GRPCKG: decode a device-specification; called by GROPEN.
   !
   ! Returns:
   !  GRPARS (output): 1 if the device-specification is
   !       acceptable; any other value indicates an error.
   !
   ! Arguments:
   !  SPEC (input): the device specification.
   !  DEV  (output):  device name or file spec.
   !  TYPE (output): device type (integer code); 0 if no device
   !       type is specified.
   !  APPEND (output): .TRUE. if /APPEND specified, .FALSE. otherwise.
   !--
   ! 23-Jul-1984 - [TJP].
   ! 19-Feb-1988 - Allow device part to be quoted [TJP].
   ! 30-Mar-1989 - Remove logical translation of device and type [TJP].
   ! 17-Jun-1991 - Ignore comments after ' (' [TJP].
   ! 19-Dec-1994 - Rewritten to scan backwards [TJP].
   !  6-Jun-1995 - Correct a zero-length string problem [TJP].
   !  4-Apr-2020 - Use modern Fortran 'index' to search in a string [EC].
   !-----------------------------------------------------------------------

   character(len=32) :: ctype, upper
   character(len=256) :: descr
   integer :: l, lc, ls

   !------ end of declarations -- execution starts hereafter  ------

   ! Default results.
   dev = ""
   type = 0
   append = .false.
   grpars = 1
   ctype = ""

   ! Null string is acceptable.
   if( len(spec) < 1 ) return
   if( spec == "" ) return

   ! On systems where it is possible, perform a "logical name" translation.
   descr = spec

   ! Discard trailing blanks: L is length of remainder.
   l = len_trim(descr)

   ! Find last slash in string (position LS or 0).
   ls = index( descr(1:l), "/", back=.true. )

   ! Check for /APPEND qualifier; if present, look again for type.
   if( ls > 0 ) then
      ctype = descr(ls+1:l)
      call grtoup(upper,ctype)
      ctype = upper
      if( ctype == "APPEND" ) then
         append = .true.
         l = ls - 1
         ls = index( descr(1:l), "/", back=.true. )
      else
         append = .false.
      end if
   end if

   ctype = descr(ls+1:l)
   lc = l - ls
   l = ls - 1

   ! Check for allowed type.
   if( lc > 0 ) then
      call grtoup(upper,ctype)
      ctype = upper
      type = grdtyp(ctype)
      if( type == 0) call grwarn("Unrecognized device type")
      if( type == -1) call grwarn("Device type is ambiguous")
   else
      type = 0
      call grwarn("Device type omitted")
   end if
   if( type == 0 ) grpars = grpars + 2

   ! Remove quotes from device if necessary.
   if( l >= 1 ) then
      if( descr(1:1) == '"' .and. descr(l:l) == '"' ) then
         dev = descr(2:l-1)
         l = l - 2
      else
         dev = descr(1:l)
      end if
   end if

end function
