! GRMKER -- draw graph markers

subroutine GRMKER( symbol, n, x, y )

   integer,          intent(in) :: symbol, n
   double precision, intent(in) :: x(*), y(*)
   !------ API end ------

   ! GRPCKG: Draw a graph marker at a set of n points in the current  window.
   ! Only few attributes (color, size) apply to markers, and others are
   ! ignored; marker size comes from the character height.
   !
   ! Arguments:
   !
   ! SYMBOL (input, integer): the marker number to be drawn.
   !       Only numbers whose absolute value is 1-26 are processed.
   !       Negative are treated by adding a white border around the symbol,
   !       but only for filled glyphs.
   ! N (input, integer): the number of points to be plotted.
   ! X, Y (input, real arrays, dimensioned at least N): the (X,Y)
   !       coordinates of the points to be plotted.
   !--
   ! (19-Mar-1983)
   ! 20-Jun-1985 - Revise to window markers whole [TJP].
   !  5-Aug-1986 - Add GREXEC support [AFT].
   !  1-Aug-1988 - Add direct use of Hershey number [TJP].
   ! 15-Dec-1988 - Standardize [TJP].
   ! 17-Dec-1990 - Add polygons [PAH/TJP].
   ! 12-Jun-1992 - [TJP]
   ! 22-Sep-1992 - Add support for hardware markers [TJP].
   !  1-Sep-1994 - Suppress driver call [TJP].
   ! 15-Feb-1994 - Fix bug (expanding viewport!) [TJP].
   ! 04-Sep-2004 - NaN values are not displayed.
   ! 01-Jul-2018 - New 21 markers numbering (to be consistent to those of EPS
   !               and PDF drivers). Filled markers with a negative number
   !               have a white border around them.
   ! 16-Aug-2018 - Remove angle, and computation of cos and sin,
   !               since angle is always equal to zero.
   !  3-Sep-2018 - For X11, markers are now drawn using symbols of a specific
   !               Type-1 PostScript font, via the libXft library (antialiased
   !               display). White border around symbol is also supported.
   ! 29-Feb-2020 - Use now double precision instead of single precision.
   !  1-Apr-2020 - Update explanation in the header. Old part treating markers
   !               as Hershey symbols is removed; therefore polygons having
   !               an arbitrary number of sides are no longer available.
   !  6-Apr-2020 - Removed ABSXY argument (FALSE in all calling routines);
   !               calling syntax to GRTXY0 updated.
   ! 16-Mar-2021 - Changed name of GRTXY0 to GR_WLD_TO_DEV.
   ! 14-Apr-2021 - Take care of integer overflow in the X11 driver
   !               (In Xlib, XPoint coordinates are 'short int').
   ! 15-Apr-2021 - Reintroduced Quick Return for symbol located outside
   !               the viewport.
   ! 29-Oct-2021 - Added 5 new symbols. Total number is now 26 instead of 21.
   !-----------------------------------------------------------------------

   integer :: c, symbol0, color, i, k
   logical :: white_border
   double precision :: factor, factor_wb, font_size_std, font_size_wb
   double precision :: xx, yy, xmin, xmax, ymin, ymax

   double precision :: rbuf(1)
   integer :: ibuf(5), lchr
   character(len=128) :: chr

   character(len=80) :: fontname

   logical, external :: grisnan

   !------ end of declarations -- execution starts hereafter  ------

   ! Check that there is something to be plotted.
   if( n <= 0 ) return

   ! Check that a device is selected.
   if( grcide < 1 ) then
      call grwarn('GRMKER - no graphics device is active.')
      return
   end if

   ! For the clipping...
   if( CLIPPING_IN_AXES ) then
      xmin = grxmin(grcide)
      xmax = grxmax(grcide)
      ymin = grymin(grcide)
      ymax = grymax(grcide)
!### TODO: la valeur constante utilisée ci-dessous n'est pas adaptée à tous
!          les cas. Il faudrait tenir compte de la taille des symboles.
      xmin = xmin - 0.01d0
      xmax = xmax + 0.01d0
      ymin = ymin - 0.01d0
      ymax = ymax + 0.01d0
   end if

   ! We must not modify 'symbol'
   symbol0 = symbol

   if( symbol0 < 0 ) then
      symbol0 = abs(symbol0)
      if( 14 <= symbol0 .and. symbol0 <= 21 ) then
         ! Border is white only for filled glyphs
         white_border = .true.
      else
         white_border = .false.
      end if
   else
      white_border = .false.
   end if

   factor = grmsiz(grcide)

   if( grgtyp == X11_DRIVER_NUM ) then ! XWINDOW
      factor = factor*0.425d0! 0.425d0 = old value of grcfac(grcide)/2.5d0 (X11)
      ! Pre call: we have to send the angle and the clipping status
      rbuf(1) = 0. ! angle is always 0 for markers
      if( CLIPPING_IN_AXES ) then
         ibuf(1) = 1
         ! Pass the clipping box (actually the whole viewport)
         ibuf(2) = nint(grxmin(grcide))
         ibuf(3) = nint(grymax(grcide))
         ibuf(4) = nint(grxmax(grcide))
         ibuf(5) = nint(grymin(grcide))
      else
         ibuf(1) = 0
      end if
      call grexec( grgtyp, SET_ANG_CLIP_XFT, rbuf, ibuf, chr, lchr )
      ! The multiplicative coefficient below has been determined to obtain
      ! the same character dimension (width and height) as in EPS and PDF.
      font_size_std = factor * 22.9d0 * 0.75d0
      font_size_wb = font_size_std * 1.2d0
   else ! NULL, EPS or PDF
      factor = factor*6.15d0! 6.15d0 = old value of grcfac(grcide)/2.5d0 (EPS+PDF)
      factor_wb = factor * 1.2d0
   endif

   if( 1 <= symbol0 .and. symbol0 <= 26 ) then

      if( .not. grpltd(grcide) ) call grbpic

      ! Symbol number
      ibuf(1) = symbol0
      lchr = 0
      do k = 1, n

         if( grisnan(x(k)) .or. grisnan(y(k)) ) cycle

         ! Convert to device coordinates
         call gr_wld_to_dev( x(k), y(k), xx, yy )
         ! Quick return (for all devices)
         if( CLIPPING_IN_AXES ) then
            if( xx < xmin ) cycle
            if( xx > xmax ) cycle
            if( yy < ymin ) cycle
            if( yy > ymax ) cycle
         end if

         if( grgtyp == X11_DRIVER_NUM ) then ! XWINDOW
            if( white_border ) then
               ! Save current color
               call grqci(color)
               ! Set color to background
               call grsci(0)
               ! Enlarge slightly the size

               ! Pre call: we have to send the fontname. Be aware that the
               ! fontname contains also some attributes (fontsize and fontstyle).
               write(fontname,"(F7.2)") font_size_wb
               fontname = "CeMaSP-" // adjustl(fontname)
               i = index( fontname, "*" )
               if( i /= 0 ) then
                  call grwarn('(Muesli FGL:) grmker: marker scale too large!')
                  print *, "  -> unexpected results may occur..."
               end if
               ! Send the font (name and attributes)
               chr = trim(fontname) // char(0)
               lchr = len_trim(fontname) + 1
               ibuf(1) = 0 ! not utf8
               call grexec( grgtyp, SET_FONT_XFT, rbuf, ibuf, chr, lchr )
               ibuf(1) = nint(xx)
               ibuf(2) = nint(yy)
               chr = char(64+symbol0) // char(0) ! A, B, C, ...
               lchr = 2
               call grexec( grgtyp, DRAW_MARKER, rbuf, ibuf, chr, lchr )

               ! Restore color
               call grsci(color)
            end if
            ! Pre call: we have to send the fontname. Be aware that the
            ! fontname contains also some attributes (fontsize and fontstyle).
            write(fontname,"(F7.2)") font_size_std
            fontname = "CeMaSP-" // adjustl(fontname)
            i = index( fontname, "*" )
            if( i /= 0 ) then
               call grwarn('(Muesli FGL:) grmker: marker scale too large!')
               print *, "  -> unexpected results may occur..."
            end if
            ! Send the font (name and attributes)
            chr = trim(fontname) // char(0)
            lchr = len_trim(fontname) + 1
            ibuf(1) = 0 ! not utf8
            call grexec( grgtyp, SET_FONT_XFT, rbuf, ibuf, chr, lchr )
            ! Overflow in integers may lead to a change of sign; in such a case
            ! some drawn lines may appear as wrong. Device coordinates must be
            ! restricted to [−32,767, +32,767], which is the range for the XPoint
            ! coordinates in Xlib. See also the 'restrict_to_short' function in
            ! the MOD_GRPLOT module.
            ibuf(1) = restrict_to_short(xx)
            ibuf(2) = restrict_to_short(yy)
            chr = char(64+symbol0) // char(0) ! A, B, C, ...
            lchr = 2
            call grexec( grgtyp, DRAW_MARKER, rbuf, ibuf, chr, lchr )

         else ! NULL, EPS or PDF

            ibuf(2) = xx
            ibuf(3) = yy
            if( white_border ) then
               ! Save current color
               call grqci(color)
               ! Set color to background
               call grsci(0)
               ! Enlarge slightly the size
               rbuf(1) = factor_wb
               call grexec( grgtyp, DRAW_MARKER, rbuf, ibuf, chr, lchr )
               ! Restore color
               call grsci(color)
            end if
            ! Scale factor
            rbuf(1) = factor
            call grexec( grgtyp, DRAW_MARKER, rbuf, ibuf, chr, lchr )

         end if

      end do

   else

      print "(/,A)",    "(MFPLOT:) ***internal error ***"
      print "(A,I0,A)", "          GRMKER: marker symbol ", symbol, " unkown!"
      pause "only for debugging purpose"

   end if

end subroutine
