! GRCURS -- read cursor position

integer function GRCURS( cursor, ix, iy, ixref, iyref,                  &
                         mode, posn, ch, inside,                        &
                         pg_xorg, pg_yorg, pg_xscl, pg_yscl,            &
                         axis_scale_x, axis_scale_y,                    &
                         control_key_down, read_cursor_valid )

   integer,          intent(in)     :: cursor
   integer,          intent(in out) :: ix, iy
   integer,          intent(in)     :: ixref, iyref, mode, posn
   character(len=*), intent(out)    :: ch
   integer,          intent(in)     :: inside
   double precision, optional :: pg_xorg, pg_yorg, pg_xscl, pg_yscl
   integer, optional :: axis_scale_x, axis_scale_y
   integer, optional :: control_key_down, read_cursor_valid
   !------ API end ------

   ! GRPCKG: Read the cursor position and a character typed by the user.
   ! The position is returned in absolute device coordinates (pixels).
   ! GRCURS positions the cursor at the position specified, and allows the
   ! user to move the cursor using the mouse or arrow keys or whatever is
   ! available on the device. When he has positioned the cursor, the user
   ! types a single character on his keyboard; GRCURS then returns this
   ! character and the new cursor position.
   !
   ! "Rubber band" feedback of cursor movement can be requested (although
   ! it may not be supported on some devices). If MODE=1, a line from the
   ! anchor point to the current cursor position is displayed as the cursor
   ! is moved. If MODE=2, a rectangle with vertical and horizontal sides
   ! and one vertex at the anchor point and the opposite vertex at the
   ! current cursor position is displayed as the cursor is moved.
   !
   ! Returns:
   !
   ! GRCURS (integer): 1 if the call was successful; 0 if the device
   !      has no cursor or some other error occurs.
   !
   ! Arguments:
   !
   ! CURSOR (integer, input):  cursor shape
   !                           (0:no change (live ?)
   !                            1:crosshair, 2:fleur, 3:arrow)
   !                           (modif É. Canot) 10 Juin 2006
   !
   ! IX    (integer, in/out): the device x-coordinate of the cursor.
   ! IY    (integer, in/out): the device y-coordinate of the cursor.
   ! IXREF (integer, input):  x-coordinate of anchor point.
   ! IYREF (integer, input):  y-coordinate of anchor point.
   ! MODE  (integer, input):  type of rubber-band feedback (0 to 8: cf PGBAND).
   ! POSN  (integer, input):  cursor position (True if > 0).
   ! CH    (char,    output): the character typed by the user; if the devic
   !      has no cursor or if some other error occurs, the value CHAR(0)
   !      [ASCII NUL character] is returned.
   ! IMIN, IMAX, JMIN, JMAX, INSIDE (integer, input):  see PGBAND (mode=8)
   !--
   !  1-Aug-1984 - Extensively revised [TJP].
   ! 29-Jan-1985 - Add ARGS and HP2648 devices (?) [KS/TJP].
   !  5-Aug-1986 - Add GREXEC support [AFT].
   ! 11-Jun-1987 - Remove built-ins [TJP].
   ! 15-Feb-1988 - Remove test for batch jobs; leave this to the device
   !               handler [TJP].
   ! 13-Dec-1990 - Remove code to abort after 10 cursor errors [TJP].
   !  7-Sep-1994 - Add support for rubber-band modes [TJP].
   ! 17-Jan-1995 - Start picture if necessary [TJP].
   ! 10-Jun-2006 - New arg added (first): CURSOR.
   ! 11-Aug-2009 - Changes for mode=8 of PGBAND.
   !               five args added : imin, imax, jmin, jmax, inside
   ! 29-Feb-2020 - Use now double precision instead of single precision.
   !  9-May-2021 - Remove args {imin, imax, jmin, jmax}, already defined in
   !               the X11 driver.
   !  9-Jul-2022 - Update w.r.t. changes made in PGBAND.
   !-----------------------------------------------------------------------

   integer :: icurs, errcnt = 0
   character :: c

   double precision :: rbuf(4)
   integer :: ibuf(12), lchr
   character(len=16) :: chr

   !------ end of declarations -- execution starts hereafter  ------

   ! Begin picture if necessary.
   if( .not. grpltd(grcide) ) call grbpic

   ! Make sure cursor is on view surface. (It does not
   ! have to be in the viewport.)
   ix = max(0,min(grxmxa(grcide),ix))
   iy = max(0,min(grymxa(grcide),iy))

   ! Does the device have a cursor?
   c = grgcap(grcide)(2:2)
   icurs = 0
   if( c == 'C' .or. c == 'X' ) icurs = 1

   if( icurs > 0 ) then ! Device does have a cursor.

      ! initial position of cursor
      ibuf(1) = ix
      ibuf(2) = iy
      ! reference point for rubber band
      ibuf(3) = ixref
      ibuf(4) = iyref
      ! rubber band mode
      ibuf(5) = mode
      ! position cursor?
      ibuf(6) = posn
      ! cursor shape
      ibuf(7) = cursor

      if( mode == 8 ) then
         ibuf(8) = inside
      end if

      if( present(pg_xorg) ) then
         ibuf(10) = 1 ! want the position output in the bottom-left corner
         rbuf(1) = pg_xorg
         rbuf(2) = pg_yorg
         rbuf(3) = pg_xscl
         rbuf(4) = pg_yscl
         if( present(axis_scale_x) ) then
            ibuf(11) = axis_scale_x ! 1=lin, 2=log
            ibuf(12) = axis_scale_y ! 1=lin, 2=log
         else
            ibuf(11) = 1
            ibuf(12) = 1
         end if
      else
         ibuf(10) = 0 ! don't want the position output in the bottom-left corner
      end if

      call grexec( grgtyp, READ_CURSOR, rbuf, ibuf, chr, lchr )
      ix = ibuf(1)
      iy = ibuf(2)
      if( mode == 100 .or. mode == 101 ) then
         if( present(control_key_down) ) then
            control_key_down = ibuf(3)
         end if
         if( present(read_cursor_valid) ) then
            read_cursor_valid = ibuf(4)
         end if
      else
         if( present(control_key_down) ) then
            control_key_down = -1
         end if
      end if
      ch = chr(1:1)
      grcurs = 1
      ! error if driver returns NULL
      if( ichar(ch) == 0 ) then
         grcurs = 0
      endif

   else ! Other devices are illegal.

      call grexec( grgtyp, GET_DEV_NAME, rbuf, ibuf, chr, lchr )
      lchr = index(chr,' ')
      if( errcnt <= 10 ) call                                           &
          grwarn('Output device has no cursor: '//chr(:lchr))
      ch = char(0)
      grcurs = 0
      errcnt = errcnt + 1

   end if

end function
