/*.......................................................................
 * Do a window resize and change some parameters accordingly
 *-----------------------------------------------------------------------
 *
 * Input:
 *  xw    XWdev * The MFPLOT /xw device descriptor.
 *  flag    int   0: the routine enters in an interactive mode
 *                1: the window size (width,height) is given by the user
 *  width   int   (not referenced if flag=0)
 *  height  int   (not referenced if flag=0)
 *
 * Output:
 *  return    int      0 - OK.
 *                     1 - Error.
 */

static int xw_win_resize ( XWdev *xw, int flag, int width, int height ) {

  XSizeHints *hints;
  int finished;
  XEvent event;

  /*
   * Device error?
   */
  if( xw->bad_device ) return 1;

  hints = XAllocSizeHints();
  if(!hints) {
    return 1;
  }

  if( flag ) {

    /*
     * Quick return if the window has already the requested size
     */
    if( width == xw->geom.width & height == xw->geom.height ) return 0;

    /* Be aware that, below, the XResizeWindow must be called AFTER the
     * WM request. If you swap the two following blocks, you will obtained
     * a blank window on time over two... */
    hints->flags = PMinSize | PMaxSize;
    hints->min_width  = width;
    hints->max_width  = width;
    hints->min_height = height;
    hints->max_height = height;
    XSetWMNormalHints( xw->display, xw->window, hints );
    XFlush( xw->display );

    XResizeWindow( xw->display, xw->window, width, height );
    // Send an Expose event
    XClearArea( xw->display, xw->window, 0, 0, 0, 0, True );

    XSelectInput( xw->display, xw->window, ExposureMask );
    finished = 0;
    while( !finished ) {
      XNextEvent( xw->display, & event );
      if( event.type == Expose ) {
        finished = 1;
      }
    }

    xw->geom.width  = width;
    xw->geom.height = height;

  } else {

    XWindowAttributes attr;
    char key;

    /* Allow the user to access to resize action (included maximizing the window) */
    hints->flags = PMinSize;
    hints->min_width  = XW_MIN_WIDTH;
    hints->min_height = XW_MIN_HEIGHT;
    XSetWMNormalHints( xw->display, xw->window, hints );
    XFlush( xw->display );

    xw_set_cursor( xw, XW_RESIZE_CURSOR );

    /* Event loop */
    XSelectInput( xw->display, xw->window, ExposureMask | KeyPressMask );
    finished = 0;
    while( !finished ) {
      XNextEvent( xw->display, &event );
      switch( event.type ) {

      case Expose:
        /* Choosing the second Graphic Context xw->gc2 is more secure because,
         * contrarily to xw->gc, it never uses clipping */
        XCopyArea( xw->display, xw->pixmap, xw->window, xw->gc2,
                   event.xexpose.x, event.xexpose.y,
                   (unsigned) event.xexpose.width, (unsigned) event.xexpose.height,
                   event.xexpose.x, event.xexpose.y );
        XFlush( xw->display );
        break;

      case KeyPress:
        {
          char buffer[10];   /* Buffer to read key definition into */
          KeySym keysym = 0; /* Key code of pressed keyboard key */
          int nret;          /* The number of characters in buffer[] */
          /* Gets the ASCII encoding associated with the key. */
          nret = XLookupString( (XKeyEvent *)&event, buffer,
                                (int)(sizeof(buffer)/sizeof(char)), &keysym,
                                NULL );
          if ( nret == 1 ) {
            key = buffer[0];
            if ( key == 27 ) { /* ESCAPE */
              finished = 1;
            }
          }
          break;
          }

      default:
        break;

      }
    }

    xw_set_cursor( xw, XW_LEFT_ARROW_CURSOR );

    /* Get new size */
    XGetWindowAttributes( xw->display, xw->window, &attr );
    XFlush( xw->display );

    xw->geom.width = attr.width;
    xw->geom.height = attr.height;

    /* Disable again the resizing of the window */
    hints->flags = PMinSize | PMaxSize;
    hints->min_width = xw->geom.width;
    hints->max_width = xw->geom.width;
    hints->min_height = xw->geom.height;
    hints->max_height = xw->geom.height;
    XSetWMNormalHints( xw->display, xw->window, hints );
    XFlush( xw->display );

  }

  XFree((char *)hints);

  /* If needed, change the pixmap size accordingly */
  xw_get_pixmap ( xw );
  pixmap_ID = xw->pixmap;

  return 0;
}

