/*.......................................................................
 * Assigns a given RGB color representation to a given color index.
 *
 * Input:
 *  xw    XWdev *  The /xw device descriptor.
 *  ci      int    The color index to assign the color to. Out of range
 *                 indexes are quietly ignored.
 *  red   double   The fractional red brightness 0..1.
 *  green double   The fractional green brightness 0..1.
 *  blue  double   The fractional blue brightness 0..1.
 * Output:
 *  return  int    0 - OK.
 *                 1 - Error.
 */
static int xw_set_rgb ( XWdev *xw, int ci,
                        double red, double green, double blue ) {
  double gray;   /* Gray-scale intensity */
  XColor *xc;   /* The descriptor of the new color */
  /*
   * Device error?
   */
  if(xw->bad_device) return 1;
  /*
   * Limits RGB values to be between 0 and 1.
   */
  if(red   < 0.0) red   = 0.0;
  if(green < 0.0) green = 0.0;
  if(blue  < 0.0) blue  = 0.0;
  if(red   > 1.0) red   = 1.0;
  if(green > 1.0) green = 1.0;
  if(blue  > 1.0) blue  = 1.0;
  /*
   * Color index in range?
   */
  if(!xw->color.monochrome && ci >= 0 && ci < xw->color.ncol) {
    /*
     * Gets the color representation descriptor.
     */
    xc = &xw->color.xcolor[ci];
    /*
     * Gets the pixel to be assigned the new color representation.
     */
    xc->pixel = 0; /* will be assigned in xw_update_colors */
    xc->flags = DoRed | DoGreen | DoBlue;
    xc->pad   = 0;
    /*
     * Determines the appropriate RGB values for the type of colormap.
     */
    switch(xw->color.vi->class) {
    case PseudoColor:
    case StaticColor:
    case DirectColor:
    case TrueColor:
      xc->red   = xw_rgb_to_xcolor(red);
      xc->green = xw_rgb_to_xcolor(green);
      xc->blue  = xw_rgb_to_xcolor(blue);
      break;
    case GrayScale:
    case StaticGray:
    /*
     * For gray-scale colormaps the red, green and blue intensities must all
     * be equal. Weight the colors so that what is brightest to the eye, is
     * also brighter in grayscale, and so that different colors of equal
     * intensity appear different in grayscale. Note that the 3 weights must
     * add up to 1.0.
     * The black and white TV standard says to use 0.3*R+0.59*G+0.11*B.
     * Unfortunately blue pretty much disappears in this scheme. The following
     * is a compromise between making all colors visible and making different
     * colors look different in grayscale.
     */
      gray = 0.35*red + 0.40*green + 0.25*blue;
      xc->red = xc->green = xc->blue = xw_rgb_to_xcolor(gray);
      break;
    };
    /*
     * Updates the recorded range of color indexes whose color representations
     * have been changed since the last call to xw_update_colors().
     */
    if(xw->color.nbuff<=0) {
      xw->color.sbuff = ci;
      xw->color.nbuff = 1;
    } else if(ci < xw->color.sbuff) {
      xw->color.nbuff += xw->color.sbuff - ci;
      xw->color.sbuff = ci;
    } else if(ci > xw->color.sbuff + xw->color.nbuff-1) {
      xw->color.nbuff = ci - xw->color.sbuff + 1;
    };
    /*
     * Registers xw_update_colors() to be called to flush the colors to the
     * window.
     */
    xw->flush_opcode_fn = (Flush_Opcode_fn) xw_update_colors;
  };

  return 0;
}
