/*.......................................................................
 * Presents the active cursor, wait for the user to press a button or
 * keyboard key, then retrack the active cursor and return the cursor
 * position and key pressed.
 *
 * Multispot (multi-zone) version of xw_read_cursor_dyn.
 *
 * Input:
 *   xw    XWdev *   The MFPLOT /xw device descriptor.
 *   ispot[], jspot[] (int) : coords of few active points where the cursor
 *                            has to be changed
 *   nbspot (int)           : number of spots
 *
 *   shape  (int)           : Cursor shape
 *
 * Output:
 *   selectedSpot (int *)   : The spot which has been clicked.
 *   pos  (XPoint *)        : The start position of the cursor. On output
 *                            this is the selected position of the cursor.
 *   key    (char *)        : If key!=NULL, the selection key will be assigned
 *                            to the caller's variable pointed to by key.
 *   return  (int)          : 0 - OK.      1 - Error.
 */

static int xw_read_cursor_dyn_multispot ( XWdev *xw,
                 int *ispot, int *jspot, int nbspot, int shape,
                 int *selectedSpot, XPoint *pos, char *key) {
  int finished = 0;       /* True when cursor succesfully read */
  XEvent event;           /* The latest event */
  int inside;             /* 1 if cursor is inside the required BBOX */
  Window p_child;         /* The child of /xw (None) containing the pointer */
  int p_win_x, p_win_y;   /* The pointer coordinates in xw->window */
  int p_root_x, p_root_y; /* The pointer coordinates in the root window */
  Window p_root_win;      /* The root window containing the cursor */
  unsigned int p_mask;    /* Bit mask of button states etc.. */
  int itest, partial_t;
  int pixel_half_width = 8;
  int k, imin, imax, jmin, jmax;

  /*
   * Device error?
   */
  if(xw->bad_device) return 1;
  /*
   * Makes sure that the window is up to date.
   */
  if(xw_flush(xw)) return xw_end_cursor(xw, NULL, 1);
  /*
   * De-iconifies and bring the window to the foreground.
   */
  XMapRaised(xw->display, xw->window);
  if(xw->bad_device) return xw_end_cursor(xw, NULL, 1);
  XSync(xw->display, False);
  if(xw->bad_device) return xw_end_cursor(xw, NULL, 1);
  /*
   * (from xw_read_cursor)
   */
  {
    long event_mask = ExposureMask | KeyPressMask | KeyReleaseMask |
                      ButtonPressMask | ButtonReleaseMask |
                      EnterWindowMask | PointerMotionMask |
                      LeaveWindowMask;
    xw_add_events(xw, event_mask);
  }
  /*
   * Discards un-handled events.
   */

  /* Warning: la modification ici ne fonctionne que si on effectue la même
   *          chose pour 'event_mask' dans la routine 'xw_new_Band' !
   */
  while(xw_check_window_event(xw, xw->window, (long)
       (ButtonPressMask | KeyPressMask | KeyReleaseMask),
        &event));

  if(xw->bad_device) return xw_end_cursor(xw, NULL, 1);

  XQueryPointer( xw->display, xw->window, &p_root_win, &p_child,
                 &p_root_x, &p_root_y, &p_win_x, &p_win_y, &p_mask );
  itest = 0;
  *selectedSpot = 0;
  for( k=0; k<nbspot; k++ ) {
    imin = ispot[k] - pixel_half_width;
    imax = ispot[k] + pixel_half_width;
    jmin = jspot[k] - pixel_half_width;
    jmax = jspot[k] + pixel_half_width;
    partial_t = imin<p_win_x && p_win_x<imax &&
                jmin<p_win_y && p_win_y<jmax;
    if( partial_t ) *selectedSpot = k+1;
    itest = partial_t || itest;
  }
  if( itest ) {
    /*
     * Presents the active cursor.
     */
    if(xw_set_cursor(xw, shape))
      return xw_end_cursor(xw, NULL, 1);
    inside = 1;
  } else {
    if(xw_set_cursor(xw, XW_LEFT_ARROW_CURSOR))
      return xw_end_cursor(xw, NULL, 1);
    inside = 0;
  }
  /*
   * Loop for cursor events.
   */
  while(!finished) {
    /*
     * Handles the next selected event.
     */

    if(xw_next_event(xw, &event))
      return xw_end_cursor(xw, NULL, 1);

    XQueryPointer( xw->display, xw->window, &p_root_win, &p_child,
                   &p_root_x, &p_root_y, &p_win_x, &p_win_y, &p_mask );
    itest = 0;
    *selectedSpot = 0;
    for( k=0; k<nbspot; k++ ) {
      imin = ispot[k] - pixel_half_width;
      imax = ispot[k] + pixel_half_width;
      jmin = jspot[k] - pixel_half_width;
      jmax = jspot[k] + pixel_half_width;
      partial_t = imin<p_win_x && p_win_x<imax &&
                  jmin<p_win_y && p_win_y<jmax;
      if( partial_t ) *selectedSpot = k+1;
      itest = partial_t || itest;
    }
    if( itest ) {
      if(!inside) {
        /*
         * Presents the active cursor.
         */
        if(xw_set_cursor(xw, shape))
          return xw_end_cursor(xw, NULL, 1);
        inside = 1;
      }
    } else { /* outside */
      if(inside) {
        xw_set_cursor(xw, XW_LEFT_ARROW_CURSOR);
        inside = 0;
      }
    }

    switch(event.type) {

    case Expose:
      if(xw_expose(xw, &event))
        return xw_end_cursor(xw, NULL, 1);
      break;

    case KeyPress:
      {
        char buffer[10];   /* Buffer to read key definition into */
        KeySym keysym = 0; /* Key code of pressed keyboard key */
        int nret;          /* The number of characters in buffer[] */
        /*
         * Gets the ASCII encoding associated with the key.
         */
        nret = XLookupString((XKeyEvent *)&event, buffer,
                   (int) (sizeof(buffer)/sizeof(char)), &keysym, NULL);
        if(xw->bad_device) return xw_end_cursor(xw, NULL, 1);
        /*
         * Detects the Ctrl key down.
         */
        if( keysym == XK_Control_L || keysym == XK_Control_R ) {
          CTRL_KEY_DOWN = 1;
        }
        /*
         * Ignores modifier keys and all but single character keys.
         */
        if(nret==1 && (keysym < XK_Shift_L || keysym > XK_Hyper_R)) {
          pos->x = event.xkey.x;
          pos->y = event.xkey.y;
          if(key) *key = buffer[0];
          if(inside) finished = 1;

          /* modif E.C. */
          if(*key == 27) { /* ESCAPE */
            CTRL_KEY_DOWN = 0;
            xw_set_cursor(xw,XW_LEFT_ARROW_CURSOR);
            finished = 1;
            break;
          }

        };
        /*
         * Checks for arrow keys.
         */
        switch(keysym) {
#ifdef XK_KP_Left
        case XK_KP_Left:
#endif
        case XK_Left:
#ifdef XK_KP_Right
        case XK_KP_Right:
#endif
        case XK_Right:
#ifdef XK_KP_Up
        case XK_KP_Up:
#endif
        case XK_Up:
#ifdef XK_KP_Down
        case XK_KP_Down:
#endif
        case XK_Down:
          if(xw_shift_cursor(xw, keysym, event.xkey.state))
            return xw_end_cursor(xw, NULL, 1);
          break;
        };
      };
      break;

    case KeyRelease:
      {
        char buffer[10];   /* Buffer to read key definition into */
        KeySym keysym = 0; /* Key code of pressed keyboard key */
        /*
         * Gets the ASCII encoding associated with the key.
         */
        XLookupString( (XKeyEvent *)&event, buffer,
                       (int) (sizeof(buffer)/sizeof(char)), &keysym, NULL );
        if(xw->bad_device) return xw_end_cursor(xw, NULL, 1);
        /*
         * modif E.C. to detect the Ctrl key up.
         */
        if( keysym == XK_Control_L || keysym == XK_Control_R ) {
          CTRL_KEY_DOWN = 0;
        }
      };
      break;

    case ButtonPress:
      /*
       * Returns the position at which the cursor was selected.
       */
      pos->x = event.xbutton.x;
      pos->y = event.xbutton.y;
      /*
       * Returns the key alias of the button that selected the cursor.
       */
      if(key) {
        switch(event.xbutton.button) {
        case Button1:
          *key = 'L';
          if( CTRL_KEY_DOWN ) *key = '4';
          break;
        case Button2:
          *key = 'M';
          if( CTRL_KEY_DOWN ) *key = '5';
          break;
        case Button3:
          *key = 'R';
          if( CTRL_KEY_DOWN ) *key = '6';
          break;
        case Button4:
          *key = '8'; /* Top */
          if( CTRL_KEY_DOWN ) *key = '9';
          break;
        case Button5:
          *key = '2'; /* Bottom */
          if( CTRL_KEY_DOWN ) *key = '3';
          break;
        default:
          *key = 'X';
          if( CTRL_KEY_DOWN ) *key = 'Y';
          break;
        };
      };
      if(inside) finished = 1;
      break;

    default:
      break;

    };

  };

  /*
   * Clean up.
   */
  return xw_end_cursor(xw, NULL, xw->bad_device!=0);
}

