/*.......................................................................
 * Presents the active cursor, wait for the user to press a button or
 * keyboard key, then retrack the active cursor and return the cursor
 * position and key pressed.
 *
 * Input:
 *  xw    XWdev *   The MFPLOT /xw device descriptor.
 *  mode    int     0 - No rubber banding.
 *                  1 - Maintain a line between (xin,yin) and the cursor.
 *                  2 - Maintain a rectangle outline with opposing
 *                      vertices at (xin,yin) and the cursor.
 *                  3..8 see 'xw_new_Band'
 *
 *  posn    int     0 - Don't attempt to position the cursor.
 *                  1 - Do try to pre-position the cursor.
 *
 *  shape  int      Cursor shape
 *
 *  ref  XPoint *   The reference position of the cursor (can be the same
 *                  as 'pos').
 * Input/Output:
 *  pos  XPoint *   The start position of the cursor. On output this is the
 *                  selected position of the cursor.
 * Output:
 *  key    char *   If key!=NULL, the selection key will be assigned to
 *                  the caller's variable pointed to by key.
 *  return  int     0 - OK.
 *                  1 - Error.
 */
static int xw_read_cursor ( XWdev *xw, int mode, int posn, int shape,
          XPoint *ref, XPoint *pos, char *key,
          int write_pos,
          double pgxorg, double pgyorg, double pgxscl, double pgyscl,
          int axis_scale_x, int axis_scale_y ) {
  int finished = 0;    /* True when cursor succesfully read */
  XEvent event;        /* The latest event */
  XPoint last;         /* Last recorded position of cursor */
  Band *bc = NULL;     /* Band-cursor descriptor */
  int warped = 0;      /* Zero until the cursor has been positioned */
  Window p_child;         /* The child of /xw (None) containing the pointer */
  int p_win_x, p_win_y;   /* The pointer coordinates in xw->window */
  int p_root_x, p_root_y; /* The pointer coordinates in the root window */
  Window p_root_win;      /* The root window containing the cursor */
  unsigned int p_mask;    /* Bit mask of button states etc.. */

  /*
   * Device error?
   */
  if ( xw->bad_device ) return 1;

  if( mode == 100 || mode == 101 ) {
     CTRL_KEY_DOWN_OLD = -1;
  }

  /*
   * Ensures that the input positions are within the pixmap and window bounds.
   */
  if ( xw_bound_cursor(xw, ref) || xw_bound_cursor(xw, pos) ) return 1;
  /*
   * Presents the active cursor.
   */
  if( xw_set_cursor(xw, shape) ) return xw_end_cursor(xw, bc, 1);
  /*
   * Makes sure that the window is up to date.
   */
  if( xw_flush(xw) ) return xw_end_cursor(xw, bc, 1);
  /*
   * De-iconifies and brings the window to the foreground.
   */
  XMapRaised( xw->display, xw->window );
  if( xw->bad_device ) return xw_end_cursor(xw, bc, 1);
  XSync(xw->display, False);
  if( xw->bad_device ) return xw_end_cursor(xw, bc, 1);
  /*
   * Sets up for modes that maintain elastic lines following the cursor.
   */
  if( (bc=xw_new_Band(xw, mode, ref))==NULL )
     return xw_end_cursor(xw, bc, 1);
  /*
   * If the cursor is in the window, locate its position,
   * after warping if requested.
   */
  if( xw_locate_cursor(xw, pos, posn, &last) ) {
    warped = 1;
    /*
     * Draws the cursor.
     */
    if( xw->bad_device || xw_bound_cursor(xw, &last) ||
        xw_draw_cursor(xw, bc, &last) ) return xw_end_cursor(xw, bc, 1);
  };
  /*
   * Discards un-handled events.
   */
  while( xw_check_window_event(xw, xw->window, (long)
         (ButtonPressMask | ButtonReleaseMask | KeyPressMask | KeyReleaseMask |
          PointerMotionMask), &event) );

  if( xw->bad_device ) return xw_end_cursor(xw, bc, 1);
  /*
   * Loop for cursor events.
   */
  while( !finished ) {
    /*
     * Handles the next selected event.
     */
    if( xw_next_event(xw, &event) ) return xw_end_cursor(xw, bc, 1);

    /* Write the world coords (x,y) in the bottom right corner of the window */
    if( write_pos ) {
      int x, y;
      double xx, yy;
      char string[32];
      XPoint pos;
      if( mode == 101 ) {
        sprintf(string,"(%0.4g, %0.4g)", X_OUT_WLD_COORD, Y_OUT_WLD_COORD);
      } else {
        XQueryPointer( xw->display, xw->window, &p_root_win, &p_child,
                       &p_root_x, &p_root_y, &p_win_x, &p_win_y, &p_mask );
        /* convert X11 coords to device coords */
        x = p_win_x - xw->geom.xmin;
        y = xw->geom.ymax - p_win_y;
        /* convert device coords to world coords */
        xx = (x*1.0 - pgxorg)/pgxscl;
        yy = (y*1.0 - pgyorg)/pgyscl;
        if( axis_scale_x == 2 ) xx = pow( 10.0, xx );
        if( axis_scale_y == 2 ) yy = pow( 10.0, yy );
        sprintf(string,"(%0.4g, %0.4g)",xx,yy);
      }
      pos.x = 7;
      pos.y = xw->geom.height - 7;
      xw_erase_output_area( xw );
      xw_direct_draw_string ( xw, string, &pos );
    }

    switch( event.type ) {

    case Expose:
      if( xw_expose(xw, &event) ) return xw_end_cursor(xw, bc, 1);
      break;

    case ButtonRelease:
      /*
       * Returns the position at which the cursor was selected.
       * Only for button 1 to 5
       */
      /* Makes initialization of key [valgrind test]*/
      *key = ' ';
      switch(event.xbutton.button) {
      case Button1: case Button2: case Button3:
                    case Button4: case Button5:
        pos->x = event.xbutton.x;
        pos->y = event.xbutton.y;
        break;
      default:
        break;
      };
      finished = 1;
      break;

    case KeyPress:
      {
        char buffer[10];   /* Buffer to read key definition into */
        KeySym keysym = 0; /* Key code of pressed keyboard key */
        int nret;          /* The number of characters in buffer[] */
        /*
         * Gets the ASCII encoding associated with the key.
         */
        nret = XLookupString( (XKeyEvent *)&event, buffer,
                              (int) (sizeof(buffer)/sizeof(char)), &keysym,
                              NULL );
        if( xw->bad_device ) return xw_end_cursor(xw, bc, 1);
        /*
         * Detects the Ctrl key down.
         */
        if( keysym == XK_Control_L || keysym == XK_Control_R ) {
          CTRL_KEY_DOWN = 1;
        }
        /*
         * Ignores modifier keys and all but single character keys.
         */
        if( nret==1 && (keysym < XK_Shift_L || keysym > XK_Hyper_R) ) {
          pos->x = event.xkey.x;
          pos->y = event.xkey.y;
          if(key) *key = buffer[0];
          finished = 1;

          /* modif E.C. */
          if( *key == 27 ) { /* ESCAPE */
            CTRL_KEY_DOWN = 0;
            xw_set_cursor(xw,XW_LEFT_ARROW_CURSOR); /* normal shape */
          }

        };
        /*
         * Checks for arrow keys.
         */
        switch(keysym) {
#ifdef XK_KP_Left
        case XK_KP_Left:
#endif
        case XK_Left:
#ifdef XK_KP_Right
        case XK_KP_Right:
#endif
        case XK_Right:
#ifdef XK_KP_Up
        case XK_KP_Up:
#endif
        case XK_Up:
#ifdef XK_KP_Down
        case XK_KP_Down:
#endif
        case XK_Down:
          if(xw_shift_cursor(xw, keysym, event.xkey.state))
            return xw_end_cursor(xw, bc, 1);
          break;
        };
      };
      break;

    case KeyRelease:
      {
        char buffer[10];   /* Buffer to read key definition into */
        KeySym keysym = 0; /* Key code of pressed keyboard key */
        /*
         * Gets the ASCII encoding associated with the key.
         */
        XLookupString( (XKeyEvent *)&event, buffer,
                       (int) (sizeof(buffer)/sizeof(char)), &keysym, NULL );
        if(xw->bad_device) return xw_end_cursor(xw, bc, 1);
        /*
         * modif E.C. to detect the Ctrl key up.
         */
        if( keysym == XK_Control_L || keysym == XK_Control_R ) {
          CTRL_KEY_DOWN = 0;
        }
      };
      break;

    case ButtonPress:
      /*
       * Returns the position at which the cursor was selected.
       */
      pos->x = event.xbutton.x;
      pos->y = event.xbutton.y;
      /*
       * Returns the key alias of the button that selected the cursor.
       */
      if(key) {
        switch(event.xbutton.button) {
        case Button1:
          *key = 'L';
          if( CTRL_KEY_DOWN ) *key = '4';
          break;
        case Button2:
          *key = 'M';
          if( CTRL_KEY_DOWN ) *key = '5';
          break;
        case Button3:
          *key = 'R';
          if( CTRL_KEY_DOWN ) *key = '6';
          break;
        case Button4:
          *key = '8'; /* Top */
          if( CTRL_KEY_DOWN ) *key = '9';
          break;
        case Button5:
          *key = '2'; /* Bottom */
          if( CTRL_KEY_DOWN ) *key = '3';
          break;
        default:
          *key = 'X';
          if( CTRL_KEY_DOWN ) *key = 'Y';
          break;
        };
      };
      finished = 1;
      break;

    case EnterNotify:
      /*
       * The cursor may still be drawn if a button was pressed when the
       * cursor was last moved out of the window. The resulting
       * passive grab will have continued to deliver motion events to
       * the MFPLOT window.
       */
      if(xw_erase_cursor(xw, bc)) return xw_end_cursor(xw, bc, 1);
      /*
       * If the cursor is in the window, locate its position. If this is
       * the first time that the cursor has been in the window and warping
       * has been requested, this also involves pre-positioning the cursor
       * and setting input focus.
       */
      if(xw_locate_cursor(xw, pos, posn && !warped, &last)) {
        warped = 1;
        /*
         * Draws the cursor.
         */
        if(xw->bad_device || xw_bound_cursor(xw, &last) ||
           xw_draw_cursor(xw, bc, &last))
          return xw_end_cursor(xw, bc, 1);
      };
      break;

    case LeaveNotify:
      if(xw_erase_cursor(xw, bc)) return xw_end_cursor(xw, bc, 1);
      break;

    case MotionNotify:
      /*
       * Discards all but the last MotionNotify event.
       */
      while(xw_check_window_event(xw, xw->window,
                                  (long)(PointerMotionMask),
                                  &event));
      if( xw->bad_device || xw_erase_cursor(xw, bc) )
        return xw_end_cursor(xw, bc, 1);
      last.x = event.xmotion.x;
      last.y = event.xmotion.y;
      if( xw_bound_cursor(xw, &last) || xw_draw_cursor(xw, bc, &last) )
        return xw_end_cursor(xw, bc, 1);
      break;

    default:
      break;

    };

  };

  /* here, 'read_cursor' finished */
  if( mode == 8 ) {
    /* correction */
    pos->x = bc->ref.x + bc->width/2;
    pos->y = bc->ref.y + bc->height/2;
  }

  /*
   * Clean up.
   */
  return xw_end_cursor(xw, bc, xw->bad_device!=0);
}
