/*.......................................................................
 * Agrees upon a window geometry.
 *
 * Input:
 *  xw         XWdev *  The MFPLOT /xw device descriptor. Only the display
 *                      and screen members are required.
 *  x            int    The left edge of the window.
 *  y            int    The top edge of the window.
 *  width   unsigned    The width of the window.
 *  height  unsigned    The height of the window.
 * Output:
 *  xw->geom  XWgeom    The new window geometry.
 *  return       int    0 - OK.
 *                      1 - Error.
 */
static int xw_new_geom ( XWdev *xw, int x, int y, unsigned int width,
                         unsigned int height ) {
  unsigned int d_pix_width;   /* Display width in pixels */
  unsigned int d_pix_height;  /* Display height in pixels */
  unsigned int d_mm_width;    /* Display width in mm */
  unsigned int d_mm_height;   /* Display height in mm */

  /*
   * Device error?
   */
  if(xw->bad_device) return 1;

  /*
   * Records the window size.
   */
  xw->geom.width = width;
  xw->geom.height = height;

  /*
   * Determines the current display width and height in mm and pixels.
   */
  d_pix_width  = XDisplayWidth(    xw->display, xw->screen );
  d_mm_width   = XDisplayWidthMM(  xw->display, xw->screen );
  d_pix_height = XDisplayHeight(   xw->display, xw->screen );
  d_mm_height  = XDisplayHeightMM( xw->display, xw->screen );
  /*
   * Determines the device resolution in pixels per inch. (actually, it will
   * be most of time 96 dpi, because Xlib is not able to access to the true
   * physical dimension of the screen(s))
   * To adapt itself to High DPI screens, we should send a request to xrandr.
   *
   * (modif. E.C. : round to the nearest integer)
   */
  xw->geom.xpix_per_inch = xw_nint( 25.4*((double)d_pix_width  / (double)d_mm_width) );
  xw->geom.ypix_per_inch = xw_nint( 25.4*((double)d_pix_height / (double)d_mm_height) );
  /*
   * Determines the number of pixels needed to form a 1/4" margin around
   * the plot area.
   */
  xw->geom.xmargin = xw_nint( 0.25*xw->geom.xpix_per_inch );
  xw->geom.ymargin = xw_nint( 0.25*xw->geom.ypix_per_inch );
  /*
   * Determines the pixel coordinates that enclose an area bounded by 1/4"
   * margins. (~ 6.3 mm)
   */
  xw->geom.xmin = xw->geom.xmargin;
  xw->geom.xmax = xw->geom.width - xw->geom.xmargin;
  xw->geom.ymin = xw->geom.ymargin;
  xw->geom.ymax = xw->geom.height - xw->geom.ymargin;

  /* Reveals the window */
  XMapRaised(xw->display, xw->window);
  XFlush( xw->display );

  return 0;
}
