/*.......................................................................
 * Opens a X11 window and return an initialized MFPLOT device descriptor.
 *
 * Input:
 *  ident    int   The MFPLOT identifier for the window
 *  x, y, width, height (int) : the wished geometry of the window
 * Output:
 *  return    XWdev *   The MFPLOT /xw device descriptor, or NULL on error.
 */
static XWdev * xw_new_XWdev ( int ident, int x, int y,
                              int width, int height ) {
  XWdev *xw;     /* The descriptor to be returned */
  /*
   * Allocates the descriptor.
   */
  xw = (XWdev *) malloc(sizeof(XWdev));
  if( xw==NULL ) return NULL;
  /*
   * Initializes all members of the descriptor at least to the point at which
   * the descriptor can safely be sent to xw_del_XWdev(xw,0). All pointers
   * must be assigned NULL and XIDs assigned None, so that xw_del_XWdev()
   * knows what hasn't been allocated yet.
   */
  xw->display = NULL;
  xw->parent = None;
  xw->window = None;
  xw->number = ident;
  xw->screen = 0;
  xw->bad_device = 0;
  xw->last_error = 0;
  xw->xftdraw_no_pixmap = NULL;
  xw->xftdraw_pixmap = NULL;
  xw->xftdraw_pixmap_aux_1 = NULL;
  xw->xftdraw_pixmap_aux_2 = NULL;
  xw->pixmap = None;
  xw->pixmap_aux_1 = None;
  xw->pixmap_aux_2 = None;
  xw->font = NULL;
  xw->xft_angle = 1000.;
  xw->xft_fontname[0] = '\0';
  xw->color.cmap = None;
  xw->norm_cursor = None;
  xw->live_cursor = None;
  xw->crosshair = 0;
  xw->resize_cursor = None;
  xw->watch_cursor = None;
  xw->hand1_cursor = None;
  xw->hand2_cursor = None;
  xw->zoom_cursor = None;
  xw->poly.points = NULL;
  xw->poly.ndone = xw->poly.npoint = 0;
  xw->gc = NULL;
  xw->gc2 = NULL;
  xw->color.vi = NULL;
  xw->color.cmap = None;
  xw->color.ncol = 0;
  xw->color.monochrome = 1;
  xw->color.pixel = NULL;
  xw->color.xcolor = NULL;
  xw->color.initialized = 0;
  xw->color.nbuff = 0;
  xw->color.sbuff = 0;
  xw->update.modified = 0;
  xw->event.mask = NoEventMask;
  xw->event.no_buttons = 0;
  xw->image.xi = NULL;
  xw->last_opcode = 0;
  xw->flush_opcode_fn = (Flush_Opcode_fn) 0;

  /*
   * Opens a connection to the X display server.
   */
  xw->display = XOpenDisplay( NULL );
  if( xw->display==NULL ) {
    fprintf( stderr, "%s: cannot connect to X server [%s]\n", XW_IDENT,
             XDisplayName(NULL) );
    return xw_del_XWdev(xw,0);
  };
  /*
   * Installs an error handler for non-fatal errors. If we don't do this then
   * Xlib will do its own error handling, which includes killing the program.
   */
  XSetErrorHandler(xw_error);
  /*
   * Gets the index of the screen cited in the display string.
   */
  xw->screen = DefaultScreen(xw->display);
  /*
   * Also records the parent window ID.
   */
  xw->parent = RootWindow(xw->display, xw->screen);

  /*
   * Gets a new MFPLOT window.
   */
  if( xw_get_window(xw, x, y, width, height) == None ) return xw_del_XWdev(xw,0);
  /*
   * We want to know if the MFPLOT window gets destroyed.
   */
  if( xw_add_events(xw, (long) StructureNotifyMask) ) return xw_del_XWdev(xw,0);
  /*
   * Gets the visual and colormap of the window.
   */
  if( xw_get_visual(xw) ) return xw_del_XWdev(xw,0);

  if( xw_new_geom(xw, x, y, width, height) ) return xw_del_XWdev(xw,0);
  /*
   * Instates the normal cursor for the window.
   */
  if( xw_set_cursor(xw, XW_LEFT_ARROW_CURSOR) ) return xw_del_XWdev(xw,0);
  /*
   * Creates and initialize a graphical context (GC) descriptor. This is
   * where line widths, line styles, fill styles, plot color, etc. are
   * recorded.
   */
  xw->gcv.line_width = 1;
  xw->gcv.fill_rule = EvenOddRule;
  xw->gcv.graphics_exposures = False;
  xw->gcv.foreground = WhitePixel(xw->display, xw->screen);
  xw->gc = XCreateGC ( xw->display, xw->window, (unsigned long) (GCLineWidth |
                       GCFillRule | GCGraphicsExposures | GCForeground),
                       &xw->gcv );
  if( xw->gc==NULL || xw->bad_device ) {
    fprintf( stderr, "%s: Failed to allocate graphical context.\n", XW_IDENT );
    return xw_del_XWdev(xw,0);
  };

  /*
   * Creates a second graphical context (GC) descriptor, which will never
   * have a ClipMask -- see xw_flush().
   */
  xw->gc2 = XCreateGC ( xw->display, xw->window,
                        (unsigned long) (GCGraphicsExposures | GCForeground),
                        &xw->gcv );
  if( xw->gc2==NULL || xw->bad_device ) {
    fprintf( stderr, "%s: Failed to allocate the 2nd graphical context.\n", XW_IDENT );
    return xw_del_XWdev(xw,0);
  };

  /*
   * Allocates the buffers that will be used to compose a line of pixels.
   */
  if( xw_get_image(xw, XW_IMAGE_LEN) ) return xw_del_XWdev(xw,0);

  /*
   * Set the window title.
   */
  {
    XTextProperty window_name;
    char name[128];
    char * w_name;
    if( strlen(EXECUTABLE_NAME) == 0 ) {
      if( strlen(TITLE_DESCRIPTION) == 0 ) {
        sprintf( name, "Figure %i", MF_WIN_ID );
      } else {
        if( FIGURE_NUM_IN_TITLE ) {
          sprintf( name, "Figure %i: %s", MF_WIN_ID, TITLE_DESCRIPTION );
        } else {
          sprintf( name, "%s", TITLE_DESCRIPTION );
        }
      }
    } else {
      if( strlen(TITLE_DESCRIPTION) == 0 ) {
        sprintf( name, "%s - Figure %i", EXECUTABLE_NAME, MF_WIN_ID );
      } else {
        if( FIGURE_NUM_IN_TITLE ) {
          sprintf( name, "%s - Figure %i: %s", EXECUTABLE_NAME, MF_WIN_ID, TITLE_DESCRIPTION );
        } else {
          sprintf( name, "%s - %s", EXECUTABLE_NAME, TITLE_DESCRIPTION );
        }
      }
    };
    w_name = name;
    if( XStringListToTextProperty(&w_name, 1, &window_name) == 0 ) {
      fprintf( stderr, "%s: Error allocating window name.\n", XW_IDENT );
      return xw_del_XWdev(xw,0);
    };
    XSetWMName( xw->display, xw->window, &window_name );
    XFree((char *)window_name.value);
  };

  /*
   * Initialize MFPLOT basic colors.
   */
  xw_init_colors ( xw );

  /*
   * Return the focus to the terminal.
   */
  xw_set_focus_win ( xw, 0 );

  /*
   * Returns the initialized descriptor for use.
   */
  return xw;
}

