/*.......................................................................
 * Moves an image interactively in the window
 *  1 - copies (white is transparent, due to the use of a ClipMask)
 *      'pixmap_aux_1' (created elsewhere) at initial position 'TL'.
 *  2 - using the mouse, you can move interactively 'pixmap_aux_1' inside
 *      the window, until the ESC key is pressed.
 *  3 - returns the new position in 'TL'.
 *
 * During the move, the overwritten portion of the image is continuously
 * saved in a local pixmap ('pixmap_save') to restore the drawing
 * after each move.
 * If available, another pixmap (pixmap_aux_2, which contains the foreground
 * drawing) is continuously drawn over the main pixmap at position TL2,
 * after the image, using also white as a transparent color.
 * The object to move may be clipped at the viewport: it uses CLIPPED_OBJ,
 * which has not the same meaning as CLIPPING.
 * Also, uses the 2nd GC, which never have a ClipMask.
 *-----------------------------------------------------------------------
 *
 * Input:
 *  xw     XWdev *  The MFPLOT /xw device descriptor.
 *  TL     XPoint   The starting point (Top-Left corner in the window)
 *  TL2    XPoint   The starting point of the foreground part (if needed)
 *
 */

static void xw_move_rect_in_pixmap ( XWdev *xw,
                                     XPoint *TL, XPoint *TL2 ) {
  Pixmap pixmap_save, clipMask, clipMask_clp;
  Window root;
  int x, y;
  unsigned int width, height, border, depth;
  GC gc, gc_clp;
  XRectangle rectangles[1]; /* Array of rectangles (here, only one) */

  /* variable declaration for the foreground drawing */
  Pixmap clipMask2;
  unsigned int width2, height2;
  GC gc2;

  struct my_struct_arg *arg;

  /*
   * Device error?
   */
  if ( xw->bad_device ) return;

  if ( xw->pixmap_aux_1 == None ) return;

  /* Gets width and height from pixmap_aux_1 */
  XGetGeometry ( xw->display, xw->pixmap_aux_1, &root, &x, &y,
                 &width, &height, &border, &depth );

  /* Saves the whole main pixmap */
  pixmap_save = XCreatePixmap ( xw->display, root,
                                xw->geom.width, xw->geom.height,
                                (unsigned) xw->color.vi->depth );
  XCopyArea ( xw->display, xw->pixmap, pixmap_save, xw->gc2,
              0, 0, xw->geom.width, xw->geom.height, 0, 0);

  /* Creates a bitmap mask from pixmap_aux_1 (depth=1) */
  clipMask = xw_bitmap_from_pixmap ( xw, xw->pixmap_aux_1 );

  /* Defines the clipping mask for this graphics context */
  gc = XCreateGC ( xw->display, xw->pixmap, 0, NULL ); /* 0 -> use defaults */

  if ( CLIPPED_OBJ ) {
    /* define another mask, clipped itself at the viewport */
    clipMask_clp = XCreatePixmap ( xw->display, root,
                                   width, height, 1 );
    /* Very important: must use a special GC (not the default one)
     * adapted to the depth of the bitmap */
    gc_clp = XCreateGC ( xw->display, clipMask_clp, 0, NULL ); /* 0 -> use defaults */
    XSetForeground( xw->display, gc_clp, 0x000000 ); /* black */
    XFillRectangle ( xw->display, clipMask_clp, gc_clp, 0, 0, width, height );
    /* Define the rectangle for the clipping zone;
     * (the shift added constrains the clipping zone to be strictly located
     *  inside the axes) */
    rectangles[0].x = 0;
    rectangles[0].y = 0;
    rectangles[0].width  = BRC.x - TLC.x + 1 -2;
    rectangles[0].height = BRC.y - TLC.y + 1 -2;
    XSetClipRectangles ( xw->display, gc_clp,
                         TLC.x - TL->x +1, TLC.y - TL->y +1, rectangles,
                         1, Unsorted );
    XCopyArea ( xw->display, clipMask, clipMask_clp, gc_clp,
                0, 0, width, height, 0, 0 );
    XSetClipMask ( xw->display, gc, clipMask_clp );
  } else { /* no clipping at viewport */
    XSetClipMask ( xw->display, gc, clipMask );
  }

  XSetClipOrigin ( xw->display, gc, TL->x, TL->y );

  /* Copies the moving object at the prescribed initial location;
   * the clipping mask makes the white transparent */
  XCopyArea ( xw->display, xw->pixmap_aux_1, xw->pixmap, gc,
              0, 0, width, height, TL->x, TL->y );
  xw_mark_modified ( xw, TL->x, TL->y, 1 );
  xw_mark_modified ( xw, TL->x+width, TL->y+height, 1 );

  if( xw->pixmap_aux_2 != None ) {
    XGetGeometry ( xw->display, xw->pixmap_aux_2, &root, &x, &y,
                   &width2, &height2, &border, &depth );
    clipMask2 = xw_bitmap_from_pixmap ( xw, xw->pixmap_aux_2 );
    gc2 = XCreateGC ( xw->display, xw->pixmap, 0, NULL ); /* 0 -> use defaults */
    XSetClipMask ( xw->display, gc2, clipMask2 );
    XSetClipOrigin ( xw->display, gc2, TL2->x, TL2->y );
    XCopyArea ( xw->display, xw->pixmap_aux_2, xw->pixmap, gc2,
                0, 0, width2, height2, TL2->x, TL2->y );
    xw_mark_modified ( xw, TL2->x, TL2->y, 1 );
    xw_mark_modified ( xw, TL2->x+width2, TL2->y+height2, 1 );
  }

  xw_flush ( xw );

  ADC_xmin = TL->x - 1;
  ADC_ymin = TL->y - 1;
  ADC_width = width + 1; /* strange: I thought it should be +2, but +1 only is correct */
  ADC_height = height + 1;

  arg = malloc(sizeof(struct my_struct_arg));
  arg->xw = xw;

  pthread_create ( &ADC_thread, NULL, xw_animate_dashed_contour, arg );

  while(1) {
    XPoint pos, pos_ref;
    int imin, imax, jmin, jmax;
    char key;
    int dx, dy; // shifted pixels for the image moving

    imin = TL->x;
    imax = TL->x+width;
    jmin = TL->y;
    jmax = TL->y+height;

    /* waiting for a click (mouse down) or a key pressed... */
    xw_read_cursor_dyn ( xw, imin,imax,jmin,jmax,
                         0,0,0,0, 0,
                         XW_OPENED_HAND_CURSOR, &pos, &key,
                         0, 0,0,0,0, 0,0 );

    if(key == 27) { /* ESCAPE */
      ADC_finished = 1;
      pthread_join ( ADC_thread, NULL );
      break;
    } else if(key == 'L') { /* left button click (down) */
      xw_set_cursor(xw,XW_CLOSED_HAND_CURSOR);
      /* Registers current pointer position */
      xw_current_pos(xw, &pos_ref);
      while(1) {
        /* Get position */
        if ( xw_current_pos(xw, &pos) == 1 ) { /* inside the window */
          /* Moves the image if there is at least one pixel of difference,
           * and only if the pointer is inside the window */
          dx = pos.x - pos_ref.x;
          dy = pos.y - pos_ref.y;
          if ( abs(dx)>0 || abs(dy)>0 ) {

            /* Restores pixmap_save -- difficult to optimize because the
             * background color is transparent.
             * Small extension of 2 pixels around the grobj, in order
             * to be sure to cover the additional animated contour */
            XCopyArea ( xw->display, pixmap_save, xw->pixmap, xw->gc2,
                        TL->x-2, TL->y-2, width+4, height+4, TL->x-2, TL->y-2 );
            xw_mark_modified ( xw, TL->x-2, TL->y-2, 1 );
            xw_mark_modified ( xw, TL->x+width+2, TL->y+height+2, 1 );

            if( CLIPPED_OBJ ) {
              XFillRectangle ( xw->display, clipMask_clp, gc_clp, 0, 0, width, height );
              XSetClipRectangles ( xw->display, gc_clp,
                                   TLC.x - (TL->x+dx) +1, TLC.y - (TL->y+dy) +1,
                                   rectangles, 1, Unsorted );
              XCopyArea ( xw->display, clipMask, clipMask_clp, gc_clp,
                          0, 0, width, height, 0, 0 );
              XSetClipMask ( xw->display, gc, clipMask_clp );
            }
            /* Moves clipping mask in the graphics context */
            XSetClipOrigin ( xw->display, gc, TL->x+dx, TL->y+dy );

            TL->x += dx;
            TL->y += dy;
            ADC_xmin = TL->x - 1;
            ADC_ymin = TL->y - 1;

            /* Copies image at the new position */
            XCopyArea ( xw->display, xw->pixmap_aux_1, xw->pixmap, gc,
                        0, 0, width, height, TL->x, TL->y );
            xw_mark_modified ( xw, TL->x, TL->y, 1 );
            xw_mark_modified ( xw, TL->x+width, TL->y+height, 1 );

            /* If needed, copies also the foreground portion */
            if( xw->pixmap_aux_2 != None ) {
              XCopyArea ( xw->display, xw->pixmap_aux_2, xw->pixmap, gc2,
                          0, 0, width2, height2, TL2->x, TL2->y );
              xw_mark_modified ( xw, TL2->x, TL2->y, 1 );
              xw_mark_modified ( xw, TL2->x+width2, TL2->y+height2, 1 );
            }

            /* Flushes XW buffer because we want a continuously updated window */
            xw_flush ( xw );

            pos_ref = pos;
          }
          if ( xw_get_click(xw) == 1 ) {
             /* got a click (actually mouse up) */
             break;
          }
        }
      }
    }
  }

  XFreePixmap( xw->display, pixmap_save );
  XFreeGC( xw->display, gc );
  if( CLIPPED_OBJ ) {
    XFreeGC( xw->display, gc_clp );
  }
  if( xw->pixmap_aux_2 != None ) {
    XFreeGC( xw->display, gc2 );
  }

}

