/*.......................................................................
 * This function is called by X whenever a non-fatal error occurs
 * on a given display connection. For the moment it does nothing but
 * count such errors in an internal static error counter. This counter
 * can then be queried and reset by sending a NULL error event pointer.
 *
 * Input:
 *  display    Display *  The display connection on which the error occured.
 *  event  XErrorEvent *  The descriptor of the error event, or NULL to
 *                        request that the error counter be queried and reset.
 * Output:
 *  return         int    The return value is not specified by Xlib, so
 *                        for Xlib calls we will simply return 0. For
 *                        none Xlib calls (distinguishable by sending
 *                        event==NULL), the value of the error counter
 *                        is returned.
 */
static int xw_handle_error(Display *display, XErrorEvent *event) {
  static int error_count = 0;
/*
 * To query and reset the error counter, this program calls xw_handle_error()
 * with a NULL error event pointer. This distinguishes it from a call
 * from Xlib.
 */
  if(!event) {
    int ret_count = error_count;
    error_count = 0;   /* Reset the error counter */
    return ret_count;  /* Return the pre-reset value of the error counter */
#ifdef DEBUG
  } else {
    char errtxt[81]; /* Buffer to receive error message in */
/*
 * Get a message describing the error.
 */
    XGetErrorText(display, (int)event->error_code, errtxt, (int)sizeof(errtxt));
    fprintf(stderr, "xw_handle_error: XErrorEvent: %s\n", errtxt);
/*
 * Report the operation that caused it. These opcode numbers are listed in
 * <X11/Xproto.h>.
 */
    fprintf(stderr, "xw_handle_error: Major opcode: %d, Resource ID: 0x%lx%s.\n",
            (int) event->request_code,
            (unsigned long) event->resourceid,
            (event->resourceid==DefaultRootWindow(display)?" (Root window)":""));
#endif
  };
/*
 * Keep a record of the number of errors that have occurred since the
 * error counter was last cleared.
 */
  error_count++;
  return 0;
}

