/*.......................................................................
 * Gets a new MFPLOT window.
 *
 * Input:
 *  xw       XWdev *  The MFPLOT /xw device descriptor.
 * Output:
 *  xw->window        The new window ID.
 *  return  Window    The window ID, or 'None' on error.
 */
static Window xw_get_window ( XWdev *xw, int x, int y, unsigned int width,
                              unsigned int height ) {

  /*
   * Device error?
   */
  if(xw->bad_device) return None;

  /*
   * Get the configuration defaults for the window.
   */
  xw->color.ncol = XW_MAX_COLORS;

  /*
   * Initialize the color descriptor.
   */
  xw->color.vi = NULL;
  xw->color.cmap = None;
  xw->color.ncol = 2;
  xw->color.pixel = NULL;
  xw->color.monochrome = 1;
  /*
   * Get the XVisualInfo structure for the default visual.
   */
  xw->color.vi = xw_visual_info ( xw->display, xw->screen,
                                  DefaultVisual(xw->display, xw->screen) );
  /*
   * Set the window size.
   */
  xw->geom.width  = width;
  xw->geom.height = height;

  /*
   * Create the window.
   */
  {
    XSetWindowAttributes attr;
    unsigned long mask = CWEventMask | CWDontPropagate | CWBackingStore |
                         CWBorderPixel | CWBackPixel | CWCursor;
    attr.event_mask    = ExposureMask | StructureNotifyMask |
                         ResizeRedirectMask;
    attr.do_not_propagate_mask = ButtonPressMask | ButtonReleaseMask |
                                 KeyPressMask | KeyReleaseMask;

    if( BLACK_ON_WHITE == 1 ) {
       attr.border_pixel  = BlackPixel(xw->display, xw->screen);
       attr.background_pixel = WhitePixel(xw->display, xw->screen);
    } else {
       attr.border_pixel  = WhitePixel(xw->display, xw->screen);
       attr.background_pixel = BlackPixel(xw->display, xw->screen);
    };

    attr.backing_store = Always;

    attr.cursor = xw->norm_cursor;
    mask |= CWColormap;
    attr.colormap = xw->color.cmap;
    /*
     * Bracket the window acquisition with xw_sync_error() calls, to
     * determine whether any window creation errors occur.
     */
    xw_sync_error(xw);
    xw->window = XCreateWindow( xw->display, xw->parent, x, y,
                                xw->geom.width, xw->geom.height,
                                XW_BORDER, xw->color.vi->depth, InputOutput,
                                xw->color.vi->visual, mask, &attr );
    if(xw_sync_error(xw) || xw->window == None) {
      fprintf( stderr, "Failed to create window with visual: id=0x%lx class=%d depth=%u.\n",
               (unsigned long)xw->color.vi->visualid,
               xw->color.vi->class, xw->color.vi->depth);
      fprintf( stderr, "Colormap id=0x%lx.\n", (unsigned long) xw->color.cmap);
      xw->window = None;
      return (Window) xw_del_XWdev(xw,0);
    };
  };

  /*
   * Creates the cursors for the window.
   */
  xw_new_cursors( xw );
  XDefineCursor( xw->display, xw->window, xw->norm_cursor );

  /*
   * Arrange to be informed of window manager "delete window" actions,
   * and do nothing when it will be catched.
   */
  {
    Atom WM_message;
    WM_message = XInternAtom( xw->display, "WM_DELETE_WINDOW", False );
    XSetWMProtocols ( xw->display, xw->window, &WM_message, 1 );
  }

  /*
   * Tell the window-manager geometry hints for the current window.
   */
  {
    XSizeHints *hints = XAllocSizeHints();
    if(hints) {
      hints->flags = USPosition | USSize | PMinSize | PMaxSize;
      hints->x = x;
      hints->y = y;
      hints->width = xw->geom.width;
      hints->height = xw->geom.height;
      /* This tip is to avoid that the user change the window size
       * (no event loop to detect a resize request and redraw the window
       *  accordingly) */
      hints->min_width = xw->geom.width;
      hints->max_width = xw->geom.width;
      hints->min_height = xw->geom.height;
      hints->max_height = xw->geom.height;
      /*
       * Instate the new size hints in the WM_NORMAL_HINTS property.
       */
      XSetWMNormalHints( xw->display, xw->window, hints );
      XFree((char *)hints);
    };
  };

  XFlush( xw->display );

  return xw->window;
}

