/*.......................................................................
 * Check the pixmap size, and create a new one if required.
 *
 * Input:
 *  xw       XWdev *  The MFPLOT /xw device descriptor.
 * Output:
 *  xw->pixmap        The pixmap ID.
 *  return  Pixmap    The pixmap ID, or 'None' on error.
 */
static Pixmap xw_get_pixmap ( XWdev *xw ) {

  unsigned long white = 0xFFFFFF, black = 0x0, background;

  /*
   * Device error?
   */
  if(xw->bad_device) return None;

  /*
   * Delete the current pixmap if it doesn't have the required size.
   */
  if(xw->pixmap != None) {
    Window root;
    int x, y;
    unsigned width, height, border, depth;
    /*
     * Determine the size of the existing pixmap.
     */
    XGetGeometry( xw->display, xw->pixmap, &root, &x, &y, &width, &height,
                  &border, &depth );
    /*
     * If the pixmap doesn't have a size equal to that requested, delete it.
     */
    if( width != xw->geom.width || height != xw->geom.height ) {
      XFreePixmap( xw->display, xw->pixmap );
      xw->pixmap = None;
    } else {
      return xw->pixmap;
    };
  };

  /*
   * Create a new pixmap if necessary.
   */
  if( xw->pixmap == None ) {
    /*
     * Bracket the pixmap acquisition with xw_sync_error() calls, to
     * determine whether any errors occur.
     */
    xw_sync_error(xw);
    xw->pixmap = XCreatePixmap( xw->display, xw->window,
                                xw->geom.width, xw->geom.height,
                                (unsigned) xw->color.vi->depth );
    if( xw_sync_error(xw) || xw->pixmap==None ) {
      fprintf( stderr, "MFPLOT: Failed to allocate %dx%d pixmap.\n",
               xw->geom.width, xw->geom.height );
      xw->pixmap = None;
    };
    /* If an XftDraw surface was attached to this pixmap, update it */
    if( xw->xftdraw_pixmap ) {
      XftDrawChange ( xw->xftdraw_pixmap, xw->pixmap );
    }
  };

  /*
   * Set the specified fill-color.
   */
  if( BLACK_ON_WHITE ) {
    background = white;
  } else {
    background = black;
  }
  XSetForeground( xw->display, xw->gc, background );
  /*
   * Clear the pixmap.
   */
  if( xw->pixmap != None )
    XFillRectangle( xw->display, xw->pixmap, xw->gc, 0, 0,
                    xw->geom.width, xw->geom.height );
  /*
   * Clear the window.
   */
  XClearWindow( xw->display, xw->window );
  XFlush( xw->display );

  return xw->pixmap;
}

