/*.......................................................................
 * Locates an executable by searching for it in a list of directories.
 * The list of directories is a string containing directory paths
 * separated by ':'s. If the first or last character
 * in the path is one of these terminators, or if two of these terminators
 * are adjacent in the path, then the current directory is included in
 * the search. Each directory in the path is searched in order and the
 * first match is returned.
 *
 * Note that these semantics are identical to the UNIX Bourne-shell
 * treatment of the PATH environment variable.
 *
 * In order that getenv() can be used directly as an argument to this
 * function without invoking error messages when getenv() returns
 * NULL, either or both of the 'path' and 'program' arguments can be
 * NULL. In this case xw_find_exe() quietly returns NULL as though it had
 * searched for the executable and failed to find it.
 *
 * Input:
 *  program  char *   The name of the program to be located. In the case
 *                    of this being NULL, xw_find_exe() will quietly abort
 *                    and return NULL. This allows one to use getenv()
 *                    without worrying about the NULL return case.
 *  path     char *   A colon-separated, '\0' terminated list of directory
 *                    paths to search for the program in.
 * Output:
 *  return   char *   The full name of the executable, or NULL if not found.
 *                    The returned pointer is malloc()d memory and should
 *                    be free()d by the caller when no longer required.
 */

static char * xw_find_exe ( char *path, char *program ) {
  char *dir;       /* Pointer to start of directory in 'path' */
  char *buf=NULL;  /* A buffer used to compile file names in */
  int buflen=0;    /* The size of the dynamic buffer in char's */
  int prog_len;    /* Length of program name */
  int dirlen;      /* Length of directory name pointed to by 'dir' */
  int path_len;    /* Length of latest path name */
  char *exe = "";    /* UNIX doesn't add extensions */
  char *sep = "/";   /* UNIX directory/file separator */
  int term  = ':';   /* Directory separator in path (in addition to '\0') */
  /*
   * No path or executable?
   */
  if(path==NULL || program==NULL) return NULL;
  /*
   * Allocates memory for the filename buffer.
   */
  buflen = strlen(program) + 40;
  buf = (char *) malloc(sizeof(char) * (buflen+1));
  if(buf==NULL) {
    fprintf( stderr, "%s: Insufficient memory to locate program: %s\n",
             XW_IDENT, program );
    return buf;
  };
  /*
   * Determines the length of the program name.
   */
  prog_len = strlen(program);
  /*
   * Seeks the program in each 'term' separated path name.
   */
  do {
    /*
     * Maintains a pointer to the start of the directory path.
     */
    dir = path;
    /*
     * Finds the directory terminator.
     */
    while(*path && *path != term) path++;
    /*
     * Records the path length.
     */
    dirlen = path - dir;
    /*
     * Skips the trailing terminator unless at the end of the path.
     */
    if(*path) path++;
    /*
     * Combines the directory and command file name into a full program
     * name.
     */
    path_len = dirlen + strlen(sep) + prog_len + strlen(exe) ;
    if(path_len > buflen) {
      char *new_buf = realloc(buf, (path_len+1) * sizeof(char));
      if(new_buf==NULL) {
        fprintf( stderr, "%s: Insufficient memory to locate program: %s\n",
                 XW_IDENT, program );
        free(buf);
        return buf;
      };
      buf = new_buf;
    };
    sprintf(buf, "%.*s%s%s%s", dirlen, dir, dirlen==0 ? "":sep, program, exe);
    /*
     * See if the executable file exists.
     */
#ifndef X_OK
#define X_OK 1
#endif
#ifndef R_OK
#define R_OK 4
#endif
    if(access(buf, X_OK)==0) return buf;
  } while(*path);
  /*
   * Executable file not found.
   */
  free(buf);
  return NULL;
}
