
/*.......................................................................
 * Draws the current cursor.
 *
 * Input:
 *  xw   XWdev *   The MFPLOT /xw device descriptor.
 *  bc    Band *   A cursor descriptor returned by xw_new_band().
 *  end XPoint *   The current cursor position.
 * Output:
 *  return int     0 - OK.
 *                 1 - Error.
 */
static int xw_draw_cursor ( XWdev *xw, Band *bc, XPoint *end ) {
  XGCValues gr_values;
  /*
   * Stores the new end point.
   */
  bc->end = *end;
  /*
   * Inverses video (added by E.C.)
   */
  gr_values.function = GXinvert;
  XChangeGC( xw->display, xw->gc, GCFunction, &gr_values );
  /*
   * Draws the cursor.
   */
  switch(bc->mode) {
  case 0: default:
    if(xw->gc!=NULL && xw->crosshair) {
      XDrawLine( xw->display, xw->window, xw->gc, 0, bc->end.y,
                 (int)xw->geom.width-1, bc->end.y );
      XDrawLine( xw->display, xw->window, xw->gc, bc->end.x, 0,
                 bc->end.x, (int)xw->geom.height-1 );
    };
    break;
  case 1:
    XDrawLine( xw->display, xw->window, xw->gc, bc->ref.x, bc->ref.y,
               bc->end.x, bc->end.y );
    break;
  case 2:  /* Draws a rectangle */
    {
      int x = bc->ref.x < bc->end.x ? bc->ref.x : bc->end.x;
      int y = bc->ref.y < bc->end.y ? bc->ref.y : bc->end.y;
      unsigned int width = (unsigned int) abs(bc->ref.x - bc->end.x);
      unsigned int height = (unsigned int) abs(bc->ref.y - bc->end.y);
      XDrawRectangle( xw->display, xw->window, xw->gc, x, y, width, height );
    };
    break;
  case 3:  /* Two horizontal lines */
    XDrawLine( xw->display, xw->window, xw->gc, 0, bc->end.y,
               (int)xw->geom.width-1, bc->end.y );
    XDrawLine( xw->display, xw->window, xw->gc, 0, bc->ref.y,
               (int)xw->geom.width-1, bc->ref.y );
    break;
  case 4:  /* Two vertical lines */
    XDrawLine( xw->display, xw->window, xw->gc, bc->end.x, 0,
               bc->end.x, (int)xw->geom.height-1 );
    XDrawLine( xw->display, xw->window, xw->gc, bc->ref.x, 0,
               bc->ref.x, (int)xw->geom.height-1 );
    break;
  case 5: /* One horizontal line through the cursor */
    XDrawLine( xw->display, xw->window, xw->gc, 0, bc->end.y,
               (int)xw->geom.width-1, bc->end.y );
    break;
  case 6: /* One vertical line through the cursor */
    XDrawLine( xw->display, xw->window, xw->gc, bc->end.x, 0,
               bc->end.x, (int)xw->geom.height-1 );
    break;
  case 7: /* Cross hair */
    XDrawLine( xw->display, xw->window, xw->gc, 0, bc->end.y,
               (int)xw->geom.width-1, bc->end.y );
    XDrawLine( xw->display, xw->window, xw->gc, bc->end.x, 0,
               bc->end.x, (int)xw->geom.height-1 );
    break;
  case 8:  /* Draws a rectangle centered on the current pointer */
    {
      if( bc->inside == 1 ) {
        int x1 = bc->end.x - (int)bc->width/2;
        int x2 = bc->end.x + (int)bc->width/2;
        if( x1 < bc->xmin ) {
          bc->ref.x = bc->xmin;
          bc->end.x = bc->ref.x + bc->width;
        } else {
          if( x2 > bc->xmax ) {
            bc->end.x = bc->xmax;
            bc->ref.x = bc->end.x - bc->width;
          } else {
            bc->ref.x = x1;
            bc->end.x = bc->ref.x + bc->width;
          }
        }
        int y1 = bc->end.y - (int)bc->height/2;
        int y2 = bc->end.y + (int)bc->height/2;
        if( y1 < bc->ymin ) {
          bc->ref.y = bc->ymin;
          bc->end.y = bc->ref.y + bc->height;
        } else {
          if( y2 > bc->ymax ) {
            bc->end.y = bc->ymax;
            bc->ref.y = bc->end.y - bc->height;
          } else {
            bc->ref.y = y1;
            bc->end.y = bc->ref.y + bc->height;
          }
        }
      } else {
        bc->ref.x = bc->end.x - (int)bc->width/2;
        bc->end.x = bc->ref.x + bc->width;
        bc->ref.y = bc->end.y - (int)bc->height/2;
        bc->end.y = bc->ref.y + bc->height;
      }
      XDrawRectangle( xw->display, xw->window, xw->gc,
                      bc->ref.x, bc->ref.y, bc->width, bc->height );
    };
    break;
  case 100:  /* Display a custom cursor */
    {
      if( CTRL_KEY_DOWN != CTRL_KEY_DOWN_OLD ) {
        /* uniquement dans ce cas, on doit mettre à jour 'bc', car
         * le curseur a changé */
        if( CTRL_KEY_DOWN == 0 ) {
          /* User cursor #1 */
          xw_set_cursor(xw, XW_USER1_CURSOR);
        } else {
          /* User cursor #2 */
          xw_set_cursor(xw, XW_USER2_CURSOR);
        }
      }
      CTRL_KEY_DOWN_OLD = CTRL_KEY_DOWN;
    };
    break;
  case 101:  /* Display a custom cursor + usage of a magnetic grid */
    {
      int ictrl_key = 1 + CTRL_KEY_DOWN;
      int ix_in, iy_in, ix_out, iy_out;
      int ivalid;

      /* convert (x,y) from X11 to device coords */
      ix_in = bc->end.x - xw->geom.xmin;
      iy_in = xw->geom.ymax - bc->end.y;

      /* calling the Fortran 'magnetic_grid_rule' routine */
#if defined _GNU_GFC
      __mod_pgplot_MOD_magnetic_grid_rule_pgplot( &ictrl_key,
                                     &ix_in, &iy_in, &ix_out, &iy_out,
                                     &X_OUT_WLD_COORD, &Y_OUT_WLD_COORD,
                                     &ivalid );
#elif defined _INTEL_IFC
      mod_pgplot_mp_magnetic_grid_rule_pgplot_( &ictrl_key,
                                     &ix_in, &iy_in, &ix_out, &iy_out,
                                     &X_OUT_WLD_COORD, &Y_OUT_WLD_COORD,
                                     &ivalid );
#else
      "UNKNOWN compiler."
#endif

      if( ivalid == 1 ) {
         READ_CURSOR_VALID = 1;
         xw_set_cursor(xw, XW_CROSSHAIR_CURSOR);
      } else {
          /* Normal cursor (left arrow) */
         READ_CURSOR_VALID = 0;
         xw_set_cursor(xw, XW_LEFT_ARROW_CURSOR);
         break;
      }

      /* convert (x,y) from device to X11 coords  */
      /* added a shift of 8 for each coord because hotspot is taken
       * as (0,0) during XCopyPlane (below) */
      bc->end.x = xw->geom.xmin + ix_out - 8;
      bc->end.y = xw->geom.ymax - iy_out - 8;

      XSetClipOrigin ( xw->display, bc->gc, bc->end.x, bc->end.y );

      if( CTRL_KEY_DOWN != CTRL_KEY_DOWN_OLD ) {
        /* uniquement dans ce cas, on doit mettre à jour 'bc', car
         * le curseur a changé */
        if( CTRL_KEY_DOWN == 0 ) {
          /* User cursor #1 */
          PIX_USER_CURSOR      = PIX_USER1_CURSOR;
          PIX_USER_CURSOR_MASK = PIX_USER1_CURSOR_MASK;
          PIX_USER_COLOR       = PIX_USER1_COLOR;
        } else {
          /* User cursor #2 */
          PIX_USER_CURSOR      = PIX_USER2_CURSOR;
          PIX_USER_CURSOR_MASK = PIX_USER2_CURSOR_MASK;
          PIX_USER_COLOR       = PIX_USER2_COLOR;
        }
        XSetForeground( xw->display, bc->gc, PIX_USER_COLOR.pixel );
        XSetClipMask( xw->display, bc->gc, PIX_USER_CURSOR_MASK );
      }
      CTRL_KEY_DOWN_OLD = CTRL_KEY_DOWN;

      /* XCopyPlane required to copy a depth-1 bitmap (the source) to
       * the window which have the default depth of root -- usually not 1.
       * XCopyArea cannot be used.
       * On the other hand, the new pixmap for simulating the cursor must not
       * be centered, else the update of the transparent cursor restore
       * the original pixmap and erase the cursor image. */
      XCopyPlane( xw->display, PIX_USER_CURSOR, xw->window, bc->gc,
                  0, 0, 16, 16, bc->end.x, bc->end.y, 1 );
    }
    break;
  };
  /*
   * Standard video (after inverse video)
   */
  gr_values.function = GXcopy;
  XChangeGC( xw->display, xw->gc, GCFunction, &gr_values );

  XFlush( xw->display );
  return 0;
}
