/*.......................................................................
 * Restores the pixels under a given line.
 *
 * Input:
 *  xw    XWdev *  The MFPLOT /xw device descriptor.
 *  xa, ya  int    The start pixel of the line.
 *  xb, yb  int    The end pixel of the line.
 * Output:
 *  return  int    0 - OK.
 *                 1 - Error.
 */
static int xw_cursor_line ( XWdev *xw, int xa, int ya, int xb, int yb ) {
  int xlen = xb - xa;  /* X-axis displacement of line */
  int ylen = yb - ya;  /* Y-axis displacement of line */
  int xmin,xmax;       /* Min/max X-axis end points */
  int ymin,ymax;       /* Min/max Y-axis end points */
#define PIXINC 51
  /*
   * Device error?
   */
  if(xw->bad_device) return 1;
  /*
   * Silently ignores the call if a pixmap is not available.
   */
  if(xw->pixmap != None) {
    /*
     * Gets sorted versions of xa and xb.
     */
    if(xlen > 0) {
      xmin = xa;
      xmax = xb;
    } else {
      xmin = xb;
      xmax = xa;
    };
    /*
     * Gets sorted versions of ya and yb.
     */
    if(ylen > 0) {
      ymin = ya;
      ymax = yb;
    } else {
      ymin = yb;
      ymax = ya;
    };
    /*
     * Vertical line?
     */
    if(xlen==0) {
      XCopyArea(xw->display, xw->pixmap, xw->window, xw->gc, xmin, ymin,
                (unsigned) 1, (unsigned) (ymax-ymin+1), xmin, ymin);
    }
    /*
     * Horizontal line?
     */
    else if(ylen==0) {
      XCopyArea(xw->display, xw->pixmap, xw->window, xw->gc, xmin, ymin,
                (unsigned) (xmax-xmin+1), (unsigned) 1, xmin, ymin);
    }
    /*
     * Diagonal line encompasing fewer x-axis lines that y-axis lines?
     */
    else if(abs(xlen) <= abs(ylen)) {
      int x;       /* The X coordinate of the line of pixels being drawn */
      int y1,y2;   /* The start and end Y coordinates of the pixel line */
      double yperx = (double) ylen / (double) xlen;
      double yhalf = 0.5 * yperx;         /* Y-step over half a pixel */
      double ydelt = (PIXINC+0.5) * yperx; /* Y-step over PIXINC+0.5 pixels */
      double ylo = yperx > 0 ? yhalf : -ydelt;
      double yhi = yperx > 0 ? ydelt : -yhalf;
      /*
       * Draws the block of pixels that encompases the line between X-axis
       * pixels the outer edges of pixels x -> x+PIXINC, for each consecutive
       * block of PIXINC pixels along X.
       */
      for(x=xmin; x <= xmax; x += PIXINC+1) {
        double ycent = ya + (x - xa) * yperx;
        y1 = (int)(ycent - ylo);    /* Note round-down semantics */
        y2 = (int)(ycent + yhi+0.5);/* Note round-up semantics */
        XCopyArea(xw->display, xw->pixmap, xw->window, xw->gc,
                  x, y1, (unsigned) (PIXINC+1), (unsigned) (y2-y1+1), x, y1);
      };
      /*
       * Diagonal line encompasing fewer y-axis lines that x-axis lines?
       */
    } else {
      int y;       /* The Y coordinate of the line of pixels being drawn */
      int x1,x2;   /* The start and end X coordinates of the pixel line */
      double xpery = (double) xlen / (double) ylen;
      double xhalf = 0.5 * xpery;         /* X-step over half a pixel */
      double xdelt = (PIXINC+0.5) * xpery; /* X-step over PIXINC+0.5 pixels */
      double xlo = xpery > 0 ? xhalf : -xdelt;
      double xhi = xpery > 0 ? xdelt : -xhalf;
      /*
       * Draws the block of pixels that encompases the line between Y-axis
       * pixels the outer edges of pixels y -> y+PIXINC, for each consecutive
       * block of PIXINC pixels along Y.
       */
      for(y=ymin; y <= ymax; y += PIXINC+1) {
        double xcent = xa + (y - ya) * xpery;
        x1 = (int)(xcent - xlo);    /* Note round-down semantics */
        x2 = (int)(xcent + xhi+0.5);/* Note round-up semantics */
        XCopyArea(xw->display, xw->pixmap, xw->window, xw->gc,
                  x1, y, (unsigned) (x2-x1+1), (unsigned) (PIXINC+1), x1, y);
      };
    };
  };
  /*
   * Checks for device errors.
   */
  if(xw->bad_device) return 1;
  return 0;
}
