/*...........................................................................
 * MFPLOT driver for X Windows (Xlib).
 *
 *---------------------------------------------------------------------------
 *
 *  Scope: This driver should work with all unix workstations running
 *         X Windows (Version 11).
 *  Color: Visual color maps of types, PseudoColor, StaticColor, GrayScale
 *         and StaticGray are supported. Where insufficient colors are
 *         available in the default colormap, a private colormap is
 *         requested. If this fails, the device is treated as
 *         monochrome.
 * Cursor: The cursor is controlled by a mouse or equivalent. Buttons
 *         1 2 3 are mapped to characters A D X. The cursor can also
 *         be moved horizontally and vertically with the arrow keys.
 *         Each press of an arrow key moves the cursor one pixel. This
 *         can be increased to 10 pixels by pressing the shift key.
 *   Size: The initial size and position of the window number #,
 *         is determined with the following hierachy of specifications,
 *         missing details at each level are supplied by those below.
 *
 *          1. X-resource: mfxwin.win#.geometry:   WIDTHxHEIGHT+X+Y
 *          2. X-resource: mfxwin.Win.geometry:    WIDTHxHEIGHT+X+Y
 *          3. Environment variables: MFPLOT_XW_WIDTH, MFPLOT_XW_HEIGHT
 *          4. #define's: XW_DEF_WIDTH, XW_DEF_HEIGHT, XW_DEF_ASPECT
 *
 *---------------------------------------------------------------------------
 * History of Changes: (For previous changes see before 2021-06-28)
 * 28-Jun-2021 - Remove mfxwin_server. Windows creation is managed now
 *               by the present driver.
 *  1-Jul-2021 - Opcodes 54 (Unmap Window), 55 (Map and Raise Window) and
 *               56 (Iconify Window) removed: they were never called
 *               (no Fortran callers in MFPLOT).
 *  2-Jul-2021 - MFPLOT windows are now not resizeable. The only way to
 *               change their size is to call xw_win_resize, via the
 *               opcode 56 (Window Resize).
 *  4-Nov-2021 - Added opcode 72 for the cleaning of FontConfig (to get
 *               rid of a great number of small memory leaks). Called from
 *               the msExitFgl() FGL routine.
 *  9-Nov-2021 - The function 'FcFini' of FontConfig, used to clean the
 *               memory, was buggy! Change for another way, called in the
 *               new 'XftClean' function of Xft (added by myself -- see
 *               opcode 72).
 * 11-Nov-2021 - Modified the XWdev structure to include XftDraw and the
 *               Xft font. Also, the two auxiliary pixmaps are now attached
 *               to each XW device.
 * 12-Nov-2021 - Removed opcode 70, now that the auxiliary pixmaps are
 *               attached to each XW device. Cleaning of many GC of Xlib.
 * 15-Nov-2021 - The initial setting of cap_style and join_style in the
 *               'xw_new_XWdev' function have been removed, because these
 *               two items will be changed by appropriate new opcodes.
 *               New opcode 25 to set these line attributes.
 *  1-Dec-2021 - Implemented hardware dash in opcode 19.
 *               (opcode 4 now returns, on request, that dash is available)
 *               Opcode 13 moved to number 18.
 *               New opcode 13 for drawing a polyline (opcode 12 remains
 *               valid, but only used now to draw a segment of two points).
 *  4-Dec-2021 - Added opcode 70, to get the dash pattern period.
 * 10-Dec-2021 - Added four opcodes (160-163) to manage "outside" legends,
 *               drawn in a small, additional window.
 * 11-Dec-2021 - Added opcode 164 to read cursor without waiting (with,
 *               Optionally, the code of the key pressed).
 * 15-Dec-2021 - Changed all numerical target labels of the select case to
 *               predefined character strings.
 *  9-Jul-2022 - Add two custom cursors, registered by new opcode
 *               DEF_CUSTOM_CURSOR_SHAPE.
 *               Add also a transparent cursor (i.e. no visible)
 * 22-Mar-2023 - Inverted dashes implemented in SET_LINE_STYLE.
 * 12-Apr-2024 - Added a test to call XInitThreads function only if the
 *               environment variable MFPLOT_X11_DEVICE is set to 1.
 *               This is to avoid a small warning from XInitThreads
 *               when checking memory leaks with valgrind (usually with
 *               MFPLOT_X11_DEVICE=0).
 * 20-Jun-2024 - Add a call to XPending before getting the next event
 *               in xw_read_cursor_no_wait().
 *               Indeed, XNextEvent may block the program if there is
 *               no event in the queue.
 *---------------------------------------------------------------------------
 */

#include "x11_driver.h"

/*.......................................................................
 * This is the only external entry point to the /xw device driver.
 * It is called by MFPLOT to open, perform operations on, return
 * information about and close /xw windows.
 *
 * Input:
 *  ifunc   int *  The MFPLOT operation code to be executed.
 * Input/output:
 *  rbuf double *  A general buffer for input/output of double values.
 *  ibuf    int *  A general buffer for input/output of integer values.
 *                 Where relevant this is used to return the number of
 *                 elements in rbuf[]. Also used on input to specify
 *                 number of pixels in the line-of-pixels primitive.
 *  chr    char *  A general buffer for string I/O.
 *  lchr    int *  Where relevant this is used to send and return the
 *                 number of significant characters in chr.
 * Input:
 *  len     int    Added to the call line by the FORTRAN compiler.
 *                 It contains the declared size of chr[].
 */
void X11_DRIVER ( int *ifunc, double *rbuf, int *ibuf, char *chr, int *lchr,
                  int len ) {

  static XWdev *xw = NULL; /* The descriptor of the currently selected device */
  int i;

  int x;
  int y;
  unsigned int width;
  unsigned int height;

  /* Must be called at the beginning, before any other Xlib functions,
   * only one time */
  if ( !InitThreadsCalled ) {
    char *e; int ok;
    e = getenv( "MFPLOT_X11_DEVICE" );
    if( e ) {
//      printf( "in x11_driver: MFPLOT_X11_DEVICE=%s\n", e );
      ok = atoi( e );
    } else {
      ok = 1;
    }
    if( ok == 1 ) {
      InitThreadsCalled = 1;
      XInitThreads();
    }
  }

  /*
   * If there is a buffered opcode and the latest opcode is not the same as
   * the last opcode, call the given flush function for the buffered opcode.
   */
  if(xw && !xw->bad_device) {
    if(xw->last_opcode != *ifunc) {
      if(xw->flush_opcode_fn != (Flush_Opcode_fn) 0) {
        (*xw->flush_opcode_fn)(xw);
        xw->flush_opcode_fn = (Flush_Opcode_fn) 0;
      }
      /*
       * Record the current opcode for next time.
       */
      xw->last_opcode = *ifunc;
    }
  }

  /* Branch on opcode. */

  switch(*ifunc) {

  /*--- Return device name ---------------------------------------*/

  case GET_DEV_NAME:
    {
      char *dev_name;
      dev_name = "XWINDOW (X window window@node:display.screen/xw)";
      strncpy(chr, dev_name, len);
      *lchr = strlen(dev_name);
      for(i = *lchr; i < len; i++)
        chr[i] = ' ';
    }
    break;

  /*--- Return range of color indexes ----------------------------*/

  case GET_COL_IND_RANGE:
    ibuf[0] =  0;
    ibuf[1] = (xw && !xw->bad_device) ? xw->color.ncol-1 : 1;
    break;

  /*--- Return device resolution ---------------------------------*/

  case GET_DEV_RESOL:
    if( xw_ok(xw) ) {
      ibuf[0] = xw->geom.xpix_per_inch;
      ibuf[1] = xw->geom.ypix_per_inch;
    } else {
      ibuf[0] = 1;
      ibuf[1] = 1;
    }
    ibuf[2] = 1;    /* Device coordinates per pixel */
    break;

  /*--- Return misc device info ----------------------------------*/

  case GET_MISC_INFO:
    chr[0] = 'I'; /* Interactive device (I) or Hardcopy (H) */
    chr[1] = 'C'; /* Cursor available (C) or not (N) */
    chr[2] = 'D'; /* Dashed lines available (D) or not (N) */
    chr[3] = 'A'; /* Area fill available (A) or not (N) */
    chr[4] = 'T'; /* Thick lines available (T) or not (N) */
    chr[5] = 'R'; /* Rectangle fill available (R) or not (N) */
    chr[6] = 'P'; /* Pixel device (P), Image device (Q) or nothing (N) */
    chr[7] = 'N'; /* Persistent device (Verbose prompt) or not (N) */
    chr[8] = 'Y'; /* Can return color representation? (Y) or not (N) */
    chr[9] = 'M'; /* Markers available (M) or not (N) */
    *lchr = 10;
    break;

  /*--- Return Default File Name ---------------------------------*/

  case GET_DEF_FILENAME:
    chr[0] = '\0';  /* Default name is "" (i.e. empty) */
    *lchr = 0;
    break;

  /*--- Return Physical Size of Plot -----------------------------*/

  case GET_DEF_SIZE:
    if( xw && !xw->bad_device ) {
      /*
       * Returns various sizes of the X11 window, according to rbuf[0]:
       *    rbuf[0] = 0.  ->  the physical size is returned;
       *              1.  ->  the drawing size is returned.
       *                      (drawing area = whole window minus margins)
       *              2.  ->  the margins are returned
       * Sometimes, we must wait few milliseconds in order to get the right
       * answer; in this case, look at ibuf[0]:
       *    ibuf[0] = 0   ->  no need to wait;
       *              1   ->  wait is mandatory.
       */

      /* We can retrieve width and height reliably via xw->geom,
       * because this structure is (or should) always up-to-date!  */
      if ( rbuf[0] == 1. ) {
        ibuf[0] = 0;
        ibuf[1] = xw->geom.width - 2 * xw->geom.xmargin;
        ibuf[2] = 0;
        ibuf[3] = xw->geom.height - 2 * xw->geom.ymargin;
      } else if ( rbuf[0] == 0. ) {
        ibuf[0] = 0;
        ibuf[1] = xw->geom.width;
        ibuf[2] = 0;
        ibuf[3] = xw->geom.height;
      } else if ( rbuf[0] == 2. ) {
        ibuf[0] = 0;
        ibuf[1] = xw->geom.xmargin;
        ibuf[2] = 0;
        ibuf[3] = xw->geom.ymargin;
      } else {
        fprintf( stderr, " (Muesli FGL:) x11_driver: opcode GET_DEF_SIZE: internal error.\n" );
        fprintf( stderr, "               rbuf value equal to 0, 1 or 2 required!\n" );
        fprintf( stderr, "\n -> pause for debugging purpose...\n" );
        getchar();
#if defined _GNU_GFC
#ifndef _OPTIM
        printf( "  Traceback:\n ");
        _gfortran_backtrace();
#endif
#endif
        fprintf( stderr, " *** ABORTING ***\n" );
        exit(1);
      }
    } else {
      /* Returning the default size */
      ibuf[0] = 0;
      ibuf[1] = XW_DEF_WIDTH;
      ibuf[2] = 0;
      ibuf[3] = XW_DEF_HEIGHT;
    }
    break;

  /*--- Return Character Size ------------------------------------*/

  case GET_CHAR_SIZE:
    rbuf[0] = 1.0;
    break;

  /*--- Select Plot ----------------------------------------------*/

  case SELECT_PLOT:
    xw = xw_select_device(ibuf[0]);
    if( xw->pixmap != None ) {
      pixmap_ID = xw->pixmap;
    }
    break;

  /*--- Open Device ----------------------------------------------*/

  case OPEN_DEV:
    /*
     * Assign the returned device unit number and success indicator.
     * Assume failure to open until the window is open.
     */

    /* the calling routine is always gropen_ec */
    x = ibuf[0];
    y = ibuf[1];
    width  = ibuf[2];
    height = ibuf[3];
    i = ibuf[4]; /* MFPLOT ident */

    ibuf[0] = ibuf[1] = 0;
    /*
     * Create the window.
     */
    xw = xw_new_XWdev ( i, x, y, width, height );
    if( xw==NULL ) return;
//XSynchronize( xw->display, True );
    /*
     * Insert the device in the list of open devices.
     */
    xw_insert_device(xw);
    ibuf[0] = xw->number; /* Number used to select this device */
    ibuf[1] = 1;

    break;

  /*--- Close Device --------------------------------------------*/

  case CLOSE_DEV:
    /*
     * Remove the device from the list of open devices and delete it.
     */
    xw_remove_device(xw);

    if( xw->bad_device ) return;

    /* Getting window size */
    {
      Window root;
      unsigned border, depth;
      XGetGeometry( xw->display, xw->window, &root, &x, &y, &width, &height,
                    &border, &depth );
    }

    /* For the position, don't use XGetWindowAttributes or XGetGeometry,
     * they are not reliable... some users advice to use XTranslateCoordinates
     */
    {
      Window unused;
      XTranslateCoordinates( xw->display, xw->window, DefaultRootWindow(xw->display),
                             0, 0, &x, &y, &unused);
    }

    /* Take care of the WM title height before returning the y position */

    ibuf[0] = x;
    ibuf[1] = y - XW_WM_TITLE_HEIGHT;
    ibuf[2] = width;
    ibuf[3] = height;

    xw = xw_del_XWdev(xw,0);
    break;

  /*--- Begin Picture -------------------------------------------*/

  case BEGIN_PICT:
    /* called by grbpic */
    if( xw_ok(xw) ) {
      /*
       * Convert the passed max X and Y coordinates into the total width of
       * the new window. Add 1/4" margins to the requested area.
       */
      unsigned int width  = ibuf[0] + 2*xw->geom.xmargin;
      unsigned int height = ibuf[1] + 2*xw->geom.ymargin;

      /*
       * Re-size the window if required.
       */
      xw_next_page ( xw, width, height );

    }
    break;

  /*--- Draw one straight segment -------------------------------*/

  case DRAW_LINE_SEGM:
    if( xw_ok(xw) && (pixmap_ID != None) ) {
      XPoint start;
      XPoint end;

      xw_xy_to_XPoint ( xw, &ibuf[0], &start );
      xw_xy_to_XPoint ( xw, &ibuf[2], &end );

      XDrawLine ( xw->display, pixmap_ID, xw->gc,
                  start.x, start.y, end.x, end.y );
      if ( DRAW_IN_AUX == 0 ) {
        xw_mark_modified ( xw, start.x, start.y, xw->gcv.line_width );
        xw_mark_modified ( xw, end.x, end.y, xw->gcv.line_width );
      }
    }
    break;

  /*--- Draw polyline (multiple straight segments) -------------*/

  case DRAW_POLYLINE:
    /* This opcode is necessary to get regular dashing (when using
     * DRAW_LINE_SEGM, the dash offset is reset to 0 between XDrawLine calls).
     * We use the same array of XPoint, xw->poly, as in Polygon Fill */
    if( xw_ok(xw) && (pixmap_ID != None) ) {
      int i, j;
      int xmin = 65536, ymin = 65536, xmax = -1, ymax = -1;

      xw->poly.npoint = ibuf[0];
      xw->poly.points = (XPoint *) malloc(sizeof(XPoint) * xw->poly.npoint);
      if(xw->poly.points == NULL) {
        fprintf(stderr, "%s: Insufficient memory for polyline points.\n", XW_IDENT);
        break;
      }

      for (i=0; i<xw->poly.npoint; i++) {
        j = 2*i;
        XPoint *xp = &xw->poly.points[i];
        xw_xy_to_XPoint(xw, &ibuf[j+1], xp);
        if ( xp->x < xmin ) xmin = xp->x;
        if ( xp->y < ymin ) ymin = xp->y;
        if ( xp->x > xmax ) xmax = xp->x;
        if ( xp->y > ymax ) ymax = xp->y;
      }
      XDrawLines ( xw->display, pixmap_ID, xw->gc, xw->poly.points,
                   xw->poly.npoint, CoordModeOrigin );
      free((char *)xw->poly.points);
      xw->poly.points = NULL;
      if ( DRAW_IN_AUX == 0 ) {
        xw_mark_modified( xw, xmin, ymin, xw->gcv.line_width );
        xw_mark_modified( xw, xmax, ymax, xw->gcv.line_width );
      }
    }
    break;

  /*--- End Picture ---------------------------------------------*/

  case END_PICT:
    break;

  /*--- Select Color Index --------------------------------------*/

  case SELECT_COL_IND:
    if( xw_ok(xw) ) {
      last_color_index = ibuf[0];
      xw_set_ci( xw, last_color_index );
    }
    break;

  /*--- Flush Buffer --------------------------------------------*/

  case FLUSH_BUF:
    if( xw_ok(xw) ) {
      xw_flush(xw);
    }
    break;

  /*--- Read Cursor ---------------------------------------------*/

  case READ_CURSOR:
    if( xw_ok(xw) ) {
      XPoint ref;           /* Reference cursor coordinates */
      XPoint pos;           /* Input/Output cursor coordinates */
      int mode = 0;         /* Cursor band mode */
      int posn = 1;         /* True to position the cursor */
      int shape;            /* Cursor shape */
      int custom_cursor;
      int write_pos;
      double pgxorg, pgyorg, pgxscl, pgyscl;
      int axis_scale_x, axis_scale_y;

      xw_xy_to_XPoint(xw, ibuf, &pos);
      mode = ibuf[4];

      if( mode == 8 ) {
        /* in this mode, IREF and JREF are not coordinates, but length */
        ref.x = ibuf[2];
        ref.y = ibuf[3];
        /* flag -- global variable */
        rect_inside = ibuf[7];
      } else {
        xw_xy_to_XPoint(xw, &ibuf[2], &ref);
      }

      posn = ibuf[5] > 0;
      shape = ibuf[6];
      custom_cursor = ibuf[8];
      write_pos = ibuf[9];
      if( write_pos == 1 ) {
        pgxorg = rbuf[0];
        pgyorg = rbuf[1];
        pgxscl = rbuf[2];
        pgyscl = rbuf[3];
        axis_scale_x = ibuf[10];
        axis_scale_y = ibuf[11];
      } else {
        pgxorg = 0;
        pgyorg = 0;
        pgxscl = 0;
        pgyscl = 0;
        axis_scale_x = 1;
        axis_scale_y = 1;
      }
      if( xw_read_cursor( xw, mode, posn, shape, &ref, &pos, chr,
                          write_pos, pgxorg, pgyorg, pgxscl, pgyscl,
                          axis_scale_x, axis_scale_y ) == 0 ) {
        xw_XPoint_to_xy(xw, &pos, ibuf);
        ibuf[2] = CTRL_KEY_DOWN;
        ibuf[3] = READ_CURSOR_VALID;
      } else {
        *chr = '\0';
      }
    } else {
      *chr = '\0';
    }
    *lchr = 1;
    break;

  /*--- Set X11 window ID of the terminal -----------------------*/

  case SET_X11_WIN_ID:
    MF_TERMINAL_X11_ID = (Window) ibuf[0];
    break;

  /*--- Set Line Style ------------------------------------------*/

  case SET_LINE_STYLE:
    /* (called only from grsls)
     * Below, can be found corrections, in order that n1, n2 and n3 are
     * all greater than 1 (unit is pixel) */
    if( xw_ok(xw) ) {
      int line_style = ibuf[0];
      int dash_offset = 0;

      /* Treat the special case of inverted dashes */
      if( line_style == 5 ) {
        line_style = 2;
        dash_offset = -1;
      }

      switch ( line_style ) {
        case 1: /* continuous (default) */
          {
            xw->gcv.line_style = LineSolid;
            dash_pattern_period = 0;
          }
          break;
        case 2: /* dashed */
          {
            int n1 = (int) dash_length_1*sqrt(current_linewidth);
            if( n1 == 0 ) n1 = 1;
            if( dash_offset == -1 ) dash_offset = n1;
            char dash_list[] = { n1, n1 };
            int list_length = 2;
            XSetDashes( xw->display, xw->gc, dash_offset, dash_list, list_length );
            xw->gcv.line_style = LineOnOffDash;
            dash_pattern_period = n1 + n1;
          }
          break;
        case 3: /* dotted-dashed */
          {
            int n1 = (int) dash_length_1*sqrt(current_linewidth);
            int n2 = (int) dash_length_2*sqrt(current_linewidth);
            int n3 = (int) dash_length_3*sqrt(current_linewidth);
            if( n3 == 0 ) {
               n1 = 10; n2 = 6; n3 = 1;
            }
            char dash_list[] = { n1, n2, n3, n2 };
            int list_length = 4;
            XSetDashes( xw->display, xw->gc, dash_offset, dash_list, list_length );
            xw->gcv.line_style = LineOnOffDash;
            dash_pattern_period = n1 + n2 + n3 + n2;
          }
          break;
        case 4: /* dotted */
          {
            int n2 = (int) dash_length_2*sqrt(current_linewidth);
            int n3 = (int) dash_length_3*sqrt(current_linewidth);
            if( n3 == 0 ) {
               n2 = 6; n3 = 1;
            }
            char dash_list[] = { n3, n2 };
            int list_length = 2;
            XSetDashes( xw->display, xw->gc, dash_offset, dash_list, list_length );
            xw->gcv.line_style = LineOnOffDash;
            dash_pattern_period = n3 + n2;
          }
          break;
      }
      XChangeGC( xw->display, xw->gc, GCLineStyle, &xw->gcv );
    }
    break;

  /*--- Polygon Fill --------------------------------------------*/

  case POLYGON_FILL:
    if( xw_ok(xw) && (pixmap_ID != None) ) {
      int i, j;
      int xmin = 65536, ymin = 65536, xmax = -1, ymax = -1;

      xw->poly.npoint = ibuf[0];
      xw->poly.points = (XPoint *) malloc(sizeof(XPoint) * xw->poly.npoint);
      if(xw->poly.points == NULL) {
        fprintf(stderr, "%s: Insufficient memory for polygon points.\n", XW_IDENT);
        break;
      }

      for (i=0; i<xw->poly.npoint; i++) {
        j = 2*i;
        XPoint *xp = &xw->poly.points[i];
        xw_xy_to_XPoint(xw, &ibuf[j+1], xp);
        if ( xp->x < xmin ) xmin = xp->x;
        if ( xp->y < ymin ) ymin = xp->y;
        if ( xp->x > xmax ) xmax = xp->x;
        if ( xp->y > ymax ) ymax = xp->y;
      }
      XFillPolygon ( xw->display, pixmap_ID, xw->gc, xw->poly.points,
                     xw->poly.npoint, Complex, CoordModeOrigin );
      free((char *)xw->poly.points);
      xw->poly.points = NULL;
      if ( DRAW_IN_AUX == 0 ) {
        xw_mark_modified(xw, xmin, ymin, 1);
        xw_mark_modified(xw, xmax, ymax, 1);
      }
    }
    break;

  /*--- Set Color Representation --------------------------------*/

  case SET_COL_REPRES:
    if( xw_ok(xw) ) {
      if(!xw->color.initialized) xw_init_colors(xw);
      int ci = ibuf[0];
      xw_set_rgb(xw, ci, rbuf[0],rbuf[1],rbuf[2]);
    }
    break;

  /*--- Set Line Width ------------------------------------------*/

  case SET_LINE_WIDTH:
    /*
     * The line width is provided in multiples of 0.005 inches.
     */
    if( xw_ok(xw) ) {
      current_linewidth = rbuf[0]*0.005 * xw->geom.xpix_per_inch;
      xw->gcv.line_width = (int) current_linewidth;
      XChangeGC( xw->display, xw->gc, GCLineWidth, &xw->gcv );
    }
    break;

  /*--- Escape --------------------------------------------------*/

  case ESCAPE:
    /* Not implemented: ignored */
    break;

  /*--- Rectangle Fill ------------------------------------------*/

  case RECT_FILL:
    if( xw_ok(xw) && (pixmap_ID != None) ) {
      XPoint blc;
      XPoint trc;

      xw_xy_to_XPoint(xw, &ibuf[0], &blc);
      xw_xy_to_XPoint(xw, &ibuf[2], &trc);

      XFillRectangle ( xw->display, pixmap_ID, xw->gc, blc.x, trc.y,
                       (unsigned)(trc.x-blc.x+1), (unsigned)(blc.y-trc.y+1) );
      if ( DRAW_IN_AUX == 0 ) {
        xw_mark_modified(xw, blc.x, blc.y, 1);
        xw_mark_modified(xw, trc.x, trc.y, 1);
      }
    }
    break;

  /*--- Draw Rectangle ------------------------------------------*/

  case RECT_DRAW:
    if( xw_ok(xw) && (pixmap_ID != None) ) {
      XPoint blc;
      XPoint trc;

      xw_xy_to_XPoint(xw, &ibuf[0], &blc);
      xw_xy_to_XPoint(xw, &ibuf[2], &trc);

      XDrawRectangle ( xw->display, pixmap_ID, xw->gc, blc.x, trc.y,
                       (unsigned)(trc.x-blc.x), (unsigned)(blc.y-trc.y) );
      if ( DRAW_IN_AUX == 0 ) {
        xw_mark_modified(xw, blc.x, blc.y, 1);
        xw_mark_modified(xw, trc.x, trc.y, 1);
      }
    }
    break;

  /*--- Set Line Cap and Join Style -----------------------------*/

  case SET_LINE_CAP_JOIN_STYLE:
    /*
     * ibuf[0] -> Cap style  - 0: CapButt;
     *                         1: CapRound;
     *                         2: CapProjecting.
     * ibuf[1] -> Join style - 0: JoinMiter;
     *                         1: JoinRound;
     *                         2: JoinBevel.
     */
    if( xw_ok(xw) ) {
//printf("X11 driver: line 675: Cap style, ibuf[0] = %i\n",ibuf[0]);
      switch ( ibuf[0] ) {
        case 0:
//printf("X11 driver: line 677: set cap_style to CapButt\n");
          xw->gcv.cap_style = CapButt;
          break;
        case 1:
//printf("X11 driver: line 681: set cap_style to CapRound\n");
//getchar();
          xw->gcv.cap_style = CapRound;
          break;
        case 2:
          xw->gcv.cap_style = CapProjecting;
          break;
      }
      switch ( ibuf[1] ) {
        case 0:
          xw->gcv.join_style = JoinMiter;
          break;
        case 1:
          xw->gcv.join_style = JoinRound;
          break;
        case 2:
          xw->gcv.join_style = JoinBevel;
          break;
      }

      XChangeGC( xw->display, xw->gc, GCCapStyle | GCJoinStyle, &xw->gcv );
    }
    break;

  /*--- Put Line of Pixels (via color index) --------------------*/

  case PUT_LINE_PIX_CI:
    if( xw_ok(xw) ) {
      XPoint start;
      xw_xy_to_XPoint(xw, &ibuf[1], &start);
      xw_image_line(xw, &start, &ibuf[3], ibuf[0]);
    }
    break;

  /*--- Get Line of Pixels (via RGB val.) -----------------------*/

  case GET_LINE_PIX_RGB:
    if( xw_ok(xw) ) {
      XPoint start;
      xw_xy_to_XPoint(xw, &ibuf[1], &start);
      xw_get_image_line_RGB(xw, &start, &ibuf[3], ibuf[0]);
    }
    break;

  /*--- Put Line of Pixels (via RGB val.) -----------------------*/

  case PUT_LINE_PIX_RGB:
    if( xw_ok(xw) ) {
      XPoint start;
      xw_xy_to_XPoint(xw, &ibuf[1], &start);
      xw_image_line_RGB(xw, &start, &ibuf[3], ibuf[0]);
    }
    break;

  /*--- Query Color Representation ------------------------------*/

  case GET_COL_REPRES:
    if( xw_ok(xw) ) {
      int ci = ibuf[0];
      if(!xw->color.initialized) xw_init_colors(xw);
      rbuf[0] = xw_xcolor_to_rgb(xw->color.xcolor[ci].red);
      rbuf[1] = xw_xcolor_to_rgb(xw->color.xcolor[ci].green);
      rbuf[2] = xw_xcolor_to_rgb(xw->color.xcolor[ci].blue);
      ibuf[0] = 3;
    } else {
      ibuf[0] = 0;
    }
    break;

  /*--- Set Angle and Clipping for Xft --------------------------*/

  case SET_ANG_CLIP_XFT:
    /*
     * Set angle and clipping flag on a rectangular zone,
     * for next write of a string via Xft and Freetype.
     * WARNING: BBox retrieved here (TLC,BRC) must be really that of the
     *          viewport, as with opcode 84, which set the clipping flag.
     */
    if( xw_ok(xw) ) {
      /* clockwise angle from horizontal, in radian */
      xft_angle = rbuf[0];
      xft_clipping = ibuf[0];
      if( xft_clipping == 1 ) {
        /* read the clipping bbox */
        xw_xy_to_XPoint(xw, &ibuf[1], &TLC);
        xw_xy_to_XPoint(xw, &ibuf[3], &BRC);
      }
    }
    break;

  /*--- Set Xft Font --------------------------------------------*/

  case SET_FONT_XFT:
    /*
     * Set fontname and attributes (fontsize and fontstyle), for writing
     * the string (via Xft).
     * (The fontname contains itself the attributes)
     */
    if( xw_ok(xw) ) {
      strcpy(xft_fontname, chr);
      xft_string_utf8 = ibuf[0];
    }
    break;

  /*--- Draw Character String -----------------------------------*/

  case DRAW_CHAR_STRING:
    /*
     * Write chr, a string of characters, using the antialiased feature of
     * Xft and Freetype. It is written at the position xy, using the
     * previously retrieved xft_fontname and xft_angle. It checks also
     * if clipping at viewport is needed, via xft_clipping.
     */
    if( xw_ok(xw) ) {
      XPoint xy;
      xw_xy_to_XPoint(xw, ibuf, &xy);
      xw_draw_string( xw, chr, &xy );
    }
    break;

  /*--- Set the Muesli Installation Path ------------------------*/

  case SET_MUESLI_PATH:
    strcpy(MFPLOT_DIR, chr);
    break;

  /*--- Get the BBox of a String via Xft ------------------------*/

  case GET_XFT_STRING_BBOX:
    if( xw_ok(xw) ) {
      int ex, ey, ew, eh;
      xw_get_string_bbox ( xw, chr, &ex, &ey, &ew, &eh );
      ibuf[0] = ex;
      ibuf[1] = ey;
      ibuf[2] = ex;
      ibuf[3] = ey;
    }
    break;

  /*--- Draw Marker ---------------------------------------------*/

  case DRAW_MARKER:
    /*
     * Write one marker at a time, using the antialiased feature of Xft
     * and Freetype. It is written at the position xy.
     */
    if( xw_ok(xw) ) {
      XPoint xy;
      xw_xy_to_XPoint ( xw, ibuf, &xy );
      xw_draw_marker ( xw, chr, &xy );
    }
    break;

  /*--- Get Cursor Position (without event) ---------------------*/

  case GET_CURSOR_POS:
    if( xw_ok(xw) ) {
      XPoint pos;      /* Output cursor coordinates */
      if( xw_current_pos(xw, &pos)==1 ) {
         xw_XPoint_to_xy ( xw, &pos, ibuf );
         ibuf[2] =  1;
      } else {
         ibuf[2] = -1;
      }
    }
    break;

  /*--- Get a Click in the Window -------------------------------*/

  case GET_CLICK_IN_WIN:
    if( xw_ok(xw) ) {
      if( xw_get_click(xw)==1 ) {
         ibuf[0] =  1;
      } else {
         ibuf[0] = -1;
      }
    }
    break;

  /*--- Raise Window --------------------------------------------*/

  case RAISE_WIN:
    if( xw_ok(xw) ) {
       xw_raise_win(xw);
    }
    break;

  /*--- Window Resize -------------------------------------------*/

  case WIN_RESIZE:
    if( xw_ok(xw) ) {
      xw_win_resize( xw, ibuf[0], ibuf[1], ibuf[2] );
    }
    break;

  /*--- Select Back- & Foreground Colors ------------------------*/

  case SELECT_BACK_FOREGROUND:
    if( ibuf[0] == 0 ) {
    /* black_on_white = 0 */
      BLACK_ON_WHITE = 0;
    } else {
    /* default : black on white */
      BLACK_ON_WHITE = 1;
    }
    break;

  /*--- Get the Color Depth -------------------------------------*/

  case GET_COL_DEPTH:
    if( xw_ok(xw) ) {
       ibuf[0] = xw->color.vi->depth;
    }
    break;

  /*--- Set the Executable Name ---------------------------------*/

  case SET_EXE_NAME:
    MF_WIN_ID = ibuf[0];
    if (ibuf[1] == 0) {
      TITLE_DESCRIPTION[0] = '\0';
    } else {
      int n = ibuf[1];
      strncpy(TITLE_DESCRIPTION,&chr[*lchr],n);
      TITLE_DESCRIPTION[n] = '\0';
    }
    if (*lchr == 0) {
      EXECUTABLE_NAME[0] = '\0';
    } else {
      strncpy(EXECUTABLE_NAME,chr,*lchr);
      EXECUTABLE_NAME[*lchr] = '\0';
    }
    break;

  /*--- Get a Color Correction ----------------------------------*/

  case GET_COLOR_CORR:
    /*
     * Given a user RGB-color, (rbuf[0],rbuf[1],rbuf[2]), returns in the
     * same arguments the XColor, as it is actually stored by the X11 lib;
     * this correction depends on the color depth.
     */
    if( xw_ok(xw) ) {
      if(!xw->color.initialized) xw_init_colors(xw);
      /* choose as working color index the greatest one */
      int ci = xw->color.ncol - 1;
      xw_set_rgb(xw, ci, rbuf[0],rbuf[1],rbuf[2]);
      /* very important, flushes X11 color update */
      xw_update_colors(xw);
      /* then query the color */
      rbuf[0] = xw_xcolor_to_rgb(xw->color.xcolor[ci].red);
      rbuf[1] = xw_xcolor_to_rgb(xw->color.xcolor[ci].green);
      rbuf[2] = xw_xcolor_to_rgb(xw->color.xcolor[ci].blue);
    }
    break;

  /*--- Finish a Read Cursor ------------------------------------*/

  case FINISH_READ_CURSOR:
    /*
     * Change the cursor to its normal shape (left arrow) because
     * 'xw_end_cursor' no longer calls 'xw_set_cursor'
     */
    if( xw_ok(xw) ) {
      xw_set_cursor( xw, XW_LEFT_ARROW_CURSOR );
    }
    break;

  /*--- Change the Pointer Shape --------------------------------*/

  case SET_CURSOR_SHAPE:
    /*
     * 0: normal (left arrow)  1: small crosshair      2: resize
     * 3: watch                4: hand1 (opened hand)  5: hand2 (closed hand)
     * 6: zoom                 7: user1                8: user2
     *   (numbers are defined in the 'xw_set_cursor' routine)
     */
    if( xw_ok(xw) ) {
      int shape;            /* Cursor shape */
      shape = ibuf[0];
      xw_set_cursor( xw, shape );
    }
    break;

  /*--- Read Cursor with Dynamic Change of the Cursor -----------*/

  case READ_CURSOR_DYN:
    if( xw_ok(xw) ) {
      XPoint pos;                  /* Output cursor coordinates */
      int imin, imax, jmin, jmax;  /* Device coord of the BBOX */
      int shape;                   /* Cursor shape */
      int write_pos;
      double pgxorg, pgyorg, pgxscl, pgyscl;
      int axis_scale_x, axis_scale_y;

      imin  = ibuf[0];
      imax  = ibuf[1];
      jmin  = ibuf[2];
      jmax  = ibuf[3];
      shape = ibuf[4];
      write_pos = ibuf[9];
      if( write_pos == 1 ) {
        pgxorg = rbuf[0];
        pgyorg = rbuf[1];
        pgxscl = rbuf[2];
        pgyscl = rbuf[3];
        axis_scale_x = ibuf[10];
        axis_scale_y = ibuf[11];
      } else {
        pgxorg = 0;
        pgyorg = 0;
        pgxscl = 0;
        pgyscl = 0;
        axis_scale_x = 1;
        axis_scale_y = 1;
      }
      if( xw_read_cursor_dyn( xw, imin,imax,jmin,jmax,
                              0,0,0,0, 0,
                              shape, &pos, chr,
                              write_pos, pgxorg, pgyorg, pgxscl, pgyscl,
                              axis_scale_x, axis_scale_y )==0) {
        xw_XPoint_to_xy(xw, &pos, ibuf);
      } else {
        *chr = '\0';
      }
    } else {
      *chr = '\0';
    }
    *lchr = 1;
    break;

  /*--- Return Window Size in Pixels ----------------------------*/

  case GET_WIN_SIZE_PIX:
    if( xw && !xw->bad_device ) {  /* Return the size of the current window */
      XWindowAttributes attr;
      XGetWindowAttributes(xw->display, xw->window, &attr);
      if(!xw->bad_device) {
        ibuf[0] = 0;
        ibuf[1] = (double) (attr.width);
        ibuf[2] = 0;
        ibuf[3] = (double) (attr.height);
      } else {
        ibuf[0] = 0;
        ibuf[1] = (double) xw->geom.width;
        ibuf[2] = 0;
        ibuf[3] = (double) xw->geom.height;
      }
    } else {
      ibuf[0] = 0;
      ibuf[1] = XW_DEF_WIDTH;
      ibuf[2] = 0;
      ibuf[3] = XW_DEF_HEIGHT;
    }
    break;

  /*--- Read Cursor with Dynamic Change of the Cursor.
   *    Variant with an Exclusion BBox --------------------------*/

  case READ_CURSOR_DYN_EXCL_BOX:
    if( xw_ok(xw) ) {
      XPoint pos;              /* Output cursor coordinates */
      int imin,imax,jmin,jmax; /* Device coord of the BBOX */
      int not_imin,not_imax,not_jmin,not_jmax; /* Device coord of the excluded BBOX */
      int shape;               /* Cursor shape */
      int write_pos;
      double pgxorg, pgyorg, pgxscl, pgyscl;
      int axis_scale_x, axis_scale_y;

      imin     = ibuf[0];
      imax     = ibuf[1];
      jmin     = ibuf[2];
      jmax     = ibuf[3];
      not_imin = ibuf[4];
      not_imax = ibuf[5];
      not_jmin = ibuf[6];
      not_jmax = ibuf[7];
      shape    = ibuf[8];
      write_pos = ibuf[9];
      if( write_pos == 1 ) {
        pgxorg = rbuf[0];
        pgyorg = rbuf[1];
        pgxscl = rbuf[2];
        pgyscl = rbuf[3];
        axis_scale_x = ibuf[10];
        axis_scale_y = ibuf[11];
      } else {
        pgxorg = 0;
        pgyorg = 0;
        pgxscl = 0;
        pgyscl = 0;
        axis_scale_x = 1;
        axis_scale_y = 1;
      }
      if( xw_read_cursor_dyn( xw, imin,imax,jmin,jmax,
                              not_imin,not_imax,not_jmin,not_jmax, 1,
                              shape, &pos, chr,
                              write_pos, pgxorg, pgyorg, pgxscl, pgyscl,
                              axis_scale_x, axis_scale_y )==0 ) {
        xw_XPoint_to_xy(xw, &pos, ibuf);
      } else {
        *chr = '\0';
      }
    } else {
      *chr = '\0';
    }
    *lchr = 1;
    break;

  /*--- Read Cursor with Dynamic Change of the Cursor.
   *    Variant with a Multispot Zone ---------------------------*/

  case READ_CURSOR_DYN_MULTISPOT:
    if( xw_ok(xw) ) {
      XPoint pos;              /* Output cursor coordinates */
      int shape;               /* Cursor shape */
      int ispot[5], jspot[5];
      int k, selectedSpot, nbspot = (ibuf[0]-1)/2;
      for( k=0; k<nbspot; k++ ) {
        ispot[k] = ibuf[2*k+1];
        jspot[k] = ibuf[2*k+2];
      }
      shape = ibuf[2*nbspot+1];
      if( xw_read_cursor_dyn_multispot( xw, ispot,jspot,nbspot, shape,
                                        &selectedSpot, &pos, chr )==0 ) {
        xw_XPoint_to_xy(xw, &pos, ibuf);
        ibuf[2] = selectedSpot;
      } else {
        *chr = '\0';
      }
    } else {
      *chr = '\0';
    }
    *lchr = 1;
    break;

  /*--- Read Cursor with Dynamic Change of the Cursor.
   *    Variant for a Quadrilateral Polygon ---------------------*/

  case READ_CURSOR_DYN_QUADR_POLYG:
    if( xw_ok(xw) ) {
      XPoint pos;             /* Output cursor coordinates */
      int ibox[4], jbox[4];   /* Device coords of the quadr. polygon */
      int shape;              /* Cursor shape */
      ibox[0] = ibuf[0];
      ibox[1] = ibuf[1];
      ibox[2] = ibuf[2];
      ibox[3] = ibuf[3];
      jbox[0] = ibuf[4];
      jbox[1] = ibuf[5];
      jbox[2] = ibuf[6];
      jbox[3] = ibuf[7];
      shape = ibuf[8];
      if( xw_read_cursor_dyn_quad( xw, ibox,jbox,
                                   shape, &pos, chr )==0) {
        xw_XPoint_to_xy(xw, &pos, ibuf);
      } else {
        *chr = '\0';
      }
    } else {
      *chr = '\0';
    }
    *lchr = 1;
    break;

  /*--- Get the Dash Pattern Period  ----------------------------*/

  case GET_DASH_PERIOD:
    ibuf[0] = dash_pattern_period;
    break;

  /*--- Cleaning of FontConfig  ---------------------------------*/

  case CLEAN_FONTCONFIG:
    /*
     * Found this tip in:
     * https://stackoverflow.com/questions/37251165/pangocairo-hello-world-leaks
     */
//    FcFini();         bug of FontConfig -> SIGABRT
    XftClean(); // new entry in libXft (added by É.C.)
    break;

  /*--- Select and Set Auxiliary Pixmap  ----------------------- */

  case SELECT_AUX_PIXMAP:
    /*
     * For drawing, switches between:
     *   ibuf[0] = 0: the main pixmap
     *   ibuf[0] = 1: the first auxiliary one
     *                 ibuf[1] = 0: default size = that of main pixmap
     *                 ibuf[1] = 1: size given in ibuf[2], ibuf[3]
     *   ibuf[0] = 2: the second one (default size)
     */
    DRAW_IN_AUX = ibuf[0];
    /*
     * Begins a drawing in the auxiliary pixmap
     *   -> checks that pixmap_aux is allocated and has the appropriate size.
     *   -> clears pixmap_aux
     */
    if( xw_ok(xw) ) {
      XFlush ( xw->display );
      unsigned int width, height;

      switch ( DRAW_IN_AUX ) {
        case 0:
          pixmap_ID = xw->pixmap;
          break;
        case 1:
          if( ibuf[1] == 1 ) {
            width  = ibuf[2];
            height = ibuf[3];
          } else {
            width  = 0; /* this means default size */
            height = 0;
          }
          xw_checks_pixmap_aux ( xw, &(xw->pixmap_aux_1), width, height, 1 );
          xw_clear_pixmap_aux ( xw, xw->pixmap_aux_1 );
          pixmap_ID = xw->pixmap_aux_1;
          break;
        case 2:
          xw_checks_pixmap_aux ( xw, &(xw->pixmap_aux_2), 0, 0, 2 );
          xw_clear_pixmap_aux ( xw, xw->pixmap_aux_2 );
          pixmap_ID = xw->pixmap_aux_2;
          break;
        default:
          printf("X11 driver: ERROR for opcode SELECT_AUX_PIXMAP (Select and Set Auxiliary Pixmap)\n");
          printf("            -> Argument must be 0, 1 or 2!\n");
#if defined _GNU_GFC
#ifndef _OPTIM
          printf( "  Traceback:\n ");
          _gfortran_backtrace();
#endif
#endif
          break;
      }
    } else {
      printf("X11 driver: ERROR for opcode SELECT_AUX_PIXMAP (Select and Set Auxiliary Pixmap)\n");
      printf("            -> Cannot access to an opened plotting window!\n");
      printf("            (forget to call msFigure?)\n");
#if defined _GNU_GFC
#ifndef _OPTIM
      printf( "  Traceback:\n ");
      _gfortran_backtrace();
#endif
#endif
    }
    break;

  /*--- Move Grobj from an Auxiliary Pixmap  ------------------- */

  case MOVE_GROBJ_PIXMAP:
    /*
     * The moving Grobj is always stored in pixmap_aux_1.
     * Optionally, the drawing stored in pixmap_aux_2 will overwrite the
     * drawing in pixmap_aux_1.
     * It returns new position in ibuf[0:1].
     */
    if( xw_ok(xw) ) {
      int ret_x, ret_y;
      /* Taking care of the clipping for the selected object */
      CLIPPED_OBJ = ibuf[0];
      xw_move_grobj ( xw, &ret_x, &ret_y );
      ibuf[0] = ret_x;
      ibuf[1] = ret_y;
    }
    break;

  /*--- Reduction of the 2nd auxiliary pixmap  ----------------- */

  case REDUCE_2ND_AUX_PIXMAP:
    if( xw->pixmap_aux_2 != None ) {
      Window root;
      int xmin2, ymin2, width2, height2;
      int xmax2, ymax2, empty2;
      Pixmap pixmap_tmp;
      /* Finds the bbox of the foreground part */
      xw_bbox_of_pixmap ( xw, xw->pixmap_aux_2,
                          &xmin2, &ymin2, &xmax2, &ymax2, &empty2 );
      width2  = xmax2 - xmin2 + 1;
      height2 = ymax2 - ymin2 + 1;
      /* Copies it in a smaller pixmap TODO: always? */
      root = RootWindow ( xw->display, DefaultScreen(xw->display) );
      pixmap_tmp = XCreatePixmap ( xw->display, root, width2, height2,
                                   (unsigned) xw->color.vi->depth );
      XCopyArea ( xw->display, xw->pixmap_aux_2, pixmap_tmp, xw->gc2,
                  xmin2, ymin2, width2, height2, 0, 0 );
      XFreePixmap( xw->display, xw->pixmap_aux_2 );
      /* Copies just the drawable ID (avoid a true copy) */
      xw->pixmap_aux_2 = pixmap_tmp;
      /* If an XftDraw surface was attached to this aux pixmap, update it */
      if( xw->xftdraw_pixmap_aux_2 ) {
        XftDrawChange ( xw->xftdraw_pixmap_aux_2, xw->pixmap_aux_2 );
      }
      /* Return position and size of this subpart in the main pixmap */
      ibuf[0] = xmin2;
      ibuf[1] = ymin2;
      ibuf[2] = width2;
      ibuf[3] = height2;
    }
    break;

  /*--- Set Clipping  ------------------------------------------ */

  case SET_CLIPPING:
    /*
     * ibuf[0] = 0:   no clipping
     * ibuf[0] = 1:   standard clipping (at viewport)
     * ibuf[0] = 2:   clipping at a rectangle defined by ibuf[1:4]
     */
    if( xw_ok(xw) ) {
      int clipping_requested = ibuf[0];

      if ( clipping_requested == 0 ) {

        if ( CLIPPING == 0 ) break;
        XSetClipMask ( xw->display, xw->gc, None );
        CLIPPING = clipping_requested;

      } else if ( clipping_requested == 1 ) {

        if ( CLIPPING == 1 ) break;

        xw_clip_at_viewport ( xw );
        CLIPPING = clipping_requested;

      } else if ( clipping_requested == 2 ) {

        if ( CLIPPING == 2 ) break;

        int TL_xy[2], BR_xy[2];
        XRectangle rectangles[1]; /* Array of rectangles (here, only one) */

        TL_xy[0] = ibuf[1];
        TL_xy[1] = ibuf[4];
        BR_xy[0] = ibuf[2];
        BR_xy[1] = ibuf[3];
        /* read the clipping bbox (take care that Y-axis is downward in X11) */
        xw_xy_to_XPoint(xw, &TL_xy[0], &TLC);
        xw_xy_to_XPoint(xw, &BR_xy[0], &BRC);

        /* update the graphics context */
        rectangles[0].x = 0;
        rectangles[0].y = 0;
        rectangles[0].width  = BRC.x - TLC.x + 1;
        rectangles[0].height = BRC.y - TLC.y + 1;
        XSetClipRectangles ( xw->display, xw->gc,
                             TLC.x, TLC.y, rectangles, 1, Unsorted );
        CLIPPING = clipping_requested;

      }

    }
    break;

  /*--- Set a 1-pixel correction for the viewport  ------------- */

  case SET_1_PIX_CORR:
    /*
     * Usually, the viewport clipping BBox is reduced by 1 pixel around all
     * sides, in order to avoid some colored pixel overwriting the framebox.
     * In exceptional case (X11 dump), this clipping BBox should remains
     * unchanged. The variable CLIP_VP_CORR contains this correction:
     * it must be 0 or 1. Default is one.
     */
    {
      CLIP_VP_CORR = ibuf[0];
    }
    break;

  /*--- Scroll in pixmap  -------------------------------------- */

  case SCROLL_IN_PIXMAP:
    /*
     * Actually move the auxiliary pixmap over the main one, with clipping
     * at the viewport, as usual.
     */
    {
      int Dx = ibuf[0], Dy = ibuf[1];
      int VP_with = ibuf[2], VP_height = ibuf[3];
      int pos_xy[2], ret_dx, ret_dy;
      int xmin2 = ibuf[6], ymin2 = ibuf[7], width2 = ibuf[8], height2 = ibuf[9];
      int axis_on = ibuf[10];

      /* Compute the TLC of the viewport */
      /* (shift position to take into account the PGPLOT margin, also
       *  invert y-axis to follow the X11 convention) */
      pos_xy[0] = xw->geom.xmin + ibuf[4];
      pos_xy[1] = xw->geom.ymax - ibuf[5];

      xw_scroll_rect ( xw, Dx, Dy, VP_with, VP_height, &pos_xy[0],
                       &ret_dx, &ret_dy,
                       xmin2, ymin2, width2, height2, axis_on );

      /* returning just the shift in X and Y */
      ibuf[4] = ret_dx;
      ibuf[5] = ret_dy;

    }
    break;

  /*--- Prepare arrow moving ---------------------------------- */

  case PREP_ARROW_MOVE:
    /*
     * Get arrow geometry and attributes
     *   ibuf[0] : x_start (dev. coords)  ->   ARROW_START.x  (X11 coords)
     *   ibuf[1] : y_start   "     "           ARROW_START.y    "    "
     *   ibuf[2] : x_end     "     "           ARROW_END.x      "    "
     *   ibuf[3] : y_end     "     "           ARROW_END.y      "    "
     *   ibuf[4] : color index
     *   rbuf[0] : line width
     *   rbuf[1] : length of arrowhead (dh in pgarro, but in dev. coords)
     */
    if( xw_ok(xw) ) {
      xw_xy_to_XPoint(xw, &ibuf[0], &ARROW_START);
      xw_xy_to_XPoint(xw, &ibuf[2], &ARROW_END);
      ARROW_CI = ibuf[4];
      ARROW_LW = rbuf[0];
      ARROW_DH = rbuf[1];
    }
    break;

  /*--- Move arrow -------------------------------------------- */

  case MOVE_ARROW:
    /*
     * Move an arrow whose geometric characteristics have been recorded
     * previously by opcode 102.
     * Returns the new geometry of the arrow:
     *   ibuf[0] : x_start (dev. coords)
     *   ibuf[1] : y_start   "     "
     *   ibuf[2] : x_end     "     "
     *   ibuf[3] : y_end     "     "
     */
    if( xw_ok(xw) ) {

      CLIPPED_OBJ = ibuf[0];

      xw_move_arrow ( xw );

      xw_XPoint_to_xy ( xw, &ARROW_START, &ibuf[0] );
      xw_XPoint_to_xy ( xw, &ARROW_END,   &ibuf[2] );

    }
    break;

  /*--- X11 flush policy -------------------------------------- */

  case X11_FLUSH_POLICY:
    /*
     * ibuf[0] = 1:   Enable the update of the screen by 'xw_flush' (Default).
     * ibuf[0] = 0:   Disable the update of the screen by 'xw_flush'.
     */
    if( xw_ok(xw) ) {
      FLUSH_ENABLED = ibuf[0];
    }
    break;

  /*--- Force flush buffer ------------------------------------ */

  case FORCE_FLUSH_BUFFER:
    if( xw_ok(xw) ) {
      FORCE_FLUSH = 1;
      xw_flush(xw);
      FORCE_FLUSH = 0;
    }
    break;

  /*--- Force a full update from the pixmap ------------------- */
// test to avoid sometimes a blank window after resizing (from msResizeWindow)
// => but, this doesn't prevent the bug.

  case FORCE_UPDATE_FROM_PIXMAP:
    if( xw_ok(xw) ) {

      xw->update.modified = 1;
      xw->update.xmin = 0;
      xw->update.ymin = 0;
      xw->update.xmax = xw->geom.width - 1;
      xw->update.ymax = xw->geom.height - 1;

      FORCE_FLUSH = 1;
      xw_flush(xw);
      FORCE_FLUSH = 0;

    }
    break;

  /*--- Set Input Focus --------------------------------------- */

  case SET_INPUT_FOCUS:
    /*
     * ibuf[0] = 0:   Set the focus to the terminal
     * ibuf[0] = 1:   Set the focus to the current mfplot window
     */
    if( xw_ok(xw) ) {
      int focus = ibuf[0];
      xw_set_focus_win ( xw, focus );
    }
    break;

  /*--- Set Viewport ------------------------------------------ */

  case SET_VIEWPORT:
    if( xw_ok(xw) ) {
        int TL_xy[2], BR_xy[2];

        TL_xy[0] = ibuf[0];
        TL_xy[1] = ibuf[3];
        BR_xy[0] = ibuf[1];
        BR_xy[1] = ibuf[2];
        /* read the clipping bbox (take care that Y-axis is downward in X11) */
        xw_xy_to_XPoint ( xw, &TL_xy[0], &VP_TLC );
        xw_xy_to_XPoint ( xw, &BR_xy[0], &VP_BRC );
        viewport_defined = 1;
    }
    break;

  /*--- Update the main pixmap from the auxiliary one  -------- */

  case UPDATE_MAIN_PIXMAP:
    /*
     * Update also the axes labelling and, if needed, the legend frame
     */
    if( xw_ok(xw) ) {
      int x = ibuf[0], y = ibuf[1];
      unsigned int width = ibuf[2], height = ibuf[3];
      int xmin2 = ibuf[4], ymin2 = ibuf[5], width2 = ibuf[6], height2 = ibuf[7];

      width  = width  + 1; /* Fix as in lines 22-23 of xw_scroll_rect */
      height = height + 1;

      XCopyArea ( xw->display, xw->pixmap_aux_1, xw->pixmap, xw->gc,
                  x, y, width, height, VP_TLC.x, VP_TLC.y );
      xw_mark_modified ( xw, VP_TLC.x, VP_TLC.y, 1 );
      xw_mark_modified ( xw, VP_TLC.x+width, VP_TLC.y+height, 1 );

      /* If needed, copies also the foreground portion */
      if( xw->pixmap_aux_2 != None ) {
        XCopyArea ( xw->display, xw->pixmap_aux_2, xw->pixmap, xw->gc2,
                    0, 0, width2, height2, xmin2, ymin2 );
        xw_mark_modified ( xw, xmin2, ymin2, 1 );
        xw_mark_modified ( xw, xmin2+width2, ymin2+height2, 1 );
      }

      xw_flush ( xw );
    }
    break;

  /*--- Get the labelling area around the viewport ------------ */

  case GET_LABEL_AREA_AROUND:
    /*
     * It concerns the area for the ticks and their numerical labels.
     */
    AXIS_LABEL_HEIGHT = ibuf[0];
    break;

  /*--- Erase the labelling area around the viewport ---------- */

  case ERASE_LABEL_AREA_AROUND:
    /*
     * It concerns the area for the ticks and their numerical labels.
     *   ibuf[0] = 0: usual numeric values; 1: time values for X-axis
     *   ibuf[1] = 0: usual numeric values; 1: time values for Y-axis
     */
    if( xw_ok(xw) ) {
      xw_erase_labelling_area ( xw, ibuf[0], ibuf[1] );
    }
    break;

  /*--- Get the CLIPPING status ------------------------------- */

  case GET_CLIPPING:
    /*
     * It allows the coherence check in the GRPCKG.
     */
    ibuf[0] = CLIPPING;
    break;

  /*--- Save the pixmap ID ------------------------------------ */

  case SAVE_PIXMAP_ID:
    if( xw_ok(xw) & (pixmap_ID != None) ) {
      int xmin, ymin, xmax, ymax, empty;
      pixmap_ID_save = pixmap_ID;

      /* Finds the bbox of the pixmap */
      xw_bbox_of_pixmap ( xw, pixmap_ID,
                          &xmin, &ymin, &xmax, &ymax, &empty );
      if( empty ) printf("X11 driver: line 2000: ERROR: pixmap is empty.\n");
      ibuf[0] = xmin;
      ibuf[1] = ymin;
      ibuf[2] = xmax;
      ibuf[3] = ymax;
    }
    break;

  /*--- Copy a saved pixmap ID in the current window ---------- */

  case COPY_PIXMAP_FROM_ID:
    if( xw_ok(xw) & (xw->pixmap != None) ) {
      int xmin, ymin, width, height;
      xmin = ibuf[0];
      ymin = ibuf[1];
      width  = ibuf[2] - ibuf[0] + 1;
      height = ibuf[3] - ibuf[1] + 1;
      XCopyArea ( xw->display, pixmap_ID_save, xw->window, xw->gc2,
                  xmin, ymin, width, height, 1, 1 );
      XFlush( xw->display );
    }
    break;

  /*--- Suppress the figure number in the window title -------- */

  case REM_FIG_NUM_IN_WIN_TITLE:
    /*
     * ibuf[0] = 0 : No figure number is added in the window title
     *               (convenient for shared legends)
     *           1 : (default) Figure number is added.
     */
    FIGURE_NUM_IN_TITLE = ibuf[0];
    break;

  /*--- Return the current window geometry ---------------------*/

  case GET_CURRENT_WIN_GEOM:

    if( xw->bad_device ) return;

    /* Getting window size */
    {
      Window root;
      unsigned border, depth;
      XGetGeometry( xw->display, xw->window, &root, &x, &y, &width, &height,
                    &border, &depth );
    }

    /* For the position, don't use XGetWindowAttributes or XGetGeometry,
     * they are not reliable... some users advice to use XTranslateCoordinates
     */
    {
      Window unused;
      XTranslateCoordinates( xw->display, xw->window, DefaultRootWindow(xw->display),
                             0, 0, &x, &y, &unused);
    }

    /* Take care of the WM title height before returning the y position */

    ibuf[0] = x;
    ibuf[1] = y - XW_WM_TITLE_HEIGHT;
    ibuf[2] = width;
    ibuf[3] = height;
    ibuf[4] = XW_WM_TITLE_HEIGHT; /* Return also this height, which depends
                                     on the window manager... */

    break;

  /*--- Read Cursor without Waiting ----------------------------*/

  case READ_CURSOR_WITHOUT_WAITING:
    if( xw_ok(xw) ) {

      if( xw_read_cursor_no_wait( xw, chr ) == 0 ) {
      } else {
        *chr = '\0';
      }
    } else {
      *chr = '\0';
    }
    *lchr = 1;
    break;

  /*--- Register a user-defined cursor shape -------------------*/

  case DEF_CUSTOM_CURSOR_SHAPE:
    /*
     * ibuf[0] = 1 : Main image for cursor 1 is provided
     *           2 : Mask image for cursor 1 is provided
     *           3 : Color for cursor 1 is provided
     *           4 : Main image for cursor 2 is provided
     *           5 : Mask image for cursor 2 is provided
     *           6 : Color for cursor 2 is provided
     */
    if( xw_ok(xw) ) {
      Window root = DefaultRootWindow(xw->display);
      unsigned int width, height;
      int x, y, x_hot, y_hot;
      int check;
      Colormap cmap;
      XColor white, exact;
      cmap = DefaultColormap(xw->display, xw->screen);
      XAllocNamedColor(xw->display, cmap, "white", &white, &exact);

      switch(ibuf[0]) {
        case 1:
          /* copy the filename; waiting for other data */
          strcpy( FILENAME_TMP, chr );
          break;
        case 2:
          /* set the color; waiting for other data */
          XAllocNamedColor(xw->display, cmap, chr, &PIX_USER1_COLOR, &exact);
          break;
        case 3:
          check = XReadBitmapFile( xw->display, root, chr,
                      &width, &height, &PIX_USER1_CURSOR_MASK, &x, &y );
          check = XReadBitmapFile( xw->display, root, FILENAME_TMP,
                      &width, &height, &PIX_USER1_CURSOR, &x_hot, &y_hot );
          xw->user1_cursor = XCreatePixmapCursor( xw->display,
                      PIX_USER1_CURSOR, PIX_USER1_CURSOR_MASK,
                      &PIX_USER1_COLOR, &white, x_hot, y_hot );
          break;
        case 4:
          /* just copy the filename; waiting for the mask. */
          strcpy( FILENAME_TMP, chr );
          break;
        case 5:
          /* set the color; waiting for other data */
          XAllocNamedColor(xw->display, cmap, chr, &PIX_USER2_COLOR, &exact);
          break;
        case 6:
          check = XReadBitmapFile( xw->display, root, chr,
                      &width, &height, &PIX_USER2_CURSOR_MASK, &x, &y );
          check = XReadBitmapFile( xw->display, root, FILENAME_TMP,
                      &width, &height, &PIX_USER2_CURSOR, &x_hot, &y_hot );
          xw->user2_cursor = XCreatePixmapCursor( xw->display,
                      PIX_USER2_CURSOR, PIX_USER2_CURSOR_MASK,
                      &PIX_USER2_COLOR, &white, x_hot, y_hot );
          break;
      }
    }
    break;

  /*--- Read Modified Cursor Position --------------------------*/

  case GET_MODIFIED_CURS_POS:
    if( xw_ok(xw) ) {
      rbuf[0] = X_OUT_WLD_COORD;
      rbuf[1] = Y_OUT_WLD_COORD;
    }
    break;

  /*--- --------------------------------------------------------*/
  default:
    fprintf(stderr, "Unimplemented opcode in X11 driver: %d\n", *ifunc);
    break;
  }
  /*
   * After a server error, close the connection to the display and set all
   * server resources to 'None'.
   */
  if(xw && xw->bad_device && xw->display)
    xw_del_XWdev(xw, 1);
  return;
}

#include "xwdriv_include/xw_add_events.inc"
#include "xwdriv_include/xw_animate_dashed_contour.inc"
#include "xwdriv_include/xw_animate_dashed_multispot.inc"
#include "xwdriv_include/xw_bad_device.inc"
#include "xwdriv_include/xw_bbox_of_pixmap.inc"
#include "xwdriv_include/xw_bitmap_from_pixmap.inc"
#include "xwdriv_include/xw_bound_cursor.inc"
#include "xwdriv_include/xw_checks_pixmap_aux.inc"
#include "xwdriv_include/xw_check_window_event.inc"
#include "xwdriv_include/xw_clear.inc"
#include "xwdriv_include/xw_clear_pixmap_aux.inc"
#include "xwdriv_include/xw_clip_at_viewport.inc"
#include "xwdriv_include/xw_current_pos.inc"
#include "xwdriv_include/xw_cursor_line.inc"
#include "xwdriv_include/xw_del_Band.inc"
#include "xwdriv_include/xw_del_XWdev.inc"
#include "xwdriv_include/xw_direct_draw_string.inc"
#include "xwdriv_include/xw_draw_arrow.inc"
#include "xwdriv_include/xw_draw_cursor.inc"
#include "xwdriv_include/xw_draw_marker.inc"
#include "xwdriv_include/xw_draw_string.inc"
#include "xwdriv_include/xw_end_cursor.inc"
#include "xwdriv_include/xw_erase_cursor.inc"
#include "xwdriv_include/xw_erase_labelling_area.inc"
#include "xwdriv_include/xw_erase_output_area.inc"
#include "xwdriv_include/xw_error.inc"
#include "xwdriv_include/xw_expose.inc"
#include "xwdriv_include/xw_find_exe.inc"
#include "xwdriv_include/xw_flush.inc"
#include "xwdriv_include/xw_get_click.inc"
#include "xwdriv_include/xw_get_image.inc"
#include "xwdriv_include/xw_get_image_line_RGB.inc"
#include "xwdriv_include/xw_get_pixmap.inc"
#include "xwdriv_include/xw_get_string_bbox.inc"
#include "xwdriv_include/xw_get_visual.inc"
#include "xwdriv_include/xw_get_window.inc"
#include "xwdriv_include/xw_handle_error.inc"
#include "xwdriv_include/xw_image_line.inc"
#include "xwdriv_include/xw_image_line_RGB.inc"
#include "xwdriv_include/xw_init_colors.inc"
#include "xwdriv_include/xw_insert_device.inc"
#include "xwdriv_include/xw_is_inside_quad.inc"
#include "xwdriv_include/xw_limit_pcoords.inc"
#include "xwdriv_include/xw_locate_cursor.inc"
#include "xwdriv_include/xw_mark_modified.inc"
#include "xwdriv_include/xw_move_arrow.inc"
#include "xwdriv_include/xw_move_arrow_in_pixmap.inc"
#include "xwdriv_include/xw_move_grobj.inc"
#include "xwdriv_include/xw_move_rect_in_pixmap.inc"
#include "xwdriv_include/xw_new_Band.inc"
#include "xwdriv_include/xw_new_cursors.inc"
#include "xwdriv_include/xw_new_geom.inc"
#include "xwdriv_include/xw_new_XWdev.inc"
#include "xwdriv_include/xw_next_event.inc"
#include "xwdriv_include/xw_next_page.inc"
#include "xwdriv_include/xw_nint.inc"
#include "xwdriv_include/xw_ok.inc"
#include "xwdriv_include/xw_raise_win.inc"
#include "xwdriv_include/xw_read_cursor_dyn.inc"
#include "xwdriv_include/xw_read_cursor_dyn_multispot.inc"
#include "xwdriv_include/xw_read_cursor_dyn_quad.inc"
#include "xwdriv_include/xw_read_cursor.inc"
#include "xwdriv_include/xw_read_cursor_no_wait.inc"
#include "xwdriv_include/xw_rem_events.inc"
#include "xwdriv_include/xw_remove_device.inc"
#include "xwdriv_include/xw_rgb_to_xcolor.inc"
#include "xwdriv_include/xw_scroll_rect.inc"
#include "xwdriv_include/xw_select_device.inc"
#include "xwdriv_include/xw_set_ci.inc"
#include "xwdriv_include/xw_set_cursor.inc"
#include "xwdriv_include/xw_set_focus_win.inc"
#include "xwdriv_include/xw_set_fontconfig_path.inc"
#include "xwdriv_include/xw_set_rgb.inc"
#include "xwdriv_include/xw_shift_cursor.inc"
#include "xwdriv_include/xw_show_pixmap.inc"
#include "xwdriv_include/xw_sync_error.inc"
#include "xwdriv_include/xw_update_colors.inc"
#include "xwdriv_include/xw_visual_info.inc"
#include "xwdriv_include/xw_win_resize.inc"
#include "xwdriv_include/xw_xcolor_to_rgb.inc"
#include "xwdriv_include/xw_XPoint_to_xy.inc"
#include "xwdriv_include/xw_xy_to_XPoint.inc"

