subroutine post_pdfdriv( unit, fname, ier )

   implicit none

   integer,          intent(in)  :: unit
   character(len=*), intent(in)  :: fname
   integer,          intent(out) :: ier
   !------ API end ------

   ! Post-processing for the MFPLOT driver for Portable Document Format
   ! (returns ier=0 if all is OK)
   !
   ! Objects' addresses to be inserted in the document are offset, so they
   ! begin at 0 for the first position, etc. Therefore there is a shift of
   ! one with respect to the position in the file, in term of Fortran
   ! direct access.
   !
   ! This routine has been designed to work only with PDF created by MFPLOT.
   ! It shouldn't be used to fix other PDF files; it has therefore the
   ! following constraints:
   !  * the header of all objects must contain the string " 0 obj <<" or
   !    " 0 obj ["; for the latter case, a LineFeed (char=10) must follow
   !    immediately the "[" character.
   !  * all objects' numbers must constitute a continuous list, without
   !    any hole.
   !  * total number of object cannot be greater than 'nb_obj_max',
   !    defined below as the first variable; but this parameter can be
   !    increased.
   !  * the "/Length" keyword must be indented by exactly 2 spaces from the
   !    left; it must be followed by a string which must contain at least
   !    10 chars before the next LineFeed (e.g. "0         "; a valid
   !    length don't need to be already present).
   !  * "stream", "endstream", "xref" and "startxref" tags must appear as
   !    alone on their line, without any space before or after.
   !  * no comment can be present between the opening tag "<<" and the
   !    "stream" tag.
   !  * the "trailer" keyword must begin a line and must be followed by
   !    " <<".
   !  * in the trailer dictionary, "/Size" must be located at the first
   !    position.
   !  * after the closing ">>" of the trailer dictionary, the 'startxref'
   !    keyword must follow immediately, without any blank line.
   !
   ! 02.08.2018 - Creation.
   ! 12.08.2018 - Check that the number of objects found is consistant with
   !              the size of the xref table and the size mentioned in the
   !              trailer.
   !              Check that all lines of the xref table contains 19 chars
   !              plus the LineFeed character (i.e. check that there is a
   !              space at the end of all address lines).
   ! 12.09.2018 - Initialize 'startxref_addr' variable to -1, in order
   !              to trigger an error when 'startxref' is not found.
   ! 23.11.2018 - Fix the read of nb_obj_plus_one_2 in an internal variable.
   !              GNU Fortran was ok but not INTEL Fortran 18.
   ! 03.12.2018 - Support also Windows line termination (CR+LF).
   ! 27.10.2021 - Just change layout of error messages. Update the list of
   !              constraints above.
   !---------------------------------------------------------------------------

   integer, parameter :: nb_obj_max = 5000

   integer :: obj_num(nb_obj_max), obj_addr(nb_obj_max),                &
              stream_len(nb_obj_max), len_addr(nb_obj_max),             &
              xref_addr, first_addr_in_xref, startxref_addr
   logical :: has_stream(nb_obj_max)

   character(len=1), parameter :: LF = char(10), CR = char(13)
   character(len=2), parameter :: CRLF = CR // LF
   character(len=2) :: EOL ! End-Of-Line (1 or 2 chars, according Unix or Windows)
   logical :: windows

   integer :: iostat, i, k, k_obj, pos
   integer :: nb_obj_plus_one_1, nb_obj_plus_one_2, xref_lines
   integer :: beg_stream, end_stream
   integer, parameter :: len_car = 14
   character(len=len_car) :: string
   character(len=8) :: tmp
   character(len=10) :: str10
   logical :: in_stream, items_are_present(3)

   ier = 0

!!print "(/,A,A)", "  PDF driver: Post-processing of ", trim(fname)
   ! Opening 'fname' in stream/IO
   open( unit, file=trim(fname), access="stream",                       &
         status="old", action="readwrite", iostat=iostat )

   if( iostat /= 0 ) then
      ier = 3
      print *
      print *, "(Muesli FGL:) PDF driver:"
      print *, "  Error: cannot postprocess the PDF!"
      print *, "  PDF file: """, trim(fname), """"
      print *, "  -> file deleted (or at least not writable)?"
      print "(A,I0,A)", "   (post_pdfdriv: ier=", ier, ")"
      print *
      return
   end if

   ! Check the header and the line termination of the file
   string = ""
   read( unit, pos=1 ) string(1:9)
   if( string(1:4) /= "%PDF" ) then
      ier = 1
      print *
      print *, "(Muesli FGL:) PDF driver:"
      print *, "  Error: bad PDF header!"
      print *, "  PDF file: """, trim(fname), """"
      print *, "  -> file must begin with '%PDF'"
      print "(A,I0,A)", "   (post_pdfdriv: ier=", ier, ")"
      print *
      return
   end if
   if( string(9:9) == LF ) then
      ! Unix-like line termination
!!print *, " PDF driver: PDF is recognized as a Unix file (LF line termination)."
      Windows = .false.
      EOL = LF
   else if( string(9:9) == CR ) then
      ! Windows-like line termination
!!print *, " PDF driver: PDF is recognized as a Windows file (CR+LF line termination)."
      Windows = .true.
      EOL = CRLF
   else
      ier = 2
      print *
      print *, "(Muesli FGL:) PDF driver:"
      print *, "  Error: cannot detect the line termination!"
      print *, "  PDF file: """, trim(fname), """"
      print "(A,I0,A)", "   (post_pdfdriv: ier=", ier, ")"
      print *
      return
   end if
   rewind(unit)

   has_stream(:) = .false.
   stream_len(:) = -1
   len_addr(:) = -1
   xref_addr = -1
   xref_lines = 0
   startxref_addr = -1

   items_are_present(:) = .false.

   string = ""
   k = 0
   k_obj = 0
   do

      ! 'string' is a kind of accumulator which contains 14 characters.
      ! Before each read of a new char, we have to shift present chars
      ! one position to the left...
      k = k + 1
      call shift_left( string )
      read( unit, pos=k, end=99 ) string(len_car:len_car)
!!print *, " string = ", string
!!read * ! pause without the requirement of typing 'go'

      if( string(6:14) == " 0 obj <<" .or.                              &
          string(6:14) == " 0 obj ["//EOL(1:1) ) then
!!print *, " YES ! found an object!"
         ! clean string(1:5) to remove linefeed=char(10)
         tmp = string(1:5)
         i = index( tmp, LF, back=.true. )
         if( i > 0 ) then
            tmp = tmp(i+1:)
!!print *, "      tmp = ", tmp, "    i (linefeed) = ", i
         else
            ier = 4
            print *
            print *, "(Muesli FGL:) PDF driver:"
            print *, "  Internal Error: LineFeed not found in a substring!"
            print "(A,I0,A)", "   (post_pdfdriv: ier=", ier, ")"
            print *
            return
         end if
         k_obj = k_obj + 1
         if( k_obj >= nb_obj_max ) then
            ier = 5
            print *
            print *, "(Muesli FGL:) PDF driver:"
            print *, "  Internal Error: too much object in the PDF!"
            print *, "  You must increase the value of 'nb_obj_max' in"
            print *, "  'post_pdfdriver.f90' and compile again the Muesli library"
            print *, "  Good luck!"
            print "(A,I0,A)", "   (post_pdfdriv: ier=", ier, ")"
            print *
            return
         end if
         read(tmp,*) obj_num(k_obj)
         obj_addr(k_obj) = k - 8 - 5 + i
!!print *, "      obj_num = ", obj_num(k_obj), "  obj_addr = ", obj_addr(k_obj)
      else if( string(4:14) == LF//"  /Length " ) then
         len_addr(k_obj) = k + 1
      else if( string(5:14) == ">>"//LF//"stream"//LF .or.              &
               string(3:14) == ">>"//CRLF//"stream"//CRLF ) then
         has_stream(k_obj) = .true.
         beg_stream = k + len_trim(EOL)
      else if( string(4:14) == LF//"endstream"//LF .or.                 &
               string(2:14) == CRLF//"endstream"//CRLF ) then
         end_stream = k - 9 - len_trim(EOL)
         stream_len(k_obj) = end_stream - beg_stream + 1
      else if( string(9:14) == LF//"xref"//LF .or.                      &
               string(7:14) == CRLF//"xref"//CRLF ) then
         items_are_present(1) = .true.
         xref_addr = k - 3 - len_trim(EOL)
      else if( string(1:6) == LF//"xref"//LF .or.                       &
               string(1:8) == CRLF//"xref"//CRLF ) then
         ! find the number of obj + 1
         tmp = string(7+2*len_trim(EOL):) ! this remove also the "0 " after "xref"
         i = index( tmp, EOL(1:1) )
         if( i > 0 ) then
            tmp = tmp(1:i-1)
         else
            ier = 6
            print *
            print *, "(Muesli FGL:) PDF driver:"
            print *, "  Internal Error: LineFeed not found in a substring!"
            print "(A,I0,A)", "   (post_pdfdriv: ier=", ier, ")"
            print *
            return
         end if
         read(tmp,*) nb_obj_plus_one_1
!!print *, "      nb_obj_plus_one_1 = ", nb_obj_plus_one_1
      else if( string(1:14) == LF//"0000000000 65" ) then
         items_are_present(2) = .true.
         first_addr_in_xref = k - 12
      else if( string(1:10) == " 00000 n "//LF .or.                     &
               string(1:10) == " 00000 n"//CRLF ) then
         xref_lines = xref_lines + 1
      else if( string(1:13) == "trailer <<"//LF//"  " .or.              &
               string(1:14) == "trailer <<"//CRLF//"  " ) then
         items_are_present(3) = .true.
         ! shift again to reach "/Size"
         do i = 1, 12+len_trim(EOL)
            k = k + 1
            call shift_left( string )
            read( unit, pos=k, end=99 ) string(len_car:len_car)
         end do
         ! now, we should find "/Size"
         if( string(1:5) /= "/Size" ) then
            ier = 7
            print *
            print *, "(Muesli FGL:) PDF driver:"
            print *, "  Internal Error: /Size not found in the trailer!"
            print "(A,I0,A)", "   (post_pdfdriv: ier=", ier, ")"
            print *
            return
         end if
         tmp = string(7:) ! this remove "/Size"
         i = index( tmp, EOL(1:1) ) ! we need also (for Intel Fortran) to remove the LF
         if( i == 0 ) then
            print *, "(Muesli FGL:) PDF driver:"
            print *, "  Internal Error: /Size found in the trailer,"
            print "(A,I0,A)", "   but cannot isolate the integer (LF not found)"
            print *
            return
         end if
         read(tmp(1:i-1),*) nb_obj_plus_one_2
      else if( string(2:14) == ">>"//LF//"startxref"//LF .or.           &
               string(2:14) == CRLF//"startxref"//CRLF ) then
         startxref_addr = k + 1
      end if

   end do

99 continue

!!k = k - 1
!!print "(A,I0)", "*** End-of-File found at address: ", k
!!print *, " items_are_present: ", items_are_present

   if( .not. all(items_are_present) ) then
      ier = 8
      print *
      print *, "(Muesli FGL:) PDF driver:"
      print *, "  Internal Error: some required items are not found!"
      print *, "  (among others ""xref"", ""0000000000 65"" and ""trailer"")"
      print "(A,I0,A)", "   (post_pdfdriv: ier=", ier, ")"
      print *
      return
   end if

!!print *, " xref_lines = ", xref_lines
   if( xref_lines /= nb_obj_plus_one_1 - 1 ) then
      ier = 9
      print *
      print *, "(Muesli FGL:) PDF driver:"
      print *, "  Internal Error: bad number of valid address lines in xref table!"
      print *, "  (these lines must be terminated by "" 00000 n ""; note the space at the end)"
      print "(A,I0,A)", "   (post_pdfdriv: ier=", ier, ")"
      print *
      return
   end if

   if( nb_obj_plus_one_1 /= nb_obj_plus_one_2 ) then
      ier = 10
      print *
      print *, "(Muesli FGL:) PDF driver:"
      print *, "  Internal Error: sizes not consistant in xref and trailer!"
      print "(A,I0,A)", "   (post_pdfdriv: ier=", ier, ")"
      print *
      return
   end if

!!print *, " nb obj found: ", k_obj
   if( nb_obj_plus_one_1 /= k_obj + 1 ) then
      ier = 11
      print *
      print *, "(Muesli FGL:) PDF driver:"
      print *, "  Internal Error: incorrect nb of objects!"
      print "(A,I0,A)", "   (post_pdfdriv: ier=", ier, ")"
      print *
      return
   end if

!!print *, " has stream ? ", has_stream(1:k_obj)
!!print *, " stream_len : ", stream_len(1:k_obj)
!!print *, " len_addr   : ", len_addr(1:k_obj)

!!print *, " xref_addr: ", xref_addr
!!print *, " first_addr_in_xref: ", first_addr_in_xref

!!      close( unit )
!!      open( unit, file=trim(fname), access="stream", form="formatted" )

   ! Now, it's time to write all addresses and stream lengths in the PDF
   do k = 1, k_obj

      i = obj_num(k)
      pos = first_addr_in_xref + 20*i
      write( str10, "(I10)" ) obj_addr(k) - 1
      call add_leading_zero( str10 )
      write( unit, pos=pos ) str10

      if( has_stream(k) ) then
         pos = len_addr(k)
         write( str10, "(I0)" ) stream_len(k)
         write( unit, pos=pos ) str10
      end if

   end do

   pos = startxref_addr
   if( pos < 0 ) then
      ier = 12
      print *
      print *, "(Muesli FGL:) PDF driver:"
      print *, "  Internal Error: didn't locate 'startxref' keyword!"
      print "(A,I0,A)", "   (post_pdfdriv: ier=", ier, ")"
      print *
      return
   end if
   write( str10, "(I0)" ) xref_addr - 1
   write( unit, pos=pos ) str10

   close( unit )

contains
   !____________________________________________________________________
   !
   subroutine shift_left( string )

      character(len=len_car) :: string

      integer :: i

      do i = 1, len_car-1
         string(i:i) = string(i+1:i+1)
      end do

   end subroutine shift_left
   !____________________________________________________________________
   !
   subroutine add_leading_zero( string )

      character(len=*) :: string

      integer :: i

      do i = 1, len(string)
         if( string(i:i) == " " ) string(i:i) = "0"
      end do

   end subroutine add_leading_zero
   !____________________________________________________________________
   !
end subroutine post_pdfdriv
