! MUESLI-MFPLOT EPS driver
!
subroutine EPS_DRIVER( ifunc, rbuf, ibuf, chr, lchr )

   use mod_mfarray, only: mf_muesli_version

   use mod_pgplot

   implicit none

   integer          :: ifunc, ibuf(*), lchr
   double precision :: rbuf(*)
   character(len=*) :: chr
   !------ API end ------

   !---------------------------------------------------------------------------
   ! MFPLOT driver for Encapsulated PostScript devices.
   !
   ! Édouard CANOT version :
   ! (see psdriv.info file for previous changes, before 2001)
   !
   ! 12.09.2001 - Added entry (IFUNC = 30) for Bézier curve
   !                (cf. lines between CEC+ et CEC- marks);
   !              Added a routine : GRFAO_EC at the end of the file.
   ! 14.08.2004 - Added entry (IFUNC = 31) for update BBox.
   ! 06.09.2004 - Added /inputf in PS prolog.
   ! 13.03.2005 - Fix BBOX initialization (not use huge());
   !              Write trimed string in GRPS02, from IFUNC = 30
   !                (because, length = 50000 now).
   ! 14.03.2005 - Eliminate BB update in IFUNC = 30
   !                (calling routine must use IFUNC = 31).
   ! 06.01.2006 - Max color index is now MAX_COLORS-1 (4175)
   ! 16.11.2006 - Change name of creator (line 432-433):
   !                %%Title: PGPLOT PostScript plot
   !                %%Creator: PGPLOT [PSDRIV 6.7ec]
   !              becomes
   !                %%Title: MUESLI PostScript plot
   !                %%Creator: MUESLI [PSDRIV 6.7ec]
   ! 18.12.2006 - WIDTH, HEIGHT, HOFFSET and VOFFSET cannot be any
   !              longer modified by environmental variables, because
   !              the ratio x/y of the plot must match those of the
   !              X11 window.
   ! 17.03.2007 - Fix initialisation of BBOX (IFUNC = 9), which
   !              contains too large values.
   ! 21.03.2007 - Writing BBOX (IFUNC = 10), also at the beginning of
   !              the file.
   ! 22.05.2007 - Writing PS header (IFUNC = 9): suppress the write of the
   !              USER name.
   ! 16.07.2007 - All images are written (IFUNC = 26) using compression
   !              via the zlib (should work only for Color devices).
   ! 25.07.2007 - Using 'gresc' in writing compressed images (IFUNC=26)
   !              leads to a recursive call to 'grexec': changed to 'GRPS02'
   ! 18.10.2007 - '90 rotate' is deleted; therefore the image is stored as
   !              a 'portrait EPS' and the preview in KDE is correct.
   !              New BBox obeys the change:
   !                a_new = b_old + 519
   !                b_new = 519 - c_old
   !                c_new = d_old + 519
   !                d_new = 519 - a_old
   ! 11.11.2007 - Increase EPS linewidth (IFUNC=22), because most of
   !              EPS files are intended for including in LaTex with
   !              reduction.
   ! 15.02.2008 - Added some information in the 'Title' field:
   !              executable name and figure number.
   ! 21.12.2009 - Length of OBUF (Output Buffer) increased to 50000
   !                (same length as INSTR -- Input String).
   !              Add entry point (IFUNC=32) for writing a comment
   !              in the EPS file.
   ! 07.07.2010 - Delayed the first comment added by MUESLI (IFUNC=32),
   !              after the initialization by IFUNC = 11.
   !              Specific PSDRIV version is removed:
   !                %%Creator: MUESLI [PSDRIV 6.7ec]
   !              becomes
   !                %%Creator: MUESLI <version_number>
   !              Now, psdriv.f "USE"s mod_mfarray module.
   ! 08.07.2010 - Why 50000 as length for INSTR and OBUF? Seems to be
   !              never fully employed because of test at label 800...
   !              Thus, reduce length (down to 132).
   ! 10.07.2010 - Redesign of the whole driver; in particular:
   !              1) view surface dimension is strictly proportional
   !                 to the X11 window
   !              2) drawing in the EPS is more close to the X11 plot
   !              3) only the LANDSCAPE mode is used
   !              4) only the COLOR mode is used
   !              5) only ONE page
   ! 12.07.2010 - Adding ISOLatin1 PS fonts support. Prolog comes
   !              from MATLAB.
   ! 20.07.2010 - Few corrections for conforming EPSF specifications.
   !              Added the "%%DocumentFonts:", "%%DocumentNeededFonts:"
   !              tags.
   ! 04.08.2014 - Add the 'showpage' command at the end of the PostScript
   !              file, according the value of the environmental variable
   !              MFPLOT_ADD_SHOWPAGE_TO_EPS (1=yes, 0=no). Default is no.
   !              This is not required by the PS language standard but it
   !              is due to some EPS vizualiser (Okular, Evince, ...) which,
   !              according to some bugged engine, show a black
   !              figure. See IFUNC=10 (Close workstation).
   ! 05.07.2015 - Initialize bbox_updated during declaration, and change
   !              some things about bbox_updated.
   ! 02.11.2015 - Add support for bold fonts (Std PS fonts only);
   !              DocumentFonts and DocumentNeededFonts at the end of the
   !              document contain now only the appropriate fontnames.
   ! 27.06.2018 - Only one type defined here: EPS
   !              (takes place of PS, VPS, CPS, VCPS)
   ! 04.07.2018 - Native dash patterns implemented (IFUNC=19).
   ! 10.07.2018 - New algorithm for embedding bitmap images: it can now
   !              use compressing on streams and output in ASCII85 encoding,
   !              better than HEXA. The choice between the old and the new
   !              algorithm is made via the environment variable
   !              MFPLOT_DEFLATE_A85 (0 or 1). Default is 1.
   ! 18.07.2018 - Complete redesign of coloured shading: the writting is no
   !              longer pixel-oriented, as previously, like for X11 device.
   !              The shading feature of PS 3.0 (shfill command) is used. This
   !              implies to store the whole colormap at the beginning of the
   !              EPS. Currently, there is only one colormap per EPS image.
   ! 04.08.2018 - Transparency added.
   ! 12.08.2018 - Added a boolean (EPS_driver_font_begin) to prevent selecting
   !              many times the same font (see 'grparse_ps_font').
   ! 03.09.2018 - Opcode 28 shift to 38, because for implementing markers in
   !              xwdriv, the number 28 is already in use.
   ! 09.09.2018 - Added 7 old colors (from Matlab) to the 16 predefined colors.
   !              Total number of predefined colors is now 23.
   ! 10.09.2018 - Added 7 new colors (from Matlab) to the 23 predefined colors.
   !              Total number of predefined colors is now 30.
   !              Added also 12 new colors (from the Breeze colortable).
   !              Total number of predefined colors is now 42.
   ! 14.09.2018 - Enlarge slightly the dash segment in the '.-' pattern for
   !              lines: [75 45 8 45] instead of [60 45 8 45].
   !              Added the dash pattern offset, useful for short segments
   !              (see IFUNC=19).
   ! 15.09.2018 - Added shortcut "d" for "setdash" in the prolog.
   ! 29.02.2020 - Used now double precision instead of single precision.
   ! 17.03.2020 - Enlarged all segments of all dash patterns, by a factor 2.67
   !              (see IFUNC=19).
   ! 18.03.2020 - Experimentally, we take all segment lengths of dash patterns
   !              proportional to sqrt(linewidth) (see IFUNC=19).
   ! 28.03.2020 - Fixed the dash length references.
   !  2.04.2020 - Fixed the /L shortcut in the header of the EPS: added the
   !              'newpath' command at the beginning of the sequence. If
   !              it is not present, problems can occur when clipping a
   !              dashed "line+point" (see IFUNC=9).
   ! 12.04.2020 - 'nbuf' scalar argument replaced by the 'ibuf' vector.
   ! 29.05.2020 - GRFAO_EC renamed format_string and modified to accept
   !              optional arguments. The routine is moved inside the
   !              PSDRIV routine (contained).
   ! 02.06.2020 - Comments "%%BeginProlog", "%%EndProlog", etc, are written
   !              only under condition in the EPS file.
   ! 03.06.2020 - When colors are all equal: use 'setgray' command (shortcut=G)
   !              instead of 'setrgbcolor'; this reduce the size of the EPS.
   ! 03.06.2020 - Trailing zeros after the decimal point of reals are removed;
   !              if the decimal point is at the last position, it is removed
   !              as well. This reduce the size of the EPS.
   !              Fixed also a bug concerning the filling of the string buffer
   !              at the end of the main routine: the condition to insert a
   !              space is "lobuf >= 1" and not "lobuf > 1"!
   ! 07.06.2020 - Reverted conditional write of DSC comments (those beginning
   !              with two "%").
   ! 09.06.2020 - Increased DSC comment "%%LanguageLevel" to level 3, because
   !              shading and deflate commands fall into this category.
   ! 19.02.2021 - Removed call to flush_buffer in opcode 16. Not necessary.
   ! 23.02.2021 - Changed definition of colors which have indices in [8-15].
   !              These are now only gray shades. Two particular colors:
   !              quasi-white and quasi-black, will have a future usage.
   ! 24.02.2021 - Minor fixes here and there. Introduced a black background
   !              when the writing mode is white over black.
   !              (added IFUNC=57, same as xwdriv).
   ! 24.02.2021 - Back to the flush_buffer in opcode 16. Actually, this
   !              is useful when writing comments, to avoid that the
   !              "%-- end of..." string is written before the command itself.
   ! 13.03.2021 - Rectangle fill implemented (opcode 24). It uses the
   !              PostScript operator 'rectfill', available since Level 2.
   !              Changed the returned status code (opcode 4) accordingly.
   !              'rectclip' is also used now (opcodes 20, 24, 26).
   ! 14.03.2021 - Added clipping in opcode 20 ('Polygon fill').
   ! 15.03.2021 - Removed usage of CLIPPING. Indeed, clipping can be
   !              set/unset outside of the driver (cf. mod_pgplot),
   !              and is set before drawing all Graphic Objects.
   !              Modified the computation of the final BBox of the EPS;
   !              now it is constant ('Full figure size') except if a legend
   !              is drawn outside the axes.
   ! 17.03.2021 - Changed the name of this driver to EPS_DRIVER.
   !              Introduced parameter int_not_defined=huge(1), to be used
   !              during initialization of (lasti,lastj). Previously, -1
   !              was used, but this later value could be a valid position.
   !              Introduced RC and RF as shortcut for rectclip and rectfill.
   ! 25.03.2021 - Fix the constant BBox.
   ! 29.03.2021 - Update of the BBox in the only case of a legend frame
   !              draw outside the window. Opcod is now 45 instead of 31.
   ! 30.03.2021 - Opcode 83 removed.
   !              Some information fields in the header (Title, Creator,
   !              CreationDate, Author) may be omitted by setting the
   !              environment MFPLOT_EPS_ANONYMOUS variable to 1.
   ! 31.03.2021 - Environment variable MFPLOT_ADD_SHOWPAGE_TO_EPS renamed
   !              MFPLOT_EPS_ADD_SHOWPAGE.
   !              The 'showpage' operator is always added at the end if the
   !              EPS file is empty.
   ! 29.10.2021 - Added 5 new symbols from the CeMaSP font, design by myself.
   !              Total number of symbols is now 26 instead of 21.
   !  9.11.2021 - Introduced shortcuts (taking first letter only) for 'curveto'
   !              'moveto' and 'lineto' in the Prolog.
   ! 13.11.2021 - Changed slightly the linewidth factor, in order to match
   !              the linewidth on a X11 screen (for the same window size).
   !              See opcode 22.
   ! 15.11.2021 - The initial settings of cap_style and join_style in the
   !              initialization part (opcode 9) have been removed, because
   !              these two items will be changed by appropriate new opcodes.
   !              New opcode 25 to set these line attributes.
   ! 01.12.2021 - Removed the shortcut /L and /C in the preamble. This
   !              inhibited the dash regularity. Redesign of many things:
   !              some internal subroutines removed (not very useful);
   !              new shortcut for PostScript operator (equal or close to
   !              those of PDF language); when possible, use preferably
   !              absolute coords (i.e. moveto and lineto) instead of
   !              relative coords (this is to conform to the PDF way).
   ! 04.12.2021 - Added opcode 70, to get the dash pattern period.
   !              Interior of markers is now transparent and not white.
   ! 14.12.2021 - Added shortcut /RS for /rectstroke.
   ! 15.12.2021 - Changed all numerical target labels of the select case to
   !              predefined character strings.
   ! 22.03.2023 - Inverted dashes implemented in SET_LINE_STYLE.
   !---------------------------------------------------------------------------

   character(len=*), parameter :: ctype = "EPS (Encapsulated PostScript)"

   integer, parameter :: STDERR = 0

   ! Document area in milli-inches: (WIDTH, HEIGHT) are the dimensions of
   ! the printable area for an EPS image, BBox can be defined at the end;
   ! Keep a large image; on the contrary, roundoff will destroyed very
   ! small scales... Before writing, the dimensions will be reduced.
   integer, save :: WIDTH, HEIGHT

   integer :: ier, i0, j0, i1, j1, L, L1, L2, lc, lj
   integer, save :: lasti, lastj, unit
   integer, parameter :: int_not_defined = huge(1)

   integer :: i2, j2, i3, j3
   double precision :: x1, x2, y1, y2
   integer :: x11_id

   logical :: anonymous_eps
   logical, save :: EMPTY_FILE

   integer :: ci
   integer, save :: npts, ioerr, line_style
   integer :: dash_offset
   integer :: n1, n2, n3, n4

   ! Polylines Management
   logical, save :: something_has_been_written
   logical, save :: polyline_start = .false., in_polyline = .false.
   logical :: bool_1, bool_2

   ! Dash patterns definition : the following values must match those
   ! in the PDF driver. They have been designed to match also the dash
   ! lengths of the X11 driver.
   integer, parameter :: dash_length_1 = 80,                            &
                         dash_length_2 = 48,                            &
                         dash_length_3 =  8
   ! Useful only for dashed line, to know the period of the pattern, in pixels
   integer, save :: dash_pattern_period = 0

   double precision, save :: current_linewidth = 0.0d0

   integer, save :: state = 0, nseg

   character(len=10) :: fig_number_str

   integer :: nxp, nyp, xorg, yorg, xlen, ylen, n, rgb(3)
   integer :: i, k, status

   logical, save :: stdout
   double precision, save :: lw

   integer, save :: bbox(4)

   integer, parameter :: MAX_COLORS = 4176
   double precision, save :: rvalue(0:MAX_COLORS-1),                    &
                             gvalue(0:MAX_COLORS-1),                    &
                             bvalue(0:MAX_COLORS-1)
   logical, save :: NEEDS_COLORMAP = .false.
   integer, save :: COLORMAP_CI_LOW, COLORMAP_CI_HIGH, COLORMAP_LENGTH
   integer, save :: BITS_PER_SAMPLE
   integer :: i_red, i_green, i_blue

   character(len=120) :: msg
   character(len=32) :: str_tmp
   character :: c

   ! 'obuf' is an output buffer, to keep PS commands before writing
   ! in the file
   integer, parameter :: lobuf_max = 132
   character(len=lobuf_max), save :: obuf
   integer, save :: lobuf
   character(len=lobuf_max) :: instr, instr2

   character(len=255), save :: fname
   integer, save :: lfname
   character(len=256), save :: fnameC ! same than fname, but with a NULL
                                      ! char at the end (for passing to C)
   integer, save :: lfnameC

   integer, parameter :: nb_MARK = 26
   integer :: nsym
   integer, save :: marker(1:nb_MARK), rad(1:nb_MARK)
   data rad / nb_MARK * 10 /

   double precision, save :: mfac

   integer, parameter :: NCOLORS = 42
   double precision, save :: rinit(0:NCOLORS-1), ginit(0:NCOLORS-1), binit(0:NCOLORS-1)
   data rinit /1.00d0, 0.00d0, 1.00d0, 0.00d0, 0.00d0, 0.00d0, 1.00d0, 1.00d0,   &
               0.003922d0, 0.1647d0, 0.3333d0, 0.5d0, 0.6667d0, 0.8314d0, 0.9961d0, 1.0d0, &
               0.00d0, 0.00d0, 1.00d0, 0.00d0, 0.75d0, 0.75d0, 0.25d0,           &
               0.000d0, 0.850d0, 0.929d0, 0.494d0, 0.466d0, 0.301d0, 0.635d0,    &
               0.2392d0, 0.9294d0, 0.9647d0, 0.9922d0, 0.1529d0, 0.7882d0,       &
               0.0667d0, 0.1608d0, 0.6078d0, 0.4980d0, 0.1020d0, 0.7529d0 /
   data ginit /1.00d0, 0.00d0, 0.00d0, 1.00d0, 0.00d0, 1.00d0, 0.00d0, 1.00d0,   &
               0.003922d0, 0.1647d0, 0.3333d0, 0.5d0, 0.6667d0, 0.8314d0, 0.9961d0, 1.0d0, &
               0.00d0, 0.50d0, 0.00d0, 0.75d0, 0.00d0, 0.75d0, 0.25d0,           &
               0.447d0, 0.325d0, 0.694d0, 0.184d0, 0.674d0, 0.745d0, 0.078d0,    &
               0.6824d0, 0.0824d0, 0.4549d0, 0.7373d0, 0.6824d0, 0.8078d0,       &
               0.8196d0, 0.5020d0, 0.3490d0, 0.5490d0, 0.7373d0, 0.2235d0 /
   data binit /1.00d0, 0.00d0, 0.00d0, 0.00d0, 1.00d0, 1.00d0, 1.00d0, 0.00d0,   &
               0.003922d0, 0.1647d0, 0.3333d0, 0.5d0, 0.6667d0, 0.8314d0, 0.9961d0, 1.0d0, &
               1.00d0, 0.00d0, 0.00d0, 0.75d0, 0.75d0, 0.00d0, 0.25d0,           &
               0.741d0, 0.098d0, 0.125d0, 0.556d0, 0.188d0, 0.933d0, 0.184d0,    &
               0.9137d0, 0.0824d0, 0.0000d0, 0.2941d0, 0.3765d0, 0.2314d0,       &
               0.0863d0, 0.7255d0, 0.7137d0, 0.5529d0, 0.6118d0, 0.1686d0 /

   integer :: irec_offset

   ! Deflate support (zlib)

   ! (multiple de 3 car R,G,B)
   integer, parameter :: len_in_max = 30000

   ! ATTENTION integer 1 byte (et non 4 par défaut !)
!### TODO 2: l'écriture devrait être plus standard, avec utilisation
!          de kind
   integer*1, save :: int_in(len_in_max)
   integer, save :: len_in
   character(len=len_in_max) :: string_out
   integer :: len_out, job

   interface
      subroutine deflate_stream_to_hex( in_bin_str, len_in_bin,         &
                                        out_hex_str, len_out_hex,       &
                                        job, ier )
         integer*1        :: in_bin_str(*)
         integer          :: len_in_bin
         character(len=*) :: out_hex_str
         integer          :: len_out_hex, job, ier
      end subroutine
      subroutine deflate_stream_to_a85( filename, len_filename,         &
                                        buffer, len_buffer,             &
                                        job, ier )
         character(len=*) :: filename
         integer          :: len_filename
         integer*1        :: buffer(*)
         integer          :: len_buffer, job, ier
      end subroutine
   end interface

   integer, external :: gr_real_user

   logical, save :: init_ok = .false.,                                  &
                    chr_save_to_be_printed = .false.
   character(len=132), save :: chr_save
   logical, save :: bbox_updated = .false.

   character :: value
   character(len=120) :: fonts_present

   integer, save :: X11_grxmxa = -1, X11_grymxa = -1

   character(len=10) :: str1, str2, str3

   ! 1: black on white (default), 0: white on black (PGPLOT default)
   integer, save :: BLACK_ON_WHITE = 1

   !------ end of declarations -- execution starts hereafter  ------

   ! at the end of each select case block, return if no string has been filled

   select case( ifunc )

      !-----------------------------------------------------------------
      case( GET_DEV_NAME ) ! Return device name.

      ! landscape, color
      chr = ctype
      lchr = len(ctype)

      return

      !-----------------------------------------------------------------
      case( GET_COL_IND_RANGE ) ! Return range of color indexes.

      ibuf(1) =  0
      ibuf(2) = MAX_COLORS - 1

      return

      !-----------------------------------------------------------------
      case( GET_DEV_RESOL ) ! Return device resolution.

      ibuf(1) = 1200
      ibuf(2) = 1200
      ibuf(3) = 5    ! device coords per pixel

      return

      !-----------------------------------------------------------------
      case( GET_MISC_INFO ) ! Return misc device info.

      chr( 1: 1) = "H" ! Interactive device (I) or Hardcopy (H)
      chr( 2: 2) = "N" ! Cursor available (C) or not (N)
      chr( 3: 3) = "D" ! Dashed lines available (D) or not (N)
      chr( 4: 4) = "A" ! Area fill available (A) or not (N)
      chr( 5: 5) = "T" ! Thick lines available (T) or not (N)
      chr( 6: 6) = "R" ! Rectangle fill available (R) or not (N)
      chr( 7: 7) = "Q" ! Pixel device (P), Image device (Q) or nothing (N)
      chr( 8: 8) = "N" ! Persistent device (Verbose prompt) or not (N)
      chr( 9: 9) = "Y" ! Can return color representation? (Y) or not (N)
      chr(10:10) = "M" ! Markers available (M) or not (N)

      lchr = 10

      return

      !-----------------------------------------------------------------
      case( GET_DEF_FILENAME ) ! Return default file name.

      return

      !-----------------------------------------------------------------
      case( GET_DEF_SIZE ) ! Return default physical size of plot.

      ibuf(1) = 0
      ibuf(2) = WIDTH
      ibuf(3) = 0
      ibuf(4) = HEIGHT

      return

      !-----------------------------------------------------------------
      case( GET_CHAR_SIZE ) ! Return character size.
                            ! (actually, never called)

      rbuf(1) = 8.0d0

      return

      !-----------------------------------------------------------------
      case( SELECT_PLOT ) ! Select plot.
                          ! No selection: there is only one EPS file opened.

      return

      !-----------------------------------------------------------------
      case( OPEN_DEV ) ! Open device.

      ! check for concurrent access
      if( state == 1 ) then
         call grwarn( "an EPS file is already open" )
         ibuf(1) = 0
         ibuf(2) = 0
         return
      end if

      EMPTY_FILE = .false.

      do ci = 0, NCOLORS-1
         rvalue(ci) = rinit(ci)
         gvalue(ci) = ginit(ci)
         bvalue(ci) = binit(ci)
      end do

      ! Device dimensions

      ! X11 max plotting are
      ! (eventually virtual, case /NULL, but of the same size as if
      !  X11 had been enabled)
      ! => it is a standard X11 of 90 dpi (resolution must be constant,
      !    in order to obtain exactly the same EPS images on different
      !    machines...)
      ! => avoid a difference in EPS files according to X11 active
      !    or no_X11 (i.e. the NULL device = in batch process)

      ! X11_grxmxa, X11_grymxa are get via opcode 35
      ! (X11_XMARGIN, X11_YMARGIN are global variables)
      if( X11_grxmxa < 0 .or. X11_grymxa < 0 ) then
         ! something terribly wrong arose!
         print *, "(Muesli FGL:) EPS driver: internal error"
         print *, "              negative value for at least one device dimension!"
         pause "only for debugging purpose"
         stop
      end if

      ! First unit here is X11 pixels
      bbox(1:4) = [ -X11_XMARGIN, -X11_YMARGIN,                         &
                     X11_grxmxa+X11_XMARGIN, X11_grymxa+X11_YMARGIN ]
      ! Then unit is EPS dots, so comparison with internal coordinates
      ! may be done
      bbox(1:4) = bbox(1:4) / 0.072d0

      WIDTH  = X11_grxmxa / 0.072d0
      HEIGHT = X11_grymxa / 0.072d0

      stdout = chr(1:lchr) == "-"
      if( stdout ) then
         ! machine-dependent!
         unit = 6
      else
         ! grglun (get logical unit) is defined in 'sys' folder
         call grglun( unit )
      end if
      ibuf(1) = unit
      ! save in pgplot module (used by deflate_stream_to_a85)
      MF_EPS_UNIT = unit
      if( .not. stdout ) then
         open( unit=unit, file=chr(1:lchr), status='unknown', iostat=ier )
         if( ier /= 0 ) then
            msg = "Cannot open output file for an EPS plot: " // chr(:lchr)
            call grwarn( msg )
            ibuf(2) = 0
            call grflun( unit )
            return
         else
            inquire( unit=unit, name=chr )
            lchr = len_trim(chr)
            ibuf(2) = 1
            fname = chr(:lchr)
            lfname = lchr
         end if
      else
         ibuf(2) = 1
         fname = "-"
         lfname= 1
      end if
      state = 1
      ioerr = 0
      lobuf = 0
      lw = 1
      npts = 0
      lasti = int_not_defined
      lastj = int_not_defined

      write(unit,"(A)") "%!PS-Adobe-3.0 EPSF-3.0"

      ! on réserve la place (blancs) pour écrire la BBox ultérieurement
      write(unit,"(A)") "%%BoundingBox:                                  "

      ! This part (four lines in the EPS header) may be omitted, if the
      ! user prefers to gt an anonymous EPS file (useful to compare the
      ! content with another file).
      anonymous_eps = .false.
      call grgenv( "EPS_ANONYMOUS", value, L )
      if( L /= 0 ) then
         if( value == "1" ) then
            anonymous_eps = .true.
         end if
      end if

      if( .not. anonymous_eps ) then
         write( fig_number_str, "(I0)" ) FIG_NUMBER
         if( len_trim(PG_EXE_NAME) == 0  ) then
            if( len_trim(WIN_TITLE) == 0  ) then
               write(unit,"(A)") "%%Title: "                            &
                               // "figure " // trim(fig_number_str)
            else
               write(unit,"(A)") "%%Title: "                            &
                               // "figure " // trim(fig_number_str)     &
                               // ": " // trim(WIN_TITLE)
            end if
         else
            if( len_trim(WIN_TITLE) == 0  ) then
               write(unit,"(A)") "%%Title: " // trim(PG_EXE_NAME)       &
                               // " - figure " // trim(fig_number_str)
            else
               write(unit,"(A)") "%%Title: " // trim(PG_EXE_NAME)       &
                               // " - figure " // trim(fig_number_str)  &
                               // ": " // trim(WIN_TITLE)
            end if
         end if

         instr = mf_muesli_version
         L = index( instr, "_" )
         L1 = L + 1
         L2 = len_trim(instr)
         L = L - 1
         write(unit,"(A)") "%%Creator: Muesli Library " //              &
                         instr(1:L) // " (" // instr(L1:L2) // ")"

         call grdate( instr, L )
         if( L > 0 ) write(unit,"(A)") "%%CreationDate: " // instr(1:L)

         call gruser( instr, L )
#ifndef _WINDOWS
         status = gr_real_user( instr(1:L)//char(0), instr2, L )
         if( status == 1 ) then
            ! Succeeded: we got the real user name.
            ! The full username may contain some accented character: if the
            ! system is set as UTF-8, we have to convert it to iso-8859-15
            instr = utf8_to_latin_1( instr2(1:L) )
            ! Moreover, metadata of EPS file (lines beginning by "%%" must contain
            ! ASCII chars only, therefore we downgrade some accented vowels
            ! (restricted here to "É", "é", "È", "è", "ë", "ç", "ô", "î") to ASCII
            ! equivalent.
            do i = 1, len_trim(instr)
               c = instr(i:i)
               k = ichar(c)
               if( k < 128 ) cycle
               select case( k )
                  case( 200, 201 )          ! "È", "É"
                     instr(i:i) = char(69)  ! "E"
                  case( 231 )               ! "ç"
                     instr(i:i) = char(99)  ! "c"
                  case( 232, 233, 235 )     ! "è", "é", "ë"
                     instr(i:i) = char(101) ! "e"
                  case( 238 )               ! "î"
                     instr(i:i) = char(105) ! "i"
                  case( 244 )               ! "ô"
                     instr(i:i) = char(111) ! "o"
               end select
            end do
         else
            ! Failed: keep the unix username in 'instr'
         end if
#endif
         if( L > 0 ) write(unit,"(A)") "%%For: " // instr(1:L)
      end if

      write(unit,"(A)") "%%DocumentFonts: (atend)"
      write(unit,"(A)") "%%DocumentNeededFonts: (atend)"
      write(unit,"(A)") "%%LanguageLevel: 3"
      write(unit,"(A)") "%%EndComments"
      write(unit,"(A)") ""
      write(unit,"(A)") "%%BeginProlog"
      write(unit,"(A)") "/N /newpath load def"
      write(unit,"(A)") "/m /moveto load def"
      write(unit,"(A)") "/l /lineto load def"
      write(unit,"(A)") "/c /curveto load def"
      write(unit,"(A)") "/h /closepath load def"
      write(unit,"(A)") "/S /stroke load def"
      write(unit,"(A)") "/q /gsave load def"
      write(unit,"(A)") "/Q /grestore load def"
      write(unit,"(A)") "/d /setdash load def"
      write(unit,"(A)") "/j /setlinejoin load def"
      write(unit,"(A)") "/J /setlinecap load def"
      write(unit,"(A)") "/RC /rectclip load def"
      write(unit,"(A)") "/RF /rectfill load def"
      write(unit,"(A)") "/RS /rectstroke load def"

      write(unit,"(A)") "/LW {5 mul setlinewidth} bind def"
      write(unit,"(A)") "/BP {newpath moveto} bind def"
      write(unit,"(A)") "/LP /lineto load def"
      write(unit,"(A)") "/EP {lineto closepath eofill} bind def"
      write(unit,"(A)") "/MB {gsave translate MFAC dup scale 1 setlinewidth 2 setlinecap 0 setlinejoin newpath} bind def"
      write(unit,"(A)") "/ME /grestore load def"
      write(unit,"(A)") "/CC {0 360 arc stroke} bind def"
      write(unit,"(A)") "/FC {0 360 arc fill} bind def"
      write(unit,"(A)") "/G {1024 div setgray} bind def"
      write(unit,"(A)") "/K {3 -1 roll 1024 div 3 -1 roll 1024 div 3 -1 roll 1024 div setrgbcolor} bind def"

      write(unit,"(A)") "/FontSize 0 def"
      write(unit,"(A)") "/FMS {/FontSize exch store findfont [FontSize 0 0 FontSize 0 0] makefont setfont} bind def"
      write(unit,"(A)") "/reencode {exch dup where {pop load} {pop StandardEncoding} ifelse exch dup 3 1"
      write(unit,"(A)") "roll findfont dup length dict begin { 1 index /FID ne {def}{pop pop} ifelse }"
      write(unit,"(A)") "forall /Encoding exch def currentdict end definefont pop} bind def"
      write(unit,"(A)") "/isroman {findfont /CharStrings get /Agrave known} bind def"
      write(unit,"(A)") "/FMSR {3 1 roll 1 index dup isroman {reencode} {pop pop} ifelse exch FMS} bind def"
      write(unit,"(A)") "%%EndProlog"
      write(unit,"(A)") ""
      write(unit,"(A)") "%%BeginPageSetup"

      ! unusual case: we are writing white on black, so before any drawing
      ! we define a large black area
      if( BLACK_ON_WHITE == 0 ) then
         ! black background
         write(unit,"(A)") "N -1200 -1200 m 2400 -1200 l 2400 2400 l -2400 2400 l h fill"
         ! white drawing
         write(unit,"(A)") "1024 G"
      end if

      ! unit is milli-inches
      write(unit,"(A)") "0.072 0.072 scale"
      ! The only way to enlarge a clipping area is to restore the initial
      ! clipping state (very large, I suppose); so we save it here.
      if( COMMENTS_IN_EPS ) then
         write(unit,"(A)") 'clipsave % save the initial clipping state, i.e. "no clip"'
      else
         write(unit,"(A)") 'clipsave'
      end if

      write(unit,"(A)") "%%EndPageSetup"
      write(unit,"(A)") ""

      EPS_driver_font_begin = .true.

      return

      !-----------------------------------------------------------------
      case( CLOSE_DEV ) ! Close device.

      call flush_buffer()

      ! Fix for some EPS browsers (Okular, Evince, ...) which display a
      ! black figure if there is no 'showpage' command at the end of the
      ! EPS file.
      ! After checking some Ubuntu versions, this bug arise now only for
      ! empty EPS files. As a consequence, "showpage' is always added
      ! (whatever the environment variable MFPLOT_EPS_ADD_SHOWPAGE is) for
      ! empty file.
      if( EMPTY_FILE ) then
         write(unit,"(A)") ""
         if( COMMENTS_IN_EPS ) then
            write(unit,"(A)") "% The following 'showpage' operator is not needed for EPS files (see the PostScript Language Ref. Manual);"
            write(unit,"(A)") "% It is however added here only to fix a bug of some viewers (as Okular or Evince) which show a black page"
            write(unit,"(A)") "% for empty EPS, or fail to show anything!"
         end if
         write(unit,"(A)") "showpage"
      else
         call grgenv( "EPS_ADD_SHOWPAGE", value, L )
         if( L /= 0 ) then
            if( value == "1" ) then
               write(unit,"(A)") ""
               if( COMMENTS_IN_EPS ) then
                  write(unit,"(A)") "% The following 'showpage' operator is not needed for EPS files (see the PostScript Language Ref. Manual);"
                  write(unit,"(A)") "% It is however added here only to fix a bug of some viewers (as Okular or Evince) which show a black page,"
                  write(unit,"(A)") "% or fail to show anything!"
               end if
               write(unit,"(A)") "showpage"
            end if
         end if
      end if

      write(unit,"(A)") ""
      write(unit,"(A)") "%%Trailer"

      fonts_present = ""

      ! the integer codes for Std PS Fonts must be the same as found
      ! in the 'grparse_ps_font' routine.
      if( PS_Std_Fonts_presence(1) ) then
         fonts_present = trim(fonts_present) // " Helvetica"
      end if

      if( PS_Std_Fonts_presence(2) ) then
         fonts_present = trim(fonts_present) // " Helvetica-Bold"
      end if

      if( PS_Std_Fonts_presence(3) ) then
         fonts_present = trim(fonts_present) // " Times-Italic"
      end if

      if( PS_Std_Fonts_presence(4) ) then
         fonts_present = trim(fonts_present) // " Times-BoldItalic"
      end if

      if( PS_Std_Fonts_presence(5) ) then
         fonts_present = trim(fonts_present) // " Times-Roman"
      end if

      if( PS_Std_Fonts_presence(6) ) then
         fonts_present = trim(fonts_present) // " Times-Bold"
      end if

      if( PS_Std_Fonts_presence(7) ) then
         fonts_present = trim(fonts_present) // " Symbol"
      end if

      if( PS_Std_Fonts_presence(8) ) then
         write(unit,"(A)") "%%DocumentFonts:" // trim(fonts_present)    &
                         // " English157BT-Regular"
      else
         write(unit,"(A)") "%%DocumentFonts:" // trim(fonts_present)
      end if

      write(unit,"(A)") "%%DocumentNeededFonts:" // trim(fonts_present)

      write(unit,"(A)") "%%EOF"

      if( ioerr /= 0 ) then
         call grwarn( "(MFPLOT) EPS driver: Warning: Error writing file." )
         call grwarn( "         File may be incomplete: check for device full or quota exceeded." )
         call grwarn( "         Filename: " // fname(:lfname) )
      end if

      if( .not. stdout ) then

         close( unit, iostat=ioerr )
         if( ioerr /= 0 ) then
            call grwarn( "(MFPLOT) EPS driver: Error closing file '" // fname(:lfname) // "'" )
            state = 0
            return
         end if

         ! Re-open the file for writing the BBOX at the beginning.
         open( unit, file=fname(:lfname), access="direct",              &
               form="formatted", recl=8 )

         ! The BBOX must be multiplied by the same scale as used to draw
         ! in the EPS (cf. "0.072 0.072 scale" after the prolog).
         bbox(:) = bbox(:)*0.072d0
         irec_offset = 40/8
         irec_offset = irec_offset + 1
         write( unit, "(I8)", rec=irec_offset ) bbox(1)
         irec_offset = irec_offset + 1
         write( unit, "(I8)", rec=irec_offset ) bbox(2)
         irec_offset = irec_offset + 1
         write( unit, "(I8)", rec=irec_offset ) bbox(3)
         irec_offset = irec_offset + 1
         write( unit, "(I8)", rec=irec_offset ) bbox(4)
         close( unit )

      end if
      state = 0

      ! for the next EPS (important, if we want to insert the colormap inside
      ! the EPS image only if it is actually used)
      NEEDS_COLORMAP = .false.

      return

      !-----------------------------------------------------------------
      case( BEGIN_PICT ) ! Begin picture.
                         ! (called from grbpic)

      WIDTH  = ibuf(1)
      HEIGHT = ibuf(2)

      do nsym = 1, nb_MARK
         marker(nsym) = 0
      end do
      mfac = 0.0d0

      if( chr_save_to_be_printed ) then
         call flush_buffer()
         write(unit,"(A)") trim(chr_save)
         chr_save_to_be_printed = .false.
      end if
      init_ok = .true.

      return

      !-----------------------------------------------------------------
      case( DRAW_POLYLINE ) ! Draw polyline (multiple straight segments).

      if( lchr == 1 ) then

         if( chr == "s" ) then
            polyline_start = .true.
            in_polyline = .true.
            something_has_been_written = .false.
            instr = "N"
         else if( chr == "e" ) then
            ! Force to locate "S" at the beginning of the line, to make
            ! possible to remove it, in case of continuity with the next
            ! polyline...
            if( lobuf+28 > lobuf_max ) then
               call flush_buffer()
            end if
            ! Perhaps all the points have been removed due to one or many
            ! zero-length segment(s); in such a case, we cannot write "S"
            ! alone (this would cause problem with continuation process
            ! with the next segment).
            polyline_start = .false.
            in_polyline = .false.
            if( something_has_been_written ) then
               instr = "S"
            else
               return
            end if
         end if

      else

         i0 = ibuf(1)
         j0 = ibuf(2)
         i1 = ibuf(3)
         j1 = ibuf(4)

         ! Suppress zero-length continuation segment
         ! Should NOT be reported in pgline, because here the resolution
         ! of the device is low (w.r.t. double precision) and it can occur
         ! more frequently.
         if( i0 == i1 .and. j0 == j1 ) return

         if( in_polyline ) then

            if( polyline_start ) then
               bool_1 = i0 == lasti .and. j0 == lastj
               if( lobuf > 0 ) then
                  bool_2 = obuf(lobuf:lobuf) == "S"
               else
                  bool_2 = .false.
               end if
               if( bool_1 .and. bool_2 ) then
                  ! Last command was a polyline
                  lobuf = max( 0, lobuf-2 ) ! discard the previous "S" command
                  write( instr, "(2(I0,1X),A)" ) i1, j1, "l"
               else
                  write( instr, "(2(I0,1X),A,1X,2(I0,1X),A)" )          &
                         i0, j0, "m", i1, j1, "l"
               end if
               polyline_start = .false.
            else
               ! Some points may have been removed (presence of NaN),
               ! therefore we must test the continuity...
               if( i0 == lasti .and. j0 == lastj ) then
                  write( instr, "(2(I0,1X),A)" ) i1, j1, "l"
               else
                  write( instr, "(A,1X,2(I0,1X),A,1X,2(I0,1X),A)" )     &
                         "S", i0, j0, "m", i1, j1, "l"
               end if
            end if
            lasti = i1
            lastj = j1
            something_has_been_written = .true.

         else
            ! Standard case (not optimized) -- normally rarely called
            write( instr, "(A,1X,2(I0,1X),A,1X,2(I0,1X),A)" )           &
                   "N", i0, j0, "m", i1, j1, "l S"
         end if

      end if

      !-----------------------------------------------------------------
      case( END_PICT ) ! End picture.

      return

      !-----------------------------------------------------------------
      case( SELECT_COL_IND ) ! Select color index.

      ci = ibuf(1)
      if( BLACK_ON_WHITE == 0 ) then
         ! invert index 0 and 1
         if( ci == 0 ) then
            ci = 1
         else if( ci == 1 ) then
            ci = 0
         end if
      end if
      i_red   = nint(1024.0d0*rvalue(ci))
      i_green = nint(1024.0d0*gvalue(ci))
      i_blue  = nint(1024.0d0*bvalue(ci))
      if( i_red == i_green .and. i_red == i_blue ) then
         write( instr, "(I0,1X,A)" ) i_red, "G"
      else
         write( instr, "(3(I0,1X),A)" ) i_red, i_green, i_blue, "K"
      end if

      lasti = int_not_defined

      !-----------------------------------------------------------------
      case( FLUSH_BUF ) ! Flush buffer.

      call flush_buffer()
      return

      !-----------------------------------------------------------------
      case( SET_LINE_STYLE ) ! Set line style
                             ! (called only from grsls)

      line_style = ibuf(1)
      ! Treat the special case of inverted dashes
      if( line_style == 5 ) then
        line_style = 2
        dash_offset = -1
      else
        dash_offset = 0
      end if

      select case( line_style )
         case( 1 ) ! continuous (default)
            instr = "[] 0 d"
            dash_pattern_period = 0
         case( 2 ) ! dashed
            ! for linewidth=1 (default) pattern = [200 200]
            n1 = dash_length_1*sqrt(current_linewidth)
            if( dash_offset == -1 ) dash_offset = n1
            write( instr, "(A,I0,1X,I0,A,1X,I0,1X,A)" )                 &
                   "[", n1, n1, "]", dash_offset, "d"
            dash_pattern_period = n1 + n1
         case( 3 ) ! dotted-dashed
            ! for linewidth=1 (default) pattern = [200 120 21 120]
            n1 = dash_length_1*sqrt(current_linewidth)
            n2 = dash_length_2*sqrt(current_linewidth)
            n3 = dash_length_3*sqrt(current_linewidth)
            write( instr, "(A,I0,1X,I0,1X,I0,1X,I0,A,1X,I0,1X,A)" )     &
                   "[", n1, n2, n3, n2, "]", 0, "d"
            dash_pattern_period = n1 + n2 + n3 + n2
         case( 4 ) ! dotted
            ! for linewidth=1 (default) pattern = [21 120]
            n2 = dash_length_2*sqrt(current_linewidth)
            n3 = dash_length_3*sqrt(current_linewidth)
            write( instr, "(A,I0,1X,I0,A,1X,I0,1X,A)" )                 &
                   "[", n3, n2, "]", 0, "d"
            dash_pattern_period = n3 + n2
      end select

      !-----------------------------------------------------------------
      case( POLYGON_FILL ) ! Polygon fill.
                           ! (called from grfa)

      call flush_buffer()

      npts = ibuf(1)

      ! first point
      i0 = ibuf(2)
      j0 = ibuf(3)
      write( instr, "(2(I0,1X),A)" ) i0, j0, "BP"
      call fill_buffer()

      ! intermediate points
      do i = 2, npts - 1
         i0 = ibuf(2*i)
         j0 = ibuf(2*i+1)
         write( instr, "(2(I0,1X),A)" ) i0, j0, "LP"
         call fill_buffer()
      end do

      ! last point
      i0 = ibuf(2*npts)
      j0 = ibuf(2*npts+1)
      write( instr, "(2(I0,1X),A)" ) i0, j0, "EP"
      call fill_buffer()

      lasti = int_not_defined
      lastj = int_not_defined

      return

      !-----------------------------------------------------------------
      case( SET_COL_REPRES ) ! Set color representation.

      ci = ibuf(1)
      rvalue(ci) = rbuf(1)
      gvalue(ci) = rbuf(2)
      bvalue(ci) = rbuf(3)

      return

      !-----------------------------------------------------------------
      case( SET_LINE_WIDTH ) ! Set line width.

      lw = rbuf(1)

      current_linewidth = lw ! without the factor below

      ! fix line width to match X11 rendering (same apparent width when
      ! the scaling of the figures are the same)
      lw = lw*1.3333d0

      if( int(lw) == lw ) then
         write( instr, "(I0,1X,A)" ) int(lw), "LW"
      else
         call rm_trail_zeros_in_flt( lw, "(F8.3)", str1 )
         write( instr, "(A)" ) trim(str1) // " LW"
      end if

      lasti = int_not_defined

      !-----------------------------------------------------------------
      case( ESCAPE ) ! Escape. (called by the routine 'gresc')
                 !
                 ! It is not a good idea to use the output buffer for
                 ! commands sent via 'gresc': for example, if a comment hides
                 ! some commands, then the EPS file would become corrupted!

      lasti = int_not_defined

      call flush_buffer()
      write(unit,"(A)") trim(chr)

      return

      !-----------------------------------------------------------------
      case( RECT_FILL ) ! Rectangle Fill.
                 ! Called from grrect (via grrec0)

      i0 = ibuf(1)
      j0 = ibuf(2)
      i1 = ibuf(3)
      j1 = ibuf(4)

      ! RF shortcut for rectfill
      write( instr, "(4(I0,1X),A)") i0, j0, i1-i0, j1-j0, "RF"

      !-----------------------------------------------------------------
      case( RECT_DRAW ) ! Draw Rectangle.
                 ! Called from grrect (via grrec0)

      i0 = ibuf(1)
      j0 = ibuf(2)
      i1 = ibuf(3)
      j1 = ibuf(4)

      ! RS shortcut for rectstroke
      write( instr, "(4(I0,1X),A)") i0, j0, i1-i0, j1-j0, "RS"

      !-----------------------------------------------------------------
      case( SET_LINE_CAP_JOIN_STYLE ) ! Set Line Cap and Join Style.

      lc = ibuf(1)
      lj = ibuf(2)

      write( instr, "(2(I0,1X,A,1X))" ) lc, "j", lj, "J"

      !-----------------------------------------------------------------
      case( GET_COL_REPRES ) ! Query color representation.

      ci = ibuf(1)
      rbuf(1) = rvalue(ci)
      rbuf(2) = gvalue(ci)
      rbuf(3) = bvalue(ci)
      ibuf(1) = 3

      return

      !-----------------------------------------------------------------
      case( DRAW_BEZIER_SEGM ) ! Draw Bézier segment.

      ! RBUF(1) to RBUF(8) contain x-y device coords of the four
      ! control-points of one Bézier segment.
      ! They are converted to integer in variables :
      ! (I0,J0), (I1,J1), (I2,J2), (I3,J3)

      ! Warning: BBox update: no need... there is always a clipping around
      !          the axes.

      ! flush buffer if needed because the commands are going to be written
      ! directly in the PS file.
      call flush_buffer()

      i0 = ibuf(1)
      j0 = ibuf(2)
      i1 = ibuf(3)
      j1 = ibuf(4)
      i2 = ibuf(5)
      j2 = ibuf(6)
      i3 = ibuf(7)
      j3 = ibuf(8)

      write(unit,"(A)") "N"

      write( instr, "(2(I0,1X),A)" ) i0, j0, "m"
      write(unit,"(A)") trim(instr)

      write( instr, "(6(I0,1X),A)" ) i1, j1, i2, j2, i3, j3, "c"
      write(unit,"(A)") trim(instr)

      write(unit,"(A)") "S"

      return

      !-----------------------------------------------------------------
      case( GET_DEF_SIZE_VIRT ) ! Get the user or default size (in pixels)
         ! of a virtual X11 window. This is required to get exactly the same
         ! results when X11 is On or Off.
         ! This is not the X11 window size, but the max plotting area.

      X11_grxmxa  = ibuf(1)
      X11_grymxa  = ibuf(2)

      return

      !-----------------------------------------------------------------
      case( WRITE_COLORMAP_IF ) ! Write the colormap if needed

      call flush_buffer()

      NEEDS_COLORMAP   = ibuf(1) == 1
      COLORMAP_CI_LOW  = ibuf(2)
      COLORMAP_CI_HIGH = ibuf(3)
      COLORMAP_LENGTH = COLORMAP_CI_HIGH - COLORMAP_CI_LOW + 1

      if( .not. NEEDS_COLORMAP ) return

      write(unit,"(A)") "/colormap {"
      write(unit,"(A)") "  /Function <<"
      write(unit,"(A)") "    /FunctionType 0"
      ! must adapt the best bits_per_sample value; it depends on the length
      ! of the colormap.
      if( COLORMAP_LENGTH <= 16 ) then
         BITS_PER_SAMPLE = 4 ! 'F' codes 0 to 15
      else if( COLORMAP_LENGTH <= 256 ) then
         BITS_PER_SAMPLE = 8 ! 'FF' codes 0 to 255
      else if( COLORMAP_LENGTH <= 4096 ) then
         BITS_PER_SAMPLE = 12 ! 'FFF' codes 0 to 4096
      else
         print *, "(Muesli FGL:) The EPS driver supports only colormap length <= 4096."
         pause "only for debugging purpose"
         stop "(caused by the EPS driver)"
      end if
      write(unit,"(A,I0)") "    /BitsPerSample ", BITS_PER_SAMPLE
      write(unit,"(A,I0,A)") "    /Size [ ", COLORMAP_LENGTH, " ]"
      write(unit,"(A)") "    /Domain [ 0 1 ]"
      write(unit,"(A)") "    /Range [ 0 1 0 1 0 1 ]"
      write(unit,"(A)") "    /DataSource <"
      call write_colormap()
      write(unit,"(A)") "    >"
      write(unit,"(A)") "  >>"
      write(unit,"(A)") "} bind def"

      return

      !-----------------------------------------------------------------
      case( DRAW_MARKER ) ! Draw Marker.

      call flush_buffer()

      nsym = ibuf(1)
      ! Output code for this marker if necessary
      if( marker(nsym) == 0 ) then
         call flush_buffer()
         call insert_graph_marker( ioerr, nsym, nb_MARK, unit )
         marker(nsym) = 1
      end if
      ! Output scale factor
      if( rbuf(1) /= mfac ) then
         call flush_buffer()
         mfac = rbuf(1)
         call rm_trail_zeros_in_flt(  mfac, "(F10.3)", str1 )
         instr = "/MFAC " // trim(str1) // " def"
         write(unit,"(A)") trim(instr)
      end if
      ! Output an instruction to draw one marker
      i1 = ibuf(2)
      j1 = ibuf(3)
      ! no space between M and the following number:
      write( instr, "(2(I0,1X),A,I0)" ) i1, j1, "M", nsym

      lasti = int_not_defined

      !-----------------------------------------------------------------
      case( BITMAP_IMAGE ) ! Bitmap Image.

      call flush_buffer()

      n = ibuf(1)
      if( n == 0 ) then
         ! First: setup for image
         nxp = ibuf(2)
         nyp = ibuf(3)
         ! Set clipping region (rbuf(1:4))
         xorg = nint(rbuf(1))
         xlen = nint(rbuf(2) - rbuf(1))
         yorg = nint(rbuf(3))
         ylen = nint(rbuf(4) - rbuf(3))

         if( PRINTING_EPS ) then
            if( COMMENTS_IN_EPS ) then
               write(unit,"(A)") "%-- begin of bitmap image"
            end if
         end if
         write(unit,"(A)") "q"
         write(unit, "(4(I0,1X),A)" ) xorg, yorg, xlen, ylen, "RC"

         write(unit, "(A,1X,I0,1X,A)" ) "/picstr", nxp, "string def"
         write(unit,"(A)") "/inputf"
         write(unit,"(A)") "  currentfile"
         if( MF_DEFLATE_TO_A85 ) then
            write(unit,"(A)") "  /ASCII85Decode filter"
         else
            write(unit,"(A)") "  /ASCIIHexDecode filter"
         end if
         write(unit,"(A)") "  /FlateDecode filter"
         write(unit,"(A)") "def"
         ! current matrix transformation in rbuf(5:10)
         write(unit, "(I0,1X,I0,' 8 [',6(ES10.3,1X),']')" ) nxp, nyp, rbuf(5:10)
         write(unit,"(A)") "{ inputf picstr readstring not {"
         write(unit,"(A)") "  (ERROR: end of file -- incomplete data?) == flush stop } if"
         write(unit,"(A)") "} false 3 colorimage"
         string_out = ""
         len_out = len(string_out)

         ! image 24 bits -> 8 bits/composante
         !               -> 2 car. hexa -> Z2 ou 1 octet (integer*1)

         job = 1
         ! à l'initialisation, on a besoin de 'len_in' pour allouer une
         ! fois pour toute le tableau de compression 'compr_str'
         ! (allocation faite une seule fois car toutes les lignes d'une image
         !  ont la même largeur)
         len_in = len_in_max
         if( MF_DEFLATE_TO_A85 ) then
            fnameC = trim(fname) // char(0)
            lfnameC = lfname + 1
            close( unit ) ! because the following routine writes
                          ! directly in the EPS file.
            call deflate_stream_to_a85( fnameC, lfnameC,                &
                                        int_in, 0, job, ier )
         else
            call deflate_stream_to_hex( int_in, len_in,                 &
                                        string_out, len_out,            &
                                        job, ier )
         end if
         len_in = 0
      else if( n == -1 ) then
         ! Last: terminate image
         if( len_in /= 0 ) then
            ! on compresse et on écrit dans le PS
            job = 3
            len_out = len(string_out)
            if( MF_DEFLATE_TO_A85 ) then
               call deflate_stream_to_a85( "", 0,                       &
                                           int_in, len_in, job, ier )
               ! reopen the file, close at step 1.
               open( unit, file=trim(fname), position="append" )
            else
               call deflate_stream_to_hex( int_in, len_in,              &
                                           string_out, len_out,         &
                                           job, ier )
            end if
            if( ier /= 0 ) then
               write(STDERR,*) "(MFPLOT:) psdriv/26 -- writing image"
               write(STDERR,*) "          ERROR while deflating image"
               return
            end if
            if( .not. MF_DEFLATE_TO_A85 ) then
               L = 0
               ! on écrit la chaîne résultante de ZLIB par ligne de
               ! 132 caractères
               ! (c'est la largeur utilisée dans MFPLOT pour les EPS/PDF)
               do
                  if( L >= len_out ) exit
                  if( L+132 >= len_out ) then
                     write(unit,"(A)") string_out(L+1:len_out)
                     exit
                  else
                     write(unit,"(A)") string_out(L+1:L+132)
                     L = L + 132
                  end if
               end do
            end if
         end if
         if( MF_DEFLATE_TO_A85 ) then
            write(unit,"(A)") "~>" ! End-Of-Data for EPS (ASCII85)
         else
            write(unit,"(A)") ">" ! End-Of-Data for EPS (HEXA)
         end if
         write(unit,"(A)") "Q"
         LAST_COLOR_IS_VALID = .false.
         LAST_LINEWIDTH_IS_VALID = .false.
         LAST_LINESTYLE_IS_VALID = .false.
         LAST_FONT_ATTRIB_IS_VALID = .false.

         if( PRINTING_EPS ) then
            if( COMMENTS_IN_EPS ) then
               write(unit,"(A)") "%-- end of bitmap image"
            end if
         end if
      else
         ! this part may be called many times !
         ! Middle: write N image pixels.
         do i = 1, n
            ci = ibuf(i+1)
            rgb(1) = nint(255.0d0*rvalue(ci))
            rgb(2) = nint(255.0d0*gvalue(ci))
            rgb(3) = nint(255.0d0*bvalue(ci))

            if( len_in+3 > len_in_max ) then
               ! on compresse
               job = 2
               len_out = len(string_out)
               if( MF_DEFLATE_TO_A85 ) then
                  call deflate_stream_to_a85( "", 0,                    &
                                              int_in, len_in, job, ier )
               else
                  call deflate_stream_to_hex( int_in, len_in,           &
                                              string_out, len_out,      &
                                              job, ier )
               end if
               if( ier /= 0 ) then
                  write(STDERR,*) "(MFPLOT) EPS driver -- writing image"
                  write(STDERR,*) "         ERROR while deflating image"
                  print *
                  print *, " [Return to resume...]"
                  read *
                  return
               end if

               if( .not. MF_DEFLATE_TO_A85 ) then
                  L = 0
                  ! on écrit la chaîne résultante de ZLIB par ligne de
                  ! 132 caractères
                  ! (c'est la largeur utilisée dans MFPLOT pour les EPS/PDF)
                  do
                     if( L >= len_out ) exit
                     if( L+132 >= len_out ) then
                        write(unit,"(A)") string_out(L+1:len_out)
                        exit
                     else
                        write(unit,"(A)") string_out(L+1:L+132)
                        L = L + 132
                     end if
                  end do
               end if
               len_in = 0
            end if

            ! on remplit un buffer de 32 ko
            len_in = len_in + 1
            int_in(len_in) = rgb(1)
            len_in = len_in + 1
            int_in(len_in) = rgb(2)
            len_in = len_in + 1
            int_in(len_in) = rgb(3)

         end do

      end if

      return

      !-----------------------------------------------------------------
      case( ADD_COMMENT_EPS_PDF ) ! Add a comment in the EPS file.
                                  ! (uses ibuf(1) to take a decision)

!### TODO: to be added?
!!      call flush_buffer()

      if( ibuf(1) == 1 ) then
         ! Here, ibuf(1) is just a flag to tell the driver to execute
         ! immediately the writing of the comment or not -- this way is
         ! only for the EPS driver: it is called from print_eps() to
         ! write the colormap at the beginning of the file.
         write(unit,"(A)") trim(chr)
      else
         ! if initialization not yet done (cf. IFUNC=11), push the string
         ! in a buffer
!### TODO: why this? don't remember... to be investigated...
         if( .not. init_ok ) then
            chr_save = chr
            chr_save_to_be_printed = .true.
         else
            call flush_buffer()
            write(unit,"(A)") trim(chr)
         end if
      end if

      return

      !-----------------------------------------------------------------
      case( UPDATE_BBOX ) ! Update BBox.

      bbox(1:4) = ibuf(1:4)

      return

      !-----------------------------------------------------------------
      case( SELECT_BACK_FOREGROUND ) ! Select Back- and Foreground colors.

      if( ibuf(1) == 0 ) then
         ! white on black
         BLACK_ON_WHITE = 0
      else
         ! black on white (default)
         BLACK_ON_WHITE = 1
      end if

      return

      !-----------------------------------------------------------------
      case( GET_DASH_PERIOD ) ! Get the Dash Pattern Period.

      ibuf(1) = dash_pattern_period

      return

      !-----------------------------------------------------------------
      case( SET_CLIPPING ) ! Set Clipping
                 ! Do nothing (but this opcode must be present for
                 ! compatibility reasons with the X11 driver).

      return

      !-----------------------------------------------------------------
      case( GET_INFO_EMPTY_EPS ) ! Give the information about an empty EPS file.

      if( ibuf(1) == 1 ) then
         EMPTY_FILE = .true.
      end if

      return

      !-----------------------------------------------------------------
      case default ! Error: unimplemented opcode.

      write( msg, "('Unimplemented opcode in EPS device driver: ',I0)" ) ifunc
      call grwarn( msg )
      pause "only for debugging purpose"

      return

   end select

   call fill_buffer()

contains
!_______________________________________________________________________
!
   subroutine fill_buffer()

      instr = adjustl(instr) ! sometimes, space(s) preceed(s) the text
      L = len_trim(instr)

      if( lobuf+L+1 > lobuf_max ) then
         call flush_buffer()
         obuf(1:L) = instr(1:L)
         lobuf = L
      else
         if( lobuf >= 1 ) then
            ! insert a space to separate commands
            lobuf = lobuf + 1
            obuf(lobuf:lobuf) = " "
         end if
         obuf(lobuf+1:lobuf+L) = instr(1:L)
         lobuf = lobuf + L
      end if

   end subroutine fill_buffer
!_______________________________________________________________________
!
   subroutine flush_buffer()

      if( lobuf /= 0 ) then
         write(unit,"(A)") obuf(1:lobuf)
         lobuf = 0
      end if

   end subroutine flush_buffer
!_______________________________________________________________________
!
   subroutine write_colormap()
      integer :: i, k, col(3), v
      character(len=16) :: format
      !----------------
      select case( BITS_PER_SAMPLE )
         case( 4 )
            format = "(3(Z1.1))"
         case( 8 )
            format = "(3(Z2.2))"
         case( 12 )
            format = "(3(Z3.3))"
         case default
            print *, "(Muesli FGL:) internal error in the EPS driver."
            print *, "              Bad value for BITS_PER_SAMPLE (should be 4, 8 or 12)"
            print "(15X,A,I0)", "Current value of BITS_PER_SAMPLE = ", BITS_PER_SAMPLE
            pause "only for debugging purpose"
            stop
      end select
      v = 2**BITS_PER_SAMPLE - 1

      k = 0
      write( unit, "(6X)", advance="no" )
      do i = COLORMAP_CI_LOW, COLORMAP_CI_HIGH
         col(:) = [ nint(rvalue(i)*v), nint(gvalue(i)*v), nint(bvalue(i)*v) ]
         write( unit, format, advance="no" ) col(:)
         k = k + 6
         if( k >= 132 ) then
            write( unit, * )
            write( unit, "(6X)", advance="no" )
            k = 0
         end if
      end do
      write( unit, * )
   end subroutine write_colormap
!_______________________________________________________________________
!
end subroutine eps_driver
!_______________________________________________________________________
!
subroutine insert_graph_marker( ioerr, nsym, nsym_max, unit )

   implicit none

   integer, intent(in) :: ioerr, nsym, nsym_max, unit

   ! Write PostScript instructions for drawing graph marker number NSYM
   ! on Fortran unit UNIT.
   ! The following PostScript codes must be coherent with the PDF symbols'
   ! definitions in the XOBJ files (see src/drivers/PDFLibXObj).
   !-----------------------------------------------------------------------
   character(len=132) :: t(25)
   integer :: i, n

   if( nsym < 0 .or. nsym_max < nsym ) return

   select case( nsym )

      case( 1 ) ! Dot
         t(1) = "/M1 {MB 0 0 2.198 FC ME} bind def"
         n = 1
         go to 20

      case( 2 ) ! Plus
         t(1) = "/M2 {MB 1.099 6.813 m -1.099 6.813 l -1.099 1.099 l -6.813 1.099 l -6.813 -1.099 l -1.099 -1.099 l -1.099 -6.813 l 1.099 -6.813 l"
         t(2) = "1.099 -1.099 l 6.813 -1.099 l 6.813 1.099 l 1.099 1.099 l h fill ME} bind def"
         n = 2
         go to 20

      case( 3 ) ! Asterisk
         t(1) = "/M3 {MB 1.209 6.593 m -1.209 6.593 l -1.209 1.780 l -5.275 4.374 l -6.505 2.286 l -2.220 -0.088 l -6.505 -2.462 l -5.275 -4.549 l"
         t(2) = "-1.209 -1.956 l -1.209 -6.769 l 1.209 -6.769 l 1.209 -1.956 l 5.275 -4.549 l 6.505 -2.462 l 2.220 -0.088 l 6.505 2.286 l 5.275"
         t(3) = "4.374 l 1.209 1.780 l h fill ME} bind def"
         n = 3
         go to 20

      case( 4 ) ! X
         t(1) = "/M4 {MB 5.802 3.824 m 3.824 5.802 l 0.0 1.978 l -3.824 5.802 l -5.802 3.824 l -1.978 0.0 l -5.802 -3.824 l -3.824 -5.802 l 0.0"
         t(2) = "-1.978 l 3.824 -5.802 l 5.802 -3.824 l 1.978 0.0 l h fill S ME} bind def"
         n = 2
         go to 20

      case( 5 ) ! Hash
         t(1) = "/M5 {MB -4.616 2.637 m -4.616 -2.637 l -7.473 -2.637 l -7.473 -4.615 l -4.616 -4.615 l -4.616 -7.473 l -2.638 -7.473 l -2.638"
         t(2) = "-4.615 l 2.637 -4.615 l 2.637 -7.473 l 4.615 -7.473 l 4.615 -4.615 l 7.472 -4.615 l 7.472 -2.637 l 4.615 -2.637 l 4.615 2.637"
         t(3) = "l 7.472 2.637 l 7.472 4.615 l 4.615 4.615 l 4.615 7.473 l 2.637 7.473 l 2.637 4.615 l -2.638 4.615 l -2.638 7.473 l -4.616"
         t(4) = "7.473 l -4.616 4.615 l -7.473 4.615 l -7.473 2.637 l -2.638 2.637 m 2.637 2.637 l 2.637 -2.637 l -2.638"
         t(5) = "-2.637 l h fill ME} bind def"
         n = 5
         go to 20

      case( 6 ) ! Circle
         t(1) = "/M6 {MB 0 0 7.956 0 360 arc 0 0 5.978 360 0 arcn fill ME} bind def"
         n = 1
         go to 20

      case( 7 ) ! Square
         t(1) = "/M7 {MB 6.813 -6.813 m 6.813 6.813 l -6.813 6.813 l -6.813 -6.813 l -4.835 -4.835 m -4.835 4.835 l 4.835"
         t(2) = "4.835 l 4.835 -4.835 l h fill ME} bind def"
         n = 2
         go to 20

      case( 8 ) ! Diamond
         t(1) = "/M8 {MB 0.0 -8.791 m 8.791 0.0 l 0.0 8.791 l -8.791 0.0 l 0.0 -5.978 m -5.978 0.0 l 0.0 5.978 l 5.978 0.0"
         t(2) = "l h fill ME} bind def"
         n = 2
         go to 20

      case( 9 ) ! Star
         t(1) = "/M9 {MB 0.0 10.11 m -2.352 2.857 l -9.956 2.857 l -3.824 -1.714 l -6.176 -8.967 l -0.044 -4.396 l 6.088 -8.967 l 3.736 -1.714 l"
         t(2) = "9.868 2.857 l 2.264 2.857 l -0.022 4.813 m 1.187 1.341 l 5.077 1.341 l 1.956 -0.835 l 3.143 -4.308 l"
         t(3) = "0.022 -2.132 l -3.121 -4.308 l -1.912 -0.835 l -5.055 1.341 l -1.143 1.341 l h fill ME} bind def"
         n = 3
         go to 20

      case( 10 ) ! TriangleUp
         t(1) = "/M10 {MB 9.033 -5.165 m 0.0 10.352 l -9.033 -5.165 l 0.022 6.352 m 5.56 -3.187 l -5.56 -3.187 l h"
         t(2) = "fill ME} bind def"
         n = 2
         go to 20

      case( 11 ) ! TriangleDown
         t(1) = "/M11 {MB 0.0 -10.44 m 9.033 5.231 l -9.055 5.231 l -0.022 -6.505 m -5.648 3.253 l 5.626 3.253 l h"
         t(2) = "fill ME} bind def"
         n = 2
         go to 20

      case( 12 ) ! TriangleLeft
         t(1) = "/M12 {MB -10.440 0.0 m 5.231 -9.033 l 5.231 9.055 l -6.505 0.022 m 3.253 5.648 l 3.253 -5.626 l h"
         t(2) = "fill ME} bind def"
         n = 2
         go to 20

      case( 13 ) ! TriangleRight
         t(1) = "/M13 {MB 10.440 0.0 m -5.231 9.033 l -5.231 -9.055 l 6.505 -0.022 m -3.253 -5.648 l -3.253 5.626 l h"
         t(2) = "fill ME} bind def"
         n = 2
         go to 20

      case( 14 ) ! CircleFilled
         t(1) = "/M14 {MB 0 0 7.956 FC ME} bind def"
         n = 1
         go to 20

      case( 15 ) ! SquareFilled
         t(1) = "/M15 {MB 6.813 -6.813 m 6.813 6.813 l -6.813 6.813 l -6.813 -6.813 l h fill ME} bind def"
         n = 1
         go to 20

      case( 16 ) ! DiamondFilled
         t(1) = "/M16 {MB 0 -8.791 m 8.791 0 l 0 8.791 l -8.791 0 l h fill ME} bind def"
         n = 1
         go to 20

      case( 17 ) ! StarFilled
         t(1) = "/M17 {MB 0 9.011 m -2.132 2.418 l -9.055 2.418 l -3.473 -1.736 l -5.604 -8.330 l -0.022 -4.176 l 5.560 -8.33 l 3.429 -1.736 l"
         t(2) = "9.011 2.418 l 2.088 2.418 l h fill ME} bind def"
         n = 2
         go to 20

      case( 18 ) ! TriangleUpFilled
         t(1) = "/M18 {MB 9.033 -5.165 m 0.0 10.352 l -9.033 -5.165 l h fill ME} bind def"
         n = 1
         go to 20

      case( 19 ) ! TriangleDownFilled
         t(1) = "/M19 {MB 9.033 5.165 m -9.055 5.165 l -0.022 -10.352 l h fill ME} bind def"
         n = 1
         go to 20

      case( 20 ) ! TriangleLeftFilled
         t(1) = "/M20 {MB -10.352 0 m 5.165 -9.033 l 5.165 9.055 l h fill ME} bind def"
         n = 1
         go to 20

      case( 21 ) ! TriangleRightFilled
         t(1) = "/M21 {MB 10.352 0 m -5.165 9.033 l -5.165 -9.055 l h fill ME} bind def"
         n = 1
         go to 20

      case( 22 ) ! HeavyFourBalloon
         t(1) = "/M22 {MB 0.379 0.379 m 0.379 2.568 2.737 2.526 2.737 4.884 c 2.737 6.358 1.474 7.579 0.0 7.579 c -1.474 7.579 -2.737 6.358 -2.737"
         t(2) = "4.884 c -2.737 2.526 -0.379 2.568 -0.379 0.379 c -2.568 0.379 -2.547 2.737 -4.905 2.737 c -6.379 2.737 -7.579 1.474 -7.579 0.0 c"
         t(3) = "-7.579 -1.474 -6.358 -2.737 -4.884 -2.737 c -2.526 -2.737 -2.568 -0.379 -0.379 -0.379 c -0.379 -2.568 -2.737 -2.526 -2.737 -4.884"
         t(4) = "c -2.737 -6.358 -1.474 -7.579 0.0 -7.579 c 1.474 -7.579 2.737 -6.358 2.737 -4.884 c 2.737 -2.526 0.379 -2.568 0.379 -0.379 c"
         t(5) = "2.568 -0.379 2.526 -2.737 4.884 -2.737 c 6.358 -2.737 7.579 -1.474 7.579 0.0 c 7.579 1.474 6.358 2.737 4.884 2.737 c 2.526 2.737"
         t(6) = "2.568 0.379 0.379 0.379 c h fill ME} bind def"
         n = 6
         go to 20

      case( 23 ) ! HeavyTearDrop
         t(1) = "/M23 {MB 0.242 -0.374 m 1.187 -0.923 1.582 -1.319 3.055 -3.231 c 3.780 -4.198 4.308 -4.527 5.011 -4.527 c 5.604 -4.527 6.000"
         t(2) = "-4.352 6.659 -3.802 c 6.725 -3.473 6.747 -3.187 6.747 -2.989 c 6.747 -2.571 6.593 -2.132 6.286 -1.758 c 5.956 -1.319 5.385 -1.099"
         t(3) = "4.352 -0.967 c 2.022 -0.659 1.407 -0.505 0.484 0.022 c 1.363 0.549 2.066 0.725 4.352 1.033 c 5.253 1.143 5.868 1.363 6.176 1.692"
         t(4) = "c 6.527 2.066 6.747 2.593 6.747 3.055 c 6.747 3.253 6.725 3.538 6.659 3.868 c 6.000 4.418 5.604 4.593 5.011 4.593 c 4.308 4.593"
         t(5) = "3.780 4.264 3.055 3.297 c 1.626 1.429 1.165 0.945 0.242 0.418 c 0.242 1.473 0.418 2.132 1.319 4.286 c 1.582 4.901 1.714 5.429"
         t(6) = "1.714 5.846 c 1.714 6.220 1.495 6.681 1.187 7.033 c 0.901 7.319 0.637 7.473 0.022 7.692 c -0.484 7.516 -0.747 7.385 -0.967 7.187"
         t(7) = "c -1.385 6.835 -1.670 6.264 -1.670 5.846 c -1.670 5.429 -1.538 4.901 -1.275 4.286 c -0.396 2.154 -0.198 1.473 -0.198 0.418 c"
         t(8) = "-1.121 0.945 -1.582 1.429 -3.011 3.297 c -3.736 4.264 -4.264 4.593 -4.967 4.593 c -5.560 4.593 -5.956 4.418 -6.615 3.868 c -6.681"
         t(9) = "3.538 -6.703 3.253 -6.703 3.055 c -6.703 2.637 -6.549 2.198 -6.242 1.824 c -5.912 1.385 -5.341 1.165 -4.308 1.033 c -2.000 0.725"
         t(10) = "-1.341 0.549 -0.440 0.022 c -1.407 -0.527 -1.824 -0.637 -4.308 -0.967 c -5.209 -1.077 -5.824 -1.297 -6.132 -1.626 c -6.484 -2.000"
         t(11) = "-6.703 -2.527 -6.703 -2.989 c -6.703 -3.187 -6.681 -3.473 -6.615 -3.802 c -5.956 -4.352 -5.582 -4.527 -4.967 -4.527 c -4.242"
         t(12) = "-4.527 -3.736 -4.198 -3.011 -3.231 c -1.582 -1.363 -1.121 -0.879 -0.198 -0.374 c -0.198 -1.407 -0.396 -2.088 -1.275 -4.220 c"
         t(13) = "-1.538 -4.835 -1.670 -5.363 -1.670 -5.780 c -1.670 -6.154 -1.451 -6.615 -1.143 -6.967 c -0.857 -7.253 -0.593 -7.407 0.022 -7.626"
         t(14) = "c 0.527 -7.451 0.791 -7.319 1.011 -7.121 c 1.429 -6.769 1.714 -6.220 1.714 -5.780 c 1.714 -5.363 1.582 -4.835 1.319 -4.220 c 0.418"
         t(15) = "-2.066 0.242 -1.407 0.242 -0.374 c h fill ME} bind def"
         n = 15
         go to 20

      case( 24 ) ! WhiteFlorette
         t(1) = "/M24 {MB -1.978 2.637 m"
         t(2) = "-3.451 4.022 -4.242 4.440 -5.297 4.440 c -6.857 4.440 -8.132 3.165 -8.132 1.604 c -8.132 -0.176 -6.791 -1.341 -4.725 -1.341 c"
         t(3) = "-4.374 -1.341 -3.978 -1.319 -3.231 -1.209 c -4.286 -1.824 -4.769 -2.154 -5.209 -2.593 c -5.780 -3.187 -6.088 -3.934 -6.088 -4.681 c"
         t(4) = "-6.088 -6.264 -4.813 -7.538 -3.231 -7.538 c -2.264 -7.538 -1.363 -7.033 -0.791 -6.154 c -0.418 -5.560 -0.220 -4.967 0.044 -3.604 c"
         t(5) = "0.330 -4.923 0.505 -5.495 0.857 -6.088 c 1.407 -6.989 2.330 -7.538 3.341 -7.538 c 4.923 -7.538 6.198 -6.264 6.198 -4.681 c 6.198"
         t(6) = "-3.934 5.890 -3.187 5.297 -2.593 c 4.879 -2.154 4.396 -1.824 3.341 -1.209 c 4.088 -1.319 4.484 -1.341 4.835 -1.341 c 6.901 -1.341"
         t(7) = "8.242 -0.176 8.242 1.604 c 8.242 3.143 6.945 4.440 5.407 4.440 c 4.352 4.440 3.538 4.000 2.088 2.637 c 2.703 4.000 2.901 4.659"
         t(8) = "2.901 5.407 c 2.901 7.011 1.648 8.308 0.066 8.308 c -1.516 8.308 -2.791 7.011 -2.791 5.407 c -2.791 4.659 -2.593 4.000 -1.978 2.637"
         t(9) = "c -1.978 2.637 m -0.462 2.747 m -1.187 2.462 l -1.846 3.451 -2.044 3.978 -2.044 4.857 c -2.044 6.264 -1.165"
         t(10) = "7.275 0.066 7.275 c 1.275 7.275 2.154 6.264 2.154 4.857 c 2.154 3.978 1.956 3.429 1.297 2.462 c 0.571 2.747 l 0.264 4.835 l 0.242"
         t(11) = "5.033 0.154 5.121 0.044 5.121 c -0.066 5.121 -0.132 5.033 -0.154 4.835 c 2.154 1.846 m 2.527 2.330 2.703 2.505"
         t(12) = "2.923 2.703 c 3.560 3.231 4.396 3.560 5.143 3.560 c 6.286 3.560 7.209 2.659 7.209 1.538 c 7.209 0.220 5.956 -0.791 4.308 -0.791 c"
         t(13) = "3.912 -0.791 3.626 -0.747 2.923 -0.527 c 2.967 0.242 l 4.857 1.187 l 5.011 1.253 5.077 1.341 5.077 1.429 c 5.077 1.538 4.989 1.604"
         t(14) = "4.857 1.604 c 4.813 1.604 l 4.813 1.604 4.769 1.582 4.725 1.582 c 2.659 1.231 l 2.593 -1.538 m 3.341 -1.802 3.604"
         t(15) = "-1.934 3.978 -2.242 c 4.747 -2.857 5.253 -3.736 5.253 -4.527 c 5.253 -5.648 4.352 -6.549 3.231 -6.549 c 2.396 -6.549 1.626 -6.022"
         t(16) = "1.099 -5.099 c 0.725 -4.462 0.615 -4.022 0.593 -2.989 c 1.341 -2.791 l 2.813 -4.308 l 2.901 -4.396 2.989 -4.440 3.055 -4.440 c"
         t(17) = "3.143 -4.440 3.231 -4.352 3.231 -4.264 c 3.231 -4.242 3.209 -4.176 3.143 -4.066 c 2.176 -2.198 l -2.549 1.231 m"
         t(18) = "-4.615 1.582 l -4.725 1.604 -4.725 1.604 -4.747 1.604 c -4.879 1.604 -4.967 1.538 -4.967 1.429 c -4.967 1.341 -4.901 1.253 -4.747"
         t(19) = "1.187 c -2.857 0.242 l -2.813 -0.527 l -3.516 -0.747 -3.802 -0.791 -4.198 -0.791 c -5.868 -0.791 -7.121 0.220 -7.121 1.538 c"
         t(20) = "-7.121 2.659 -6.176 3.560 -5.033 3.560 c -4.330 3.560 -3.560 3.275 -2.923 2.791 c -2.637 2.571 -2.462 2.374 -2.044 1.846 c"
         t(21) = "-2.022 -2.044 m -3.033 -4.066 l -3.099 -4.176 -3.121 -4.220 -3.121 -4.264 c -3.121 -4.352 -3.033 -4.440 -2.945"
         t(22) = "-4.440 c -2.879 -4.440 -2.791 -4.374 -2.703 -4.308 c -1.231 -2.791 l -0.484 -2.989 l -0.505 -3.956 -0.593 -4.330 -0.879 -4.923 c"
         t(23) = "-1.407 -5.956 -2.220 -6.549 -3.121 -6.549 c -4.242 -6.549 -5.143 -5.648 -5.143 -4.527 c -5.143 -3.736 -4.659 -2.857 -3.868 -2.242"
         t(24) = "c -3.495 -1.934 -3.231 -1.802 -2.484 -1.538 c h fill ME} bind def"
         n = 24
         go to 20

      case( 25 ) ! Snowflake
         t(1) = "/M25 {MB -0.593 -1.055 m -0.593 -4.220 l -2.945 -5.582 l -2.945 -6.923 l -0.593 -5.560 l -0.593 -7.736 l 0.571 -7.736 l 0.571"
         t(2) = "-5.560 l 2.923 -6.923 l 2.923 -5.582 l 0.571 -4.220 l 0.571 -1.055 l 3.319 -2.637 l 3.319 -5.341 l 4.484 -6.022 l 4.484 -3.319 l"
         t(3) = "6.374 -4.418 l 6.967 -3.407 l 5.077 -2.308 l 7.429 -0.945 l 6.264 -0.264 l 3.912 -1.626 l 1.165 -0.044 l 3.912  1.538 l 6.264"
         t(4) = "0.198 l 7.429  0.879 l 5.077  2.242 l 6.967  3.341 l 6.374  4.352 l 4.484  3.253 l 4.484  5.956 l 3.319  5.275 l 3.319  2.571 l"
         t(5) = "0.571  0.989 l 0.571  4.154 l 2.923  5.516 l 2.923  6.857 l 0.571  5.516 l 0.571  7.692 l -0.593  7.692 l -0.593  5.516 l -2.945"
         t(6) = "6.857 l -2.945  5.516 l -0.593  4.176 l -0.593  1.011 l -3.341  2.593 l -3.341  5.297 l -4.505  5.978 l -4.505  3.275 l -6.396"
         t(7) = "4.374 l -6.989  3.363 l -5.099  2.264 l -7.451  0.901 l -6.286  0.220 l -3.934  1.582 l -1.187  0.000 l -3.934 -1.582 l -6.286"
         t(8) = "-0.242 l -7.451 -0.923 l -5.099 -2.264 l -6.989 -3.363 l -6.396 -4.374 l -4.505 -3.275 l -4.505 -5.978 l -3.341 -5.297 l -3.341"
         t(9) = "-2.593 l h fill ME} bind def"
         n = 9
         go to 20

      case( 26 ) ! BlackDiamondMinusWhiteX
         t(1) = "/M26 {MB 0.0 7.648 m -3.297 4.352 l 0.0 1.055 l 3.297 4.352 l h fill 7.692 -0.044 m 4.396 3.253 l 1.099 -0.044 l 4.396"
         t(2) = "-3.341 l h fill 0.000 -7.736 m 3.297 -4.440 l 0.000 -1.143 l -3.297 -4.440 l h fill -7.692 -0.044 m -4.396 -3.341"
         t(3) = "l -1.099 -0.044 l -4.396 3.253 l h fill ME} bind def"
         n = 3
         go to 20

      case default
         print *, "(Muesli FGL:) EPS driver: unknown marker number"
         print *, "              nsym = ", nsym
         return

   end select

20 continue
   do i = 1, n
      write(unit,"(A)") trim(t(i))
   end do

end subroutine insert_graph_marker
