! f90 include file

!_______________________________________________________________________
!
   subroutine TriStream( x, y, u, v, tc, x0, y0,                        &
                         npt_max, Ltol, dLtol, backward, n, coords,     &
                         status, stop_zone )

      real(kind=MF_DOUBLE), intent(in)              :: x(:), y(:), u(:), v(:)
      type(mfTriConnect)                            :: tc
      real(kind=MF_DOUBLE), intent(in)              :: x0, y0
      integer,              intent(in)              :: npt_max
      real(kind=MF_DOUBLE), intent(in)              :: Ltol, dLtol
      logical,              intent(in)              :: backward
      integer,                          intent(out) :: n
      real(kind=MF_DOUBLE), allocatable             :: coords(:,:)
      integer,                          intent(out) :: status

      interface
         logical function stop_zone( x, y )
            import MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in) :: x, y
         end function
      end interface
      !------ API end ------

      ! Note: only one streamline is processed here; the starting point
      ! (x0,y0) is therefore scalar, contrary to the original version.

      ! (u,v) components must not contain any NaN values. The calling
      ! routine must check such a property.

      ! Output: the array 'coords' contains the coordinates of the 'n'
      ! points of the streamline, one point for each row. The size of the
      ! 'coords' array is usually greater than 'n'.
      ! The 'status' integer concerns the result of the routine:
      !   0: normal termination.
      !  -1: normal termination, but indicates a cycling curve; this
      !      avoids a second integration over the same path.
      !  -2: streamline is empty because the starting point is outside
      !      the domain.
      !-----------------------------------------------------------------
      ! [07-10-22] Fix some pathological cases (stagnation point very far
      !            from the current triangle, but not at inifinity due to
      !            roundoff errors). For that, all tests involving MF_EPS
      !            have been multiply by 10, i.e. involves 100*MF_EPS
      !            instead of 10*MF_EPS.
      ! [25-04-24] Add a new feature: the points cannot belong to a stop
      !            zone. This is to avoid an error if the vector field
      !            contains some non finite values (NaN or Inf).
      ! [22-05-24] Fix a bug is the computation of the new point Q at
      !            lines 348-349; it was sometimes inside the triangle,
      !            but we are looking for an outside point. Introducing
      !            the magnitude of the velocity fixed the problem.
      !-----------------------------------------------------------------

      real(kind=MF_DOUBLE), parameter :: dt_max = 1.0d24

      real(kind=MF_DOUBLE) :: XP, YP
      integer :: tri_ind, r

      real(kind=MF_DOUBLE) :: xx(3), yy(3), uu(3), vv(3)
      real(kind=MF_DOUBLE) :: detA, A11, A12, A13, A21, A22, A23, A31, A32, A33
      real(kind=MF_DOUBLE) :: L(3), Lprev(3), a(3), b(3), c(3)
      real(kind=MF_DOUBLE) :: B11, B12, B13, B21, B22, B23, B31, B32, B33
      real(kind=MF_DOUBLE) :: dt_c(3), delta, q, root(2), dt_c_1, incr(3)
      real(kind=MF_DOUBLE) :: B1, B2, B3, Bnrm, dt_crv, dt, Xend, Yend, h

      real(kind=MF_DOUBLE), allocatable :: coords_tmp(:,:) ! for quickly extend 'coords' array
      integer :: current_coords_size, new_coords_size
      character(len=12) :: str
      logical :: new_triangle

      real(kind=MF_DOUBLE) :: B_mat(3,3), null_B_vec(3), null_B_sum
      real(kind=MF_DOUBLE) :: null_B_sum_1, null_B_sum_2
      real(kind=MF_DOUBLE) :: L_stagn_pt(3), L_intersect(3), L_1(3), L_2(3)
      logical :: stagn_pt, stagn_pt_inside_tri, stagn_line

      real(kind=MF_DOUBLE) :: sigma(2), ratio
      real(kind=MF_DOUBLE) :: XSL_1, YSL_1, XSL_2, YSL_2
      real(kind=MF_DOUBLE) :: xproj, yproj, Lproj(3)
      type(mfArray) :: null_B
      integer :: nb_im, i
      logical :: first_point, second_point
      real(kind=MF_DOUBLE) :: uP, vP, module_u, tmp, XQ, YQ, LQ(3),     &
                              magnitude

   !------ end of declarations -- execution starts hereafter  ------

      status = 0

      XP = x0
      YP = y0

      n = 1
      coords(1,1) = XP
      coords(1,2) = YP

      first_point = .true.
      second_point = .false.

      new_triangle = .true.

main_loop: do
         if( new_triangle ) then
            tri_ind = tsearch( tc%convex_domain, tc%n_xy, tc%tri_n,     &
                               tc%face_tri, tc%tri_f, tc%n_tri,         &
                               tc%face_n, tc%faces_boundary,            &
                               tc%faces_boundary_ptr,                   &
                               dble(XP), dble(YP) )

            if( tri_ind == 0 ) then
               ! pt is outside the domain
               if( first_point ) then
                  status = -2
                  return
               else
                  if( .not. second_point ) then
                     ! Take the intersection with the triangle boundary and exit
                     call bary_intersect( Lprev, L, 0.0d0, L_intersect )
                     L = L_intersect
                     XP = L(1)*xx(1) + L(2)*xx(2) + L(3)*xx(3)
                     YP = L(1)*yy(1) + L(2)*yy(2) + L(3)*yy(3)
                     coords(n,1) = XP
                     coords(n,2) = YP
                  end if
               end if
               exit main_loop
            end if

            xx(:) = x(tc%tri_n(tri_ind,:))
            yy(:) = y(tc%tri_n(tri_ind,:))
            uu(:) = u(tc%tri_n(tri_ind,:))
            vv(:) = v(tc%tri_n(tri_ind,:))
            if( backward ) then
               uu = -uu
               vv = -vv
            end if

            detA = yy(1)*(xx(3)-xx(2)) +                                &
                   yy(2)*(xx(1)-xx(3)) +                                &
                   yy(3)*(xx(2)-xx(1))

            ! Characteristic length of the triangle
            h = sqrt(detA)

            A11 = ( yy(2)-yy(3) )/detA
            A12 = ( xx(3)-xx(2) )/detA
            A13 = ( xx(2)*yy(3)-xx(3)*yy(2) )/detA
            A21 = ( yy(3)-yy(1) )/detA
            A22 = ( xx(1)-xx(3) )/detA
            A23 = ( xx(3)*yy(1)-xx(1)*yy(3) )/detA
            A31 = ( yy(1)-yy(2) )/detA
            A32 = ( xx(2)-xx(1) )/detA
            A33 = ( xx(1)*yy(2)-xx(2)*yy(1) )/detA

            L = [ A11*XP + A12*YP + A13,                                &
                  A21*XP + A22*YP + A23,                                &
                  A31*XP + A32*YP + A33 ]

            if( first_point ) then
               ! If the starting point is exactly on any boundary,
               ! it is preferable to go slightly inside
               if( any( L < 100*MF_EPS ) ) then
                  ! Compute velocity at current point
                  uP = L(1)*uu(1) + L(2)*uu(2) + L(3)*uu(3)
                  vP = L(1)*vv(1) + L(2)*vv(2) + L(3)*vv(3)
                  ! Normalize this vector
                  tmp = hypot( uP, vP ) ! = sqrt(uP**2+vP**2) without underflow
                  uP = uP / tmp
                  vP = vP / tmp
                  xP = xP + h*Ltol*uP
                  yP = yP + h*Ltol*vP
                  n = n + 1
                  coords(n,1) = XP
                  coords(n,2) = YP
                  second_point = .true.
               end if
!### TODO: bizarre, les deux lignes suivantes devraient être
!          avant le 'end if' précédent, non ? (pas sûr...)
               first_point = .false.
               cycle
            end if
            second_point = .false.

            B11 = A11*uu(1) + A12*vv(1)
            B12 = A11*uu(2) + A12*vv(2)
            B13 = A11*uu(3) + A12*vv(3)
            B21 = A21*uu(1) + A22*vv(1)
            B22 = A21*uu(2) + A22*vv(2)
            B23 = A21*uu(3) + A22*vv(3)
            B31 = A31*uu(1) + A32*vv(1)
            B32 = A31*uu(2) + A32*vv(2)
            B33 = A31*uu(3) + A32*vv(3)

            B1 = sum( abs([B11, B21, B31]) )
            B2 = sum( abs([B12, B22, B32]) )
            B3 = sum( abs([B13, B23, B33]) )
            Bnrm = maxval([B1, B2, B3])

            B_mat = reshape( [ B11, B21, B31, B12, B22, B32, B13, B23, B33 ], &
                             [ 3, 3 ] )

            call msAssign( null_B, mfNull(mf(B_mat)) )
            if( mfIsEmpty(null_B) ) then
               print *, "(Muesli:) TriStream: internal error: null(B) is empty!"
               print *, "  -> Theoretically, B is always rank deficient!"
               pause "only for debugging purpose"
               stop
            end if
            ! Check image dimension (nb of columns of null(B))
            nb_im = size(null_B,2) ! therefore, nb_im >= 1

            if( nb_im == 1 ) then

               stagn_line = .false.
               null_B_sum = mfSum(null_B)
               if( abs(null_B_sum) > 100*MF_EPS ) then
                  ! A stagnation point exists for the local flow
                  stagn_pt = .true.
                  call PrintMessage( "Streamline", "I",                 &
                                     "A stagnation point exists (possibly outside)" )
                  ! Is it inside the triangle?
                  L_stagn_pt = null_B / null_B_sum
                  if( all(0.0d0 <= L_stagn_pt) .and.                    &
                      all(L_stagn_pt <= 1.0d0) ) then
                     stagn_pt_inside_tri = .true.
                  else
                     stagn_pt_inside_tri = .false.
                  end if
               else
                  ! A stagnation point doesn't exist (or located at infinity)
                  stagn_pt = .false.
               end if

            else if( nb_im == 2 ) then

               ! The original algorithm may or may not work, it depends on
               ! the kind of flow
               stagn_pt = .false.
               null_B_sum_1 = mfSum(mfGet(null_B,MF_ALL,1))
               null_B_sum_2 = mfSum(mfGet(null_B,MF_ALL,2))

               if( abs(null_B_sum_1) > 100*MF_EPS .and.                 &
                   abs(null_B_sum_2) > 100*MF_EPS ) then
                  ! A stagnation line exists for the local flow
                  stagn_line = .true.
                  call PrintMessage( "Streamline", "I",                 &
                                     "A stagnation line exists (possibly outside)" )
                  L_1 = mfGet(null_B,MF_ALL,1) / null_B_sum_1
                  XSL_1 = L_1(1)*xx(1) + L_1(2)*xx(2) + L_1(3)*xx(3)
                  YSL_1 = L_1(1)*yy(1) + L_1(2)*yy(2) + L_1(3)*yy(3)
                  L_2 = mfGet(null_B,MF_ALL,2) / null_B_sum_2
                  XSL_2 = L_2(1)*xx(1) + L_2(2)*xx(2) + L_2(3)*xx(3)
                  YSL_2 = L_2(1)*yy(1) + L_2(2)*yy(2) + L_2(3)*yy(3)
               else
                  ! A stagnation line doesn't exist (or located at infinity)
                  stagn_line = .false.
               end if

            else ! nb_im == 3
               call PrintMessage( "Streamline", "I",                    &
                                  "All velocities are zero in the current triangle!" )
               exit main_loop

            end if

            if( stagn_pt ) then
               ! Original value for dt_crv
               dt_crv = dLtol/Bnrm
            else
               ! Scaling to avoid a great number of points for big meshes
               dt_crv = dLtol/Bnrm / h
            end if

         end if

         new_triangle = .false.

         c = L
         b = [ B11*c(1) + B12*c(2) + B13*c(3),                          &
               B21*c(1) + B22*c(2) + B23*c(3),                          &
               B31*c(1) + B32*c(2) + B33*c(3) ]
         a = [ B11*b(1) + B12*b(2) + B13*b(3),                          &
               B21*b(1) + B22*b(2) + B23*b(3),                          &
               B31*b(1) + B32*b(2) + B33*b(3) ]*0.5d0

         dt_c(1:3) = dt_max
         do r = 1, 3
            delta = b(r)**2 - 4.0d0*a(r)*c(r)
            if( delta > 0.0d0 ) then
               q = -0.5d0*( b(r) + sign(1.0d0,b(r))*sqrt(delta) )
               root = [ q/a(r), c(r)/q ]
               if( root(1) < 0.0d0 ) root(1) = dt_max
               if( root(2) < 0.0d0 ) root(2) = dt_max
               dt_c(r) = minval(root)
            end if
         end do

         dt_c_1 = (1.0d0+Ltol)*minval(dt_c)

         if( dt_c_1 > dt_max ) then
            call PrintMessage( "Streamline", "I",                       &
                               "Stagnation point detected!" )
            exit main_loop
         end if

         if( dt_crv < dt_c_1 ) then
            dt = dt_crv
         else
            dt = dt_c_1
         end if

         if( dt <= MF_EPS*100 * h ) then
            dt = Ltol * h
            dt = min( dt, dt_crv )
         end if

         sigma = mfSVD( mf(a).vc.mf(b) )
         ratio = sigma(2)/sigma(1)

!### TODO: ci-dessous, les tests ne sont à faire que s'il existe un pt de
!          stagnation, non ? en cours... (voir ci-dessous)

         ! Better test because it doesn't depend of dt
         if( ratio > 100**2*MF_EPS ) then
            ! Usual algorithm, like the original TriStream Matlab script
            Lprev = L
            L = dt*( a*dt + b ) + c
         else
            if( nb_im == 1 ) then
               ! Are we going toward the stagnation point?
               if( all_are_positive(-b/a) ) then
                  ! Too dangerous to remain on a pathological line...
                  call PrintMessage( "Streamline", "I",                 &
                                     "Current point is on, or very close to a", &
                                     "pathological line!" )
                  if( stagn_pt ) then
                     if( stagn_pt_inside_tri ) then
                        ! Go directly to the stagnation point
                        L = L_stagn_pt
                     else
                        if( maxval(abs(L_stagn_pt)) > 1.d3 ) then
                           ! Stagnation point is too far: direction is often
                           ! very badly computed!
                           ! Compute a new point Q, in the right direction
                           ! of (uu,vv)
                           magnitude = sqrt(uu(1)**2+vv(1)**2)
                           XQ = XP + 2*h*uu(1) / magnitude
                           YQ = YP + 2*h*vv(1) / magnitude
                           call cart_to_bary( XQ, YQ, LQ )
                           ! Find the intersection of LQ with an edge
                           call bary_intersect( L, LQ, Ltol, L_intersect )
                        else
                           ! Find the intersection with an edge
                           call bary_intersect( L, L_stagn_pt, Ltol, L_intersect )
                        end if
                        Lprev = L
                        L = L_intersect
                        new_triangle = .true. ! since intersection is found
                     end if
                  else
                     ! Go on, as usual
                     Lprev = L
                     L = dt*( a*dt + b ) + c
                  end if
               else
                  ! Go on, as usual
                  Lprev = L
                  L = dt*( a*dt + b ) + c
               end if
            else ! nb_im == 2
               if( all_are_positive(-b/a) ) then

                  ! Orthogonal projection (using classical coords) of the
                  ! current point onto the stagnation line
                  if( stagn_line ) then
                     call orthog_proj( XSL_1, ySL_1, XSL_2, YSL_2, xP, yP, &
                                       xproj, yproj )
                     ! Barycentric coords
                     Lproj = [ A11*xproj + A12*yproj + A13,                &
                              A21*xproj + A22*yproj + A23,                &
                              A31*xproj + A32*yproj + A33 ]
                     if( all(0.0d0 <= Lproj) .and. all(Lproj <= 1.0d0) ) then
                        L = Lproj
                     else
                        ! Find the intersection with an edge
                        call bary_intersect( L, Lproj, Ltol, L_intersect )
                        Lprev = L
                        L = L_intersect
                        new_triangle = .true. ! since intersection is found
                     end if
                  else
                     ! Go on, as usual
                     Lprev = L
                     L = dt*( a*dt + b ) + c
                  end if

               else
                  ! Go on, as usual
                  Lprev = L
                  L = dt*( a*dt + b ) + c
               end if
            end if
         end if

         if( any(L > 1.0d0) .or. any(L < 0.0d0) ) then
            new_triangle = .true.
         end if

         XP = L(1)*xx(1) + L(2)*xx(2) + L(3)*xx(3)
         YP = L(1)*yy(1) + L(2)*yy(2) + L(3)*yy(3)

         ! Memory check
         current_coords_size = size(coords,1)
         if( n+2 > current_coords_size ) then
            ! Must increase size of coords
            new_coords_size = 2*current_coords_size
            ! Limitation from the user
            if( new_coords_size > npt_max+2 ) then
               new_coords_size = npt_max+2
            end if
            allocate( coords_tmp(new_coords_size,2) )
            write( str, "(I0)" ) new_coords_size
            call PrintMessage( "Streamline", "I",                       &
                               "Dynamic allocation: doubling the number", &
                               "of points for the current streamline!", &
                               "(max nb of points is now " // trim(str) // ")" )
            coords_tmp(1:current_coords_size,:) = coords(:,:)
            call move_alloc( from=coords_tmp, to=coords )
         end if
         ! Storing the new point (unless we reach a stop zone!)
         if( stop_zone(XP,YP) ) then
            exit main_loop
         end if
         n = n + 1
         coords(n,1) = XP
         coords(n,2) = YP

         if( hypot( coords(1,1)-XP,   coords(1,2)-YP ) <                &
             hypot( coords(n-1,1)-XP, coords(n-1,2)-YP ) ) then
            n = n + 1
            coords(n,1) = coords(1,1)
            coords(n,2) = coords(1,2)
            call PrintMessage( "Streamline", "I",                       &
                               "Cycling path detected!" )
            status = -1
            exit main_loop
         end if

         if( stagn_pt ) then
            if( mfDble(mfNorm(mf(L-L_stagn_pt))) < 0.1*Ltol ) then
               call PrintMessage( "Streamline", "I",                    &
                                  "Stagnation point detected!",         &
                                  "(reason: L and L_stagn_pt are too close)" )
               exit main_loop
            end if
         end if

         if( n >= npt_max ) then
            call PrintMessage( "Streamline", "W",                       &
                               "Dynamic allocation: max number of points for", &
                               "the current streamline is reached!" )
            exit main_loop
         end if

      end do main_loop

      call msSilentRelease( null_B )

   contains ! Internal subroutines
   !____________________________________________________________________
   !
      subroutine cart_to_bary( xP, yP, LP )

         real(kind=MF_DOUBLE), intent(in)  :: xP, yP
         real(kind=MF_DOUBLE), intent(out) :: LP(3)

         ! Given cartesian coordinates of a point P (xP,yP), computes
         ! the normalized barycentric coordinates in LP(3).

         ! Coefficients are already computed in the main subroutine.

         LP(1) = A11*xP + A12*yP + A13

         LP(2) = A21*xP + A22*yP + A23

         LP(3) = 1.0d0 - LP(1) - LP(2)

      end subroutine cart_to_bary
   !____________________________________________________________________
   !
      subroutine bary_intersect( LP, LS, Ltol, LI )

         real(kind=MF_DOUBLE), intent(in)  :: LP(3), LS(3)
         real(kind=MF_DOUBLE), intent(in)  :: Ltol
         real(kind=MF_DOUBLE), intent(out) :: LI(3)

         ! Given a point P inside the triangle, and another point S outside
         ! the same triangle, compute the location of the intersection I.
         ! If Ltol is not equal to zero, then the intersection is located
         ! slightly 'outside' the triangle.
         !
         ! The calling routine must ensure that S is outside!
         !
         ! Computation uses barycentric coordinates.

         type(mfArray) :: mfInd

         real(kind=MF_DOUBLE) :: lambda
         integer :: i, j

         call msAssign( mfInd, mfFind(mf(LS)<0.0d0) )

         if( size(mfInd) == 1 ) then
            select case( mfInt(mfInd) )
               case(1)
                  lambda = LP(1) / ( LP(1) - LS(1) ) + Ltol*h
                  LI(1) = LP(1) + lambda*( LS(1) - LP(1) )
                  LI(2) = LP(2) + lambda*( LS(2) - LP(2) )
                  LI(3) = 1.0d0 - LI(2) - LI(1)
               case(2)
                  lambda = LP(2) / ( LP(2) - LS(2) ) + Ltol*h
                  LI(2) = LP(2) + lambda*( LS(2) - LP(2) )
                  LI(3) = LP(3) + lambda*( LS(3) - LP(3) )
                  LI(1) = 1.0d0 - LI(3) - LI(2)
               case(3)
                  lambda = LP(3) / ( LP(3) - LS(3) ) + Ltol*h
                  LI(3) = LP(3) + lambda*( LS(3) - LP(3) )
                  LI(1) = LP(1) + lambda*( LS(1) - LP(1) )
                  LI(2) = 1.0d0 - LI(1) - LI(3)
            end select
         else if( size(mfInd) == 2 ) then
            do j = 1, 2
               i = mfGet(mfInd,j)
               ! Testing side #i
               select case( i )
                  case(1)
                     lambda = LP(1) / ( LP(1) - LS(1) )
                     LI(2) = LP(2) + lambda*( LS(2) - LP(2) )
                     if( LI(2) < 0.0d0 ) cycle
                     if( LI(2) > 1.0d0 ) cycle
                  case(2)
                     lambda = LP(2) / ( LP(2) - LS(2) )
                     LI(3) = LP(3) + lambda*( LS(3) - LP(3) )
                     if( LI(3) < 0.0d0 ) cycle
                     if( LI(3) > 1.0d0 ) cycle
                  case(3)
                     lambda = LP(3) / ( LP(3) - LS(3) )
                     LI(1) = LP(1) + lambda*( LS(1) - LP(1) )
                     if( LI(1) < 0.0d0 ) cycle
                     if( LI(1) > 1.0d0 ) cycle
               end select
               exit ! to avoid the second side
            end do
            ! Now that we know which side is concerned, we can use Ltol
            ! to shift the point slightly outside the triangle
            select case( i )
               case(1)
                  lambda = LP(1) / ( LP(1) - LS(1) ) + Ltol*h
                  LI(1) = LP(1) + lambda*( LS(1) - LP(1) )
                  LI(2) = LP(2) + lambda*( LS(2) - LP(2) )
                  LI(3) = 1.0d0 - LI(2) - LI(1)
               case(2)
                  lambda = LP(2) / ( LP(2) - LS(2) ) + Ltol*h
                  LI(2) = LP(2) + lambda*( LS(2) - LP(2) )
                  LI(3) = LP(3) + lambda*( LS(3) - LP(3) )
                  LI(1) = 1.0d0 - LI(3) - LI(2)
               case(3)
                  lambda = LP(3) / ( LP(3) - LS(3) ) + Ltol*h
                  LI(3) = LP(3) + lambda*( LS(3) - LP(3) )
                  LI(1) = LP(1) + lambda*( LS(1) - LP(1) )
                  LI(2) = 1.0d0 - LI(1) - LI(3)
            end select
         else ! size(mfInd) = 0
            print *, "TriStreamline: bary_intersect: internal error"
            print *, "   Point S is inside the triangle. No intersection!"
            pause "for debugging purpose only..."
            stop
         end if

         call msRelease( mfInd )

      end subroutine bary_intersect
   !____________________________________________________________________
   !
      subroutine orthog_proj( xM, yM, xM2, yM2, xC, yC, xP, yP )

         real(kind=MF_DOUBLE), intent(in)  :: xM, yM, xM2, yM2, xC, yC
         real(kind=MF_DOUBLE), intent(out) :: xP, yP

         ! Orthogonal projection of the point C on the straight line
         ! defined by the points M and M2.
         !
         ! The result is the point P.

         real(kind=MF_DOUBLE) :: v1, v2, v1sq, v2sq
         real(kind=MF_DOUBLE) :: tmp, tmp1, tmp2

         ! Compute the orientation of M-M2
         v1 = xM2 - xM
         v2 = yM2 - yM
         tmp1 = abs(v1/xM)
         tmp2 = abs(v2/yM)

         if( tmp1 < 10*MF_EPS ) then
            if( tmp2 < 10*MF_EPS ) then
               print *, "(Muesli) TriStream: internal error: M and M2 are the same point!"
               print *, "  -> To make a valid orthogonal projection, M and M2 must be distinct."
               pause "only for debugging purpose"
               stop
            else
               xP = xM
               yP = yC
            end if
         else
            if( tmp2 < 10*MF_EPS ) then
               xP = xC
               yP = yM
            else
               tmp = hypot( v1, v2 ) ! = sqrt( v1**2 + v2**2 )
                                     ! but without over- or underflow
               v1 = v1 / tmp
               v2 = v2 / tmp
               v1sq = v1**2
               v2sq = v2**2

               xP = v1sq*xC + v1*v2*(yC-yM) + v2sq*xM
               yP = v1sq*yM + v1*v2*(xC-xM) + v2sq*yC
            end if
         end if

      end subroutine orthog_proj
   !____________________________________________________________________
   !
      function all_are_positive( tab ) result( bool )

         real(kind=MF_DOUBLE), intent(in) :: tab(3)
         logical                          :: bool

         ! From Gfortran 9, all fonctions related to min and max are not
         ! guaranted to return a coherent result if a NaN is encountered.

         ! This function takes into account NaN values: it returns TRUE
         ! if all components of tab(:) are positive and not equal to NaN.
         !
         ! It is supposed to be more reliable than the expression:
         !                   all( tab > 0 )

         bool = .false.

         if( isnan( tab(1) ) ) return
         if( isnan( tab(2) ) ) return
         if( isnan( tab(3) ) ) return

         if( tab(1) <= 0.0 ) return
         if( tab(2) <= 0.0 ) return
         if( tab(3) <= 0.0 ) return

         bool = .true.

      end function all_are_positive
   !____________________________________________________________________
   !
   end subroutine TriStream
