/*
 * Part of MUESLI Numerical Library
 * Copyright É. Canot 2003-2025 -- IPR/CNRS
 */

/*
 * Floating-Point Exceptions Handling
 */

/* these constants must match those defined in the 'mod_core' f90 module */
#define OVERFLOW_EXC  1
#define DIVBYZERO_EXC 2
#define INVALID_EXC   3
#define USUAL_EXC     4
#define UNDERFLOW_EXC 5

/*====================================================================*/
/*
 *   --- GNU Linux: at least glibc 2.2 is required ---
 */
#define _GNU_SOURCE
#include <fenv.h>

/*-------------------------------------------*/

/* C <-> Fortran interoperability */
#ifdef UNDERSCORE
#define get_fpu_flags get_fpu_flags_
#endif
int get_fpu_flags ()
{
   return( fegetexcept() );
}

/* C <-> Fortran interoperability */
#ifdef UNDERSCORE
#define enable_fpu_flags enable_fpu_flags_
#endif
void enable_fpu_flags ( int *fpu_flags )
{
   feenableexcept( *fpu_flags );
}

/*-------------------------------------------*/

/* C <-> Fortran interoperability */
#ifdef UNDERSCORE
#define disable_all_FPE disable_all_fpe_
#else
#define disable_all_FPE disable_all_fpe
#endif
void disable_all_FPE ( )
{
   fedisableexcept( FE_ALL_EXCEPT );
}

/* C <-> Fortran interoperability */
#ifdef UNDERSCORE
#define clear_all_FPE clear_all_fpe_
#else
#define clear_all_FPE clear_all_fpe
#endif
void clear_all_FPE ( )
{
  feclearexcept( FE_ALL_EXCEPT );
}

/*-------------------------------------------*/

/* C <-> Fortran interoperability */
#ifdef UNDERSCORE
#define enableFPE enablefpe_
#else
#define enableFPE enablefpe
#endif
void enableFPE ( int *exception )
{
  switch (*exception) {
    case OVERFLOW_EXC :
      feenableexcept( FE_OVERFLOW );
      break;
    case DIVBYZERO_EXC :
      feenableexcept( FE_DIVBYZERO );
      break;
    case INVALID_EXC :
      feenableexcept( FE_INVALID );
      break;
    case USUAL_EXC :
      feenableexcept( FE_OVERFLOW | FE_DIVBYZERO | FE_INVALID );
      break;
    case UNDERFLOW_EXC :
      feenableexcept( FE_UNDERFLOW );
      break;
  }
}

/* C <-> Fortran interoperability */
#ifdef UNDERSCORE
#define disableFPE disablefpe_
#else
#define disableFPE disablefpe
#endif
void disableFPE ( int *exception )
{
  switch (*exception) {
    case OVERFLOW_EXC :
      fedisableexcept( FE_OVERFLOW );
      break;
    case DIVBYZERO_EXC :
      fedisableexcept( FE_DIVBYZERO );
      break;
    case INVALID_EXC :
      fedisableexcept( FE_INVALID );
      break;
    case USUAL_EXC :
      fedisableexcept( FE_OVERFLOW | FE_DIVBYZERO | FE_INVALID );
      break;
    case UNDERFLOW_EXC :
      fedisableexcept( FE_UNDERFLOW );
      break;
  }
}

/*====================================================================*/
