module int_funs

   use fml

   implicit none

contains

   function fun_sqrt( x ) result( res )
      real(kind=MF_DOUBLE), intent(in) :: x
      real(kind=MF_DOUBLE) :: res
      res = sqrt( x )
   end function fun_sqrt

   function fun_log( x ) result( res )
      real(kind=MF_DOUBLE), intent(in) :: x
      real(kind=MF_DOUBLE) :: res
      res = log( x )
   end function fun_log

   function fun_sin( x ) result( res )
      real(kind=MF_DOUBLE), intent(in) :: x
      real(kind=MF_DOUBLE) :: res
      res = sin( x )
   end function fun_sin

   function fun_one_over_x2( x ) result( res )
      real(kind=MF_DOUBLE), intent(in) :: x
      real(kind=MF_DOUBLE) :: res
      res = 1.0d0 / x**2
   end function fun_one_over_x2

   function fun_x_times_gaussian( x ) result( res )
      real(kind=MF_DOUBLE), intent(in) :: x
      real(kind=MF_DOUBLE) :: res
      res = x * exp( -x**2 )
   end function fun_x_times_gaussian

   function fun_which_gives_pi( x ) result( res )
      real(kind=MF_DOUBLE), intent(in) :: x
      real(kind=MF_DOUBLE) :: res
      res = 1.0d0 / (x+1.0d0) / sqrt(x)
   end function fun_which_gives_pi

   function fun_2_lin( x, y ) result( res )
      real(kind=MF_DOUBLE), intent(in) :: x, y
      real(kind=MF_DOUBLE) :: res
      res = x + y
   end function fun_2_lin

   function fun_2_exp( x, y ) result( res )
      real(kind=MF_DOUBLE), intent(in) :: x, y
      real(kind=MF_DOUBLE) :: res
      res = exp( -x**2 - y**2 )
   end function fun_2_exp

   function fun_2_sqrt( x, y ) result( res )
      real(kind=MF_DOUBLE), intent(in) :: x, y
      real(kind=MF_DOUBLE) :: res
      res = sqrt( x**2 + y**2 )
   end function fun_2_sqrt

   function fun_2_var( x, y ) result( res )
      real(kind=MF_DOUBLE), intent(in) :: x, y
      real(kind=MF_DOUBLE) :: res
      res = 5.0d0*x**3 * cos(y**3)
   end function fun_2_var

   function fun_xa( y ) result( res )
      real(kind=MF_DOUBLE), intent(in) :: y
      real(kind=MF_DOUBLE) :: res
      res = 0.0d0
   end function fun_xa

   function fun_xb( y ) result( res )
      real(kind=MF_DOUBLE), intent(in) :: y
      real(kind=MF_DOUBLE) :: res
      res = 2.0d0*sqrt(y)
   end function fun_xb

   function fun_ya( x ) result( res )
      real(kind=MF_DOUBLE), intent(in) :: x
      real(kind=MF_DOUBLE) :: res
      res = 0.25d0*x**2
   end function fun_ya

   function fun_yb( x ) result( res )
      real(kind=MF_DOUBLE), intent(in) :: x
      real(kind=MF_DOUBLE) :: res
      res = 2.0d0
   end function fun_yb

end module int_funs
!_______________________________________________________________________
!
module ode_funs

   use fml

   implicit none

contains

   subroutine deriv( t, y, yprime, flag )
      real(kind=MF_DOUBLE), intent(in)     :: t, y(*)
      real(kind=MF_DOUBLE), intent(out)    :: yprime(*)
      integer,              intent(in out) :: flag
      yprime(1) =  y(2)
      yprime(2) = -y(1)
   end subroutine deriv

   subroutine jac( t, y, jacobian, nrow )
      real(kind=MF_DOUBLE), intent(in)  :: t, y(*)
      integer,              intent(in)  :: nrow
      real(kind=MF_DOUBLE), intent(out) :: jacobian(nrow,*)
      ! only non zero elements need to be set
      jacobian(1,2) =  1.0d0
      jacobian(2,1) = -1.0d0
   end subroutine jac

   subroutine spjac( t, y, nrow, job, pd, ipd, jpd, nnz )
      real(kind=MF_DOUBLE), intent(in)     :: t, y(*)
      integer,              intent(in)     :: nrow, job
      real(kind=MF_DOUBLE), intent(out)    :: pd(*)
      integer,              intent(out)    :: ipd(*), jpd(*)
      integer,              intent(in out) :: nnz
      ! two kinds of call :
      ! - job=0 : initialization (i.e. allocation of vectors)
      !     -> only nnz is returned
      ! - job/=0 : subsequent standard calls (i.e. return all vectors)

      ! nnz = 2 should be sufficient, but we must assign entries
      !       for the diagonal elements (even if they are null)
      !
      ! row entries (in ipd) must be sorted in ascending order
      nnz = 4
      if( job == 0 ) then
         return
      end if
      PD(1:nnz) = [ 0.0d0, -1.0d0, 1.0d0, 0.0d0 ]
      IPD(1:nnz) = [ 1, 2, 1, 2 ]
      JPD(1:nrow+1) = [ 1, 3, nnz+1 ]
   end subroutine spjac

   subroutine deriv_chemical( t, y, yprime, flag )
      real(kind=MF_DOUBLE), intent(in)     :: t, y(*)
      real(kind=MF_DOUBLE), intent(out)    :: yprime(*)
      integer,              intent(in out) :: flag
      yprime(1) = -0.04d0*y(1) + 1.0d4*y(2)*y(3)
      yprime(3) = 3.0d7*y(2)**2
      yprime(2) = -yprime(1) - yprime(3)
   end subroutine deriv_chemical

   subroutine deriv_illegal_condition( t, y, yprime, flag )
      real(kind=MF_DOUBLE), intent(in)     :: t, y(*)
      real(kind=MF_DOUBLE), intent(out)    :: yprime(*)
      integer,              intent(in out) :: flag
      !
      ! Regular ODE. Solution is: y(t) = t**2
      !
      if( y(1) > 1.0d0 ) then
         ! Illegal condition
         flag = -1
         return
      end if
      yprime(1) = 2.0d0*t
   end subroutine deriv_illegal_condition

   subroutine deriv_emergency_exit( t, y, yprime, flag )
      real(kind=MF_DOUBLE), intent(in)     :: t, y(*)
      real(kind=MF_DOUBLE), intent(out)    :: yprime(*)
      integer,              intent(in out) :: flag
      !
      ! Regular ODE. Solution is: y(t) = t**2
      !
      if( y(1) > 1.0d0 ) then
         ! Emergency exit
         flag = -2
         return
      end if
      yprime(1) = 2.0d0*t
   end subroutine deriv_emergency_exit

   subroutine deriv_emergency_exit_2( t, y, yprime, flag )
      real(kind=MF_DOUBLE), intent(in)     :: t, y(*)
      real(kind=MF_DOUBLE), intent(out)    :: yprime(*)
      integer,              intent(in out) :: flag
      !
      ! Regular ODE. Solution is: y(t) = t**(3/2)
      !
      if( y(1) > 1.0d0 ) then
         ! Emergency exit
         flag = -2
         return
      end if
      yprime(1) = 1.5d0 * sqrt(t)
   end subroutine deriv_emergency_exit_2

   subroutine deriv_end_condition( t, y, yprime, flag )
      real(kind=MF_DOUBLE), intent(in)     :: t, y(*)
      real(kind=MF_DOUBLE), intent(out)    :: yprime(*)
      integer,              intent(in out) :: flag
      !
      ! Regular ODE. Solution is: y(t) = t**2
      !
      yprime(1) = 2.0d0*t
      if( y(1) > 1.0d0 ) then
         ! End condition
         flag = 3
      end if
   end subroutine deriv_end_condition

   subroutine deriv_combined_conditions( t, y, yprime, flag )
      real(kind=MF_DOUBLE), intent(in)     :: t, y(*)
      real(kind=MF_DOUBLE), intent(out)    :: yprime(*)
      integer,              intent(in out) :: flag
      !
      ! Regular ODE. Solution is: y(t) = t**2
      !
      ! This kind of combined conditions may be used to avoid
      ! some numerical cost due to repetitive reduced time steps
      ! in order to reach a point very close to y=1
      if( y(1) > 1.01d0 ) then
         ! Illegal condition
         flag = -1
         return
      end if
      yprime(1) = 2.0d0*t
      if( y(1) > 0.99d0 ) then
         ! End condition
         flag = 3
      end if
   end subroutine deriv_combined_conditions

   subroutine deriv_illegal_condition_sing( t, y, yprime, flag )
      real(kind=MF_DOUBLE), intent(in)     :: t, y(*)
      real(kind=MF_DOUBLE), intent(out)    :: yprime(*)
      integer,              intent(in out) :: flag
      !
      ! This is a singular ODE which has an infinite negative slope
      ! at t = 2/3, where y(t) = 0.
      !
      ! Exact solution is: y(t) = 1/4*(8-12*t)**(2/3)
      if( y(1) <= 0.0d0 ) then
         ! Illegal condition
         flag = -1
         return
      end if
      yprime(1) = -1.0d0 / sqrt( y(1) )
   end subroutine deriv_illegal_condition_sing

end module ode_funs
!_______________________________________________________________________
!
module dae_funs

   use fml

   implicit none

   real(kind=MF_DOUBLE), parameter :: r3_a = -0.5d0,                    &
                                      r3_b = 5.0d0,                     &
                                      r3_c = -5.0d0
! remettre la puissance 1 sur b et c...

contains

   subroutine resid( t, y, yprime, delta, flag )
      real(kind=MF_DOUBLE), intent(in)     :: t, y(*), yprime(*)
      real(kind=MF_DOUBLE), intent(out)    :: delta(*)
      integer,              intent(in out) :: flag
      delta(1) = yprime(1) - ( y(2) )
      delta(2) = yprime(2) - (-y(1) )
   end subroutine resid

   subroutine resid_2( t, y, yprime, delta, flag )
      real(kind=MF_DOUBLE), intent(in)     :: t, y(*), yprime(*)
      real(kind=MF_DOUBLE), intent(out)    :: delta(*)
      integer,              intent(in out) :: flag
      delta(1) = yprime(1) - ( y(2) )
      delta(2) = y(1)**2 + y(2)**2 - 1.0d0
   end subroutine resid_2

   subroutine resid_3( t, y, yprime, delta, flag )
      real(kind=MF_DOUBLE), intent(in)     :: t, y(*), yprime(*)
      real(kind=MF_DOUBLE), intent(out)    :: delta(*)
      integer,              intent(in out) :: flag
      delta(1) = r3_a*y(1) + r3_b*y(2)*y(3)
      delta(2) = -delta(1) + r3_c*y(2)**2 - yprime(2)
      delta(1) = delta(1) - yprime(1)
      delta(3) = y(1) + y(2) + y(3) - 1.0d0
   end subroutine resid_3

   subroutine jac_resid_3( t, y, yprime, jacobian, cj, nrow )
      real(kind=MF_DOUBLE), intent(in)  :: t, y(*), yprime(*), cj
      integer,              intent(in)  :: nrow
      real(kind=MF_DOUBLE), intent(out) :: jacobian(nrow,*)
      ! only non zero elements need to be set
      jacobian(1,1) = r3_a - cj
      jacobian(2,1) = -r3_a
      jacobian(3,1) = 1.0d0
      jacobian(1,2) = r3_b*y(3)
      jacobian(2,2) = -r3_b*y(3) + 2*r3_c*y(2) - cj
      jacobian(3,2) = 1.0d0
      jacobian(1,3) = r3_b*y(2)
      jacobian(2,3) = -r3_b*y(2)
      jacobian(3,3) = 1.0d0
   end subroutine jac_resid_3

   subroutine resid_chem_kinetics( t, y, yprime, delta, flag )
      real(kind=MF_DOUBLE), intent(in)     :: t, y(*), yprime(*)
      real(kind=MF_DOUBLE), intent(out)    :: delta(*)
      integer,              intent(in out) :: flag
      delta(1) = -0.04d0*y(1) + 1.0d4*y(2)*y(3)
      delta(2) = -delta(1) - 3.0d7*y(2)**2 - yprime(2)
      delta(1) = delta(1) - yprime(1)
      delta(3) = y(1) + y(2) + y(3) - 1.0d0
   end subroutine resid_chem_kinetics

   subroutine jac_chem_kinetics( t, y, yprime, jacobian, cj, nrow )
      real(kind=MF_DOUBLE), intent(in)  :: t, y(*), yprime(*), cj
      integer,              intent(in)  :: nrow
      real(kind=MF_DOUBLE), intent(out) :: jacobian(nrow,*)
      ! only non zero elements need to be set
      jacobian(1,1) = -0.04d0 - cj
      jacobian(2,1) = 0.04d0
      jacobian(3,1) = 1.0d0
      jacobian(1,2) = 1.0d4*y(3)
      jacobian(2,2) = -1.0d4*y(3) - 6.0d7*y(2) - cj
      jacobian(3,2) = 1.0d0
      jacobian(1,3) = 1.0d4*y(2)
      jacobian(2,3) = -1.0d4*y(2)
      jacobian(3,3) = 1.0d0
   end subroutine jac_chem_kinetics

   subroutine resid_illegal_condition( t, y, yprime, delta, flag )
      real(kind=MF_DOUBLE), intent(in)     :: t, y(*), yprime(*)
      real(kind=MF_DOUBLE), intent(out)    :: delta(*)
      integer,              intent(in out) :: flag
      !
      ! Regular ODE. Solution is: y(t) = t**2
      !
      if( y(1) > 1.0d0 ) then
         ! Illegal condition
         flag = -1
         return
      end if
      delta(1) = yprime(1) - 2.0d0*t
   end subroutine resid_illegal_condition

   subroutine resid_emergency_exit( t, y, yprime, delta, flag )
      real(kind=MF_DOUBLE), intent(in)     :: t, y(*), yprime(*)
      real(kind=MF_DOUBLE), intent(out)    :: delta(*)
      integer,              intent(in out) :: flag
      !
      ! Regular ODE. Solution is: y(t) = t**2
      !
      if( y(1) > 1.0d0 ) then
         ! Emergency exit
         flag = -2
         return
      end if
      delta(1) = yprime(1) - 2.0d0*t
   end subroutine resid_emergency_exit

   subroutine resid_end_condition( t, y, yprime, delta, flag )
      real(kind=MF_DOUBLE), intent(in)     :: t, y(*), yprime(*)
      real(kind=MF_DOUBLE), intent(out)    :: delta(*)
      integer,              intent(in out) :: flag
      !
      ! Regular ODE. Solution is: y(t) = t**2
      !
      delta(1) = yprime(1) - 2.0d0*t
      if( y(1) > 1.0d0 ) then
         ! End condition
         flag = 3
      end if
   end subroutine resid_end_condition

   subroutine resid_combined_conditions( t, y, yprime, delta, flag )
      real(kind=MF_DOUBLE), intent(in)     :: t, y(*), yprime(*)
      real(kind=MF_DOUBLE), intent(out)    :: delta(*)
      integer,              intent(in out) :: flag
      !
      ! Regular ODE. Solution is: y(t) = t**2
      !
      ! This kind of combined conditions may be used to avoid
      ! some numerical cost due to repetitive reduced time steps
      ! in order to reach a point very close to y=1
      if( y(1) > 1.01d0 ) then
         ! Illegal condition
         flag = -1
         return
      end if
      delta(1) = yprime(1) - 2.0d0*t
      if( y(1) > 0.99d0 ) then
         ! End condition
         flag = 3
      end if
   end subroutine resid_combined_conditions

   subroutine resid_illegal_condition_sing( t, y, yprime, delta, flag )
      real(kind=MF_DOUBLE), intent(in)     :: t, y(*), yprime(*)
      real(kind=MF_DOUBLE), intent(out)    :: delta(*)
      integer,              intent(in out) :: flag
      !
      ! This is a singular ODE which has an infinite negative slope
      ! at t = 2/3, where y(t) = 0.
      !
      ! Exact solution is: y(t) = 1/4*(8-12*t)**(2/3)
      if( y(1) <= 0.0d0 ) then
         ! Illegal condition
         flag = -1
         return
      end if
      delta(1) = yprime(1) - ( -1.0d0 / sqrt( y(1) ) )
   end subroutine resid_illegal_condition_sing

end module dae_funs
!_______________________________________________________________________
!
module fzero_funs

   use fml

   implicit none

contains

   function fun_sqr_minus_onehalf( x ) result( res )
      real(kind=MF_DOUBLE), intent(in) :: x
      real(kind=MF_DOUBLE) :: res
      res = x**2 - 0.5d0
   end function fun_sqr_minus_onehalf

   function fun_tan( x ) result( res )
      real(kind=MF_DOUBLE), intent(in) :: x
      real(kind=MF_DOUBLE) :: res
      res = tan(x)
   end function fun_tan

   subroutine exp_fcn( n, x, fvec, iflag )
      real(kind=MF_DOUBLE), intent(in) :: x(n)
      integer,              intent(in) :: n
      real(kind=MF_DOUBLE)             :: fvec(n)
      integer                          :: iflag
      !---
      fvec(1) = 2*x(1) - x(2) - exp(-x(1))
      fvec(2) = -x(1) + 2*x(2) - exp(-x(2))
   end subroutine exp_fcn

   subroutine exp_fcn_jac( n, x, jacobian )
      real(kind=MF_DOUBLE), intent(in) :: x(n)
      integer,              intent(in) :: n
      real(kind=MF_DOUBLE)             :: jacobian(n,n)
      !---
      jacobian(1,1) = 2 + exp(-x(1))
      jacobian(1,2) = -1
      jacobian(2,1) = -1
      jacobian(2,2) = 2 + exp(-x(2))
   end subroutine exp_fcn_jac

   subroutine fcn_oct_1( n, x, fvec, iflag )
      real(kind=MF_DOUBLE), intent(in) :: x(n)
      integer,              intent(in) :: n
      real(kind=MF_DOUBLE)             :: fvec(n)
      integer                          :: iflag
      !---
      fvec(1) = sin(x(1)) + x(2)**2 + log(x(3)) - 7
      fvec(2) = 3*x(1) + 2**x(2) - x(3)**3 + 1
      fvec(3) = x(1) + x(2) + x(3) - 5
   end subroutine fcn_oct_1

   subroutine fcn_oct_1_jac( n, x, jacobian )
      real(kind=MF_DOUBLE), intent(in) :: x(n)
      integer,              intent(in) :: n
      real(kind=MF_DOUBLE)             :: jacobian(n,n)
      !---
      jacobian(1,1) = cos(x(1))
      jacobian(1,2) = 2*x(2)
      jacobian(1,3) = 1.0d0/x(3)
      jacobian(2,1) = 3
      jacobian(2,2) = 2**x(2)*log(2.0d0)
      jacobian(2,3) = -3*x(3)**2
      jacobian(3,1) = 1
      jacobian(3,2) = 1
      jacobian(3,3) = 1
   end subroutine fcn_oct_1_jac

   subroutine fcn_oct_2( n, x, fvec, iflag )
      real(kind=MF_DOUBLE), intent(in) :: x(n)
      integer,              intent(in) :: n
      real(kind=MF_DOUBLE)             :: fvec(n)
      integer                          :: iflag
      !---
      fvec(1) = 3*x(1) + 4*x(2) + exp(x(3)+x(4)) - 1.007d0
      fvec(2) = 6*x(1) - 4*x(2) + exp(3*x(3)+x(4)) - 11
      fvec(3) = x(1)**4 - 4*x(2)**2 + 6*x(3) - 8*x(4) - 20
      fvec(4) = x(1)**2 + 2*x(2)**3 + x(3) - x(4) - 4
   end subroutine fcn_oct_2

   subroutine fcn_oct_2_jac( n, x, jacobian )
      real(kind=MF_DOUBLE), intent(in) :: x(n)
      integer,              intent(in) :: n
      real(kind=MF_DOUBLE)             :: jacobian(n,n)
      !---
      jacobian(1,1) = 3
      jacobian(1,2) = 4
      jacobian(1,3) = exp(x(3)+x(4))
      jacobian(1,4) = exp(x(3)+x(4))
      jacobian(2,1) = 6
      jacobian(2,2) = -4
      jacobian(2,3) = 3*exp(3*x(3)+x(4))
      jacobian(2,4) = exp(3*x(3)+x(4))
      jacobian(3,1) = 4*x(1)**3
      jacobian(3,2) = -8*x(2)
      jacobian(3,3) = 6
      jacobian(3,4) = -8
      jacobian(4,1) = 2*x(1)
      jacobian(4,2) = 6*x(2)**2
      jacobian(4,3) = 1
      jacobian(4,4) = -1
   end subroutine fcn_oct_2_jac

   subroutine fcn_4_ec( n, x, fvec, iflag )
      real(kind=MF_DOUBLE), intent(in) :: x(n)
      integer,              intent(in) :: n
      real(kind=MF_DOUBLE)             :: fvec(n)
      integer                          :: iflag
      !---
      fvec(1) = exp(x(3)+x(4)) - exp(2.0d0)
      fvec(2) = x(1)*x(4) - 1
      fvec(3) = x(1)*x(2)**3 - 1
      fvec(4) = x(2)**3 + x(3)**3 - 2.0d0
   end subroutine fcn_4_ec

   subroutine fcn_4_ec_spjac( n, x, job, pd, ipd, jpd, nnz )
      integer,          intent(in) :: n, job
      double precision, intent(in) :: x(n)
      double precision             :: pd(*)
      integer                      :: ipd(*), jpd(*), nnz
      ! two kinds of call :
      ! - job=0 : initialization (i.e. allocation of vectors)
      !     -> only nnz is returned
      ! - job/=0 : subsequent standard calls (i.e. return all vectors)

      ! nnz = 2 should be sufficient, but we must assign entries
      !       for the diagonal elements (even if they are null)
      !
      ! row entries (in ipd) must be sorted in ascending order

      !
      !       [    0            0          exp(x(3)+x(4))   exp(x(3)+x(4)) ]
      ! Jac = [   x(4)          0                0               x(1)      ]
      !       [ x(2)**3   3*x(1)*x(2)**2         0                0        ]
      !       [    0        3*x(2)**2        3*x(3)**2            0        ]
      !
      nnz = 8
      if( job == 0 ) then
         return
      end if
      PD(1:nnz) = [ x(4), x(2)**3, 3*x(1)*x(2)**2, 3*x(2)**2,           &
                    exp(x(3)+x(4)), 3*x(3)**2, exp(x(3)+x(4)), x(1) ]
      IPD(1:nnz) = [ 2, 3, 3, 4, 1, 4, 1, 2 ]
      JPD(1:n+1) = [ 1, 3, 5, 7, nnz+1 ]
   end subroutine fcn_4_ec_spjac

end module fzero_funs
!_______________________________________________________________________
!
module optim_funs_2

   use fml

   implicit none

contains

   subroutine exp_min( m, n, p, fvec, iflag )
      integer,              intent(in) :: m, n
      real(kind=MF_DOUBLE), intent(in) :: p(n)
      real(kind=MF_DOUBLE)             :: fvec(m)
      integer                          :: iflag
      !---
      integer :: i
      if( n /= 2 ) then
         iflag = -1
         return
      end if
      do i = 1, m
         fvec(i) = 2.0d0 + 2.0d0*i                                      &
                 - exp(i/dble(m)*p(1)) - exp(sqrt(i/dble(m))*p(2))
      end do
   end subroutine exp_min

   subroutine exp_min_jac( m, n, p, fjac )
      integer,              intent(in) :: m, n
      real(kind=MF_DOUBLE), intent(in) :: p(n)
      real(kind=MF_DOUBLE)             :: fjac(m,n)
      !---
      integer :: i
      do i = 1, m
         fjac(i,1) = - i/dble(m) * exp(i/dble(m)*p(1))
         fjac(i,2) = - sqrt(i/dble(m)) * exp(sqrt(i/dble(m))*p(2))
      end do
   end subroutine exp_min_jac

   subroutine lin_regress( m, n, p, fvec, flag )
      integer,              intent(in) :: m, n
      real(kind=MF_DOUBLE), intent(in) :: p(n)
      real(kind=MF_DOUBLE)             :: fvec(m)
      integer                          :: flag
      !---
      integer :: i
      if( n /= 3 ) then
         flag = -1
         return
      end if
      fvec(1) = -p(3)*1.0d-2 + p(2) - 1.5d0
      fvec(2) =  p(3)*1.0d-2 + p(2) - 0.5d0
      fvec(3) =  p(1)*1.0d-6 + p(2) - 1.0d0
   end subroutine lin_regress

   subroutine lin_regress_jac( m, n, p, fjac )
      integer,              intent(in) :: m, n
      real(kind=MF_DOUBLE), intent(in) :: p(n)
      real(kind=MF_DOUBLE)             :: fjac(m,n)
      !---
      fjac(1,1) = 0.0d0
      fjac(1,2) = 1.0d0
      fjac(1,3) = -1.0d-2
      fjac(2,1) = 0.0d0
      fjac(2,2) = 1.0d0
      fjac(2,3) = 1.0d-2
      fjac(3,1) = 1.0d-6
      fjac(3,2) = 1.0d0
      fjac(3,3) = 0.0d0
   end subroutine lin_regress_jac

end module optim_funs_2
