module mod_ops ! Operators

   ! Part of MUESLI Numerical Library
   ! Copyright É. Canot 2003-2025 -- IPR/CNRS

!-----------------------------------------------------------------------
!                             used modules
!-----------------------------------------------------------------------

   use mod_elfun

#ifndef _DEVLP
   use mod_mfarray ! required for the 2nd pass of the double compilation
   use mod_physunits ! required for the 2nd pass of the double compilation
#endif

#ifndef _WINDOWS
   !$ use omp_lib
#endif

   implicit none

#ifndef _DEVLP
   private
#endif

   interface mfColon
      module procedure mfColon_int
      module procedure mfColon_single
      module procedure mfColon_double
   end interface mfColon
   !------ API end ------

   interface operator(.t.)
      module procedure mfTranspose_mfArray
      module procedure mfTranspose_vec_int
      module procedure mfTranspose_vec_real8
      module procedure mfTranspose_array_int
      module procedure mfTranspose_array_real8
   end interface operator(.t.)
   !------ API end ------

   interface operator(.h.)
      module procedure mfCTranspose_mfArray
      module procedure mfCTranspose_vec_cmplx8
      module procedure mfCTranspose_array_cmplx8
   end interface operator(.h.)
   !------ API end ------

   interface operator(.x.)
      module procedure mfMul_A_B
   end interface operator(.x.)
   !------ API end ------

   interface operator(+)
      module procedure plus_mfArray
      module procedure mfAdd_mfArray_mfArray
      module procedure mfAdd_mfArray_int
      module procedure mfAdd_int_mfArray
      module procedure mfAdd_mfArray_real4
      module procedure mfAdd_real4_mfArray
      module procedure mfAdd_mfArray_real8
      module procedure mfAdd_real8_mfArray
      module procedure mfAdd_mfArray_cmplx4
      module procedure mfAdd_cmplx4_mfArray
      module procedure mfAdd_mfArray_cmplx8
      module procedure mfAdd_cmplx8_mfArray
      ! for Physical Units :
      module procedure mfAdd_mfArray_mfUnit
      module procedure mfAdd_mfUnit_mfArray
   end interface operator(+)
   !------ API end ------

   interface operator(-)
      module procedure minus_mfArray
      module procedure mfSub_mfArray_mfArray
      module procedure mfSub_mfArray_int
      module procedure mfSub_int_mfArray
      module procedure mfSub_mfArray_real4
      module procedure mfSub_real4_mfArray
      module procedure mfSub_mfArray_real8
      module procedure mfSub_real8_mfArray
      module procedure mfSub_mfArray_cmplx4
      module procedure mfSub_cmplx4_mfArray
      module procedure mfSub_mfArray_cmplx8
      module procedure mfSub_cmplx8_mfArray
      ! for Physical Units :
      module procedure mfSub_mfArray_mfUnit
      module procedure mfSub_mfUnit_mfArray
   end interface operator(-)
   !------ API end ------

   interface operator(*)
      module procedure mfMul_mfArray_mfArray
      module procedure mfMul_mfArray_int
      module procedure mfMul_int_mfArray
      module procedure mfMul_mfArray_real4
      module procedure mfMul_real4_mfArray
      module procedure mfMul_mfArray_real8
      module procedure mfMul_real8_mfArray
      module procedure mfMul_mfArray_cmplx4
      module procedure mfMul_cmplx4_mfArray
      module procedure mfMul_mfArray_cmplx8
      module procedure mfMul_cmplx8_mfArray
      ! for Physical Units :
      module procedure mfMul_mfArray_mfUnit
      module procedure mfMul_mfUnit_mfArray
   end interface operator(*)
   !------ API end ------

   interface operator(/)
      module procedure mfDiv_mfArray_mfArray
      module procedure mfDiv_mfArray_int
      module procedure mfDiv_int_mfArray
      module procedure mfDiv_mfArray_real4
      module procedure mfDiv_real4_mfArray
      module procedure mfDiv_mfArray_real8
      module procedure mfDiv_real8_mfArray
      module procedure mfDiv_mfArray_cmplx4
      module procedure mfDiv_cmplx4_mfArray
      module procedure mfDiv_mfArray_cmplx8
      module procedure mfDiv_cmplx8_mfArray
      ! for Physical Units :
      module procedure mfDiv_mfArray_mfUnit
      module procedure mfDiv_mfUnit_mfArray ! really used ?
   end interface
   !------ API end ------

   interface operator(**)
      module procedure mfPow_mfArray_int
      module procedure mfPow_mfArray_real8
      module procedure mfPow_int_mfArray
      module procedure mfPow_real8_mfArray
      module procedure mfPow_mfArray_mfArray
   end interface operator(**)
   !------ API end ------

   interface operator(.not.)
      module procedure not_mfArray ! -> boolean mfArray
   end interface operator(.not.)
   !------ API end ------

   interface operator(.and.)
      module procedure and_mfArray_mfArray ! -> boolean mfArray
   end interface operator(.and.)
   !------ API end ------

   interface operator(.or.)
      module procedure or_mfArray_mfArray ! -> boolean mfArray
   end interface operator(.or.)
   !------ API end ------

   interface operator(.eqv.)
      module procedure eqv_mfArray_mfArray ! -> boolean mfArray
   end interface operator(.eqv.)
   !------ API end ------

   interface operator(.neqv.)
      module procedure neqv_mfArray_mfArray ! -> boolean mfArray
   end interface operator(.neqv.)
   !------ API end ------

   interface operator(==)
      module procedure mfArray_equal_mfArray ! -> boolean mfArray
      module procedure mfArray_equal_int     ! -> boolean mfArray
      module procedure mfArray_equal_real8   ! -> boolean mfArray
      module procedure mfArray_equal_cmplx   ! -> boolean mfArray
   end interface operator(==)
   !------ API end ------

   interface operator(/=)
      module procedure mfArray_not_equal_mfArray ! -> boolean mfArray
      module procedure mfArray_not_equal_int    ! -> boolean mfArray
      module procedure mfArray_not_equal_real8   ! -> boolean mfArray
      module procedure mfArray_not_equal_cmplx   ! -> boolean mfArray
   end interface operator(/=)
   !------ API end ------

   interface operator(>=)
      module procedure mfGreater_mfArray_mfArray ! -> boolean mfArray
      module procedure mfGreater_mfArray_real8   ! -> boolean mfArray
      module procedure mfGreater_mfArray_int     ! -> boolean mfArray
   end interface operator(>=)
   !------ API end ------

   interface operator(>)
      module procedure mfGreaterStrict_mfArray_mfArray ! -> boolean mfArray
      module procedure mfGreaterStrict_mfArray_real8   ! -> boolean mfArray
      module procedure mfGreaterStrict_mfArray_int     ! -> boolean mfArray
   end interface operator(>)
   !------ API end ------

   interface operator(<=)
      module procedure mfLess_mfArray_mfArray ! -> boolean mfArray
      module procedure mfLess_mfArray_real8   ! -> boolean mfArray
      module procedure mfLess_mfArray_int     ! -> boolean mfArray
   end interface operator(<=)
   !------ API end ------

   interface operator(<)
      module procedure mfLessStrict_mfArray_mfArray ! -> boolean mfArray
      module procedure mfLessStrict_mfArray_real8   ! -> boolean mfArray
      module procedure mfLessStrict_mfArray_int     ! -> boolean mfArray
   end interface operator(<)
   !------ API end ------

   interface mfMul
      module procedure mfMul_A_B
      module procedure mfMul_transp
   end interface mfMul
   !------ API end ------

   public :: operator(==), &
             operator(/=), &
             mfColon, &
             operator(.t.), &
             operator(.h.), &
             operator(+), &
             operator(-), &
             operator(*), &
             operator(/), &
             operator(**), &
             operator(>=), &
             operator(>), &
             operator(<=), &
             operator(<), &
             operator(.not.), &
             operator(.and.), &
             operator(.or.), &
             operator(.eqv.), &
             operator(.neqv.), &
             mfAll, &
             mfAny, &
             mfMul, &
             operator(.x.), &
             mfCross, &
             mfColPerm, msColPerm, &
             mfRowPerm, msRowPerm, &
             mfColScale, msColScale, &
             mfRowScale, msRowScale, &
             mfInvPerm, &
             mfUnique

   logical, private :: discard_phys_unit = .false.

   private :: Mul_At_B, Mul_A_Bt

contains
!_______________________________________________________________________
!
#include "fml_ops/Add.inc"
!_______________________________________________________________________
!
#include "fml_ops/Sub.inc"
!_______________________________________________________________________
!
#include "fml_ops/Mul.inc"
!_______________________________________________________________________
!
#include "fml_ops/Div.inc"
!_______________________________________________________________________
!
#include "fml_ops/Pow.inc"
!_______________________________________________________________________
!
#include "fml_ops/Colon.inc"
!_______________________________________________________________________
!
#include "fml_ops/Transpose.inc"
!_______________________________________________________________________
!
#include "fml_ops/CTranspose.inc"
!_______________________________________________________________________
!
#include "fml_ops/Greater.inc"
!_______________________________________________________________________
!
#include "fml_ops/GreaterStrict.inc"
!_______________________________________________________________________
!
#include "fml_ops/Less.inc"
!_______________________________________________________________________
!
#include "fml_ops/LessStrict.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/nnzamub.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/amub.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/colperm.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/rowperm.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/colscale.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/rowscale.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/col_auto_scale.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/row_auto_scale.inc"
!_______________________________________________________________________
!
#include "fml_ops/matmul.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/nnzatmub.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/atmub.inc"
!_______________________________________________________________________
!
   function plus_mfArray( A ) result( out )

      type(mfArray), intent(in) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( A%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "operator(+)", "E",                         &
                            "array is empty!" )
         return
      end if

      if( A%data_type == MF_DT_BOOL .or. A%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "operator(+)", "E",                         &
                            "this function cannot be applied to a logical!" )
         call msAutoRelease(A)
         return
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "operator(+)", "E",                         &
                            "cannot be applied to a permutation vector!" )
         call msAutoRelease(A)
         return
      end if

      out%data_type = A%data_type
      out%shape = A%shape

      select case( A%data_type )
         case( MF_DT_DBLE )
            out%double => A%double
         case( MF_DT_CMPLX )
            out%cmplx => A%cmplx
         case( MF_DT_SP_DBLE )
            out%a => A%a
            out%i => A%i
            out%j => A%j
         case( MF_DT_SP_CMPLX )
            out%z => A%z
            out%i => A%i
            out%j => A%j
         case default
            call PrintMessage( "operator(+)", "W",                      &
                               "incorrect use of this ops!" )
      end select

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu
      out%prop%symm = A%prop%symm
      out%prop%posd = A%prop%posd

      if( mf_phys_units ) then
         out%units(:) = a%units(:)
      end if

      out%status_temporary = A%status_temporary

#endif
   end function plus_mfArray
!_______________________________________________________________________
!
   function minus_mfArray( A ) result( out )

      type(mfArray), intent(in) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( A%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "operator(-)", "E",                         &
                            "array is empty!" )
         return
      end if

      if( A%data_type == MF_DT_BOOL .or. A%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "operator(-)", "E",                         &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "operator(-)", "E",                         &
                            "cannot be applied to a permutation vector!" )
         call msAutoRelease(A)
         return
      end if

      out%data_type = A%data_type
      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         allocate( out%double(out%shape(1),out%shape(2)) )

         out%double = -A%double
      else if( A%data_type == MF_DT_CMPLX ) then
         allocate( out%cmplx(out%shape(1),out%shape(2)) )

         out%cmplx = -A%cmplx
      else if( A%data_type == MF_DT_SP_DBLE ) then
         allocate( out%a(size(A%a)) )

         out%a = -A%a
         allocate( out%i(size(A%i)) )

         out%i = A%i
         allocate( out%j(size(A%j)) )

         out%j = A%j
      else if( A%data_type == MF_DT_SP_CMPLX ) then
         allocate( out%z(size(A%z)) )

         out%z = -A%z
         allocate( out%i(size(A%i)) )

         out%i = A%i
         allocate( out%j(size(A%j)) )

         out%j = A%j
      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu
      out%prop%symm = A%prop%symm
      out%prop%posd = UNKNOWN

      if( mf_phys_units ) then
         out%units(:) = a%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msAutoRelease(A)

#endif
   end function minus_mfArray
!_______________________________________________________________________
!
   recursive function mfArray_equal_mfArray( a, b ) result( out )

      ! Tests if two arrays are numerically equal, whatever the
      ! structure is (i.e. sparse or dense, real or complex).
      ! In case of mixed structures (ex. dense versus sparse),
      ! the sparse array is converted in a dense one, and returns a
      ! dense boolean mfArray; so it is not efficient for large
      ! matrices : it therefore emits a warning.
      !
      ! The shapes must match, and should be non zero.
      !
      ! Comparison between boolean mfArray are excluded.
      !
      ! '==' returns a boolean mfArray,
      ! whereas 'mfIsEqual' returns a scalar logical (in 'mod_core' module)

      type(mfArray), intent(in) :: a, b
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( a, b )

      if( mfIsEmpty(a) ) then
         call PrintMessage( "==", "W",                                  &
                            "left arg is an empty mfArray!" )
         go to 99
      end if

      if( mfIsEmpty(b) ) then
         call PrintMessage( "==", "W",                                  &
                            "right arg is an empty mfArray!" )
         go to 99
      end if

      if( a%data_type == MF_DT_BOOL .or. A%data_type == MF_DT_SP_BOOL .or. &
          b%data_type == MF_DT_BOOL .or. B%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "==", "E",                                  &
                            "this operator cannot be applied to a logical!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC .or. B%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "==", "E",                                  &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      ! mixed structure
      if( mfIsSparse(a) .neqv. mfIsSparse(b) ) then
         ! recursive call
         call PrintMessage( "==", "W",                                  &
                            "comparing mixed matrices structure (dense/sparse)",&
                            "The sparse one is converted in a dense one.",&
                            "For large matrices, it is very inefficient." )
         if( mfIsSparse(a) ) then
            out = mfArray_equal_mfArray( mfFull(a), b )
         else if( mfIsSparse(b) ) then
            out = mfArray_equal_mfArray( a, mfFull(b) )
         end if
         out%status_temporary = .true.
         go to 99
      end if

      if( any( a%shape /= b%shape ) ) then
         call PrintMessage( "==", "E",                                  &
                            "args must have the same shape!" )
         go to 99
      end if

      out%data_type = MF_DT_BOOL
      out%shape = a%shape
      allocate( out%double(out%shape(1),out%shape(2)) )

      if( a%data_type == MF_DT_DBLE .and. b%data_type == MF_DT_DBLE ) then
         where( a%double == b%double )
            out%double = TRUE
         elsewhere
            out%double = FALSE
         end where
      else if( a%data_type == MF_DT_CMPLX .and. b%data_type == MF_DT_CMPLX ) then
         where( a%cmplx == b%cmplx )
            out%double = TRUE
         elsewhere
            out%double = FALSE
         end where
      else if( a%data_type == MF_DT_DBLE .and. b%data_type == MF_DT_CMPLX ) then
         where( a%double == b%cmplx )
            out%double = TRUE
         elsewhere
            out%double = FALSE
         end where
      else if( a%data_type == MF_DT_CMPLX .and. b%data_type == MF_DT_DBLE ) then
         where( a%cmplx == b%double )
            out%double = TRUE
         elsewhere
            out%double = FALSE
         end where
      else
         call PrintMessage( "==", "E",                                  &
                            "cannot use sparse mfArrays!",              &
                            "(if you just want to test equality between two mfArray's",&
                            "you should use the 'mfIsEqual' function)" )
         go to 99
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, b%units, status )
         if( status /= 0 ) then
            call PrintMessage( "==", "E",                               &
                               "the physical dimensions of the two mfArray's",&
                               "are not consistent!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( a, b )
      call msAutoRelease( a, b )

#endif
   end function mfArray_equal_mfArray
!_______________________________________________________________________
!
   function mfArray_equal_int( a, x ) result( out )

      type(mfArray), intent(in) :: a
      integer,       intent(in) :: x
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( a )

      if( mfIsEmpty(a) ) then
         call PrintMessage( "==", "W",                                  &
                            "left arg is an empty mfArray!" )
         go to 99
      end if

      if( mfIsSparse(a) ) then
         call PrintMessage( "==", "E",                                  &
                            "sparse matrices not yet supported!" )
         go to 99
      end if

      if( a%data_type == MF_DT_BOOL .or. A%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "==", "E",                                  &
                            "this operator cannot be applied to a logical!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "==", "E",                                  &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%data_type = MF_DT_BOOL
      out%shape = a%shape
      allocate( out%double(out%shape(1),out%shape(2)) )

      if( a%data_type == MF_DT_DBLE ) then
         where( a%double == dble(x) )
            out%double = TRUE
         elsewhere
            out%double = FALSE
         end where
      else if( a%data_type == MF_DT_CMPLX ) then
         where( a%cmplx == cmplx(x,kind=MF_DOUBLE) )
            out%double = TRUE
         elsewhere
            out%double = FALSE
         end where
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "==", "E",                               &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( a )

      call msAutoRelease( a )

#endif
   end function mfArray_equal_int
!_______________________________________________________________________
!
   function mfArray_equal_real8( a, x ) result( out )

      type(mfArray), intent(in) :: a
      real(kind=MF_DOUBLE), intent(in) :: x
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( a )

      if( mfIsEmpty(a) ) then
         call PrintMessage( "==", "W",                                  &
                            "left arg is an empty mfArray!" )
         go to 99
      end if

      if( a%data_type == MF_DT_BOOL .or. A%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "==", "E",                                  &
                            "this operator cannot be applied to a logical!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "==", "E",                                  &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( a%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_BOOL
         out%shape = a%shape
         allocate( out%double(out%shape(1),out%shape(2)) )
         where( a%double == x )
            out%double = TRUE
         elsewhere
            out%double = FALSE
         end where
      else if( a%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_BOOL
         out%shape = a%shape
         allocate( out%double(out%shape(1),out%shape(2)) )
         where( a%cmplx == x )
            out%double = TRUE
         elsewhere
            out%double = FALSE
         end where
      else if( a%data_type == MF_DT_SP_DBLE ) then
         out%data_type = MF_DT_SP_BOOL
         out%shape = a%shape
         !### sorting is required before calling 'apply_bin_ops_to_sp_dble'
         call msRowSort( A )
         call apply_bin_ops_to_sp_dble( A%shape(1), A%shape(2),         &
                                        A%a, A%i, A%j,                  &
                                        ".eq.", x, out%a, out%i, out%j )
      else if( a%data_type == MF_DT_SP_CMPLX ) then
         out%data_type = MF_DT_SP_BOOL
         out%shape = a%shape
!!         call apply_log_ops_to_sp_cmplx( ... )
stop "[DBG FML] mod_ops.F90: line 714: mfArray_equal_real8 not finished for data type MF_DT_SP_CMPLX"
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "==", "E",                               &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( a )

      call msAutoRelease( a )

#endif
   end function mfArray_equal_real8
!_______________________________________________________________________
!
   function mfArray_equal_cmplx( a, x ) result( out )

      type(mfArray), intent(in) :: a
      complex(kind=MF_DOUBLE), intent(in) :: x
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( a )

      if( mfIsEmpty(a) ) then
         call PrintMessage( "==", "W",                                  &
                            "left arg is an empty mfArray!" )
         go to 99
      end if

      if( mfIsSparse(a) ) then
         call PrintMessage( "==", "E",                                  &
                            "sparse matrices not yet supported!" )
         go to 99
      end if

      if( a%data_type == MF_DT_BOOL .or. A%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "==", "E",                                  &
                            "this operator cannot be applied to a logical!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "==", "E",                                  &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%data_type = MF_DT_BOOL
      out%shape = a%shape
      allocate( out%double(out%shape(1),out%shape(2)) )

      if( a%data_type == MF_DT_DBLE ) then
         where( a%double == x )
            out%double = TRUE
         elsewhere
            out%double = FALSE
         end where
      else if( a%data_type == MF_DT_CMPLX ) then
         where( a%cmplx == x )
            out%double = TRUE
         elsewhere
            out%double = FALSE
         end where
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "==", "E",                               &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( a )

      call msAutoRelease( a )

#endif
   end function mfArray_equal_cmplx
!_______________________________________________________________________
!
   recursive function mfArray_not_equal_mfArray( a, b ) result( out )

      ! Tests if two arrays are numerically different, whatever the
      ! structure is (i.e. sparse or dense, real or complex).
      ! In case of mixed structures (ex. dense versus sparse),
      ! the sparse array is converted in a dense one : so it is
      ! sometimes not very efficient; it therefore emits a warning.
      !
      ! the shapes must match, and should be non zero.
      !
      ! comparison between boolean mfArray are excluded.
      !
      ! '/=' returns a boolean mfArray,
      ! whereas 'mfIsNotEqual' returns a scalar logical (in 'mod_core' module)

      type(mfArray), intent(in) :: a, b
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( a, b )

      if( mfIsEmpty(a) ) then
         call PrintMessage( "/=", "W",                                  &
                            "left arg is an empty mfArray!" )
         go to 99
      end if

      if( mfIsEmpty(b) ) then
         call PrintMessage( "/=", "W",                                  &
                            "right arg is an empty mfArray!" )
         go to 99
      end if

      if( a%data_type == MF_DT_BOOL .or. a%data_type == MF_DT_SP_BOOL .or. &
          b%data_type == MF_DT_BOOL .or. b%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "/=", "E",                                  &
                            "this operator cannot be applied to a logical!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC .or. B%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "/=", "E",                                  &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      ! mixed structure
      if( mfIsSparse(a) .neqv. mfIsSparse(b) ) then
         ! recursive call
         call PrintMessage( "/=", "W",                                  &
                            "comparing mixed matrices structure (dense/sparse)",&
                            "The sparse one is converted in a dense one.",&
                            "For large matrices, it is very inefficient." )
         if( mfIsSparse(a) ) then
            out = mfArray_not_equal_mfArray( mfFull(a), b )
         else if( mfIsSparse(b) ) then
            out = mfArray_not_equal_mfArray( a, mfFull(b) )
         end if
         out%status_temporary = .true.
         go to 99
      end if

      if( any( a%shape /= b%shape ) ) then
         call PrintMessage( "/=", "E",                                  &
                            "args must have the same shape!" )
         go to 99
      end if

      out%data_type = MF_DT_BOOL
      out%shape = a%shape
      allocate( out%double(out%shape(1),out%shape(2)) )

      if( a%data_type == MF_DT_DBLE .and. b%data_type == MF_DT_DBLE ) then
         where( a%double /= b%double )
            out%double = TRUE
         elsewhere
            out%double = FALSE
         end where
      else if( a%data_type == MF_DT_CMPLX .and. b%data_type == MF_DT_CMPLX ) then
         where( a%cmplx /= b%cmplx )
            out%double = TRUE
         elsewhere
            out%double = FALSE
         end where
      else if( a%data_type == MF_DT_DBLE .and. b%data_type == MF_DT_CMPLX ) then
         where( a%double /= b%cmplx )
            out%double = TRUE
         elsewhere
            out%double = FALSE
         end where
      else if( a%data_type == MF_DT_CMPLX .and. b%data_type == MF_DT_DBLE ) then
         where( a%cmplx /= b%double )
            out%double = TRUE
         elsewhere
            out%double = FALSE
         end where
      else
         call PrintMessage( "/=", "E",                                  &
                            "cannot use sparse mfArrays!",              &
                            "(if you just want to test equality between two mfArray's",&
                            "you should use the '.not. mfIsEqual' combination)" )
         go to 99
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, b%units, status )
         if( status /= 0 ) then
            call PrintMessage( "/=", "E",                               &
                               "the physical dimensions of the two mfArray's",&
                               "are not consistent!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( a, b )

      call msAutoRelease( a, b )

#endif
   end function mfArray_not_equal_mfArray
!_______________________________________________________________________
!
   function mfArray_not_equal_int( a, x ) result( out )

      type(mfArray), intent(in) :: a
      integer,       intent(in) :: x
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( a )

      if( mfIsEmpty(a) ) then
         call PrintMessage( "/=", "W",                                  &
                            "left arg is an empty mfArray!" )
         go to 99
      end if

      if( mfIsSparse(a) ) then
         call PrintMessage( "/=", "E",                                  &
                            "sparse matrices not yet supported!" )
         go to 99
      end if

      if( a%data_type == MF_DT_BOOL .or. A%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "/=", "E",                                  &
                            "this operator cannot be applied to a logical!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "/=", "E",                                  &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%data_type = MF_DT_BOOL
      out%shape = a%shape
      allocate( out%double(out%shape(1),out%shape(2)) )

      if( a%data_type == MF_DT_DBLE ) then
         where( a%double /= dble(x) )
            out%double = TRUE
         elsewhere
            out%double = FALSE
         end where
      else if( a%data_type == MF_DT_CMPLX ) then
         where( a%cmplx /= cmplx(x,kind=MF_DOUBLE) )
            out%double = TRUE
         elsewhere
            out%double = FALSE
         end where
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "/=", "E",                               &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( a )

      call msAutoRelease( a )

#endif
   end function mfArray_not_equal_int
!_______________________________________________________________________
!
   function mfArray_not_equal_real8( a, x ) result( out )

      type(mfArray), intent(in) :: a
      real(kind=MF_DOUBLE), intent(in) :: x
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( a )

      if( mfIsEmpty(a) ) then
         call PrintMessage( "/=", "W",                                  &
                            "left arg is an empty mfArray!" )
         go to 99
      end if

      if( mfIsSparse(a) ) then
         call PrintMessage( "/=", "E",                                  &
                            "sparse matrices not yet supported!" )
         go to 99
      end if

      if( a%data_type == MF_DT_BOOL .or. A%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "/=", "E",                                  &
                            "this operator cannot be applied to a logical!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "/=", "E",                                  &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%data_type = MF_DT_BOOL
      out%shape = a%shape
      allocate( out%double(out%shape(1),out%shape(2)) )

      if( a%data_type == MF_DT_DBLE ) then
         where( a%double /= x )
            out%double = TRUE
         elsewhere
            out%double = FALSE
         end where
      else if( a%data_type == MF_DT_CMPLX ) then
         where( a%cmplx /= x )
            out%double = TRUE
         elsewhere
            out%double = FALSE
         end where
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "/=", "E",                               &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( a )

      call msAutoRelease( a )

#endif
   end function mfArray_not_equal_real8
!_______________________________________________________________________
!
   function mfArray_not_equal_cmplx( a, x ) result( out )

      type(mfArray), intent(in) :: a
      complex(kind=MF_DOUBLE), intent(in) :: x
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( a )

      if( mfIsEmpty(a) ) then
         call PrintMessage( "/=", "W",                                  &
                            "left arg is an empty mfArray!" )
         go to 99
      end if

      if( mfIsSparse(a) ) then
         call PrintMessage( "/=", "E",                                  &
                            "sparse matrices not yet supported!" )
         go to 99
      end if

      if( a%data_type == MF_DT_BOOL .or. A%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "/=", "E",                                  &
                            "this operator cannot be applied to a logical!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "/=", "E",                                  &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%data_type = MF_DT_BOOL
      out%shape = a%shape
      allocate( out%double(out%shape(1),out%shape(2)) )

      if( a%data_type == MF_DT_DBLE ) then
         where( a%double /= x )
            out%double = TRUE
         elsewhere
            out%double = FALSE
         end where
      else if( a%data_type == MF_DT_CMPLX ) then
         where( a%cmplx /= x )
            out%double = TRUE
         elsewhere
            out%double = FALSE
         end where
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "/=", "E",                               &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( a )

      call msAutoRelease( a )

#endif
   end function mfArray_not_equal_cmplx
!_______________________________________________________________________
!
   function not_mfArray( a ) result( out )

      type(mfArray), intent(in) :: a
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( a )

      if( mfIsEmpty(a) ) then
         call PrintMessage( ".not.", "W",                               &
                            "arg is an empty mfArray!" )
         go to 99
      end if

      if( a%data_type /= MF_DT_BOOL .and. a%data_type /= MF_DT_SP_BOOL ) then
         call PrintMessage( ".not.", "E",                               &
                            "arg must be a logical mfArray!" )
         go to 99
      end if

      if( a%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( ".not.", "E",                               &
                            "not yet implemented for sparse boolean matrix!" )
         go to 99
      end if

      out%data_type = MF_DT_BOOL
      out%shape = a%shape
      allocate( out%double(out%shape(1),out%shape(2)) )

      where( a%double == FALSE )
         out%double = TRUE
      elsewhere
         out%double = FALSE
      end where

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( a )

      call msAutoRelease( a )

#endif
   end function not_mfArray
!_______________________________________________________________________
!
   function and_mfArray_mfArray( a, b ) result( out )

      type(mfArray), intent(in) :: a, b
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( a, b )

      if( isEndIndex(a) ) then
         call PrintMessage( ".and.", "E",                               &
                            "index sequences must be grouped inside parenthesis!" )
         go to 99
      end if

      if( isEndIndex(b) ) then
         call PrintMessage( ".and.", "E",                               &
                            "index sequences must be grouped inside parenthesis!" )
         go to 99
      end if

      if( mfIsEmpty(a) ) then
         call PrintMessage( ".and.", "W",                               &
                            "left arg is an empty mfArray!" )
         go to 99
      end if

      if( mfIsEmpty(b) ) then
         call PrintMessage( ".and.", "W",                               &
                            "right arg is an empty mfArray!" )
         go to 99
      end if

      if( (a%data_type /= MF_DT_BOOL .and. a%data_type /= MF_DT_SP_BOOL) .or. &
          (b%data_type /= MF_DT_BOOL .and. b%data_type /= MF_DT_SP_BOOL) ) then
         call PrintMessage( ".and.", "E",                               &
                            "args must be logical mfArrays!" )
         go to 99
      end if

      if( a%data_type == MF_DT_SP_BOOL .or. b%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( ".and.", "E",                               &
                            "not yet implemented for sparse boolean matrix!" )
         go to 99
      end if

      if( any( a%shape /= b%shape ) ) then
         call PrintMessage( ".and.", "E",                               &
                            "args must have the same shape!" )
         go to 99
      end if

      out%data_type = MF_DT_BOOL
      out%shape = a%shape
      allocate( out%double(out%shape(1),out%shape(2)) )

      where( a%double /= FALSE .and. b%double /= FALSE )
         out%double = TRUE
      elsewhere
         out%double = FALSE
      end where

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( a, b )

      call msAutoRelease( a, b )

#endif
   end function and_mfArray_mfArray
!_______________________________________________________________________
!
   function or_mfArray_mfArray( a, b ) result( out )

      type(mfArray), intent(in) :: a, b
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( a, b )

      if( mfIsEmpty(a) ) then
         call PrintMessage( ".or.", "W",                                &
                            "left arg is an empty mfArray!" )
         go to 99
      end if

      if( mfIsEmpty(b) ) then
         call PrintMessage( ".or.", "W",                                &
                            "right arg is an empty mfArray!" )
         go to 99
      end if

      if( (a%data_type /= MF_DT_BOOL .and. a%data_type /= MF_DT_SP_BOOL) .or. &
          (b%data_type /= MF_DT_BOOL .and. b%data_type /= MF_DT_SP_BOOL) ) then
         call PrintMessage( ".or.", "E",                                &
                            "args must be logical mfArrays!" )
         go to 99
      end if

      if( a%data_type == MF_DT_SP_BOOL .or. b%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( ".or.", "E",                               &
                            "not yet implemented for sparse boolean matrix!" )
         go to 99
      end if

      if( any( a%shape /= b%shape ) ) then
         call PrintMessage( ".or.", "E",                                &
                            "args must have the same shape!" )
         go to 99
      end if

      out%data_type = MF_DT_BOOL
      out%shape = a%shape
      allocate( out%double(out%shape(1),out%shape(2)) )

      where( a%double /= FALSE .or. b%double /= FALSE )
         out%double = TRUE
      elsewhere
         out%double = FALSE
      end where

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( a, b )

      call msAutoRelease( a, b )

#endif
   end function or_mfArray_mfArray
!_______________________________________________________________________
!
   function eqv_mfArray_mfArray( a, b ) result( out )

      type(mfArray), intent(in) :: a, b
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( a, b )

      if( mfIsEmpty(a) ) then
         call PrintMessage( ".eqv.", "W",                               &
                            "left arg is an empty mfArray!" )
         go to 99
      end if

      if( mfIsEmpty(b) ) then
         call PrintMessage( ".eqv.", "W",                               &
                            "right arg is an empty mfArray!" )
         go to 99
      end if

      if( (a%data_type /= MF_DT_BOOL .and. a%data_type /= MF_DT_SP_BOOL) .or. &
          (b%data_type /= MF_DT_BOOL .and. b%data_type /= MF_DT_SP_BOOL) ) then
         call PrintMessage( ".eqv.", "E",                               &
                            "args must be logical mfArrays!" )
         go to 99
      end if

      if( a%data_type == MF_DT_SP_BOOL .or. b%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( ".eqv.", "E",                               &
                            "not yet implemented for sparse boolean matrix!" )
         go to 99
      end if

      if( any( a%shape /= b%shape ) ) then
         call PrintMessage( ".eqv.", "E",                               &
                            "args must have the same shape!" )
         go to 99
      end if

      out%data_type = MF_DT_BOOL
      out%shape = a%shape
      allocate( out%double(out%shape(1),out%shape(2)) )

      where( a%double /= FALSE .eqv. b%double /= FALSE )
         out%double = TRUE
      elsewhere
         out%double = FALSE
      end where

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( a, b )

      call msAutoRelease( a, b )

#endif
   end function eqv_mfArray_mfArray
!_______________________________________________________________________
!
   function neqv_mfArray_mfArray( a, b ) result( out )

      type(mfArray), intent(in) :: a, b
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( a, b )

      if( mfIsEmpty(a) ) then
         call PrintMessage( ".neqv.", "W",                              &
                            "left arg is an empty mfArray!" )
         go to 99
      end if

      if( mfIsEmpty(b) ) then
         call PrintMessage( ".neqv.", "W",                              &
                            "right arg is an empty mfArray!" )
         go to 99
      end if

      if( (a%data_type /= MF_DT_BOOL .and. a%data_type /= MF_DT_SP_BOOL) .or. &
          (b%data_type /= MF_DT_BOOL .and. b%data_type /= MF_DT_SP_BOOL) ) then
         call PrintMessage( ".neqv.", "E",                              &
                            "args must be logical mfArrays!" )
         go to 99
      end if

      if( a%data_type == MF_DT_SP_BOOL .or. b%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( ".neqv.", "E",                               &
                            "not yet implemented for sparse boolean matrix!" )
         go to 99
      end if

      if( any( a%shape /= b%shape ) ) then
         call PrintMessage( ".neqv.", "E",                              &
                            "args must have the same shape!" )
         go to 99
      end if

      out%data_type = MF_DT_BOOL
      out%shape = a%shape
      allocate( out%double(out%shape(1),out%shape(2)) )

      where( a%double /= FALSE .neqv. b%double /= FALSE )
         out%double = TRUE
      elsewhere
         out%double = FALSE
      end where

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( a, b )

      call msAutoRelease( a, b )

#endif
   end function neqv_mfArray_mfArray
!_______________________________________________________________________
!
   function mfAll( a, dim ) result( out )

      type(mfArray), intent(in) :: a
      integer, intent(in), optional :: dim
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j, dim_loc

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( a )

      if( mfIsEmpty(a) ) then
         call PrintMessage( "mfAll", "W",                               &
                            "arg is an empty mfArray!" )
         go to 99
      end if

      if( a%data_type /= MF_DT_BOOL ) then
         call PrintMessage( "mfAll", "E",                               &
                            "args must be logical mfArrays!" )
         go to 99
      end if

      if( present(dim) ) then
         dim_loc = dim
      else
         dim_loc = 2 ! default : mfAll works down the columns
      end if

      out%data_type = MF_DT_BOOL
      if( a%shape(1)==1 ) then
         ! A is a row vector -> a scalar is returned
         out%shape = [ 1, 1 ]
         allocate( out%double(1,1) )

         if( all(a%double(1,:)/=0) ) then
            out%double(1,1) = TRUE
         else
            out%double(1,1) = FALSE
         end if
      else if( a%shape(2)==1 ) then
         ! A is a column vector -> a scalar is returned
         out%shape = [ 1, 1 ]
         allocate( out%double(1,1) )

         if( all(a%double(:,1)/=0) ) then
            out%double(1,1) = TRUE
         else
            out%double(1,1) = FALSE
         end if
      else if( a%shape(1)>1 .and. a%shape(2)>1 ) then
         ! A is a matrix -> the result will be ...
         if( dim_loc == 1 ) then
            ! ... a column vector
            out%shape = [ a%shape(1), 1 ]
            allocate( out%double(out%shape(1),out%shape(2)) )

            ! processing rows
            do i = 1, a%shape(1)
               if( all(a%double(i,:)/=0) ) then
                  out%double(i,1) = TRUE
               else
                  out%double(i,1) = FALSE
               end if
            end do
         else
            ! ... a row vector
            out%shape = [ 1, a%shape(2) ]
            allocate( out%double(out%shape(1),out%shape(2)) )

            ! processing columns
            do j = 1, a%shape(2)
               if( all(a%double(:,j)/=0) ) then
                  out%double(1,j) = TRUE
               else
                  out%double(1,j) = FALSE
               end if
            end do
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( a )

      call msAutoRelease( a )

#endif
   end function mfAll
!_______________________________________________________________________
!
   function mfAny( a, dim ) result( out )

      type(mfArray), intent(in) :: a
      integer, intent(in), optional :: dim
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j, dim_loc

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( a )

      if( mfIsEmpty(a) ) then
         call PrintMessage( "mfAny", "W",                               &
                            "arg is an empty mfArray!" )
         go to 99
      end if

      if( a%data_type /= MF_DT_BOOL ) then
         call PrintMessage( "mfAny", "E",                               &
                            "args must be logical mfArrays!" )
         go to 99
      end if

      if( present(dim) ) then
         dim_loc = dim
      else
         dim_loc = 2 ! default : mfAll works down the columns
      end if

      out%data_type = MF_DT_BOOL
      if( a%shape(1)==1 ) then
         ! A is a row vector -> a scalar is returned
         out%shape = [ 1, 1 ]
         allocate( out%double(1,1) )

         if( any(a%double(1,:)/=0) ) then
            out%double(1,1) = TRUE
         else
            out%double(1,1) = FALSE
         end if
      else if( a%shape(2)==1 ) then
         ! A is a column vector -> a scalar is returned
         out%shape = [ 1, 1 ]
         allocate( out%double(1,1) )

         if( any(a%double(:,1)/=0) ) then
            out%double(1,1) = TRUE
         else
            out%double(1,1) = FALSE
         end if
      else if( a%shape(1)>1 .and. a%shape(2)>1 ) then
         ! A is a matrix -> the result will be ...
         if( dim_loc == 1 ) then
            ! ... a column vector
            out%shape = [ a%shape(1), 1 ]
            allocate( out%double(out%shape(1),out%shape(2)) )

            ! processing rows
            do i = 1, a%shape(1)
               if( any(a%double(i,:)/=0) ) then
                  out%double(i,1) = TRUE
               else
                  out%double(i,1) = FALSE
               end if
            end do
         else
            ! ... a row vector
            out%shape = [ 1, a%shape(2) ]
            allocate( out%double(out%shape(1),out%shape(2)) )

            ! processing columns
            do j = 1, a%shape(2)
               if( any(a%double(:,j)/=0) ) then
                  out%double(1,j) = TRUE
               else
                  out%double(1,j) = FALSE
               end if
            end do
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( a )

      call msAutoRelease( a )

#endif
   end function mfAny
!_______________________________________________________________________
!
   function mfCross( a, b ) result( out )

      type(mfArray), intent(in) :: a, b
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! 'a' and 'b' must be real vectors, in dimension 3.

      integer :: idim

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( a, b )

      if( mfIsEmpty(a) .or. mfIsEmpty(b) ) then
         call PrintMessage( "mfCross", "W",                             &
                            "args must not be empty mfArray(s)!" )
         go to 99
      end if

      if( .not. mfIsVector(a) .or. .not. mfIsVector(b) ) then
         call PrintMessage( "mfCross", "E",                             &
                            "args must be vectors!" )
         go to 99
      end if

      if( .not. mfIsReal(a) .or. .not. mfIsReal(b) ) then
         call PrintMessage( "mfCross", "E",                             &
                            "args must be reals!" )
         go to 99
      end if

      if( any( a%shape /= b%shape ) ) then
         call PrintMessage( "mfCross", "E",                             &
                            "args must have the same shape!" )
         go to 99
      end if

      if( a%shape(1) == 1 ) then
         idim = 2
      else
         idim = 1
      end if

      if( a%shape(idim) /= 3 ) then
         call PrintMessage( "mfCross", "E",                             &
                            "vectors must have 3 components!" )
         go to 99
      end if

      out%data_type = MF_DT_DBLE
      out%shape = a%shape
      allocate( out%double(out%shape(1),out%shape(2)) )

      if( idim == 1 ) then
         out%double(1,1) = a%double(2,1)*b%double(3,1) -                &
                           a%double(3,1)*b%double(2,1)
         out%double(2,1) = a%double(3,1)*b%double(1,1) -                &
                           a%double(1,1)*b%double(3,1)
         out%double(3,1) = a%double(1,1)*b%double(2,1) -                &
                           a%double(2,1)*b%double(1,1)
      else
         out%double(1,1) = a%double(1,2)*b%double(1,3) -                &
                           a%double(1,3)*b%double(1,2)
         out%double(1,2) = a%double(1,3)*b%double(1,1) -                &
                           a%double(1,1)*b%double(1,3)
         out%double(1,3) = a%double(1,1)*b%double(1,2) -                &
                           a%double(1,2)*b%double(1,1)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( a, b )

      call msAutoRelease( a, b )

#endif
   end function mfCross
!_______________________________________________________________________
!
   function mfColPerm( A, p ) result( out )

      type(mfArray), intent(in) :: A, p
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! Compute A(:,p), i.e.:
      ! apply the permutation 'p' to the columns of the matrix 'A';
      ! 'p' must be a mfArray vector, of length ncol, the number of
      ! columns of 'A'.

      ! mfArray 'A' may be of any type.

      ! mfArray 'p' may be:
      !  1) a real vector (row or col.) of type MF_DT_DBLE
      !  2) an object of type MF_DT_PERM_VEC (integer col. vector)

      integer, pointer :: perm(:)
      integer :: idim, ncol, nnz

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A, p )

      if( mfIsEmpty(A) .or. mfIsEmpty(p) ) then
         call PrintMessage( "mfColPerm", "W",                           &
                            "args must not be empty mfArray(s)!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfColPerm", "E",                           &
                            "matrix 'A' cannot be a permutation vector!" )
         go to 99
      end if

      if( (.not. (p%data_type == MF_DT_DBLE)) .and.                     &
          (.not. (p%data_type == MF_DT_PERM_VEC)) ) then
         call PrintMessage( "mfColPerm", "E",                           &
                            "arg. p must be a dense real vector or",    &
                            "a integer permutation!" )
         go to 99
      end if

      if( p%shape(1) == 1 ) then
         idim = 2
      else if( p%shape(2) == 1 ) then
         idim = 1
      else
         call PrintMessage( "mfColPerm", "E",                           &
                            "permutation 'p' must be a vector!" )
         go to 99
      end if

      ncol = A%shape(2)
      if( p%shape(idim) /= ncol ) then
         call PrintMessage( "mfColPerm", "E",                           &
                            "permutation 'p' must have the same length than", &
                            "the number of columns of A" )
         go to 99
      end if

      if( p%data_type == MF_DT_DBLE ) then
         allocate( perm(ncol) )
         if( idim == 1 ) then
            perm(:) = nint( p%double(:,1) )
         else
            perm(:) = nint( p%double(1,:) )
         end if
      else ! p%data_type == MF_DT_PERM_VEC
         perm => p%i
      end if

      select case( A%data_type )
         case( MF_DT_DBLE, MF_DT_BOOL )
            out%data_type = A%data_type
            out%shape = A%shape
            allocate( out%double(out%shape(1),out%shape(2)) )

            out%double(:,:) = A%double(:,perm)
         case( MF_DT_CMPLX )
            out%data_type = A%data_type
            out%shape = A%shape
            allocate( out%cmplx(out%shape(1),out%shape(2)) )

            out%cmplx(:,:) = A%cmplx(:,perm)
         case( MF_DT_SP_DBLE )
            out%data_type = A%data_type
            out%shape = A%shape
            nnz = A%j(ncol+1) - 1
            allocate( out%a(nnz) )

            allocate( out%i(nnz) )

            allocate( out%j(ncol+1) )

            call colperm( ncol, A%a, A%i, A%j, out%a, out%i, out%j, perm )
         case( MF_DT_SP_CMPLX )
            out%data_type = A%data_type
            out%shape = A%shape
            nnz = A%j(ncol+1) - 1
            allocate( out%z(nnz) )

            allocate( out%i(nnz) )

            allocate( out%j(ncol+1) )

            call colperm_cmplx( ncol, A%z, A%i, A%j, out%z, out%i, out%j, perm )
         case default
            call PrintMessage( "mfColPerm", "E",                        &
                               "unknown data type for arg. A" )
         go to 99
      end select

      if( p%data_type == MF_DT_DBLE ) then
         deallocate( perm )

      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A, p )

      call msAutoRelease( A, p )

#endif
   end function mfColPerm
!_______________________________________________________________________
!
   subroutine msColPerm( A, p )

      type(mfArray), intent(in out) :: A
      type(mfArray), intent(in) :: p
      !------ API end ------

#ifdef _DEVLP
      ! Compute A(:,p), i.e.:
      ! apply the permutation 'p' to the columns of the matrix 'A';
      ! 'p' must be a mfArray vector, of length ncol, the number of
      ! columns of 'A'.

      ! mfArray 'A' may be of any type.

      ! mfArray 'p' may be:
      !  1) a real vector (row or col.) of type MF_DT_DBLE
      !  2) an object of type MF_DT_PERM_VEC (integer col. vector)

      ! operation is made 'in-place'

      integer, pointer :: perm(:)
      integer :: idim, nrow, ncol

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A, p )

      if( mfIsEmpty(A) .or. mfIsEmpty(p) ) then
         call PrintMessage( "msColPerm", "W",                           &
                            "args must not be empty mfArray(s)!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "msColPerm", "E",                           &
                            "matrix 'A' cannot be a permutation vector!" )
         go to 99
      end if

      if( (.not. (p%data_type == MF_DT_DBLE)) .and.                     &
          (.not. (p%data_type == MF_DT_PERM_VEC)) ) then
         call PrintMessage( "msColPerm", "E",                           &
                            "arg. p must be a dense real vector or",    &
                            "a integer permutation!" )
         go to 99
      end if

      if( p%shape(1) == 1 ) then
         idim = 2
      else if( p%shape(2) == 1 ) then
         idim = 1
      else
         call PrintMessage( "msColPerm", "E",                           &
                            "permutation 'p' must be a vector!" )
         go to 99
      end if

      ncol = A%shape(2)
      if( p%shape(idim) /= ncol ) then
         call PrintMessage( "msColPerm", "E",                           &
                            "permutation 'p' must have the same length than", &
                            "the number of columns of A" )
         go to 99
      end if

      if( p%data_type == MF_DT_DBLE ) then
         allocate( perm(ncol) )

         if( idim == 1 ) then
            perm(:) = nint( p%double(:,1) )
         else
            perm(:) = nint( p%double(1,:) )
         end if
      else ! p%data_type == MF_DT_PERM_VEC
         perm => p%i
      end if

      select case( A%data_type )
         case( MF_DT_DBLE, MF_DT_BOOL )
            nrow = A%shape(1)
            call matcolperm( nrow, ncol, A%double, perm )

         case( MF_DT_CMPLX )
            nrow = A%shape(1)
            call matcolperm_cmplx( nrow, ncol, A%cmplx, perm )

         case( MF_DT_SP_DBLE )
            call colperm2( ncol, A%a, A%i, A%j, perm )

         case( MF_DT_SP_CMPLX )
            call colperm2_cmplx( ncol, A%z, A%i, A%j, perm )

         case default
            call PrintMessage( "msColPerm", "E",                        &
                               "unknown data type for arg. A" )
         go to 99
      end select

      if( p%data_type == MF_DT_DBLE ) then
         deallocate( perm )

      end if

 99   continue

      call msFreeArgs( A, p )

      call msAutoRelease( A, p )

#endif
   end subroutine msColPerm
!_______________________________________________________________________
!
   function mfRowPerm( A, p ) result( out )

      type(mfArray), intent(in) :: A, p
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! Compute A(p,:), i.e.:
      ! apply the permutation 'p' to the rows of the matrix 'A';
      ! 'p' must be a mfArray vector, of length nrow, the number of
      ! rows of 'A'.

      ! mfArray 'A' may be of any type.

      ! mfArray 'p' may be:
      !  1) a real vector (row or col.) of type MF_DT_DBLE
      !  2) an object of type MF_DT_PERM_VEC (integer col. vector)

      integer, pointer :: perm(:)
      integer :: idim, nrow, ncol, nnz

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A, p )

      if( mfIsEmpty(A) .or. mfIsEmpty(p) ) then
         call PrintMessage( "mfRowPerm", "W",                           &
                            "args must not be empty mfArray(s)!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfRowPerm", "E",                           &
                            "matrix 'A' cannot be a permutation vector!" )
         go to 99
      end if

      if( (.not. (p%data_type == MF_DT_DBLE)) .and.                     &
          (.not. (p%data_type == MF_DT_PERM_VEC)) ) then
         call PrintMessage( "mfRowPerm", "E",                           &
                            "arg. p must be a dense real vector or",    &
                            "a integer permutation!" )
         go to 99
      end if

      if( p%shape(1) == 1 ) then
         idim = 2
      else if( p%shape(2) == 1 ) then
         idim = 1
      else
         call PrintMessage( "mfRowPerm", "E",                           &
                            "permutation 'p' must be a vector!" )
         go to 99
      end if

      nrow = A%shape(1)
      if( p%shape(idim) /= nrow ) then
         call PrintMessage( "mfRowPerm", "E",                           &
                            "permutation 'p' must have the same length than", &
                            "the number of rows of A" )
         go to 99
      end if

      if( p%data_type == MF_DT_DBLE ) then
         allocate( perm(nrow) )

         if( idim == 1 ) then
            perm(:) = nint( p%double(:,1) )
         else
            perm(:) = nint( p%double(1,:) )
         end if
      else ! p%data_type == MF_DT_PERM_VEC
         perm => p%i
      end if

      select case( A%data_type )
         case( MF_DT_DBLE, MF_DT_BOOL )
            out%data_type = A%data_type
            out%shape = A%shape
            allocate( out%double(out%shape(1),out%shape(2)) )

            out%double(:,:) = A%double(perm,:)
         case( MF_DT_CMPLX )
            out%data_type = A%data_type
            out%shape = A%shape
            allocate( out%cmplx(out%shape(1),out%shape(2)) )

            out%cmplx(:,:) = A%cmplx(perm,:)
         case( MF_DT_SP_DBLE )
            ncol = A%shape(2)
            out%data_type = A%data_type
            out%shape = A%shape
            nnz = A%j(ncol+1) - 1
            allocate( out%a(nnz) )

            allocate( out%i(nnz) )

            allocate( out%j(ncol+1) )

            call rowperm( ncol, A%a, A%i, A%j, out%a, out%i, out%j, perm )
         case( MF_DT_SP_CMPLX )
            ncol = A%shape(2)
            out%data_type = A%data_type
            out%shape = A%shape
            nnz = A%j(ncol+1) - 1
            allocate( out%z(nnz) )

            allocate( out%i(nnz) )

            allocate( out%j(ncol+1) )

            call rowperm_cmplx( ncol, A%z, A%i, A%j, out%z, out%i, out%j, perm )
         case default
            call PrintMessage( "mfRowPerm", "E",                        &
                               "unknown data type for arg. A" )
         go to 99
      end select

      if( p%data_type == MF_DT_DBLE ) then
         deallocate( perm )

      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A, p )

      call msAutoRelease( A, p )

#endif
   end function mfRowPerm
!_______________________________________________________________________
!
   subroutine msRowPerm( A, p )

      type(mfArray), intent(in out) :: A
      type(mfArray), intent(in) :: p
      !------ API end ------

#ifdef _DEVLP
      ! Compute A(:,p), i.e.:
      ! apply the permutation 'p' to the rows of the matrix 'A';
      ! 'p' must be a mfArray vector, of length nrow, the number of
      ! rows of 'A'.

      ! mfArray 'A' may be of any type.

      ! mfArray 'p' may be:
      !  1) a real vector (row or col.) of type MF_DT_DBLE
      !  2) an object of type MF_DT_PERM_VEC (integer col. vector)

      ! operation is made 'in-place'

      integer, pointer :: perm(:)
      integer :: idim, nrow, ncol

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A, p )

      if( mfIsEmpty(A) .or. mfIsEmpty(p) ) then
         call PrintMessage( "msRowPerm", "W",                           &
                            "args must not be empty mfArray(s)!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "msRowPerm", "E",                           &
                            "matrix 'A' cannot be a permutation vector!" )
         go to 99
      end if

      if( (.not. (p%data_type == MF_DT_DBLE)) .and.                     &
          (.not. (p%data_type == MF_DT_PERM_VEC)) ) then
         call PrintMessage( "msRowPerm", "E",                           &
                            "arg. p must be a dense real vector or",    &
                            "a integer permutation!" )
         go to 99
      end if

      if( p%shape(1) == 1 ) then
         idim = 2
      else if( p%shape(2) == 1 ) then
         idim = 1
      else
         call PrintMessage( "msRowPerm", "E",                           &
                            "permutation 'p' must be a vector!" )
         go to 99
      end if

      nrow = A%shape(1)
      ncol = A%shape(2)
      if( p%shape(idim) /= nrow ) then
         call PrintMessage( "msRowPerm", "E",                           &
                            "permutation 'p' must have the same length than", &
                            "the number of rows of A" )
         go to 99
      end if

      if( p%data_type == MF_DT_DBLE ) then
         allocate( perm(nrow) )

         if( idim == 1 ) then
            perm(:) = nint( p%double(:,1) )
         else
            perm(:) = nint( p%double(1,:) )
         end if
      else ! p%data_type == MF_DT_PERM_VEC
         perm => p%i
      end if

      select case( A%data_type )
         case( MF_DT_DBLE, MF_DT_BOOL )
            ! the following routine is definitely less efficient than
            ! matcolperm() because it permute memory areas which are
            ! not contiguous.
            call matrowperm( nrow, ncol, A%double, perm )

         case( MF_DT_CMPLX )
            ! the following routine is definitely less efficient than
            ! matcolperm() because it permute memory areas which are
            ! not contiguous.
            call matrowperm_cmplx( nrow, ncol, A%cmplx, perm )

         case( MF_DT_SP_DBLE, MF_DT_SP_CMPLX )
            call rowperm2( ncol, A%i, A%j, perm )

         case default
            call PrintMessage( "msRowPerm", "E",                        &
                               "unknown data type for arg. A" )
         go to 99
      end select

      if( p%data_type == MF_DT_DBLE ) then
         deallocate( perm )

      end if

 99   continue

      call msFreeArgs( A, p )

      call msAutoRelease( A, p )

#endif
   end subroutine msRowPerm
!_______________________________________________________________________
!
   function mfInvPerm( p ) result( out )

      type(mfArray), intent(in) :: p
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! inverse the permutation vector 'p'

      integer :: i, n

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( p )

      if( mfIsEmpty(p) ) then
         call PrintMessage( "mfInvPerm", "W",                           &
                            "argument must not be an empty mfArray!" )
         go to 99
      end if

      if( .not. mfIsPerm(p) ) then
         call PrintMessage( "mfInvPerm", "E",                           &
                            "argument must be a permutation vector!" )
         go to 99
      end if

      n = p%shape(1)
      out%data_type = MF_DT_PERM_VEC
      out%shape = [n,1]
      allocate( out%i(n) )

      out%i(p%i(:)) = [ ( i, i = 1, n ) ]

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( p )

      call msAutoRelease( p )

#endif
   end function mfInvPerm
!_______________________________________________________________________
!
   function mfColScale( A, s ) result( out )

      type(mfArray), intent(in) :: A, s
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! Apply a column scaling to A(:,:), i.e.:
      ! multiply column A(:,j) by s(j)

      ! mfArray 'A' must be numeric.
      ! scaling 's' must be real.

      real(kind=MF_DOUBLE), allocatable :: scal(:)
      integer :: j, idim, ncol, nnz

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A, s )

      if( mfIsEmpty(A) .or. mfIsEmpty(s) ) then
         call PrintMessage( "mfColScale", "W",                          &
                            "args must not be empty mfArray(s)!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfColScale", "E",                          &
                            "matrix 'A' cannot be a permutation vector!" )
         go to 99
      end if

      if( .not. mfIsReal(s) ) then
         call PrintMessage( "mfColScale", "E",                          &
                            "arg. s must be a dense real vector!" )
         go to 99
      end if

      if( s%shape(1) == 1 ) then
         idim = 2
      else if( s%shape(2) == 1 ) then
         idim = 1
      else
         call PrintMessage( "mfColScale", "E",                          &
                            "scaling 's' must be a vector!" )
         go to 99
      end if

      ncol = A%shape(2)
      if( s%shape(idim) /= ncol ) then
         call PrintMessage( "mfColScale", "E",                          &
                            "scaling 's' must have the same length than", &
                            "the number of columns of A" )
         go to 99
      end if

      allocate( scal(ncol) )
      if( idim == 1 ) then
         scal(:) = s%double(:,1)
      else
         scal(:) = s%double(1,:)
      end if

      select case( A%data_type )
         case( MF_DT_DBLE, MF_DT_BOOL )
            out%data_type = A%data_type
            out%shape = A%shape
            allocate( out%double(out%shape(1),out%shape(2)) )

            do j = 1, ncol
               out%double(:,j) = scal(j)*A%double(:,j)
            end do
         case( MF_DT_CMPLX )
            out%data_type = A%data_type
            out%shape = A%shape
            allocate( out%cmplx(out%shape(1),out%shape(2)) )

            do j = 1, ncol
               out%cmplx(:,j) = scal(j)*A%cmplx(:,j)
            end do
         case( MF_DT_SP_DBLE )
            out%data_type = A%data_type
            out%shape = A%shape
            nnz = A%j(ncol+1) - 1
            allocate( out%a(nnz) )

            allocate( out%i(nnz) )

            allocate( out%j(ncol+1) )

            call colscale( ncol, A%a, A%i, A%j, out%a, out%i, out%j, scal )
         case( MF_DT_SP_CMPLX )
            out%data_type = A%data_type
            out%shape = A%shape
            nnz = A%j(ncol+1) - 1
            allocate( out%z(nnz) )

            allocate( out%i(nnz) )

            allocate( out%j(ncol+1) )

            call colscale_cmplx( ncol, A%z, A%i, A%j, out%z, out%i, out%j, scal )
         case default
            call PrintMessage( "mfColScale", "E",                       &
                               "unknown data type for arg. A" )
         go to 99
      end select

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A, s )

      call msAutoRelease( A, s )

#endif
   end function mfColScale
!_______________________________________________________________________
!
   subroutine msColScale( A, s )

      type(mfArray), intent(in out) :: A
      type(mfArray), intent(in) :: s
      !------ API end ------

#ifdef _DEVLP
      ! Apply a column scaling to A(:,:), i.e.:
      ! multiply column A(:,j) by s(j)

      ! mfArray 'A' must be numeric.
      ! scaling 's' must be real.

      ! operation is made 'in-place'

      real(kind=MF_DOUBLE), allocatable :: scal(:)
      integer :: j, idim, ncol

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A, s )

      if( mfIsEmpty(A) .or. mfIsEmpty(s) ) then
         call PrintMessage( "msColScale", "W",                          &
                            "args must not be empty mfArray(s)!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "msColScale", "E",                          &
                            "matrix 'A' cannot be a permutation vector!" )
         go to 99
      end if

      if( .not. mfIsReal(s) ) then
         call PrintMessage( "msColScale", "E",                          &
                            "arg. s must be a dense real vector!" )
         go to 99
      end if

      if( s%shape(1) == 1 ) then
         idim = 2
      else if( s%shape(2) == 1 ) then
         idim = 1
      else
         call PrintMessage( "msColScale", "E",                          &
                            "scaling 's' must be a vector!" )
         go to 99
      end if

      ncol = A%shape(2)
      if( s%shape(idim) /= ncol ) then
         call PrintMessage( "msColScale", "E",                          &
                            "scaling 's' must have the same length than", &
                            "the number of columns of A" )
         go to 99
      end if

      allocate( scal(ncol) )
      if( idim == 1 ) then
         scal(:) = s%double(:,1)
      else
         scal(:) = s%double(1,:)
      end if

      select case( A%data_type )
         case( MF_DT_DBLE, MF_DT_BOOL )
            do j = 1, ncol
               A%double(:,j) = scal(j)*A%double(:,j)
            end do

         case( MF_DT_CMPLX )
            do j = 1, ncol
               A%cmplx(:,j) = scal(j)*A%cmplx(:,j)
            end do

         case( MF_DT_SP_DBLE )
            ncol = A%shape(2)
            call colscale2( ncol, A%a, A%i, A%j, scal )

         case( MF_DT_SP_CMPLX )
            ncol = A%shape(2)
            call colscale2_cmplx( ncol, A%z, A%i, A%j, scal )

         case default
            call PrintMessage( "msColScale", "E",                       &
                               "unknown data type for arg. A" )
         go to 99
      end select

 99   continue

      call msFreeArgs( A, s )

      call msAutoRelease( A, s )

#endif
   end subroutine msColScale
!_______________________________________________________________________
!
   function mfRowScale( A, s ) result( out )

      type(mfArray), intent(in) :: A, s
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! Apply a row scaling to A(:,:), i.e.:
      ! multiply row A(j,:) by s(j)

      ! mfArray 'A' must be numeric.
      ! scaling 's' must be real.

      real(kind=MF_DOUBLE), allocatable :: scal(:)
      integer :: i, idim, nrow, ncol, nnz

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A, s )

      if( mfIsEmpty(A) .or. mfIsEmpty(s) ) then
         call PrintMessage( "mfRowScale", "W",                          &
                            "args must not be empty mfArray(s)!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfRowScale", "E",                          &
                            "matrix 'A' cannot be a permutation vector!" )
         go to 99
      end if

      if( .not. mfIsReal(s) ) then
         call PrintMessage( "mfRowScale", "E",                          &
                            "arg. s must be a dense real vector!" )
         go to 99
      end if

      if( s%shape(1) == 1 ) then
         idim = 2
      else if( s%shape(2) == 1 ) then
         idim = 1
      else
         call PrintMessage( "mfRowScale", "E",                          &
                            "scaling 's' must be a vector!" )
         go to 99
      end if

      nrow = A%shape(1)
      if( s%shape(idim) /= nrow ) then
         call PrintMessage( "mfRowScale", "E",                          &
                            "scaling 's' must have the same length than", &
                            "the number of rows of A" )
         go to 99
      end if

      allocate( scal(nrow) )
      if( idim == 1 ) then
         scal(:) = s%double(:,1)
      else
         scal(:) = s%double(1,:)
      end if

      select case( A%data_type )
         case( MF_DT_DBLE, MF_DT_BOOL )
            out%data_type = A%data_type
            out%shape = A%shape
            allocate( out%double(out%shape(1),out%shape(2)) )

            do i = 1, nrow
               out%double(i,:) = scal(i)*A%double(i,:)
            end do
         case( MF_DT_CMPLX )
            out%data_type = A%data_type
            out%shape = A%shape
            allocate( out%cmplx(out%shape(1),out%shape(2)) )

            do i = 1, nrow
               out%cmplx(1,:) = scal(i)*A%cmplx(i,:)
            end do
         case( MF_DT_SP_DBLE )
            ncol = A%shape(2)
            out%data_type = A%data_type
            out%shape = A%shape
            nnz = A%j(ncol+1) - 1
            allocate( out%a(nnz) )

            allocate( out%i(nnz) )

            allocate( out%j(ncol+1) )

            call rowscale( ncol, A%a, A%i, A%j, out%a, out%i, out%j, scal )
         case( MF_DT_SP_CMPLX )
            ncol = A%shape(2)
            out%data_type = A%data_type
            out%shape = A%shape
            nnz = A%j(ncol+1) - 1
            allocate( out%z(nnz) )

            allocate( out%i(nnz) )

            allocate( out%j(ncol+1) )

            call rowscale_cmplx( ncol, A%z, A%i, A%j, out%z, out%i, out%j, scal )
         case default
            call PrintMessage( "mfRowScale", "E",                       &
                               "unknown data type for arg. A" )
         go to 99
      end select

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A, s )

      call msAutoRelease( A, s )

#endif
   end function mfRowScale
!_______________________________________________________________________
!
   subroutine msRowScale( A, s )

      type(mfArray), intent(in out) :: A
      type(mfArray), intent(in) :: s
      !------ API end ------

#ifdef _DEVLP
      ! Apply a row scaling to A(:,:), i.e.:
      ! multiply row A(j,:) by s(j)

      ! mfArray 'A' must be numeric.
      ! scaling 's' must be real.

      ! operation is made 'in-place'

      real(kind=MF_DOUBLE), allocatable :: scal(:)
      integer :: i, idim, nrow, ncol

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A, s )

      if( mfIsEmpty(A) .or. mfIsEmpty(s) ) then
         call PrintMessage( "msRowScale", "W",                          &
                            "args must not be empty mfArray(s)!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "msRowScale", "E",                          &
                            "matrix 'A' cannot be a permutation vector!" )
         go to 99
      end if

      if( .not. mfIsReal(s) ) then
         call PrintMessage( "msRowScale", "E",                          &
                            "arg. s must be a dense real vector!" )
         go to 99
      end if

      if( s%shape(1) == 1 ) then
         idim = 2
      else if( s%shape(2) == 1 ) then
         idim = 1
      else
         call PrintMessage( "msRowScale", "E",                          &
                            "scaling 's' must be a vector!" )
         go to 99
      end if

      nrow = A%shape(1)
      if( s%shape(idim) /= nrow ) then
         call PrintMessage( "msRowScale", "E",                          &
                            "scaling 's' must have the same length than", &
                            "the number of rows of A" )
         go to 99
      end if

      allocate( scal(nrow) )
      if( idim == 1 ) then
         scal(:) = s%double(:,1)
      else
         scal(:) = s%double(1,:)
      end if

      select case( A%data_type )
         case( MF_DT_DBLE, MF_DT_BOOL )
            do i = 1, nrow
               A%double(i,:) = scal(i)*A%double(i,:)
            end do

         case( MF_DT_CMPLX )
            do i = 1, nrow
               A%cmplx(1,:) = scal(i)*A%cmplx(i,:)
            end do

         case( MF_DT_SP_DBLE )
            ncol = A%shape(2)
            call rowscale2( ncol, A%a, A%i, A%j, scal )

         case( MF_DT_SP_CMPLX )
            ncol = A%shape(2)
            call rowscale2_cmplx( ncol, A%z, A%i, A%j, scal )

         case default
            call PrintMessage( "msRowScale", "E",                       &
                               "unknown data type for arg. A" )
         go to 99
      end select

 99   continue

      call msFreeArgs( A, s )

      call msAutoRelease( A, s )

#endif
   end subroutine msRowScale
!_______________________________________________________________________
!
   subroutine msColAutoScale( out, A )

      type(mfArray) :: A

      type(mf_Out) :: out
      !------ API end ------

#ifdef _DEVLP
      ! Scales the columns of A(:,:) in such a way that maximum
      ! magnitude of each col is 1.

      ! Operation is made 'in-place' ; scaling factors are returned
      ! in the vector 's'

      type(mfArray), pointer :: s
      integer :: j, ncol

      character(len=*), parameter :: ROUTINE_NAME = "msColAutoScale"

   !------ end of declarations -- execution starts hereafter  ------

      ! 1 out-arg must be specified
      if( out%n /= 1 ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "one output args required!" )
         return
      end if

      s => out%ptr1
      call msSilentRelease( s )

      if( A%data_type == MF_DT_EMPTY ) then
         call PrintMessage( ROUTINE_NAME, "W",                          &
                            "arg. is empty!" )
         return
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "matrix 'A' cannot be a permutation vector!" )
         return
      end if

      call msInitArgs( A )

      if( .not. mfIsReal(A) ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "A must be a dense real vector!" )
         go to 99
      end if

      ncol = A%shape(2)

      s%data_type = MF_DT_DBLE
      s%shape = [ 1, ncol ]
      allocate( s%double(1,ncol) )

      select case( A%data_type )
         case( MF_DT_DBLE )
            do j = 1, ncol
               s%double(1,j) = maxval( abs( A%double(:,j) ) )
               s%double(1,j) = 1.0d0 / s%double(1,j)
               A%double(:,j) = s%double(1,j) * A%double(:,j)
            end do
         case( MF_DT_SP_DBLE )
            call col_auto_scale2( ncol, A%a, A%i, A%j, s%double(1,:) )
         case default
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "unknown data type for arg. A" )
         go to 99
      end select

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end subroutine msColAutoScale
!_______________________________________________________________________
!
   subroutine msRowAutoScale( out, A )

      type(mfArray) :: A

      type(mf_Out) :: out
      !------ API end ------

#ifdef _DEVLP
      ! Scales the rows of A(:,:) in such a way that maximum
      ! magnitude of each row is 1.

      ! Operation is made 'in-place' ; scaling factors are returned
      ! in the vector 's'

      type(mfArray), pointer :: s
      integer :: i, nrow, ncol

      character(len=*), parameter :: ROUTINE_NAME = "msRowAutoScale"

   !------ end of declarations -- execution starts hereafter  ------

      ! 1 out-arg must be specified
      if( out%n /= 1 ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "one output args required!" )
         return
      end if

      s => out%ptr1
      call msSilentRelease( s )

      if( A%data_type == MF_DT_EMPTY ) then
         call PrintMessage( ROUTINE_NAME, "W",                          &
                            "arg. is empty!" )
         return
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "matrix 'A' cannot be a permutation vector!" )
         return
      end if

      call msInitArgs( A )

      if( .not. mfIsReal(A) ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "A must be a dense real vector!" )
         go to 99
      end if

      nrow = A%shape(1)
      ncol = A%shape(2)

      s%data_type = MF_DT_DBLE
      s%shape = [ nrow, 1 ]
      allocate( s%double(nrow,1) )

      select case( A%data_type )
         case( MF_DT_DBLE )
            do i = 1, nrow
               s%double(i,1) = maxval( abs( A%double(i,:) ) )
               s%double(i,1) = 1.0d0 / s%double(i,1)
               A%double(i,:) = s%double(i,1) * A%double(i,:)
            end do
         case( MF_DT_SP_DBLE )
            call row_auto_scale2( nrow, ncol, A%a, A%i, A%j, s%double(:,1) )
         case default
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "unknown data type for arg. A" )
         go to 99
      end select

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end subroutine msRowAutoScale
!_______________________________________________________________________
!
   function mfUnique( A, order, occurrence, tol ) result( out )

      type(mfArray)                              :: A
      character(len=*),     intent(in), optional :: order, occurrence
      real(kind=MF_DOUBLE), intent(in), optional :: tol
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j, dim, n, n2
      real(kind=MF_DOUBLE), allocatable :: tmp(:)
      integer, allocatable :: ind(:), ind2(:)

      character(len=3) :: order_0, mode
      character(len=5) :: occurrence_0
      real(kind=MF_DOUBLE) :: tol_0

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfUnique", "E",                            &
                            "sparse matrices not handled!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfUnique", "E",                            &
                            "cannot be applied to boolean vectors!" )
         go to 99
      end if

      if( mfIsPerm(A) ) then
         call PrintMessage( "mfUnique", "E",                            &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%data_type = A%data_type

      if( A%shape(1) == 1 ) then
         if( A%shape(2) == 1 ) then
            ! A is a scalar
            out = A
            go to 98
         else ! A%shape(2) > 1
            ! row vector
            dim = 2
         end if
      else ! A%shape(1) > 1
         if( A%shape(2) == 1 ) then
            ! column vector
            dim = 1
         else ! A%shape(2) > 1
            call PrintMessage( "mfUnique", "E",                         &
                               "'A' cannot be a matrix!" )
            go to 99
         end if
      end if

      if( present(order) ) then
         order_0 = order
         if( order_0 /= "asc" .and. order_0 /= "des" .and.              &
             order_0 /= "no" ) then
            call PrintMessage( "mfUnique", "E",                         &
                               "'order' must be equal to 'asc', 'des' or 'no'!" )
            go to 99
         end if
         if( order_0 == "no" ) then
            if( .not. present(occurrence) ) then
               call PrintMessage( "mfUnique", "E",                      &
                                  "when 'order' is equal to 'no', the optional", &
                                  "argument 'occurrence' must be present!" )
               go to 99
            end if
         end if
      else
         order_0 = "asc"
      end if
      if( order_0 == "asc" .or. order_0 == "no" ) then
         mode = "asc"
      else
         mode = "des"
      end if

      if( present(occurrence) ) then
         occurrence_0 = occurrence
         if( occurrence_0 /= "first" .and. occurrence_0 /= "last" ) then
            call PrintMessage( "mfUnique", "E",                         &
                               "'occurrence' must be equal to 'first' or 'last'!" )
            go to 99
         end if
      else
         occurrence_0 = occurrence
      end if

      if( present(tol) ) then
         tol_0 = tol
         if( tol_0 < 0.0d0 ) then
            call PrintMessage( "mfUnique", "E",                         &
                               "'tol' must be a positive real number!" )
            go to 99
         end if
      else
         tol_0 = 0.0d0
      end if

      n = A%shape(dim)
      allocate( tmp(n), ind(n) )
      if( dim == 1 ) then
         tmp(:) = A%double(:,1)
      else ! dim == 2
         tmp(:) = A%double(1,:)
      end if

      if( order_0 == "no" ) then
         ind(:) = [ (i, i = 1, n) ]
         call quick_sort_3( mode, tmp, ind )
      else
         call quick_sort_1( mode, tmp )
      end if

      ! Count the number of unique values
      j = 1
      do i = 2, n
         if( abs(tmp(i)-tmp(i-1)) > tol_0 ) j = j + 1
      end do
      n2 = j

      if( dim == 1 ) then
         out%shape = [ n2, 1 ]
      else ! dim == 2
         out%shape = [ 1, n2 ]
      end if

      allocate( out%double(out%shape(1),out%shape(2)) )

      ! Get the elements
      if( order_0 == "no" ) then
         if( occurrence_0 == "first" ) then
            call fix_order( "asc", tmp, ind )
         else ! "last"
            call fix_order( "des", tmp, ind )
         end if
         allocate( ind2(n2) )
         j = 1
         ind2(j) = ind(1)
         do i = 2, n
            if( abs(tmp(i)-tmp(i-1)) > tol_0 ) then
               j = j + 1
               ind2(j) = ind(i)
            end if
         end do
         call quick_sort_1_int( "asc", ind2 )
         do j = 1, n2
            if( dim == 1 ) then
               out%double(j,1) = A%double(ind2(j),1)
            else ! dim == 2
               out%double(1,j) = A%double(1,ind2(j))
            end if
         end do
      else
         j = 1
         out%double(1,1) = tmp(1)
         do i = 2, n
            if( abs(tmp(i)-tmp(i-1)) > tol_0 ) then
               j = j + 1
               if( dim == 1 ) then
                  out%double(j,1) = tmp(i)
               else ! dim == 2
                  out%double(1,j) = tmp(i)
               end if
            end if
         end do
      end if

 98   continue

      out%prop%symm = FALSE

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

   contains
   !____________________________________________________________________
   !
      subroutine fix_order( mode, tmp, ind )

         character(len=3),     intent(in)     :: mode
         real(kind=MF_DOUBLE), intent(in out) :: tmp(:)
         integer,              intent(in out) :: ind(:)
         !------ API end ------

         ! For a sub-set of tmp(:) containing the same value, the corresponding
         ! indices in ind(:) must be ordered in ascending  or descending order,
         ! according to the value of 'mode'.

         integer :: i, j, n, i_beg, i_end
         logical :: correct

         n = size(tmp)

         i_beg = 1
         i_end = 1
         do i = 2, n

            if( abs(tmp(i)-tmp(i-1)) > tol_0 ) then
               if( i_end-i_beg > 0 ) then
                  ! set of at least 2 elements
                  correct = .true.
                  do j = i_beg+1, i_end
                     if( mode == "asc" ) then
                        if( ind(j) < ind(j-1) ) then
                           correct = .false.
                           exit
                        end if
                     else ! "des"
                        if( ind(j) > ind(j-1) ) then
                           correct = .false.
                           exit
                        end if
                     end if
                  end do
                  if( .not. correct ) then
                     call quick_sort_1_int( mode, ind(i_beg:i_end) )
                  end if
               end if
               i_beg = i
            else
               i_end = i
            end if

         end do

      end subroutine fix_order
   !____________________________________________________________________
   !
#endif
   end function mfUnique
!_______________________________________________________________________
!
end module mod_ops
