module mod_mfdebug

   ! Part of MUESLI Numerical Library
   ! Copyright É. Canot 2003-2025 -- IPR/CNRS

!=======================================================================
!    utilitaries for user-debugging
!    (traces for error/warning)
!=======================================================================

#if defined _INTEL_IFC
   use ifcore, only: tracebackqq ! (muesli_trace)
   use ifport, only: beepqq ! (muesli_trace)
#endif

   implicit none

#ifndef _DEVLP
   private
#endif

!-----------------------------------------------------------------------
!                           global variables
!-----------------------------------------------------------------------

   ! not parameters: these variable can be redefined by the user
   ! (via msSetStdIO)
   integer :: STDERR = 0,                                               &
              STDIN  = 5,                                               &
              STDOUT = 6

#ifndef _OPTIM
   logical :: MF_NUMERICAL_CHECK = .true.
#else
   logical :: MF_NUMERICAL_CHECK = .false.
#endif

   integer :: mf_message_level     = 2       ! cf. 'msSetMsgLevel()'
   integer :: mf_traceback_level   = 1       ! cf. 'msSetTrbLevel()'
   logical :: mf_message_displayed = .false. ! used in 'muesli_trace()'
   logical :: mf_colored_messages = .true.

   logical :: mf_debug_user_bool = .false.

   integer :: mf_fp_rounding_mode = 1 ! cf. 'msSetRoundingMode()'

!-----------------------------------------------------------------------
!                             parameters
!-----------------------------------------------------------------------

   integer, parameter :: MF_SINGLE = kind(1.0)
   integer, parameter :: MF_DOUBLE = kind(1.0d0)

   ! this is actually used only in some checks in QR.inc
   ! (theoretically, we should use a 128bit integer for 64bit indexes
   !  in sparse matrices... I doubt that MUESLI will be used one day
   !  for such enormous matrices)
   integer, parameter :: MF_LONG_INT = selected_int_kind(18) ! 64 bits

   ! MF_ADDRESS is a kind for integers which store addresses from C
   ! MF_UF_LONG is a kind for integers matching UF_long in SuiteSparse
#ifdef _64_BITS
   ! max int in 64 bits is ~ 1.8446744074E+019
   integer, parameter :: MF_ADDRESS = selected_int_kind(18) ! 64 bits
   integer, parameter :: MF_UF_LONG = selected_int_kind(18) ! 64 bits
#else
   ! max int in 32 bits is ~ 2.1474836480E+009
   integer, parameter :: MF_ADDRESS = selected_int_kind(8)  ! 32 bits
   integer, parameter :: MF_UF_LONG = selected_int_kind(8)  ! 32 bits
#endif

   public :: STDERR, STDIN, STDOUT, MF_DOUBLE, MF_ADDRESS, MF_UF_LONG
   public :: MF_NUMERICAL_CHECK

   ! kind for one-byte integer
   integer, parameter :: kind_1 = selected_int_kind(r=2)

   integer(kind=kind_1), parameter :: UNKNOWN = -1,                     &
                                      FALSE   =  0,                     &
                                      TRUE    =  1

   private :: mf_traceback

   public :: msSetColoredMsg, &
             msMuesliTrace

contains
#define COLOR_RED    achar(27)//"[31m"
#define COLOR_ORANGE achar(27)//"[38;5;202m"
#define COLOR_YELLOW achar(27)//"[33m"
#define COLOR_GREY   achar(27)//"[37m"//achar(27)//"[3m"
#define NO_COLOR     achar(27)//"[0m"
!_______________________________________________________________________
!
   subroutine msSetColoredMsg( status )

      character(len=*), intent(in) :: status
      !------ API end ------

#ifdef _DEVLP
      if( status == "on" ) then
         mf_colored_messages = .true.
      else if( status == "off" ) then
         mf_colored_messages = .false.
      else
         write(STDERR,*) "(MUESLI msSetColoredMsg:) ERROR: bad argument value."
         write(STDERR,*) '                          Only "on" and "off" are valid args.'
         stop
      end if

#endif
   end subroutine msSetColoredMsg
!_______________________________________________________________________
!
   subroutine msMuesliTrace( pause )

      character(len=*), intent(in) :: pause
      !------ API end ------

#ifdef _DEVLP
      ! this is a user routine; for internal coding, use muesli_trace()

   !------ end of declarations -- execution starts hereafter  ------

      if( mf_colored_messages ) then
         write(STDERR,*) COLOR_GREY
      else
         write(STDERR,*)
      end if
      call mf_traceback()
      if( mf_colored_messages ) then
         write(STDERR,*) NO_COLOR
      else
         write(STDERR,*)
      end if

      if( pause == "yes" .and. mf_message_level /= 0 ) then
         if( mf_colored_messages ) then
            write(STDERR,*) COLOR_GREY
         else
            write(STDERR,*)
         end if
         write(STDERR,*) "(type [RETURN] to resume)"//NO_COLOR
         read *
         write(STDERR,*)
      end if

#endif
   end subroutine msMuesliTrace
!_______________________________________________________________________
!
   subroutine muesli_trace( pause )

      character(len=*), intent(in) :: pause
      !------ API end ------

#ifdef _DEVLP
      ! this is an internal routine; for a user routine use msMuesliTrace()

   !------ end of declarations -- execution starts hereafter  ------

      ! the current routine must be active only if a message
      ! has been really displayed
      if( .not. mf_message_displayed ) then
         return
      end if

      if( mf_message_level /= 0 ) then
         if( mf_traceback_level == 2 .or.                               &
             ( mf_traceback_level == 1 .and. pause == "yes" ) ) then
            if( mf_colored_messages ) then
               write(STDERR,*) COLOR_GREY
            else
               write(STDERR,*)
            end if
            call mf_traceback()
            if( mf_colored_messages ) then
               write(STDERR,*) NO_COLOR
            else
               write(STDERR,*)
            end if
         else
            if( mf_colored_messages ) then
               write(STDERR,*) COLOR_RED//"[traceback disabled]"//NO_COLOR
            else
               write(STDERR,*) "[traceback disabled]"
            end if
         end if
      end if

      if( pause == "yes" .and. mf_message_level /= 0 ) then
         if( mf_colored_messages ) then
            write(STDERR,*) COLOR_GREY
            write(STDERR,*) "(type [RETURN] to resume)"//NO_COLOR
         else
            write(STDERR,*)
            write(STDERR,*) "(type [RETURN] to resume)"
         end if
         read *
         write(STDERR,*)
      end if

      mf_message_displayed = .false.

#endif
   end subroutine muesli_trace
!_______________________________________________________________________
!
   subroutine mf_traceback( )

#ifdef _DEVLP
#if defined _GNU_GFC
#ifdef _HAVE_SHOW_BACKTRACE
      external :: show_backtrace
#endif
#endif

   ! This is an internal routine, called either by the library itself
   ! (from 'muesli_trace'), either by the user (from 'msMuesliTrace').
   !
   ! In all cases, don't restrict the traceback to the DEBUG mode only,
   ! because the user may employ only a precompiled Muesli library in
   ! OPTIM mode. By adding himself the appropriate flag (usually '-g'),
   ! he can get useful information...

   !------ end of declarations -- execution starts hereafter  ------

#if defined _INTEL_IFC
      write(STDERR,*)
      call tracebackqq( string="  traceback requested from MUESLI:", &
                        user_exit_code=-1 )
      write(STDERR,*)
#elif defined _GNU_GFC
#ifdef _SHOW_BACKTRACE
! SHOW_BACKTRACE is for GCC versions with no BACKTRACE extension;
! This required the static version of libgfortran
      write(STDERR,*)
      write(STDERR,"(A)") "  traceback requested from MUESLI:"
      call flush(STDERR)
      call show_backtrace()
      write(STDERR,*)
#else
#ifdef _STD_BACKTRACE
! As of version 4.8, GCC has the BACKTRACE extension available
! (output is to STDERR)
      write(STDERR,*)
      write(STDERR,"(A)") "  traceback requested from MUESLI:"
      call flush(STDERR)
      call backtrace()
      write(STDERR,*)
#else
      write(STDERR,*) "[no traceback with this compiler]"
#endif
#endif
#else
      write(STDERR,*) "[no traceback with this compiler]"
#endif

#endif
   end subroutine mf_traceback
!_______________________________________________________________________
!
end module mod_mfdebug
