module mod_matfun ! Matrix Functions

   ! Part of MUESLI Numerical Library
   ! Copyright É. Canot 2003-2025 -- IPR/CNRS

!-----------------------------------------------------------------------
!                             used modules
!-----------------------------------------------------------------------

   use mod_specfun

#ifndef _DEVLP
   use mod_mfdebug ! required for the 2nd pass of the double compilation
   use mod_mfarray ! required for the 2nd pass of the double compilation
   use mod_core ! required for the 2nd pass of the double compilation
#endif

   use mod_arpack

   implicit none

#ifndef _DEVLP
   private
#endif

!-----------------------------------------------------------------------
!                             interfaces
!-----------------------------------------------------------------------

! "misc/suitesparse/umf4_f90wrapper.inc" already included via modules

#include "misc/suitesparse/cholmod_f90wrapper.inc"

#include "misc/suitesparse/spqr_f90wrapper.inc"

   interface operator(.i.)
      module procedure mfInv
   end interface operator(.i.)
   !------ API end ------

   interface operator(.ix.)
      module procedure mfLDiv_A
   end interface operator(.ix.)
   !------ API end ------

   interface operator(.xi.)
      module procedure mfRDiv_A
   end interface operator(.xi.)
   !------ API end ------

   interface mfNorm
      module procedure mfNorm_int
      module procedure mfNorm_char
   end interface mfNorm
   !------ API end ------

   interface mfFunm
      module procedure mfFunm_intr
      module procedure mfFunm_user
   end interface mfFunm
   !------ API end ------

   interface msFunm
      module procedure msFunm_intr
      module procedure msFunm_user
   end interface msFunm
   !------ API end ------

   interface mfEigs
      module procedure mfEigs_which
      module procedure mfEigs_sigma_real
      module procedure mfEigs_sigma_cmplx
   end interface mfEigs
   !------ API end ------

   interface msEigs
      module procedure msEigs_which
      module procedure msEigs_sigma_real
      module procedure msEigs_sigma_cmplx
   end interface msEigs
   !------ API end ------

   interface mfSVDS
      module procedure mfSVDS_which
   end interface mfSVDS
   !------ API end ------

   interface msSVDS
      module procedure msSVDS_which
   end interface msSVDS
   !------ API end ------

   interface mfLDiv
      module procedure mfLDiv_two_mf
      module procedure mfLDiv_three_mf
      module procedure mfLDiv_four_mf
      module procedure mfLDiv_mfMatFactor
   end interface mfLDiv
   !------ API end ------

   interface mfRDiv
      module procedure mfRDiv_two_mf
      module procedure mfRDiv_three_mf
      module procedure mfRDiv_four_mf
      module procedure mfRDiv_mfMatFactor
   end interface mfRDiv
   !------ API end ------

   interface mfPowm
      module procedure mfPowm_integer
      module procedure mfPowm_real
   end interface mfPowm
   !------ API end ------

   !--------------------------------------------------------------------

   public :: mfTrace, &
             mfDet, &
             mfSvd, msSvd, &
             mfNorm, &
             mfRank, &
             mfCond, &
             mfRcond, &
             mfCondEst, &
             mfNormEst, &
             mfNormEst1, &
             msLU, &
             msLDLT, &
             mfInv, &
             operator(.i.), &
             mfPseudoInv, &
             mfChol, msChol, &
             mfBalance, msBalance, &
             mfQR, msQR, &
             mfQleft, mfQright, &
             mfEig, msEig, &
             mfHess, msHess, &
             mfSchur, msSchur, &
             mfExpm, &
             mfPowm, &
             mfLogm, &
             mfSqrtm, &
             mfFunm, msFunm, &
             mfLDiv, &
             operator(.ix.), &
             mfRDiv, &
             operator(.xi.), &
             msRref, &
             mfNull, &
             mfOrth, &
             mfEigs, msEigs, &
             mfSVDS, msSVDS, &
             mfIsSymm, &
             mfIsPosDef, &
             mfIsDiagDomCol, &
             mfIsStrictDiagDomCol, &
             mfTolForSymm, &
             mfIsFullRank, &
             mfIsDiag, &
             mfIsTril, &
             mfIsTriu

   public :: msCholSpSymb, &
             msCholSpNum

   private :: mfCheckSymmPattern, &
              mfCheckSymm, &
              mfCheckPosDef, &
              change_pos_def, &
              mfCheckDiagPattern, &
              mfCheckTrilPattern, &
              mfCheckTriuPattern

   private :: mfEigs_which, &
              msEigs_which, &
              mfEigs_sigma_real, &
              mfEigs_sigma_cmplx, &
              msEigs_sigma_real, &
              msEigs_sigma_cmplx

   ! global variable for communication between 'mfCond' and
   ! 'mfIsFullRank'
   real(kind=MF_DOUBLE), private :: MFCOND_LARGEST

   ! global variable for communication between 'msLU_mfMatFactor' and
   ! 'mfCondEst'
   integer, private :: UMF4NUM_SINGULAR_MATRIX

   ! Parameters from the SuiteSparse library
   ! (should be verified at each upgrade of SuiteSparse, though they should
   !  remain identical -- see. UserGuide.pdf of UMFPACK, § 5.11, Error codes)
   integer, parameter :: UMFPACK_STATUS = 0 + 1 ! fortran-based index
   integer, parameter :: UMFPACK_WARNING_singular_matrix = 1

contains
!_______________________________________________________________________
!
#include "fml_matfun/Balance.inc"
!_______________________________________________________________________
!
#include "fml_matfun/Chol.inc"
!_______________________________________________________________________
!
#include "fml_matfun/CondEst.inc"
!_______________________________________________________________________
!
#include "fml_matfun/Eig.inc"
!_______________________________________________________________________
!
#include "fml_matfun/Expm.inc"
!_______________________________________________________________________
!
#include "fml_matfun/Funm.inc"
!_______________________________________________________________________
!
#include "fml_matfun/Hess.inc"
!_______________________________________________________________________
!
#include "fml_matfun/Is_Pos_Def.inc"
!_______________________________________________________________________
!
#include "fml_matfun/LDiv.inc"
!_______________________________________________________________________
!
#include "fml_matfun/LDiv_aux.inc"
!_______________________________________________________________________
!
#include "fml_matfun/Logm.inc"
!_______________________________________________________________________
!
#include "fml_matfun/LDLT.inc"
!_______________________________________________________________________
!
#include "fml_matfun/LU.inc"
!_______________________________________________________________________
!
#include "fml_matfun/Norm.inc"
!_______________________________________________________________________
!
#include "fml_matfun/NormEst.inc"
!_______________________________________________________________________
!
#include "fml_matfun/NormEst1.inc"
!_______________________________________________________________________
!
#include "fml_matfun/Null.inc"
!_______________________________________________________________________
!
#include "fml_matfun/Orth.inc"
!_______________________________________________________________________
!
#include "fml_matfun/Powm.inc"
!_______________________________________________________________________
!
#include "fml_matfun/QR.inc"
!_______________________________________________________________________
!
#include "fml_matfun/Rank.inc"
!_______________________________________________________________________
!
#include "fml_matfun/RCond.inc"
!_______________________________________________________________________
!
#include "fml_matfun/RDiv.inc"
!_______________________________________________________________________
!
#include "fml_matfun/RDiv_aux.inc"
!_______________________________________________________________________
!
#include "fml_matfun/Rref.inc"
!_______________________________________________________________________
!
#include "fml_matfun/Schur.inc"
!_______________________________________________________________________
!
#include "fml_matfun/Sqrtm.inc"
!_______________________________________________________________________
!
#include "fml_matfun/Svd.inc"
!_______________________________________________________________________
!
! cannot be located in mod_sparse !
#include "fml_matfun/mat_vec.inc"
!_______________________________________________________________________
!
#include "fml_matfun/Eigs_which.inc"
!_______________________________________________________________________
!
#include "fml_matfun/Eigs_sigma.inc"
!_______________________________________________________________________
!
#include "fml_matfun/SVDS.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/max_diag_at_a.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/norm.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/symmpattern.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/isdiag.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/istril.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/istriu.inc"
!_______________________________________________________________________
!
   function mfTrace( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, n, nrow, ncol
      real(kind=MF_DOUBLE), allocatable :: diag(:)
      complex(kind=MF_DOUBLE), allocatable :: cdiag(:)

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfTrace", "E",                             &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfTrace", "E",                             &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfTrace", "E",                             &
                            "'A' cannot be a permutation vector!" )
         go to 99
      end if

      nrow = A%shape(1)
      ncol = A%shape(2)
      n = min( nrow, ncol )
      out%shape = [ 1, 1 ]
      if( A%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_DBLE
         allocate( out%double(1,1) )

         out%double(1,1) = 0.0d0
         do i = 1, n
            out%double(1,1) = out%double(1,1) + A%double(i,i)
         end do
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX
         allocate( out%cmplx(1,1) )

         out%cmplx(1,1) = (0.0d0,0.0d0)
         do i = 1, n
            out%cmplx(1,1) = out%cmplx(1,1) + A%cmplx(i,i)
         end do
      else if( A%data_type == MF_DT_SP_DBLE ) then
         allocate( diag(n) )
         call getdia( nrow, ncol, A%a, A%i, A%j, diag, 0 )
         out%data_type = MF_DT_DBLE
         allocate( out%double(1,1) )

         out%double(1,1) = 0.0d0
         do i = 1, n
            out%double(1,1) = out%double(1,1) + diag(i)
         end do
      else if( A%data_type == MF_DT_SP_CMPLX ) then
         allocate( cdiag(n) )
         call getdia_cmplx( nrow, ncol, A%z, A%i, A%j, cdiag, 0 )
         out%data_type = MF_DT_CMPLX
         allocate( out%cmplx(1,1) )

         out%cmplx(1,1) = (0.0d0,0.0d0)
         do i = 1, n
            out%cmplx(1,1) = out%cmplx(1,1) + cdiag(i)
         end do
      end if

      out%prop%symm = TRUE

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfTrace
!_______________________________________________________________________
!
   function mfDet( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, n, lda, info
      integer, allocatable :: ipiv(:)
      type(mfArray) :: A_copy

      integer :: signum

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfDet", "E",                               &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfDet", "E",                               &
                            "sparse matrices not yet handled!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfDet", "E",                               &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfDet", "E",                               &
                            "'A' cannot be a permutation vector!" )
         go to 99
      end if

      ! square matrix ?
      if( A%shape(1) /= A%shape(2) ) then
         call PrintMessage( "mfDet", "E",                               &
                            "'A' must be a square matrix!" )
         go to 99
      end if

      ! making copies because LAPACK overwrites A
      A_copy = A

      out%data_type = a%data_type
      out%shape = [ 1, 1 ]
      if( a%data_type == MF_DT_DBLE ) then
         allocate( out%double(1,1) )

         ! General system -- real square matrix : DGETRF
         n = A_copy%shape(1)
         lda = A_copy%shape(1)
         allocate( ipiv(n) )

         call dgetrf( n, n, A_copy%double(1,1), lda, ipiv(1), info )
         if( info /= 0 ) then
            ! det is null
            out%double(1,1) = 0.0d0
         else

            signum = 1
            do i = 1, n
               if( ipiv(i)>i ) signum = -signum
            end do

            out%double(1,1) = signum
            do i = 1, n
               out%double(1,1) = out%double(1,1) * A_copy%double(i,i)
            end do

         end if
      else if( a%data_type == MF_DT_CMPLX ) then
         allocate( out%cmplx(1,1) )

         ! General system -- complex square matrix : ZGETRF
         n = A_copy%shape(1)
         lda = A_copy%shape(1)
         allocate( ipiv(n) )

         call zgetrf( n, n, A_copy%cmplx(1,1), lda, ipiv(1), info )
         if( info /= 0 ) then
            ! det is null
            out%cmplx(1,1) = (0.0d0,0.0d0)
         else

            signum = 1
            do i = 1, n
               if( ipiv(i)>i ) signum = -signum
            end do

            out%cmplx(1,1) = signum
            do i = 1, n
               out%cmplx(1,1) = out%cmplx(1,1) * A_copy%cmplx(i,i)
            end do

         end if
      end if

      call msSilentRelease( A_copy )

      out%prop%symm = TRUE

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfDet
!_______________________________________________________________________
!
   function mfCond( a ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      type(mfArray) :: S, S_largest, S_smallest, flag
      integer :: m, n
      real(kind=MF_DOUBLE) :: largest, smallest, tol

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfCond", "E",                              &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfCond", "E",                              &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfCond", "E",                              &
                            "'A' cannot be a permutation vector!" )
         go to 99
      end if

      if( A%shape(1) <= 1 .or. A%shape(2) <= 1 ) then
         call PrintMessage( "mfCond", "E",                              &
                            "mfArray 'A' should be a matrix!" )
         go to 99
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ 1, 1 ]
      allocate( out%double(1,1) )

      if( mfIsSparse(A) ) then

         ! calling twice mfSVDS, because use of option "which='BE'"
         ! in ARPACK often leads to bad convergence.

         tol = MF_EPS*10.0d0
         call msSVDS( mfOut(S_largest,flag),                            &
                      A, 1, 'LM', tol=tol )
         if( .not. all(flag) ) then
            out%double(1,1) = MF_NAN
            out%status_temporary = .true.
            call PrintMessage( "mfCond", "E",                           &
                               "ARPACK:",                               &
                               "Cannot compute largest singular value", &
                               "with enough accuracy!" )
            go to 99
         end if
         MFCOND_LARGEST = S_largest%double(1,1)

         call msSVDS( mfOut(S_smallest,flag),                           &
                      A, 1, 'SM', tol=tol )
         if( .not. all(flag) ) then
            out%double(1,1) = MF_NAN
            out%status_temporary = .true.
            call PrintMessage( "mfCond", "E",                           &
                               "ARPACK:",                               &
                               "Cannot compute smallest singular value", &
                               "with enough accuracy!" )
            go to 99
         end if

         if( S_smallest%double(1,1) == 0.0d0 ) then
            out%double(1,1) = MF_INF
         else
            out%double(1,1) = S_largest%double(1,1) / S_smallest%double(1,1)
         end if

         call msSilentRelease( S_largest, S_smallest, flag )

      else

         m = A%shape(1)
         n = A%shape(2)

         call msAssign(S, mfSvd( A ))
         ! first compute Reciprocal Condition Number (for 2-norm)
         largest = S%double(1,1)
         MFCOND_LARGEST = largest
         smallest = S%double( min(m,n), 1 )
         if( largest == 0.0d0 ) then
            out%double(1,1) = 0.0d0
         else
            out%double(1,1) = smallest / largest
         end if
         call msSilentRelease( S )

         ! then compute the inverse
         if( out%double(1,1) == 0.0d0 ) then
            out%double(1,1) = MF_INF
         else
            out%double(1,1) = 1.0d0 / out%double(1,1)
         end if

      end if

      out%prop%symm = TRUE

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfCond
!_______________________________________________________________________
!
   function mfInv( A ) result( out )

      type(mfArray), intent(in) :: A
      type(mfArray)             :: out
      !------ API end ------

#ifdef _DEVLP
      type(mfArray) :: b
      integer :: n, rank

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfInv", "E",                               &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfInv", "E",                               &
                            "sparse matrices not yet handled!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfInv", "E",                               &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfInv", "E",                               &
                            "'A' cannot be a permutation vector!" )
         go to 99
      end if

      if( A%shape(1) <= 1 .or. A%shape(2) <= 1 ) then
         call PrintMessage( "mfInv", "E",                               &
                            "mfArray 'A' should be a matrix!" )
         go to 99
      end if

      n = a%shape(1)

      ! square matrix ?
      if( a%shape(2) /= n ) then
         call PrintMessage( "mfInv", "E",                               &
                            "'A' must be a square matrix!" )
         go to 99
      end if

      ! compute the rank of the matrix A
      rank = mfInt( mfRank(A) )

      if( rank /= n ) then ! Warning, singular matrix

         call PrintMessage( "mfInv", "W",                               &
                            "A is not inversible!",                     &
                            "You could use the Moore-Penrose pseudo-inverse instead", &
                            "(routine 'mfPseudoInv')" )
         call msAssign( out, MF_INF*mfOnes(n,n) )

         out%prop%symm = TRUE

      else ! full rank

         ! solve linear system of the form : A x = b,
         ! where b is the identity matrix
         call msAssign(b, mfEye(n))
         call msAssign(out, mfLDiv( A, b ))
         call msSilentRelease( b )

         out%prop%symm = A%prop%symm
         out%prop%posd = A%prop%posd

      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu
      if( mf_phys_units ) then
! bug de INTEL-ifort !
! (cf. explanation in rational_numbers/rational_numbers.F90, line 72
#if defined _INTEL_IFC
         out%units(:) = rational_neg( a%units(:) )
#else
         out%units(:) = - a%units(:)
#endif
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfInv
!_______________________________________________________________________
!
   function mfPseudoInv( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      type(mfArray) :: U, S, V, SI
      integer :: i, m, n, rank

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfPseudoInv", "E",                         &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfPseudoInv", "E",                         &
                            "sparse matrices not yet handled!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfPseudoInv", "E",                         &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfPseudoInv", "E",                         &
                            "'A' cannot be a permutation vector!" )
         go to 99
      end if

      if( A%shape(1) <= 1 .or. A%shape(2) <= 1 ) then
         call PrintMessage( "mfPseudoInv", "E",                         &
                            "mfArray 'A' should be a matrix!" )
         go to 99
      end if

      m = a%shape(1)
      n = a%shape(2)

      ! compute the rank of the matrix A
      rank = mfInt( mfRank(A) )

      call msSvd( mfOut(U,S,V), A )
      ! construct SI from S
      call msAssign(SI, mfZeros(n,m))
      do i = 1, rank
         SI%double(i,i) = 1.0d0 / S%double(i,i)
      end do
      if( A%data_type == MF_DT_DBLE ) then
         call msAssign(out, mfMul(V,mfMul(SI,.t.U)))
      else if( A%data_type == MF_DT_CMPLX ) then
         call msAssign(out, mfMul(V,mfMul(SI,.h.U)))
      end if
      call msSilentRelease( U, S, V, SI )

      out%prop%symm = A%prop%symm
      out%prop%posd = A%prop%posd

      if( mf_phys_units ) then
! bug de INTEL-ifort !
! (cf. explanation in rational_numbers/rational_numbers.F90, line 72
#if defined _INTEL_IFC
         out%units(:) = rational_neg( a%units(:) )
#else
         out%units(:) = - a%units(:)
#endif
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfPseudoInv
!_______________________________________________________________________
!
   function mfIsSymm( A, option, tol ) result( bool )

      type(mfArray) :: A
      character(len=*), intent(in), optional :: option
      real(kind=MF_DOUBLE), intent(in), optional :: tol
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( present(option) ) then
         if( to_upper(option) /= "PATTERN" ) then
            call PrintMessage( "mfIsSymm", "E",                         &
                               "optional arg. must be equal to 'PATTERN'!" )
            go to 99
         end if
      end if

      bool = .false.

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfIsSymm", "W",                            &
                            "mfArray is empty!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfIsSymm", "E",                            &
                            "'A' cannot be a permutation vector!" )
         go to 99
      end if

      if( A%shape(1) /= A%shape(2) ) then
         go to 99
      end if

      if( A%prop%symm == TRUE ) then
         ! if A is already symmetric, it will remain symmetric for
         ! any tolerance
         bool = .true.
      else if( A%prop%symm == UNKNOWN .or. present(tol) ) then
         ! for a sparse matrix, check first if pattern is symmetric
         if( mfIsSparse(A) ) then
            bool = mfCheckSymmPattern(A)
            if( .not. bool ) go to 99
         end if
         if( present(option) ) then
            if( to_upper(option) == "PATTERN" ) then
               go to 99
            end if
         end if
         ! be careful: a matrix not symmetric may become symmetric for
         ! a less strong tolerance!
         if( present(tol) ) then
            bool = mfCheckSymm( A, tol )
         else
            bool = mfCheckSymm( A )
         end if
      else ! FALSE
         if( present(option) ) then
            if( to_upper(option) == "PATTERN" ) then
               if( mfIsSparse(A) ) then
                  bool = mfCheckSymmPattern(A)
               else
                  call PrintMessage( "mfIsSymm", "E",                   &
                                     "optional arg. 'PATTERN' must be present only for sparse matrices!" )
               end if
            end if
            go to 99
         end if
      end if

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfIsSymm
!_______________________________________________________________________
!
   function mfCheckSymmPattern( A ) result( bool )

      type(mfArray) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      ! internal routine
      ! works only for a sparse matrix.

      integer :: nrow, ncol

   !------ end of declarations -- execution starts hereafter  ------

      ! first, matrix must be sorted and must not contain duplicated
      ! entries
      if( .not. mfIsRowSorted(A) ) then
         call msRowSort(A)
      end if

      nrow = A%shape(1)
      ncol = A%shape(2)
      bool = symmpattern( nrow, ncol, A%i, A%j )

#endif
   end function mfCheckSymmPattern
!_______________________________________________________________________
!
   function mfCheckSymm( A, tol ) result( bool )

      type(mfArray) :: A
      real(kind=MF_DOUBLE), intent(in), optional :: tol
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      ! internal routine;
      ! it doesn't depend on the structure of the matrix,
      ! dense or sparse.

      real(kind=MF_DOUBLE) :: norm_A, norm_D, norm_DI, tol0
      type(mfArray) :: A_minus_t_A, D, DI
      logical :: mfnorm_int_symm

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( present(tol) ) then
         tol0 = tol
         if( tol0 <= 0.0d0 ) then
            tol0 = MF_EPS
         end if
      else
         tol0 = MF_EPS
      end if

      if( A%data_type == MF_DT_DBLE .or.                                &
          A%data_type == MF_DT_SP_DBLE ) then
         ! use of norm "1" which is more economic than norm "2" !
         ! -> with a new optional arg., we are able to know
         ! the symmetry of A (can avoid, in some cases, a redonduncy)
         if( A%data_type == MF_DT_DBLE ) then
            norm_A = mfDble( mfNorm_int(A,1,mfnorm_int_symm,A_minus_t_A) )
            if( mfIsEmpty(A_minus_t_A) ) then
               bool = mfnorm_int_symm
            else
               if( norm_A == 0.0d0 .or. mfnorm_int_symm ) then
                  bool = .true.
               else
                  bool = mfDble( mfNorm(A_minus_t_A,1) / norm_A ) < 2.0d0*tol0
               end if
               call msRelease( A_minus_t_A )
            end if
         else ! sparse
            norm_A = mfDble( mfNorm_int(A,1) )
            if( norm_A == 0.0d0 ) then
               bool = .true.
            else
               bool = mfDble( mfNorm(A-(.t.A),1) / norm_A ) < 2.0d0*tol0
            end if
         end if
         if( A%level_locked == 0 .and. .not. A%status_restricted .and.  &
             .not. present(tol) ) then
            if( bool ) then
               A%prop%symm = TRUE
            else
               A%prop%symm = FALSE
            end if
         else
            if( A%level_locked > 0 ) then
               call PrintMessage( "mfCheckSymm", "I",                   &
                                  "cannot update the 'symm' property!", &
                                  "[mfArray has its properties locked,", &
                                  " because you can modify it via 'msPointer']" )
            end if
            if( A%status_restricted ) then
               call PrintMessage( "mfCheckSymm", "I",                   &
                                  "cannot update the 'symm' property!", &
                                  "[mfArray has its properties locked,", &
                                  " because you can modify it via 'msEquiv']" )
            end if
         end if
      else if( A%data_type == MF_DT_CMPLX .or.                          &
               A%data_type == MF_DT_SP_CMPLX ) then
         ! use of norm "1" which is more economic than norm "2" !
         ! -> with a new optional arg., we are able to know
         ! the symmetry of A (can avoid, in some cases, a redonduncy)
         ! ( actually, in the following call, mfNorm_int returns :
         !  A-(.h.A) )

         call msAssign( D, mfDiag(A) )
         norm_D = mfNorm(D,1)
         if( norm_D /= 0.0d0 ) then
            call msAssign( DI, mfImag(D) )
            norm_DI = mfNorm(DI,1)
            ! quick test : main diagonal must have a null imaginary part
            if( norm_DI / norm_D >= 2.0d0*tol0 ) then
               bool = .false.
            else
               if( A%data_type == MF_DT_CMPLX ) then
                  norm_A = mfDble( mfNorm_int(A,1,mfnorm_int_symm,A_minus_t_A) )
                  if( mfIsEmpty(A_minus_t_A) ) then
                     bool = mfnorm_int_symm
                  else
                     if( norm_A == 0.0d0 .or. mfnorm_int_symm ) then
                        bool = .true.
                     else
                        bool = mfDble( mfNorm(A_minus_t_A,1) / norm_A ) < 2.0d0*tol0
                     end if
                     call msRelease( A_minus_t_A )
                  end if
               else ! sparse
                  norm_A = mfDble( mfNorm_int(A,1) )
                  if( norm_A == 0.0d0 ) then
                     bool = .true.
                  else
                     bool = mfDble( mfNorm(A-(.h.A),1) / norm_A ) < 2.0d0*tol0
                  end if
               end if
            end if
         end if
         call msRelease( D, DI )

         if( A%level_locked == 0 .and. .not. A%status_restricted .and.  &
             .not. present(tol) ) then
            if( bool ) then
               A%prop%symm = TRUE
            else
               A%prop%symm = FALSE
            end if
         else
            if( A%level_locked > 0 ) then
               call PrintMessage( "mfCheckSymm", "I",                   &
                                  "cannot update the 'symm' property!", &
                                  "[mfArray has its properties locked,", &
                                  " because you can modify it via 'msPointer']" )
            end if
            if( A%status_restricted ) then
               call PrintMessage( "mfCheckSymm", "I",                   &
                                  "cannot update the 'symm' property!", &
                                  "[mfArray has its properties locked,", &
                                  " because you can modify it via 'msEquiv']" )
            end if
         end if
      else
         call PrintMessage( "mfCheckSymm", "E",                         &
                            "unknown data type!" )
         go to 99
      end if

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfCheckSymm
!_______________________________________________________________________
!
   function mfTolForSymm( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      real(kind=MF_DOUBLE) :: norm_A

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfTolForSymm", "E",                             &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      norm_A = mfDble( mfNorm_int(A,1) )
      if( norm_A == 0.0d0 ) then
         out = MF_NAN
      else
         if( mfIsReal(A) ) then
            call msAssign( out, mfNorm(A-(.t.A),1) / (2.0d0*norm_A) )
         else if( mfIsComplex(A) ) then
            call msAssign( out, mfNorm(A-(.h.A),1) / (2.0d0*norm_A) )
         end if
      end if

      out%prop%symm = TRUE

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfTolForSymm
!_______________________________________________________________________
!
   function mfIsPosDef( A ) result( bool )

      type(mfArray) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      bool = .false.

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfIsPosDef", "W",                          &
                            "mfArray is empty!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfIsPosDef", "E",                          &
                            "'A' cannot be a permutation vector!" )
         go to 99
      end if

      if( A%shape(1) /= A%shape(2) ) then
         call PrintMessage( "mfIsPosDef", "E",                          &
                            "matrix should be square!" )
         go to 99
      end if

      if( A%prop%posd == TRUE ) then
         bool = .true.
      else if( A%prop%posd == UNKNOWN ) then
         bool = mfCheckPosDef(A)
      end if

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfIsPosDef
!_______________________________________________________________________
!
   function mfCheckPosDef( A ) result( bool )

      type(mfArray) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      ! internal routine;
      ! here, A is a square matrix

      ! from: http://mathworld.wolfram.com/PositiveDefiniteMatrix.html, for
      ! unsymmetric matrices, it is sufficient that the symmetric part: A+A'
      ! is pos. definite.
      ! (but this is disputed by the wikipedia article at:
      !  http://en.wikipedia.org/wiki/Positive-definite_matrix)

      type(mfArray) :: A_sym

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( A%data_type == MF_DT_DBLE ) then

         if( mfIsSymm(A) ) then
            !... check if A is Pos. Def.
            bool = is_pos_def_real( A )
         else
            call msAssign(A_sym, A + .t. A)
            A_sym%prop%symm = TRUE
            !... check if A_sym is Pos. Def.
            bool = is_pos_def_real( A_sym )
         end if

         if( A%level_locked == 0 .and. .not. A%status_restricted ) then
            if( bool ) then
               A%prop%posd = TRUE
            else
               A%prop%posd = FALSE
            end if
         else
            if( A%level_locked > 0 ) then
               call PrintMessage( "mfCheckPosDef", "I",                 &
                                  "cannot update the 'pos_def' property!", &
                                  "[mfArray has its properties locked,", &
                                  " because you can modify it via 'msPointer'!" )
            end if
            if( A%status_restricted ) then
               call PrintMessage( "mfCheckPosDef", "I",                 &
                                  "cannot update the 'pos_def' property!", &
                                  "[mfArray has its properties locked,", &
                                  " because you can modify it via 'msEquiv'!" )
            end if
         end if

      else if( A%data_type == MF_DT_CMPLX ) then

         if( mfIsSymm(A) ) then
            !... check if A is Pos. Def.
            bool = is_pos_def_complex( A )
         else
            call msAssign(A_sym, A + .h. A)
            A_sym%prop%symm = TRUE
            !... check if A_sym is Pos. Def.
            bool = is_pos_def_complex( A_sym )
         end if

         if( A%level_locked == 0 .and. .not. A%status_restricted ) then
            if( bool ) then
               A%prop%posd = TRUE
            else
               A%prop%posd = FALSE
            end if
         else
            if( A%level_locked > 0 ) then
               call PrintMessage( "mfCheckPosDef", "I",                 &
                                  "cannot update the 'pos_def' property!", &
                                  "[mfArray has its properties locked,", &
                                  " because you can modify it via 'msPointer'!" )
            end if
            if( A%status_restricted ) then
               call PrintMessage( "mfCheckPosDef", "I",                 &
                                  "cannot update the 'pos_def' property!", &
                                  "[mfArray has its properties locked,", &
                                  " because you can modify it via 'msEquiv'!" )
            end if
         end if

      else if( A%data_type == MF_DT_SP_DBLE ) then

         if( mfIsSymm(A) ) then
            !... check if A is Pos. Def.
            bool = is_pos_def_sp_real( A )
         else
            call msAssign(A_sym, A + .t. A)
            A_sym%prop%symm = TRUE
            !... check if A_sym is Pos. Def.
            bool = is_pos_def_sp_real( A_sym )
         end if

         if( A%level_locked == 0 .and. .not. A%status_restricted ) then
            if( bool ) then
               A%prop%posd = TRUE
            else
               A%prop%posd = FALSE
            end if
         end if
         if( A%level_locked > 0 ) then
            call PrintMessage( "mfCheckPosDef", "W",                    &
                               "mfArray has its properties locked,",    &
                               "because you can modify it via 'msPointer'!" )
         end if

      else if( A%data_type == MF_DT_SP_CMPLX ) then

         if( mfIsSymm(A) ) then
            !... check if A is Pos. Def.
            bool = is_pos_def_sp_complex( A )
         else
            call msAssign(A_sym, A + .h. A)
            A_sym%prop%symm = TRUE
            !... check if A_sym is Pos. Def.
            bool = is_pos_def_sp_complex( A_sym )
         end if

         if( A%level_locked == 0 .and. .not. A%status_restricted ) then
            if( bool ) then
               A%prop%posd = TRUE
            else
               A%prop%posd = FALSE
            end if
         end if
         if( A%level_locked > 0 ) then
            call PrintMessage( "mfCheckPosDef", "W",                    &
                               "mfArray has its properties locked,",    &
                               "because you can modify it via 'msPointer'!" )
         end if

      end if

      call msSilentRelease( A_sym )

!! 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfCheckPosDef
!_______________________________________________________________________
!
   function mfIsDiagDomCol( A ) result( bool )

      type(mfArray) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      ! Is the matrix A is diagonally dominant by columns ?
      !
      ! real matrix only

      integer :: m, n, i, j, k
      real(kind=MF_DOUBLE) :: diag, offdiag

   !------ end of declarations -- execution starts hereafter  ------

      bool = .false.

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfIsDiagDomCol", "W",                      &
                            "mfArray is empty!" )
         go to 99
      end if

      if( .not. mfIsReal(A) ) then
         call PrintMessage( "mfIsDiagDomCol", "E",                      &
                            "mfArray must be real!" )
         go to 99
      end if

      m = A%shape(1)
      n = A%shape(2)
      if( m /= n ) then
         call PrintMessage( "mfIsDiagDomCol", "E",                      &
                            "matrix should be square!" )
         go to 99
      end if

      if( A%data_type == MF_DT_DBLE ) then
         do j = 1, n
            diag = abs(A%double(j,j))
            offdiag = sum( abs(A%double(:,j)) ) - diag
            if( diag < offdiag ) go to 99
         end do
      else if( A%data_type == MF_DT_SP_DBLE ) then
         do j = 1, n
            offdiag = 0.0d0
            do k = A%j(j), A%j(j+1)-1
               i = A%i(k)
               if( i == j ) then
                  diag = A%a(k)
               else
                  offdiag = offdiag + A%a(k)
               end if
            end do
            if( diag < offdiag ) go to 99
         end do
      end if

      bool = .true.

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfIsDiagDomCol
!_______________________________________________________________________
!
   function mfIsStrictDiagDomCol( A ) result( bool )

      type(mfArray) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      ! Is the matrix A is strictly diagonally dominant by columns ?
      !
      ! real matrix only

      integer :: m, n, i, j, k
      real(kind=MF_DOUBLE) :: diag, offdiag

   !------ end of declarations -- execution starts hereafter  ------

      bool = .false.

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfIsStrictDiagDomCol", "W",                &
                            "mfArray is empty!" )
         go to 99
      end if

      if( .not. mfIsReal(A) ) then
         call PrintMessage( "mfIsStrictDiagDomCol", "E",                &
                            "mfArray must be real!" )
         go to 99
      end if

      m = A%shape(1)
      n = A%shape(2)
      if( m /= n ) then
         call PrintMessage( "mfIsStrictDiagDomCol", "E",                &
                            "matrix should be square!" )
         go to 99
      end if

      if( A%data_type == MF_DT_DBLE ) then
         do j = 1, n
            diag = abs(A%double(j,j))
            offdiag = sum( abs(A%double(:,j)) ) - diag
            if( diag <= offdiag ) go to 99
         end do
      else if( A%data_type == MF_DT_SP_DBLE ) then
         do j = 1, n
            offdiag = 0.0d0
            do k = A%j(j), A%j(j+1)-1
               i = A%i(k)
               if( i == j ) then
                  diag = A%a(k)
               else
                  offdiag = offdiag + A%a(k)
               end if
            end do
            if( diag <= offdiag ) go to 99
         end do
      end if

      bool = .true.

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfIsStrictDiagDomCol
!_______________________________________________________________________
!
   function mfIsFullRank( A, tol ) result( bool )

      type(mfArray) :: A
      real(kind=MF_DOUBLE), intent(in), optional :: tol
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      real(kind=MF_DOUBLE) :: C, etol

   !------ end of declarations -- execution starts hereafter  ------

      bool = .false.

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfIsFullRank", "W",                        &
                            "mfArray is empty!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfIsFullRank", "E",                        &
                            "'A' cannot be a permutation vector!" )
         go to 99
      end if

      if( mfIsVector(A) ) then
         bool = .true.
         go to 99
      end if

      C = mfCond(A)
      if( present(tol) ) then
         if( C < 1.0d0/tol ) then
            bool = .true.
         end if
      else
         ! MFCOND_LARGEST is initialized in mfCond(), called just before
         etol = maxval(A%shape) * MFCOND_LARGEST * MF_EPS
         if( C < 1.0d0/etol ) then
            bool = .true.
         end if
      end if

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfIsFullRank
!_______________________________________________________________________
!
   subroutine change_pos_def( A, val )

      type(mfArray) :: A
      integer(kind=kind_1), intent(in) :: val
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      A%prop%posd = val

#endif
   end subroutine change_pos_def
!_______________________________________________________________________
!
   function mfIsDiag( A ) result( bool )

      type(mfArray) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      ! user-level test routine for diag pattern.
      !
      ! at the end of this routine, the 'prop' flag may have
      ! been changed or not (via mfCheckDiagPattern).

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      bool = .false.

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfIsDiag", "W",                            &
                            "mfArray is empty!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfIsDiag", "E",                            &
                            "'A' cannot be a permutation vector!" )
         go to 99
      end if

      if( A%prop%tril == TRUE .and. A%prop%triu == TRUE ) then
         bool = .true.
      else if( A%prop%tril == TRUE .and. A%prop%triu == UNKNOWN ) then
         bool = mfCheckTriuPattern( A )
      else if( A%prop%tril == UNKNOWN .and. A%prop%triu == TRUE ) then
         bool = mfCheckTrilPattern( A )
      else if( A%prop%tril == UNKNOWN .or. A%prop%triu == UNKNOWN ) then
         bool = mfCheckDiagPattern( A )
      end if

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfIsDiag
!_______________________________________________________________________
!
   function mfCheckDiagPattern( A ) result( bool )

      type(mfArray) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      ! internal routine, for both dense and sparse matrices.
      !
      ! if necessary, changes the 'prop' flag of mfArray A.

      integer :: nrow, ncol, i, j

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      bool = .true.

      nrow = A%shape(1)
      ncol = A%shape(2)

      if( mfIsSparse(A) ) then
         bool = isdiag( nrow, ncol, A%i, A%j )
      else ! A is dense
         if( mfIsReal(A) ) then
 loop_1:    do j = 1, ncol
               do i = 1, nrow
                  if( i == j ) cycle
                  if( A%double(i,j) /= 0.0d0 ) then
                     bool = .false.
                     exit loop_1
                  end if
               end do
            end do loop_1
         else ! A is complex
 loop_2:    do j = 1, ncol
               do i = 1, nrow
                  if( i == j ) cycle
                  if( A%cmplx(i,j) /= (0.0d0,0.0d0) ) then
                     bool = .false.
                     exit loop_2
                  end if
               end do
            end do loop_2
         end if
      end if

      if( A%level_locked == 0 .and. .not. A%status_restricted ) then
         if( bool ) then
            A%prop%tril = TRUE
            A%prop%triu = TRUE
         end if
      else
         if( A%level_locked > 0 ) then
            call PrintMessage( "mfCheckDiagPattern", "I",               &
                               "cannot update pattern properties!",     &
                               "[mfArray has its properties locked,",   &
                               " because you can modify it via 'msPointer']" )
         end if
         if( A%status_restricted ) then
            call PrintMessage( "mfCheckDiagPattern", "I",               &
                               "cannot update pattern properties!",     &
                               "[mfArray has its properties locked,",   &
                               " because you can modify it via 'msEquiv']" )
         end if
      end if

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfCheckDiagPattern
!_______________________________________________________________________
!
   function mfIsTril( A ) result( bool )

      type(mfArray) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      ! user-level test routine for diag pattern.
      !
      ! at the end of this routine, the 'pattern' flag may have
      ! been changed or not (via mfCheckDiagPattern).

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      bool = .false.

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfIsTril", "W",                            &
                            "mfArray is empty!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfIsTril", "E",                            &
                            "'A' cannot be a permutation vector!" )
         go to 99
      end if

      if( A%prop%tril == TRUE ) then
         bool = .true.
      else if( A%prop%tril == UNKNOWN ) then
         bool = mfCheckTrilPattern( A )
      end if

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfIsTril
!_______________________________________________________________________
!
   function mfCheckTrilPattern( A ) result( bool )

      type(mfArray) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      ! internal routine, for both dense and sparse matrices.
      !
      ! if necessary, changes the 'pattern' flag of mfArray A.

      integer :: nrow, ncol, i, j
      integer :: row_sorted

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      bool = .true.

      nrow = A%shape(1)
      ncol = A%shape(2)

      if( mfIsSparse(A) ) then
         ! we should be able to use the 'transfer' Fortran 90 intrinsic,
         ! but... some compilers bugs (e.g. valgrind claims an "uninitialized
         ! value" for GNU-gfortran 4.6.2)
         row_sorted = A%row_sorted
         bool = istril( nrow, ncol, A%i, A%j, row_sorted )
      else ! A is dense
         if( mfIsReal(A) ) then
 loop_1:    do j = 2, min(nrow,ncol)
               do i = 1, j-1
                  if( A%double(i,j) /= 0.0d0 ) then
                     bool = .false.
                     exit loop_1
                  end if
               end do
            end do loop_1
            do j = min(nrow,ncol), ncol
               if( any(A%double(:,j) /= 0.0d0) ) then
                  bool = .false.
                  exit
               end if
            end do
         else ! A is complex
 loop_2:    do j = 2, min(nrow,ncol)
               do i = 1, j-1
                  if( A%cmplx(i,j) /= (0.0d0,0.0d0) ) then
                     bool = .false.
                     exit loop_2
                  end if
               end do
            end do loop_2
            do j = min(nrow,ncol), ncol
               if( any(A%cmplx(:,j) /= (0.0d0,0.0d0)) ) then
                  bool = .false.
                  exit
               end if
            end do
         end if
      end if

      if( A%level_locked == 0 .and. .not. A%status_restricted ) then
         if( bool ) then
            A%prop%tril = TRUE
         else
            A%prop%tril = FALSE
         end if
      else
         if( A%level_locked > 0 ) then
            call PrintMessage( "mfCheckTrilPattern", "I",               &
                               "cannot update pattern properties!",     &
                               "[mfArray has its properties locked,",   &
                               " because you can modify it via 'msPointer']" )
         end if
         if( A%status_restricted ) then
            call PrintMessage( "mfCheckTrilPattern", "I",               &
                               "cannot update pattern properties!",     &
                               "[mfArray has its properties locked,",   &
                               " because you can modify it via 'msEquiv']" )
         end if
      end if

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfCheckTrilPattern
!_______________________________________________________________________
!
   function mfIsTriu( A ) result( bool )

      type(mfArray) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      ! user-level test routine for diag pattern.
      !
      ! at the end of this routine, the 'pattern' flag may have
      ! been changed or not (via mfCheckDiagPattern).

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      bool = .false.

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfIsTriu", "W",                            &
                            "mfArray is empty!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfIsTriu", "E",                            &
                            "'A' cannot be a permutation vector!" )
         go to 99
      end if

      if( A%prop%triu == TRUE ) then
         bool = .true.
      else if( A%prop%triu == UNKNOWN ) then
         bool = mfCheckTriuPattern( A )
      end if

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfIsTriu
!_______________________________________________________________________
!
   function mfCheckTriuPattern( A ) result( bool )

      type(mfArray) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      ! internal routine, for both dense and sparse matrices.
      !
      ! if necessary, changes the 'pattern' flag of mfArray A.

      integer :: nrow, ncol, i, j
      integer :: row_sorted

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      bool = .true.

      nrow = A%shape(1)
      ncol = A%shape(2)

      if( mfIsSparse(A) ) then
         ! we should be able to use the 'transfer' Fortran 90 intrinsic,
         ! but... some compilers bugs (e.g. valgrind claims an "uninitialized
         ! value" for GNU-gfortran 4.6.2)
         row_sorted = A%row_sorted
         bool = istriu( nrow, ncol, A%i, A%j, row_sorted )
      else ! A is dense
         if( mfIsReal(A) ) then
 loop_1:    do j = 1, min(nrow,ncol)-1
               do i = j+1, nrow
                  if( A%double(i,j) /= 0.0d0 ) then
                     bool = .false.
                     exit loop_1
                  end if
               end do
            end do loop_1
         else ! A is complex
 loop_2:    do j = 1, min(nrow,ncol)-1
               do i = j+1, nrow
                  if( A%cmplx(i,j) /= (0.0d0,0.0d0) ) then
                     bool = .false.
                     exit loop_2
                  end if
               end do
            end do loop_2
         end if
      end if

      if( A%level_locked == 0 .and. .not. A%status_restricted ) then
         if( bool ) then
            A%prop%triu = TRUE
         else
            A%prop%triu = FALSE
         end if
      else
         if( A%level_locked > 0 ) then
            call PrintMessage( "mfCheckTriuPattern", "I",               &
                               "cannot update pattern properties!",     &
                               "[mfArray has its properties locked,",   &
                               " because you can modify it via 'msPointer']" )
         end if
         if( A%status_restricted ) then
            call PrintMessage( "mfCheckTriuPattern", "I",               &
                               "cannot update pattern properties!",     &
                               "[mfArray has its properties locked,",   &
                               " because you can modify it via 'msEquiv']" )
         end if
      end if

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfCheckTriuPattern
!_______________________________________________________________________
!
end module mod_matfun
