module mod_funfun ! Function functions

   ! Part of MUESLI Numerical Library
   ! Copyright É. Canot 2003-2025 -- IPR/CNRS

!-----------------------------------------------------------------------
!                             used modules
!-----------------------------------------------------------------------

   use mod_matfun

   use mod_fileio

#ifndef _DEVLP
   use mod_mfarray ! required for the 2nd pass of the double compilation
   use mod_core ! required for the 2nd pass of the double compilation
#endif

   use mod_slatec_solve

   ! don't import the whole module, because this will lead errors in
   ! the tests (or at least, rename some too short variables, to avoid
   ! clash with other symbols).
   use mod_ddebd2, ddebd2_nnz => nnz

   ! inside routines:
   !   use minpack

   implicit none

#ifndef _DEVLP
   private
#endif

   interface mfTrapz
      module procedure mfTrapz_one_arg
      module procedure mfTrapz_two_args
   end interface mfTrapz
   !------ API end ------

   interface mfCumTrapz
      module procedure mfCumTrapz_one_arg
      module procedure mfCumTrapz_two_args
   end interface mfCumTrapz
   !------ API end ------

   interface mfSimpson
      module procedure mfSimpson_one_arg
      module procedure mfSimpson_two_args
   end interface mfSimpson
   !------ API end ------

   interface mfDblQuad
      module procedure mfDblQuad_cte
      module procedure mfDblQuad_funx
      module procedure mfDblQuad_funy
   end interface mfDblQuad
   !------ API end ------

   interface msDblQuad
      module procedure msDblQuad_cte
      module procedure msDblQuad_funx
      module procedure msDblQuad_funy
   end interface msDblQuad
   !------ API end ------

   interface mfOdeSolve
      module procedure mfOdeSolve_JacDF
      module procedure mfOdeSolve_JacUser
      module procedure mfOdeSolve_JacUserSP
   end interface mfOdeSolve
   !------ API end ------

   interface msOdeSolve
      module procedure msOdeSolve_JacDF
      module procedure msOdeSolve_JacUser
      module procedure msOdeSolve_JacUserSP
      module procedure msOdeSolve_SaveRest_full
      module procedure msOdeSolve_SaveRest_SP
      module procedure msOdeSolve_Finalize
   end interface msOdeSolve
   !------ API end ------

   interface mfDaeSolve
      module procedure mfDaeSolve_JacDF
      module procedure mfDaeSolve_JacUser
      module procedure mfDaeSolve_JacUserSP
   end interface mfDaeSolve
   !------ API end ------

   interface msDaeSolve
      module procedure msDaeSolve_JacDF
      module procedure msDaeSolve_JacUser
      module procedure msDaeSolve_JacUserSP
      module procedure msDaeSolve_SaveRest_mf
      module procedure msDaeSolve_SaveRest_SP_mf
      module procedure msDaeSolve_Finalize
   end interface msDaeSolve
   !------ API end ------

   interface mfFSolve
      module procedure mfFSolve_JacDF
      module procedure mfFSolve_JacUser
      module procedure mfFSolve_JacUserSP
   end interface mfFSolve
   !------ API end ------

   interface msFSolve
      module procedure msFSolve_JacDF
      module procedure msFSolve_JacUser
      module procedure msFSolve_JacUserSP
   end interface msFSolve
   !------ API end ------

   public :: mfTrapz, &
             mfSimpson, &
             mfQuad, msQuad, &
             mfDblQuad, msDblQuad, &
             mfOdeSolve, msOdeSolve, &
             mfDaeSolve, msDaeSolve, &
             mfFZero, msFZero, &
             mfFSolve, msFSolve

   private :: mfTrapz_one_arg, mfTrapz_two_args, &
              mfSimpson_one_arg, mfSimpson_two_args, &
              mfDblQuad_cte, mfDblQuad_funx, mfDblQuad_funy, &
              msDblQuad_cte, msDblQuad_funx, msDblQuad_funy, &
              mfOdeSolve_JacDF, mfOdeSolve_JacUser, mfOdeSolve_JacUserSP, &
              msOdeSolve_JacDF, msOdeSolve_JacUser, msOdeSolve_JacUserSP, &
              mfDaeSolve_JacDF, mfDaeSolve_JacUser, mfDaeSolve_JacUserSP, &
              msDaeSolve_JacDF, msDaeSolve_JacUser, msDaeSolve_JacUserSP

   ! following default tolerance are the same as in MATLAB
   real(kind=MF_DOUBLE), parameter :: ODE_rtol_def = 1.0d-3,            &
                                      ODE_atol_def = 1.0d-6

   real(kind=MF_DOUBLE), parameter :: DAE_rtol_def = 1.0d-3,            &
                                      DAE_atol_def = 1.0d-6

   ! following constants are related to each method employed
   real(kind=MF_DOUBLE), parameter :: RKF_rtol_min = 1.0d-12,           &
                                      ABM_rtol_min = 1.0d-15,           &
                                      BDF_rtol_min = 2.2d-14

   private :: ODE_rtol_def, &
              ODE_atol_def, &
              DAE_rtol_def, &
              DAE_atol_def, &
              RKF_rtol_min, &
              ABM_rtol_min, &
              BDF_rtol_min

   real(kind=MF_DOUBLE), private :: rtol_min

   ! definition of a derived type designed as a container for
   ! most of options of FSolve and LsqNonLin (MINPACK-1)
   type, public :: mf_NL_Options
      type(mfArray)        :: tol
      integer              :: max_iter = 50
      type(mfArray)        :: epsfcn
      real(kind=MF_DOUBLE) :: init_step_bound_factor = 1.0d2
      logical              :: print = .false.,                          &
                              print_using_transf = .false.,             &
                              reuse_spjac_struct  = .false.
      type(func_ptr), allocatable :: f_inv_transf(:)
      integer              :: check_jac       = 0,                      &
                              print_check_jac = 0
      logical              :: box_constrained = .false.
      type(mfArray)        :: lower_bounds,                             &
                              upper_bounds
      real(kind=MF_DOUBLE) :: sing_jac_tol = epsilon(1.0d0)
      logical              :: print_sing_val = .false.
   end type mf_NL_Options

   ! internal vars for DAE_CIC
   integer, private :: sls_daecic_call_nb
   logical, private :: free_y_full, free_yp_full

   ! definition of a derived type designed as a container for
   ! most of options of OdeSolve and DaeSolve (SLATEC)
   type, public :: mf_DE_Options
      logical              :: continuation        = .false.
      character(len=3)     :: method = ""
      type(mfArray)        :: tol, y_ind_out, band
      type(mfArray)        :: non_neg, y0_ind, yp0_ind
      logical              :: IC_known            = .true.,             &
                              print_progress      = .false.,            &
                              disp_times          = .false.,            &
                              jac_symm_pos_def    = .false.,            &
                              spjac_const_struct  = .true.,             &
                              reuse_spjac_struct  = .false.,            &
                              jac_investig        = .false.,            &
                              rational_null_basis = .false.,            &
                              pseudo_inverse      = .false.
      real(kind=MF_DOUBLE) :: jac_rcond_min       = MF_EPS

      logical              :: save_sing_jac       = .false.
      type(mfArray)        :: monitor_y_ind,                            &
                              monitor_yp_ind
      logical              :: monitor_pause       = .false.
      integer              :: check_jac           = 0,                  &
                              print_check_jac     = 0
      type(mf_DE_Named_Group), allocatable :: named_eqn(:), named_var(:)
   end type mf_DE_Options

   interface msRelease
      module procedure mf_DE_free
      module procedure mf_NL_free
   end interface msRelease
   !------ API end ------

   public :: msRelease

   interface mfLsqNonLin
      module procedure mfLsqNonLin_JacDF
      module procedure mfLsqNonLin_JacUser
   end interface mfLsqNonLin
   !------ API end ------

   interface msLsqNonLin
      module procedure msLsqNonLin_JacDF
      module procedure msLsqNonLin_JacUser
   end interface msLsqNonLin
   !------ API end ------

   public :: mfLsqNonLin,                                               &
             msLsqNonLin

   private :: mfLsqNonLin_JacDF, mfLsqNonLin_JacUser,                   &
              msLsqNonLin_JacDF, msLsqNonLin_JacUser

   private :: dae_cic_dense, dae_cic_band, dae_cic_sparse

contains
!_______________________________________________________________________
!
#include "fml_funfun/Quad.inc"
!_______________________________________________________________________
!
#include "fml_funfun/mfOdeSolve_JacDF.inc"
!_______________________________________________________________________
!
#include "fml_funfun/mfOdeSolve_JacUser.inc"
!_______________________________________________________________________
!
#include "fml_funfun/mfOdeSolve_JacUserSP.inc"
!_______________________________________________________________________
!
#include "fml_funfun/msOdeSolve_JacDF.inc"
!_______________________________________________________________________
!
#include "fml_funfun/msOdeSolve_JacUser.inc"
!_______________________________________________________________________
!
#include "fml_funfun/msOdeSolve_JacUserSP.inc"
!_______________________________________________________________________
!
#include "fml_funfun/OdeSolve_aux.inc"
!_______________________________________________________________________
!
#include "fml_funfun/mfDaeSolve_JacDF.inc"
!_______________________________________________________________________
!
#include "fml_funfun/mfDaeSolve_JacUser.inc"
!_______________________________________________________________________
!
#include "fml_funfun/mfDaeSolve_JacUserSP.inc"
!_______________________________________________________________________
!
#include "fml_funfun/msDaeSolve_JacDF.inc"
!_______________________________________________________________________
!
#include "fml_funfun/msDaeSolve_JacUser.inc"
!_______________________________________________________________________
!
#include "fml_funfun/msDaeSolve_JacUserSP.inc"
!_______________________________________________________________________
!
#include "fml_funfun/DaeSolve_aux.inc"
!_______________________________________________________________________
!
#include "fml_funfun/dae_cic_dense.inc"
!_______________________________________________________________________
!
#include "fml_funfun/dae_cic_band.inc"
!_______________________________________________________________________
!
#include "fml_funfun/dae_cic_sparse.inc"
!_______________________________________________________________________
!
#include "fml_funfun/FZero.inc"
!_______________________________________________________________________
!
#include "fml_funfun/FSolve.inc"
!_______________________________________________________________________
!
#include "fml_funfun/LsqNonLin.inc"
!_______________________________________________________________________
!
   subroutine mf_DE_free( options )

      type(mf_DE_Options) :: options
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      options%continuation       = .false.
      options%method             = ""
      call msSilentRelease( options%tol, options%y_ind_out, options%band )
      call msSilentRelease( options%non_neg, options%y0_ind, options%yp0_ind )
      options%IC_known           = .true.
      options%print_progress     = .false.
      options%disp_times         = .false.
      options%spjac_const_struct = .true.
      options%reuse_spjac_struct = .false.
      options%jac_investig       = .false.
      options%jac_rcond_min      = 0.0d0
      options%save_sing_jac      = .false.
      call msSilentRelease( options%monitor_y_ind, options%monitor_yp_ind )
      options%monitor_pause      = .false.
      options%check_jac          = 0
      options%print_check_jac    = 0

#endif
   end subroutine mf_DE_free
!_______________________________________________________________________
!
   subroutine mf_NL_free( options )

      type(mf_NL_Options) :: options
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      call msSilentRelease( options%tol )
      options%max_iter               = 50
      call msSilentRelease( options%epsfcn )
      options%init_step_bound_factor = 1.0d2
      options%print                  = .false.
      options%print_using_transf     = .false.
      if( allocated( options%f_inv_transf) ) then
         deallocate( options%f_inv_transf )
      end if
      options%check_jac              = 0
      options%print_check_jac        = 0
      options%box_constrained        = .false.
      call msSilentRelease( options%lower_bounds )
      call msSilentRelease( options%upper_bounds )
      options%sing_jac_tol           = epsilon(1.0d0)
      options%print_sing_val         = .false.

#endif
   end subroutine mf_NL_free
!_______________________________________________________________________
!
   function mfTrapz_one_arg( y ) result( out )

      type(mfArray) :: y
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! for unit spacing
      ! (if the user needs non unit spacing, the result must be
      !  multiplied by this spacing)

      integer :: m, n, dim, i
      real(kind=MF_DOUBLE) :: res

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( y )

      if( mfIsEmpty(y) ) then
         go to 99
      end if

      if( y%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfTrapz", "E",                             &
                            "y must be real array!" )
         go to 99
      end if

      m = y%shape(1)
      n = y%shape(2)
      if( m == 1 ) then
         dim = 2
      else if( n == 1 ) then
         dim = 1
      else
         call PrintMessage( "mfTrapz", "E",                             &
                            "y must be a vector!" )
         go to 99
      end if

      if( m == 1 .and. n == 1 ) then
         call PrintMessage( "mfTrapz", "E",                             &
                            "y cannot be scalar!" )
         go to 99
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ 1, 1 ]
      allocate( out%double(1,1) )

      res = 0.0d0
      if( dim == 1 ) then
         do i = 2, m
            res = res + 0.5d0*(y%double(i-1,1)+y%double(i,1))
         end do
      else ! dim == 2
         do i = 2, n
            res = res + 0.5d0*(y%double(1,i-1)+y%double(1,i))
         end do
      end if

      out%double(1,1) = res

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( y )
      call msAutoRelease( y )

#endif
   end function mfTrapz_one_arg
!_______________________________________________________________________
!
   function mfCumTrapz_one_arg( y ) result( out )

      type(mfArray) :: y
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! for unit spacing
      ! (if the user needs non unit spacing, the result must be
      !  multiplied by this spacing)

      integer :: m, n, dim, i

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( y )

      if( mfIsEmpty(y) ) then
         go to 99
      end if

      if( y%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfCumTrapz", "E",                          &
                            "y must be real array!" )
         go to 99
      end if

      m = y%shape(1)
      n = y%shape(2)
      if( m == 1 ) then
         dim = 2
      else if( n == 1 ) then
         dim = 1
      else
         call PrintMessage( "mfCumTrapz", "E",                          &
                            "y must be a vector!" )
         go to 99
      end if

      if( m == 1 .and. n == 1 ) then
         call PrintMessage( "mfCumTrapz", "E",                          &
                            "y cannot be scalar!" )
         go to 99
      end if

      out%data_type = MF_DT_DBLE
      out%shape = y%shape
      allocate( out%double(out%shape(1),out%shape(2)) )
      out%double(1,1) = 0.0d0

      if( dim == 1 ) then
         do i = 2, m
            out%double(i,1) = out%double(i-1,1)                         &
                              + 0.5d0*(y%double(i-1,1)+y%double(i,1))
         end do
      else ! dim == 2
         do i = 2, n
            out%double(1,i) = out%double(1,i-1)                         &
                              + 0.5d0*(y%double(1,i-1)+y%double(1,i))
         end do
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( y )
      call msAutoRelease( y )

#endif
   end function mfCumTrapz_one_arg
!_______________________________________________________________________
!
   function mfTrapz_two_args( x, y ) result( out )

      type(mfArray) :: x, y
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! for non uniform spacing

      integer :: m, n, dim, i
      real(kind=MF_DOUBLE) :: res

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x, y )

      if( mfIsEmpty(x) .or. mfIsEmpty(y) ) then
         go to 99
      end if

      if( x%data_type /= MF_DT_DBLE .or. y%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfTrapz", "E",                             &
                            "x and y must be real arrays!" )
         go to 99
      end if

      if( any(x%shape/=y%shape) ) then
         call PrintMessage( "mfTrapz", "E",                             &
                            "x and y must have the same shape!" )
         go to 99
      end if

      m = x%shape(1)
      n = x%shape(2)
      if( m == 1 ) then
         dim = 2
      else if( n == 1 ) then
         dim = 1
      else
         call PrintMessage( "mfTrapz", "E",                             &
                            "x and y must be vectors!" )
         go to 99
      end if

      if( m == 1 .and. n == 1 ) then
         call PrintMessage( "mfTrapz", "E",                             &
                            "x and y cannot be scalars!" )
         go to 99
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ 1, 1 ]
      allocate( out%double(1,1) )

      res = 0.0d0
      if( dim == 1 ) then
         do i = 2, m
            res = res + 0.5d0*(y%double(i-1,1)+y%double(i,1))           &
                             *(x%double(i,1)-x%double(i-1,1))
         end do
      else ! dim == 2
         do i = 2, n
            res = res + 0.5d0*(y%double(1,i-1)+y%double(1,i))           &
                             *(x%double(1,i)-x%double(1,i-1))
         end do
      end if

      out%double(1,1) = res

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( x, y )
      call msAutoRelease( x, y )

#endif
   end function mfTrapz_two_args
!_______________________________________________________________________
!
   function mfCumTrapz_two_args( x, y ) result( out )

      type(mfArray) :: x, y
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! for non uniform spacing

      integer :: m, n, dim, i

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x, y )

      if( mfIsEmpty(x) .or. mfIsEmpty(y) ) then
         go to 99
      end if

      if( x%data_type /= MF_DT_DBLE .or. y%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfCumTrapz", "E",                          &
                            "x and y must be real arrays!" )
         go to 99
      end if

      if( any(x%shape/=y%shape) ) then
         call PrintMessage( "mfCumTrapz", "E",                          &
                            "x and y must have the same shape!" )
         go to 99
      end if

      m = x%shape(1)
      n = x%shape(2)
      if( m == 1 ) then
         dim = 2
      else if( n == 1 ) then
         dim = 1
      else
         call PrintMessage( "mfCumTrapz", "E",                          &
                            "x and y must be vectors!" )
         go to 99
      end if

      if( m == 1 .and. n == 1 ) then
         call PrintMessage( "mfCumTrapz", "E",                          &
                            "x and y cannot be scalars!" )
         go to 99
      end if

      out%data_type = MF_DT_DBLE
      out%shape = y%shape
      allocate( out%double(out%shape(1),out%shape(2)) )
      out%double(1,1) = 0.0d0

      if( dim == 1 ) then
         do i = 2, m
            out%double(i,1) = out%double(i-1,1)                         &
                              + 0.5d0*(y%double(i-1,1)+y%double(i,1))   &
                                     *(x%double(i,1)-x%double(i-1,1))
         end do
      else ! dim == 2
         do i = 2, n
            out%double(1,i) = out%double(1,i-1)                         &
                              + 0.5d0*(y%double(1,i-1)+y%double(1,i))   &
                                     *(x%double(1,i)-x%double(1,i-1))
         end do
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( x, y )
      call msAutoRelease( x, y )

#endif
   end function mfCumTrapz_two_args
!_______________________________________________________________________
!
   function mfSimpson_one_arg( y ) result( out )

      type(mfArray) :: y
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! for unit spacing
      ! (if the user needs non unit spacing, the result must be
      !  multiplied by this spacing)

      ! number of points must be odd

      integer :: m, n, dim, i
      real(kind=MF_DOUBLE) :: res, sum_even, sum_odd

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( y )

      if( mfIsEmpty(y) ) then
         go to 99
      end if

      if( y%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfSimpson", "E",                           &
                            "y must be real array!" )
         go to 99
      end if

      m = y%shape(1)
      n = y%shape(2)
      if( m == 1 ) then
         dim = 2
      else if( n == 1 ) then
         dim = 1
      else
         call PrintMessage( "mfSimpson", "E",                           &
                            "y must be a vector!" )
         go to 99
      end if

      if( ( dim == 1 .and. mod(m,2) == 0 ) .or.                         &
          ( dim == 2 .and. mod(n,2) == 0 )  ) then
         call PrintMessage( "mfSimpson", "E",                           &
                            "y must have an odd number of values!" )
         go to 99
      end if

      if( m < 3 .and. n < 3 ) then
         call PrintMessage( "mfSimpson", "E",                           &
                            "y must have at least 3 values!" )
         go to 99
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ 1, 1 ]
      allocate( out%double(1,1) )

      sum_even = 0.0d0
      sum_odd = 0.0d0
      if( dim == 1 ) then
         do i = 2, m-1, 2
            sum_even = sum_even + y%double(i,1)
         end do
         do i = 3, m-2, 2
            sum_odd = sum_odd + y%double(i,1)
         end do
         res = ( y%double(1,1) + 4.0d0*sum_even                         &
                               + 2.0d0*sum_odd + y%double(m,1) ) / 3.0d0
      else ! dim == 2
         do i = 2, n-1, 2
            sum_even = sum_even + y%double(1,i)
         end do
         do i = 3, n-2, 2
            sum_odd = sum_odd + y%double(1,i)
         end do
         res = ( y%double(1,1) + 4.0d0*sum_even                         &
                               + 2.0d0*sum_odd + y%double(1,n) ) / 3.0d0
      end if

      out%double(1,1) = res

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( y )
      call msAutoRelease( y )

#endif
   end function mfSimpson_one_arg
!_______________________________________________________________________
!
   function mfSimpson_two_args( x, y ) result( out )

      type(mfArray) :: x, y
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! for non uniform spacing

      ! number of points must be odd

      integer :: m, n, dim, i
      real(kind=MF_DOUBLE) :: res, h1, h2, s

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x, y )

      if( mfIsEmpty(x) .or. mfIsEmpty(y) ) then
         go to 99
      end if

      if( x%data_type /= MF_DT_DBLE .or. y%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfSimpson", "E",                           &
                            "x and y must be real arrays!" )
         go to 99
      end if

      if( any(x%shape/=y%shape) ) then
         call PrintMessage( "mfSimpson", "E",                           &
                            "x and y must have the same shape!" )
         go to 99
      end if

      m = x%shape(1)
      n = x%shape(2)
      if( m == 1 ) then
         dim = 2
      else if( n == 1 ) then
         dim = 1
      else
         call PrintMessage( "mfSimpson", "E",                           &
                            "x and y must be vectors!" )
         go to 99
      end if

      if( ( dim == 1 .and. mod(m,2) == 0 ) .or.                         &
          ( dim == 2 .and. mod(n,2) == 0 )  ) then
         call PrintMessage( "mfSimpson", "E",                           &
                            "y must have an odd number of values!" )
         go to 99
      end if

      if( m < 3 .and. n < 3 ) then
         call PrintMessage( "mfSimpson", "E",                           &
                            "y must have at least 3 values!" )
         go to 99
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ 1, 1 ]
      allocate( out%double(1,1) )

      res = 0.0d0
      if( dim == 1 ) then
         do i = 1, m-2, 2
            h1 = x%double(i+1,1)-x%double(i,1)
            h2 = x%double(i+2,1)-x%double(i+1,1)
            s = h1 + h2
            res = res + s*(2*h1-h2)/h1*y%double(i,1)                    &
                      + s**3/h1/h2*y%double(i+1,1)                      &
                      + s*(2*h2-h1)/h2*y%double(i+2,1)
         end do
      else ! dim == 2
         do i = 1, n-2, 2
            h1 = x%double(1,i+1)-x%double(1,i)
            h2 = x%double(1,i+2)-x%double(1,i+1)
            s = h1 + h2
            res = res + s*(2*h1-h2)/h1*y%double(1,i)                    &
                      + s**3/h1/h2*y%double(1,i+1)                      &
                      + s*(2*h2-h1)/h2*y%double(1,i+2)
         end do
      end if

      out%double(1,1) = res/6.0d0

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( x, y )
      call msAutoRelease( x, y )

#endif
   end function mfSimpson_two_args
!_______________________________________________________________________
!
end module mod_funfun
