#ifdef _HAS_LOC
#define _MF_LOC_ANY_OBJ_ loc
#else
#define _MF_LOC_ANY_OBJ_ mf_loc_any_obj
#endif

module mod_core ! Core

   ! Part of MUESLI Numerical Library
   ! Copyright É. Canot 2003-2025 -- IPR/CNRS

!-----------------------------------------------------------------------
!                             used modules
!-----------------------------------------------------------------------

   use mod_ieee

   use mod_physunits

#ifndef _DEVLP
   use mod_mfarray ! required for the 2nd pass of the double compilation
#endif

   ! 'use mod_mfdebug' present at line 3898 [partial use]

#if defined _INTEL_IFC
   use iflport, only: ttynam, flush, system, getpid ! (msFlush)
#endif

   implicit none

#ifndef _DEVLP
   private
#endif

   public :: STDERR, STDIN, STDOUT

   public :: MF_DOUBLE, mfArray
   public :: MF_EMPTY, MF_COLON, MF_END,                                &
             MF_PI, MF_EPS, MF_E, MF_REALMAX, MF_REALMIN,               &
             MF_I
   public :: MF_INF, MF_NAN

   integer, parameter :: f90readline_max_length = 1024

!-----------------------------------------------------------------------
!                            derived types
!-----------------------------------------------------------------------

#include "fml_core/mf_Out.inc"

   public :: mf_Out, mfOut

   ! definition of a named group (of equations or variables)
   ! (used in mod_funfun, but mut be accessed by slatec routines)
   type, public :: mf_DE_Named_Group
      character(len=132) :: name
      integer            :: begin, last
   end type

!-----------------------------------------------------------------------
!                        fml global variables
!-----------------------------------------------------------------------

   ! the following two variables are only modified via
   ! the 'msFormat' routine
   logical :: mf_short_mantissa = .true.
   character(len=4) :: mf_display_exponent = "auto"
   logical :: mf_display_hexa = .false.

   integer, save :: ncol_term_val
   logical, save :: ncol_term_is_known = .false.

   integer :: mf_display_head_length,                                   &
              mf_display_tail_length
   logical :: mf_display_line_head_or_tail

   integer*8, save :: mf_papi_flpins_old
   integer, save :: mf_papi_available = UNKNOWN
   logical, save :: mf_papi_init_ok = .false.

   integer :: crc

   logical, save :: mf_auto_complex = .true.

   real(kind=MF_DOUBLE) :: mf_out_of_range_filling = MF_NAN

   ! global variable to access the current DE_Options
   logical :: NAMED_EQN_PRESENCE = .false.
   type(mf_DE_Named_Group), pointer :: NAMED_EQN_PTR(:) => null()
   logical :: NAMED_VAR_PRESENCE = .false.
   type(mf_DE_Named_Group), pointer :: NAMED_VAR_PTR(:) => null()

   ! avoid grenv multiple times for getting the MUESLI install path
   character(len=256), save :: MFPLOT_DIR = ""

!-----------------------------------------------------------------------
!                             interfaces
!-----------------------------------------------------------------------

#include "misc/suitesparse/umf4_f90wrapper.inc"

   interface assignment(=)
      module procedure msAssign_scalar_int
      module procedure msAssign_vector_int
      module procedure msAssign_matrix_int
      module procedure msAssign_scalar_single
      module procedure msAssign_vector_single
      module procedure msAssign_matrix_single
      module procedure msAssign_scalar_double
      module procedure msAssign_vector_double
      module procedure msAssign_matrix_double
      module procedure msAssign_scalar_cmplx
      module procedure msAssign_vector_cmplx
      module procedure msAssign_matrix_cmplx
      module procedure msAssign_scalar_double_cmplx
      module procedure msAssign_vector_double_cmplx
      module procedure msAssign_matrix_double_cmplx

      module procedure msAssign_mfArray_copy   ! always copy

      module procedure mfArray_to_scalar_int   ! int  = mfArray
      module procedure mfArray_to_vector_int
      module procedure mfArray_to_matrix_int

      module procedure mfArray_to_scalar_real  ! real = mfArray
      module procedure mfArray_to_vector_real
      module procedure mfArray_to_matrix_real

      module procedure mfArray_to_scalar_dble  ! dble = mfArray
      module procedure mfArray_to_vector_dble
      module procedure mfArray_to_matrix_dble

      module procedure mfArray_to_scalar_cmplx ! cmplx = mfArray
      module procedure mfArray_to_vector_cmplx
      module procedure mfArray_to_matrix_cmplx

      module procedure mfArray_to_scalar_dble_cmplx ! dble_cmplx = mfArray
      module procedure mfArray_to_vector_dble_cmplx
      module procedure mfArray_to_matrix_dble_cmplx

      module procedure mfArray_to_scalar_bool  ! bool = mfArray
      module procedure mfArray_to_vector_bool
      module procedure mfArray_to_matrix_bool

      module procedure msAssign_scalar_bool
      module procedure msAssign_vector_bool
      module procedure msAssign_matrix_bool

      module procedure msAssign_mfUnit
   end interface assignment(=)
   !------ API end ------

   interface msAssign
      module procedure msAssign_mfArray        ! no copy if possible
   end interface msAssign
   !------ API end ------

   interface mf
      module procedure mf__scal_int
      module procedure mf__vect_int
      module procedure mf__matr_int
      module procedure mf__scal_single
      module procedure mf__vect_single
      module procedure mf__matr_single
      module procedure mf__scal_double
      module procedure mf__vect_double
      module procedure mf__matr_double
      module procedure mf__scal_cmplx
      module procedure mf__vect_cmplx
      module procedure mf__matr_cmplx
      module procedure mf__scal_double_cmplx
      module procedure mf__vect_double_cmplx
      module procedure mf__matr_double_cmplx

      module procedure mf__scal_bool
      module procedure mf__vect_bool
      module procedure mf__matr_bool

      module procedure mf__mfUnit
   end interface mf
   !------ API end ------

   interface msDisplay
      module procedure msDisplay_mfArray
      module procedure msDisplay_int
      module procedure msDisplay_vec_int
      module procedure msDisplay_array_int
      module procedure msDisplay_single
      module procedure msDisplay_vec_single
      module procedure msDisplay_array_single
      module procedure msDisplay_double
      module procedure msDisplay_vec_double
      module procedure msDisplay_array_double
      module procedure msDisplay_cmplx
      module procedure msDisplay_vec_cmplx
      module procedure msDisplay_array_cmplx
      module procedure msDisplay_double_cmplx
      module procedure msDisplay_vec_double_cmplx
      module procedure msDisplay_array_double_cmplx
      module procedure msDisplay_mfUnit
      module procedure msDisplay_mf_Int_List
      module procedure msDisplay_mf_Int_List_vec
      module procedure msDisplay_mf_Real_List
      module procedure msDisplay_mf_Real_List_vec
      module procedure msDisplay_with_legend
   end interface msDisplay
   !------ API end ------

   interface msPointer
      module procedure msPointer_matrix_real8
      module procedure msPointer_vector_real8
      module procedure msPointer_matrix_cmplx
      module procedure msPointer_vector_cmplx
   end interface msPointer
   !------ API end ------

   interface msFreePointer
      module procedure msFreePointer_matrix_real8
      module procedure msFreePointer_LongColumn_real8
      module procedure msFreePointer_matrix_cmplx
      module procedure msFreePointer_LongColumn_cmplx
   end interface msFreePointer
   !------ API end ------

   interface msEquiv
      module procedure msEquiv_matrix_real8
      module procedure msEquiv_LongColumn_real8
      module procedure msEquiv_matrix_cmplx
      module procedure msEquiv_LongColumn_cmplx
   end interface msEquiv
   !------ API end ------

   interface mfGet
      module procedure mfGet_element
      module procedure mfGet_element_in_vec
      module procedure mfGet_section_in_vec
      module procedure mfGet_section_in_vseq
      module procedure mfGet_elems_in_array
      module procedure mfGet_section_vec_vec
      module procedure mfGet_section_vseq_vseq
      module procedure mfGet_section_int_vseq
      module procedure mfGet_section_vseq_int
      module procedure mfGet_section_array_array
      module procedure mfGet_section_array_vec
      module procedure mfGet_section_array_vseq
      module procedure mfGet_section_vec_array
      module procedure mfGet_section_vseq_array
      module procedure mfGet_section_array_int
      module procedure mfGet_section_int_array
!
! il manque pour mfGet_section_* un certain nombre de combinaisons
! entre les 4 types:
!     int, vec, vseq, array
! (mais il y a une telle redondance dans le code qu'il faudrait tout
!  restructurer, afin d'appeler une ou deux routines basiques...)
!
   end interface mfGet
   !------ API end ------

   interface msSet
      module procedure msSet_element_real8
      module procedure msSet_element_cmplx
      module procedure msSet_array_element
      module procedure msSet_element_real8_in_vec
      module procedure msSet_element_cmplx_in_vec
      module procedure msSet_array_element_in_vec
      module procedure msSet_spread_real8_in_vec_section
      module procedure msSet_spread_cmplx_in_vec_section
      module procedure msSet_spread_real8_in_vec_sect2
      module procedure msSet_spread_cmplx_in_vec_sect2
      module procedure msSet_spread_real8_in_vec_vseq
      module procedure msSet_array_in_vec_vseq
      module procedure msSet_array_in_vec_section
      module procedure msSet_array_in_vec_sect2
      module procedure msSet_array_section_vec_vec
      module procedure msSet_arr_sect_vseq_vseq
      module procedure msSet_array_section_array_array
      module procedure msSet_spread_real8_in_section
      module procedure msSet_spread_cmplx_in_section
      module procedure msSet_spread_real8_in_colon_vec
      module procedure msSet_spread_cmplx_in_colon_vec
      module procedure msSet_spread_real8_in_colon_int
      module procedure msSet_spread_cmplx_in_colon_int
      module procedure msSet_spread_real8_in_vec_colon
      module procedure msSet_spread_cmplx_in_vec_colon
      module procedure msSet_spread_real8_in_int_colon
      module procedure msSet_array_section_colon_vec
      module procedure msSet_array_section_vec_colon
      module procedure msSet_array_section_colon_vseq ! new
      module procedure msSet_array_section_vseq_colon ! new
      module procedure msSet_array_section_colon_int
      module procedure msSet_array_section_int_colon
      module procedure msSet_sprd_r8_vseq_vseq
      module procedure msSet_sprd_r8_int_vseq
      module procedure msSet_sprd_r8_vseq_int
      module procedure msSet_r8_array_array
      module procedure msSet_cmplx_array_array
   end interface msSet
   !------ API end ------

   interface shape
      module procedure Shape_mfArray
      module procedure Shape_mfMatFactor
   end interface shape
   !------ API end ------

   interface mfShape
      module procedure mfShape_mfArray
      module procedure mfShape_mfMatFactor
   end interface mfShape
   !------ API end ------

   interface size
      module procedure Size_mfArray
      module procedure Size_mfMatFactor
   end interface size
   !------ API end ------

   interface mfSize
      module procedure mfSize_mfArray
      module procedure mfSize_mfMatFactor
   end interface mfSize
   !------ API end ------

   interface all
      module procedure All_mfArray
   end interface all
   !------ API end ------

   interface any
      module procedure Any_mfArray
   end interface any
   !------ API end ------

#if !defined(_WINDOWS) && !defined(_DARWIN)
   interface msFlops
      module procedure msFlops_init
      module procedure msFlops_count
   end interface msFlops
   !------ API end ------
#endif

   interface msRelease
      module procedure msRelease_mfArray
      module procedure msRelease_mf_Int_List
      module procedure msRelease_mf_Int_List_vec
      module procedure msRelease_mf_Real_List
      module procedure msRelease_mf_Real_List_vec
   end interface msRelease
   !------ API end ------

   interface msSetTermWidth
      module procedure msSetTermWidth_real
      module procedure msSetTermWidth_char
   end interface msSetTermWidth
   !------ API end ------

   interface mfOut
      module procedure mfOut_mfArray
      module procedure mfOut_mfMatFactor
      module procedure mfOut_empty
   end interface mfOut
   !------ API end ------

   interface msPrepHashes
      module procedure msPrepHashes_int
      module procedure msPrepHashes_dble
   end interface msPrepHashes
   !------ API end ------

   interface msPrintHashes
      module procedure msPrintHashes_int
      module procedure msPrintHashes_dble
   end interface msPrintHashes
   !------ API end ------

!-----------------------------------------------------------------------
!                           public variables
!-----------------------------------------------------------------------

   public :: mf, &
             msAssign, &
             assignment(=), &
             msDisplay, &
             msPause, &
             msFormat, &
             mfIsEmpty, &
             mfIsEqual, &
             mfFull, &
             mfIsNotEqual, &
             mfIsLogical, &
             mfIsReal, &
             mfIsComplex, &
             mfIsNumeric, &
             mfIsDense, &
             mfIsSparse, &
             mfIsPerm, &
             msInitArgs, &
             msFreeArgs, &
             msSetMsgLevel, &
             mfGetMsgLevel, &
             msSetTrbLevel, &
             mfGetTrbLevel, &
             msRelease, &
             msAutoRelease, &
             msReturnArray, &
             mfIsTempoArray, &
             msPointer, &
             msFreePointer, &
             mfNbPointers, &
             msEquiv, &
             msGetStdIO, &
             msSetStdIO, &
             msFlush, &
             mfGet, &
             msSet, &
             mfCount

   public :: Shape, &
             Size, &
             mfShape, &
             mfSize, &
             mfIsScalar, &
             mfIsVector, &
             mfIsRow, &
             mfIsColumn, &
             mfIsMatrix, &
             All, &
             Any, &
             mfInt, &
             mfDble, &
             mfCmplx, &
             args_mfout_ok, &
             msUsePhysUnits, &
             mfNnz, &
             msRowSort, &
             msSetAsParameter, &
             mfDisplayColumns, &
             msPrepHashes, &
             msPrintHashes, &
             msPostHashes, &
             msSetAutoComplex, &
             mfGetAutoComplex, &
             mfIsVersion, &
             msGetBlasLib, &
             msGetLapackLib, &
             msGetSuiteSparseLib, &
             MF_LAPACK_VERSION

   public :: msSetAutoFilling, &
             mfGetAutoFilling

#ifndef _WINDOWS

#ifndef _DARWIN

   public :: msEnableFPE, &
             msDisableFPE

   ! Floating-Point Exceptions
   ! USUAL_EXCEPTS = OVERFLOW_EXC | DIVBYZERO_EXC | INVALID_EXC
   !
   integer, parameter :: MF_OVERFLOW_EXC  = 1, &
                         MF_DIVBYZERO_EXC = 2, &
                         MF_INVALID_EXC   = 3, &
                         MF_USUAL_EXCEPTS = 4, &
                         MF_UNDERFLOW_EXC = 5

   public :: msSetRoundingMode, &
             mfGetRoundingMode

   public :: mfIsFlopsOk, &
             mfFlops, &
             msFlops

   private :: msFlops_init, &
              msFlops_count

#if defined _READLINE
   public :: mfReadline, &
             msRemoveLastEntryInHistory, &
             msAddEntryInHistory, &
             msClearHistory, &
             msWriteHistoryFile, &
             msReadHistoryFile
#endif

#endif

   public :: msSetTermWidth, &
             mfGetTermWidth

#endif

#ifndef _OPTIM
   public :: print_debug_mfArray
#endif

#ifndef _DEVLP
   private :: can_use_memory
#endif

   private :: msAssign_scalar_int, &
              msAssign_vector_int, &
              msAssign_matrix_int, &
              msAssign_scalar_single, &
              msAssign_vector_single, &
              msAssign_matrix_single, &
              msAssign_scalar_double, &
              msAssign_vector_double, &
              msAssign_matrix_double, &
              msAssign_scalar_cmplx, &
              msAssign_vector_cmplx, &
              msAssign_matrix_cmplx, &
              msAssign_scalar_double_cmplx, &
              msAssign_vector_double_cmplx, &
              msAssign_matrix_double_cmplx, &
              msAssign_mfArray_copy, &
              mfArray_to_scalar_int, &
              mfArray_to_vector_int, &
              mfArray_to_matrix_int, &
              mfArray_to_scalar_real, &
              mfArray_to_vector_real, &
              mfArray_to_matrix_real, &
              mfArray_to_scalar_dble, &
              mfArray_to_vector_dble, &
              mfArray_to_matrix_dble, &
              mfArray_to_scalar_cmplx, &
              mfArray_to_vector_cmplx, &
              mfArray_to_matrix_cmplx, &
              mfArray_to_scalar_dble_cmplx, &
              mfArray_to_vector_dble_cmplx, &
              mfArray_to_matrix_dble_cmplx, &
              mfArray_to_scalar_bool, &
              mfArray_to_vector_bool, &
              mfArray_to_matrix_bool, &
              msAssign_scalar_bool, &
              msAssign_vector_bool, &
              msAssign_matrix_bool, &
              msAssign_mfUnit, &
              msAssign_mfArray

   private :: mf__scal_int, &
              mf__vect_int, &
              mf__matr_int, &
              mf__scal_single, &
              mf__vect_single, &
              mf__matr_single, &
              mf__scal_double, &
              mf__vect_double, &
              mf__matr_double, &
              mf__scal_cmplx, &
              mf__vect_cmplx, &
              mf__matr_cmplx, &
              mf__scal_double_cmplx, &
              mf__vect_double_cmplx, &
              mf__matr_double_cmplx, &
              mf__scal_bool, &
              mf__vect_bool, &
              mf__matr_bool, &
              mf__mfUnit

   private :: msDisplay_mfArray, &
              msDisplay_int, &
              msDisplay_vec_int, &
              msDisplay_array_int, &
              msDisplay_single, &
              msDisplay_vec_single, &
              msDisplay_array_single, &
              msDisplay_double, &
              msDisplay_vec_double, &
              msDisplay_array_double, &
              msDisplay_cmplx, &
              msDisplay_vec_cmplx, &
              msDisplay_array_cmplx, &
              msDisplay_double_cmplx, &
              msDisplay_vec_double_cmplx, &
              msDisplay_array_double_cmplx, &
              msDisplay_mfUnit, &
              msDisplay_mf_Int_List, &
              msDisplay_mf_Int_List_vec, &
              msDisplay_mf_Real_List, &
              msDisplay_mf_Real_List_vec, &
              msDisplay_with_legend

   private :: msPointer_matrix_real8, &
              msPointer_vector_real8, &
              msPointer_matrix_cmplx, &
              msPointer_vector_cmplx, &
              msFreePointer_matrix_real8, &
              msFreePointer_LongColumn_real8, &
              msFreePointer_matrix_cmplx, &
              msFreePointer_LongColumn_cmplx

   private :: msEquiv_matrix_real8, &
              msEquiv_LongColumn_real8, &
              msEquiv_matrix_cmplx, &
              msEquiv_LongColumn_cmplx

   private :: mfGet_element, &
              mfGet_element_in_vec, &
              mfGet_section_in_vec, &
              mfGet_section_in_vseq, &
              mfGet_elems_in_array, &
              mfGet_section_vec_vec, &
              mfGet_section_vseq_vseq, &
              mfGet_section_int_vseq, &
              mfGet_section_vseq_int, &
              mfGet_section_array_array, &
              mfGet_section_array_vec, &
              mfGet_section_array_vseq, &
              mfGet_section_vec_array, &
              mfGet_section_vseq_array, &
              mfGet_section_array_int, &
              mfGet_section_int_array

   private :: msSet_element_real8, &
              msSet_element_cmplx, &
              msSet_array_element, &
              msSet_element_real8_in_vec, &
              msSet_element_cmplx_in_vec, &
              msSet_array_element_in_vec, &
              msSet_spread_real8_in_vec_section, &
              msSet_spread_cmplx_in_vec_section, &
              msSet_spread_real8_in_vec_sect2, &
              msSet_spread_cmplx_in_vec_sect2, &
              msSet_spread_real8_in_vec_vseq, &
              msSet_array_in_vec_section, &
              msSet_array_in_vec_sect2, &
              msSet_array_section_vec_vec, &
              msSet_arr_sect_vseq_vseq, &
              msSet_array_section_array_array, &
              msSet_spread_real8_in_section, &
              msSet_spread_cmplx_in_section, &
              msSet_spread_real8_in_colon_vec, &
              msSet_spread_real8_in_colon_int, &
              msSet_spread_cmplx_in_colon_vec, &
              msSet_spread_cmplx_in_colon_int, &
              msSet_spread_real8_in_vec_colon, &
              msSet_spread_real8_in_int_colon, &
              msSet_spread_cmplx_in_vec_colon, &
              msSet_spread_cmplx_in_int_colon, &
              msSet_array_section_colon_vec, &
              msSet_array_section_vec_colon, &
              msSet_array_section_colon_int, &
              msSet_sprd_r8_vseq_vseq, &
              msSet_sprd_r8_int_vseq, &
              msSet_sprd_r8_vseq_int

   private :: Shape_mfArray, &
              Shape_mfMatFactor, &
              mfShape_mfArray, &
              mfShape_mfMatFactor, &
              Size_mfArray, &
              Size_mfMatFactor, &
              mfSize_mfArray, &
              mfSize_mfMatFactor, &
              All_mfArray, &
              Any_mfArray, &
              msSetTermWidth_real, &
              msSetTermWidth_char, &
              mfOut_mfArray, &
              mfOut_mfMatFactor, &
              mfOut_empty

   private :: Implem_msAutoRelease, &
              Implem_msInitArgs, &
              Implem_msFreeArgs, &
              msRelease_mfArray, &
              msRelease_mf_Int_List, &
              msRelease_mf_Int_List_vec, &
              msRelease_mf_Real_List, &
              msRelease_mf_Real_List_vec, &
              Implem_msRelease, &
              Implem_msSilentRelease, &
              do_data_transfer_dble, &
              do_data_transfer_cmplx, &
              Implem_msSetAsParameter, &
              get_addr_mfarray, &
              get_addr_mfarray_ptr

   ! Here, overloading of the operators for type(seq_def)

   interface operator(.from.)
      module procedure seq_from_mfarray
   end interface
   !------ API end ------

   interface operator(.to.)
      module procedure seq_from_int_to_mfarray
      module procedure seq_from_mfarray_to_int
      module procedure seq_from_mfarray_to_mfarray
      module procedure seq_range_to_mfarray
   end interface
   !------ API end ------

   interface operator(.but.)
      module procedure seq_range_but_mfarray
      module procedure seq_mfarray_but_int
      module procedure seq_mfarray_but_mfarray
   end interface
   !------ API end ------

   private :: seq_from_mfarray, &
              seq_from_int_to_mfarray, &
              seq_from_mfarray_to_int, &
              seq_from_mfarray_to_mfarray, &
              seq_range_to_mfarray, &
              seq_range_but_mfarray, &
              seq_mfarray_but_int, &
              seq_mfarray_but_mfarray

   public :: operator(.from.), &
             operator(.to.), &
             operator(.step.), &
             operator(.by.), &
             operator(.but.), &
             operator(.and.)

   public :: msRequMuesliVer, msSetTermColor, msPrintColoredMsg

contains
#define COLOR_GREEN  achar(27)//"[32m"
#define COLOR_RED    achar(27)//"[31m"
#define COLOR_ORANGE achar(27)//"[38;5;202m"
#define COLOR_YELLOW achar(27)//"[33m"
#define COLOR_GREY   achar(27)//"[37m"//achar(27)//"[3m"
#define NO_COLOR     achar(27)//"[0m"
!_______________________________________________________________________
!
   subroutine msSetTermColor( color )

      character(len=*), intent(in) :: color
      !------ API end ------

#ifdef _DEVLP
      select case( to_lower(color) )
         case( "green" )
            print *, COLOR_GREEN
         case( "red" )
            print *, COLOR_RED
         case( "orange" )
            print *, COLOR_ORANGE
         case( "yellow" )
            print *, COLOR_YELLOW
         case( "grey" )
            print *, COLOR_GREY
         case( "normal" )
            print *, NO_COLOR
         case default
            call PrintMessage( "msSetTermColor", "W",                   &
                               "bad color!" )
      end select
#endif
   end subroutine msSetTermColor
!_______________________________________________________________________
!
   subroutine msPrintColoredMsg( fmt, msg, color )

      character(len=*), intent(in) :: fmt, msg, color
      !------ API end ------

#ifdef _DEVLP
      character(len=:), allocatable :: col_msg
      character(len=*), parameter :: ROUTINE_NAME = "msPrintColoredMsg"

      select case( to_lower(color) )
         case( "green" )
            col_msg = COLOR_GREEN // trim(msg) // NO_COLOR
         case( "red" )
            col_msg = COLOR_RED // trim(msg) // NO_COLOR
         case( "orange" )
            col_msg = COLOR_ORANGE // trim(msg) // NO_COLOR
         case( "yellow" )
            col_msg = COLOR_YELLOW // trim(msg) // NO_COLOR
         case( "grey" )
            col_msg = COLOR_GREY // trim(msg) // NO_COLOR
         case default
            call PrintMessage( trim(ROUTINE_NAME), "W",                 &
                               "cannot plot: no window created!" )
            col_msg = trim(msg)
      end select

      print fmt, trim(col_msg)
#endif
   end subroutine msPrintColoredMsg
!_______________________________________________________________________
!
#include "fml_core/Array.inc"
!_______________________________________________________________________
!
#include "fml_core/Assign.inc"
!_______________________________________________________________________
!
#include "fml_core/Display.inc"
!_______________________________________________________________________
!
#include "fml_core/DisplayColumns.inc"
!_______________________________________________________________________
!
#include "fml_core/Display_bool.inc"
!_______________________________________________________________________
!
#include "fml_core/Display_double.inc"
!_______________________________________________________________________
!
#include "fml_core/Display_complex.inc"
!_______________________________________________________________________
!
#include "fml_core/Display_sparse_bool.inc"
!_______________________________________________________________________
!
#include "fml_core/Display_sparse_double.inc"
!_______________________________________________________________________
!
#include "fml_core/Display_sparse_complex.inc"
!_______________________________________________________________________
!
#include "fml_core/Display_unit.inc"
!_______________________________________________________________________
!
#include "fml_core/Display_mf_int_list.inc"
!_______________________________________________________________________
!
#include "fml_core/Display_mf_int_list_vec.inc"
!_______________________________________________________________________
!
#include "fml_core/Display_mf_real_list.inc"
!_______________________________________________________________________
!
#include "fml_core/Display_mf_real_list_vec.inc"
!_______________________________________________________________________
!
#include "fml_core/Display_with_legend.inc"
!_______________________________________________________________________
!
#include "fml_core/Get.inc"
!_______________________________________________________________________
!
#include "fml_core/Set.inc"
!_______________________________________________________________________
!
#include "fml_core/seq_def_EndIndex.inc"
!_______________________________________________________________________
!
#include "fml_core/Hashes.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/getelm.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/submat.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/setelm.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/xtrcol.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/nnzxtrcols.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/xtrcols.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/xtrrow.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/chgcol.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/sort.inc"
!_______________________________________________________________________
!
#include "fml_sparse/CSC_aux.inc"
!_______________________________________________________________________
!
   subroutine msPause( message, indent, duration )

#if defined _INTEL_IFC
      use iflport, only: system
#endif

      character(len=*),     intent(in), optional :: message
      integer,              intent(in), optional :: indent
      real(kind=MF_DOUBLE), intent(in), optional :: duration
      !------ API end ------

#ifdef _DEVLP
      character(len=36) :: cmd
      character(len=:), allocatable :: indent_char
      integer :: status, indent_0, i

   !------ end of declarations -- execution starts hereafter  ------

      if( present(duration) ) then
         if( present(indent) ) then
            call PrintMessage( "msPause", "W",                          &
                               "arg. 'indent' should not be used in conjonction with 'duration'", &
                               "-> ignoring it." )
         end if
         if( present(message) ) then
            write(STDERR,"(a)") message
         end if
         write(cmd,*) duration
         cmd = "sleep " // trim(cmd)
#if defined _INTEL_IFC | defined _GNU_GFC
         status = system( trim(cmd) )
#else
  '(MUESLI msPause:) compiler not defined!'
#endif
         if( status /= 0 ) then
            ! prefer now generate an error. Perhaps the user pressed
            ! Ctrl-C to stop his program; in such a case, previous behavior
            ! just return to the calling unit.
            call PrintMessage( "msPause", "E",                          &
                               "'system' call returns bad status!",     &
                               "('sleep' command not found ?)" )
            return
         end if
      else
         if( present(message) ) then
            write(STDERR,"(a)") message
         end if
         if( present(indent) ) then
            indent_0 = max( indent, 0 )
         else
            indent_0 = 0
         end if
         allocate( character(indent_0) :: indent_char )
         do i = 1, indent_0
            indent_char(i:i) = " "
         end do
#if defined _INTEL_IFC
         ! as mentionned in msFlush, the workaround for the bug of 'flush'
         ! doesn't work when writing on the same line.
         call put_string_on_term_no_adv( indent_char //                 &
                                         "[Return] to continue . . ."   &
                                         // char(0) )
#else
         write(STDERR,"(a)",advance="no") indent_char //                &
                                          "[Return] to continue . . ."
         call msFlush(STDERR)
#endif
         read(STDIN,*)
         write(STDERR,*)
      end if

#endif
   end subroutine msPause
!_______________________________________________________________________
!
   function mfIsEmpty( A ) result( bool )

      type(mfArray) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      ! WARNING: This function no longer considers MF_COLON (= MF_ALL),
      !          MF_END or MF_NO_ARG as empty. So be carefull when making
      !          internal checks.

   !------ end of declarations -- execution starts hereafter  ------

      bool = .true.

      if( A%data_type == MF_DT_EMPTY ) then
         if( all( A%shape == 0 ) ) then ! avoids matching special mfArrays
            go to 99
         end if
         if( all( A%shape < 0 ) ) then ! for matching special mfArrays
            call PrintMessage( "mfIsEmpty", "W",                        &
                               "You are currently checking a special mfArray (MF_COLON=MF_ALL, MF_END or", &
                               "MF_NO_ARG), which is not recommended!" )
         end if
      end if

      ! This case considers an mfArray which has only one dimension = 0
      if( any( A%shape == 0 ) ) then
         go to 99
      end if

      bool = .false.

 99   continue

      call msAutoRelease( A )

#endif
   end function mfIsEmpty
!_______________________________________________________________________
!
   function mfIsEqual( a1, a2 ) result( bool )

      type(mfArray), intent(in) :: a1, a2
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      ! Tests if two arrays are numerically equal, whatever the
      ! structure is (i.e. sparse or dense, real or complex).
      ! In case of mixed structures (ex. dense versus sparse),
      ! the sparse array is not converted in a dense one.
      ! In case of two sparse matrices, they are eventually row sorted.
      !
      ! The shapes can mismatch, and can take the zero value.
      !
      ! Comparison between boolean mfArray are excluded.
      !
      ! 'mfIsEqual' returns a scalar logical,
      ! whereas '==' returns a boolean mfArray (in 'mod_ops' module)

      integer :: ncol, nnz, i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      bool = .false.

      call msInitArgs( a1, a2 )

      ! attention : MF_EMPTY and MF_COLON must be distinguishable
      !             (they are both empty !)
      ! -> don't use 'mfIsEmpty' !
      if( a1%data_type == MF_DT_EMPTY .and. a2%data_type == MF_DT_EMPTY ) then
         if( all( a1%shape == a2%shape ) ) then
            bool = .true.
         end if
         go to 99
      end if

      if( a1%data_type == MF_DT_EMPTY .or. a2%data_type == MF_DT_EMPTY ) then
         ! no Warning : testing whether an mfArray is empty can be
         !              arise very often !
         go to 99
      end if

      if( mfIsLogical(a1) .or. mfIsLogical(a2) ) then
         call PrintMessage( "mfIsEqual", "W",                           &
                            "this function cannot be applied to logical!", &
                            "Result will be FALSE." )
         go to 99
      end if

      if( any( a1%shape /= a2%shape ) ) then
         ! no warning
         go to 99
      end if

      ! hereafter, shapes are the same
      if( a1%data_type == MF_DT_DBLE .and. a2%data_type == MF_DT_DBLE ) then
         if( any( a1%double /= a2%double ) ) go to 99
      else if( a1%data_type == MF_DT_DBLE .and. a2%data_type == MF_DT_CMPLX ) then
         if( any( a1%double /= a2%cmplx ) ) go to 99
      else if( a1%data_type == MF_DT_CMPLX .and. a2%data_type == MF_DT_DBLE ) then
         if( any( a1%cmplx /= a2%double ) ) go to 99
      else if( a1%data_type == MF_DT_CMPLX .and. a2%data_type == MF_DT_CMPLX ) then
         if( any( a1%cmplx /= a2%cmplx ) ) go to 99
      else if( a1%data_type == MF_DT_PERM_VEC .and. a2%data_type == MF_DT_PERM_VEC ) then
         if( any( a1%i /= a2%i ) ) go to 99
      else if( a1%data_type == MF_DT_SP_DBLE .and. a2%data_type == MF_DT_SP_DBLE ) then
         ncol = a1%shape(2)
         nnz = a1%j(ncol+1)-1
         if( nnz /= a2%j(ncol+1)-1 ) go to 99
         ! lines must be sorted !
         if( a1%row_sorted /= TRUE ) then
            call msRowSort(a1)
         end if
         if( a2%row_sorted /= TRUE ) then
            call msRowSort(a2)
         end if
         ! values must be the same, even if nzmax doesn't match !
         if( any( a1%a(1:nnz) /= a2%a(1:nnz) ) ) go to 99
         if( any( a1%i(1:nnz) /= a2%i(1:nnz) ) ) go to 99
         if( any( a1%j /= a2%j ) ) go to 99
      else if( a1%data_type == MF_DT_SP_CMPLX .and. a2%data_type == MF_DT_SP_CMPLX ) then
         ncol = a1%shape(2)
         nnz = a1%j(ncol+1)-1
         if( nnz /= a2%j(ncol+1)-1 ) go to 99
         ! lines must be sorted !
         if( a1%row_sorted /= TRUE ) then
            call msRowSort(a1)
         end if
         if( a2%row_sorted /= TRUE ) then
            call msRowSort(a2)
         end if
         ! values must be the same, even if nzmax doesn't match !
         if( any( a1%z(1:nnz) /= a2%z(1:nnz) ) ) go to 99
         if( any( a1%i(1:nnz) /= a2%i(1:nnz) ) ) go to 99
         if( any( a1%j /= a2%j ) ) go to 99
      else if( mfIsSparse(a1) .neqv. mfIsSparse(a2) ) then
         ! mixed structure
         if( mfIsSparse(a1) ) then
            ncol = a1%shape(2)
            nnz = a1%j(ncol+1)-1
            ! getting the NNZ of the dense matrix
            if( mfNnz(a2) == nnz ) then
               ! parsing the sparse structure
               do j = 1, ncol
                  do i = a1%j(j), a1%j(j+1)-1
                     if( a1%data_type == MF_DT_SP_DBLE .and.            &
                         a2%data_type == MF_DT_DBLE ) then
                        if( a1%a(i) /= a2%double(i,j) ) then
                           go to 99
                        end if
                     else if( a1%data_type == MF_DT_SP_CMPLX .and.      &
                              a2%data_type == MF_DT_CMPLX ) then
                        if( a1%z(i) /= a2%cmplx(i,j) ) then
                           go to 99
                        end if
                     else if( a1%data_type == MF_DT_SP_DBLE .and.       &
                              a2%data_type == MF_DT_CMPLX ) then
                        if( cmplx(a1%a(i),kind=MF_DOUBLE) /= a2%cmplx(i,j) ) then
                           go to 99
                        end if
                     else if( a1%data_type == MF_DT_SP_CMPLX .and.      &
                              a2%data_type == MF_DT_DBLE ) then
                        if( a1%z(i) /= cmplx(a2%double(i,j),kind=MF_DOUBLE) ) then
                           go to 99
                        end if
                     end if
                  end do
               end do
            else
               go to 99
            end if
         else if( mfIsSparse(a2) ) then
            ncol = a2%shape(2)
            nnz = a2%j(ncol+1)-1
            ! getting the NNZ of the dense matrix
            if( mfNnz(a1) == nnz ) then
               ! parsing the sparse structure
               do j = 1, ncol
                  do i = a2%j(j), a2%j(j+1)-1
                     if( a2%data_type == MF_DT_SP_DBLE .and.            &
                         a1%data_type == MF_DT_DBLE ) then
                        if( a2%a(i) /= a1%double(i,j) ) then
                           go to 99
                        end if
                     else if( a2%data_type == MF_DT_SP_CMPLX .and.      &
                              a1%data_type == MF_DT_CMPLX ) then
                        if( a2%z(i) /= a1%cmplx(i,j) ) then
                           go to 99
                        end if
                     else if( a2%data_type == MF_DT_SP_DBLE .and.       &
                              a1%data_type == MF_DT_CMPLX ) then
                        if( cmplx(a2%a(i),kind=MF_DOUBLE) /= a1%cmplx(i,j) ) then
                           go to 99
                        end if
                     else if( a2%data_type == MF_DT_SP_CMPLX .and.      &
                              a1%data_type == MF_DT_DBLE ) then
                        if( a2%z(i) /= cmplx(a1%double(i,j),kind=MF_DOUBLE) ) then
                           go to 99
                        end if
                     end if
                  end do
               end do
            else
               go to 99
            end if
         end if
      else
         call PrintMessage( "mfIsEqual", "E",                           &
                            "case not supported!",                      &
                            "(perhaps sparse real/cmplx, you should convert one mfArray)" )
         go to 99
      end if

      bool = .true.

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a1%units, a2%units, status )
         if( status /= 0 ) then
            call PrintMessage( "mfIsEqual", "W",                        &
                               "the physical dimensions of the two mfArray's", &
                               "are not consistent!" )
            go to 99
         end if
      end if

 99   continue

      call msFreeArgs( a1, a2 )

      call msAutoRelease( a1, a2 )

#endif
   end function mfIsEqual
!_______________________________________________________________________
!
   function mfIsNotEqual( a1, a2 ) result( bool )

      type(mfArray), intent(in) :: a1, a2
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( a1, a2 )

      if( mfIsEqual(a1,a2) ) then
         bool = .false.
      else
         bool = .true.
      end if

      call msFreeArgs( a1, a2 )

      call msAutoRelease( a1, a2 )

#endif
   end function mfIsNotEqual
!_______________________________________________________________________
!
   function mfFull( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      if( A%data_type == MF_DT_EMPTY ) then

         call PrintMessage( "mfFull", "W",                              &
                            "arg. is empty." )

      else if( A%data_type == MF_DT_DBLE .or.                           &
               A%data_type == MF_DT_BOOL      ) then

         call PrintMessage( "mfFull", "W",                              &
                            "arg. is already full." )

         out%data_type = A%data_type
         out%shape = A%shape
         if( A%status_temporary .and. .not. A%status_restricted ) then
            out%double => A%double
            A%status_temporary = .false.
         else
            allocate( out%double(out%shape(1),out%shape(2)) )

            out%double(:,:) = A%double(:,:)
         end if

      else if( A%data_type == MF_DT_CMPLX ) then

         call PrintMessage( "mfFull", "W",                              &
                            "arg. is already full." )

         out%data_type = A%data_type
         out%shape = A%shape
         if( A%status_temporary .and. .not. A%status_restricted ) then
            out%cmplx => A%cmplx
            A%status_temporary = .false.
         else
            allocate( out%cmplx(out%shape(1),out%shape(2)) )

            out%cmplx(:,:) = A%cmplx(:,:)
         end if

      else if( A%data_type == MF_DT_SP_DBLE ) then

         out%data_type = MF_DT_DBLE
         out%shape = A%shape
         allocate( out%double(out%shape(1),out%shape(2)) )

         out%double(:,:) = 0.0d0
         do j = 1, A%shape(2)
            do i = A%j(j), A%j(j+1)-1
               out%double( A%i(i), j ) = A%a(i)
            end do
         end do

      else if( A%data_type == MF_DT_SP_BOOL ) then

         out%data_type = MF_DT_BOOL
         out%shape = A%shape
         allocate( out%double(out%shape(1),out%shape(2)) )

         out%double(:,:) = 0.0d0
         do j = 1, A%shape(2)
            do i = A%j(j), A%j(j+1)-1
               out%double( A%i(i), j ) = A%a(i)
            end do
         end do

      else if( A%data_type == MF_DT_SP_CMPLX ) then

         out%data_type = MF_DT_CMPLX
         out%shape = A%shape
         allocate( out%cmplx(out%shape(1),out%shape(2)) )

         out%cmplx(:,:) = 0.0d0
         do j = 1, A%shape(2)
            do i = A%j(j), A%j(j+1)-1
               out%cmplx( A%i(i), j ) = A%z(i)
            end do
         end do

      else

         call PrintMessage( "mfFull", "E",                              &
                            "unknown data type!" )
         return
      end if

      out%prop = A%prop

      if( mf_phys_units ) then
         out%units(:) = a%units(:)
      end if

      out%status_temporary = .true.

      call msAutoRelease( A )

#endif
   end function mfFull
!_______________________________________________________________________
!
   function mfIsLogical( A ) result( bool )

      type(mfArray), intent(in) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( A%data_type == MF_DT_BOOL .or. A%data_type == MF_DT_SP_BOOL ) then
         bool = .true.
      else
         bool = .false.
      end if

      call msAutoRelease( A )

#endif
   end function mfIsLogical
!_______________________________________________________________________
!
   function mfIsReal( A ) result( bool )

      type(mfArray), intent(in) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( A%data_type == MF_DT_DBLE .or.                                &
          A%data_type == MF_DT_SP_DBLE ) then
         bool = .true.
      else
         bool = .false.
      end if

      call msAutoRelease( A )

#endif
   end function mfIsReal
!_______________________________________________________________________
!
   function mfIsComplex( A ) result( bool )

      type(mfArray), intent(in) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( A%data_type == MF_DT_CMPLX .or.                               &
          A%data_type == MF_DT_SP_CMPLX ) then
         bool = .true.
      else
         bool = .false.
      end if

      call msAutoRelease( A )

#endif
   end function mfIsComplex
!_______________________________________________________________________
!
   function mfIsNumeric( A ) result( bool )

      type(mfArray), intent(in) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( A%data_type == MF_DT_DBLE .or. A%data_type == MF_DT_CMPLX .or. &
          A%data_type == MF_DT_SP_DBLE .or. A%data_type == MF_DT_SP_CMPLX ) then
         bool = .true.
      else
         bool = .false.
      end if

      call msAutoRelease( A )

#endif
   end function mfIsNumeric
!_______________________________________________________________________
!
   function mfIsDense( A ) result( bool )

      type(mfArray) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( A%data_type == MF_DT_DBLE .or.                                &
          A%data_type == MF_DT_CMPLX .or.                               &
          A%data_type == MF_DT_BOOL )   then
         bool = .true.
      else
         bool = .false.
      end if

      call msAutoRelease( A )

#endif
   end function mfIsDense
!_______________________________________________________________________
!
   function mfIsSparse( A ) result( bool )

      type(mfArray) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( A%data_type == MF_DT_SP_DBLE .or.                             &
          A%data_type == MF_DT_SP_CMPLX .or.                            &
          A%data_type == MF_DT_SP_BOOL )   then
         bool = .true.
      else
         bool = .false.
      end if

      call msAutoRelease( A )

#endif
   end function mfIsSparse
!_______________________________________________________________________
!
   function mfIsPerm( A ) result( bool )

      type(mfArray) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( A%data_type == MF_DT_PERM_VEC )   then
         bool = .true.
      else
         bool = .false.
      end if

      call msAutoRelease( A )

#endif
   end function mfIsPerm
!_______________________________________________________________________
!
   subroutine msFormat( mantissa, exponent )

      character(len=*), intent(in), optional :: mantissa
      character(len=*), intent(in), optional :: exponent
      !------ API end ------

#ifdef _DEVLP
      character(len=5) :: mant

   !------ end of declarations -- execution starts hereafter  ------

      mf_display_hexa = .false.

      if( present(mantissa) ) then
         mant = to_lower(mantissa)
         if( mant /= "short" .and. mant /= "long" .and. mant /= "hex" ) then
            call PrintMessage( "msFormat", "E",                         &
                               "mantissa bad value:",                   &
                               "this optional arg. must be equal to 'short', 'long' or 'hex'" )
            return
         end if
      else
         mant = "short"
      end if

      if( mant == "long" ) then
         mf_short_mantissa = .false.
      else if( mant == "short" ) then
         mf_short_mantissa = .true.
      else if( mant == "hex" ) then
         mf_display_hexa = .true.
         if( present(exponent) ) then
            call PrintMessage( "msFormat", "E",                         &
                               "'exponent' arg. ignored:",              &
                               "'exponent' arg. cannot be present for hexadecimal format" )
         end if
         return
      end if

      if( present(exponent) ) then
         mf_display_exponent = to_lower(exponent)
         if( mf_display_exponent /= "auto" .and.                        &
             mf_display_exponent /= "sci"  .and.                        &
             mf_display_exponent /= "eng" ) then
            call PrintMessage( "msFormat", "E",                         &
                               "exponent bad value:",                   &
                               "this optional arg. must be equal to 'auto', 'sci' or 'eng'" )
            return
         end if
      else
         mf_display_exponent = "auto"
      end if

#endif
   end subroutine msFormat
!_______________________________________________________________________
!
   subroutine msSetMsgLevel( level )

      integer, intent(in) :: level
      !------ API end ------

#ifdef _DEVLP
      ! Msg level = 3 : all messages are printed (verbose mode)
      !             2 : messages of kind 'ERROR' and 'Warning' are
      !                 printed [default]
      !             1 : only messages of kind 'ERROR' are printed
      !             0 : nothing is printed (quiet mode)
      !                 (pause are ignored !)

   !------ end of declarations -- execution starts hereafter  ------

      if( 0 <= level .and. level <= 3 ) then
         mf_message_level = level
      else
         call PrintMessage( "msSetMsgLevel", "W",                       &
                            "bad value for message level!",             &
                            "[message level reset to default value: 2]" )
         mf_message_level = 2
      end if

#endif
   end subroutine msSetMsgLevel
!_______________________________________________________________________
!
   function mfGetMsgLevel() result( level )

      integer :: level
      !------ API end ------

#ifdef _DEVLP
      ! cf. msSetMsgLevel

   !------ end of declarations -- execution starts hereafter  ------

      level = mf_message_level

#endif
   end function mfGetMsgLevel
!_______________________________________________________________________
!
   subroutine msSetTrbLevel( when, level )

      character(len=*), intent(in), optional :: when
      integer,          intent(in), optional :: level
      !------ API end ------

#ifdef _DEVLP
      ! when = "none"  : 0           no traceback
      !        "auto"  : 1 [default] traceback only for ERRORS
      !        "all"   : 3           traceback for ERRORS, Warning and infos
      !
      ! To trigger a traceback for ERRORS and Warnings, the only
      ! possibity is to use the 'level' argument with the value 2.

   !------ end of declarations -- execution starts hereafter  ------

      if( present(when) ) then

         if( present(level) ) then

            call PrintMessage( "msSetTrbLevel", "E",                    &
                               "only one argument is allowed!" )
            return

         else ! .not. present(level)

            select case( to_lower(when) )
               case( "none" )
                  mf_traceback_level = 0
               case( "auto" )
                  mf_traceback_level = 1
               case( "all" )
                  mf_traceback_level = 2
               case default
                  call PrintMessage( "msSetTrbLevel", "W",              &
                                     "bad value for argument!",         &
                                     "[traceback level reset to default value: 'auto']" )
                  mf_traceback_level = 1
            end select

         end if

      else ! .not. present(when)

         if( present(level) ) then

            if( level < 0 .or. 3 < level ) then
               call PrintMessage( "msSetTrbLevel", "E",                 &
                                  "'level' arg is out-of-range!" )
               return
            end if
            mf_traceback_level = level

         else ! .not. present(level)

            call PrintMessage( "msSetTrbLevel", "E",                    &
                               "one argument is expected!" )
            return

         end if

      end if

#endif
   end subroutine msSetTrbLevel
!_______________________________________________________________________
!
   function mfGetTrbLevel() result( when )

      character(len=4) :: when
      !------ API end ------

#ifdef _DEVLP
      ! cf. msSetMsgTrbck

   !------ end of declarations -- execution starts hereafter  ------

      select case( mf_traceback_level )
         case( 0 )
            when = "none"
         case( 1 )
            when = "auto"
         case( 2 )
            when = "all"
      end select

#endif
   end function mfGetTrbLevel
!_______________________________________________________________________
!
   subroutine msAutoRelease( x1,                                        &
                             x2, x3, x4, x5, x6, x7 )

      type(mfArray) :: x1
      type(mfArray), optional :: x2, x3, x4, x5, x6, x7
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      ! conditional Release (only if tempo)

      call Implem_msAutoRelease( x1 )

      ! this routine may be called a great number of times:
      ! the following design for testing optional arguments
      ! is efficient, but the routine cannot be called by the
      ! keywords 'xi' any more.

      if( present(x2) ) then
         call Implem_msAutoRelease( x2 )
         if( present(x3) ) then
            call Implem_msAutoRelease( x3 )
            if( present(x4) ) then
               call Implem_msAutoRelease( x4 )
               if( present(x5) ) then
                  call Implem_msAutoRelease( x5 )
                  if( present(x6) ) then
                     call Implem_msAutoRelease( x6 )
                     if( present(x7) ) then
                        call Implem_msAutoRelease( x7 )
                     end if
                  end if
               end if
            end if
         end if
      end if

#endif
   end subroutine msAutoRelease
!_______________________________________________________________________
!
   subroutine Implem_msAutoRelease( x )

      type(mfArray) :: x
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( x%status_temporary .and. (x%level_protected==0)               &
                             .and. (.not. x%status_restricted) ) then

         if( x%crc_stored .or. x%level_locked > 0 ) then
            call PrintMessage( "msAutoRelease", "E",                    &
                               "attempting to release an mfArray",      &
                               "while it is yet pointed by an f90 array!" )
            return
         end if

         if( associated( x%double ) ) then
            deallocate( x%double )

         else if( associated( x%cmplx ) ) then
            deallocate( x%cmplx )

         else if( associated( x%a ) ) then
            deallocate( x%a )

            deallocate( x%i )

            deallocate( x%j )

         else if( associated( x%z ) ) then
            deallocate( x%z )

            deallocate( x%i )

            deallocate( x%j )

         else
            call PrintMessage( "msAutoRelease", "I",                    &
                               "mfArray not allocated!" )
         end if

         ! for a vector permutation
         if( x%data_type == MF_DT_PERM_VEC ) then
            deallocate( x%i )

         end if

         x%data_type         = MF_DT_EMPTY
         x%shape             = [ 0, 0 ]
         x%row_sorted        = UNKNOWN
         x%prop              = PROP_ALL_UNKNOWN
         x%level_locked      = zero_kind_1
         x%crc_stored        = .false.
         x%crc               = 0
         x%status_temporary  = .false.
         x%level_protected   = zero_kind_1
         x%status_restricted = .false.
         x%units             = UNIT_ZERO_DIM

      end if

#endif
   end subroutine Implem_msAutoRelease
!_______________________________________________________________________
!
   subroutine msInitArgs( x1,                                           &
                          x2, x3, x4, x5, x6, x7 )

      type(mfArray) :: x1
      type(mfArray), optional :: x2, x3, x4, x5, x6, x7
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      call Implem_msInitArgs( x1 )

      ! this routine may be called a great number of times:
      ! the following design for testing optional arguments
      ! is efficient, but the routine cannot be called by the
      ! keywords 'xi' any more.

      if( present(x2) ) then
         call Implem_msInitArgs( x2 )
         if( present(x3) ) then
            call Implem_msInitArgs( x3 )
            if( present(x4) ) then
               call Implem_msInitArgs( x4 )
               if( present(x5) ) then
                  call Implem_msInitArgs( x5 )
                  if( present(x6) ) then
                     call Implem_msInitArgs( x6 )
                     if( present(x7) ) then
                        call Implem_msInitArgs( x7 )
                     end if
                  end if
               end if
            end if
         end if
      end if

#endif
   end subroutine msInitArgs
!_______________________________________________________________________
!
   subroutine Implem_msInitArgs( x )

      type(mfArray) :: x
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      ! Parameter mfArrays are not modified...
      if( x%parameter ) return

      ! ...but all other mfArray, including with special shapes
      ! (as EndIndex objects) may be modified
      x%level_protected = x%level_protected + one_kind_1

#endif
   end subroutine Implem_msInitArgs
!_______________________________________________________________________
!
   subroutine msFreeArgs( x1,                                           &
                          x2, x3, x4, x5, x6, x7 )

      type(mfArray) :: x1
      type(mfArray), optional :: x2, x3, x4, x5, x6, x7
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      call Implem_msFreeArgs( x1 )

      ! This routine may be called a great number of times:
      ! the following design for testing optional arguments
      ! is efficient, but the routine cannot be called by the
      ! keywords 'xi' any more.

      if( present(x2) ) then
         call Implem_msFreeArgs( x2 )
         if( present(x3) ) then
            call Implem_msFreeArgs( x3 )
            if( present(x4) ) then
               call Implem_msFreeArgs( x4 )
               if( present(x5) ) then
                  call Implem_msFreeArgs( x5 )
                  if( present(x6) ) then
                     call Implem_msFreeArgs( x6 )
                     if( present(x7) ) then
                        call Implem_msFreeArgs( x7 )
                     end if
                  end if
               end if
            end if
         end if
      end if

#endif
   end subroutine msFreeArgs
!_______________________________________________________________________
!
   subroutine Implem_msFreeArgs( x )

      type(mfArray) :: x
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

!!      ! avoid MF_EMPTY or MF_COLON, because these two objects are
!!      ! parameters (hence, non modifiable !)
!!      if( x%shape(1) <= 0 .or. x%shape(2) <= 0 ) return

! now (2017-04-23), all Muesli constants (MF_EMPTY, MF_COLON=MF_ALL,
! MF_END, MF_NO_ARG, ... have their 'parameter' field set to TRUE:
if( x%parameter ) return

      if( x%level_protected == 0 ) then
         call PrintMessage( "msFreeArgs", "E",                          &
                            "level_protected is null!",                 &
                            "forgot to call msInitArgs?" )
         return
      end if

      x%level_protected = x%level_protected - one_kind_1

#endif
   end subroutine Implem_msFreeArgs
!_______________________________________________________________________
!
   subroutine msSilentRelease( x1,                                      &
                               x2, x3, x4, x5, x6, x7 )

      type(mfArray) :: x1
      type(mfArray), optional :: x2, x3, x4, x5, x6, x7
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      ! Unconditional Release,
      ! without warning if some mfArrays are already empty.

      call Implem_msSilentRelease( x1, 1 )

      ! This routine may be called a great number of times:
      ! the following design for testing optional arguments is efficient,
      ! but the routine cannot be called by the keywords 'xi' any more.

      if( present(x2) ) then
         call Implem_msSilentRelease( x2, 2 )
         if( present(x3) ) then
            call Implem_msSilentRelease( x3, 3 )
            if( present(x4) ) then
               call Implem_msSilentRelease( x4, 4 )
               if( present(x5) ) then
                  call Implem_msSilentRelease( x5, 5 )
                  if( present(x6) ) then
                     call Implem_msSilentRelease( x6, 6 )
                     if( present(x7) ) then
                        call Implem_msSilentRelease( x7, 7 )
                     end if
                  end if
               end if
            end if
         end if
      end if

#endif
   end subroutine msSilentRelease
!_______________________________________________________________________
!
   subroutine Implem_msSilentRelease( x, i )

      type(mfArray) :: x
      integer, intent(in) :: i
      !------ API end ------

#ifdef _DEVLP
      character(len=1) :: arg_nb

   !------ end of declarations -- execution starts hereafter  ------

      write( arg_nb, "(I1)" ) i

      if( x%parameter ) then
         call PrintMessage( "msSilentRelease", "E",                     &
                            "cannot release an mfArray (arg #" // arg_nb // ")!", &
                            "it is a protected array (pseudo-parameter)." )
         return
      end if

      if( x%crc_stored .or. x%level_locked > 0 ) then
         call PrintMessage( "msSilentRelease", "E",                     &
                            "attempting to release an mfArray (arg #" // arg_nb // ")", &
                            "while it is yet pointed by an f90 array!", &
                            "[please use 'msFreePointer']" )
         return
      end if

      if( associated( x%double ) ) then
         if( .not. x%status_restricted ) then
            deallocate( x%double )

         else
            x%double => null()
         end if
      else if( associated( x%cmplx ) ) then
         if( .not. x%status_restricted ) then
            deallocate( x%cmplx )

         else
            x%cmplx => null()
         end if
      else if( associated( x%a ) ) then
         deallocate( x%a )

         deallocate( x%i )

         deallocate( x%j )

         if( associated( x%umf4_ptr_numeric ) ) then
            call umf4fnum_d( x%umf4_ptr_numeric )
            deallocate( x%umf4_ptr_numeric )

         end if
      else if( associated( x%z ) ) then
         deallocate( x%z )

         deallocate( x%i )

         deallocate( x%j )

         if( associated( x%umf4_ptr_numeric ) ) then
            call umf4fnum_z( x%umf4_ptr_numeric )
            deallocate( x%umf4_ptr_numeric )

         end if
      end if

      ! for a vector permutation
      if( x%data_type == MF_DT_PERM_VEC ) then
         deallocate( x%i )

      end if

      x%data_type         = MF_DT_EMPTY
      x%shape             = [ 0, 0 ]
      x%row_sorted        = UNKNOWN
      x%prop              = PROP_ALL_UNKNOWN
      x%level_locked      = zero_kind_1
      x%crc_stored        = .false.
      x%crc               = 0
      x%status_temporary  = .false.
      x%level_protected   = zero_kind_1
      x%status_restricted = .false.
      x%units             = UNIT_ZERO_DIM

#endif
   end subroutine Implem_msSilentRelease
!_______________________________________________________________________
!
   subroutine msRelease_mfArray( x1,                                    &
                                 x2, x3, x4, x5, x6, x7 )

      type(mfArray)           :: x1
      type(mfArray), optional :: x2, x3, x4, x5, x6, x7
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      ! Unconditional Release.

      call Implem_msRelease( x1, 1 )

      ! This routine may be called a great number of times:
      ! the following design for testing optional arguments is efficient,
      ! but the routine cannot be called by the keywords 'xi' any more.

      if( present(x2) ) then
         call Implem_msRelease( x2, 2 )
         if( present(x3) ) then
            call Implem_msRelease( x3, 3 )
            if( present(x4) ) then
               call Implem_msRelease( x4, 4 )
               if( present(x5) ) then
                  call Implem_msRelease( x5, 5 )
                  if( present(x6) ) then
                     call Implem_msRelease( x6, 6 )
                     if( present(x7) ) then
                        call Implem_msRelease( x7, 7 )
                     end if
                  end if
               end if
            end if
         end if
      end if

#endif
   end subroutine msRelease_mfArray
!_______________________________________________________________________
!
   subroutine Implem_msRelease( x, i )

      type(mfArray) :: x
      integer, intent(in) :: i
      !------ API end ------

#ifdef _DEVLP
      character(len=24) :: msg

   !------ end of declarations -- execution starts hereafter  ------

      if( x%parameter ) then
         call PrintMessage( "msRelease", "E",                           &
                            "cannot release mfArray 'x'!",              &
                            "it is a protected array (pseudo-parameter)." )
         return
      end if

      if( x%crc_stored .or. x%level_locked > 0 ) then
         call PrintMessage( "msRelease", "E",                           &
                            "attempting to release an mfArray",         &
                            "while it is yet pointed by an f90 array!", &
                            "[please use 'msFreePointer']" )
         return
      end if

      if( associated( x%double ) ) then
         if( .not. x%status_restricted ) then
            deallocate( x%double )

         else
            x%double => null()
         end if
      else if( associated( x%cmplx ) ) then
         if( .not. x%status_restricted ) then
            deallocate( x%cmplx )

         else
            x%cmplx => null()
         end if
      else if( associated( x%a ) ) then
         deallocate( x%a )

         deallocate( x%i )

         deallocate( x%j )

         if( associated( x%umf4_ptr_numeric ) ) then
            call umf4fnum_d( x%umf4_ptr_numeric )
            deallocate( x%umf4_ptr_numeric )

         end if
      else if( associated( x%z ) ) then
         deallocate( x%z )

         deallocate( x%i )

         deallocate( x%j )

         if( associated( x%umf4_ptr_numeric ) ) then
            call umf4fnum_z( x%umf4_ptr_numeric )
            deallocate( x%umf4_ptr_numeric )

         end if
      else
         write(msg,"(I0)") i
         msg = "arg #" // trim(msg) // " is empty."
         call PrintMessage( "msRelease", "I",                           &
                            msg )
      end if

      ! for a vector permutation
      if( x%data_type == MF_DT_PERM_VEC ) then
         deallocate( x%i )

      end if

      x%data_type         = MF_DT_EMPTY
      x%shape             = [ 0, 0 ]
      x%row_sorted        = UNKNOWN
      x%prop              = PROP_ALL_UNKNOWN
      x%level_locked      = zero_kind_1
      x%crc_stored        = .false.
      x%crc               = 0
      x%status_temporary  = .false.
      x%level_protected   = zero_kind_1
      x%status_restricted = .false.
      x%units             = UNIT_ZERO_DIM

#endif
   end subroutine Implem_msRelease
!_______________________________________________________________________
!
   subroutine msRelease_mf_Int_List( x )

      type(mf_Int_List) :: x
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( allocated(x%list)) then
         deallocate( x%list )
      end if

#endif
   end subroutine msRelease_mf_Int_List
!_______________________________________________________________________
!
   subroutine msRelease_mf_Int_List_vec( x )

      type(mf_Int_List) :: x(:)
      !------ API end ------

#ifdef _DEVLP
      integer :: n, i

   !------ end of declarations -- execution starts hereafter  ------

      n = size(x)

      do i = 1, n
         if( allocated(x(i)%list)) then
            deallocate( x(i)%list )
         end if
      end do

#endif
   end subroutine msRelease_mf_Int_List_vec
!_______________________________________________________________________
!
   subroutine msRelease_mf_Real_List( x )

      type(mf_Real_List) :: x
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( allocated(x%list)) then
         deallocate( x%list )
      end if

#endif
   end subroutine msRelease_mf_Real_List
!_______________________________________________________________________
!
   subroutine msRelease_mf_Real_List_vec( x )

      type(mf_Real_List) :: x(:)
      !------ API end ------

#ifdef _DEVLP
      integer :: n, i

   !------ end of declarations -- execution starts hereafter  ------

      n = size(x)

      do i = 1, n
         if( allocated(x(i)%list)) then
            deallocate( x(i)%list )
         end if
      end do

#endif
   end subroutine msRelease_mf_Real_List_vec
!_______________________________________________________________________
!
   subroutine set_status_tempo_to_false( a )

      type(mfArray) :: a
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      ! used in routines which must have intent(in) arguments
      ! cf. e.g. 'horiz_concat_mfArray_mfArray' in mod_ops
! BUG (?): actually, (all ?) compilers don't verify that arguments
! are not modified at internal levels...

      a%status_temporary = .false.

#endif
   end subroutine set_status_tempo_to_false
!_______________________________________________________________________
!
   subroutine manual_dealloc_vec( vec )

      real(kind=MF_DOUBLE), pointer :: vec(:)
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      ! used in routines which must have intent(in) arguments
      ! cf. e.g. 'horiz_concat_mfArray_mfArray' in mod_ops
! BUG (?): actually, (all ?) compilers don't verify that arguments
! are not modified at internal levels...

#ifndef _TRACE_MEM_ALLOC
      deallocate( vec )
#else
      call mf_deallocate( array=vec,                                    &
                          file="mod_core.F90", line="???",              &
                          symb="vec", unit="manual_dealloc_vec" )
#endif

#endif
   end subroutine manual_dealloc_vec
!_______________________________________________________________________
!
   subroutine msPointer_matrix_real8( A, f90_ptr, no_crc, intern_call )

      type(mfArray) :: A
      real(kind=MF_DOUBLE), pointer :: f90_ptr(:,:)
      logical, intent(in), optional :: no_crc, intern_call
      !------ API end ------

#ifdef _DEVLP
      ! ~ equiv. to : f90_ptr => A
      !
      ! A cannot be a tempo, because at the end of the current routine,
      ! it will be deleted, and f90_ptr will become hangling.
      !
      ! Optional arg. 'no_crc' should not be used by ordinary users
      ! (therefore, it is intentionally not documented); by setting
      ! no_crc=.true. the user claims that data in A will not be modified
      ! during all the time of pointing, and that therefore the checksum
      ! computation is not needed.
      !
      ! Optional arg. 'internal_call' is set to TRUE only if the call
      ! comes from the Muesli library (either FML or FGL). It allows to
      ! disable some warnings...

      logical :: internal_call

   !------ end of declarations -- execution starts hereafter  ------

      f90_ptr => null()

      if( A%parameter ) then
         call PrintMessage( "msPointer", "E",                           &
                            "'A' cannot be a protected array (pseudo-parameter).", &
                            "resulting pointer will be nullify" )
         return
      end if

      if( A%prop == PROP_ALL_UNKNOWN ) then
         A%crc_stored = .false.
      else
         if( present(no_crc) ) then
            A%crc_stored = .not. no_crc
         else
            A%crc_stored = .true.
         end if
      end if

      if( present(intern_call) ) then
         internal_call = intern_call
      else
         internal_call = .false.
      end if

      if( A%status_temporary ) then
         ! To detect an internal call, we check the 'no_crc' arg presence.
         if( present(no_crc) ) then
            ! Remove completely the message, this could confuse the user;
            ! indeed, even if there is some risk, it would work. It's under
            ! my own responsability...
!### je pourrais, à la limite, remettre le message pour retrouver les endroits
!    du code de FGL où j'appelle msPointer avec un tempo...
!!            call PrintMessage( "msPointer", "W",                        &
!!                               "target is tempo!",                      &
!!                               "this is not recommended in MUESLI",     &
!!                               "(however, this is allowed inside MUESLI)")
         else
            call PrintMessage( "msPointer", "E",                        &
                               "target is tempo!",                      &
                               "this is forbidden in MUESLI" )
            call msAutoRelease( A )
            return
         end if
      end if

      ! Take care to not call user-level routines (like mfIsEmpty): this
      ! should release a tempo A, which is allowed for internal calls...
      if( A%data_type /= MF_DT_DBLE ) then
         if( A%data_type == MF_DT_EMPTY ) then
            call PrintMessage( "msPointer", "E",                        &
                               "target is empty!",                      &
                               "resulting pointer will be nullify" )
            return
         end if

         if( A%data_type == MF_DT_CMPLX ) then
            call PrintMessage( "msPointer", "E",                        &
                               "target is complex! (cannot point from a real ptr)", &
                               "resulting pointer will be nullify" )
            return
         end if

         if( A%data_type == MF_DT_BOOL .or. A%data_type == MF_DT_SP_BOOL ) then
            call PrintMessage( "msPointer", "E",                        &
                               "target is logical!",                    &
                               "resulting pointer will be nullify" )
            return
         end if

         if( A%data_type == MF_DT_SP_DBLE .or. A%data_type == MF_DT_SP_CMPLX ) then
            call PrintMessage( "msPointer", "E",                        &
                               "target is sparse!",                     &
                               "resulting pointer will be nullify" )
            return
         end if

         if( A%data_type == MF_DT_PERM_VEC ) then
            call PrintMessage( "msPointer", "E",                        &
                               "target is a permutation vector!",       &
                               "resulting pointer will be nullify" )
            return
         end if
      end if

      if( A%level_locked == 0 ) then
         ! hide matrix properties (Warning: must be done only once!)
!### TODO: why always hide matrix properties? If no_crc=.true., it should be
!          safe to keep these properties.
         call hide_prop(A%prop)
      else ! A%level_locked > 0
         if( .not. internal_call ) then
            call PrintMessage( "msPointer", "W",                        &
                               "Target is already pointed by a Fortran array.", &
                               "While it is not forbidden to have many different", &
                               "pointers to the same mfArray, it is impossible", &
                               "to check that the pointers are different!")
         end if
      end if

      f90_ptr => A%double

      if( A%crc_stored ) then
         call PrintMessage( "msPointer", "I",                           &
                            "taking CRC-32 bits." )
         ! storing CRC
         A%crc = crc_32_bits( A%double )
      end if

      A%level_locked = A%level_locked + 1

#endif
   end subroutine msPointer_matrix_real8
!_______________________________________________________________________
!
   subroutine msPointer_vector_real8( A, f90_ptr, no_crc, intern_call )

      type(mfArray) :: A
      real(kind=MF_DOUBLE), pointer :: f90_ptr(:)
      logical, intent(in), optional :: no_crc, intern_call
      !------ API end ------

#ifdef _DEVLP
      ! ~ equiv. to : f90_ptr => A
      !
      ! A cannot be a tempo, because at the end of the current routine,
      ! it will be deleted, and f90_ptr will become hangling.
      !
      ! Optional arg. 'no_crc' should not be used by ordinary users
      ! (therefore, it is intentionally not documented); by setting
      ! no_crc=.true. the user claims that data in A will not be modified
      ! during all the time of pointing, and that therefore the checksum
      ! computation is not needed.

      integer :: n, ncol, nnz
      logical :: internal_call

   !------ end of declarations -- execution starts hereafter  ------

      f90_ptr => null()

      if( A%parameter ) then
         call PrintMessage( "msPointer", "E",                           &
                            "'A' cannot be a protected array (pseudo-parameter).", &
                            "resulting pointer will be nullify" )
         return
      end if

      if( A%prop == PROP_ALL_UNKNOWN ) then
         A%crc_stored = .false.
      else
         if( present(no_crc) ) then
            A%crc_stored = .not. no_crc
         else
            A%crc_stored = .true.
         end if
      end if

      if( present(intern_call) ) then
         internal_call = intern_call
      else
         internal_call = .false.
      end if

      if( A%status_temporary ) then
         ! To detect an internal call, we check the 'no_crc' arg presence.
         if( present(no_crc) ) then
            ! Remove completely the message, this could confuse the user;
            ! indeed, even if there is some risk, it would work. It's under
            ! my own responsability...
!!            call PrintMessage( "msPointer", "W",                        &
!!                               "target is tempo!",                      &
!!                               "this is not recommended in MUESLI",     &
!!                               "(however, this is allowed inside MUESLI)")
         else
            call PrintMessage( "msPointer", "E",                        &
                               "target is tempo!",                      &
                               "this is forbidden in MUESLI" )
            call msAutoRelease( A )
            return
         end if
      end if

      ! Take care to not call user-level routines (like mfIsEmpty): this
      ! should release a tempo A, which is allowed for internal calls...
      if( A%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "msPointer", "E",                           &
                            "target is empty!",                         &
                            "resulting pointer will be nullify" )
         return
      end if

      if( A%data_type == MF_DT_CMPLX .or. A%data_type == MF_DT_SP_CMPLX ) then
         call PrintMessage( "msPointer", "E",                           &
                            "target is complex! (cannot point from a real ptr)", &
                            "resulting pointer will be nullify" )
         return
      end if

      if( A%data_type == MF_DT_BOOL .or. A%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "msPointer", "E",                           &
                            "target is logical!",                       &
                            "resulting pointer will be nullify" )
         return
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "msPointer", "E",                           &
                            "target is a permutation vector!",          &
                            "resulting pointer will be nullify" )
         return
      end if

      if( A%level_locked == 0 ) then
         ! hide matrix properties (Warning: must be done only once!)
         call hide_prop(A%prop)
      else ! A%level_locked > 0
         if( .not. internal_call ) then
            call PrintMessage( "msPointer", "W",                        &
                               "Target is already pointed by a Fortran array.", &
                               "While it is not forbidden to have many different", &
                               "pointers to the same mfArray, it is impossible", &
                               "to check that the pointers are different!")
         end if
      end if

      if( A%data_type == MF_DT_SP_DBLE ) then

         if( .not. associated(A%a) ) then
            call PrintMessage( "msPointer", "E",                        &
                               "target is real sparse, but not associated!", &
                               "resulting pointer will be nullify" )
            return
         end if
         ncol = A%shape(2)
         nnz = A%j(ncol+1)-1
         f90_ptr => A%a(1:nnz)

         if( A%crc_stored ) then
            call PrintMessage( "msPointer", "I",                        &
                               "taking CRC-32 bits." )
            ! storing CRC
            A%crc = crc_32_bits( A%a(1:nnz) )
         end if

      else ! dense storage

         ! LongColumn pointer

         n = size( A%double )

         ! the following statement is valid only in Fortran 2003
         ! f90_ptr(1:n) => A%double

         ! the rank remapping is implemented in: mod_mfaux
         f90_ptr => rank_2_to_1_real8( A%double, n )

         if( A%crc_stored ) then
            call PrintMessage( "msPointer", "I",                        &
                               "taking CRC-32 bits." )
            ! storing CRC
            A%crc = crc_32_bits( A%double )
         end if

      end if

      A%level_locked = A%level_locked + 1

#endif
   end subroutine msPointer_vector_real8
!_______________________________________________________________________
!
   subroutine msPointer_matrix_cmplx( A, f90_ptr, no_crc )

      type(mfArray) :: A
      complex(kind=MF_DOUBLE), pointer :: f90_ptr(:,:)
      logical, intent(in), optional :: no_crc
      !------ API end ------

#ifdef _DEVLP
      ! ~ equiv. to : f90_ptr => a
      !
      ! A cannot be a tempo, because at the end of the current routine,
      ! it will be deleted, and f90_ptr will become hangling.
      !
      ! Optional arg. 'no_crc' should not be used by ordinary users
      ! (therefore, it is intentionally not documented); by setting
      ! no_crc=.true. the user claims that data in A will not be modified
      ! during all the time of pointing, and that therefore the checksum
      ! computation is not needed.

   !------ end of declarations -- execution starts hereafter  ------

      f90_ptr => null()

      if( A%parameter ) then
         call PrintMessage( "msPointer", "E",                           &
                            "'A' cannot be a protected array (pseudo-parameter).", &
                            "resulting pointer will be nullify" )
         return
      end if

      if( A%prop == PROP_ALL_UNKNOWN ) then
         A%crc_stored = .false.
      else
         if( present(no_crc) ) then
            A%crc_stored = .not. no_crc
         else
            A%crc_stored = .true.
         end if
      end if

      if( A%status_temporary ) then
         ! To detect an internal call, we check the 'no_crc' arg presence.
         if( present(no_crc) ) then
            ! Remove completely the message, this could confuse the user;
            ! indeed, even if there is some risk, it would work. It's under
            ! my own responsability...
!!            call PrintMessage( "msPointer", "W",                        &
!!                               "target is tempo!",                      &
!!                               "this is not recommended in MUESLI",     &
!!                               "(however, this is allowed inside MUESLI)")
         else
            call PrintMessage( "msPointer", "E",                        &
                               "target is tempo!",                      &
                               "this is forbidden in MUESLI" )
            call msAutoRelease( A )
            return
         end if
      end if

      ! Take care to not call user-level routines (like mfIsEmpty): this
      ! should release a tempo A, which is allowed for internal calls...
      if( A%data_type /= MF_DT_CMPLX ) then
         if( A%data_type == MF_DT_EMPTY ) then
            call PrintMessage( "msPointer", "E",                        &
                               "target is empty!",                      &
                               "resulting pointer will be nullify" )
            return
         end if

         if( A%data_type == MF_DT_DBLE ) then
            call PrintMessage( "msPointer", "E",                        &
                               "target is real! (cannot point from a complex ptr)", &
                               "resulting pointer will be nullify" )
            return
         end if

         if( A%data_type == MF_DT_BOOL .or. A%data_type == MF_DT_SP_BOOL ) then
            call PrintMessage( "msPointer", "E",                        &
                               "target is logical!",                    &
                               "resulting pointer will be nullify" )
            return
         end if

         if( A%data_type == MF_DT_SP_DBLE .or. A%data_type == MF_DT_SP_CMPLX ) then
            call PrintMessage( "msPointer", "E",                        &
                               "target is sparse!",                     &
                               "resulting pointer will be nullify" )
            return
         end if

         if( A%data_type == MF_DT_PERM_VEC ) then
            call PrintMessage( "msPointer", "E",                        &
                               "target is a permutation vector!",       &
                               "resulting pointer will be nullify" )
            return
         end if
      end if

      if( A%level_locked == 0 ) then
         ! hide matrix properties (Warning: must be done only once!)
         call hide_prop(A%prop)
      else ! A%level_locked > 0
         call PrintMessage( "msPointer", "W",                           &
                            "Target is already pointed by a Fortran array.", &
                            "While it is not forbidden to have many different", &
                            "pointers to the same mfArray, it is impossible", &
                            "to check that the pointers are different!")
      end if

      f90_ptr => A%cmplx

      if( A%crc_stored ) then
         call PrintMessage( "msPointer", "I",                           &
                            "taking CRC-32 bits." )
         ! storing CRC
         A%crc = crc_32_bits( A%cmplx )
      end if

      A%level_locked = A%level_locked + 1

#endif
   end subroutine msPointer_matrix_cmplx
!_______________________________________________________________________
!
   subroutine msPointer_vector_cmplx( A, f90_ptr, no_crc )

      type(mfArray) :: A
      complex(kind=MF_DOUBLE), pointer :: f90_ptr(:)
      logical, intent(in), optional :: no_crc
      !------ API end ------

#ifdef _DEVLP
      ! ~ equiv. to : f90_ptr => A
      !
      ! A cannot be a tempo, because at the end of the current routine,
      ! it will be deleted, and f90_ptr will become hangling.
      !
      ! Optional arg. 'no_crc' should not be used by ordinary users
      ! (therefore, it is intentionally not documented); by setting
      ! no_crc=.true. the user claims that data in A will not be modified
      ! during all the time of pointing, and that therefore the checksum
      ! computation is not needed.

      integer :: n, ncol, nnz

   !------ end of declarations -- execution starts hereafter  ------

      f90_ptr => null()

      if( A%parameter ) then
         call PrintMessage( "msPointer", "E",                           &
                            "'A' cannot be a protected array (pseudo-parameter).", &
                            "resulting pointer will be nullify" )
         return
      end if

      if( A%prop == PROP_ALL_UNKNOWN ) then
         A%crc_stored = .false.
      else
         if( present(no_crc) ) then
            A%crc_stored = .not. no_crc
         else
            A%crc_stored = .true.
         end if
      end if

      if( A%status_temporary ) then
         ! To detect an internal call, we check the 'no_crc' arg presence.
         if( present(no_crc) ) then
            ! Remove completely the message, this could confuse the user;
            ! indeed, even if there is some risk, it would work. It's under
            ! my own responsability...
!!            call PrintMessage( "msPointer", "W",                        &
!!                               "target is tempo!",                      &
!!                               "this is not recommended in MUESLI",     &
!!                               "(however, this is allowed inside MUESLI)")
         else
            call PrintMessage( "msPointer", "E",                        &
                               "target is tempo!",                      &
                               "this is forbidden in MUESLI" )
            call msAutoRelease( A )
            return
         end if
      end if

      ! Take care to not call user-level routines (like mfIsEmpty): this
      ! should release a tempo A, which is allowed for internal calls...
      if( A%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "msPointer", "E",                           &
                            "target is empty!",                         &
                            "resulting pointer will be nullify" )
         return
      end if

      if( A%data_type == MF_DT_DBLE .or. A%data_type == MF_DT_SP_DBLE ) then
         call PrintMessage( "msPointer", "E",                           &
                            "target is real! (cannot point from a complex ptr)", &
                            "resulting pointer will be nullify" )
         return
      end if

      if( A%data_type == MF_DT_BOOL .or. A%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "msPointer", "E",                           &
                            "target is logical!",                       &
                            "resulting pointer will be nullify" )
         return
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "msPointer", "E",                           &
                            "target is a permutation vector!",          &
                            "resulting pointer will be nullify" )
         return
      end if

      if( A%level_locked == 0 ) then
         ! hide matrix properties (Warning: must be done only once!)
         call hide_prop(A%prop)
      else ! A%level_locked > 0
         call PrintMessage( "msPointer", "W",                           &
                            "Target is already pointed by a Fortran array.", &
                            "While it is not forbidden to have many different", &
                            "pointers to the same mfArray, it is impossible", &
                            "to check that the pointers are different!")
      end if

      if( A%data_type == MF_DT_SP_CMPLX ) then

         if( .not. associated(A%z) ) then
            call PrintMessage( "msPointer", "E",                        &
                               "target is complex sparse, but not associated!", &
                               "resulting pointer will be nullify" )
            return
         end if
         ncol = A%shape(2)
         nnz = A%j(ncol+1)-1
         f90_ptr => A%z(1:nnz)

         if( A%crc_stored ) then
            call PrintMessage( "msPointer", "I",                        &
                               "taking CRC-32 bits." )
            ! storing CRC
            A%crc = crc_32_bits( A%z(1:nnz) )
         end if

      else ! dense, complex storage

         ! LongColumn pointer

         n = size( A%cmplx )

         ! the following statement is valid only in Fortran 2003
         ! f90_ptr(1:n) => A%cmplx

         ! the rank remapping is implemented in: mod_mfaux
         f90_ptr => rank_2_to_1_cmplx( A%cmplx, n )

         if( A%crc_stored ) then
            call PrintMessage( "msPointer", "I",                        &
                               "taking CRC-32 bits." )
            ! storing CRC
            A%crc = crc_32_bits( A%cmplx )
         end if

      end if

      A%level_locked = A%level_locked + 1

#endif
   end subroutine msPointer_vector_cmplx
!_______________________________________________________________________
!
   subroutine msFreePointer_matrix_real8( A, f90_ptr )

      type(mfArray) :: A
      real(kind=MF_DOUBLE), pointer :: f90_ptr(:,:)
      !------ API end ------

#ifdef _DEVLP
      ! ~ equiv. to : f90_ptr => null()
      !
      ! but modify a flag of the old target

   !------ end of declarations -- execution starts hereafter  ------

      f90_ptr => null()

      if( A%level_locked == 0 ) then
         call PrintMessage( "msFreePointer", "E",                       &
                            "Too much FreePointer for this mfArray!",   &
                            "(nevertheless, f90ptr has been nullified)" )
         return
      end if

      A%level_locked = A%level_locked - 1

      if( A%level_locked == 0 ) then

         ! unhide matrix properties
!### TODO: sauf peut-être si j'ai tenu compte de no_crc...
!          (cf. msPointer_matrix_real8)
         call unhide_prop(A%prop)

         if( A%crc_stored ) then
            call PrintMessage( "msFreePointer", "I",                    &
                               "comparing CRC-32 bits." )
            ! comparing CRC
            crc = crc_32_bits( A%double )
            if( crc /= A%crc ) then
               call PrintMessage( "msFreePointer", "I",                 &
                                  "data has been modified!",            &
                                  "-> erasing matrix properties" )
               call reset_prop(A%prop)
            end if
            A%crc = 0
            A%crc_stored = .false.
         end if

      else ! A%level_locked > 0

         call PrintMessage( "msFreePointer", "I",                       &
                            "Cannot unlock the targeted mfArray A, because this", &
                            "latter is still pointed by another f90 pointer!" )

      end if

#endif
   end subroutine msFreePointer_matrix_real8
!_______________________________________________________________________
!
   subroutine msFreePointer_LongColumn_real8( A, f90_ptr )

      type(mfArray) :: A
      real(kind=MF_DOUBLE), pointer :: f90_ptr(:)
      !------ API end ------

#ifdef _DEVLP
      ! ~ equiv. to : f90_ptr => null()
      !
      ! but modify a flag of the old target

      integer :: ncol, nnz

   !------ end of declarations -- execution starts hereafter  ------

      f90_ptr => null()

      if( A%level_locked == 0 ) then
         call PrintMessage( "msFreePointer", "E",                       &
                            "too much FreePointer for this mfArray!",   &
                            "(nevertheless, f90ptr has been nullified)" )
         return
      end if

      A%level_locked = A%level_locked - 1

      if( A%level_locked == 0 ) then

         ! unhide matrix properties
         call unhide_prop(A%prop)

         if( A%crc_stored ) then
            call PrintMessage( "msFreePointer", "I",                    &
                               "comparing CRC-32 bits." )

            if( A%data_type == MF_DT_SP_DBLE ) then
               ncol = A%shape(2)
               nnz = A%j(ncol+1)-1
               ! comparing CRC
               crc = crc_32_bits( A%a(1:nnz) )
            else
               ! comparing CRC
               crc = crc_32_bits( A%double )
            end if

            if( crc /= A%crc ) then
               call PrintMessage( "msFreePointer", "I",                 &
                                  "data has been modified!",            &
                                  "-> erasing matrix properties" )
               call reset_prop(A%prop)
            end if
            A%crc = 0
            A%crc_stored = .false.
         end if

      else ! A%level_locked > 0

         call PrintMessage( "msFreePointer", "I",                       &
                            "Cannot unlock the targeted mfArray A, because this", &
                            "latter is still pointed by another f90 pointer!" )

      end if

#endif
   end subroutine msFreePointer_LongColumn_real8
!_______________________________________________________________________
!
   subroutine msFreePointer_matrix_cmplx( A, f90_ptr )

      type(mfArray) :: A
      complex(kind=MF_DOUBLE), pointer :: f90_ptr(:,:)
      !------ API end ------

#ifdef _DEVLP
      ! ~ equiv. to : f90_ptr => null()
      !
      ! but modify a flag of the old target

   !------ end of declarations -- execution starts hereafter  ------

      f90_ptr => null()

      if( A%level_locked == 0 ) then
         call PrintMessage( "msFreePointer", "E",                       &
                            "too much FreePointer for this mfArray!",   &
                            "(nevertheless, f90ptr has been nullified)" )
         return
      end if

      A%level_locked = A%level_locked - 1

      if( A%level_locked == 0 ) then

         ! unhide matrix properties
         call unhide_prop(A%prop)

         if( A%crc_stored ) then
            call PrintMessage( "msFreePointer", "I",                    &
                               "comparing CRC-32 bits." )
            ! comparing CRC
            crc = crc_32_bits( A%cmplx )
            if( crc /= A%crc ) then
               call PrintMessage( "msFreePointer", "I",                 &
                                  "data has been modified!",            &
                                  "-> erasing matrix properties" )
               call reset_prop(A%prop)
            end if
            A%crc = 0
            A%crc_stored = .false.
         end if

      else ! A%level_locked > 0

         call PrintMessage( "msFreePointer", "I",                       &
                            "Cannot unlock the targeted mfArray A, because this", &
                            "latter is still pointed by another f90 pointer!" )

      end if

#endif
   end subroutine msFreePointer_matrix_cmplx
!_______________________________________________________________________
!
   subroutine msFreePointer_LongColumn_cmplx( A, f90_ptr )

      type(mfArray) :: A
      complex(kind=MF_DOUBLE), pointer :: f90_ptr(:)
      !------ API end ------

#ifdef _DEVLP
      ! ~ equiv. to : f90_ptr => null()
      !
      ! but modify a flag of the old target

      integer :: ncol, nnz

   !------ end of declarations -- execution starts hereafter  ------

      f90_ptr => null()

      if( A%level_locked == 0 ) then
         call PrintMessage( "msFreePointer", "E",                       &
                            "too much FreePointer for this mfArray!",   &
                            "(nevertheless, f90ptr has been nullified)" )
         return
      end if

      A%level_locked = A%level_locked - 1

      if( A%level_locked == 0 ) then

         ! unhide matrix properties
         call unhide_prop(A%prop)

         if( A%crc_stored ) then
            call PrintMessage( "msFreePointer", "I",                    &
                               "comparing CRC-32 bits." )

            if( mfIsSparse(A) ) then
               ncol = A%shape(2)
               nnz = A%j(ncol+1)-1
               ! comparing CRC
               crc = crc_32_bits( A%z(1:nnz) )
            else
               ! comparing CRC
               crc = crc_32_bits( A%cmplx )
            end if

            if( crc /= A%crc ) then
               call PrintMessage( "msFreePointer", "I",                 &
                                  "data has been modified!",            &
                                  "-> erasing matrix properties" )
               call reset_prop(A%prop)
            end if
            A%crc = 0
            A%crc_stored = .false.
         end if

      else ! A%level_locked > 0

         call PrintMessage( "msFreePointer", "I",                       &
                            "Cannot unlock the targeted mfArray A, because this", &
                            "latter is still pointed by another f90 pointer!" )

      end if

#endif
   end subroutine msFreePointer_LongColumn_cmplx
!_______________________________________________________________________
!
   subroutine set_status_restr_to_false( A )

      type(mfArray) :: A
      !------ API end ------

#ifdef _DEVLP
      ! This routine is used in the FGL module, in which we want to
      ! allocate an array and point it by an mfArray; so, we use
      ! 'msEquiv' followed by 'set_status_restr_to_false' !

   !------ end of declarations -- execution starts hereafter  ------

      A%status_restricted = .false.

#endif
   end subroutine set_status_restr_to_false
!_______________________________________________________________________
!
   subroutine msEquiv_matrix_real8( f90_array, A, realloc )

      real(kind=MF_DOUBLE), target :: f90_array(:,:)
      type(mfArray) :: A
      logical, optional :: realloc
      !------ API end ------

#ifdef _DEVLP
      ! ~ equiv. to : A => f90_array
      !
      ! optional arg. for special situations (e.g. FGL/msGetX11Pixmap)
      ! if realloc = .true. -> SilentRelease is not called (but we have
      !                        to unlock it, if needed)
      !
      ! (this latter facility is not referenced in the user's guide nor
      !  the reference guide; it should be however described in the
      !  'MUESLI Inside' document)

   !------ end of declarations -- execution starts hereafter  ------

      if( A%parameter ) then
         call PrintMessage( "msEquiv", "E",                             &
                            "'A' cannot be a protected array (pseudo-parameter)." )
         return
      end if

      if( present(realloc) ) then
         if( realloc ) then
            ! Perhaps A is pointed by an ordinary f90 array, so we have
            ! also to remove the lock...
            ! No need to emit a warning, because the call comes from FGL.
            A%level_locked = 0
         else
            call msSilentRelease( A ) ! no dealloc if restricted
         end if
      else
         call msSilentRelease( A ) ! no dealloc if restricted
      end if

      A%data_type = MF_DT_DBLE
      A%shape = shape(f90_array)

      A%double => f90_array
      A%status_restricted = .true.

#endif
   end subroutine msEquiv_matrix_real8
!_______________________________________________________________________
!
   subroutine msEquiv_LongColumn_real8( f90_vec, a, new_shape )

      real(kind=MF_DOUBLE), target            :: f90_vec(:)
      type(mfArray)                           :: a
      integer, optional,           intent(in) :: new_shape(2)
      !------ API end ------

#ifdef _DEVLP
      ! ~ equiv. to : A => f90_array

      integer :: n, n1, n2

   !------ end of declarations -- execution starts hereafter  ------

      if( A%parameter ) then
         call PrintMessage( "msEquiv", "E",                             &
                            "'A' cannot be a protected array (pseudo-parameter)." )
         return
      end if

      call msSilentRelease( a )

      n = size(f90_vec)
      if( present(new_shape) ) then
         n1 = new_shape(1)
         n2 = new_shape(2)
         if( n1*n2 /= n ) then
            call PrintMessage( "msEquiv", "E",                          &
                               "bad arg value for new_shape!",          &
                               "(the product of the two components of new_shape must be", &
                                "equal to the size of the vecteur f90_vec)")
            return
         end if
      else
         n1 = n
         n2 = 1
      end if

      a%data_type = MF_DT_DBLE
      a%shape = [ n1, n2 ]

      ! only in Fortran 2003
      ! a%double(n,1) => f90_vec

      ! the rank remapping is implemented in: mod_mfaux
      a%double => rank_1_to_2_real8( f90_vec, n1, n2 )

      a%status_restricted = .true.

#endif
   end subroutine msEquiv_LongColumn_real8
!_______________________________________________________________________
!
   subroutine msEquiv_matrix_cmplx( f90_array, a )

      complex(kind=MF_DOUBLE), target :: f90_array(:,:)
      type(mfArray) :: a
      !------ API end ------

#ifdef _DEVLP
      ! ~ equiv. to : A => f90_array

   !------ end of declarations -- execution starts hereafter  ------

      if( A%parameter ) then
         call PrintMessage( "msEquiv", "E",                             &
                            "'A' cannot be a protected array (pseudo-parameter)." )
         return
      end if

      call msSilentRelease( a )

      a%data_type = MF_DT_CMPLX
      a%shape = shape(f90_array)
      a%cmplx => f90_array

      a%status_restricted = .true.

#endif
   end subroutine msEquiv_matrix_cmplx
!_______________________________________________________________________
!
   subroutine msEquiv_LongColumn_cmplx( f90_vec, a, new_shape )

      complex(kind=MF_DOUBLE), target         :: f90_vec(:)
      type(mfArray)                           :: a
      integer, optional,           intent(in) :: new_shape(2)
      !------ API end ------

#ifdef _DEVLP
      ! ~ equiv. to : A => f90_array

      integer :: n, n1, n2

   !------ end of declarations -- execution starts hereafter  ------

      if( A%parameter ) then
         call PrintMessage( "msEquiv", "E",                             &
                            "'A' cannot be a protected array (pseudo-parameter)." )
         return
      end if

      call msSilentRelease( a )

      n = size(f90_vec)
      if( present(new_shape) ) then
         n1 = new_shape(1)
         n2 = new_shape(2)
         if( n1*n2 /= n ) then
            call PrintMessage( "msEquiv", "E",                          &
                               "bad arg value for new_shape!",          &
                               "(the product of the two components of new_shape must be", &
                                "equal to the size of the vecteur f90_vec)")
            return
         end if
      else
         n1 = n
         n2 = 1
      end if

      a%data_type = MF_DT_CMPLX
      a%shape = [ n1, n2 ]

      ! only in Fortran 2003
      ! a%cmplx(n,1) => f90_vec

      ! the rank remapping is implemented in: mod_mfaux
      a%cmplx => rank_1_to_2_cmplx( f90_vec, n1, n2 )

      a%status_restricted = .true.

#endif
   end subroutine msEquiv_LongColumn_cmplx
!_______________________________________________________________________
!
   function mfNbPointers( A ) result( n )

      type(mfArray) :: A
      integer :: n
      !------ API end ------

#ifdef _DEVLP
      n = A%level_locked

#endif
   end function
!_______________________________________________________________________
!
   function mfOut_mfArray( arg1, arg2, arg3, arg4, arg5, arg6, arg7,    &
                           arg8, arg9, arg10 )                          &
   result( mfOut )

      type(mfArray), target           :: arg1
      type(mfArray), target, optional :: arg2, arg3, arg4, arg5, arg6,  &
                                         arg7, arg8, arg9, arg10
      type(mf_Out) :: mfOut
      !------ API end ------

#ifdef _DEVLP
      ! the derived type 'mf_Out' is defined in 'mf_Out.inc'

   !------ end of declarations -- execution starts hereafter  ------

      mfOut%n = 1
      mfOut%ptr1 => arg1
      mfOut%arg_present(1) = .true.
      if( arg1%parameter ) then
         ! test of MF_NO_ARG
         if( all( arg1%shape == -3 ) ) then
            mfOut%ptr1 => null()
            mfOut%arg_present(1) = .false.
         else
            call PrintMessage( "mfOut", "E",                            &
                               "'arg1' cannot be a protected array (pseudo-parameter)." )
            return
         end if
      end if

      if( present(arg2) ) then
         mfOut%n = 2
         mfOut%ptr2 => arg2
         mfOut%arg_present(2) = .true.
         if( arg2%parameter ) then
            ! test of MF_NO_ARG
            if( all( arg2%shape == -3 ) ) then
               mfOut%ptr2 => null()
               mfOut%arg_present(2) = .false.
            else
               call PrintMessage( "mfOut", "E",                         &
                                  "'arg2' cannot be a protected array (pseudo-parameter)." )
               return
            end if
         end if
      else
         return
      end if

      if( present(arg3) ) then
         mfOut%n = 3
         mfOut%ptr3 => arg3
         mfOut%arg_present(3) = .true.
         if( arg3%parameter ) then
            ! test of MF_NO_ARG
            if( all( arg3%shape == -3 ) ) then
               mfOut%ptr3 => null()
               mfOut%arg_present(3) = .false.
            else
               call PrintMessage( "mfOut", "E",                         &
                                  "'arg3' cannot be a protected array (pseudo-parameter)." )
               return
            end if
         end if
      else
         return
      end if

      if( present(arg4) ) then
         mfOut%n = 4
         mfOut%ptr4 => arg4
         mfOut%arg_present(4) = .true.
         if( arg4%parameter ) then
            ! test of MF_NO_ARG
            if( all( arg4%shape == -3 ) ) then
               mfOut%ptr4 => null()
               mfOut%arg_present(4) = .false.
            else
               call PrintMessage( "mfOut", "E",                         &
                                  "'arg4' cannot be a protected array (pseudo-parameter)." )
               return
            end if
         end if
      else
         return
      end if

      if( present(arg5) ) then
         mfOut%n = 5
         mfOut%ptr5 => arg5
         mfOut%arg_present(5) = .true.
         if( arg5%parameter ) then
            ! test of MF_NO_ARG
            if( all( arg5%shape == -3 ) ) then
               mfOut%ptr5 => null()
               mfOut%arg_present(5) = .false.
            else
               call PrintMessage( "mfOut", "E",                         &
                                  "'arg5' cannot be a protected array (pseudo-parameter)." )
               return
            end if
         end if
      else
         return
      end if

      if( present(arg6) ) then
         mfOut%n = 6
         mfOut%ptr6 => arg6
         mfOut%arg_present(6) = .true.
         if( arg6%parameter ) then
            ! test of MF_NO_ARG
            if( all( arg6%shape == -3 ) ) then
               mfOut%ptr6 => null()
               mfOut%arg_present(6) = .false.
            else
               call PrintMessage( "mfOut", "E",                         &
                                  "'arg6' cannot be a protected array (pseudo-parameter)." )
               return
            end if
         end if
      else
         return
      end if

      if( present(arg7) ) then
         mfOut%n = 7
         mfOut%ptr7 => arg7
         mfOut%arg_present(7) = .true.
         if( arg7%parameter ) then
            ! test of MF_NO_ARG
            if( all( arg7%shape == -3 ) ) then
               mfOut%ptr7 => null()
               mfOut%arg_present(7) = .false.
            else
               call PrintMessage( "mfOut", "E",                         &
                                  "'arg7' cannot be a protected array (pseudo-parameter)." )
               return
            end if
         end if
      end if

      if( present(arg8) ) then
         mfOut%n = 8
         mfOut%ptr8 => arg8
         mfOut%arg_present(8) = .true.
         if( arg8%parameter ) then
            ! test of MF_NO_ARG
            if( all( arg8%shape == -3 ) ) then
               mfOut%ptr8 => null()
               mfOut%arg_present(8) = .false.
            else
               call PrintMessage( "mfOut", "E",                         &
                                  "'arg8' cannot be a protected array (pseudo-parameter)." )
               return
            end if
         end if
      end if

      if( present(arg9) ) then
         mfOut%n = 9
         mfOut%ptr9 => arg9
         mfOut%arg_present(9) = .true.
         if( arg9%parameter ) then
            ! test of MF_NO_ARG
            if( all( arg9%shape == -3 ) ) then
               mfOut%ptr9 => null()
               mfOut%arg_present(9) = .false.
            else
               call PrintMessage( "mfOut", "E",                         &
                                  "'arg9' cannot be a protected array (pseudo-parameter)." )
               return
            end if
         end if
      end if

      if( present(arg10) ) then
         mfOut%n = 10
         mfOut%ptr10 => arg10
         mfOut%arg_present(10) = .true.
         if( arg10%parameter ) then
            ! test of MF_NO_ARG
            if( all( arg10%shape == -3 ) ) then
               mfOut%ptr10 => null()
               mfOut%arg_present(10) = .false.
            else
               call PrintMessage( "mfOut", "E",                         &
                                  "'arg10' cannot be a protected array (pseudo-parameter)." )
               return
            end if
         end if
      end if

#endif
   end function mfOut_mfArray
!_______________________________________________________________________
!
   function mfOut_mfMatFactor( arg1, arg2, arg3 ) result( mfOut )

      type(mfMatFactor), target           :: arg1
      type(mfArray),     target, optional :: arg2, arg3
      type(mf_Out) :: mfOut
      !------ API end ------

#ifdef _DEVLP
      ! the derived type 'mf_Out' is defined in 'mf_Out.inc'

   !------ end of declarations -- execution starts hereafter  ------

      mfOut%factor_n = 1
      mfOut%factor_ptr1 => arg1

      mfOut%n = 0
      if( present(arg2) ) then
         if( arg2%parameter ) then
            call PrintMessage( "mfOut", "E",                            &
                               "'arg2' cannot be a protected array (pseudo-parameter)." )
            return
         end if
         mfOut%n = 1
         mfOut%ptr1 => arg2
      else
         return
      end if

      if( present(arg3) ) then
         if( arg3%parameter ) then
            call PrintMessage( "mfOut", "E",                            &
                               "'arg3' cannot be a protected array (pseudo-parameter)." )
            return
         end if
         mfOut%n = 2
         mfOut%ptr2 => arg3
      end if

#endif
   end function mfOut_mfMatFactor
!_______________________________________________________________________
!
   function mfOut_empty( ) result( mfOut )

      type(mf_Out) :: mfOut
      !------ API end ------

#ifdef _DEVLP
      ! the derived type 'mf_Out' is defined in 'mf_Out.inc'

   !------ end of declarations -- execution starts hereafter  ------

      mfOut%n = 0
      mfOut%factor_n = 0

#endif
   end function mfOut_empty
!_______________________________________________________________________
!
   function args_mfout_ok( out, mf_1, mf_2, mf_3, mf_4 )

      type(mf_Out) :: out
      type(mfArray), optional :: mf_1, mf_2, mf_3, mf_4
      logical :: args_mfout_ok
      !------ API end ------

#ifdef _DEVLP

   !------ end of declarations -- execution starts hereafter  ------

      args_mfout_ok = .true.

      ! arg 1 -------------
      if( out%arg_present(1) ) then
         if( out%ptr1%status_temporary ) then
            call PrintMessage( "mfOut", "I",                            &
                               "arg #1 is tempo!" )
            args_mfout_ok = .false.
         end if
         if( present(mf_1) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr1) == _MF_LOC_ANY_OBJ_(mf_1) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #1 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_2) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr1) == _MF_LOC_ANY_OBJ_(mf_2) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #1 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_3) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr1) == _MF_LOC_ANY_OBJ_(mf_3) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #1 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_4) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr1) == _MF_LOC_ANY_OBJ_(mf_4) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #1 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
      end if

      ! arg 2 -------------
      if( out%arg_present(2) ) then
         if( out%ptr2%status_temporary ) then
            call PrintMessage( "mfOut", "I",                            &
                               "arg #2 is tempo!" )
            args_mfout_ok = .false.
         end if
         if( present(mf_1) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr2) == _MF_LOC_ANY_OBJ_(mf_1) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #2 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_2) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr2) == _MF_LOC_ANY_OBJ_(mf_2) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #2 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_3) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr2) == _MF_LOC_ANY_OBJ_(mf_3) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #2 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_4) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr2) == _MF_LOC_ANY_OBJ_(mf_4) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #2 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
      end if

      ! arg 3 -------------
      if( out%arg_present(3) ) then
         if( out%ptr3%status_temporary ) then
            call PrintMessage( "mfOut", "I",                            &
                               "arg #3 is tempo!" )
            args_mfout_ok = .false.
         end if
         if( present(mf_1) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr3) == _MF_LOC_ANY_OBJ_(mf_1) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #3 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_2) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr3) == _MF_LOC_ANY_OBJ_(mf_2) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #3 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_3) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr3) == _MF_LOC_ANY_OBJ_(mf_3) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #3 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_4) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr3) == _MF_LOC_ANY_OBJ_(mf_4) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #3 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
      end if

      ! arg 4 -------------
      if( out%arg_present(4) ) then
         if( out%ptr4%status_temporary ) then
            call PrintMessage( "mfOut", "I",                            &
                               "arg #4 is tempo!" )
            args_mfout_ok = .false.
         end if
         if( present(mf_1) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr4) == _MF_LOC_ANY_OBJ_(mf_1) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #4 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_2) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr4) == _MF_LOC_ANY_OBJ_(mf_2) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #4 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_3) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr4) == _MF_LOC_ANY_OBJ_(mf_3) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #4 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_4) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr4) == _MF_LOC_ANY_OBJ_(mf_4) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #4 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
      end if

      ! arg 5 -------------
      if( out%arg_present(5) ) then
         if( out%ptr5%status_temporary ) then
            call PrintMessage( "mfOut", "I",                            &
                               "arg #5 is tempo!" )
            args_mfout_ok = .false.
         end if
         if( present(mf_1) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr5) == _MF_LOC_ANY_OBJ_(mf_1) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #5 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_2) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr5) == _MF_LOC_ANY_OBJ_(mf_2) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #5 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_3) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr5) == _MF_LOC_ANY_OBJ_(mf_3) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #5 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_4) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr5) == _MF_LOC_ANY_OBJ_(mf_4) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #5 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
      end if

      ! arg 6 -------------
      if( out%arg_present(6) ) then
         if( out%ptr6%status_temporary ) then
            call PrintMessage( "mfOut", "I",                            &
                               "arg #6 is tempo!" )
            args_mfout_ok = .false.
         end if
         if( present(mf_1) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr6) == _MF_LOC_ANY_OBJ_(mf_1) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #6 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_2) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr6) == _MF_LOC_ANY_OBJ_(mf_2) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #6 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_3) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr6) == _MF_LOC_ANY_OBJ_(mf_3) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #6 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_4) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr6) == _MF_LOC_ANY_OBJ_(mf_4) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #6 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
      end if

      ! arg 7 -------------
      if( out%arg_present(7) ) then
         if( out%ptr7%status_temporary ) then
            call PrintMessage( "mfOut", "I",                            &
                               "arg #7 is tempo!" )
            args_mfout_ok = .false.
         end if
         if( present(mf_1) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr7) == _MF_LOC_ANY_OBJ_(mf_1) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #7 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_2) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr7) == _MF_LOC_ANY_OBJ_(mf_2) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #7 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_3) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr7) == _MF_LOC_ANY_OBJ_(mf_3) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #7 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_4) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr7) == _MF_LOC_ANY_OBJ_(mf_4) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #7 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
      end if

      ! arg 8 -------------
      if( out%arg_present(8) ) then
         if( out%ptr8%status_temporary ) then
            call PrintMessage( "mfOut", "I",                            &
                               "arg #8 is tempo!" )
            args_mfout_ok = .false.
         end if
         if( present(mf_1) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr8) == _MF_LOC_ANY_OBJ_(mf_1) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #8 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_2) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr8) == _MF_LOC_ANY_OBJ_(mf_2) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #8 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_3) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr8) == _MF_LOC_ANY_OBJ_(mf_3) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #8 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_4) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr8) == _MF_LOC_ANY_OBJ_(mf_4) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #8 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
      end if

      ! arg 9 -------------
      if( out%arg_present(9) ) then
         if( out%ptr9%status_temporary ) then
            call PrintMessage( "mfOut", "I",                            &
                               "arg #9 is tempo!" )
            args_mfout_ok = .false.
         end if
         if( present(mf_1) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr9) == _MF_LOC_ANY_OBJ_(mf_1) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #9 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_2) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr9) == _MF_LOC_ANY_OBJ_(mf_2) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #9 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_3) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr9) == _MF_LOC_ANY_OBJ_(mf_3) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #9 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_4) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr9) == _MF_LOC_ANY_OBJ_(mf_4) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #9 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
      end if

      ! arg 10 -------------
      if( out%arg_present(10) ) then
         if( out%ptr10%status_temporary ) then
            call PrintMessage( "mfOut", "I",                            &
                               "arg #10 is tempo!" )
            args_mfout_ok = .false.
         end if
         if( present(mf_1) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr10) == _MF_LOC_ANY_OBJ_(mf_1) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #10 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_2) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr10) == _MF_LOC_ANY_OBJ_(mf_2) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #10 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_3) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr10) == _MF_LOC_ANY_OBJ_(mf_3) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #10 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_4) ) then
            if( _MF_LOC_ANY_OBJ_(out%ptr10) == _MF_LOC_ANY_OBJ_(mf_4) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #10 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
      end if

      !------------------------------------------------

      ! mfMatFactor --------------
      if( out%factor_n >= 1 ) then
         if( present(mf_1) ) then
            if( _MF_LOC_ANY_OBJ_(out%factor_ptr1) == _MF_LOC_ANY_OBJ_(mf_1) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #1 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_2) ) then
            if( _MF_LOC_ANY_OBJ_(out%factor_ptr1) == _MF_LOC_ANY_OBJ_(mf_2) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #1 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_3) ) then
            if( _MF_LOC_ANY_OBJ_(out%factor_ptr1) == _MF_LOC_ANY_OBJ_(mf_3) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #1 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
         if( present(mf_4) ) then
            if( _MF_LOC_ANY_OBJ_(out%factor_ptr1) == _MF_LOC_ANY_OBJ_(mf_4) ) then
               call PrintMessage( "mfOut", "I",                         &
                                  "arg #1 shares the same memory as another arg.!" )
               args_mfout_ok = .false.
            end if
         end if
      end if

#endif
   end function args_mfout_ok
!_______________________________________________________________________
!
   subroutine msGetStdIO( stdin, stdout, stderr )

      use mod_mfdebug, only: muesli_stdin  => STDIN,                    &
                             muesli_stdout => STDOUT,                   &
                             muesli_stderr => STDERR

      integer, intent(out), optional :: stdin, stdout, stderr
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( present(stdin) ) then
         stdin = muesli_stdin
      end if

      if( present(stdout) ) then
         stdout = muesli_stdout
      end if

      if( present(stderr) ) then
         stderr = muesli_stderr
      end if

#endif
   end subroutine msGetStdIO
!_______________________________________________________________________
!
   subroutine msSetStdIO( stdin, stdout, stderr )

      use mod_mfdebug, only: muesli_stdin  => STDIN,                    &
                             muesli_stdout => STDOUT,                   &
                             muesli_stderr => STDERR

      integer, intent(in), optional :: stdin, stdout, stderr
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( present(stdin) ) then
         muesli_stdin = stdin
      end if

      if( present(stdout) ) then
         muesli_stdout = stdout
      end if

      if( present(stderr) ) then
         muesli_stderr = stderr
      end if

#endif
   end subroutine msSetStdIO
!_______________________________________________________________________
!
   subroutine msFlush( unit )

      integer, intent(in) :: unit
      !------ API end ------

#ifdef _DEVLP
#if defined _INTEL_IFC
      character(len=72) :: tty
#endif

   !------ end of declarations -- execution starts hereafter  ------

#if defined _INTEL_IFC
      if( unit == 6 ) then
         ! work-around because standard 'flush' routine doesn't work
         ! with INTEL-ifort for STDOUT...
! WARNING: doesn't work when printing on the same line (ex. ms*PrintProgress)
!          use instead the routine in 'print_aux.c' file.
         ! Defining again 'flush' for unit=6
         call ttynam(tty,unit) ! get current device name of the terminal
         close(unit)
         open(unit,file=trim(tty))
      else
         call flush(unit)
      end if
#else
      ! we suppose that other compilers have a good 'flush' routine
      call flush(unit)
#endif

#endif
   end subroutine msFlush
!_______________________________________________________________________
!
   subroutine print_debug_mfArray( A, symb_name, disp_array )

      type(mfArray), intent(in) :: A
      character(len=*) :: symb_name
      logical, optional :: disp_array
      !------ API end ------

#ifdef _DEVLP
      integer :: ncol, nnz

   ! Caution: writing 32bit addresses use Z8.8 in hexadecimal,
   !          but must use Z16.16 in 64bit (two characters for on byte)
#ifdef _64_BITS
      character(len=10) :: hexa_format = "(A,Z16.16)"
#else
      character(len=10) :: hexa_format = "(A,Z8.8)"
#endif

   !------ end of declarations -- execution starts hereafter  ------

#ifdef _LOC_DISABLED
      call PrintMessage( "print_debug_mfArray", "W",                    &
                         "routine 'loc()' disabled!" )
#else
      write(STDOUT,"(A)")            "--------------------------------------------------"
      write(STDOUT,"(A,A)")          "        symbol name : ", trim(symb_name)
      write(STDOUT,hexa_format)      "       mfArray addr : 0x", _MF_LOC_ANY_OBJ_(a)
      write(STDOUT,"(A,I0,2x,I0)")   "        array shape : ", A%shape
      select case( A%data_type )
         case( MF_DT_EMPTY )
            write(STDOUT,"(A,I0,A)") "          data type : ", A%data_type, " (empty)"
         case( MF_DT_DBLE )
            write(STDOUT,"(A,I0,A)") "          data type : ", A%data_type, " (dense, real)"
         case( MF_DT_CMPLX )
            write(STDOUT,"(A,I0,A)") "          data type : ", A%data_type, " (dense, complex)"
         case( MF_DT_BOOL )
            write(STDOUT,"(A,I0,A)") "          data type : ", A%data_type, " (boolean)"
         case( MF_DT_SP_BOOL )
            write(STDOUT,"(A,I0,A)") "          data type : ", A%data_type, " (sparse, boolean)"
         case( MF_DT_SP_DBLE )
            write(STDOUT,"(A,I0,A)") "          data type : ", A%data_type, " (sparse, real)"
         case( MF_DT_SP_CMPLX )
            write(STDOUT,"(A,I0,A)") "          data type : ", A%data_type, " (sparse, complex)"
         case( MF_DT_PERM_VEC )
            write(STDOUT,"(A,I0,A)") "          data type : ", A%data_type, " (permutation vector)"
         case default
            write(STDOUT,"(A)")      "          data type : *** ERROR (unknown value) ***"
      end select
      if( associated(A%double) ) then
#ifdef _64_BITS
         write(STDOUT,"(A)")         "         array addr : 0x????????????????"
#else
         write(STDOUT,"(A)")         "         array addr : 0x????????"
#endif
         write(STDOUT,hexa_format)   "             array => 0x", _MF_LOC_ANY_OBJ_(a%double)
         if( present(disp_array) ) then
            if( disp_array ) then
               write(STDOUT,*)       "          array val : ", A%double
            end if
         end if
      else if( associated(A%cmplx) ) then
#ifdef _64_BITS
         write(STDOUT,"(A)")         "         array addr : 0x????????????????"
#else
         write(STDOUT,"(A)")         "         array addr : 0x????????"
#endif
         write(STDOUT,hexa_format)   "             array => 0x", _MF_LOC_ANY_OBJ_(a%cmplx)
         if( present(disp_array) ) then
            if( disp_array ) then
               write(STDOUT,*)       "          array val : ", A%cmplx
            end if
         end if
      else if( associated(A%a) ) then
#ifdef _64_BITS
         write(STDOUT,"(A)")         "         array addr : 0x????????????????"
#else
         write(STDOUT,"(A)")         "         array addr : 0x????????"
#endif
         write(STDOUT,hexa_format)   "             array => 0x", _MF_LOC_ANY_OBJ_(a%a)
         write(STDOUT,"(A,I0)")      " actual size of a() : ", size(A%a)
         write(STDOUT,"(A,I0)")      " actual size of i() : ", size(A%i)
         write(STDOUT,"(A,I0)")      " actual size of j() : ", size(A%j)
         if( present(disp_array) ) then
            ncol = A%shape(2)
            nnz = A%j(ncol+1)-1
            if( disp_array ) then
               write(STDOUT,*)       "          array val : ", A%a(1:nnz)
            end if
         end if
      else if( associated(A%z) ) then
#ifdef _64_BITS
         write(STDOUT,"(A)")         "         array addr : 0x????????????????"
#else
         write(STDOUT,"(A)")         "         array addr : 0x????????"
#endif
         write(STDOUT,hexa_format)   "             array => 0x", _MF_LOC_ANY_OBJ_(a%z)
         write(STDOUT,"(A,I0)")      "actual size of z(:) : ", size(A%z)
         write(STDOUT,"(A,I0)")      "actual size of i(:) : ", size(A%i)
         write(STDOUT,"(A,I0)")      "actual size of j(:) : ", size(A%j)
         if( present(disp_array) ) then
            ncol = A%shape(2)
            nnz = A%j(ncol+1)-1
            if( disp_array ) then
               write(STDOUT,*)       "          array val : ", A%z(1:nnz)
            end if
         end if
      else if( associated(A%i) ) then
         write(STDOUT,hexa_format)   "         array addr : 0x????????"
         write(STDOUT,hexa_format)   "             array => 0x", _MF_LOC_ANY_OBJ_(a%i)
         if( present(disp_array) ) then
            if( disp_array ) then
               write(STDOUT,*)       "          array val : ", A%i
            end if
         end if
      else
         write(STDOUT,"(A)")         "         array addr : <not associated>"
         if( present(disp_array) ) then
            if( disp_array ) then
               write(STDOUT,"(A)")   "          array val : <not available>"
            end if
         end if
      end if
      select case( A%row_sorted )
         case( TRUE )
            write(STDOUT,"(A,I0,A)") "         row_sorted : ", A%row_sorted, " (True)"
         case( FALSE )
            write(STDOUT,"(A,I0,A)") "         row_sorted : ", A%row_sorted, " (False)"
         case( UNKNOWN )
            write(STDOUT,"(A,I0,A)") "         row_sorted : ", A%row_sorted, " (Unknown)"
         case default
            write(STDOUT,"(A)")      "         row_sorted : *** ERROR (unknown value) ***"
      end select
      select case( A%prop%tril )
         case( TRUE )
            write(STDOUT,"(A,I0,A)") "      lower triang. : ", A%prop%tril, " (True)"
         case( FALSE )
            write(STDOUT,"(A,I0,A)") "      lower triang. : ", A%prop%tril, " (False)"
         case( UNKNOWN )
            write(STDOUT,"(A,I0,A)") "      lower triang. : ", A%prop%tril, " (Unknown)"
         case default
            write(STDOUT,"(A)")      "      lower triang. : *** ERROR (unknown value) ***"
      end select
      select case( A%prop%triu )
         case( TRUE )
            write(STDOUT,"(A,I0,A)") "      upper triang. : ", A%prop%triu, " (True)"
         case( FALSE )
            write(STDOUT,"(A,I0,A)") "      upper triang. : ", A%prop%triu, " (False)"
         case( UNKNOWN )
            write(STDOUT,"(A,I0,A)") "      upper triang. : ", A%prop%triu, " (Unknown)"
         case default
            write(STDOUT,"(A)")      "      upper triang. : *** ERROR (unknown value) ***"
      end select
      select case( A%prop%symm )
         case( TRUE )
            write(STDOUT,"(A,I0,A)") "      symm. / herm. : ", A%prop%symm, " (True)"
         case( FALSE )
            write(STDOUT,"(A,I0,A)") "      symm. / herm. : ", A%prop%symm, " (False)"
         case( UNKNOWN )
            write(STDOUT,"(A,I0,A)") "      symm. / herm. : ", A%prop%symm, " (Unknown)"
         case default
            write(STDOUT,"(A)")      "      symm. / herm. : *** ERROR (unknown value) ***"
      end select
      select case( A%prop%posd )
         case( TRUE )
            write(STDOUT,"(A,I0,A)") "          pos. def. : ", A%prop%posd, " (True)"
         case( FALSE )
            write(STDOUT,"(A,I0,A)") "          pos. def. : ", A%prop%posd, " (False)"
         case( UNKNOWN )
            write(STDOUT,"(A,I0,A)") "          pos. def. : ", A%prop%posd, " (Unknown)"
         case default
            write(STDOUT,"(A)")      "          pos. def. : *** ERROR (unknown value) ***"
      end select
      write(STDOUT,"(A,I0)")         "       level_locked : ", A%level_locked
      write(STDOUT,"(A,1X,I0)")      "                crc : ", A%crc
      write(STDOUT,*)                 "  status_temporary : ", A%status_temporary
      write(STDOUT,"(A,1X,I0)")      "    level_protected : ", A%level_protected
      write(STDOUT,*)                 " status_restricted : ", A%status_restricted
      write(STDOUT,*)                 "         parameter : ", A%parameter
      write(STDOUT,"(A,14(I0,1X))")  "              units : ", A%units
      write(STDOUT,"(A)")            "--------------------------------------------------"
#endif

#endif
   end subroutine print_debug_mfArray
!_______________________________________________________________________
!
   function same_address( a, b ) result( bool )

      type(mfArray), intent(in) :: a, b
      integer :: bool
      !------ API end ------

#ifdef _DEVLP
      ! bool = 0 : false
      !        1 : true
      !       -1 : undefined

      logical :: test

   !------ end of declarations -- execution starts hereafter  ------

      bool = -1

#ifdef _LOC_DISABLED
      call PrintMessage( "print_debug_mfArray", "W",                    &
                         "routine 'loc()' disabled!",                  &
                         "-> cannot resolve some internal tests." )
#else

      if( associated(a%double) .and. associated(b%double) ) then
         test = _MF_LOC_ANY_OBJ_(a%double(1,1)) == _MF_LOC_ANY_OBJ_(b%double(1,1))
         if( test ) then
            bool = 1
         else
            bool = 0
         end if
      else if( associated(a%cmplx) .and. associated(b%cmplx) ) then
         test = _MF_LOC_ANY_OBJ_(a%cmplx(1,1)) == _MF_LOC_ANY_OBJ_(b%cmplx(1,1))
         if( test ) then
            bool = 1
         else
            bool = 0
         end if
      else if( associated(a%a) .and. associated(b%a) ) then
         test = _MF_LOC_ANY_OBJ_(a%a(1)) == _MF_LOC_ANY_OBJ_(b%a(1))
         if( test ) then
            bool = 1
         else
            bool = 0
         end if
      else if( associated(a%z) .and. associated(b%z) ) then
         test = _MF_LOC_ANY_OBJ_(a%z(1)) == _MF_LOC_ANY_OBJ_(b%z(1))
         if( test ) then
            bool = 1
         else
            bool = 0
         end if
      else
         call PrintMessage( "same_address", "I",                        &
                            "it appears that vars 'a' and 'b' has not the same", &
                            "data type, or at least one var is not allocated." )
      end if
#endif

#endif
   end function same_address
!_______________________________________________________________________
!
   function Shape_mfArray( x ) result( out )

      type(mfArray) :: x
      integer :: out(2)
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      out = x%shape

      call msAutoRelease( x )

#endif
   end function Shape_mfArray
!_______________________________________________________________________
!
   function Shape_mfMatFactor( x ) result( out )

      type(mfMatFactor) :: x
      integer :: out(2)
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( x%package /= 3 ) then
         call PrintMessage( "Shape", "E",                               &
                            "bad type of factor in an mfMatFactor!",    &
                            "(valid only for Qhouse factor)" )
         return
      end if
      out = x%shape

#endif
   end function Shape_mfMatFactor
!_______________________________________________________________________
!
   function mfShape_mfArray( x ) result( out )

      type(mfArray) :: x
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      out = x%shape

      out%prop%symm = FALSE

      out%status_temporary = .true.

      call msAutoRelease( x )

#endif
   end function mfShape_mfArray
!_______________________________________________________________________
!
   function mfShape_mfMatFactor( x ) result( out )

      type(mfMatFactor) :: x
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( x%package /= 3 ) then
         call PrintMessage( "mfShape", "E",                             &
                            "bad type of factor in an mfMatFactor!",    &
                            "(valid only for Qhouse factor)" )
         return
      end if
      out = x%shape

      out%prop%symm = FALSE

      out%status_temporary = .true.

#endif
   end function mfShape_mfMatFactor
!_______________________________________________________________________
!
   function Size_mfArray( x, idim ) result( out )

      type(mfArray) :: x
      integer, optional :: idim
      integer :: out
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( present(idim) ) then
         if( idim == 1 ) then
            out = x%shape(1)
         else if( idim == 2 ) then
            out = x%shape(2)
         else
            call PrintMessage( "Size", "E",                             &
                               "bad argument value for : idim!" )
            go to 99
         end if
      else
         out = x%shape(1) * x%shape(2)
      end if

 99   continue

      call msAutoRelease( x )

#endif
   end function Size_mfArray
!_______________________________________________________________________
!
   function Size_mfMatFactor( x, idim ) result( out )

      type(mfMatFactor) :: x
      integer, optional :: idim
      integer :: out
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( x%package /= 3 ) then
         call PrintMessage( "Size", "E",                                &
                            "bad type of factor in an mfMatFactor!",    &
                            "(valid only for Qhouse factor)" )
         return
      end if
      if( present(idim) ) then
         if( idim == 1 ) then
            out = x%shape(1)
         else if( idim == 2 ) then
            out = x%shape(2)
         else
            call PrintMessage( "Size", "E",                             &
                               "bad argument value for : idim!" )
            return
         end if
      else
         out = x%shape(1) * x%shape(2)
      end if

#endif
   end function Size_mfMatFactor
!_______________________________________________________________________
!
   function mfSize_mfArray( x, idim ) result( out )

      type(mfArray) :: x
      integer, optional :: idim
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( present(idim) ) then
         if( idim == 1 ) then
            out = x%shape(1)
         else if( idim == 2 ) then
            out = x%shape(2)
         else
            call PrintMessage( "mfSize", "E",                           &
                               "bad argument value for : idim!" )
            go to 99
         end if
      else
         out = x%shape(1) * x%shape(2)
      end if

      out%prop%symm = TRUE

      out%status_temporary = .true.

 99   continue

      call msAutoRelease( x )

#endif
   end function mfSize_mfArray
!_______________________________________________________________________
!
   function mfSize_mfMatFactor( x, idim ) result( out )

      type(mfMatFactor) :: x
      integer, optional :: idim
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( x%package /= 3 ) then
         call PrintMessage( "mfSize", "E",                              &
                            "bad type of factor in an mfMatFactor!",    &
                            "(valid only for Qhouse factor)" )
         return
      end if
      if( present(idim) ) then
         if( idim == 1 ) then
            out = x%shape(1)
         else if( idim == 2 ) then
            out = x%shape(2)
         else
            call PrintMessage( "mfSize", "E",                           &
                               "bad argument value for : idim!" )
            return
         end if
      else
         out = x%shape(1) * x%shape(2)
      end if

      out%prop%symm = TRUE

      out%status_temporary = .true.

#endif
   end function mfSize_mfMatFactor
!_______________________________________________________________________
!
   function mfIsScalar( A ) result( bool )

      type(mfArray), intent(in) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      ! info: a scalar is not a vector, neither a matrix

   !------ end of declarations -- execution starts hereafter  ------

      if( A%shape(1) == 1 .and. A%shape(2) == 1 ) then
         bool = .true.
      else
         bool = .false.
      end if

      call msAutoRelease( A )

#endif
   end function mfIsScalar
!_______________________________________________________________________
!
   function mfIsVector( A ) result( bool )

      type(mfArray), intent(in) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      ! info: a vector is not a matrix

   !------ end of declarations -- execution starts hereafter  ------

      if( A%shape(1) == 1 .neqv. A%shape(2) == 1 ) then
         bool = .true.
      else
         bool = .false.
      end if

      call msAutoRelease( A )

#endif
   end function mfIsVector
!_______________________________________________________________________
!
   function mfIsRow( A ) result( bool )

      type(mfArray), intent(in) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( A%shape(1) == 1 .and. A%shape(2) > 1 ) then
         bool = .true.
      else
         bool = .false.
      end if

      call msAutoRelease( A )

#endif
   end function mfIsRow
!_______________________________________________________________________
!
   function mfIsColumn( A ) result( bool )

      type(mfArray), intent(in) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( A%shape(1) > 1 .and. A%shape(2) == 1 ) then
         bool = .true.
      else
         bool = .false.
      end if

      call msAutoRelease( A )

#endif
   end function mfIsColumn
!_______________________________________________________________________
!
   function mfIsMatrix( A ) result( bool )

      type(mfArray), intent(in) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      ! info: a strict matrix

   !------ end of declarations -- execution starts hereafter  ------

      if( A%shape(1) > 1 .and. A%shape(2) > 1 ) then
         bool = .true.
      else
         bool = .false.
      end if

      call msAutoRelease( A )

#endif
   end function mfIsMatrix
!_______________________________________________________________________
!
   function All_mfArray( A ) result( bool )

      type(mfArray), intent(in) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      integer :: nrow, ncol, nnz

   !------ end of declarations -- execution starts hereafter  ------

      bool = .false.

      if( A%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "All", "W",                                 &
                            "arg is an empty mfArray!" )
         go to 99
      end if

      if( A%data_type /= MF_DT_BOOL .and. A%data_type /= MF_DT_SP_BOOL ) then
         call PrintMessage( "All", "E",                                 &
                            "arg must be a boolean mfArray!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         if( all(A%double/=0) ) bool = .true.
      else ! MF_DT_SP_BOOL
         nrow = A%shape(1)
         ncol = A%shape(2)
         nnz = A%j(ncol+1) - 1
         if( nnz == nrow*ncol ) bool = .true.
      end if

 99   continue

      call msAutoRelease( A )

#endif
   end function All_mfArray
!_______________________________________________________________________
!
   function Any_mfArray( A ) result( bool )

      type(mfArray), intent(in) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      integer :: ncol, nnz

   !------ end of declarations -- execution starts hereafter  ------

      bool = .false.

      if( A%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "Any", "W",                                 &
                            "arg is an empty mfArray!" )
         go to 99
      end if

      if( A%data_type /= MF_DT_BOOL .and. A%data_type /= MF_DT_SP_BOOL ) then
         call PrintMessage( "Any", "E",                                 &
                            "arg must be a boolean mfArray!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         if( any(A%double/=0) ) bool = .true.
      else ! MF_DT_SP_BOOL
         ncol = A%shape(2)
         nnz = A%j(ncol+1) - 1
         if( nnz >= 1 ) bool = .true.
      end if

 99   continue

      call msAutoRelease( A )

#endif
   end function Any_mfArray
!_______________________________________________________________________
!
   subroutine msReturnArray( x )

      type(mfArray) :: x
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      x%status_temporary = .true.

#endif
   end subroutine msReturnArray
!_______________________________________________________________________
!
   function mfIsTempoArray( x ) result( bool )

      type(mfArray) :: x
      logical       :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      bool = x%status_temporary

#endif
   end function mfIsTempoArray
!_______________________________________________________________________
!
   function mfInt( mf_scalar ) result( out )

      type(mfArray), intent(in) :: mf_scalar
      integer :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( mf_scalar )

      if( mfIsEmpty(mf_scalar) ) then
         call PrintMessage( "mfInt", "E",                               &
                            "mfArray not allocated!" )
         go to 99
      end if

      if( mf_scalar%data_type == MF_DT_BOOL .or. mf_scalar%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "mfInt", "E",                               &
                            "cannot be applied to boolean!" )
         go to 99
      end if

      if( mfIsSparse(mf_scalar) ) then
         call PrintMessage( "mfInt", "E",                               &
                            "cannot be applied to sparse!" )
         go to 99
      end if

      if( .not. mfIsReal(mf_scalar) ) then
         call PrintMessage( "mfInt", "E",                               &
                            "mfArray must be real!" )
         go to 99
      end if

      if( mfIsPerm(mf_scalar) ) then
         call PrintMessage( "mfInt", "E",                               &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( any( mf_scalar%shape /= [1,1] ) ) then
         call PrintMessage( "mfInt", "E",                               &
                            "mfArray must be a scalar!" )
         go to 99
      end if

      out = nint( mf_scalar%double(1,1) )
      if( dble(out) /= mf_scalar%double(1,1) ) then
         call PrintMessage( "mfInt", "W",                               &
                            "inexact result when converting a double to integer!" )
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( mf_scalar%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfInt", "I",                            &
                               "the physical unit of the mfArray is lost!" )
         end if
      end if

 99   continue

      call msFreeArgs( mf_scalar )

      call msAutoRelease( mf_scalar )

#endif
   end function mfInt
!_______________________________________________________________________
!
   function mfDble( mf_scalar ) result( out )

      type(mfArray)        :: mf_scalar
      real(kind=MF_DOUBLE) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( mf_scalar )

      if( mfIsEmpty(mf_scalar) ) then
         call PrintMessage( "mfDble", "E",                              &
                            "mfArray not allocated!" )
         go to 99
      end if

      if( mf_scalar%data_type == MF_DT_BOOL .or. mf_scalar%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "mfDble", "E",                              &
                            "cannot be applied to boolean!" )
         go to 99
      end if

      if( mfIsSparse(mf_scalar) ) then
         call PrintMessage( "mfDble", "E",                              &
                            "cannot be applied to sparse!" )
         go to 99
      end if

      if( mfIsPerm(mf_scalar) ) then
         call PrintMessage( "mfDble", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( any( mf_scalar%shape /= [1,1] ) ) then
         call PrintMessage( "mfDble", "E",                              &
                            "mfArray must be a scalar!" )
         go to 99
      end if

      if( mf_scalar%data_type == MF_DT_DBLE ) then
         out = mf_scalar%double(1,1)
      else if( mf_scalar%data_type == MF_DT_CMPLX ) then
         out = real( mf_scalar%cmplx(1,1) )
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( mf_scalar%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfDble", "I",                           &
                               "the physical unit of the mfArray is lost!" )
         end if
      end if

 99   continue

      call msFreeArgs( mf_scalar )

      call msAutoRelease( mf_scalar )

#endif
   end function mfDble
!_______________________________________________________________________
!
   function mfCmplx( mf_scalar ) result( out )

      type(mfArray), intent(in) :: mf_scalar
      complex(kind=MF_DOUBLE) :: out
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( mf_scalar )

      if( mfIsEmpty(mf_scalar) ) then
         call PrintMessage( "mfCmplx", "E",                             &
                            "mfArray not allocated!" )
         go to 99
      end if

      if( mf_scalar%data_type == MF_DT_BOOL .or. mf_scalar%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "mfCmplx", "E",                             &
                            "cannot be applied to boolean!" )
         go to 99
      end if

      if( mfIsSparse(mf_scalar) ) then
         call PrintMessage( "mfCmplx", "E",                             &
                            "cannot be applied to sparse!" )
         go to 99
      end if

      if( mfIsPerm(mf_scalar) ) then
         call PrintMessage( "mfCmplx", "E",                             &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( any( mf_scalar%shape /= [1,1] ) ) then
         call PrintMessage( "mfCmplx", "E",                             &
                            "mfArray must be a scalar!" )
         go to 99
      end if

      if( mf_scalar%data_type == MF_DT_DBLE ) then
         out = mf_scalar%double(1,1)
      else if( mf_scalar%data_type == MF_DT_CMPLX ) then
         out = mf_scalar%cmplx(1,1)
      end if

 99   continue

      call msFreeArgs( mf_scalar )

      call msAutoRelease( mf_scalar )

#endif
   end function mfCmplx
!_______________________________________________________________________
!
   function can_use_memory( A ) result( bool )

      type(mfArray), intent(in) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      ! the following old test was NEVER true... because the current
      ! routine is always called from another routine which has itself
      ! get up a protection!
      !! if( A%status_temporary .and. A%level_protected == 0 ) then

      if( A%status_temporary .and. A%level_protected == 1 ) then
         bool = .true.
      else
         bool = .false.
      end if

#endif
   end function can_use_memory
!_______________________________________________________________________
!
   subroutine do_data_transfer_dble( new, tempo )

      type(mfArray) :: new, tempo
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      new%double => tempo%double
      call set_status_tempo_to_false( tempo )

#endif
   end subroutine do_data_transfer_dble
!_______________________________________________________________________
!
   subroutine do_data_transfer_cmplx( new, tempo )

      type(mfArray) :: new, tempo
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      new%cmplx => tempo%cmplx
      call set_status_tempo_to_false( tempo )

#endif
   end subroutine do_data_transfer_cmplx
!_______________________________________________________________________
!
#if !defined(_WINDOWS) && !defined(_DARWIN)
   subroutine msFlops_init( init )

      integer, intent(in) :: init
      !------ API end ------

#ifdef _DEVLP
      ! returns the number of Floating-Point Operations made by
      ! the processor.
      !
      ! requires a specific version of the PAPI library
      ! (cf. in papi/version)

#include "papi/fpapi.inc"

      ! API :
      !
      !   call msFlops(0) or
      !   call msFlops(init), with 'init' any argument of type integer:
      !                        initialize the hardware counter
      !
      !   call msFlops(count) : returns the flops since the last
      !                         initialization;
      !                         'count' argument of type integer*8
      !
      !   count >= 0, except if the program has been linked with the
      !   dummy version of PAPI (dummy_papi.o), or if the OS doesn't
      !   support hardware counting (e.g. the linux kernel has not
      !   been patched with 'perfctr'); in the latter case, msFlops
      !   returns the value '-1'.

      integer :: retval
      real :: proc_time, mflops, real_time
      integer*8 :: flpins

   !------ end of declarations -- execution starts hereafter  ------

      if( mf_papi_available == FALSE ) then

         call PrintMessage( "msFlops", "W",                             &
                            "PAPI not available!",                      &
                            "-> bad version of PAPI?",                  &
                            "-> old linux kernel not patched with the PerfCtr module?" )
         return

      else if( mf_papi_available == UNKNOWN ) then

         retval = PAPI_VER_CURRENT
         call PAPIf_library_init( retval )
         if( retval /= PAPI_VER_CURRENT ) then
            call PrintMessage( "msFlops", "W",                          &
                               "initialization failed!",                &
                               "-> bad version of PAPI?",               &
                               "-> old linux kernel not patched with the PerfCtr module?" )
            return
         end if

         call PAPIf_query_event( PAPI_FP_INS, retval )
!!         call PAPIf_query_event( PAPI_FP_OPS, retval )
         if( retval /= PAPI_OK ) then
            call PrintMessage( "msFlops", "W",                          &
                               "initialization failed!" )
            return
         end if

      end if

      ! Setup PAPI library and begin collecting data from the
      ! hardware counters
      call PAPIf_flips( real_time, proc_time, flpins, mflops, retval )
      if( retval /= PAPI_OK ) then
         call PrintMessage( "msFlops", "W",                             &
                            "PAPI not ready!" )
         return
      end if

      if( mf_papi_init_ok ) then
         mf_papi_flpins_old = flpins
      else
         mf_papi_flpins_old = 0
         mf_papi_init_ok = .true.
      end if

#endif
   end subroutine msFlops_init
!_______________________________________________________________________
!
   subroutine msFlops_count( count )

      integer*8, intent(out) :: count
      !------ API end ------

#ifdef _DEVLP
#include "papi/fpapi.inc"

      integer :: retval
      real :: proc_time, mflops, real_time
      integer*8 :: flpins

   !------ end of declarations -- execution starts hereafter  ------

      if( .not. mf_papi_init_ok ) then
         call PrintMessage( "msFlops", "W",                             &
                            "PAPI is not initialized!" )
         count = -1
         return
      end if

      call PAPIf_flips( real_time, proc_time, flpins, mflops, retval )
      if( retval /= PAPI_OK ) then
         call PrintMessage( "msFlops", "W",                             &
                            "PAPI is not ready!" )
         count = -1
      else
         count = flpins - mf_papi_flpins_old
      end if

#endif
   end subroutine msFlops_count
!_______________________________________________________________________
!
   function mfIsFlopsOk( ) result( bool )

      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      ! tests if the Flops (via PAPI) function is available
      !
      ! requires a specific version of the PAPI library
      ! (cf. in papi/version)

#include "papi/fpapi.inc"

      integer :: retval

   !------ end of declarations -- execution starts hereafter  ------

      bool = .false.

      if( mf_papi_init_ok ) then
         bool = .true.
         return
      end if

      if( mf_papi_available == TRUE ) then
         bool = .true.
         return
      else if( mf_papi_available == FALSE ) then
         return
      else ! mf_papi_available == UNKNOWN

         retval = PAPI_VER_CURRENT
         call PAPIf_library_init( retval )
         if( retval /= PAPI_VER_CURRENT ) then
            return
         end if

         call PAPIf_query_event( PAPI_FP_INS, retval )
!!         call PAPIf_query_event( PAPI_FP_OPS, retval )
         if( retval /= PAPI_OK ) then
            return
         end if

      end if

      bool = .true.
      mf_papi_available = TRUE

#endif
   end function mfIsFlopsOk
!_______________________________________________________________________
!
   function mfFlops( ) result( out )

      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
#include "papi/fpapi.inc"

      integer :: retval
      real :: proc_time, mflops, real_time
      integer*8 :: flpins

   !------ end of declarations -- execution starts hereafter  ------

      if( .not. mf_papi_init_ok ) then
         call PrintMessage( "mfFlops", "W",                             &
                            "PAPI is not initialized!" )
         out = -1
         go to 99
      end if

      call PAPIf_flips( real_time, proc_time, flpins, mflops, retval )
      if( retval /= PAPI_OK ) then
         call PrintMessage( "mfFlops", "W",                             &
                            "PAPI is not ready!" )
         out = -1
      else
         out = dble(flpins - mf_papi_flpins_old)
      end if

 99   continue

      out%status_temporary = .true.

#endif
   end function mfFlops
#endif
!_______________________________________________________________________
!
   subroutine msUsePhysUnits( mode )

      character(len=*), intent(in) :: mode
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( to_lower(trim(mode)) == "on" ) then
         mf_phys_units = .true.
      else if( to_lower(trim(mode)) == "off" ) then
         mf_phys_units = .false.
      else
         call PrintMessage( "msUsePhysUnits", "E",                      &
                            "arg. must be 'on' or 'off'!" )
      end if

#endif
   end subroutine msUsePhysUnits
!_______________________________________________________________________
!
   function mfNnz( A ) result( out )

      type(mfArray) :: A
      integer       :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: ncol

   !------ end of declarations -- execution starts hereafter  ------

      if( A%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "mfNnz", "W",                               &
                            "arg. is empty." )
         return
      end if

      call msInitArgs( A )

      out = 0

      if( A%data_type == MF_DT_BOOL .or. A%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "mfNnz", "E",                               &
                            "cannot be applied to a logical!",          &
                            "(use mfCount instead)" )
         go to 99
      end if

      if( mfIsPerm(A) ) then
         call PrintMessage( "mfNnz", "E",                               &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( mfIsSparse(A) ) then

         ncol = A%shape(2)
         out = A%j(ncol+1) - 1

      else

         if( A%data_type == MF_DT_DBLE ) then
            out = count( A%double /= 0.0d0 )
         else if( A%data_type == MF_DT_CMPLX ) then
            out = count( A%cmplx /= (0.0d0,0.0d0) )
         end if

      end if


 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfNnz
!_______________________________________________________________________
!
   subroutine msRowSort( A, struct_only )

      type(mfArray) :: A
      logical, intent(in), optional :: struct_only
      !------ API end ------

#ifdef _DEVLP
      integer :: nrow, ncol, nb_removed, nnz
      logical :: unique
      character(len=15) :: nb_char
      logical :: structure_only

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "msRowSort", "W",                           &
                            "mfArray is empty." )
         go to 99
      end if

      if( .not. mfIsSparse(A) ) then
         call PrintMessage( "msRowSort", "E",                           &
                            "mfArray is not sparse!" )
         go to 99
      end if

      if( A%parameter ) then
         call PrintMessage( "msRowSort", "E",                           &
                            "'A' is a protected array (pseudo-parameter)." )
         go to 99
      end if

      nrow = A%shape(1)
      ncol = A%shape(2)

      nnz = A%j(ncol+1) - 1
      if( nnz <= 1 ) then
         call PrintMessage( "msRowSort", "I",                           &
                            "mfArray is obviously sorted (0 or 1 element)!" )
         A%row_sorted = TRUE
         go to 99
      end if

      if( present(struct_only) ) then
         structure_only = struct_only
      else
         structure_only = .false.
      end if

      if( A%data_type == MF_DT_SP_DBLE ) then
         if( structure_only ) then
            call row_sort_struct_only(nrow,ncol,A%i,A%j)
            unique = unique_entries(ncol,A%i,A%j)
            if( .not. unique ) then
               call remove_dupl_entries_struct_only(ncol,A%i,A%j,nb_removed)
               write(nb_char,"(1X,I0)") nb_removed
               call PrintMessage( "msRowSort", "I",                        &
                                  trim(adjustl(nb_char)) // " duplicated entries have been removed." )
            end if
         else
            call row_sort(nrow,ncol,A%a,A%i,A%j)
            unique = unique_entries(ncol,A%i,A%j)
            if( .not. unique ) then
               call remove_dupl_entries(ncol,A%a,A%i,A%j,nb_removed)
               write(nb_char,"(1X,I0)") nb_removed
               call PrintMessage( "msRowSort", "I",                     &
                                  trim(adjustl(nb_char)) // " duplicated entries have been removed." )
            end if
         end if
      else if( A%data_type == MF_DT_SP_CMPLX ) then
         if( structure_only ) then
            call row_sort_struct_only(nrow,ncol,A%i,A%j)
            unique = unique_entries(ncol,A%i,A%j)
            if( .not. unique ) then
               call remove_dupl_entries_struct_only(ncol,A%i,A%j,nb_removed)
               write(nb_char,"(1X,I0)") nb_removed
               call PrintMessage( "msRowSort", "I",                     &
                                  trim(adjustl(nb_char)) // " duplicated entries have been removed." )
            end if
         else
            call row_sort_cmplx(nrow,ncol,A%z,A%i,A%j)
            unique = unique_entries(ncol,A%i,A%j)
            if( .not. unique ) then
               call remove_dupl_entries_cmplx(ncol,A%z,A%i,A%j,nb_removed)
               write(nb_char,"(1X,I0)") nb_removed
               call PrintMessage( "msRowSort", "I",                     &
                                  trim(adjustl(nb_char)) // " duplicated entries have been removed." )
            end if
         end if
      else
         call PrintMessage( "msRowSort", "E",                           &
                            "unknown data type for 'A'!" )
         go to 99
      end if

      A%row_sorted = TRUE

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end subroutine msRowSort
!_______________________________________________________________________
!
   elemental pure function lep( n, x ) result( y )

      integer, intent(in) :: n
      real(kind=MF_DOUBLE), intent(in) :: x
      real(kind=MF_DOUBLE) :: y
      !------ API end ------

#ifdef _DEVLP
      ! computes the Legendre polynomial of degree 'n'
      ! at point 'x'

      real(kind=MF_DOUBLE) :: y0, y1, y2
      integer :: k

   !------ end of declarations -- execution starts hereafter  ------

      if( n < 0 ) then
         y = MF_NAN
         return
      else if( n == 0 ) then
         y2 = 1.0d0
      else if( n == 1 ) then
         y2 = x
      else
         y1 = 1.0d0
         y2 = x
         do k = 2, n
            y0 = y1
            y1 = y2
            y2 = ( (2*k-1)*x*y1 - (k-1)*y0 ) / k
         end do
      end if

      ! if normalized, we should add the following line:
      !! y = sqrt((2*n+1)/2.0d0)*y2
      y = y2

#endif
   end function lep
!_______________________________________________________________________
!
   function mfCount( A, dim ) result( out )

      type(mfArray)                 :: A
      integer, intent(in), optional :: dim
      type(mfArray)                 :: out
      !------ API end ------

#ifdef _DEVLP
      ! counts the number of true values in a boolean mfArray

      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         out = 0.0d0
         out%status_temporary = .true.
         go to 99
      end if

      if( A%data_type /= MF_DT_BOOL .and. A%data_type /= MF_DT_SP_BOOL ) then
         call PrintMessage( "mfCount", "E",                             &
                            "arg. must be boolean!" )
         go to 99
      end if

      out%data_type = MF_DT_DBLE

      if( A%data_type == MF_DT_BOOL ) then

         if( present(dim) ) then
            ! A is always considered as a matrix
            if( dim == 1 ) then
               ! sum by column (as in Matlab)
               out%shape = [ 1, A%shape(2) ]
               allocate( out%double(out%shape(1),out%shape(2)) )
               do j = 1, A%shape(2)
                  out%double(1,j) = sum(A%double(:,j))
               end do
            else if( dim == 2 ) then
               ! sum by row
               out%shape = [ A%shape(1), 1 ]
               allocate( out%double(out%shape(1),out%shape(2)) )
               do i = 1, a%shape(1)
                  out%double(i,1) = sum(A%double(i,:))
               end do
            else
               call PrintMessage( "mfCount", "E",                       &
                                  "dim must be equal to 1 or 2!" )
               go to 99
            end if
         else
            out%shape = [ 1, 1 ]
            allocate( out%double(1,1) )
            out%double(1,1) = sum( A%double )
         end if

      else ! sparse boolean

         if( present(dim) ) then
            ! A is always considered as a matrix
!### TODO
print *, "mod_core.F90: line 5232: internal error"
print *, "  'dim' cannot be present: not yet implemented!"
pause "for debugging purpose only..."
stop
         else
            if( A%shape(1) == 1 .or. A%shape(2) == 1 ) then
               ! vector
!### TODO
print *, "mod_core.F90: line 5240: internal error"
print *, "  sparse vector: not yet implemented!"
pause "for debugging purpose only..."
stop
            else
               ! matrix
               out%shape = [ 1, 1 ]
               allocate( out%double(1,1) )
               out%double(1,1) = A%j(A%shape(2)+1) - 1
            end if
         end if

      end if

      out%prop%symm = FALSE

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfCount
!_______________________________________________________________________
!
   subroutine msSetAsParameter( x1,                                     &
                                x2, x3, x4, x5, x6, x7,                 &
                                param )

      type(mfArray)           :: x1
      type(mfArray), optional :: x2, x3, x4, x5, x6, x7
      logical                 :: param
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      call Implem_msSetAsParameter( x1, param )

      if( present(x2) ) then
         call Implem_msSetAsParameter( x2, param )
         if( present(x3) ) then
            call Implem_msSetAsParameter( x3, param )
            if( present(x4) ) then
               call Implem_msSetAsParameter( x4, param )
               if( present(x5) ) then
                  call Implem_msSetAsParameter( x5, param )
                  if( present(x6) ) then
                     call Implem_msSetAsParameter( x6, param )
                     if( present(x7) ) then
                        call Implem_msSetAsParameter( x7, param )
                     end if
                  end if
               end if
            end if
         end if
      end if

#endif
   end subroutine msSetAsParameter
!_______________________________________________________________________
!
   subroutine Implem_msSetAsParameter( x, bool )

      type(mfArray) :: x
      logical       :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( x%status_temporary ) then
         call PrintMessage( "msSetAsParameter", "E",                    &
                            "mfArray 'x' cannot be temporary!" )
         return
      end if

      if( bool ) then
         x%parameter = .true.
      else ! bool == .false.
         x%parameter = .false.
      end if

#endif
   end subroutine Implem_msSetAsParameter
!_______________________________________________________________________
!
   subroutine msSetTermWidth_real( width )

      integer, intent(in) :: width
      !------ API end ------

#ifdef _DEVLP
      ! Manually set the character width of the terminal, for pretty
      ! print with msDisplay

   !------ end of declarations -- execution starts hereafter  ------

      if( width < 40 ) then
         call PrintMessage( "msSetTermWidth", "E",                        &
                            "string value must be equal to 'auto'" )
         return
      end if

      ncol_term_val = width

      ncol_term_is_known = .true.

#endif
   end subroutine msSetTermWidth_real
!_______________________________________________________________________
!
   subroutine msSetTermWidth_char( mode )

      character(len=*), intent(in) :: mode
      !------ API end ------

#ifdef _DEVLP
      ! Returns to the automatic detection mode of the character width
      ! of the terminal

   !------ end of declarations -- execution starts hereafter  ------

      if( mode /= "auto" ) then
         call PrintMessage( "msSetTermWidth", "E",                        &
                            "string value must be equal to 'auto'" )
         return
      end if

      ncol_term_is_known = .false.

#endif
   end subroutine msSetTermWidth_char
!_______________________________________________________________________
!
   function mfGetTermWidth() result ( out )

      integer :: out
      !------ API end ------

#ifdef _DEVLP
      ! returns the current nb of columns (i.e. character width) of the
      ! terminal (actually, at the time of the begin of the program
      ! because this value is never updated)

      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      if( ncol_term_is_known ) then
         out = ncol_term_val
         return
      end if

      call term_col( out, status )

      if( status /= 0 ) then
         call PrintMessage( "mfGetTermWidth", "I",                        &
                            "value hasn't been determined automatically." )
      end if

      ncol_term_val = out
      ncol_term_is_known = .true.

#endif
   end function mfGetTermWidth
!_______________________________________________________________________
!
   subroutine msSetAutoComplex( bool )

      logical, intent(in) :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      mf_auto_complex = bool

#endif
   end subroutine msSetAutoComplex
!_______________________________________________________________________
!
   function mfGetAutoComplex() result ( out )

      logical :: out
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      out = mf_auto_complex

#endif
   end function mfGetAutoComplex
!_______________________________________________________________________
!
   function get_addr_mfarray( A ) result ( out )

      type(mfArray) :: A
      integer(kind=MF_ADDRESS) :: out
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      out =  _MF_LOC_ANY_OBJ_(A)

#endif
   end function get_addr_mfarray
!_______________________________________________________________________
!
   function get_addr_mfarray_ptr( A ) result ( out )

      type(mfArray), pointer :: A
      integer(kind=MF_ADDRESS) :: out
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( associated(A) ) then
         out =  _MF_LOC_ANY_OBJ_(A)
      else
         out = 0
      end if

#endif
   end function get_addr_mfarray_ptr
!_______________________________________________________________________
!
   subroutine get_address_of_out_args( int_arr, pos, out, n )

      integer(kind=MF_ADDRESS) :: int_arr(:)
      integer :: pos, n
      type(mf_Out) :: out
      !------ API end ------

#ifdef _DEVLP
      ! stores the address of the 'n' first elements of out
      ! inside the integer array 'int_arr' from position 'pos'

   !------ end of declarations -- execution starts hereafter  ------

      if( n > 10 ) then
         write(STDERR,*)
         write(STDERR,*) "(MUESLI get_address_of_out_args:) internal error:"
         write(STDERR,*) "  -> currently, n cannot be greater than 10."
         mf_message_displayed = .true.
         call muesli_trace( pause="yes" )
         stop
      end if

      int_arr(pos) = get_addr_mfarray_ptr( out%ptr1 )
      if( n == 1 ) return

      int_arr(pos+1) = get_addr_mfarray_ptr( out%ptr2 )
      if( n == 2 ) return

      int_arr(pos+2) = get_addr_mfarray_ptr( out%ptr3 )
      if( n == 3 ) return

      int_arr(pos+3) = get_addr_mfarray_ptr( out%ptr4 )
      if( n == 4 ) return

      int_arr(pos+4) = get_addr_mfarray_ptr( out%ptr5 )
      if( n == 5 ) return

      int_arr(pos+5) = get_addr_mfarray_ptr( out%ptr6 )
      if( n == 6 ) return

      int_arr(pos+6) = get_addr_mfarray_ptr( out%ptr7 )
      if( n == 7 ) return

      int_arr(pos+7) = get_addr_mfarray_ptr( out%ptr8 )
      if( n == 8 ) return

      int_arr(pos+8) = get_addr_mfarray_ptr( out%ptr9 )
      if( n == 9 ) return

      int_arr(pos+9) = get_addr_mfarray_ptr( out%ptr10 )

#endif
   end subroutine get_address_of_out_args
!_______________________________________________________________________
!
   subroutine get_address_of_in_args( int_arr, pos, a1,                 &
                                      a2, a3, a4, a5 )

      integer(kind=MF_ADDRESS) :: int_arr(:)
      integer :: pos
      type(mfArray) :: a1
      type(mfArray), optional :: a2, a3, a4, a5
      !------ API end ------

#ifdef _DEVLP
      ! stores the address of some mfArrays (a1, a2, ...)
      ! inside the integer array 'int_arr' from position 'pos'

   !------ end of declarations -- execution starts hereafter  ------

      int_arr(pos) = get_addr_mfarray( a1 )

      if( present(a2) ) then
         int_arr(pos+1) = get_addr_mfarray( a2 )
         if( present(a3) ) then
            int_arr(pos+2) = get_addr_mfarray( a3 )
            if( present(a4) ) then
               int_arr(pos+3) = get_addr_mfarray( a4 )
               if( present(a5) ) then
                  int_arr(pos+4) = get_addr_mfarray( a5 )
               end if
            end if
         end if
      end if

#endif
   end subroutine get_address_of_in_args
!_______________________________________________________________________
!
#if !defined(_WINDOWS) && !defined(_DARWIN)
   subroutine msSetRoundingMode( rounding_mode )

      character(len=*), intent(in) :: rounding_mode
      !------ API end ------

#ifdef _DEVLP
      ! rounding_mode =
      !   "to_zero" : 0           IEEE 754 rounding to zero
      !   "nearest" : 1 [default] IEEE 754 rounding to nearest
      !   "up"      : +99         IEEE 754 rounding to pos. infinity
      !   "down"    : -99         IEEE 754 rounding to neg. infinity

   !------ end of declarations -- execution starts hereafter  ------

      select case( to_lower(rounding_mode) )
         case( "to_zero" )
            call round_towardzero()
            mf_fp_rounding_mode = 0
         case( "nearest" )
            call round_nearest()
            mf_fp_rounding_mode = 1
         case( "up" )
            call round_upward()
            mf_fp_rounding_mode = 99
         case( "down" )
            call round_downward()
            mf_fp_rounding_mode = -99
         case default
            call PrintMessage( "msSetRoundingMode", "E",                &
                               "bad value for argument!",               &
                               "[rounding_mode unchanged]" )
      end select

#endif
   end subroutine msSetRoundingMode
!_______________________________________________________________________
!
   function mfGetRoundingMode() result( rounding_mode )

      character(len=7) :: rounding_mode
      !------ API end ------

#ifdef _DEVLP
      ! cf. msSetRoundingMode

   !------ end of declarations -- execution starts hereafter  ------

      select case( mf_fp_rounding_mode )
         case( 0 )
            rounding_mode = "to_zero"
         case( 1 )
            rounding_mode = "nearest"
         case( 99 )
            rounding_mode = "up"
         case( -99 )
            rounding_mode = "down"
      end select

#endif
   end function mfGetRoundingMode
#endif
!_______________________________________________________________________
!
#if !defined(_WINDOWS) && !defined(_DARWIN)
   subroutine msEnableFPE( exception, full_trapping )

      character(len=*), intent(in)           :: exception
      logical,          intent(in), optional :: full_trapping
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      call clear_all_FPE()

      select case( exception )
         case( "overflow" )
            call enableFPE( MF_OVERFLOW_EXC )
         case( "zero_divide" )
            call enableFPE( MF_DIVBYZERO_EXC )
         case( "invalid" )
            call enableFPE( MF_INVALID_EXC )
         case( "usual_exceptions" )
            call enableFPE( MF_USUAL_EXCEPTS )
         case( "underflow" )
            call enableFPE( MF_UNDERFLOW_EXC )
         case default
            call PrintMessage( "msEnableFPE", "E",                      &
                               "bad character(len=*) argument!" )
      end select

      if( present(full_trapping) ) then
         mf_full_trapping = full_trapping
      end if

#endif
   end subroutine msEnableFPE
!_______________________________________________________________________
!
   subroutine msDisableFPE( exception )

      character(len=*), intent(in) :: exception
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      select case( exception )
         case( "overflow" )
            call disableFPE( MF_OVERFLOW_EXC )
         case( "zero_divide" )
            call disableFPE( MF_DIVBYZERO_EXC )
         case( "invalid" )
            call disableFPE( MF_INVALID_EXC )
         case( "usual_exceptions" )
            call disableFPE( MF_USUAL_EXCEPTS )
         case( "underflow" )
            call disableFPE( MF_UNDERFLOW_EXC )
         case default
            call PrintMessage( "msDisableFPE", "E",                     &
                               "bad character(len=*) argument!" )
      end select

#endif
   end subroutine msDisableFPE
!_______________________________________________________________________
!
#if defined _READLINE
   function mfReadline( prompt ) result( res )

      character(len=*), intent(in) :: prompt
      character(len=f90readline_max_length) :: res
      !------ API end ------

#ifdef _DEVLP
      integer :: long

   !------ end of declarations -- execution starts hereafter  ------

      res = ""
      call mf_readline_c( long, res, prompt//char(0) )
      if( long == -1 ) then
         res = "Ctrl-D"
      else
         res(long+1:) = " "
      end if

#endif
   end function mfReadline
!_______________________________________________________________________
!
   subroutine msRemoveLastEntryInHistory( )

   !------ end of declarations -- execution starts hereafter  ------

#ifdef _DEVLP
      call ms_remove_last_entry_c( )

#endif
   end subroutine msRemoveLastEntryInHistory
!_______________________________________________________________________
!
   subroutine msAddEntryInHistory( string )

      character(len=*) :: string
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      call ms_add_history_c( trim(string)//char(0) )

#endif
   end subroutine msAddEntryInHistory
!_______________________________________________________________________
!
   subroutine msClearHistory( )

   !------ end of declarations -- execution starts hereafter  ------

#ifdef _DEVLP
      call ms_clear_history_c( )

#endif
   end subroutine msClearHistory
!_______________________________________________________________________
!
   subroutine msWriteHistoryFile( filename )

      character(len=*), intent(in), optional :: filename
      !------ API end ------

#ifdef _DEVLP
      integer :: mypid
      character(len=16) :: str_mypid

   !------ end of declarations -- execution starts hereafter  ------

      if( present(filename) ) then
         call ms_write_history_c( filename//char(0) )
      else
         call ms_write_history_c( "~/.mfreadline.history"//char(0) )
      end if

#endif
   end subroutine msWriteHistoryFile
!_______________________________________________________________________
!
   subroutine msReadHistoryFile( filename )

      character(len=*), intent(in), optional :: filename
      !------ API end ------

#ifdef _DEVLP
      integer :: mypid
      character(len=16) :: str_mypid

   !------ end of declarations -- execution starts hereafter  ------

      if( present(filename) ) then
         call ms_read_history_c( filename//char(0) )
      else
         call ms_read_history_c( "~/.mfreadline.history"//char(0) )
      end if

#endif
   end subroutine msReadHistoryFile
#endif
#endif
!_______________________________________________________________________
!
   function mfIsVersion( v_1, op, v_2 ) result( bool )

      character(len=*), intent(in) :: v_1, op, v_2
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      ! in order to be able to process MF_MUESLI_VERSION, the second part
      ! (after the "_"), if any, is discarded

      integer :: i, j
      integer :: len1, len2, n, v1, v2
      character(len=2) :: oper
      character(len=8) :: v_1_short, v_2_short

   !------ end of declarations -- execution starts hereafter  ------

      if( len(op) > 2 ) then
         call PrintMessage( "mfIsVersion", "E",                         &
                            "string length for op must be <= than 2!" )
         return
      end if
      oper = adjustl(op)

      if( len_trim(v_1) < 5 .or. len_trim(v_2) < 5 ) then
         call PrintMessage( "mfIsVersion", "E",                         &
                            "string length for v_1 and v_2 must be >= 5!" )
         return
      end if

      ! search for "_" in v_1 and v_2...
      i = index( v_1, "_" )
      if( i > 0 ) then
         v_1_short = v_1(1:i-1)
      else
         v_1_short = v_1
      end if
      i = index( v_2, "_" )
      if( i > 0 ) then
         v_2_short = v_2(1:i-1)
      else
         v_2_short = v_2
      end if

      len1 = len_trim(v_1_short)
      ! extracting a version number from v_1_short
      i = index(v_1_short,".")
      if( i <= 1 ) then
         call PrintMessage( "mfIsVersion", "E",                         &
                            "bad version number for arg v_1!" )
         return
      else
         j = index(v_1_short,".",back=.true.)
         if( j == 0 .or. j==i .or. j == len1 ) then
            call PrintMessage( "mfIsVersion", "E",                      &
                               "bad version number for arg v_1!" )
            return
         end if
      end if
      read(v_1_short(1:i-1),*,ERR=99) n
      v1 = n*10000
      read(v_1_short(i+1:j-1),*,ERR=99) n
      v1 = v1 + n*100
      read(v_1_short(j+1:),*,ERR=99) n
      v1 = v1 + n

      len2 = len_trim(v_2_short)
      ! extracting a version number from v_2_short
      i = index(v_2_short,".")
      if( i <= 1 ) then
         call PrintMessage( "mfIsVersion", "E",                         &
                            "bad version number for arg v_2!" )
         return
      else
         j = index(v_2_short,".",back=.true.)
         if( j == 0 .or. j==i .or. j == len2 ) then
            call PrintMessage( "mfIsVersion", "E",                      &
                               "bad version number for arg v_2!" )
            return
         end if
      end if
      read(v_2_short(1:i-1),*,ERR=99) n
      v2 = n*10000
      read(v_2_short(i+1:j-1),*,ERR=99) n
      v2 = v2 + n*100
      read(v_2_short(j+1:),*,ERR=99) n
      v2 = v2 + n

      select case( oper )
         case( "==" )
            if( v1 == v2 ) then
               bool = .true.
            else
               bool = .false.
            end if
         case( "<=" )
            if( v1 <= v2 ) then
               bool = .true.
            else
               bool = .false.
            end if
         case( "< " )
            if( v1 < v2 ) then
               bool = .true.
            else
               bool = .false.
            end if
         case( ">=" )
            if( v1 >= v2 ) then
               bool = .true.
            else
               bool = .false.
            end if
         case( "> " )
            if( v1 > v2 ) then
               bool = .true.
            else
               bool = .false.
            end if
         case default
            call PrintMessage( "mfIsVersion", "E",                      &
                               "bad value for arg op!" )
            return
      end select

      return

 99   continue
      call PrintMessage( "mfIsVersion", "E",                            &
                         "bad version number for arg v_1 or v_2!" )

#endif
   end function mfIsVersion
!_______________________________________________________________________
!
   subroutine msRequMuesliVer( version_min, version_max )

      character(len=*), intent(in)           :: version_min
      character(len=*), intent(in), optional :: version_max
      !------ API end ------

#ifdef _DEVLP
      logical :: ok
      integer :: i
      character(len=8) :: muesli_short_version

      ! Check a minimum version of Muesli to run a program.
      ! If MF_MUESLI_VERSION is not sufficient, the program stops.

   !------ end of declarations -- execution starts hereafter  ------

      ok = mfIsVersion( MF_MUESLI_VERSION, ">=", version_min )
      if( ok ) then
         if( present(version_max) ) then
            ok = mfIsVersion( MF_MUESLI_VERSION, "<=", version_max )
         end if
      end if

      if( .not. ok ) then
         ! shortering MF_MUESLI_VERSION...
         i = index( MF_MUESLI_VERSION, "_" )
         if( i > 0 ) then
            muesli_short_version = MF_MUESLI_VERSION(1:i-1)
         else
            print *, "msRequMuesliVer: internal error"
            print *, "                 no underscore found in MF_MUESLI_VERSION!"
            pause "for debugging purpose only..."
            stop
         end if

         if( present(version_max) ) then
            call PrintMessage( "msRequMuesliVer", "E",                  &
                               "Your program requires a Muesli version ranged in [" &
                               // trim(version_min) // "," // trim(version_max), &
                               "but you are using Muesli " // trim(muesli_short_version) )
         else
            call PrintMessage( "msRequMuesliVer", "E",                  &
                               "Your program requires a Muesli version at least equal to " &
                               // trim(version_min),                    &
                               "but you are using Muesli " // trim(muesli_short_version) )
         end if
      end if
#endif
   end subroutine msRequMuesliVer
!_______________________________________________________________________
!
   subroutine msGetBlasLib( string )

      character(len=mf_blas_lib_strlen) :: string
      !------ API end ------

#ifdef _DEVLP
      ! Get dynamically the BLAS Library Implementation

      ! The current routine is a subroutine and not a function!
      ! The call of the 'system' command cannot be used inside a function...
      ! (got some hangs with GNU_GFC)

      character(len=128) :: cmd, tmp_file
      integer :: status, mypid, unit
      character(len=16) :: str_mypid

#if defined _INTEL_IFC
#ifdef _HAS_MKL
      integer :: i1, i2
#endif
#endif

   !------ end of declarations -- execution starts hereafter  ------

#if defined _INTEL_IFC
#ifdef _HAS_MKL
      call mkl_get_version_string( string )
      ! make the string shorter
      i1 = index( string, "Version" )
      i2 = index( string, "Product" )
      if( i1 /= 0 .and.i2 /= 0 .and. i2>i1 ) then
         string = string(1:i2-1)
      end if

      return
#endif
#endif

#ifdef _WINDOWS
      ! to be updated by hand before compiling under Windows!
      string = "Reference (http://www.netlib.org/lapack)"
      return
#endif

      ! default returned string
      string = "Unknown"

      mypid = getpid()
      write( str_mypid, "(I0)" ) mypid
      tmp_file = "/tmp/muesli_blas_implementation_" // trim(str_mypid)

      cmd = "muesli-config --blas-implem > " // trim(tmp_file)
#if defined _GNU_GFC | defined _INTEL_IFC
      status = system( trim(cmd) )
#else
  '(MUESLI msGetBlasLib:) compiler not defined!'
#endif
      if( status == 0 ) then
         call find_unit(unit)
         open( unit, file=trim(tmp_file) )
         read(unit,"(A)") string
      else
         write(STDERR,*) ""
         write(STDERR,*) "(MUESLI FML:) ERROR:"
         write(STDERR,*) "  It appears that the 'muesli-config' script is not available (forgot to"
         write(STDERR,*) "  update your PATH variable?) or that the library is not yet installed..."
         write(STDERR,*) ""
         write(STDERR,*) "  The inquiring of the BLAS and LAPACK implementation, via the routines"
         write(STDERR,*) "  'msGetBlasLib' and 'msGetLapackLib', requires the access to the"
         write(STDERR,*) "  'muesli-config' script."
         write(STDERR,*) ""
         return
      end if
      status = system( "rm -f " // trim(tmp_file) )

#endif
   end subroutine msGetBlasLib
!_______________________________________________________________________
!
   subroutine msGetLapackLib( string )

      character(len=mf_lapack_lib_strlen) :: string
      !------ API end ------

#ifdef _DEVLP
      ! Get dynamically the LAPACK Library Implementation

      ! The current routine is a subroutine and not a function!
      ! The call of the 'system' command cannot be used inside a function...
      ! (got some hangs with GNU_GFC)

      character(len=128) :: cmd, tmp_file
      integer :: status, mypid, unit
      character(len=16) :: str_mypid

#if defined _INTEL_IFC
#ifdef _HAS_MKL
      integer :: i1, i2
#endif
#endif

   !------ end of declarations -- execution starts hereafter  ------

#if defined _INTEL_IFC
#ifdef _HAS_MKL
      call mkl_get_version_string( string )
      ! make the string shorter
      i1 = index( string, "Version" )
      i2 = index( string, "Product" )
      if( i1 /= 0 .and.i2 /= 0 .and. i2>i1 ) then
         string = string(1:i2-1)
      end if

      return
#endif
#endif

#ifdef _WINDOWS
      ! to be updated by hand before compiling under Windows!
      string = "Reference (http://www.netlib.org/lapack)"
      return
#endif

      ! default returned string
      string = "Unknown"

      mypid = getpid()
      write( str_mypid, "(I0)" ) mypid
      tmp_file = "/tmp/muesli_lapack_implementation_" // trim(str_mypid)

      cmd = "muesli-config --lapack-implem > " // trim(tmp_file)
#if defined _GNU_GFC | defined _INTEL_IFC
      status = system( trim(cmd) )
#else
  '(MUESLI msGetLapackLib:) compiler not defined!'
#endif
      if( status == 0 ) then
         call find_unit(unit)
         open( unit, file=trim(tmp_file) )
         read(unit,"(A)") string
      else
         write(STDERR,*) ""
         write(STDERR,*) "(MUESLI FML:) ERROR:"
         write(STDERR,*) "  It appears that the 'muesli-config' script is not available (forgot to"
         write(STDERR,*) "  update your PATH variable?) or that the library is not yet installed..."
         write(STDERR,*) ""
         write(STDERR,*) "  The inquiring of the BLAS and LAPACK implementation, via the routines"
         write(STDERR,*) "  'msGetBlasLib' and 'msGetLapackLib', requires the access to the"
         write(STDERR,*) "  'muesli-config' script."
         write(STDERR,*) ""
         return
      end if
      status = system( "rm -f " // trim(tmp_file) )

#endif
   end subroutine msGetLapackLib
!_______________________________________________________________________
!
   subroutine msGetSuiteSparseLib( string )

      character(len=mf_version_length) :: string ! 8 chars
      !------ API end ------

#ifdef _DEVLP
      ! Get dynamically the SuiteSparse Library information

      ! The current routine is a subroutine and not a function!
      ! The call of the 'system' command cannot be used inside a function...
      ! (got some hangs with GNU_GFC)

      integer :: iver(3)
      character(len=2) :: c1, c2, c3

   !------ end of declarations -- execution starts hereafter  ------

#ifdef _WINDOWS
      ! to be updated by hand before compiling under Windows!
      string = "5.4.0"
      return
#endif

      ! default returned string
      string = "Unknown"

      call get_suitesparse_version( iver )
      write(c1,"(I0)") iver(1)
      write(c2,"(I0)") iver(2)
      write(c3,"(I0)") iver(3)
      string = trim(c1) // "." // trim(c2) // "." // trim(c3)

#endif
   end subroutine msGetSuiteSparseLib
!_______________________________________________________________________
!
   function MF_LAPACK_VERSION() result( string )

      character(len=mf_version_length) :: string
      !------ API end ------

#ifdef _DEVLP
      integer :: major, minor, patch

   !------ end of declarations -- execution starts hereafter  ------

      call ilaver( major, minor, patch )

      write(string,"(I0,A,I0,A,I0)") major, ".", minor, ".", patch

#endif
   end function MF_LAPACK_VERSION
!_______________________________________________________________________
!
   subroutine check_named_groups( named_groups, named_groups_status )

      type(mf_DE_Named_Group), intent(in)  :: named_groups(:)
      integer,                 intent(out) :: named_groups_status
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j, ind

      named_groups_status = 0

      ! check that indices don't overlap together...
      do i = 1, size( named_groups )
         do j = 1, size( named_groups )
            if( j == i ) cycle
            ! here j /= i
            ind = named_groups(i)%begin
            if( named_groups(j)%begin <= ind .and. ind <= named_groups(j)%last ) then
               named_groups_status = -1
               return
            end if
            ind = named_groups(i)%last
            if( named_groups(j)%begin <= ind .and. ind <= named_groups(j)%last ) then
               named_groups_status = -1
               return
            end if
         end do
      end do

#endif
   end subroutine check_named_groups
!_______________________________________________________________________
!
   subroutine search_index_in_eqn_groups( ind, i_named_group, i_group_ind )

      integer, intent(in)  :: ind
      integer, intent(out) :: i_named_group, i_group_ind
      !------ API end ------

#ifdef _DEVLP
      ! works only for EQUATION groups

      integer :: i

      ! default value which means that 'ind' is not found in a named group
      i_group_ind = 0

      ! search a group such that: begin <= ind <= last
      do i = 1, size( NAMED_EQN_PTR )
         if( NAMED_EQN_PTR(i)%begin <= ind .and.                     &
                    ind <= NAMED_EQN_PTR(i)%last ) then
            i_named_group = i
            i_group_ind = ind - NAMED_EQN_PTR(i)%begin + 1
            return
         end if
      end do

#endif
   end subroutine search_index_in_eqn_groups
!_______________________________________________________________________
!
   subroutine search_index_in_var_groups( ind, i_named_group, i_group_ind )

      integer, intent(in)  :: ind
      integer, intent(out) :: i_named_group, i_group_ind
      !------ API end ------

#ifdef _DEVLP
      ! works only for VARIABLE groups

      integer :: i

      ! default value which means that 'ind' is not found in a named group
      i_group_ind = 0

      ! search a group such that: begin <= ind <= last
      do i = 1, size( NAMED_VAR_PTR )
         if( NAMED_VAR_PTR(i)%begin <= ind .and.                     &
                    ind <= NAMED_VAR_PTR(i)%last ) then
            i_named_group = i
            i_group_ind = ind - NAMED_VAR_PTR(i)%begin + 1
            return
         end if
      end do

#endif
   end subroutine search_index_in_var_groups
!_______________________________________________________________________
!
   subroutine msSetAutoFilling( r )

      real(kind=MF_DOUBLE), intent(in) :: r
      !------ API end ------

#ifdef _DEVLP
      mf_out_of_range_filling = r

#endif
   end subroutine msSetAutoFilling
!_______________________________________________________________________
!
   function mfGetAutoFilling( ) result( r )

      real(kind=MF_DOUBLE) :: r
      !------ API end ------

#ifdef _DEVLP
      r = mf_out_of_range_filling

#endif
   end function mfGetAutoFilling
!_______________________________________________________________________
!
end module mod_core
