! f90 include file
!
! list of Fortran 90 interfaces for the CHOLMOD wrapper routines,
! stored in the file: 'cholmod_f90wrapper.c'.
!
! WARNING: before using these interfaces, please read some assumptions
!          written at the beginning of 'cholmod_f90wrapper.c'.
!
! Copyright É. Canot 2003-2025 -- IPR/CNRS

interface
!-----------------------------------------------------------------------
!                      R E A L     V A L U E S
!-----------------------------------------------------------------------

!_______________________________________________________________________
!
subroutine cholmod_pos_def( n, nnz, Ap, Ai, Ax,                         &
                            pos_def )
   import :: MF_DOUBLE
   integer, intent(in) :: n ! size of the square matrix A (sparse)
                            ! [only the upper triangular part]
   integer, intent(in) :: nnz ! nb of non-zero element in A
   integer, intent(in) :: Ap(*), Ai(*) ! indexes and elements of A
   real(kind=MF_DOUBLE), intent(in) :: Ax(*)   ! (CSC storage with Fortran
                                               !  indexes, i.e. 1-based)
   integer, intent(out) :: pos_def ! 1: A is positive definite
                                   ! 0: otherwise
   ! this routine just tries to make a Choleski factorisation:
   !                    A = L L'
   ! in order to determine the positiveness of A;
   ! be aware that the L factor is not saved.
end subroutine
!_______________________________________________________________________
!
subroutine cholmod_llt_prep( n, nnz, Ap, Ai, Ax,                        &
                             c_addr, LL_addr, S_addr, lnz )
   import :: MF_ADDRESS, MF_DOUBLE
   integer, intent(in) :: n ! size of the square matrix A (sparse)
                            ! [only the upper triangular part]
   integer, intent(in) :: nnz ! nb of non-zero element in A
   integer, intent(in) :: Ap(*), Ai(*) ! indexes and elements of A
   real(kind=MF_DOUBLE), intent(in) :: Ax(*)   ! (CSC storage with Fortran
                                               !  indexes, i.e. 1-based)
   integer(kind=MF_ADDRESS), intent(out) :: c_addr ! address of the cholmod_common
                                                   ! specific to the factorisation
   integer(kind=MF_ADDRESS), intent(out) :: LL_addr ! address of the cholmod_factor
                                                    ! (special internal format)
   integer(kind=MF_ADDRESS), intent(out) :: S_addr ! address of the sparse L factor
                                                   ! (CSC format, 0-based)
                                                   ! [may be null for the simplicial
                                                   ! case]
   integer, intent(out) :: lnz ! 0: A is not positive definite,
                               !    otherwise contains nnz(L)
   ! this routine tries to make a Choleski factorisation:
   !                    A = L L'
   ! and saves the L factor, both in internal and CSC format.
   ! Indicates also if the matrix A is positive definite,
   ! via the 'lnz' output.
end subroutine
!_______________________________________________________________________
!
subroutine cholmod_llt_prep2( n, nnz, Ap, Ai, Ax,                       &
                              c_addr, LL_addr, status )
   import :: MF_ADDRESS, MF_DOUBLE
   integer, intent(in) :: n ! size of the square matrix A (sparse)
                            ! [only the upper triangular part]
   integer, intent(in) :: nnz ! nb of non-zero element in A
   integer, intent(in) :: Ap(*), Ai(*) ! indexes and elements of A
   real(kind=MF_DOUBLE), intent(in) :: Ax(*)   ! (CSC storage with Fortran
                                               !  indexes, i.e. 1-based)
   integer(kind=MF_ADDRESS), intent(out) :: c_addr ! address of the cholmod_common
                                                   ! specific to the factorisation
   integer(kind=MF_ADDRESS), intent(out) :: LL_addr ! address of the cholmod_factor
                                                    ! (special internal format)
   integer, intent(out) :: status ! 0: A is positive definite,
                                  ! 1: otherwise
   ! this routine tries to make a Choleski factorisation:
   !                    A = L L'
   ! and saves the L factor, only in internal format.
   ! Indicates also if the matrix A is positive definite,
   ! via the 'status' output.
end subroutine
!_______________________________________________________________________
!
subroutine cholmod_llt_symb( n, nnz, Ap, Ai,                            &
                             c_addr, LL_addr, AA_addr )
   import :: MF_ADDRESS
   integer, intent(in) :: n ! size of the square matrix A (sparse)
                            ! [only the upper triangular part]
   integer, intent(in) :: nnz ! nb of non-zero element in A
   integer, intent(in) :: Ap(*), Ai(*) ! indexes of A (CSC storage
                                       ! with Fortran indexes, i.e. 1-based)
   integer(kind=MF_ADDRESS), intent(out) :: c_addr ! address of the cholmod_common
                                                   ! specific to the factorisation
   integer(kind=MF_ADDRESS), intent(out) :: LL_addr ! address of the cholmod_factor
                                                    ! (special internal format)
   integer(kind=MF_ADDRESS), intent(out) :: AA_addr ! copy of with C-indexes
   ! this routine makes the symbolic part of a Choleski factorisation:
   !                    A = L L'
   ! (it also stores AA, the copy of A with C-indexes, i.e. 0-based)
end subroutine
!_______________________________________________________________________
!
subroutine cholmod_llt_num( n, nnz, Ax, c_addr, LL_addr, AA_addr,       &
                            status )
   import :: MF_ADDRESS, MF_DOUBLE
   integer, intent(in) :: n ! size of the square matrix A (sparse)
                            ! [only the upper triangular part]
   integer, intent(in) :: nnz ! nb of non-zero element in A
   real(kind=MF_DOUBLE), intent(in) :: Ax(*) ! numerical entries of A (CSC)
   integer(kind=MF_ADDRESS), intent(in) :: c_addr ! address of the cholmod_common
                                                  ! specific to the factorisation
   integer(kind=MF_ADDRESS), intent(in out) :: LL_addr ! address of the cholmod_factor
                                                       ! (special internal format)
   integer(kind=MF_ADDRESS), intent(in) :: AA_addr ! copy of with C-indexes
   integer, intent(out) :: status ! 0: A is positive definite,
                                  ! 1: otherwise
   ! this routine makes the numerical factorisation:
   !                    A = L L'
   ! (so the L factor is updated)
   ! Indicates also if the matrix A is positive definite,
   ! via the 'status' output.
end subroutine
!_______________________________________________________________________
!
subroutine cholmod_ldlt_prep( n, nnz, Ap, Ai, Ax,                       &
                              c_addr, LL_addr, S_addr, lnz )
   import :: MF_ADDRESS, MF_DOUBLE
   integer, intent(in) :: n ! size of the square matrix A (sparse)
                            ! [only the upper triangular part]
   integer, intent(in) :: nnz ! nb of non-zero element in A
   integer, intent(in) :: Ap(*), Ai(*) ! indexes and elements of A
   real(kind=MF_DOUBLE), intent(in) :: Ax(*)   ! (CSC storage with Fortran
                                               !  indexes, i.e. 1-based)
   integer(kind=MF_ADDRESS), intent(out) :: c_addr ! address of the cholmod_common
                                                   ! specific to the factorisation
   integer(kind=MF_ADDRESS), intent(out) :: LL_addr ! address of the cholmod_factor
                                                    ! (special internal format)
   integer(kind=MF_ADDRESS), intent(out) :: S_addr ! address of the sparse L factor
                                                   ! (CSC format, 0-based)
   integer, intent(out) :: lnz ! 0: A is not positive definite,
                               !    otherwise contains nnz(L)
   ! this routine tries to make a Choleski factorisation:
   !                    A = L D L'
   ! and saves the L factor, both in internal and CSC format.
   ! Indicates also if the matrix A is positive definite,
   ! via the 'lnz' output.
   !
   ! Be aware that this kind of decomposition may fail.
end subroutine
!_______________________________________________________________________
!
subroutine cholmod_get_factor( c_addr, LL_addr, S_addr, n, lnz,         &
                               Lp, Li, Lx, perm )
   import :: MF_ADDRESS, MF_DOUBLE
   integer(kind=MF_ADDRESS), intent(in) :: c_addr ! address of the cholmod_common
                                                  ! specific to the factorisation
   integer(kind=MF_ADDRESS), intent(in) :: LL_addr ! address of the cholmod_factor
                                                   ! (special internal format)
   integer(kind=MF_ADDRESS), intent(in) :: S_addr ! address of the sparse L factor
                                                  ! (CSC format, 0-based)
   integer, intent(in) :: n ! size of the square matrix A (sparse)
                            ! [only the upper triangular part]
   integer, intent(in) :: lnz !  nnz(L)
   integer, intent(in) :: Lp(*), Li(*) ! indexes and elements of L
   real(kind=MF_DOUBLE), intent(in) :: Lx(*)   ! (CSC storage with Fortran
                                               !  indexes, i.e. 1-based)
   integer, intent(out) :: perm(*) ! permutation vector
   ! this routine gets the L factor of the Choleski factorisation:
   !                    A = L L'
   ! gets also the permutation vector applied to A before the
   ! factorisation (reducing the fill-in).
end subroutine
!_______________________________________________________________________
!
subroutine cholmod_get_diag( c_addr, LL_addr, S_addr, n, lnz,           &
                             diag )
   import :: MF_ADDRESS, MF_DOUBLE
   integer(kind=MF_ADDRESS), intent(in) :: c_addr ! address of the cholmod_common
                                                  ! specific to the factorisation
   integer(kind=MF_ADDRESS), intent(in) :: LL_addr ! address of the cholmod_factor
                                                   ! (special internal format)
   integer(kind=MF_ADDRESS), intent(in) :: S_addr ! address of the sparse L factor
                                                  ! (CSC format, 0-based)
   integer, intent(in) :: n ! size of the square matrix A (sparse)
                            ! [only the upper triangular part]
   integer, intent(in) :: lnz !  nnz(L)
   real(kind=MF_DOUBLE), intent(out) :: diag(*) ! diagonal of the L factor
   ! this routine extracts the diagonal of the L factor.
end subroutine
!_______________________________________________________________________
!
subroutine cholmod_solve_f90( n, nnz, Ap, Ai, Ax, b,                    &
                              x, status )
   import :: MF_DOUBLE
   integer, intent(in) :: n ! size of the square matrix A (sparse)
                            ! [only the upper triangular part]
   integer, intent(in) :: nnz ! nb of non-zero element in A
   integer, intent(in) :: Ap(*), Ai(*) ! indexes and elements of A
   real(kind=MF_DOUBLE), intent(in) :: Ax(*)   ! (CSC storage with Fortran
                                               !  indexes, i.e. 1-based)
   real(kind=MF_DOUBLE), intent(in) :: b(*) ! RHS of the equation
   real(kind=MF_DOUBLE), intent(out) :: x(*) ! vector solution
   integer, intent(out) :: status !  0 : ok, A is positive definite,
                                  !  1 : failed
                                  ! -1 : internal failed (out of memory, ...)
   ! this routine solves the following system:
   !                    A x = b
   ! where A is symmetric positive definite.
   !
end subroutine
!_______________________________________________________________________
!
subroutine cholmod_free_factor_f90( c_addr, LL_addr )
   import :: MF_ADDRESS
   integer(kind=MF_ADDRESS), intent(in) :: c_addr ! address of the cholmod_common
                                                  ! specific to the factorisation
   integer(kind=MF_ADDRESS), intent(in) :: LL_addr ! address of the cholmod_factor
                                                   ! (special internal format)
   ! this routine frees internal structures used by the CHOLMOD
   ! solver.
end subroutine
!_______________________________________________________________________
!
subroutine cholmod_free_factor_2( c_addr, LL_addr, S_addr )
   import :: MF_ADDRESS
   integer(kind=MF_ADDRESS), intent(in) :: c_addr ! address of the cholmod_common
                                                  ! specific to the factorisation
   integer(kind=MF_ADDRESS), intent(in) :: LL_addr ! address of the cholmod_factor
                                                   ! (special internal format)
   integer(kind=MF_ADDRESS), intent(in) :: S_addr ! address of the sparse L factor
                                                  ! (CSC format, 0-based)
   ! this routine frees internal structures used by the CHOLMOD
   ! solver.
end subroutine
!_______________________________________________________________________
!
subroutine cholmod_solve_factor( n, c_addr, L_addr, b,                  &
                                 x )
   import :: MF_ADDRESS, MF_DOUBLE
   integer, intent(in) :: n ! size of the square matrix A (sparse)
   integer(kind=MF_ADDRESS), intent(in) :: c_addr ! address of the cholmod_common
                                                  ! specific to the factorisation
   integer(kind=MF_ADDRESS), intent(in) :: L_addr ! address of the cholmod_factor
                                                  ! (special internal format)
   real(kind=MF_DOUBLE), intent(in) :: b(*) ! RHS of the equation
   real(kind=MF_DOUBLE), intent(out) :: x(*) ! vector solution
   ! this routine solves the following system:
   !                    L L' x = b
   ! where L is the Choleski factor obtained by another routine.
end subroutine
!_______________________________________________________________________
!
subroutine cholmod_solve_factor_nRHS( n, c_addr, L_addr, nrhs, b,       &
                                      x )
   import :: MF_ADDRESS, MF_DOUBLE
   integer, intent(in) :: n ! size of the square matrix A (sparse)
   integer(kind=MF_ADDRESS), intent(in) :: c_addr ! address of the cholmod_common
                                                  ! specific to the factorisation
   integer(kind=MF_ADDRESS), intent(in) :: L_addr ! address of the cholmod_factor
                                                  ! (special internal format)
   integer, intent(in) :: nrhs ! number of columns of the RHS
   real(kind=MF_DOUBLE), intent(in) :: b(n,*) ! RHS of the equation
   real(kind=MF_DOUBLE), intent(out) :: x(n,*) ! vector solution
   ! this routine solves the following system:
   !                    L L' x = b
   ! where L is the Choleski factor obtained by another routine.
   ! It can handle multiple RHS simultaneously.
end subroutine
!_______________________________________________________________________
!
subroutine cholmod_spsol_fact_nRHS_prep( n, c_addr, L_addr, nrhs,       &
                                         bnz, bp, bi, bx,               &
                                         x_addr, xnz )
   import :: MF_ADDRESS, MF_DOUBLE
   integer, intent(in) :: n ! size of the square matrix A (sparse)
   integer(kind=MF_ADDRESS), intent(in) :: c_addr ! address of the cholmod_common
                                                  ! specific to the factorisation
   integer(kind=MF_ADDRESS), intent(in) :: L_addr ! address of the cholmod_factor
                                                  ! (special internal format)
   integer, intent(in) :: nrhs ! number of columns of the RHS
   integer, intent(in) :: bnz ! number of nonzero elements in RHS
   integer, intent(in) :: bp(*), bi(*) ! indexes and elements of b
   real(kind=MF_DOUBLE), intent(in) :: bx(*)   ! (CSC storage with Fortran
                                               !  indexes, i.e. 1-based)
   integer(kind=MF_ADDRESS), intent(out) :: x_addr ! address of the cholmod_sparse
                                                   ! (special internal format) solution
   integer, intent(out) :: xnz ! nnz of solution
   ! this routine solves the following system:
   !                    L L' x = b
   ! where L is the Choleski factor obtained by another routine;
   ! The RHS 'b' should have multiple columns.
   ! This routine returns only the address of 'x' and its number
   ! of non-zero elements.
end subroutine
!_______________________________________________________________________
!
subroutine cholmod_get_sol_nRHS( c_addr, x_addr, m, n, xnz,             &
                                 xp, xi, xx )
   import :: MF_ADDRESS, MF_DOUBLE
   integer(kind=MF_ADDRESS), intent(in) :: c_addr ! address of the cholmod_common
                                                  ! specific to the factorisation
   integer(kind=MF_ADDRESS), intent(in) :: x_addr ! address of the cholmod_sparse
                                                  ! (special internal format) solution
   integer, intent(in) :: m ! number of rows of the solution
   integer, intent(in) :: n ! number of columns of the solution
   integer, intent(in) :: xnz ! number of nonzero elements in solution
   integer, intent(out) :: xp(*), xi(*) ! indexes and elements of x
   real(kind=MF_DOUBLE), intent(out) :: xx(*)   ! (CSC storage with Fortran
                                                !  indexes, i.e. 1-based)
   ! this routine gets the elements of a CHOLMOD_SPARSE structure
   ! (usually after calling 'cholmod_spsol_fact_nRHS_prep')
end subroutine

!-----------------------------------------------------------------------
!                   C O M P L E X     V A L U E S
!-----------------------------------------------------------------------

!_______________________________________________________________________
!
subroutine cholmod_cmplx_pos_def( n, nnz, Ap, Ai, Ax,                   &
                                  pos_def )
   import :: MF_DOUBLE
   integer, intent(in) :: n ! size of the square matrix A (sparse)
                            ! [only the upper triangular part]
   integer, intent(in) :: nnz ! nb of non-zero element in A
   integer, intent(in) :: Ap(*), Ai(*)  ! indexes and elements of A
   complex(kind=MF_DOUBLE), intent(in) :: Ax(*) ! (CSC storage with Fortran
                                                !  indexes, i.e. 1-based)
   integer, intent(out) :: pos_def ! 1: A is positive definite
                                   ! 0: otherwise
   ! this routine just tries to make a Choleski factorisation:
   !                    A = L L'
   ! in order to determine the positiveness of A;
   ! be aware that the L factor is not saved.
end subroutine
!_______________________________________________________________________
!
subroutine cholmod_cmplx_llt_prep( n, nnz, Ap, Ai, Ax,                  &
                                   c_addr, LL_addr, S_addr, lnz )
   import :: MF_ADDRESS, MF_DOUBLE
   integer, intent(in) :: n ! size of the square matrix A (sparse)
                            ! [only the upper triangular part]
   integer, intent(in) :: nnz ! nb of non-zero element in A
   integer, intent(in) :: Ap(*), Ai(*)  ! indexes and elements of A
   complex(kind=MF_DOUBLE), intent(in) :: Ax(*) ! (CSC storage with Fortran
                                                !  indexes, i.e. 1-based)
   integer(kind=MF_ADDRESS), intent(out) :: c_addr ! address of the cholmod_common
                                                   ! specific to the factorisation
   integer(kind=MF_ADDRESS), intent(out) :: LL_addr ! address of the cholmod_factor
                                                    ! (special internal format)
   integer(kind=MF_ADDRESS), intent(out) :: S_addr ! address of the sparse L factor
                                                   ! (CSC format, 0-based)
                                                   ! [may be null for the simplicial
                                                   ! case]
   integer, intent(out) :: lnz ! 0: A is not positive definite,
                               !    otherwise contains nnz(L)
   ! this routine tries to make a Choleski factorisation:
   !                    A = L L'
   ! and saves the L factor, both in internal and CSC format.
   ! Indicates also if the matrix A is positive definite,
   ! via the 'lnz' output.
end subroutine
!_______________________________________________________________________
!
subroutine cholmod_cmplx_llt_prep2( n, nnz, Ap, Ai, Ax,                 &
                                    c_addr, LL_addr, status )
   import :: MF_ADDRESS, MF_DOUBLE
   integer, intent(in) :: n ! size of the square matrix A (sparse)
                            ! [only the upper triangular part]
   integer, intent(in) :: nnz ! nb of non-zero element in A
   integer, intent(in) :: Ap(*), Ai(*)  ! indexes and elements of A
   complex(kind=MF_DOUBLE), intent(in) :: Ax(*) ! (CSC storage with Fortran
                                                !  indexes, i.e. 1-based)
   integer(kind=MF_ADDRESS), intent(out) :: c_addr ! address of the cholmod_common
                                                   ! specific to the factorisation
   integer(kind=MF_ADDRESS), intent(out) :: LL_addr ! address of the cholmod_factor
                                                    ! (special internal format)
   integer, intent(out) :: status ! 0: A is positive definite,
                                  ! 1: otherwise
   ! this routine tries to make a Choleski factorisation:
   !                    A = L L'
   ! and saves the L factor, only in internal format.
   ! Indicates also if the matrix A is positive definite,
   ! via the 'status' output.
end subroutine
!_______________________________________________________________________
!
subroutine cholmod_cmplx_llt_symb( n, nnz, Ap, Ai,                      &
                                   c_addr, LL_addr, AA_addr )
   import :: MF_ADDRESS
   integer, intent(in) :: n ! size of the square matrix A (sparse)
                            ! [only the upper triangular part]
   integer, intent(in) :: nnz ! nb of non-zero element in A
   integer, intent(in) :: Ap(*), Ai(*) ! indexes of A (CSC storage
                                       ! with Fortran indexes, i.e. 1-based)
   integer(kind=MF_ADDRESS), intent(out) :: c_addr ! address of the cholmod_common
                                                   ! specific to the factorisation
   integer(kind=MF_ADDRESS), intent(out) :: LL_addr ! address of the cholmod_factor
                                                    ! (special internal format)
   integer(kind=MF_ADDRESS), intent(out) :: AA_addr ! copy of with C-indexes
   ! this routine makes the symbolic part of a Choleski factorisation:
   !                    A = L L'
   ! it also stores AA, the copy of A with C-indexes, i.e. 0-based)
end subroutine
!_______________________________________________________________________
!
subroutine cholmod_cmplx_llt_num( n, nnz, Ax, c_addr, LL_addr, AA_addr, &
                                  status )
   import :: MF_ADDRESS, MF_DOUBLE
   integer, intent(in) :: n ! size of the square matrix A (sparse)
                            ! [only the upper triangular part]
   integer, intent(in) :: nnz ! nb of non-zero element in A
   complex(kind=MF_DOUBLE), intent(in) :: Ax(*) ! numerical entries of A (CSC)
   integer(kind=MF_ADDRESS), intent(in) :: c_addr ! address of the cholmod_common
                                                  ! specific to the factorisation
   integer(kind=MF_ADDRESS), intent(in out) :: LL_addr ! address of the cholmod_factor
                                                       ! (special internal format)
   integer(kind=MF_ADDRESS), intent(in) :: AA_addr ! copy of with C-indexes
   integer, intent(out) :: status ! 0: A is positive definite,
                                  ! 1: otherwise
   ! this routine makes the numerical factorisation:
   !                    A = L L'
   ! (so the L factor is updated)
   ! Indicates also if the matrix A is positive definite,
   ! via the 'status' output.
end subroutine
!_______________________________________________________________________
!
subroutine cholmod_cmplx_get_factor( c_addr, LL_addr, S_addr, n, lnz,   &
                                     Lp, Li, Lx, perm )
   import :: MF_ADDRESS, MF_DOUBLE
   integer(kind=MF_ADDRESS), intent(in) :: c_addr ! address of the cholmod_common
                                                  ! specific to the factorisation
   integer(kind=MF_ADDRESS), intent(in) :: LL_addr ! address of the cholmod_factor
                                                   ! (special internal format)
   integer(kind=MF_ADDRESS), intent(in) :: S_addr ! address of the sparse L factor
                                                  ! (CSC format, 0-based)
   integer, intent(in) :: n ! size of the square matrix A (sparse)
                            ! [only the upper triangular part]
   integer, intent(in) :: lnz !  nnz(L)
   integer, intent(in) :: Lp(*), Li(*)  ! indexes and elements of L
   complex(kind=MF_DOUBLE), intent(in) :: Lx(*) ! (CSC storage with Fortran
                                                !  indexes, i.e. 1-based)
   integer, intent(out) :: perm(*) ! permutation vector
   ! this routine gets the L factor of the Choleski factorisation:
   !                    A = L L'
   ! gets also the permutation vector applied to A before the
   ! factorisation (reducing the fill-in).
end subroutine
!_______________________________________________________________________
!
subroutine cholmod_cmplx_solve( n, nnz, Ap, Ai, Ax, b,                  &
                                x, status )
   import :: MF_DOUBLE
   integer, intent(in) :: n ! size of the square matrix A (sparse)
                            ! [only the upper triangular part]
   integer, intent(in) :: nnz ! nb of non-zero element in A
   integer, intent(in) :: Ap(*), Ai(*)  ! indexes and elements of A
   complex(kind=MF_DOUBLE),  intent(in) :: Ax(*) ! (CSC storage with Fortran
                                                 !  indexes, i.e. 1-based)
   complex(kind=MF_DOUBLE),  intent(in) :: b(*)  ! RHS of the equation
   complex(kind=MF_DOUBLE), intent(out) :: x(*)  ! vector solution
   integer, intent(out) :: status !  0 : ok, A is positive definite,
                                  !  1 : failed
                                  ! -1 : internal failed (out of memory, ...)
   ! this routine solves the following system:
   !                    A x = b
   ! where A is symmetric positive definite.
   !
end subroutine
!_______________________________________________________________________
!
subroutine cholmod_cmplx_solve_factor( n, c_addr, L_addr, b,            &
                                       z )
import :: MF_ADDRESS, MF_DOUBLE
   integer, intent(in) :: n ! size of the square matrix A (sparse)
   integer(kind=MF_ADDRESS), intent(in) :: c_addr ! address of the cholmod_common
                                                  ! specific to the factorisation
   integer(kind=MF_ADDRESS), intent(in) :: L_addr ! address of the cholmod_factor
                                                  ! (special internal format)
   complex(kind=MF_DOUBLE),  intent(in) :: b(*)   ! RHS of the equation
   complex(kind=MF_DOUBLE), intent(out) :: z(*)   ! vector solution
   ! this routine solves the following system:
   !                    L L' z = b
   ! where L is the Choleski factor obtained by another routine.
end subroutine
!_______________________________________________________________________
!
subroutine cholmod_cmplx_solve_factor_nrhs( n, c_addr, L_addr, nrhs, b, &
                                            z )
   import :: MF_ADDRESS, MF_DOUBLE
   integer, intent(in) :: n ! size of the square matrix A (sparse)
   integer(kind=MF_ADDRESS), intent(in) :: c_addr ! address of the cholmod_common
                                                  ! specific to the factorisation
   integer(kind=MF_ADDRESS), intent(in) :: L_addr ! address of the cholmod_factor
                                                  ! (special internal format)
   integer, intent(in) :: nrhs ! number of columns of the RHS
   complex(kind=MF_DOUBLE),  intent(in) :: b(n,*) ! RHS of the equation
   complex(kind=MF_DOUBLE), intent(out) :: z(n,*) ! vector solution
   ! this routine solves the following system:
   !                    L L' z = b
   ! where L is the Choleski factor obtained by another routine.
end subroutine
!_______________________________________________________________________
!
end interface
