/*
 * CHOLMOD: Fortran 90 wrapper
 *
 * Copyright É. Canot 2005-2016 -- IPR/CNRS
 *
 * December 17, 2015
 *
 *   addresses are stored in 32bit or 64bit int, according to the OS used
 *
 *   CAUTION: use only the 'SuiteSparse_long' definition,
 *            which automatically changes according the OS.
 */

/*
 * WARNING: we assume that all SYMMETRIC matrices which are passed
 *          to CHOLMOD contain at least the UPPER part of the matrix
 *          (most of time, the 'stype' arg is 1).
 *          Moreover, they must be row-sorted.
 *
 * In the calls to the 'cholmod_l_allocate_sparse' routine, the args
 * 4 to 7 concerning a sparse matrix A are the following:
 *   sorted   TRUE if columns of A are sorted, FALSE otherwise
 *   packed   TRUE if A will be packed, FALSE otherwise
 *   stype    storage type of A
 *              0: matrix is unsymmetric; use both upper and lower
 *                 triangular parts. May be square or rectangular.
 *             >0: matrix is square and symmetric, use the upper
 *                 triangular part.
 *             <0: matrix is square and symmetric, use the lower
 *                 triangular part.
 *   xtype    CHOLMOD_REAL or CHOLMOD_COMPLEX
 *
 */

#include "cholmod.h"

#define TRUE 1
#define FALSE 0

/* Address type: ADDRESS */
#ifdef _64_BITS
#define ADDRESS long long int
#else
#define ADDRESS int
#endif

/*--------------------------------------------------------------------*/
/*                     R E A L     V A L U E S                        */
/*--------------------------------------------------------------------*/

/*----------------------------------------------------------------------
 *
 * Fortran call :
 *   call cholmod_pos_def( n, nnz, Ap, Ai, Ax,
 *                         pos_def )
 *
 *     pos_def = TRUE if matrix is positive definite
 *               FALSE otherwise.
 *
 *     factor L is not saved;
 *     doesn't need any clean after this call.
 */
#ifdef UNDERSCORE
#define CHOLMOD_POS_DEF cholmod_pos_def_
#else
#define CHOLMOD_POS_DEF cholmod_pos_def
#endif
void CHOLMOD_POS_DEF(
     int* n, int* nnz, int Ap[], int Ai[], double Ax[],
     int* pos_def
)
{
/*
 *  Warning: in Fortran, all indexes are 1-based.
 *  The shift is done in this routine.
 */
   SuiteSparse_long i;

   cholmod_sparse *AA;
   SuiteSparse_long *AAp, *AAi;
   double *AAx;
   cholmod_factor *L;
   cholmod_common c;

   cholmod_l_start ( &c );

   AA = cholmod_l_allocate_sparse( (SuiteSparse_long)*n, (SuiteSparse_long)*n, (SuiteSparse_long)*nnz,
                                   TRUE, TRUE, 1, CHOLMOD_REAL,
                                   &c );

   AAp = AA->p;
   for( i=0; i<*n+1; i++ ){
      AAp[i] = Ap[i] - 1; /* shift */
   }

   AAi = AA->i;
   AAx = AA->x;
   for( i=0; i<*nnz; i++ ){
      AAi[i] = Ai[i] - 1; /* shift */
      AAx[i] = Ax[i];
   }

   L = cholmod_l_analyze( AA, &c );

   c.quick_return_if_not_posdef = TRUE;  /* for supernodal only */
   c.final_ll = TRUE;                    /* L*L' decomposition */
   c.print = 0;                          /* QUIET mode */
   cholmod_l_factorize( AA, L, &c );

   if( L->minor == *n ){
      *pos_def = TRUE;
   } else {
      *pos_def = FALSE;
   }

   cholmod_l_free_factor( &L, &c );        /* free matrices */
   cholmod_l_free_sparse( &AA, &c );
   cholmod_l_finish( &c );                 /* finish CHOLMOD */
}

/*----------------------------------------------------------------------
 *
 * Fortran call :
 *   call cholmod_llt_prep( n, nnz, Ap, Ai, Ax,
 *                          c_addr, LL_addr, S_addr, lnz )
 *
 * lnz may be seen as a return status:
 *     lnz > 0 : ok, the matrix is positive definite
 *         = 0 : failed (not positive definite)
 */
#ifdef UNDERSCORE
#define CHOLMOD_LLT_PREP cholmod_llt_prep_
#else
#define CHOLMOD_LLT_PREP cholmod_llt_prep
#endif
void CHOLMOD_LLT_PREP(
     int* n, int* nnz, int Ap[], int Ai[], double Ax[],
     ADDRESS* c_addr, ADDRESS* LL_addr, ADDRESS* S_addr, int* lnz
)
{
/*
 *  Warning: in Fortran, all indexes are 1-based.
 *  The shift is done in this routine.
 */
   SuiteSparse_long i;

   cholmod_sparse *AA, *S;
   SuiteSparse_long *AAp, *AAi;
   double *AAx;

   cholmod_factor *LL;
   SuiteSparse_long *LLp;

   cholmod_common *c;

   *c_addr = (ADDRESS) malloc( sizeof(cholmod_common) );
   c = (cholmod_common*)*c_addr;
   cholmod_l_start ( c );

   AA = cholmod_l_allocate_sparse( (SuiteSparse_long)*n, (SuiteSparse_long)*n, (SuiteSparse_long)*nnz,
                                   TRUE, TRUE, 1, CHOLMOD_REAL,
                                   c );

   AAp = AA->p;
   for( i=0; i<*n+1; i++ ){
      AAp[i] = Ap[i] - 1; /* shift */
   }

   AAi = AA->i;
   AAx = AA->x;
   for( i=0; i<*nnz; i++ ){
      AAi[i] = Ai[i] - 1; /* shift */
      AAx[i] = Ax[i];
   }

   *LL_addr = (ADDRESS) cholmod_l_analyze( AA, c );
   LL = (cholmod_factor*)*LL_addr;

   c->quick_return_if_not_posdef = TRUE;  /* for supernodal only */
   c->final_ll = TRUE;                    /* L*L' decomposition */
   c->print = 0;                          /* QUIET mode */

   cholmod_l_factorize( AA, LL, c );

   if( LL->minor == *n ){
      /* getting number of non-zeros */
      if( LL->is_super ){ /* supernodal */
         S = cholmod_l_factor_to_sparse( LL, c );
         *S_addr = (ADDRESS) S;
         LLp = S->p;
      } else {           /* simplicial */
         *S_addr = 0;
         LLp = LL->p;
      }
      *lnz = LLp[*n];
   } else {
      *S_addr = 0;
      cholmod_l_free_sparse( &AA, c );
      cholmod_l_free_factor( &LL, c );
      *LL_addr = 0;
      cholmod_l_finish( c );
      free( c );
      *lnz = 0;
   }

   if( AA != 0 ){
      cholmod_l_free_sparse( &AA, c );
   }
}

/*----------------------------------------------------------------------
 *
 * Fortran call :
 *   call cholmod_llt_prep2( n, nnz, Ap, Ai, Ax,
 *                           c_addr, LL_addr, status )
 *
 *     status = 0 if matrix is positive definite
 *              1 otherwise.
 */
#ifdef UNDERSCORE
#define CHOLMOD_LLT_PREP2 cholmod_llt_prep2_
#else
#define CHOLMOD_LLT_PREP2 cholmod_llt_prep2
#endif
void CHOLMOD_LLT_PREP2(
     int* n, int* nnz, int Ap[], int Ai[], double Ax[],
     ADDRESS* c_addr, ADDRESS* LL_addr, int* status
)
{
/*
 *  Warning: in Fortran, all indexes are 1-based.
 *  The shift is done in this routine.
 */
   SuiteSparse_long i;

   cholmod_sparse *AA;
   SuiteSparse_long *AAp, *AAi;
   double *AAx;

   cholmod_factor *LL;

   cholmod_common *c;

   *c_addr = (ADDRESS) malloc( sizeof(cholmod_common) );
   c = (cholmod_common*)*c_addr;
   cholmod_l_start ( c );

   AA = cholmod_l_allocate_sparse( (SuiteSparse_long)*n, (SuiteSparse_long)*n, (SuiteSparse_long)*nnz,
                                   TRUE, TRUE, 1, CHOLMOD_REAL,
                                   c );

   AAp = AA->p;
   for( i=0; i<*n+1; i++ ){
      AAp[i] = Ap[i] - 1; /* shift */
   }

   AAi = AA->i;
   AAx = AA->x;
   for( i=0; i<*nnz; i++ ){
      AAi[i] = Ai[i] - 1; /* shift */
      AAx[i] = Ax[i];
   }

   *LL_addr = (ADDRESS) cholmod_l_analyze( AA, c );
   LL = (cholmod_factor*)*LL_addr;

   c->quick_return_if_not_posdef = TRUE;  /* for supernodal only */
   c->final_ll = TRUE;                    /* L*L' decomposition */
   c->print = 0;                          /* QUIET mode */

   cholmod_l_factorize( AA, LL, c );

   if( LL->minor == *n ){
      *status = 0;   /* ok */
   } else {
      cholmod_l_free_sparse( &AA, c );
      cholmod_l_free_factor( &LL, c );
      *LL_addr = 0;
      cholmod_l_finish( c );
      free( c );
      *status = 1;   /* matrix is not symm. pos. def. */
   }

   if( AA != 0 ){
      cholmod_l_free_sparse( &AA, c );
   }
}

/*----------------------------------------------------------------------
 *
 * Fortran call :
 *   call cholmod_llt_symb( n, nnz, Ap, Ai,
 *                          c_addr, LL_addr, AA_addr )
 *
 * makes only the symbolic factorisation;
 * returns also the copy of A in 0-based indexes.
 *
 */
#ifdef UNDERSCORE
#define CHOLMOD_LLT_SYMB cholmod_llt_symb_
#else
#define CHOLMOD_LLT_SYMB cholmod_llt_symb
#endif
void CHOLMOD_LLT_SYMB(
     int* n, int* nnz, int Ap[], int Ai[],
     ADDRESS* c_addr, ADDRESS* LL_addr, ADDRESS* AA_addr
)
{
/*
 *  Warning: in Fortran, all indexes are 1-based.
 *  The shift is done in this routine.
 */
   SuiteSparse_long i;

   cholmod_sparse *AA;
   SuiteSparse_long *AAp, *AAi;

   cholmod_factor *LL;

   cholmod_common *c;

   *c_addr = (ADDRESS) malloc( sizeof(cholmod_common) );
   c = (cholmod_common*)*c_addr;
   cholmod_l_start ( c );

   AA = cholmod_l_allocate_sparse( (SuiteSparse_long)*n, (SuiteSparse_long)*n, (SuiteSparse_long)*nnz,
                                   TRUE, TRUE, 1, CHOLMOD_REAL,
                                   c );

   AAp = AA->p;
   for( i=0; i<*n+1; i++ ){
      AAp[i] = Ap[i] - 1; /* shift */
   }

   AAi = AA->i;
   for( i=0; i<*nnz; i++ ){
      AAi[i] = Ai[i] - 1; /* shift */
   }

   *LL_addr = (ADDRESS) cholmod_l_analyze( AA, c );
   LL = (cholmod_factor*)*LL_addr;

   *AA_addr = (ADDRESS) AA;
}

/*----------------------------------------------------------------------
 *
 * Fortran call :
 *   call cholmod_llt_num( n, nnz, Ax, c_addr, LL_addr, AA_addr,
 *                         status )
 *
 *     status = 0 if matrix is positive definite
 *              1 otherwise.
 */
#ifdef UNDERSCORE
#define CHOLMOD_LLT_NUM cholmod_llt_num_
#else
#define CHOLMOD_LLT_NUM cholmod_llt_num
#endif
void CHOLMOD_LLT_NUM(
     int* n, int* nnz, double Ax[], ADDRESS* c_addr, ADDRESS* LL_addr,
     ADDRESS* AA_addr,
     int* status
)
{
/*
 *  Warning: in Fortran, all indexes are 1-based.
 *  The shift is done in this routine.
 */
   SuiteSparse_long i;

   cholmod_sparse *AA;
   SuiteSparse_long *AAp, *AAi;
   double *AAx;

   cholmod_factor *LL;

   cholmod_common *c;

   c = (cholmod_common *)*c_addr;

   AA = (cholmod_sparse*)*AA_addr;

   /* update numerical entries of A */
   AAx = AA->x;
   for( i=0; i<*nnz; i++ ){
      AAx[i] = Ax[i];
   }

   LL = (cholmod_factor*)*LL_addr;

   c->quick_return_if_not_posdef = TRUE;  /* for supernodal only */
   c->final_ll = TRUE;                    /* L*L' decomposition */
   c->print = 0;                          /* QUIET mode */

   cholmod_l_factorize( AA, LL, c );

   if( LL->minor == *n ){
      *status = 0;   /* ok */
   } else {
      *status = 1;   /* matrix is not symm. pos. def. */
   }

}

/*----------------------------------------------------------------------
 *
 * Fortran call :
 *   call cholmod_ldlt_prep( n, nnz, Ap, Ai, Ax,
 *                           c_addr, LL_addr, S_addr, lnz )
 *
 * LDLT decomposition may fail.
 * lnz may be seen as a return status:
 *      lnz > 0 : ok
 *          = 0 : failed
 */
#ifdef UNDERSCORE
#define CHOLMOD_LDLT_PREP cholmod_ldlt_prep_
#else
#define CHOLMOD_LDLT_PREP cholmod_ldlt_prep
#endif
void CHOLMOD_LDLT_PREP(
     int* n, int* nnz, int Ap[], int Ai[], double Ax[],
     ADDRESS* c_addr, ADDRESS* LL_addr, ADDRESS* S_addr, int* lnz
)
{
/*
 *  Warning: in Fortran, all indexes are 1-based.
 *  The shift is done in this routine.
 */
   SuiteSparse_long i;

   cholmod_sparse *AA, *S;
   SuiteSparse_long *AAp, *AAi;
   double *AAx;

   cholmod_factor *LL;
   SuiteSparse_long *LLp;

   cholmod_common *c;

   *c_addr = (ADDRESS) malloc( sizeof(cholmod_common) );
   c = (cholmod_common*)*c_addr;
   cholmod_l_start ( c );

   AA = cholmod_l_allocate_sparse( (SuiteSparse_long)*n, (SuiteSparse_long)*n, (SuiteSparse_long)*nnz,
                                   TRUE, TRUE, 1, CHOLMOD_REAL,
                                   c );

   AAp = AA->p;
   for( i=0; i<*n+1; i++ ){
      AAp[i] = Ap[i] - 1; /* shift */
   }

   AAi = AA->i;
   AAx = AA->x;
   for( i=0; i<*nnz; i++ ){
      AAi[i] = Ai[i] - 1; /* shift */
      AAx[i] = Ax[i];
   }

   *LL_addr = (ADDRESS) cholmod_l_analyze( AA, c );
   LL = (cholmod_factor*)*LL_addr;

   c->final_ll = FALSE;                   /* L*D*L' decomposition */
   c->print = 0;                          /* QUIET mode */

   cholmod_l_factorize( AA, LL, c );

   cholmod_l_free_sparse( &AA, c );

   if( LL->minor == *n ){
      /* getting number of non-zeros */
      if( LL->is_super ){ /* supernodal */
         S = cholmod_l_factor_to_sparse( LL, c );
         *S_addr = (ADDRESS) S;
         LLp = S->p;
      } else {           /* simplicial */
         *S_addr = 0;
         LLp = LL->p;
      }
      *lnz = LLp[*n];
   } else {
      *S_addr = 0;
      *LL_addr = 0;
      cholmod_l_free_factor( &LL, c );
      cholmod_l_finish( c );
      free( c );
      *lnz = 0;
   }

}

/*----------------------------------------------------------------------
 *
 * Fortran call :
 *   call cholmod_get_factor( c_addr, LL_addr, S_addr, n, lnz,
 *                            Lp, Li, Lx, perm )
 */
#ifdef UNDERSCORE
#define CHOLMOD_GET_FACTOR cholmod_get_factor_
#else
#define CHOLMOD_GET_FACTOR cholmod_get_factor
#endif
void CHOLMOD_GET_FACTOR(
     ADDRESS* c_addr, ADDRESS* LL_addr, ADDRESS* S_addr, int* n, int* lnz,
     int Lp[], int Li[], double Lx[], int perm[]
)
{
/*
 *  Warning: in Fortran, all indexes are 1-based.
 *  The shift is done in this routine.
 */
   SuiteSparse_long i;

   cholmod_sparse *S;
   cholmod_common *c;
   cholmod_factor *LL;
   SuiteSparse_long *LLp, *LLi, *LLperm;
   double *LLx;

   c = (cholmod_common *)*c_addr;
   LL = (cholmod_factor *)*LL_addr;
   S = (cholmod_sparse *)*S_addr;

   if( S != 0 ){ /* supernodal */
      LLp = S->p;
      LLi = S->i;
      LLx = S->x;
   } else {      /* simplicial */
      LLp = LL->p;
      LLi = LL->i;
      LLx = LL->x;
   }
   for( i=0; i<*n+1; i++ ){
      Lp[i] = LLp[i] + 1; /* shift */
   }
   for( i=0; i<*lnz; i++ ){
      Li[i] = LLi[i] + 1; /* shift */
      Lx[i] = LLx[i];
   }
   LLperm = LL->Perm;
   for( i=0; i<*n; i++ ){
      perm[i] = LLperm[i] + 1; /* shift */
   }

   cholmod_l_free_factor( &LL, c );
   *LL_addr = 0;
   if( S != 0 ){
      cholmod_l_free_sparse( &S, c );
      *S_addr = 0;
   }
   cholmod_l_finish( c );
   free( c );

}

/*----------------------------------------------------------------------
 *
 * Fortran call :
 *   call cholmod_get_diag( c_addr, LL_addr, S_addr, n, lnz,
 *                          diag )
 */
#ifdef UNDERSCORE
#define CHOLMOD_GET_DIAG cholmod_get_diag_
#else
#define CHOLMOD_GET_DIAG cholmod_get_diag
#endif
void CHOLMOD_GET_DIAG(
     ADDRESS* c_addr, ADDRESS* LL_addr, ADDRESS* S_addr, int* n, int* lnz,
     double diag[]
)
{

   SuiteSparse_long j, k;

   cholmod_sparse *S;
   cholmod_common *c;
   cholmod_factor *LL;
   SuiteSparse_long *LLp, *LLi;
   double *LLx;

   c = (cholmod_common *)*c_addr;
   LL = (cholmod_factor *)*LL_addr;
   S = (cholmod_sparse *)*S_addr;

   if( S != 0 ){ /* supernodal */
      LLp = S->p;
      LLi = S->i;
      LLx = S->x;
   } else {      /* simplicial */
      LLp = LL->p;
      LLi = LL->i;
      LLx = LL->x;
   }
   for( j=0; j<*n; j++ ){
      for( k=LLp[j]; k<LLp[j+1]; k++ ){
         if( j==LLi[k] ) diag[j] = LLx[k];
      }
   }

   cholmod_l_free_factor( &LL, c );
   *LL_addr = 0;
   if( S != 0 ){
      cholmod_l_free_sparse( &S, c );
      *S_addr = 0;
   }
   cholmod_l_finish( c );
   free( c );

}

/*----------------------------------------------------------------------
 *
 * Fortran call :
 *   call cholmod_solve_f90( n, nnz, Ap, Ai, Ax, b,
 *                           x, status )
 *
 *    status =  0 : ok
 *           =  1 : failed
 *           = -1 : internal fail (out of memory, ...)
 *
 * The failure can be detected just after the factorization.
 */
#ifdef UNDERSCORE
#define CHOLMOD_SOLVE cholmod_solve_f90_
#else
#define CHOLMOD_SOLVE cholmod_solve_f90
#endif
void CHOLMOD_SOLVE(
     int* n, int* nnz, int Ap[], int Ai[], double Ax[], double b[],
     double x[], int* status
)
{
/*
 *  Warning: in Fortran, all indexes are 1-based.
 *  The shift is done in this routine.
 */
   SuiteSparse_long i;
   cholmod_sparse *AA;
   SuiteSparse_long *AAp, *AAi;
   double *AAx;
   cholmod_factor *L;
   cholmod_common c;
   cholmod_dense *bb, *xx;
   double *array;

   cholmod_l_start ( &c );

/* choice for ordering: see Cholmod Documentation, § 11.1 and 11.2 */
c.nmethods = 1;                        /* only one method is tested */
c.method[0].ordering = CHOLMOD_AMD;    /* it is AMD */
c.postorder = FALSE;                   /* without postorder */

   AA = cholmod_l_allocate_sparse( (SuiteSparse_long)*n, (SuiteSparse_long)*n, (SuiteSparse_long)*nnz,
                                   TRUE, TRUE, 1, CHOLMOD_REAL,
                                   &c );

   AAp = AA->p;
   for( i=0; i<*n+1; i++ ){
      AAp[i] = Ap[i] - 1; /* shift */
   }

   AAi = AA->i;
   AAx = AA->x;
   for( i=0; i<*nnz; i++ ){
      AAi[i] = Ai[i] - 1; /* shift */
      AAx[i] = Ax[i];
   }

   L = cholmod_l_analyze( AA, &c );
   c.final_ll = TRUE;                    /* L*L' decomposition */
   /* default: print some info only when something fails... */
   c.print = 0;                          /* QUIET mode */
   cholmod_l_factorize( AA, L, &c );

   cholmod_l_free_sparse( &AA, &c );

   if( L->minor == *n ){
      *status = 0;
      bb = (cholmod_dense *) malloc( sizeof(cholmod_dense) );
      bb->nrow = *n;
      bb->ncol = 1;
      bb->nzmax = *n;
      bb->d = *n;
      bb->x = b;
      bb->xtype = CHOLMOD_REAL;
      bb->dtype = CHOLMOD_DOUBLE;

      xx = cholmod_l_solve( CHOLMOD_A, L, bb, &c );
      if( xx == NULL ){
        printf("(MUESLI cholmod_l_solve:) CHOLMOD ERROR: failed!\n");
        *status = -1;
      } else {
        array = xx->x;
        for( i=0; i<*n; i++ ){
          x[i] = array[i];
        }
      }

      bb->x = NULL;
      cholmod_l_free_dense( &bb, &c );
      cholmod_l_free_dense( &xx, &c );
   } else {
      *status = 1;
   }
   cholmod_l_free_factor( &L, &c );
   cholmod_l_finish( &c );

}

/*----------------------------------------------------------------------
 *
 * Fortran call :
 *   call cholmod_free_factor_f90( c_addr, LL_addr )
 */
#ifdef UNDERSCORE
#define CHOLMOD_FREE_FACTOR cholmod_free_factor_f90_
#else
#define CHOLMOD_FREE_FACTOR cholmod_free_factor_f90
#endif
void CHOLMOD_FREE_FACTOR(
     ADDRESS* c_addr, ADDRESS* L_addr
)
{
   cholmod_common *c;
   cholmod_factor *L;

   c = (cholmod_common *)*c_addr;
   L = (cholmod_factor *)*L_addr;

   if( L != 0 ){
      cholmod_l_free_factor( &L, c );
      *L_addr = 0;
   }
   if( c != 0 ){
      cholmod_l_finish( c );
      free( c );
      *c_addr = 0;
   }

}

/*----------------------------------------------------------------------
 *
 * Fortran call :
 *   call cholmod_free_factor_2( c_addr, LL_addr, S_addr )
 */
#ifdef UNDERSCORE
#define CHOLMOD_FREE_FACTOR_2 cholmod_free_factor_2_
#else
#define CHOLMOD_FREE_FACTOR_2 cholmod_free_factor_2
#endif
void CHOLMOD_FREE_FACTOR_2(
     ADDRESS* c_addr, ADDRESS* L_addr, ADDRESS* S_addr
)
{
   cholmod_common *c;
   cholmod_factor *L;
   cholmod_sparse *S;

   c = (cholmod_common *)*c_addr;
   L = (cholmod_factor *)*L_addr;
   S = (cholmod_sparse *)*S_addr;

   if( L != 0 ){
      cholmod_l_free_factor( &L, c );
      *L_addr = 0;
   }
   if( S != 0 ){
      cholmod_l_free_sparse( &S, c );
      *S_addr = 0;
   }
   if( c != 0 ){
      cholmod_l_finish( c );
      free( c );
      *c_addr = 0;
   }

}

/*----------------------------------------------------------------------
 *
 * Fortran call :
 *   call cholmod_solve_factor( n, c_addr, L_addr, b,
 *                              x )
 *
 */
#ifdef UNDERSCORE
#define CHOLMOD_SOLVE_FACTOR cholmod_solve_factor_
#else
#define CHOLMOD_SOLVE_FACTOR cholmod_solve_factor
#endif
void CHOLMOD_SOLVE_FACTOR(
     int* n, ADDRESS* c_addr, ADDRESS* L_addr, double b[],
     double x[]
)
{
   SuiteSparse_long i;
   cholmod_common *c;
   cholmod_factor *L;
   cholmod_dense *bb, *xx;
   double *array;

   c = (cholmod_common *)*c_addr;
   L = (cholmod_factor *)*L_addr;

   bb = (cholmod_dense *) malloc( sizeof(cholmod_dense) );
   bb->nrow = *n;
   bb->ncol = 1;
   bb->nzmax = *n;
   bb->d = *n;
   bb->x = b;
   bb->xtype = CHOLMOD_REAL;
   bb->dtype = CHOLMOD_DOUBLE;

   xx = cholmod_l_solve( CHOLMOD_A, L, bb, c );

   array = xx->x;
   for( i=0; i<*n; i++ ){
      x[i] = array[i];
   }

   bb->x = NULL;
   cholmod_l_free_dense( &bb, c );
   cholmod_l_free_dense( &xx, c );

}

/*----------------------------------------------------------------------
 *
 * Fortran call :
 *   call cholmod_solve_factor_nrhs( n, c_addr, L_addr, nrhs, b,
 *                                   x )
 *
 */
#ifdef UNDERSCORE
#define CHOLMOD_SOLVE_FACTOR_NRHS cholmod_solve_factor_nrhs_
#else
#define CHOLMOD_SOLVE_FACTOR_NRHS cholmod_solve_factor_nrhs
#endif
void CHOLMOD_SOLVE_FACTOR_NRHS(
     int* n, ADDRESS* c_addr, ADDRESS* L_addr, int* nrhs, double b[],
     double x[]
)
{
   SuiteSparse_long i;
   cholmod_common *c;
   cholmod_factor *L;
   cholmod_dense *bb, *xx;
   double *array;

   c = (cholmod_common *)*c_addr;
   L = (cholmod_factor *)*L_addr;

   bb = (cholmod_dense *) malloc( sizeof(cholmod_dense) );
   bb->nrow = *n;
   bb->ncol = *nrhs;
   bb->nzmax = (*n)*(*nrhs);
   bb->d = *n;
   bb->x = b;
   bb->xtype = CHOLMOD_REAL;
   bb->dtype = CHOLMOD_DOUBLE;

   xx = cholmod_l_solve( CHOLMOD_A, L, bb, c );

   array = xx->x;
   for( i=0; i<(*n)*(*nrhs); i++ ){
      x[i] = array[i];
   }

   bb->x = NULL;
   cholmod_l_free_dense( &bb, c );
   cholmod_l_free_dense( &xx, c );

}

/*----------------------------------------------------------------------
 *
 * Fortran call :
 *   call cholmod_spsol_fact_nRHS_prep( n, c_addr, L_addr, nrhs,
 *                                      bnz, bp, bi, bx,
 *                                      x_addr, xnz )
 *
 */
#ifdef UNDERSCORE
#define CHOLMOD_SPSOL_FACT_NRHS_PREP cholmod_spsol_fact_nrhs_prep_
#else
#define CHOLMOD_SPSOL_FACT_NRHS_PREP cholmod_spsol_fact_nrhs_prep
#endif
void CHOLMOD_SPSOL_FACT_NRHS_PREP(
     int* n, ADDRESS* c_addr, ADDRESS* L_addr, int* nrhs,
     int* bnz, int bp[], int bi[], double bx[],
     ADDRESS* x_addr, int* xnz
)
{
/*
 *  Warning: in Fortran, all indexes are 1-based.
 *  The shift is done in this routine.
 */
   SuiteSparse_long i;
   cholmod_common *c;
   cholmod_factor *L;
   cholmod_sparse *bb, *xx;
   SuiteSparse_long *bbp, *bbi;
   double *bbx;

   c = (cholmod_common *)*c_addr;
   L = (cholmod_factor *)*L_addr;

   bb = cholmod_l_allocate_sparse( (SuiteSparse_long)*n, (SuiteSparse_long)*nrhs, (SuiteSparse_long)*bnz,
                                   TRUE, TRUE, 0, CHOLMOD_REAL,
                                   c );

   bbp = bb->p;
   for( i=0; i<*nrhs+1; i++ ){
      bbp[i] = bp[i] - 1; /* shift */
   }

   bbi = bb->i;
   bbx = bb->x;
   for( i=0; i<*bnz; i++ ){
      bbi[i] = bi[i] - 1; /* shift */
      bbx[i] = bx[i];
   }

   xx = cholmod_l_spsolve( CHOLMOD_A, L, bb, c );

   *x_addr = (ADDRESS) xx;

   *xnz = xx->nzmax;
   cholmod_l_free_sparse( &bb, c );

}

/*----------------------------------------------------------------------
 *
 * Fortran call :
 *   call cholmod_get_sol_nrhs( c_addr, S_addr, m, n, snz,
 *                              Sp, Si, Sx )
 */
#ifdef UNDERSCORE
#define CHOLMOD_GET_SOL_NRHS cholmod_get_sol_nrhs_
#else
#define CHOLMOD_GET_SOL_NRHS cholmod_get_sol_nrhs
#endif
void CHOLMOD_GET_SOL_NRHS(
     ADDRESS* c_addr, ADDRESS* S_addr, int* m, int* n, int* snz,
     int Sp[], int Si[], double Sx[]
)
{
/*
 *  Warning: in Fortran, all indexes are 1-based.
 *  The shift is done in this routine.
 */
   SuiteSparse_long i;
   SuiteSparse_long *SSp, *SSi;
   double *SSx;

   cholmod_sparse *S;
   cholmod_common *c;

   c = (cholmod_common *)*c_addr;
   S = (cholmod_sparse *)*S_addr;

   SSp = S->p;
   SSi = S->i;
   SSx = S->x;
   for( i=0; i<*n+1; i++ ){
      Sp[i] = SSp[i] + 1; /* shift */
   }
   for( i=0; i<*snz; i++ ){
      Si[i] = SSi[i] + 1; /* shift */
      Sx[i] = SSx[i];
   }
   cholmod_l_free_sparse( &S, c );

}

/*--------------------------------------------------------------------*/
/*                  C O M P L E X     V A L U E S                     */
/*--------------------------------------------------------------------*/

/* ===========================================================
 *  WARNING: in CHOLMOD 1.0, supernodal representation
 *  is not possible with ZOMPLEX storage mode; only
 *  with COMPLEX storage mode (BLAS and LAPACK compatibility)
 * ===========================================================
 */

/*----------------------------------------------------------------------
 *
 * Fortran call :
 *   call cholmod_cmplx_pos_def( n, nnz, Ap, Ai, Ax,
 *                               pos_def )
 *
 *     pos_def = TRUE if matrix is positive definite
 *               FALSE otherwise.
 */
#ifdef UNDERSCORE
#define CHOLMOD_CMPLX_POS_DEF cholmod_cmplx_pos_def_
#else
#define CHOLMOD_CMPLX_POS_DEF cholmod_cmplx_pos_def
#endif
void CHOLMOD_CMPLX_POS_DEF(
     int* n, int* nnz, int Ap[], int Ai[], double Ax[],
     int* pos_def
)
{
/*
 *  Warning: in Fortran, all indexes are 1-based.
 *  The shift is done in this routine.
 */
   SuiteSparse_long i;

   cholmod_sparse *AA;
   SuiteSparse_long *AAp, *AAi;
   double *AAx;

   cholmod_factor *L;

   cholmod_common c;

   cholmod_l_start ( &c );

   AA = cholmod_l_allocate_sparse( (SuiteSparse_long)*n, (SuiteSparse_long)*n, (SuiteSparse_long)*nnz,
                                   TRUE, TRUE, 1, CHOLMOD_COMPLEX,
                                   &c );

   AAp = AA->p;
   for( i=0; i<*n+1; i++ ){
      AAp[i] = Ap[i] - 1; /* shift */
   }

   AAi = AA->i;
   AAx = AA->x;
   for( i=0; i<*nnz; i++ ){
      AAi[i] = Ai[i] - 1; /* shift */
   }
   for( i=0; i<2*(*nnz); i++ ){
      AAx[i] = Ax[i];
   }

   L = cholmod_l_analyze( AA, &c );

   c.quick_return_if_not_posdef = TRUE;  /* for supernodal only */
   c.final_ll = TRUE;                    /* L*L' decomposition */
   c.print = 0;                          /* QUIET mode */
   cholmod_l_factorize( AA, L, &c );

   if( L->minor == *n ){
      *pos_def = TRUE;
   } else {
      *pos_def = FALSE;
   }

   cholmod_l_free_factor( &L, &c );        /* free matrices */
   cholmod_l_free_sparse( &AA, &c );
   cholmod_l_finish( &c );                 /* finish CHOLMOD */
}

/*----------------------------------------------------------------------
 *
 * Fortran call :
 *   call cholmod_cmplx_llt_prep( n, nnz, Ap, Ai, Ax,
 *                                c_addr, LL_addr, S_addr, lnz )
 */
#ifdef UNDERSCORE
#define CHOLMOD_CMPLX_LLT_PREP cholmod_cmplx_llt_prep_
#else
#define CHOLMOD_CMPLX_LLT_PREP cholmod_cmplx_llt_prep
#endif
void CHOLMOD_CMPLX_LLT_PREP(
     int* n, int* nnz, int Ap[], int Ai[], double Ax[],
     ADDRESS* c_addr, ADDRESS* LL_addr, ADDRESS* S_addr, int* lnz
)
{
/*
 *  Warning: in Fortran, all indexes are 1-based.
 *  The shift is done in this routine.
 */
   SuiteSparse_long i;

   cholmod_sparse *AA, *S;
   SuiteSparse_long *AAp, *AAi;
   double *AAx;

   cholmod_factor *LL;
   SuiteSparse_long *LLp;

   cholmod_common *c;

   *c_addr = (ADDRESS) malloc( sizeof(cholmod_common) );
   c = (cholmod_common*)*c_addr;
   cholmod_l_start ( c );

   AA = cholmod_l_allocate_sparse( (SuiteSparse_long)*n, (SuiteSparse_long)*n, (SuiteSparse_long)*nnz,
                                   TRUE, TRUE, 1, CHOLMOD_COMPLEX,
                                   c );

   AAp = AA->p;
   for( i=0; i<*n+1; i++ ){
      AAp[i] = Ap[i] - 1; /* shift */
   }

   AAi = AA->i;
   AAx = AA->x;
   for( i=0; i<*nnz; i++ ){
      AAi[i] = Ai[i] - 1; /* shift */
   }
   for( i=0; i<2*(*nnz); i++ ){
      AAx[i] = Ax[i];
   }

   *LL_addr = (ADDRESS) cholmod_l_analyze( AA, c );
   LL = (cholmod_factor*)*LL_addr;

   c->quick_return_if_not_posdef = TRUE;  /* for supernodal only */
   c->final_ll = TRUE;                    /* L*L' decomposition */
   c->print = 0;                          /* QUIET mode */
   cholmod_l_factorize( AA, LL, c );

   cholmod_l_free_sparse( &AA, c );

   if( LL->minor == *n ){
      /* getting number of non-zeros */
      if( LL->is_super ){
         S = cholmod_l_factor_to_sparse( LL, c );
         *S_addr = (ADDRESS) S;
         LLp = S->p;
      } else {
         *S_addr = 0;
         LLp = LL->p;
      }
      *lnz = LLp[*n];
   } else {
      *S_addr = 0;
      cholmod_l_free_factor( &LL, c );
      *LL_addr = 0;
      cholmod_l_finish( c );
      free( c );
      *lnz = 0;
   }

}

/*----------------------------------------------------------------------
 *
 * Fortran call :
 *   call cholmod_cmplx_llt_prep2( n, nnz, Ap, Ai, Ax,
 *                                 c_addr, LL_addr, status )
 */
#ifdef UNDERSCORE
#define CHOLMOD_CMPLX_LLT_PREP2 cholmod_cmplx_llt_prep2_
#else
#define CHOLMOD_CMPLX_LLT_PREP2 cholmod_cmplx_llt_prep2
#endif
void CHOLMOD_CMPLX_LLT_PREP2(
     int* n, int* nnz, int Ap[], int Ai[], double Ax[],
     ADDRESS* c_addr, ADDRESS* LL_addr, int* status
)
{
/*
 *  Warning: in Fortran, all indexes are 1-based.
 *  The shift is done in this routine.
 */
   SuiteSparse_long i;

   cholmod_sparse *AA, *S;
   SuiteSparse_long *AAp, *AAi;
   double *AAx;

   cholmod_factor *LL;

   cholmod_common *c;

   *c_addr = (ADDRESS) malloc( sizeof(cholmod_common) );
   c = (cholmod_common*)*c_addr;
   cholmod_l_start ( c );

   AA = cholmod_l_allocate_sparse( (SuiteSparse_long)*n, (SuiteSparse_long)*n, (SuiteSparse_long)*nnz,
                                   TRUE, TRUE, 1, CHOLMOD_COMPLEX,
                                   c );

   AAp = AA->p;
   for( i=0; i<*n+1; i++ ){
      AAp[i] = Ap[i] - 1; /* shift */
   }

   AAi = AA->i;
   AAx = AA->x;
   for( i=0; i<*nnz; i++ ){
      AAi[i] = Ai[i] - 1; /* shift */
   }
   for( i=0; i<2*(*nnz); i++ ){
      AAx[i] = Ax[i];
   }

   *LL_addr = (ADDRESS) cholmod_l_analyze( AA, c );
   LL = (cholmod_factor*)*LL_addr;

   c->quick_return_if_not_posdef = TRUE;  /* for supernodal only */
   c->final_ll = TRUE;                    /* L*L' decomposition */
   c->print = 0;                          /* QUIET mode */
   cholmod_l_factorize( AA, LL, c );

   cholmod_l_free_sparse( &AA, c );

   if( LL->minor == *n ){
      *status = 0;   /* ok */
   } else {
      cholmod_l_free_factor( &LL, c );
      *LL_addr = 0;
      cholmod_l_finish( c );
      free( c );
      *status = 1;   /* matrix is not symm. pos. def. */
   }

}

/*----------------------------------------------------------------------
 *
 * Fortran call :
 *   call cholmod_cmplx_llt_symb( n, nnz, Ap, Ai,
 *                                c_addr, LL_addr, AA_addr )
 *
 * makes only the symbolic factorisation;
 * returns also the copy of A in 0-based indexes.
 *
 */
#ifdef UNDERSCORE
#define CHOLMOD_CMPLX_LLT_SYMB cholmod_cmplx_llt_symb_
#else
#define CHOLMOD_CMPLX_LLT_SYMB cholmod_cmplx_llt_symb
#endif
void CHOLMOD_CMPLX_LLT_SYMB(
     int* n, int* nnz, int Ap[], int Ai[],
     ADDRESS* c_addr, ADDRESS* LL_addr, ADDRESS* AA_addr
)
{
/*
 *  Warning: in Fortran, all indexes are 1-based.
 *  The shift is done in this routine.
 */
   SuiteSparse_long i;

   cholmod_sparse *AA;
   SuiteSparse_long *AAp, *AAi;

   cholmod_factor *LL;

   cholmod_common *c;

   *c_addr = (ADDRESS) malloc( sizeof(cholmod_common) );
   c = (cholmod_common*)*c_addr;
   cholmod_l_start ( c );

   AA = cholmod_l_allocate_sparse( (SuiteSparse_long)*n, (SuiteSparse_long)*n, (SuiteSparse_long)*nnz,
                                   TRUE, TRUE, 1, CHOLMOD_COMPLEX,
                                   c );

   AAp = AA->p;
   for( i=0; i<*n+1; i++ ){
      AAp[i] = Ap[i] - 1; /* shift */
   }

   AAi = AA->i;
   for( i=0; i<*nnz; i++ ){
      AAi[i] = Ai[i] - 1; /* shift */
   }

   *LL_addr = (ADDRESS) cholmod_l_analyze( AA, c );
   LL = (cholmod_factor*)*LL_addr;

   *AA_addr = (ADDRESS) AA;
}

/*----------------------------------------------------------------------
 *
 * Fortran call :
 *   call cholmod_cmplx_llt_num( n, nnz, Ax, c_addr, LL_addr, AA_addr,
 *                               status )
 *
 *     status = 0 if matrix is positive definite
 *              1 otherwise.
 */
#ifdef UNDERSCORE
#define CHOLMOD_CMPLX_LLT_NUM cholmod_cmplx_llt_num_
#else
#define CHOLMOD_CMPLX_LLT_NUM cholmod_cmplx_llt_num
#endif
void CHOLMOD_CMPLX_LLT_NUM(
     int* n, int* nnz, double Ax[], ADDRESS* c_addr, ADDRESS* LL_addr,
     ADDRESS* AA_addr,
     int* status
)
{
/*
 *  Warning: in Fortran, all indexes are 1-based.
 *  The shift is done in this routine.
 */
   SuiteSparse_long i;

   cholmod_sparse *AA;
   SuiteSparse_long *AAp, *AAi;
   double *AAx;

   cholmod_factor *LL;

   cholmod_common *c;

   c = (cholmod_common *)*c_addr;

   AA = (cholmod_sparse*)*AA_addr;

   /* update numerical entries of A */
   AAx = AA->x;
   for( i=0; i<2*(*nnz); i++ ){
      AAx[i] = Ax[i];
   }

   LL = (cholmod_factor*)*LL_addr;

   c->quick_return_if_not_posdef = TRUE;  /* for supernodal only */
   c->final_ll = TRUE;                    /* L*L' decomposition */
   c->print = 0;                          /* QUIET mode */

   cholmod_l_factorize( AA, LL, c );

   if( LL->minor == *n ){
      *status = 0;   /* ok */
   } else {
      *status = 1;   /* matrix is not symm. pos. def. */
   }

}

/*----------------------------------------------------------------------
 *
 * Fortran call :
 *   call cholmod_cmplx_get_factor( c_addr, LL_addr, S_addr, n, lnz,
 *                                  Lp, Li, Lx, perm )
 */
#ifdef UNDERSCORE
#define CHOLMOD_CMPLX_GET_FACTOR cholmod_cmplx_get_factor_
#else
#define CHOLMOD_CMPLX_GET_FACTOR cholmod_cmplx_get_factor
#endif
void CHOLMOD_CMPLX_GET_FACTOR(
     ADDRESS* c_addr, ADDRESS* LL_addr, ADDRESS* S_addr, int* n, int* lnz,
     int Lp[], int Li[], double Lx[], int perm[]
)
{
/*
 *  Warning: in Fortran, all indexes are 1-based.
 *  The shift is done in this routine.
 */
   SuiteSparse_long i;

   cholmod_sparse *S;
   cholmod_common *c;
   cholmod_factor *LL;
   SuiteSparse_long *LLp, *LLi, *LLperm;
   double *LLx;

   c = (cholmod_common *)*c_addr;
   LL = (cholmod_factor *)*LL_addr;
   S = (cholmod_sparse *)*S_addr;

   if( S != 0 ){ /* supernodal */
      LLp = S->p;
      LLi = S->i;
      LLx = S->x;
   } else {
      LLp = LL->p;
      LLi = LL->i;
      LLx = LL->x;
   }
   for( i=0; i<*n+1; i++ ){
      Lp[i] = LLp[i] + 1; /* shift */
   }
   for( i=0; i<*lnz; i++ ){
      Li[i] = LLi[i] + 1; /* shift */
   }
   for( i=0; i<2*(*lnz); i++ ){
      Lx[i] = LLx[i];
   }
   LLperm = LL->Perm;
   for( i=0; i<*n; i++ ){
      perm[i] = LLperm[i] + 1; /* shift */
   }

   cholmod_l_free_factor( &LL, c );
   *LL_addr = 0;
   if( S != 0 ){
      cholmod_l_free_sparse( &S, c );
      *S_addr = 0;
   }
   cholmod_l_finish( c );
   free( c );

}

/*----------------------------------------------------------------------
 *
 * Fortran call :
 *   call cholmod_cmplx_solve( n, nnz, Ap, Ai, Az, b,
 *                             z, status )
 *
 *    status =  0 : ok
 *           =  1 : failed
 *           = -1 : internal fail (out of memory, ...)
 *
 * In this routine, 'Az', 'b' and 'z' are declared as 'double' but
 * are complex on the Fortran side; so the arrays are of twice length.
 * (COMPLEX is used here, as opposed to other routines where ZOMPLEX
 *  is used instead)
 *
 */
#ifdef UNDERSCORE
#define CHOLMOD_CMPLX_SOLVE cholmod_cmplx_solve_
#else
#define CHOLMOD_CMPLX_SOLVE cholmod_cmplx_solve
#endif
void CHOLMOD_CMPLX_SOLVE(
     int* n, int* nnz, int Ap[], int Ai[], double Az[], double b[],
     double z[], int* status
)
{
/*
 *  Warning: in Fortran, all indexes are 1-based.
 *  The shift is done in this routine.
 */
   SuiteSparse_long i;
   cholmod_sparse *AA;
   SuiteSparse_long *AAp, *AAi;
   double *AAz;
   cholmod_factor *L;
   cholmod_common c;
   cholmod_dense *bb, *zz;
   double *array;

   cholmod_l_start ( &c );

   AA = cholmod_l_allocate_sparse( (SuiteSparse_long)*n, (SuiteSparse_long)*n, (SuiteSparse_long)*nnz,
                                   TRUE, TRUE, 1, CHOLMOD_COMPLEX,
                                   &c );

   AAp = AA->p;
   for( i=0; i<*n+1; i++ ){
      AAp[i] = Ap[i] - 1; /* shift */
   }

   AAi = AA->i;
   AAz = AA->x;
   for( i=0; i<*nnz; i++ ){
      AAi[i] = Ai[i] - 1; /* shift */
   }
   for( i=0; i<2*(*nnz); i++ ){
      AAz[i] = Az[i];
   }

   L = cholmod_l_analyze( AA, &c );
   c.final_ll = TRUE;                    /* L*L' decomposition */
   /* default: print some info only when something fails... */
   c.print = 0;                          /* QUIET mode */
   cholmod_l_factorize( AA, L, &c );

   cholmod_l_free_sparse( &AA, &c );

   if( L->minor == *n ){
      *status = 0;
      bb = (cholmod_dense *) malloc( sizeof(cholmod_dense) );
      bb->nrow = *n;
      bb->ncol = 1;
      bb->nzmax = *n;
      bb->d = *n;
      bb->x = b;
      bb->xtype = CHOLMOD_COMPLEX;
      bb->dtype = CHOLMOD_DOUBLE;

      zz = cholmod_l_solve( CHOLMOD_A, L, bb, &c );
if( zz == NULL ){
  printf("(MUESLI cholmod_cmplx_solve:) CHOLMOD ERROR: failed!\n");
  *status = -1;
} else {
      array = zz->x;
      for( i=0; i<2*(*n); i++ ){
         z[i] = array[i];
      }
}

      bb->x = NULL;
      cholmod_l_free_dense( &bb, &c );
      cholmod_l_free_dense( &zz, &c );
   } else {
      *status = 1;
   }
   cholmod_l_free_factor( &L, &c );
   cholmod_l_finish( &c );

}

/*----------------------------------------------------------------------
 *
 * Fortran call :
 *   call cholmod_cmplx_solve_factor( n, c_addr, L_addr, b,
 *                                    z )
 *
 */
#ifdef UNDERSCORE
#define CHOLMOD_CMPLX_SOLVE_FACTOR cholmod_cmplx_solve_factor_
#else
#define CHOLMOD_CMPLX_SOLVE_FACTOR cholmod_cmplx_solve_factor
#endif
void CHOLMOD_CMPLX_SOLVE_FACTOR(
     int* n, ADDRESS* c_addr, ADDRESS* L_addr, double b[],
     double z[]
)
{
   SuiteSparse_long i;
   cholmod_common *c;
   cholmod_factor *L;
   cholmod_dense *bb, *zz;
   double *array;

   c = (cholmod_common *)*c_addr;
   L = (cholmod_factor *)*L_addr;

   bb = (cholmod_dense *) malloc( sizeof(cholmod_dense) );
   bb->nrow = *n;
   bb->ncol = 1;
   bb->nzmax = *n;
   bb->d = *n;
   bb->x = b;
   bb->xtype = CHOLMOD_COMPLEX;
   bb->dtype = CHOLMOD_DOUBLE;

   zz = cholmod_l_solve( CHOLMOD_A, L, bb, c );

   array = zz->x;
   for( i=0; i<2*(*n); i++ ){
      z[i] = array[i];
   }

   bb->x = NULL;
   cholmod_l_free_dense( &bb, c );
   cholmod_l_free_dense( &zz, c );

}

/*----------------------------------------------------------------------
 *
 * Fortran call :
 *   call cholmod_cmplx_solve_factor_nrhs( n, c_addr, L_addr, nrhs, b,
 *                                         z )
 *
 */
#ifdef UNDERSCORE
#define CHOLMOD_CMPLX_SOLVE_FACTOR_NRHS cholmod_cmplx_solve_factor_nrhs_
#else
#define CHOLMOD_CMPLX_SOLVE_FACTOR_NRHS cholmod_cmplx_solve_factor_nrhs
#endif
void CHOLMOD_CMPLX_SOLVE_FACTOR_NRHS(
     int* n, ADDRESS* c_addr, ADDRESS* L_addr, int* nrhs, double b[],
     double z[]
)
{
   SuiteSparse_long i;
   cholmod_common *c;
   cholmod_factor *L;
   cholmod_dense *bb, *zz;
   double *array;

   c = (cholmod_common *)*c_addr;
   L = (cholmod_factor *)*L_addr;

   bb = (cholmod_dense *) malloc( sizeof(cholmod_dense) );
   bb->nrow = *n;
   bb->ncol = *nrhs;
   bb->nzmax = (*n)*(*nrhs);
   bb->d = *n;
   bb->x = b;
   bb->xtype = CHOLMOD_COMPLEX;
   bb->dtype = CHOLMOD_DOUBLE;

   zz = cholmod_l_solve( CHOLMOD_A, L, bb, c );

   array = zz->x;
   for( i=0; i<2*(*n)*(*nrhs); i++ ){
      z[i] = array[i];
   }

   bb->x = NULL;
   cholmod_l_free_dense( &bb, c );
   cholmod_l_free_dense( &zz, c );

}

