! f90 include file

!_______________________________________________________________________
!
   subroutine fix_tetra_orientation( nb_pts, x, y, z,                   &
                                     nb_tetra, tetra_n )

      integer,              intent(in) :: nb_pts, nb_tetra
      real(kind=MF_DOUBLE), intent(in) :: x(nb_pts), y(nb_pts), z(nb_pts)
      integer,              intent(in out) :: tetra_n( nb_tetra, 4 )
      !------ API end ------

#ifdef _DEVLP
      ! called only by 'mfDelaunay3D'

      real(kind=MF_DOUBLE) :: x1, y1, z1, x2, y2, z2, x3, y3, z3, x4, y4, z4
      integer :: i
      real(kind=MF_DOUBLE) :: det, tmp

      do i = 1, nb_tetra
         x1 = x(tetra_n(i,1))
         y1 = y(tetra_n(i,1))
         z1 = z(tetra_n(i,1))
         x2 = x(tetra_n(i,2))
         y2 = y(tetra_n(i,2))
         z2 = z(tetra_n(i,2))
         x3 = x(tetra_n(i,3))
         y3 = y(tetra_n(i,3))
         z3 = z(tetra_n(i,3))
         x4 = x(tetra_n(i,4))
         y4 = y(tetra_n(i,4))
         z4 = z(tetra_n(i,4))
         det = ((z3-z4)*y2+(-z2+z4)*y3+(-z3+z2)*y4)*x1 +                &
               ((-z3+z4)*y1+(-z4+z1)*y3+(z3-z1)*y4)*x2 +                &
               ((z2-z4)*y1+(z4-z1)*y2-y4*(z2-z1))*x3 -                  &
               ((-z3+z2)*y1+(z3-z1)*y2-y3*(z2-z1))*x4
         if( det > 0.0d0 ) then
            ! swap two points in order to get a direct orientation
            tmp = tetra_n(i,1)
            tetra_n(i,1) = tetra_n(i,2)
            tetra_n(i,2) = int( tmp )
         end if
      end do


#endif
   end subroutine fix_tetra_orientation
!_______________________________________________________________________
!
   subroutine build_tetra_conn( n_xyz, tetra_n, tetra_neighbors,        &
                                nodes_sorted_x, nodes_sorted_y, nodes_sorted_z, &
                                face_n, face_tetra, tetra_f, n_tetra )

      ! Build a Tetrahedral Connectivity structure (and more, actually)
      ! for searching in a 3D triangulation.
      !--------------------------------------------------------------------

      double precision, intent(in) :: n_xyz(:,:)
      integer, pointer :: tetra_n(:,:)           ! intent(in)
      integer, pointer :: tetra_neighbors(:,:)
      integer, pointer :: nodes_sorted_x(:)
      integer, pointer :: nodes_sorted_y(:)
      integer, pointer :: nodes_sorted_z(:)
      integer, pointer :: face_n(:,:)
      integer, pointer :: face_tetra(:,:)
      integer, pointer :: tetra_f(:,:)
      integer, pointer :: n_tetra(:)
      !------ API end ------

#ifdef _DEVLP
      integer :: nn, ne, i, j, k, nf, nf_max, i1, i2, i3, inode
      integer, allocatable :: face_n_0(:,:)
      integer, allocatable :: face_tetra_0(:,:)

      logical, allocatable :: tetra_visited(:)
      integer :: i_neigh
      logical :: found

   !------ end of declarations -- execution starts hereafter  ------

      nn = size(n_xyz,1)
      ne = size(tetra_n,1)

      allocate( tetra_visited(ne) )
      tetra_visited(:) = .false.

      ! Optimistic evaluation for total nomber of faces
      ! (worst case: each tetra is connected to others by only two faces;
      ! we discard the pathological case where all tetrahedra are disconnected)
      nf_max = 3*ne + 1 ! = 4*ne - (ne-1)
      allocate( face_n_0(nf_max,3) )
      ! No need to initialize face_n to zero.
      allocate( face_tetra_0(nf_max,2) )
      face_tetra_0(:,:) = 0 ! Very important (0 means that face is on the edge)
#ifndef _TRACE_MEM_ALLOC
      allocate( tetra_f(ne,4) )
#else
      call mf_allocate( array=tetra_f, m=ne, n=4,                       &
                        file="tetra.inc", line="???",                   &
                        symb="tetra_f", unit="build_tetra_conn" )
#endif
      tetra_f(:,:) = 0

      nf = 0

      do i = 1, ne

         ! Neighbors are  already sorted by 'tetgen': neighbor #i is the
         ! tetrahedron opposed to node #i.
         ! We have just to replace the 'no neighbor' index -1 (tetgen)
         ! by 0 (value used in Muesli).
         ! In Muesli, no neighbor is marked by '0'
         do j = 1, 4
            i_neigh = tetra_neighbors(i,j)
            if( i_neigh < 0 ) then
               tetra_neighbors(i,j) = 0
            end if
         end do

         ! Mark current tetra as already visited
         tetra_visited(i) = .true.

         do j = 1, 4
            i_neigh = tetra_neighbors(i,j)
            if( i_neigh /= 0 ) then
               if( tetra_visited(i_neigh) ) cycle
            end if
            ! (i1,i2,i3) is the common face
            select case( j )
               case( 1 )
                  i1 = tetra_n(i,2)
                  i2 = tetra_n(i,3)
                  i3 = tetra_n(i,4)
               case( 2 )
                  i1 = tetra_n(i,1)
                  i2 = tetra_n(i,3)
                  i3 = tetra_n(i,4)
               case( 3 )
                  i1 = tetra_n(i,1)
                  i2 = tetra_n(i,2)
                  i3 = tetra_n(i,4)
               case( 4 )
                  i1 = tetra_n(i,1)
                  i2 = tetra_n(i,2)
                  i3 = tetra_n(i,3)
            end select
            ! Recording the new face (i1,i2,i3)
            nf = nf + 1
            face_n_0(nf,1) = i1
            face_n_0(nf,2) = i2
            face_n_0(nf,3) = i3
            face_tetra_0(nf,1) = i
            face_tetra_0(nf,2) = i_neigh
            ! Storing nf at the position j
            tetra_f(i,j) = nf
            if( i_neigh == 0 ) cycle
            found = .false.
            do k = 1, 4
               if( i1/=tetra_n(i_neigh,k) .and. i2/=tetra_n(i_neigh,k) .and. &
                   i3/=tetra_n(i_neigh,k) ) then
                  tetra_f(i_neigh,k) = nf
                  found = .true.
                  exit
               end if
            end do
            if( .not. found ) then
               write(STDERR,*)
               write(STDERR,*) "(MUESLI build_tetra_conn:) internal error:"
               write(STDERR,*) "      Severe error: k not found for storing nf in tetra_f(i_neigh,?)"
               write(STDERR,*) "      Please report this bug to: Edouard.Canot@univ-rennes.fr"
               mf_message_displayed = .true.
               call muesli_trace( pause ="yes" )
               stop
            end if
         end do

      end do

#ifndef _TRACE_MEM_ALLOC
      allocate( face_n(nf,3) )
#else
      call mf_allocate( array=face_n, m=nf, n=2,                        &
                        file="tetra.inc", line="???",                   &
                        symb="face_n", unit="build_tetra_conn" )
#endif
      face_n(:,:) = face_n_0(1:nf,:)
      deallocate( face_n_0 )
#ifndef _TRACE_MEM_ALLOC
      allocate( face_tetra(nf,2) )
#else
      call mf_allocate( array=face_tetra, m=nf, n=2,                    &
                        file="tetra.inc", line="???",                   &
                        symb="face_tetra", unit="build_tetra_conn" )
#endif
      face_tetra(:,:) = face_tetra_0(1:nf,:)
      deallocate( face_tetra_0 )

      call build_nodes_sorted_3D( n_xyz, nodes_sorted_x, nodes_sorted_y, &
                                  nodes_sorted_z )

      ! Building 'n_tetra' array : one triangle (any) for each node
      !                            (useful for 'tsearch', see below)
#ifndef _TRACE_MEM_ALLOC
      allocate( n_tetra(nn) )
#else
      call mf_allocate( array=n_tetra, n=nn,                            &
                        file="tetra.inc", line="???",                   &
                        symb="n_tetra", unit="build_tetra_conn" )
#endif
      do k = 1, 4
         do i = 1, ne
            inode = tetra_n(i,k)
            n_tetra(inode) = i
         end do
      end do

#endif
   end subroutine build_tetra_conn
!_______________________________________________________________________
!
   subroutine build_nodes_sorted_3D( n_xyz, nodes_sorted_x,             &
                                     nodes_sorted_y, nodes_sorted_z )

      double precision, intent(in) :: n_xyz(:,:)
      integer, pointer :: nodes_sorted_x(:)
      integer, pointer :: nodes_sorted_y(:)
      integer, pointer :: nodes_sorted_z(:)
      !------ API end ------

#ifdef _DEVLP
      double precision, allocatable :: tmp(:)
      integer :: nn, i

   !------ end of declarations -- execution starts hereafter  ------

      nn = size( n_xyz, 1 )

#ifndef _TRACE_MEM_ALLOC
      allocate( nodes_sorted_x(nn) )
#else
      call mf_allocate( array=nodes_sorted_x, n=nn,                     &
                        file="tetra.inc", line="???",                   &
                        symb="nodes_sorted_x", unit="build_nodes_sorted_3D" )
#endif
#ifndef _TRACE_MEM_ALLOC
      allocate( nodes_sorted_y(nn) )
#else
      call mf_allocate( array=nodes_sorted_y, n=nn,                     &
                        file="tetra.inc", line="???",                   &
                        symb="nodes_sorted_y", unit="build_nodes_sorted_3D" )
#endif
#ifndef _TRACE_MEM_ALLOC
      allocate( nodes_sorted_z(nn) )
#else
      call mf_allocate( array=nodes_sorted_z, n=nn,                     &
                        file="tetra.inc", line="???",                   &
                        symb="nodes_sorted_z", unit="build_nodes_sorted_3D" )
#endif

      nodes_sorted_x(:) = [ (i, i = 1, nn) ]
      nodes_sorted_y(:) = [ (i, i = 1, nn) ]
      nodes_sorted_z(:) = [ (i, i = 1, nn) ]

      ! Sorting coord. 'x'
      allocate( tmp(nn) )
      tmp(:) = n_xyz(:,1)
      call quick_sort( "asc", tmp, nodes_sorted_x(:) )
      deallocate( tmp )

      ! Sorting coord. 'y'
      allocate( tmp(nn) )
      tmp(:) = n_xyz(:,2)
      call quick_sort( "asc", tmp, nodes_sorted_y(:) )
      deallocate( tmp )

      ! Sorting coord. 'z'
      allocate( tmp(nn) )
      tmp(:) = n_xyz(:,3)
      call quick_sort( "asc", tmp, nodes_sorted_z(:) )
      deallocate( tmp )

#endif
   end subroutine build_nodes_sorted_3D
!_______________________________________________________________________
!
   function tsearch_3D( n_xyz, tetra_n, nodes_sorted_x, nodes_sorted_y, &
                        nodes_sorted_z, face_tetra, tetra_f, n_tetra,   &
                        xi, yi, zi, previous )                          &
   result( num )

      double precision, pointer :: n_xyz(:,:)
      integer,          pointer :: tetra_n(:,:)
      integer,          pointer :: nodes_sorted_x(:)
      integer,          pointer :: nodes_sorted_y(:)
      integer,          pointer :: nodes_sorted_z(:)
      integer,          pointer :: face_tetra(:,:)
      integer,          pointer :: tetra_f(:,:)
      integer,          pointer :: n_tetra(:)
      double precision, intent(in) :: xi, yi, zi
      logical, optional :: previous
      integer :: num
      !------ API end ------

#ifdef _DEVLP
      ! The 'previous' option is useful when the search is repeated;
      ! this is the case when the calling function is 'mfGridData3D'

      integer :: k, face
      logical :: previous_d
      integer, save :: num_previous = 0

   !------ end of declarations -- execution starts hereafter  ------

      if( present(previous) ) then
         previous_d = previous
      else
         previous_d = .false.
      end if

      if( previous_d .and. num_previous/=0 ) then
         num = num_previous
      else
         ! First, we must detect a node sufficiently closed to (xi,yi,zi),
         ! in order to determine a starting tetrahedron...
         k = find_node_3D( n_xyz, nodes_sorted_x, nodes_sorted_y,       &
                           nodes_sorted_z, xi, yi, zi )
         num = n_tetra(k)
      end if

      do
         ! Does the current tetrahedron contain the final point?
         face = is_in_tetrahedron( n_xyz, tetra_n, num, xi, yi, zi )
         if( face == 0 ) then
            ! Tetrahedron found
            num_previous = num
            return
         else
            num = adj_tetrahedron( face_tetra, tetra_f, num, face )
            if( num == 0 ) then
               ! Tetrahedron not found
               num_previous = num
               return
            end if
         end if
      end do

#endif
   end function tsearch_3D
!_______________________________________________________________________
!
   function find_node_3D( n_xyz, nodes_sorted_x, nodes_sorted_y,        &
                          nodes_sorted_z, xi, yi, zi )                  &
            result( inode )

      double precision, pointer :: n_xyz(:,:)
      integer,          pointer :: nodes_sorted_x(:)
      integer,          pointer :: nodes_sorted_y(:)
      integer,          pointer :: nodes_sorted_z(:)
      double precision, intent(in) :: xi, yi, zi
      integer :: inode
      !------ API end ------

#ifdef _DEVLP
      integer :: nn, i_x, i_y, i_z, i1, i2, im
      integer :: i, j, k, iref
      integer :: i_x_1, i_x_2, i_y_1, i_y_2, i_z_1, i_z_2

   !------ end of declarations -- execution starts hereafter  ------

      nn = size( n_xyz, 1 )

      if( xi < n_xyz(nodes_sorted_x(1),1) ) then
         i_x = 1
      else if( n_xyz(nodes_sorted_x(nn),1) <= xi ) then
         i_x = nn
      else
         ! Binary search for xi in nodes_sorted_x
         i1 = 1
         i2 = nn
         do while( i2-i1 > 1 )
            im = (i1+i2)/2
            if( xi < n_xyz(nodes_sorted_x(im),1) ) then
               i2 = im
            else
               i1 = im
            end if
         end do
         i_x = i1
      end if

      if( yi < n_xyz(nodes_sorted_y(1),2) ) then
         i_y = 1
      else if( n_xyz(nodes_sorted_y(nn),2) <= yi ) then
         i_y = nn
      else
         ! Binary search for yi in nodes_sorted_y
         i1 = 1
         i2 = nn
         do while( i2-i1 > 1 )
            im = (i1+i2)/2
            if( yi < n_xyz(nodes_sorted_y(im),2) ) then
               i2 = im
            else
               i1 = im
            end if
         end do
         i_y = i1
      end if

      if( zi < n_xyz(nodes_sorted_z(1),3) ) then
         i_z = 1
      else if( n_xyz(nodes_sorted_z(nn),3) <= zi ) then
         i_z = nn
      else
         ! Binary search for zi in nodes_sorted_z
         i1 = 1
         i2 = nn
         do while( i2-i1 > 1 )
            im = (i1+i2)/2
            if( zi < n_xyz(nodes_sorted_z(im),3) ) then
               i2 = im
            else
               i1 = im
            end if
         end do
         i_z = i1
      end if

      if( i_x == i_y .and. i_y == i_z ) then
         inode = i_x
         return
      end if

      i_x_1 = max(i_x-1,1)
      i_x_2 = min(nn,i_x+1)
      i_y_1 = max(i_y-1,1)
      i_y_2 = min(nn,i_y+1)
      i_z_1 = max(i_z-1,1)
      i_z_2 = min(nn,i_z+1)
      do
         ! Searching the intersection between : nodes_sorted_x(i_x_1:i_x_2)
         !                                and : nodes_sorted_y(i_y_1:i_y_2)
         !                                and : nodes_sorted_z(i_z_1:i_z_2)
         ! this requires another loop
         do i = i_x_1, i_x_2
            iref = nodes_sorted_x(i)
            do j = i_y_1, i_y_2
               if( nodes_sorted_y(j) == iref ) then
                  do k = i_z_1, i_z_2
                     if( nodes_sorted_z(k) == iref ) then
                        inode = iref
                        return
                     end if
                  end do
               else
                  cycle
               end if
            end do
         end do
         i_x_1 = max(i_x_1-1,1)
         i_x_2 = min(nn,i_x_2+1)
         i_y_1 = max(i_y_1-1,1)
         i_y_2 = min(nn,i_y_2+1)
         i_z_1 = max(i_z_1-1,1)
         i_z_2 = min(nn,i_z_2+1)
      end do

#endif
   end function find_node_3D
!_______________________________________________________________________
!
   function is_in_tetrahedron( n_xyz, tetra_n, num, x, y, z )           &
            result( res )

      double precision, pointer :: n_xyz(:,:)
      integer,          pointer :: tetra_n(:,:)
      integer, intent(in) :: num
      double precision, intent(in) :: x, y, z
      integer :: res
      !------ API end ------

#ifdef _DEVLP
      ! Returns 0 if the point (x,y,z) is inside the tetrahedron 'num'
      !
      ! else, returns the first face for which the node is exterior.

      double precision :: x1, y1, z1, x2, y2, z2,                       &
                          x3, y3, z3, x4, y4, z4

   !------ end of declarations -- execution starts hereafter  ------

      x1 = n_xyz(tetra_n(num,1),1)
      y1 = n_xyz(tetra_n(num,1),2)
      z1 = n_xyz(tetra_n(num,1),3)

      x2 = n_xyz(tetra_n(num,2),1)
      y2 = n_xyz(tetra_n(num,2),2)
      z2 = n_xyz(tetra_n(num,2),3)

      x3 = n_xyz(tetra_n(num,3),1)
      y3 = n_xyz(tetra_n(num,3),2)
      z3 = n_xyz(tetra_n(num,3),3)

      x4 = n_xyz(tetra_n(num,4),1)
      y4 = n_xyz(tetra_n(num,4),2)
      z4 = n_xyz(tetra_n(num,4),3)

      ! The tetrahedron is defined by a direct triangle (P1,P2,P3), and
      ! the P4 point is above this triangle.
      !
      ! Considering the plane (P1,P2,P3), a point P is below (i.e. outside
      ! the tetrahedron), if : ( P1P2 x P1P3 ).P1P < 0
      ! ('x' is the cross-product and '.' is the dot-product)
      !
      ! Tests for other planes are obtained by permutation over the indices,
      ! using an appropriate order: (P2,P4,P3), (P4,P1,P3) and (P1,P4,P2).

      if( ((y2-y1)*(z3-z1)-(y3-y1)*(z2-z1))*(x-x1) +                    &
          ((z2-z1)*(x3-x1)-(z3-z1)*(x2-x1))*(y-y1) +                    &
          ((x2-x1)*(y3-y1)-(x3-x1)*(y2-y1))*(z-z1) < 0.0d0 ) then
         res = 4
         return
      end if

      if( ((y4-y2)*(z3-z2)-(y3-y2)*(z4-z2))*(x-x2) +                    &
          ((z4-z2)*(x3-x2)-(z3-z2)*(x4-x2))*(y-y2) +                    &
          ((x4-x2)*(y3-y2)-(x3-x2)*(y4-y2))*(z-z2) < 0.0d0 ) then
         res = 1
         return
      end if

      if( ((y1-y4)*(z3-z4)-(y3-y4)*(z1-z4))*(x-x4) +                    &
          ((z1-z4)*(x3-x4)-(z3-z4)*(x1-x4))*(y-y4) +                    &
          ((x1-x4)*(y3-y4)-(x3-x4)*(y1-y4))*(z-z4) < 0.0d0 ) then
         res = 2
         return
      end if

      if( ((y4-y1)*(z2-z1)-(y2-y1)*(z4-z1))*(x-x1) +                    &
          ((z4-z1)*(x2-x1)-(z2-z1)*(x4-x1))*(y-y1) +                    &
          ((x4-x1)*(y2-y1)-(x2-x1)*(y4-y1))*(z-z1) < 0.0d0 ) then
         res = 3
         return
      end if

      res = 0

#endif
   end function is_in_tetrahedron
!_______________________________________________________________________
!
   function adj_tetrahedron( face_tetra, tetra_f, num, face )           &
            result( num_new )

      integer, pointer :: face_tetra(:,:)
      integer, pointer :: tetra_f(:,:)
      integer, intent(in) :: num, face
      integer :: num_new
      !------ API end ------

#ifdef _DEVLP
      ! Returns the number of the tetrahedron adjacent to the side
      ! mentionned by 'face' (1 to 4)

      integer :: iface

   !------ end of declarations -- execution starts hereafter  ------

      if( face == 0 ) then
         write(STDERR,*)
         write(STDERR,*) "(MUESLI adj_tetrahedron:) internal error:"
         write(STDERR,*) "        Severe error: face == 0"
         write(STDERR,*) "        Please report this bug to: Edouard.Canot@univ-rennes.fr"
         mf_message_displayed = .true.
         call muesli_trace( pause ="yes" )
         stop
      end if

      iface = tetra_f(num,face)
      num_new = face_tetra(iface,1)
      if( num_new == num ) then
         num_new = face_tetra(iface,2)
      end if

#endif
   end function adj_tetrahedron
!_______________________________________________________________________
!
   subroutine build_3d_node_neighbors( tetra_nodes, node_neighbors )

      double precision :: tetra_nodes(:,:) ! actually integers
      type(mf_Int_List) :: node_neighbors(:) ! allocated in the calling unit
      !------ API end ------

#ifdef _DEVLP
      integer :: nn, ne, i, n1, n2, n3, n4, nb, inode, k
      logical :: found

      ! First col is the number of non zero indices
      integer, allocatable :: node_neighbors_tmp(:,:)

   !------ end of declarations -- execution starts hereafter  ------

      nn = size(node_neighbors,1)
      ne = size(tetra_nodes,1)

      allocate( node_neighbors_tmp(nn,60) ) ! 15 should be sufficient (?)
      node_neighbors_tmp(:,1) = 0

      do i = 1, ne

         n1 = int( tetra_nodes(i,1) )
         n2 = int( tetra_nodes(i,2) )
         n3 = int( tetra_nodes(i,3) )
         n4 = int( tetra_nodes(i,4) )

         ! Processing n1
         nb = node_neighbors_tmp(n1,1)
         !   storing n2?
         found = .false.
         do k = 2, nb+1
            if( n2 == node_neighbors_tmp(n1,k) ) then
               found = .true.
               exit
            end if
         end do
         if( .not. found ) then
            nb = nb + 1
            node_neighbors_tmp(n1,1) = nb
            node_neighbors_tmp(n1,nb+1) = n2
         end if
         !   storing n3?
         found = .false.
         do k = 2, nb+1
            if( n3 == node_neighbors_tmp(n1,k) ) then
               found = .true.
               exit
            end if
         end do
         if( .not. found ) then
            nb = nb + 1
            node_neighbors_tmp(n1,1) = nb
            node_neighbors_tmp(n1,nb+1) = n3
         end if
         !   storing n4?
         found = .false.
         do k = 2, nb+1
            if( n4 == node_neighbors_tmp(n1,k) ) then
               found = .true.
               exit
            end if
         end do
         if( .not. found ) then
            nb = nb + 1
            node_neighbors_tmp(n1,1) = nb
            node_neighbors_tmp(n1,nb+1) = n4
         end if

         ! Processing n2
         nb = node_neighbors_tmp(n2,1)
         !   storing n1?
         found = .false.
         do k = 2, nb+1
            if( n1 == node_neighbors_tmp(n2,k) ) then
               found = .true.
               exit
            end if
         end do
         if( .not. found ) then
            nb = nb + 1
            node_neighbors_tmp(n2,1) = nb
            node_neighbors_tmp(n2,nb+1) = n1
         end if
         !   storing n3?
         found = .false.
         do k = 2, nb+1
            if( n3 == node_neighbors_tmp(n2,k) ) then
               found = .true.
               exit
            end if
         end do
         if( .not. found ) then
            nb = nb + 1
            node_neighbors_tmp(n2,1) = nb
            node_neighbors_tmp(n2,nb+1) = n3
         end if
         !   storing n4?
         found = .false.
         do k = 2, nb+1
            if( n4 == node_neighbors_tmp(n2,k) ) then
               found = .true.
               exit
            end if
         end do
         if( .not. found ) then
            nb = nb + 1
            node_neighbors_tmp(n2,1) = nb
            node_neighbors_tmp(n2,nb+1) = n4
         end if

         ! Processing n3
         nb = node_neighbors_tmp(n3,1)
         !   storing n1?
         found = .false.
         do k = 2, nb+1
            if( n1 == node_neighbors_tmp(n3,k) ) then
               found = .true.
               exit
            end if
         end do
         if( .not. found ) then
            nb = nb + 1
            node_neighbors_tmp(n3,1) = nb
            node_neighbors_tmp(n3,nb+1) = n1
         end if
         !   storing n2?
         found = .false.
         do k = 2, nb+1
            if( n2 == node_neighbors_tmp(n3,k) ) then
               found = .true.
               exit
            end if
         end do
         if( .not. found ) then
            nb = nb + 1
            node_neighbors_tmp(n3,1) = nb
            node_neighbors_tmp(n3,nb+1) = n2
         end if
         !   storing n4?
         found = .false.
         do k = 2, nb+1
            if( n4 == node_neighbors_tmp(n3,k) ) then
               found = .true.
               exit
            end if
         end do
         if( .not. found ) then
            nb = nb + 1
            node_neighbors_tmp(n3,1) = nb
            node_neighbors_tmp(n3,nb+1) = n4
         end if

         ! Processing n4
         nb = node_neighbors_tmp(n4,1)
         !   storing n1?
         found = .false.
         do k = 2, nb+1
            if( n1 == node_neighbors_tmp(n4,k) ) then
               found = .true.
               exit
            end if
         end do
         if( .not. found ) then
            nb = nb + 1
            node_neighbors_tmp(n4,1) = nb
            node_neighbors_tmp(n4,nb+1) = n1
         end if
         !   storing n2?
         found = .false.
         do k = 2, nb+1
            if( n2 == node_neighbors_tmp(n4,k) ) then
               found = .true.
               exit
            end if
         end do
         if( .not. found ) then
            nb = nb + 1
            node_neighbors_tmp(n4,1) = nb
            node_neighbors_tmp(n4,nb+1) = n2
         end if
         !   storing n3?
         found = .false.
         do k = 2, nb+1
            if( n3 == node_neighbors_tmp(n4,k) ) then
               found = .true.
               exit
            end if
         end do
         if( .not. found ) then
            nb = nb + 1
            node_neighbors_tmp(n4,1) = nb
            node_neighbors_tmp(n4,nb+1) = n3
         end if

      end do

      do inode = 1, nn
         nb = node_neighbors_tmp(inode,1)
         ! Transfers to the output structure
         allocate( node_neighbors(inode)%list(nb) )
         node_neighbors(inode)%list(:) = node_neighbors_tmp(inode,2:nb+1)
      end do

      ! Deallocation of node_neighbors_tmp(:,:) is automatic

#endif
   end subroutine build_3d_node_neighbors
