program cmp_ifort_mod

   ! The INTEL-ifort precompiled module file (*.mod) of the same source
   ! file contains randomly occurrences of the following special ASCII
   ! characters: <HT>, <LF>, <VT> and <CR>.
   ! This version of the UNIX 'cmp' command, doesn't make any
   ! difference between these four characters.
   !
   ! Moreover, this program skips bytes 49-52, which contain the
   ! timestamp of the file. (since version 10 at least, and up to recent
   ! versions, like ifx-2025)
   !
   ! This program works with INTEL Fortran
   ! (ifort 10.x to 19.x -- ifx 2023 to 2025)
   !
   ! By default, this program runs silently if the files are the same;
   ! if they differ, only the byte at which the first difference
   ! occured is reported.
   !
   ! This program returns as value to the shell:
   !
   !  0: The files are identical (then they have the same size).
   !
   !  1: The files are different; this includes the case where one file
   !     is identical to the first part of the other.
   !
   !  2: An error ocurred.
   !--------------------------------------------------------------------

   ! É. Canot -- IPR/CNRS -- November 26, 2024

   ! must be compiled with a f90 compiler which supports the following
   ! routines: exit(), fstat()    [e.g. INTEL-ifort,
   !                                    GNU-gfortran]

   ! WARNING: the unit length for RECL is 4 (bytes) in INTEL-fortran,
   !          but is equal to 1 (byte) in GNU-fortran.
   ! Therefore, when using the INTEL compiler, we have to compile with
   ! the '-assume byterecl' option!

#if defined _INTEL_IFC
   use ifport
#endif

   implicit none

   character(len=80) :: file1, file2
   logical :: exist, verbose = .true.
   character(kind=1) :: c1, c2
#if defined _INTEL_IFC
   integer :: sarray1(12), sarray2(12)
#else
#if defined _GNU_GFC
   integer :: sarray1(13), sarray2(13)
#endif
#endif
   integer :: status, rec, size, n_args, n
   character(len=10) :: option

   n_args = iargc()
   n = 1
   if( n_args < 2 ) then
      print *
      print *, " usage: cmp_ifort_mod [-s] file1 file2"
      print *, "        the -s option makes the command silently."
      call exit(2)
   else if( n_args == 3 ) then
      call getarg( n, option )
      if( trim(option) == "-s" ) then
         verbose = .false.
      else
         print *, " cmp_ifort_mod: bad option. The only option is '-s'."
         call exit(2)
      end if
      n = n + 1
   end if

   call getarg( n, file1 )
   n = n + 1
   ! verifying the existence of the precompiled module file
   inquire( file=trim(file1), exist=exist )
   if( .not. exist ) then
      if( verbose ) then
         print *, " cmp_ifort_mod: " // trim(file1) // " doesn't exist."
      end if
      call exit(2)
   end if

   call getarg( n, file2 )
   ! verifying the existence of the precompiled module file
   inquire( file=trim(file2), exist=exist )
   if( .not. exist ) then
      if( verbose ) then
         print *, " cmp_ifort_mod: " // trim(file2) // " doesn't exist."
      end if
      call exit(2)
   end if

   ! opening the module file 1 in 'direct access' mode and defining
   ! the record length as 1 byte
   open( unit=11, file=trim(file1), action="read", status="old",        &
         access="direct", recl=1 )

   ! opening the module file 2 in 'direct access' mode and defining
   ! the record length as 1 byte
   open( unit=12, file=trim(file2), action="read", status="old",        &
         access="direct", recl=1 )

   ! getting the size of the two files; then exit if they differ
   status = fstat( 11, sarray1 )
   size = sarray1(8)
   status = fstat( 12, sarray2 )
   if( sarray2(8) /= size ) then
      if( verbose ) then
         print *, " cmp_ifort_mod: " // trim(file1) // " and " //       &
                  trim(file2) // " differ in size."
      end if
      call exit(1)
   end if

   do rec = 1, size
      if( 49 <= rec .and. rec <= 52 ) cycle
      read( 11, rec=rec ) c1
      select case( ichar(c1) )
         case( 9, 10, 11 )
            ! <HT>, <LF> and <VT> are changing to <CR>
            c1 = char(13)
      end select
      read( 12, rec=rec ) c2
      select case( ichar(c2) )
         case( 9, 10, 11 )
            ! <HT>, <LF> and <VT> are changing to <CR>
            c2 = char(13)
      end select
      if( c1 /= c2 ) then
         if( verbose ) then
            print "(2X,A,I0,A)", "cmp_ifort_mod: " // trim(file1) //    &
                  " and " // trim(file2) // " differ at byte ", rec, "."
         end if
         call exit(1)
      end if
   end do

   ! ok, EOF reach, normal termination
   call exit(0)

end program
