! errors are not processed => Fortran STOP !

module lib_aux

   use fml

   implicit none

contains
!_______________________________________________________________________
!
   function sym_part( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out

      ! returns the symmetric part of the mfArray 'A'

      integer :: s(2), i, j, n
      type(mfArray) :: alpha, beta

      call msInitArgs( A )

      !-----------------------------------------------------------------

      ! matrix cannot be empty
      if( mfIsEmpty(A) ) then
         stop "sym_part: empty matrix!"
      end if

      ! matrix must be square
      s = Shape(A)
      if( s(1) /= s(2) ) then
         stop "sym_part: matrix is not square!"
      end if

      n = s(1)
      if( n == 1 ) then
         ! trivial case
         out = A
      else
         call msAssign( out, mfZeros(n) )
         do i = 1, n
            do j = 1, i-1
               call msAssign( alpha, mfGet(A,i,j) )
               call msAssign( beta, mfGet(A,j,i) )
               call msSet( 0.5d0*(alpha+beta), out, i, j )
               call msSet( 0.5d0*(alpha+beta), out, j, i )
            end do
            ! diagonal
            call msSet( mfGet(A,i,i), out, i, i )
         end do
      end if

      call msRelease( alpha, beta )

      !-----------------------------------------------------------------

      call msFreeArgs( A )

      call msAutoRelease( A )

      call msReturnArray( out )

   end function sym_part
!_______________________________________________________________________
!
   subroutine sym_decmp ( out, A )

      type(mfArray) :: A
      type(mf_Out) :: out

      ! compute both the symmetric part and the antisymmetric part of
      ! the mfArray 'A'

      integer :: s(2), i, j, n
      type(mfArray), pointer :: sym, antisym
      type(mfArray) :: alpha, beta

      call msInitArgs( A )

      !-----------------------------------------------------------------

      ! matrix cannot be empty
      if( mfIsEmpty(A) ) then
         stop "sym_decmp: empty matrix!"
      end if

      ! matrix must be square
      s = Shape(A)
      if( s(1) /= s(2) ) then
         stop "sym_decmp: matrix is not square!"
      end if

      ! must have two output arguments
      if( out%n /= 2 ) then
         print *, "(sym_decmp:) two output args required!"
         print *, "             syntax is : call sym_decmp ( mfOut(sym,asym), A )"
         stop
      end if

      ! check args of mfOut
      if( .not. args_mfout_ok( out, A ) ) then
         write(STDERR,*) "(sym_decmp:) ERROR: output arguments cannot be tempo, or cannot share"
         write(STDERR,*) "                    same memory as another input argument."
         call msMuesliTrace( pause="yes" )
         go to 99
      end if

      sym     => out%ptr1
      antisym => out%ptr2
      call msRelease(sym)
      call msRelease(antisym)

      n = s(1)
      if( n == 1 ) then
         ! trivial case
         sym = A
         antisym = 0.0d0
      else
         call msAssign( sym, mfZeros(n) )
         call msAssign( antisym, mfZeros(n) )
         do i = 1, n
            do j = 1, i-1
               call msAssign( alpha, mfGet(A,i,j) )
               call msAssign( beta, mfGet(A,j,i) )
               call msSet( 0.5d0*(alpha+beta), sym, i, j )
               call msSet( 0.5d0*(alpha+beta), sym, j, i )
               call msSet( 0.5d0*(alpha-beta), antisym, i, j )
               call msSet( 0.5d0*(beta-alpha), antisym, j, i )
            end do
            ! diagonal
            call msSet( mfGet(A,i,i), sym, i, i )
         end do
      end if

      call msRelease( alpha, beta )

      !-----------------------------------------------------------------

99    continue

      call msFreeArgs( A )

      call msAutoRelease( A )

   end subroutine sym_decmp
!_______________________________________________________________________
!
   subroutine check_small( A, string, threshold )

      type(mfArray)                    :: A
      character(len=*),     intent(in) :: string ! to be displayed
      real(kind=MF_DOUBLE), intent(in) :: threshold

      ! verify if mfArray A has all elements smaller than a threshold;
      ! useful for many tests

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      !-----------------------------------------------------------------

      ! matrix cannot be empty
      if( mfIsEmpty(A) ) then
         call msPause( "check_small: empty matrix!" )
      end if

      print "()"

100   format(1X,A,ES9.2,A)

      if( mfIsScalar(A) ) then

         write(*,100,advance="no") "Is " // trim(string) // " smaller than", &
                                   threshold, "? "

      else

         write(*,100,advance="no") "Does " // trim(string) // " have all its elements smaller than", &
                                   threshold, "? "

      end if

      if( All( A <= threshold ) ) then
         print "(A)", "Yes"
      else
         print "(A)", "No (see below)"
         print "()"
         call msDisplay( A, string )
         call msPause( "check_small: test failed!" )
      end if

      print "()"

      !-----------------------------------------------------------------

      call msFreeArgs( A )

      call msAutoRelease( A )

      call mf_restore_fpe( )

   end subroutine check_small
!_______________________________________________________________________
!
   subroutine zeroes_small_values( A, threshold, mode )

      type(mfArray)                              :: A
      real(kind=MF_DOUBLE), intent(in)           :: threshold
      character(len=3),     intent(in), optional :: mode

      ! cancels very small elements of the mfArray A
      ! (by default threshold is absolute; it may be relative
      !  to the magnitude of A, when mode="rel")
      ! (magnitude is a pseudo-norm; it is the largest finite
      !  element found in A -- so, Inf and NaN are discarded)
      !
      ! mode = "abs" [default] or "rel"
!### TODO 2:
! mode = "rel" is not yet implemented (so magnitude is not yet computed)
      !
      ! useful to eliminate different writing for negative zero

      real(kind=MF_DOUBLE) :: magnitude
      integer :: i, j, nrow, ncol, nnz
      real(kind=MF_DOUBLE), pointer :: f90_ptr(:,:), vec_ptr(:)
      complex(kind=MF_DOUBLE), pointer :: f90_c_ptr(:,:), vec_c_ptr(:)

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      !-----------------------------------------------------------------

      if( mfIsTempoArray(A) ) then
         stop "zeroes_small_values: temporary object!"
      end if

      if( mfIsEmpty(A) ) then
         stop "zeroes_small_values: empty matrix!"
      end if

      if( threshold < 0.0d0 ) then
         stop "zeroes_small_values: 'threshold' must be positive!"
      end if

      if( mfIsSparse(A) ) then

         if( mfIsReal(A) ) then

            call msPointer( A, vec_ptr )
            nnz = Size( vec_ptr )

            do i = 1, nnz
               if( abs(vec_ptr(i)) <= threshold ) then
                  vec_ptr(i) = 0.0d0
               end if
            end do

            call msFreePointer( A, vec_ptr )

         else if( mfIsComplex(A) ) then

            call msPointer( A, vec_c_ptr )
            nnz = Size( vec_c_ptr )

            do i = 1, nnz
               if( abs(real(vec_c_ptr(i))) <= threshold ) then
                  vec_c_ptr(i) = cmplx(0.0d0,aimag(vec_c_ptr(i)),kind=MF_DOUBLE)
               end if
               if( abs(aimag(vec_c_ptr(i))) <= threshold ) then
                  vec_c_ptr(i) = cmplx(real(vec_c_ptr(i)),0.0d0,kind=MF_DOUBLE)
               end if
            end do

            call msFreePointer( A, vec_c_ptr )

         end if

      else ! dense case

         if( mfIsReal(A) ) then

            call msPointer( A, f90_ptr )
            nrow = Size( A, 1 )
            ncol = Size( A, 2 )

            do i = 1, nrow
               do j = 1, ncol
                  if( abs(f90_ptr(i,j)) <= threshold ) then
                     f90_ptr(i,j) = 0.0d0
                  end if
               end do
            end do

            call msFreePointer( A, f90_ptr )

         else if( mfIsComplex(A) ) then

            call msPointer( A, f90_c_ptr )
            nrow = Size( A, 1 )
            ncol = Size( A, 2 )

            do i = 1, nrow
               do j = 1, ncol
                  if( abs(real(f90_c_ptr(i,j))) <= threshold ) then
                     f90_c_ptr(i,j) = cmplx(0.0d0,aimag(f90_c_ptr(i,j)),kind=MF_DOUBLE)
                  end if
                  if( abs(aimag(f90_c_ptr(i,j))) <= threshold ) then
                     f90_c_ptr(i,j) = cmplx(real(f90_c_ptr(i,j)),0.0d0,kind=MF_DOUBLE)
                  end if
               end do
            end do

            call msFreePointer( A, f90_c_ptr )

         end if

      end if

      !-----------------------------------------------------------------

      call msFreeArgs( A )

      call msAutoRelease( A )

      call mf_restore_fpe( )

   end subroutine zeroes_small_values
!_______________________________________________________________________
!
   subroutine round_values( A, d )

      type(mfArray)       :: A
      integer, intent(in) :: d

      ! round all elements of a matrix A to 'd' decimals
      !
      ! 'd' must be ranged in [2,15]

      integer :: i, j, nrow, ncol, nnz, expo
      real(kind=MF_DOUBLE) :: tmp, factor, re, im
      real(kind=MF_DOUBLE), pointer :: f90_ptr(:,:), vec_ptr(:)
      complex(kind=MF_DOUBLE), pointer :: f90_c_ptr(:,:), vec_c_ptr(:)

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      !-----------------------------------------------------------------

      if( mfIsTempoArray(A) ) then
         stop "round_values: temporary object!"
      end if

      if( mfIsEmpty(A) ) then
         stop "round_values: empty matrix!"
      end if

      if( d < 2 .or. 15 < d ) then
         stop "round_values: 'd' must be ranged in [2,15]!"
      end if

      if( mfIsSparse(A) ) then
!### TODO 2:
stop 'lib_aux/round_values: not yet available for sparse mfArray'
      else ! dense case

         if( mfIsReal(A) ) then

            call msPointer( A, f90_ptr )
            nrow = Size( A, 1 )
            ncol = Size( A, 2 )

            do i = 1, nrow
               do j = 1, ncol
                  if( f90_ptr(i,j) == 0.0d0 ) cycle
                  expo = log10( abs(f90_ptr(i,j)) )
                  factor = 10.0d0**(d-1-expo)
                  tmp = f90_ptr(i,j)*factor
                  f90_ptr(i,j) = anint(tmp)/factor
               end do
            end do

            call msFreePointer( A, f90_ptr )

         else if( mfIsComplex(A) ) then

            call msPointer( A, f90_c_ptr )
            nrow = Size( A, 1 )
            ncol = Size( A, 2 )

            do i = 1, nrow
               do j = 1, ncol
                  re = real(f90_c_ptr(i,j))
                  if( re == 0.0d0 ) go to 10
                  expo = log10( abs(re) )
                  factor = 10.0d0**(d-1-expo)
                  tmp = re*factor
                  re = anint(tmp)/factor
 10               continue
                  im = aimag(f90_c_ptr(i,j))
                  if( im == 0.0d0 ) go to 20
                  expo = log10( abs(im) )
                  factor = 10.0d0**(d-1-expo)
                  tmp = im*factor
                  im = anint(tmp)/factor
 20               continue
                  f90_c_ptr(i,j) = cmplx(re,im,kind=MF_DOUBLE)
               end do
            end do

            call msFreePointer( A, f90_c_ptr )

         end if

      end if

      !-----------------------------------------------------------------

      call msFreeArgs( A )

      call msAutoRelease( A )

      call mf_restore_fpe( )

   end subroutine round_values
!_______________________________________________________________________
!
end module lib_aux
