/* from GNU-GCC libgfortran/runtime/backtrace.c and
		libgfortran/libgfortran.h of gcc-4.5.0
   modified by É. Canot for MUESLI Fortran Library
   20 Feb 2014
   (compatible with 4.4 to 4.9, but not with 4.3)
   (4.8 has its own backtrace routine)
   NOT compatible with version 6
 */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <inttypes.h>
#include <unistd.h>
#include <execinfo.h>
#include <sys/wait.h>
#include <ctype.h>

#define st_printf _gfortrani_st_printf

#define gfc_xtoa _gfortrani_gfc_xtoa

#define full_exe_path _gfortrani_full_exe_path

typedef struct _IO_FILE FILE;

#define GFC_LARGEST_BUF 16
#define GFC_XTOA_BUF_SIZE (GFC_LARGEST_BUF * 2 + 1)

#ifdef _64_BITS
typedef __uint128_t GFC_UINTEGER_16;
#define GFC_UINTEGER_LARGEST GFC_UINTEGER_16
#else
typedef uint64_t GFC_UINTEGER_8;
#define GFC_UINTEGER_LARGEST GFC_UINTEGER_8
#endif

extern const char *gfc_xtoa( GFC_UINTEGER_LARGEST, char *, size_t );

extern char * full_exe_path( void );

/*----------------------------------------------------------------------------*/

static void
dump_glibc_backtrace( int depth, char *str[] )
{
  int i;

  for (i = 0; i < depth; i++)
    st_printf("  + %s\n", str[i]);

  free (str);
}

/* show_backtrace displays the backtrace, currently obtained by means of
   the glibc backtrace* functions.
   adding an underscore: show_backtrace will be called from GNU Fortran */
void
show_backtrace_( void )
{

#define DEPTH 50
#define BUFSIZE 1024

  void *trace[DEPTH];
  char **str;
  int depth;

  depth = backtrace (trace, DEPTH);
  if (depth <= 0)
    return;

  str = backtrace_symbols (trace, depth);

#define STDIN_FILENO 0
#define STDOUT_FILENO 1
#define STDERR_FILENO 2

  /* We attempt to extract file and line information from addr2line. */
  do
  {
    /* Local variables. */
    int f[2], pid, line, i;
    FILE *output;
    char addr_buf[DEPTH][GFC_XTOA_BUF_SIZE], func[BUFSIZE], file[BUFSIZE];
    char *p, *end;
    const char *addr[DEPTH];

    /* Write the list of addresses in hexadecimal format. */
    for (i = 0; i < depth; i++)
      addr[i] = gfc_xtoa( (GFC_UINTEGER_LARGEST) (intptr_t) trace[i],
			  addr_buf[i], sizeof (addr_buf[i]) );

    /* Don't output an error message if something goes wrong, we'll simply
       fall back to the pstack and glibc backtraces. */
    if (pipe (f) != 0)
      break;
    if ((pid = fork ()) == -1)
      break;

    if (pid == 0)
    {
      /* Child process. */
#define NUM_FIXEDARGS 5
      char *arg[DEPTH+NUM_FIXEDARGS+1];

      close (f[0]);
      close (STDIN_FILENO);
      close (STDERR_FILENO);

      if (dup2 (f[1], STDOUT_FILENO) == -1)
	_exit (0);
      close (f[1]);

      arg[0] = (char *) "addr2line";
      arg[1] = (char *) "-e";
      arg[2] = full_exe_path ();
      arg[3] = (char *) "-f";
      arg[4] = (char *) "-s";
      for (i = 0; i < depth; i++)
	arg[NUM_FIXEDARGS+i] = (char *) addr[i];
      arg[NUM_FIXEDARGS+depth] = NULL;
      execvp (arg[0], arg);
      _exit (0);
#undef NUM_FIXEDARGS
    }

    /* Father process. */
    close (f[1]);
    wait (NULL);
    output = fdopen (f[0], "r");
    i = -1;

    if( fgets (func, sizeof(func), output) )
    {
      int first = 1;
      st_printf("\n  Backtrace: (Innermost first)\n");

      do
      {
	if( ! fgets (file, sizeof(file), output) )
	  goto fallback;

	i++;

	for (p = func; *p != '\n' && *p != '\r'; p++)
	  ;

	*p = '\0';

	/* Try to recognize the internal libgfortran functions. */
	if( strncmp(func, "*_gfortran", 10) == 0 ||
	    strncmp(func, "_gfortran", 9) == 0 ||
	    strcmp(func, "main") == 0 ||
	    strcmp(func, "_start") == 0 ||
	    strcmp(func, "_gfortrani_handler") == 0
	   )
	  continue;

	if( strstr(str[i], "libgfortran.so") != NULL ||
	    strstr(str[i], "libgfortran.a") != NULL ||
	    strstr(str[i], "__libc_start_main") != NULL
	   )
	  continue;

	/* If we only have the address, use the glibc backtrace. */
	if( func[0] == '?' && func[1] == '?' && file[0] == '?' &&
	    file[1] == '?' )
	{
	  st_printf("  + %s\n", str[i]);
	  continue;
	}

	/* Extract the line number. */
	for( end = NULL, p = file; *p; p++ )
	  if( *p == ':' )
	    end = p;
	if( end != NULL )
	{
	  *end = '\0';
	  line = atoi (++end);
	}
	else
	  line = -1;

	if( strcmp(func, "MAIN__") == 0 )
	  st_printf("  + in the main program\n");
	else
          /* Skip some predefined routines. */
          /* Leave the routine names with and without modules ! Curiously enough
           * gcc-4.7 and gcc-4.8 behave differently...
           */
          if( strcmp(func, "show_backtrace_") == 0 ||
              strcmp(func,                   "mf_traceback") == 0 ||
              strcmp(func, "__mod_mfdebug_MOD_mf_traceback") == 0 ||
              strcmp(func,                   "muesli_trace") == 0 ||
              strcmp(func, "__mod_mfdebug_MOD_muesli_trace") == 0 ||
              strcmp(func,                   "msmueslitrace") == 0 ||
              strcmp(func, "__mod_mfdebug_MOD_msmueslitrace") == 0 ||
              strcmp(func,                   "printmessage") == 0 ||
              strcmp(func, "__mod_mfarray_MOD_printmessage") == 0
            )
            continue;
	if( first == 1 )
	{
	  st_printf("    routine %s\n", func);
	  first = 0;
	} else {
	  st_printf("  + routine %s\n", func);
	}

	if( line <= 0 && strcmp(file, "??") == 0 )
	  continue;

	if( line <= 0 )
	  st_printf("      from file %s\n", file);
	else
	  st_printf("      at line %d of file %s\n", line, file);
      }
      while( fgets (func, sizeof(func), output));

      free(str);
      return;

fallback:
      st_printf("** Something went wrong while running addr2line. **\n"
		"** Falling back to a simpler backtrace scheme.   **\n");
    }
  }
  while(0);

#undef DEPTH
#undef BUFSIZE

  /* Try to call pstack. */
  do
  {
    /* Local variables. */
    int pid;

    /* Don't output an error message if something goes wrong, we'll simply
       fall back to the pstack and glibc backtraces. */
    if( (pid = fork ()) == -1 )
      break;

    if( pid == 0 )
    {
      /* Child process. */
#define NUM_ARGS 2
      char *arg[NUM_ARGS+1];
      char buf[20];

      st_printf("\nBacktrace for this error:\n");
      arg[0] = (char *) "pstack";
      snprintf( buf, sizeof(buf), "%d", (int) getppid () );
      arg[1] = buf;
      arg[2] = NULL;
      execvp (arg[0], arg);
#undef NUM_ARGS

      /* pstack didn't work, so we fall back to dumping the glibc
	 backtrace if we can. */
      dump_glibc_backtrace (depth, str);

      _exit (0);
    }

    /* Father process. */
    wait (NULL);
    return;
  }
  while(0);

  /* Fallback to the glibc backtrace. */
  st_printf("\nBacktrace for this error:\n");
  dump_glibc_backtrace(depth, str);
}
