module mod_specfun ! Specialized Math Functions

   ! Part of MUESLI Numerical Library
   ! Copyright É. Canot 2003-2025 -- IPR/CNRS

!-----------------------------------------------------------------------
!                             used modules
!-----------------------------------------------------------------------

   use mod_datafun

#ifndef _DEVLP
   use mod_mfdebug ! required for the 2nd pass of the double compilation
   use mod_mfarray ! required for the 2nd pass of the double compilation
   use mod_core ! required for the 2nd pass of the double compilation
#endif

   implicit none

#ifndef _DEVLP
   private
#endif

   public :: mfErf, &
             mfErfInv, &
             mfErfC, &
             mfErfCInv, &
             mfErfCScaled, &
             mfExpInt, &
             mfGamma, &
             mfGammaLn, &
             mfBesselJ, &
             mfBesselY, &
             mfBesselI, &
             mfBesselK, &
             mfAiry, &
             msEllipKE, &
             mfIsPrime, &
             mfFactor, &
             msCleanPrimeNumbers, &
             msRat

   ! for Prime Numbers generation
   integer, parameter :: MF_PN_MAX_LIMIT = 2147000000 ! max is near huge(1)

   integer, save :: MF_PN_MAX = 0, MF_PN_NB = 0
   integer, save, allocatable :: prime_numbers(:)

   private :: prime_numbers_initialize, &
              real_to_rat

   real(kind=MF_DOUBLE), parameter :: MF_BESSEL_J0_ROOTS(30) =          &
       [ 2.404825557695772d0, 5.520078110286311d0, 8.653727912911011d0, &
         11.79153443901428d0, 14.93091770848779d0, 18.07106396791092d0, &
         21.21163662987926d0, 24.35247153074930d0, 27.49347913204025d0, &
         30.63460646843197d0, 33.77582021357357d0, 36.91709835366404d0, &
         40.05842576462823d0, 43.19979171317673d0, 46.34118837166181d0, &
         49.48260989739782d0, 52.62405184111499d0, 55.76551075501998d0, &
         58.90698392608094d0, 62.04846919022717d0, 65.18996480020685d0, &
         68.33146932985680d0, 71.47298160359372d0, 74.61450064370183d0, &
         77.75602563038805d0, 80.89755587113763d0, 84.03909077693818d0, &
         87.18062984364114d0, 90.32217263721047d0, 93.46371878194476d0 ]

   real(kind=MF_DOUBLE), parameter :: MF_BESSEL_J1_ROOTS(30) =          &
       [ 3.831705970207512d0, 7.015586669815618d0, 10.17346813506272d0, &
         13.32369193631422d0, 16.47063005087763d0, 19.61585851046824d0, &
         22.76008438059277d0, 25.90367208761838d0, 29.04682853491686d0, &
         32.18967991097440d0, 35.33230755008386d0, 38.47476623477162d0, &
         41.61709421281445d0, 44.75931899765282d0, 47.90146088718544d0, &
         51.04353518357151d0, 54.18555364106132d0, 57.32752543790101d0, &
         60.46945784534749d0, 63.61135669848123d0, 66.75322673409849d0, &
         69.89507183749578d0, 73.03689522557383d0, 76.17869958464145d0, &
         79.32048717547630d0, 82.46225991437355d0, 85.60401943635023d0, &
         88.74576714492630d0, 91.88750425169498d0, 95.02923180804469d0 ]

   public :: MF_BESSEL_J0_ROOTS, MF_BESSEL_J1_ROOTS

contains
!_______________________________________________________________________
!
#include "fml_specfun/cei_double.inc"
!_______________________________________________________________________
!
#include "fml_specfun/erf.inc"
!_______________________________________________________________________
!
   function mfExpInt( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! Exponential Integrale function:
      !   EXPINT(x) = int( (exp(-t)/t), t = x..+Inf)
      !   defined for x > 0

      ! special values:
      !   EXPINT(0) = +Inf
      !   EXPINT(+Inf) = 0
      !   EXPINT(x < 0) = NaN

      ! Restriction: limited to real, positive arg.

      ! Based on E1(X), from SLATEC

      integer :: status

      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfExpInt", "E",                            &
                            "this function cannot be",                  &
                            "applied to sparse matrix!" )
         go to 99
      end if

      if( .not. mfIsReal(A) ) then
         call PrintMessage( "mfExpInt", "E",                            &
                            "arg. must be real!" )
         go to 99
      end if

      out%data_type = MF_DT_DBLE
      out%shape = a%shape
      allocate( out%double(a%shape(1),a%shape(2)) )

      do j = 1, a%shape(2)
         do i = 1, a%shape(1)
            if( mf_isnan( a%double(i,j)) ) then
               out%double(i,j) = MF_NAN
            else if( a%double(i,j) == MF_INF ) then
               out%double(i,j) = 0.0d0
            else if( a%double(i,j) < 0.0d0 ) then
               out%double(i,j) = MF_NAN
               call PrintMessage( "mfExpInt", "W",                      &
                                  "negative argument: value set to NaN" )
            else if( a%double(i,j) == 0.0d0 ) then
               out%double(i,j) = MF_INF
            else
               out%double(i,j) = de1( a%double(i,j) )
            end if
         end do
      end do

      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfExpInt", "E",                         &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfExpInt
!_______________________________________________________________________
!
   function mfGamma( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! Gamma function

      ! Restriction: limited to real arg.

      integer :: status

      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfGamma", "E",                             &
                            "this function cannot (yet) be",            &
                            "applied to sparse matrix!" )
         go to 99
      end if

      if( .not. mfIsReal(A) ) then
         call PrintMessage( "mfGamma", "E",                             &
                            "arg. must be real!" )
         go to 99
      end if

      out%data_type = MF_DT_DBLE
      out%shape = a%shape
      allocate( out%double(a%shape(1),a%shape(2)) )

      do j = 1, a%shape(2)
         do i = 1, a%shape(1)
            if( mf_isnan( a%double(i,j)) ) then
               out%double(i,j) = MF_NAN
            else if( a%double(i,j) == MF_INF ) then
               out%double(i,j) = MF_INF
            else if( a%double(i,j) == -MF_INF ) then
               out%double(i,j) = MF_NAN
            else if( a%double(i,j) == 0.0d0 ) then
               out%double(i,j) = MF_NAN
            else
               if( a%double(i,j) > 0.0d0 ) then
#if defined _GNU_GFC | defined _INTEL_IFC
                  out%double(i,j) = gamma( a%double(i,j) )
#else
                  out%double(i,j) = dgamma( a%double(i,j) )
#endif
               else
                  if( dble(nint(a%double(i,j))) == a%double(i,j) ) then
                     out%double(i,j) = MF_NAN
                  else
#if defined _GNU_GFC | defined _INTEL_IFC
                     out%double(i,j) = gamma( a%double(i,j) )
#else
                     out%double(i,j) = dgamma( a%double(i,j) )
#endif
                  end if
               end if
            end if
         end do
      end do

      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfGamma", "E",                          &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfGamma
!_______________________________________________________________________
!
   function mfGammaLn( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! Logarithm of the Gamma function

      ! Restriction: limited to real arg. > 0

      integer :: status

      integer :: i, j
#if defined _GNU_GFC | defined _INTEL_IFC
#else
      integer :: ierr
#endif

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfGammaLn", "E",                           &
                            "this function cannot (yet) be",            &
                            "applied to sparse matrix!" )
         go to 99
      end if

      if( .not. mfIsReal(A) ) then
         call PrintMessage( "mfGammaLn", "E",                           &
                            "arg. must be real!" )
         go to 99
      end if

      out%data_type = MF_DT_DBLE
      out%shape = a%shape
      allocate( out%double(a%shape(1),a%shape(2)) )

      do j = 1, a%shape(2)
         do i = 1, a%shape(1)
            if( mf_isnan( a%double(i,j)) ) then
               out%double(i,j) = MF_NAN
            else if( a%double(i,j) == MF_INF ) then
               out%double(i,j) = MF_INF
            else if( a%double(i,j) < 0.0d0 ) then
               out%double(i,j) = MF_NAN
            else if( a%double(i,j) == 0.0d0 ) then
               out%double(i,j) = MF_INF
            else
#if defined _GNU_GFC | defined _INTEL_IFC
               out%double(i,j) = log_gamma( a%double(i,j) )
#else
               out%double(i,j) = dgamln( a%double(i,j), ierr )
#endif
            end if
         end do
      end do

      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfGammaLn", "E",                        &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfGammaLn
!_______________________________________________________________________
!
   function mfBesselJ( alpha, A ) result( out )

      real(kind=MF_DOUBLE), intent(in) :: alpha
      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      !-----------------------------------------------------------------
      ! Bessel functions of the first kind: J(alpha,x)
      !
      ! Restriction: 'x' must be real.
      !
      ! Library used: SLATEC
      !   (except intrinsic extensions for integer orders,
      !    for some compilers)
      !
      ! implementation:
      !   if order is integer:
      !      J(alpha,x) is computed, with alpha>=0, x>=0
      !                              otherwise:
      !      J(alpha,-x) = +/- J(alpha,x), according alpha parity
      !      J(-alpha,x) = (-1)^alpha J(alpha,x)
      !   if order is fractional:
      !      J(alpha,x) is computed, with alpha>=0, any real x
      !                              otherwise:
      !      J(-alpha,x) = J(alpha,x)*cos(alpha*pi)
      !                  - Y(alpha,x)*sin(alpha*pi)
      !
      !      exception: J(-alpha,0) = +/- Inf
      !-----------------------------------------------------------------

      integer :: i, j, status, nz, ierr, nu
      real(kind=MF_DOUBLE) :: y(1), cyr(1), cyi(1), y1(1), y2(1),       &
                              cwrkr(1), cwrki(1)
      complex(kind=MF_DOUBLE) :: z1(1), z2(1)


   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfBesselJ", "E",                           &
                            "this function cannot (yet) be",            &
                            "applied to sparse matrix!" )
         go to 99
      end if

      if( .not. mfIsReal(A) ) then
         call PrintMessage( "mfBesselJ", "E",                           &
                            "arg. must be real!" )
         go to 99
      end if

      out%shape = a%shape

      nu = nint(alpha)
      if( dble(nu) == alpha ) then ! integer order
         ! result will be real
         out%data_type = MF_DT_DBLE
         allocate( out%double(a%shape(1),a%shape(2)) )

         if( nu < 0 ) then ! negative integer order
            nu = -nu
            do j = 1, a%shape(2)
               do i = 1, a%shape(1)
#if defined _GNU_GFC | defined _INTEL_IFC
                  if( a%double(i,j) >= 0.0d0 ) then
                     out%double(i,j) = bessel_jn(nu,a%double(i,j))
                  else
                     if( mod(nu,2) == 0 ) then
                        out%double(i,j) = bessel_jn(nu,-a%double(i,j))
                     else
                        out%double(i,j) = -bessel_jn(nu,-a%double(i,j))
                     end if
                  end if
#else
                  if( mf_isnan( a%double(i,j)) ) then
                     out%double(i,j) = MF_NAN
                  else if( abs(a%double(i,j)) == MF_INF ) then
                     out%double(i,j) = 0.0d0
                  else if( a%double(i,j) >= 0.0d0 ) then
                     ! slatec: dbesj(x,alpha,n,y,nz)
                     call dbesj(a%double(i,j),-alpha,1,y,nz)
                     out%double(i,j) = y(1)
                  else
                     ! slatec: dbesj(x,alpha,n,y,nz)
                     call dbesj(-a%double(i,j),-alpha,1,y,nz)
                     if( mod(nu,2) == 0 ) then
                        out%double(i,j) = y(1)
                     else
                        out%double(i,j) = -y(1)
                     end if
                  end if
#endif
               end do
            end do
            out%double(:,:) = (-1)**nu * out%double(:,:)
         else ! positive integer order
            do j = 1, a%shape(2)
               do i = 1, a%shape(1)
#if defined _GNU_GFC
                  if( a%double(i,j) >= 0.0d0 ) then
                     out%double(i,j) = dbesjn(nu,a%double(i,j))
                  else
                     if( mod(nu,2) == 0 ) then
                        out%double(i,j) = dbesjn(nu,-a%double(i,j))
                     else
                        out%double(i,j) = -dbesjn(nu,-a%double(i,j))
                     end if
                  end if
#else
                  if( mf_isnan( a%double(i,j)) ) then
                     out%double(i,j) = MF_NAN
                  else if( abs(a%double(i,j)) == MF_INF ) then
                     out%double(i,j) = 0.0d0
                  else if( a%double(i,j) >= 0.0d0 ) then
                     ! slatec: dbesj(x,alpha,n,y,nz)
                     call dbesj(a%double(i,j),alpha,1,y,nz)
                     out%double(i,j) = y(1)
                  else
                     ! slatec: dbesj(x,alpha,n,y,nz)
                     call dbesj(-a%double(i,j),alpha,1,y,nz)
                     if( mod(nu,2) == 0 ) then
                        out%double(i,j) = y(1)
                     else
                        out%double(i,j) = -y(1)
                     end if
                  end if
#endif
               end do
            end do
         end if

      else ! fractional order

         if( alpha > 0.0d0 ) then ! positive fractional order

            if( any( A < 0.0d0 ) ) then
               ! result will be complex
               out%data_type = MF_DT_CMPLX
               allocate( out%cmplx(a%shape(1),a%shape(2)) )

               do j = 1, a%shape(2)
                  do i = 1, a%shape(1)
                     if( mf_isnan( a%double(i,j)) ) then
                        out%cmplx(i,j) = MF_NAN
                     else if( abs(a%double(i,j)) == MF_INF ) then
                        out%cmplx(i,j) = (0.0d0,0.0d0)
                     else
                        ! slatec: zbesj(zr,zi,fnu,kode,n,cyr,cyi,nz,ierr)
                        call zbesj(a%double(i,j),0.0d0,alpha,1,1,cyr,cyi,nz,ierr)
                        out%cmplx(i,j) = cmplx(cyr(1),cyi(1),kind=MF_DOUBLE)
                     end if
                  end do
               end do
            else
               ! result will be real
               out%data_type = MF_DT_DBLE
               allocate( out%double(a%shape(1),a%shape(2)) )

               do j = 1, a%shape(2)
                  do i = 1, a%shape(1)
                     if( mf_isnan( a%double(i,j)) ) then
                        out%double(i,j) = MF_NAN
                     else if( a%double(i,j) == MF_INF ) then
                        out%double(i,j) = 0.0d0
                     else
                        ! slatec: dbesj(x,alpha,n,y,nz)
                        call dbesj(a%double(i,j),alpha,1,y,nz)
                        out%double(i,j) = y(1)
                     end if
                  end do
               end do
            end if

         else ! negative fractional order

            if( any( A < 0.0d0 ) ) then
               ! result will be complex
               out%data_type = MF_DT_CMPLX
               allocate( out%cmplx(a%shape(1),a%shape(2)) )

               do j = 1, a%shape(2)
                  do i = 1, a%shape(1)
                     if( mf_isnan( a%double(i,j)) ) then
                        out%cmplx(i,j) = MF_NAN
                     else if( abs(a%double(i,j)) == MF_INF ) then
                        out%cmplx(i,j) = (0.0d0,0.0d0)
                     else if( a%double(i,j) == 0.0d0 ) then
                        out%cmplx(i,j) = MF_INF*(sin(-alpha*MF_PI)-MF_I)
                     else
                        ! slatec: zbesj(zr,zi,fnu,kode,n,cyr,cyi,nz,ierr)
                        call zbesj(a%double(i,j),0.0d0,-alpha,1,1,cyr,cyi,nz,ierr)
                        z1 = cmplx(cyr,cyi,kind=MF_DOUBLE)
                        ! slatec: zbesy(zr,zi,fnu,kode,n,cyr,cyi,nz,cwrkr,cwrki,ierr)
                        call zbesy(a%double(i,j),0.0d0,-alpha,1,1,cyr,cyi,nz,cwrkr,cwrki,ierr)
                        z2 = cmplx(cyr,cyi,kind=MF_DOUBLE)
                        out%cmplx(i,j) = z1(1)*cos(-alpha*MF_PI) -      &
                                         z2(1)*sin(-alpha*MF_PI)
                     end if
                  end do
               end do
            else
               ! result will be real
               out%data_type = MF_DT_DBLE
               allocate( out%double(a%shape(1),a%shape(2)) )

               do j = 1, a%shape(2)
                  do i = 1, a%shape(1)
                     if( mf_isnan( a%double(i,j)) ) then
                        out%double(i,j) = MF_NAN
                     else if( a%double(i,j) == MF_INF ) then
                        out%double(i,j) = 0.0d0
                     else if( a%double(i,j) == 0.0d0 ) then
                        out%double(i,j) = - (-MF_INF)*sin(-alpha*MF_PI)
                     else
                        ! slatec: dbesj(x,alpha,n,y,nz)
                        call dbesj(a%double(i,j),-alpha,1,y1,nz)
                        ! slatec: dbesy(x,fnu,n,y)
                        call dbesy(a%double(i,j),-alpha,1,y2)
                        out%double(i,j) = y1(1)*cos(-alpha*MF_PI) -     &
                                          y2(1)*sin(-alpha*MF_PI)
                     end if
                  end do
               end do
            end if

         end if

      end if

      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfBesselJ", "E",                        &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfBesselJ
!_______________________________________________________________________
!
   function mfBesselY( alpha, A ) result( out )

      real(kind=MF_DOUBLE), intent(in) :: alpha
      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      !-----------------------------------------------------------------
      ! Bessel functions of the second kind: Y(alpha,x)

      ! Restriction: 'x' must be real.
      !
      ! Library used: SLATEC
      !   (except intrinsic extensions for integer orders,
      !    for some compilers)
      !
      ! implementation:
      !   if order is integer:
      !      Y(alpha,x) is computed, with alpha>=0, x>=0
      !                              otherwise:
      !      Y(-alpha,x) = (-1)^alpha Y(alpha,x)
      !      Y(alpha,-x) is complex and computed as for fractional orders
      !   if order is fractional:
      !      Y(alpha,x) is computed, with alpha>=0, any real x
      !                              otherwise:
      !      Y(-alpha,x) = Y(alpha,x)*cos(alpha*pi)
      !                  + J(alpha,x)*sin(alpha*pi)
      !
      !      exception: Y(-alpha,0) = 0
      !-----------------------------------------------------------------

      integer :: i, j, status, nz, ierr, nu
      real(kind=MF_DOUBLE) :: y(1), cyr(1), cyi(1), y1(1), y2(1),       &
                              cwrkr(1), cwrki(1)
      complex(kind=MF_DOUBLE) :: z1(1), z2(1)


   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfBesselY", "E",                           &
                            "this function cannot (yet) be",            &
                            "applied to sparse matrix!" )
         go to 99
      end if

      if( .not. mfIsReal(A) ) then
         call PrintMessage( "mfBesselY", "E",                           &
                            "arg. must be real!" )
         go to 99
      end if

      out%shape = a%shape

      nu = nint(alpha)
      if( dble(nu) == alpha ) then ! integer order

         if( nu < 0 ) then ! negative integer order
            nu = -nu

            if( any( A < 0.0d0 ) ) then ! result will be complex
               out%data_type = MF_DT_CMPLX
               allocate( out%cmplx(a%shape(1),a%shape(2)) )

               do j = 1, a%shape(2)
                  do i = 1, a%shape(1)
                     if( mf_isnan( a%double(i,j)) ) then
                        out%cmplx(i,j) = MF_NAN
                     else if( abs(a%double(i,j)) == MF_INF ) then
                        out%cmplx(i,j) = 0.0d0
                     else if( a%double(i,j) == 0.0d0 ) then
                        out%cmplx(i,j) = -MF_INF
                     else
                        ! slatec: zbesy(zr,zi,fnu,kode,n,cyr,cyi,nz,cwrkr,cwrki,ierr)
                        call zbesy(a%double(i,j),0.0d0,-alpha,1,1,cyr,cyi,nz,cwrkr,cwrki,ierr)
                        out%cmplx(i,j) = (-1)**nu * cmplx(cyr(1),cyi(1),kind=MF_DOUBLE)
                     end if
                  end do
               end do

            else ! result will be real
               out%data_type = MF_DT_DBLE
               allocate( out%double(a%shape(1),a%shape(2)) )

               do j = 1, a%shape(2)
                  do i = 1, a%shape(1)
#if defined _GNU_GFC | defined _INTEL_IFC
                     out%double(i,j) = (-1)**nu * bessel_yn(nu,a%double(i,j))
#else
                     if( mf_isnan( a%double(i,j)) ) then
                        out%double(i,j) = MF_NAN
                     else if( a%double(i,j) == MF_INF ) then
                        out%double(i,j) = 0.0d0
                     else if( a%double(i,j) == 0.0d0 ) then
                        out%double(i,j) = -MF_INF * (-1)**nu
                     else
                        ! slatec: dbesy(x,fnu,n,y)
                        call dbesy(a%double(i,j),-alpha,1,y)
                        out%double(i,j) = (-1)**nu * y(1)
                     end if
#endif
                  end do
               end do

            end if

         else ! positive integer order

            if( any( A < 0.0d0 ) ) then ! result will be complex
               out%data_type = MF_DT_CMPLX
               allocate( out%cmplx(a%shape(1),a%shape(2)) )

               do j = 1, a%shape(2)
                  do i = 1, a%shape(1)
                     if( mf_isnan( a%double(i,j)) ) then
                        out%cmplx(i,j) = MF_NAN
                     else if( abs(a%double(i,j)) == MF_INF ) then
                        out%cmplx(i,j) = 0.0d0
                     else if( a%double(i,j) == 0.0d0 ) then
                        if( mod(nu,2) == 0 ) then
                           out%cmplx(i,j) = -MF_INF
                        else
                           out%cmplx(i,j) = MF_INF
                        end if
                     else
                        ! slatec: zbesy(zr,zi,fnu,kode,n,cyr,cyi,nz,cwrkr,cwrki,ierr)
                        call zbesy(a%double(i,j),0.0d0,alpha,1,1,cyr,cyi,nz,cwrkr,cwrki,ierr)
                        out%cmplx(i,j) = cmplx(cyr(1),cyi(1),kind=MF_DOUBLE)
                     end if
                  end do
               end do
            else ! result will be real
               out%data_type = MF_DT_DBLE
               allocate( out%double(a%shape(1),a%shape(2)) )

               do j = 1, a%shape(2)
                  do i = 1, a%shape(1)
#if defined _GNU_GFC | defined _INTEL_IFC
                     out%double(i,j) = bessel_yn(nu,a%double(i,j))
#else
                     if( mf_isnan( a%double(i,j)) ) then
                        out%double(i,j) = MF_NAN
                     else if( a%double(i,j) == MF_INF ) then
                        out%double(i,j) = 0.0d0
                     else if( a%double(i,j) == 0.0d0 ) then
                        out%double(i,j) = -MF_INF
                     else
                        ! slatec: dbesy(x,fnu,n,y)
                        call dbesy(a%double(i,j),alpha,1,y)
                        out%double(i,j) = y(1)
                     end if
#endif
                  end do
               end do
            end if
         end if

      else ! fractional order

         if( alpha > 0.0d0 ) then ! positive fractional order

            if( any( A < 0.0d0 ) ) then ! result will be complex
               out%data_type = MF_DT_CMPLX
               allocate( out%cmplx(a%shape(1),a%shape(2)) )

               do j = 1, a%shape(2)
                  do i = 1, a%shape(1)
                     if( mf_isnan( a%double(i,j)) ) then
                        out%cmplx(i,j) = MF_NAN
                     else if( abs(a%double(i,j)) == MF_INF ) then
                        out%cmplx(i,j) = (0.0d0,0.0d0)
                     else if( a%double(i,j) == 0.0d0 ) then
                        nu = floor(alpha)
                        out%cmplx(i,j) = cmplx(-MF_INF,MF_INF*(-1)**nu,kind=MF_DOUBLE)
                     else
                        ! slatec: zbesy(zr,zi,fnu,kode,n,cyr,cyi,nz,cwrkr,cwrki,ierr)
                        call zbesy(a%double(i,j),0.0d0,alpha,1,1,cyr,cyi,nz,cwrkr,cwrki,ierr)
                        out%cmplx(i,j) = cmplx(cyr(1),cyi(1),kind=MF_DOUBLE)
                     end if
                  end do
               end do
            else ! result will be real
               out%data_type = MF_DT_DBLE
               allocate( out%double(a%shape(1),a%shape(2)) )

               do j = 1, a%shape(2)
                  do i = 1, a%shape(1)
                     if( mf_isnan( a%double(i,j)) ) then
                        out%double(i,j) = MF_NAN
                     else if( a%double(i,j) == MF_INF ) then
                        out%double(i,j) = 0.0d0
                     else if( a%double(i,j) == 0.0d0 ) then
                        out%double(i,j) = -MF_INF
                     else
                        ! slatec: dbesy(x,fnu,n,y)
                        call dbesy(a%double(i,j),alpha,1,y)
                        out%double(i,j) = y(1)
                     end if
                  end do
               end do
            end if

         else ! negative fractional order

            if( any( A < 0.0d0 ) ) then
               ! result will be complex
               out%data_type = MF_DT_CMPLX
               allocate( out%cmplx(a%shape(1),a%shape(2)) )

               do j = 1, a%shape(2)
                  do i = 1, a%shape(1)
                     if( mf_isnan( a%double(i,j)) ) then
                        out%cmplx(i,j) = MF_NAN
                     else if( abs(a%double(i,j)) == MF_INF ) then
                        out%cmplx(i,j) = (0.0d0,0.0d0)
                     else if( a%double(i,j) == 0.0d0 ) then
                        out%cmplx(i,j) = (0.0d0,0.0d0)
                     else
                        ! slatec: zbesy(zr,zi,fnu,kode,n,cyr,cyi,nz,
                        !               cwrkr,cwrki,ierr)
                        call zbesy(a%double(i,j),0.0d0,-alpha,1,1,cyr,cyi,nz,cwrkr,cwrki,ierr)
                        z1 = cmplx(cyr,cyi,kind=MF_DOUBLE)
                        ! slatec: zbesj(zr,zi,fnu,kode,n,cyr,cyi,nz,ierr)
                        call zbesj(a%double(i,j),0.0d0,-alpha,1,1,cyr,cyi,nz,ierr)
                        z2 = cmplx(cyr,cyi,kind=MF_DOUBLE)
                        out%cmplx(i,j) = z1(1)*cos(-alpha*MF_PI) +      &
                                         z2(1)*sin(-alpha*MF_PI)
                     end if
                  end do
               end do
            else
               ! result will be real
               out%data_type = MF_DT_DBLE
               allocate( out%double(a%shape(1),a%shape(2)) )

               do j = 1, a%shape(2)
                  do i = 1, a%shape(1)
                     if( mf_isnan( a%double(i,j)) ) then
                        out%double(i,j) = MF_NAN
                     else if( a%double(i,j) == MF_INF ) then
                        out%double(i,j) = 0.0d0
                     else if( a%double(i,j) == 0.0d0 ) then
                        out%double(i,j) = 0.0d0
                     else
                        ! slatec: dbesy(x,fnu,n,y)
                        call dbesy(a%double(i,j),-alpha,1,y1)
                        ! slatec: dbesj(x,alpha,n,y,nz)
                        call dbesj(a%double(i,j),-alpha,1,y2,nz)
                        out%double(i,j) = y1(1)*cos(-alpha*MF_PI) +     &
                                          y2(1)*sin(-alpha*MF_PI)
                     end if
                  end do
               end do
            end if

         end if

      end if

      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfBesselY", "E",                        &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfBesselY
!_______________________________________________________________________
!
   function mfBesselI( alpha, A ) result( out )

      real(kind=MF_DOUBLE), intent(in) :: alpha
      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      !-----------------------------------------------------------------
      ! Modified Bessel functions of the first kind: I(alpha,x)
      !
      ! Restriction: 'x' must be real.
      !
      ! Library used: SLATEC
      !
      ! implementation:
      !   if order is integer:
      !      I(alpha,x) is computed, with alpha>=0, x>=0
      !                              otherwise:
      !      I(alpha,-x) = +/- I(alpha,x), according alpha parity
      !      I(-alpha,x) = I(alpha,x)
      !   if order is fractional:
      !      I(alpha,x) is computed, with alpha>=0, any real x
      !                              otherwise:
      !      I(-alpha,x) = I(alpha,x) + (2/pi)*sin(alpha*pi)*K(alpha,x)
      !
      !      exception: I(-alpha,0) = +/-Inf
      !-----------------------------------------------------------------

      integer :: i, j, status, nz, ierr, nu
      real(kind=MF_DOUBLE) :: y(1), cyr(1), cyi(1), y1(1), y2(1),       &
                              cwrkr(1), cwrki(1)
      complex(kind=MF_DOUBLE) :: z1(1), z2(1)


   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfBesselI", "E",                           &
                            "this function cannot (yet) be",            &
                            "applied to sparse matrix!" )
         go to 99
      end if

      if( .not. mfIsReal(A) ) then
         call PrintMessage( "mfBesselI", "E",                           &
                            "arg. must be real!" )
         go to 99
      end if

      out%shape = a%shape

      nu = nint(alpha)
      if( dble(nu) == alpha ) then ! integer order
         ! result will be real
         out%data_type = MF_DT_DBLE
         allocate( out%double(a%shape(1),a%shape(2)) )

         if( nu < 0 ) then ! negative integer order
            nu = -nu
            do j = 1, a%shape(2)
               do i = 1, a%shape(1)
                  if( mf_isnan( a%double(i,j)) ) then
                     out%double(i,j) = MF_NAN
                  else if( abs(a%double(i,j)) == MF_INF ) then
                     out%double(i,j) = -MF_INF * (-1)**nu
                  else if( a%double(i,j) >= 0.0d0 ) then
                     ! slatec: dbesi(x,alpha,kode,n,y,nz)
                     call dbesi(a%double(i,j),-alpha,1,1,y,nz)
                     out%double(i,j) = y(1)
                  else
                     ! slatec: dbesi(x,alpha,kode,n,y,nz)
                     call dbesi(-a%double(i,j),-alpha,1,1,y,nz)
                     out%double(i,j) = y(1)
                  end if
               end do
            end do
            out%double(:,:) = (-1)**nu * out%double(:,:)
         else ! positive integer order
            do j = 1, a%shape(2)
               do i = 1, a%shape(1)
                  if( mf_isnan( a%double(i,j)) ) then
                     out%double(i,j) = MF_NAN
                  else if( a%double(i,j) == MF_INF ) then
                     out%double(i,j) = MF_INF
                  else if( a%double(i,j) == -MF_INF ) then
                     out%double(i,j) = MF_INF * (-1)**nu
                  else if( a%double(i,j) >= 0.0d0 ) then
                     ! slatec: dbesi(x,alpha,kode,n,y,nz)
                     call dbesi(a%double(i,j),alpha,1,1,y,nz)
                     out%double(i,j) = y(1)
                  else
                     ! slatec: dbesi(x,alpha,kode,n,y,nz)
                     call dbesi(-a%double(i,j),alpha,1,1,y,nz)
                     if( mod(nu,2) == 0 ) then
                        out%double(i,j) = y(1)
                     else
                        out%double(i,j) = -y(1)
                     end if
                  end if
               end do
            end do
         end if

      else ! fractional order

         if( alpha > 0.0d0 ) then ! positive fractional order

            if( any( A < 0.0d0 ) ) then
               ! result will be complex
               out%data_type = MF_DT_CMPLX
               allocate( out%cmplx(a%shape(1),a%shape(2)) )

               do j = 1, a%shape(2)
                  do i = 1, a%shape(1)
                     if( mf_isnan( a%double(i,j)) ) then
                        out%cmplx(i,j) = MF_NAN
                     else if( a%double(i,j) == -MF_INF ) then
                        nu = floor(alpha)
                        out%cmplx(i,j) = cmplx(0.0d0,MF_INF*(-1)**nu,kind=MF_DOUBLE)
                     else if( a%double(i,j) == MF_INF ) then
                        out%cmplx(i,j) = MF_INF
                     else
                        ! slatec: zbesi(zr,zi,fnu,kode,n,cyr,cyi,nz,ierr)
                        call zbesi(a%double(i,j),0.0d0,alpha,1,1,cyr,cyi,nz,ierr)
                        out%cmplx(i,j) = cmplx(cyr(1),cyi(1),kind=MF_DOUBLE)
                     end if
                  end do
               end do
            else
               ! result will be real
               out%data_type = MF_DT_DBLE
               allocate( out%double(a%shape(1),a%shape(2)) )

               do j = 1, a%shape(2)
                  do i = 1, a%shape(1)
                     if( mf_isnan( a%double(i,j)) ) then
                        out%double(i,j) = MF_NAN
                     else if( a%double(i,j) == MF_INF ) then
                        out%double(i,j) = MF_INF
                     else
                        ! slatec: dbesi(x,alpha,kode,n,y,nz)
                        call dbesi(a%double(i,j),alpha,1,1,y,nz)
                        out%double(i,j) = y(1)
                     end if
                  end do
               end do
            end if

         else ! negative fractional order

            if( any( A < 0.0d0 ) ) then
               ! result will be complex
               out%data_type = MF_DT_CMPLX
               allocate( out%cmplx(a%shape(1),a%shape(2)) )

               do j = 1, a%shape(2)
                  do i = 1, a%shape(1)
                     if( mf_isnan( a%double(i,j)) ) then
                        out%cmplx(i,j) = MF_NAN
                     else if( a%double(i,j) == MF_INF ) then
                        out%cmplx(i,j) = MF_INF
                     else if( a%double(i,j) == -MF_INF ) then
                        nu = floor(alpha)
                        out%cmplx(i,j) = cmplx(0.0d0,-MF_INF*(-1)**nu,kind=MF_DOUBLE)
                     else if( a%double(i,j) == 0.0d0 ) then
                        out%cmplx(i,j) = MF_INF*(sin(-alpha*MF_PI)-MF_I)
                     else
                        ! slatec: zbesi(zr,zi,fnu,kode,n,cyr,cyi,nz,ierr)
                        call zbesi(a%double(i,j),0.0d0,-alpha,1,1,cyr,cyi,nz,ierr)
                        z1 = cmplx(cyr,cyi,kind=MF_DOUBLE)
                        ! slatec: zbesk(zr,zi,fnu,kode,n,cyr,cyi,nz,ierr)
                        call zbesk(a%double(i,j),0.0d0,-alpha,1,1,cyr,cyi,nz,ierr)
                        z2 = cmplx(cyr,cyi,kind=MF_DOUBLE)
                        out%cmplx(i,j) = z1(1) +                        &
                                         z2(1)*sin(-alpha*MF_PI)*2.0d0/MF_PI
                     end if
                  end do
               end do
            else
               ! result will be real
               out%data_type = MF_DT_DBLE
               allocate( out%double(a%shape(1),a%shape(2)) )

               do j = 1, a%shape(2)
                  do i = 1, a%shape(1)
                     if( mf_isnan( a%double(i,j)) ) then
                        out%double(i,j) = MF_NAN
                     else if( a%double(i,j) == MF_INF ) then
                        out%double(i,j) = MF_INF
                     else if( a%double(i,j) == 0.0d0 ) then
                        nu = floor(alpha)
                        out%double(i,j) = -MF_INF*(-1)**nu
                     else
                        ! slatec: dbesi(x,alpha,kode,n,y,nz)
                        call dbesi(a%double(i,j),-alpha,1,1,y1,nz)
                        ! slatec: dbesk(x,fnu,kode,n,y,nz)
                        call dbesk(a%double(i,j),-alpha,1,1,y2,nz)
                        out%double(i,j) = y1(1) +                       &
                                          y2(1)*sin(-alpha*MF_PI)*2.0d0/MF_PI
                     end if
                  end do
               end do
            end if

         end if

      end if

      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfBesselI", "E",                        &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfBesselI
!_______________________________________________________________________
!
   function mfBesselK( alpha, A ) result( out )

      real(kind=MF_DOUBLE), intent(in) :: alpha
      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      !-----------------------------------------------------------------
      ! Modified Bessel functions of the second kind: K(alpha,x)
      !
      ! Restriction: 'x' must be real.
      !
      ! Library used: SLATEC
      !
      ! implementation:
      !   if order is integer:
      !      K(alpha,x) is computed, with alpha>=0, x>=0
      !                              otherwise:
      !      K(alpha,-x) = +/- K(alpha,x), according alpha parity
      !      K(-alpha,x) = K(alpha,x)
      !   if order is fractional:
      !      K(alpha,x) is computed, with alpha>=0, any real x
      !                              otherwise:
      !      K(-alpha,x) = K(alpha,x)
      !
      !      exception: K(-alpha,0) = NaN or Inf
      !-----------------------------------------------------------------

      integer :: i, j, status, nz, ierr, nu
      real(kind=MF_DOUBLE) :: y(1), cyr(1), cyi(1), y1(1), y2(1),       &
                              cwrkr(1), cwrki(1)
      complex(kind=MF_DOUBLE) :: z1(1), z2(1)


   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfBesselK", "E",                           &
                            "this function cannot (yet) be",            &
                            "applied to sparse matrix!" )
         go to 99
      end if

      if( .not. mfIsReal(A) ) then
         call PrintMessage( "mfBesselK", "E",                           &
                            "arg. must be real!" )
         go to 99
      end if

      out%shape = a%shape

      nu = nint(alpha)
      if( dble(nu) == alpha ) then ! integer order

         if( any( A < 0.0d0 ) ) then ! result will be complex
            out%data_type = MF_DT_CMPLX
            allocate( out%cmplx(a%shape(1),a%shape(2)) )

            if( nu < 0 ) then ! negative integer order
               nu = -nu

               do j = 1, a%shape(2)
                  do i = 1, a%shape(1)
                     if( mf_isnan( a%double(i,j)) ) then
                        out%cmplx(i,j) = MF_NAN
                     else if( a%double(i,j) == MF_INF ) then
                        out%cmplx(i,j) = 0.0d0
                     else if( a%double(i,j) == -MF_INF ) then
                        out%cmplx(i,j) = cmplx(0.0d0,-MF_INF,kind=MF_DOUBLE)
                     else if( a%double(i,j) == 0.0d0 ) then
                        if( mod(nu,2) == 0 ) then
                           out%cmplx(i,j) = MF_INF
                        else
                           out%cmplx(i,j) = MF_NAN
                        end if
                     else if( a%double(i,j) > 0.0d0 ) then
                        ! slatec: dbesk(x,fnu,kode,n,y,nz)
                        call dbesk(a%double(i,j),-alpha,1,1,y,nz)
                        out%cmplx(i,j) = y(1)
                     else ! x < 0
                        ! slatec: zbesk(zr,zi,fnu,kode,n,cyr,cyi,nz,ierr)
                        call zbesk(a%double(i,j),0.0d0,-alpha,1,1,cyr,cyi,nz,ierr)
                        out%cmplx(i,j) = cmplx(cyr(1),cyi(1),kind=MF_DOUBLE)
                     end if
                  end do
               end do

            else ! positive integer order

               do j = 1, a%shape(2)
                  do i = 1, a%shape(1)
                     if( mf_isnan( a%double(i,j)) ) then
                        out%cmplx(i,j) = MF_NAN
                     else if( a%double(i,j) == MF_INF ) then
                        out%cmplx(i,j) = 0.0d0
                     else if( a%double(i,j) == -MF_INF ) then
                        out%cmplx(i,j) = cmplx(0.0d0,-MF_INF,kind=MF_DOUBLE)
                     else if( a%double(i,j) == 0.0d0 ) then
                        if( nu == 0 ) then
                           out%cmplx(i,j) = cmplx(MF_INF,MF_NAN,kind=MF_DOUBLE)
                        else if( mod(nu,2) == 0 ) then
                           out%cmplx(i,j) = MF_INF
                        else
                           out%cmplx(i,j) = MF_NAN
                        end if
                     else if( a%double(i,j) > 0.0d0 ) then
                        ! slatec: dbesk(x,fnu,kode,n,y,nz)
                        call dbesk(a%double(i,j),alpha,1,1,y,nz)
                        out%cmplx(i,j) = y(1)
                     else ! x < 0
                        ! slatec: zbesk(zr,zi,fnu,kode,n,cyr,cyi,nz,ierr)
                        call zbesk(a%double(i,j),0.0d0,alpha,1,1,cyr,cyi,nz,ierr)
                        out%cmplx(i,j) = cmplx(cyr(1),cyi(1),kind=MF_DOUBLE)
                     end if
                  end do
               end do

            end if

         else ! result will be real
            out%data_type = MF_DT_DBLE
            allocate( out%double(a%shape(1),a%shape(2)) )

            if( nu < 0 ) then ! negative integer order
               nu = -nu

               do j = 1, a%shape(2)
                  do i = 1, a%shape(1)
                     if( mf_isnan( a%double(i,j)) ) then
                        out%double(i,j) = MF_NAN
                     else if( a%double(i,j) == MF_INF ) then
                        out%double(i,j) = 0.0d0
                     else if( a%double(i,j) == 0.0d0 ) then
                        out%double(i,j) = MF_INF
                     else
                        ! slatec: dbesk(x,fnu,kode,n,y,nz)
                        call dbesk(a%double(i,j),-alpha,1,1,y,nz)
                        out%double(i,j) = y(1)
                     end if
                  end do
               end do

            else ! positive integer order

               do j = 1, a%shape(2)
                  do i = 1, a%shape(1)
                     if( mf_isnan( a%double(i,j)) ) then
                        out%double(i,j) = MF_NAN
                     else if( a%double(i,j) == MF_INF ) then
                        out%double(i,j) = 0.0d0
                     else if( a%double(i,j) == 0.0d0 ) then
                        out%double(i,j) = MF_INF
                     else
                        ! slatec: dbesk(x,fnu,kode,n,y,nz)
                        call dbesk(a%double(i,j),alpha,1,1,y,nz)
                        out%double(i,j) = y(1)
                     end if
                  end do
               end do

            end if

         end if

      else ! fractional order

         if( any( A < 0.0d0 ) ) then
            ! result will be complex
            out%data_type = MF_DT_CMPLX
            allocate( out%cmplx(a%shape(1),a%shape(2)) )

            if( alpha > 0.0d0 ) then ! positive fractional order

               do j = 1, a%shape(2)
                  do i = 1, a%shape(1)
                     if( mf_isnan( a%double(i,j)) ) then
                        out%cmplx(i,j) = MF_NAN
                     else if( abs(a%double(i,j)) == MF_INF ) then
                        out%cmplx(i,j) = 0.0d0
                     else if( a%double(i,j) == 0.0d0 ) then
                        out%cmplx(i,j) = MF_NAN
                     else
                        ! slatec: zbesk(zr,zi,fnu,kode,n,cyr,cyi,nz,ierr)
                        call zbesk(a%double(i,j),0.0d0,alpha,1,1,cyr,cyi,nz,ierr)
                        out%cmplx(i,j) = cmplx(cyr(1),cyi(1),kind=MF_DOUBLE)
                     end if
                  end do
               end do

            else ! negative fractional order

               do j = 1, a%shape(2)
                  do i = 1, a%shape(1)
                     if( mf_isnan( a%double(i,j)) ) then
                        out%cmplx(i,j) = MF_NAN
                     else if( abs(a%double(i,j)) == MF_INF ) then
                        out%cmplx(i,j) = 0.0d0
                     else if( a%double(i,j) == 0.0d0 ) then
                        out%cmplx(i,j) = MF_NAN
                     else
                        ! slatec: zbesk(zr,zi,fnu,kode,n,cyr,cyi,nz,ierr)
                        call zbesk(a%double(i,j),0.0d0,-alpha,1,1,cyr,cyi,nz,ierr)
                        out%cmplx(i,j) = cmplx(cyr(1),cyi(1),kind=MF_DOUBLE)
                     end if
                  end do
               end do

            end if

         else ! result will be real
            out%data_type = MF_DT_DBLE
            allocate( out%double(a%shape(1),a%shape(2)) )

            if( alpha > 0.0d0 ) then ! positive fractional order

               do j = 1, a%shape(2)
                  do i = 1, a%shape(1)
                     if( mf_isnan( a%double(i,j)) ) then
                        out%double(i,j) = MF_NAN
                     else if( a%double(i,j) == MF_INF ) then
                        out%double(i,j) = 0.0d0
                     else if( a%double(i,j) == 0.0d0 ) then
                        out%double(i,j) = MF_INF
                     else
                        ! slatec: dbesk(x,fnu,kode,n,y,nz)
                        call dbesk(a%double(i,j),alpha,1,1,y,nz)
                        out%double(i,j) = y(1)
                     end if
                  end do
               end do

            else ! negative fractional order

               do j = 1, a%shape(2)
                  do i = 1, a%shape(1)
                     if( mf_isnan( a%double(i,j)) ) then
                        out%double(i,j) = MF_NAN
                     else if( a%double(i,j) == MF_INF ) then
                        out%double(i,j) = 0.0d0
                     else if( a%double(i,j) == 0.0d0 ) then
                        out%double(i,j) = MF_INF
                     else
                        ! slatec: dbesk(x,fnu,kode,n,y,nz)
                        call dbesk(a%double(i,j),-alpha,1,1,y,nz)
                        out%double(i,j) = y(1)
                     end if
                  end do
               end do
            end if

         end if

      end if

      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfBesselK", "E",                        &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfBesselK
!_______________________________________________________________________
!
   function mfAiry( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! computes the Airy function, for real argument only.

      integer :: i, j, status

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfAiry", "E",                              &
                            "this function cannot be applied to sparse matrix!" )
         go to 99
      end if

      if( A%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfAiry", "E",                              &
                            "arg. must be real!" )
         go to 99
      end if

      out%shape = a%shape

      out%data_type = MF_DT_DBLE
      allocate( out%double(out%shape(1),out%shape(2)) )

      do j = 1, A%shape(2)
         do i = 1, A%shape(1)
            out%double(i,j) = dai(A%double(i,j))
         end do
      end do

      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfAiry", "E",                           &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfAiry
!_______________________________________________________________________
!
   subroutine msEllipKE( out, A )

      type(mfArray) :: A
      type(mf_Out) :: out
      !------ API end ------

#ifdef _DEVLP
      ! computes the Complete Elliptic Integrals of First and Second Kind,
      ! as in Matlab.

      type(mfArray), pointer :: K, E
      integer :: status, i, j
      real(kind=MF_DOUBLE) :: vk, ve

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      ! we must have two output arguments
      if( out%n /= 2 ) then
         call PrintMessage( "msEllipKE", "E",                           &
                            "two output args required!",               &
                            "syntax is : call mfEllipKE ( mfOut(K,E), A )" )
         go to 99
      end if

      ! internal check for all mfOut args
      if( .not. args_mfout_ok( out, A ) ) then
         call PrintMessage( "msEllipKE", "E",                           &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      K => out%ptr1
      E => out%ptr2
      call msSilentRelease( K, E )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( A%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "msEllipKE", "E",                           &
                            "real array required!" )
         go to 99
      end if

      K%data_type = MF_DT_DBLE
      E%data_type = MF_DT_DBLE
      K%shape = A%shape
      E%shape = A%shape
      allocate( K%double(A%shape(1),A%shape(2)) )

      allocate( E%double(A%shape(1),A%shape(2)) )

      do j = 1, A%shape(2)
         do i = 1, A%shape(1)
            call cei_double( 1.0d0-A%double(i,j), vk, ve )
            K%double(i,j) = vk
            E%double(i,j) = ve
         end do
      end do

      K%prop%symm = A%prop%symm
      E%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "msEllipKE", "E",                        &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end subroutine msEllipKE
!_______________________________________________________________________
!
   function mfIsPrime( A ) result( out )

      type(mfArray), intent(in) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: nrow, ncol, i, j, k, n
      character(len=20) :: string

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfIsPrime", "W",                           &
                            "arg is an empty mfArray!" )
         go to 99
      end if

      if( .not. mfIsReal(A) ) then
         call PrintMessage( "mfIsPrime", "E",                           &
                            "arg must be real!" )
         go to 99
      end if

      if( .not. mfIsDense(A) ) then
         call PrintMessage( "mfIsPrime", "E",                           &
                            "arg must be a dense matrix!" )
         go to 99
      end if

      nrow = A%shape(1)
      ncol = A%shape(2)

      ! all elements in A must be storable in 32-bit integers
      do j = 1, ncol
         do i = 1, nrow
            n = A%double(i,j)
            if( dble(n) /= A%double(i,j) .or. n < 1 ) then
               call PrintMessage( "mfIsPrime", "E",                     &
                                  "all elements in the mfArray must be strictly positive integer!" )
               go to 99
            end if
         end do
      end do

      n = mfMax(mfMax(A))
      if( n > MF_PN_MAX_LIMIT ) then
         write(string,*) MF_PN_MAX_LIMIT
         call PrintMessage( "mfIsPrime", "E",                           &
                            "arg must be less than " // trim(string) )
         go to 99
      end if

      ! if needed, generate prime numbers less than n
      call prime_numbers_initialize(n)

      out%shape = A%shape
      out%data_type = MF_DT_BOOL
      allocate( out%double(nrow,ncol) )

      do j = 1, ncol
         do i = 1, nrow

            n = A%double(i,j)

            out%double(i,j) = 0.0d0
            do k = 1, MF_PN_NB
               if( n == prime_numbers(k) ) then
                  out%double(i,j) = 1.0d0
                  exit
               end if
            end do

         end do
      end do

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfIsPrime
!_______________________________________________________________________
!
   function mfFactor( A ) result( out )

      type(mfArray), intent(in) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! for n = MF_PN_MAX_LIMIT, 31 factors are then sufficient
      ! (the worse case is when n is a power of 2, so 31 factors are sufficient,
      !  because 2^31 = huge(1) > MF_PN_MAX_LIMIT)
      integer :: p(31)

      integer :: i, k, n
      integer(kind=8) :: n_8
      character(len=20) :: string

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfFactor", "W",                            &
                            "arg is an empty mfArray!" )
         go to 99
      end if

      if( .not. mfIsScalar(A) ) then
         call PrintMessage( "mfFactor", "E",                            &
                            "arg must be a scalar mfArray!" )
         go to 99
      end if

      if( .not. mfIsReal(A) ) then
         call PrintMessage( "mfFactor", "E",                            &
                            "arg must be real!" )
         go to 99
      end if

      n = A

      if( n < 1 ) then
         call PrintMessage( "mfFactor", "E",                            &
                            "arg must be greater or equal than 1" )
         go to 99
      end if
      if( n > MF_PN_MAX_LIMIT ) then
         write(string,*) MF_PN_MAX_LIMIT
         call PrintMessage( "mfFactor", "E",                            &
                            "arg must be less than " // trim(string) )
         go to 99
      end if

      ! if needed, generate prime numbers less than n
      call prime_numbers_initialize(n)

      ! working with long integer to avoid loss of precision
      n_8 = n

      k = 0
      do i = 1, MF_PN_NB
         do while( mod(n_8,prime_numbers(i)) == 0 )
            k = k + 1
            p(k) = prime_numbers(i)
            n_8 = n_8/prime_numbers(i)
         end do
         if( n_8 == 1 ) exit
      end do
      call msAssign( out, mfZeros(1,k) )
      do i = 1, k
         out%double(1,i) = p(i)
      end do

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfFactor
!_______________________________________________________________________
!
   subroutine prime_numbers_initialize( n )

      integer, intent(in) :: n
      !------ API end ------

#ifdef _DEVLP
      ! Build or update the list of prime numbers.
      !
      ! Method used is an improved "Sieve of Eratosthenes"
      ! (Matlab R2017a, 2013)

      ! it is recommended to use the smallest memory size
      logical(kind=1), allocatable :: is_prime(:)

      integer :: pn_nb_estim
      integer :: i, imax, j, ii, iijj, k

   !------ end of declarations -- execution starts hereafter  ------

      if( n <= MF_PN_MAX ) return

      if( allocated(prime_numbers) ) deallocate(prime_numbers)

      MF_PN_MAX = n

!### TODO: incremental generation?
      pn_nb_estim = 1.2*MF_PN_MAX/log(real(MF_PN_MAX)) + 2
      allocate( prime_numbers(pn_nb_estim) )

      ! only odd integers are taken into accont
      allocate( is_prime(MF_PN_MAX/2+1) )
      is_prime(:) = .true.

      imax = sqrt(dble(MF_PN_MAX))

      do i = 3, imax, 2
         ii = (i+1)/2
         if( is_prime(ii) ) then
            j = i
            do while( i*j <= MF_PN_MAX )
               iijj = (i*j+1)/2
               is_prime(iijj) = .false.
               j = j + 2
            end do
         end if
      end do

      prime_numbers(1) = 2
      k = 1
      do i = 3, MF_PN_MAX, 2
         ii = (i+1)/2
         if( is_prime(ii) ) then
            k = k + 1
            prime_numbers(k) = i
         end if
      end do
      MF_PN_NB = k

#endif
   end subroutine prime_numbers_initialize
!_______________________________________________________________________
!
   subroutine msCleanPrimeNumbers()
      !------ API end ------

#ifdef _DEVLP
      ! Useful to economize memory

   !------ end of declarations -- execution starts hereafter  ------

      if( allocated(prime_numbers) ) deallocate(prime_numbers)
      MF_PN_MAX = 0
      MF_PN_NB = 0

#endif
   end subroutine msCleanPrimeNumbers
!_______________________________________________________________________
!
   subroutine real_to_rat( n, d, x, tol )

      real(kind=MF_DOUBLE), intent(out) :: n, d
      real(kind=MF_DOUBLE), intent(in)           :: x
      real(kind=MF_DOUBLE), intent(in), optional :: tol
      !------ API end ------

#ifdef _DEVLP
      ! conversion from a real number to its rational form
      ! (after Matlab 7.1 : specfun/rat.m, version 2004/07/05)

      ! the current implementation differs from those of the
      ! 'rational_numbers' module.

      ! Warning, the tolerance test done here concerns the absolute
      ! error. The calling routine (e.g. 'msRref') must compute
      ! the tolerance to be relative to:
      !  - the value of abs(x)
      !  - another value (from a matrix norm), when we work on a matrix

      real(kind=MF_DOUBLE) :: n_old, d_old, n_try, d_try, xx, dd, tol1
      integer*8 :: i8

   !------ end of declarations -- execution starts hereafter  ------

      ! special cases
      if( mf_isinf(x) ) then
         n = sign(1.0d0,x)
         d = 0.0d0
         return
      end if
      if( mf_isnan(x) ) then
         n = 0.0d0
         d = 0.0d0
         return
      end if

      if( present(tol) ) then
         tol1 = max(tol,1.0d-12) ! tol should not be too small
      else
         tol1 = 1.0d-06
      end if

      n_old = 0.0d0
      d_old = 1.0d0

      n = 1.0d0
      d = 0.0d0

      xx = x
      do
         dd = nint(xx,kind=kind(i8)) ! integer part
         xx = xx - dd  ! fractional part
         n_try = n*dd + n_old
         d_try = d*dd + d_old
         if( abs(n_try/d_try - x) < tol1 ) then
            n = n_try
            d = d_try
            exit
         end if
         n_old = n
         d_old = d
         n = n_try
         d = d_try
         if( xx == 0.0d0 ) then
            exit
         end if
         xx = 1.0d0 / xx
      end do

      if( d < 0.0d0 ) then
         d = -d
         n = -n
      end if

#endif
   end subroutine real_to_rat
!_______________________________________________________________________
!
   subroutine msRat( out, A, tol )

      type(mfArray) :: A
      real(kind=MF_DOUBLE), intent(in), optional :: tol
      type(mf_Out) :: out
      !------ API end ------

#ifdef _DEVLP
      ! returns two mfArrays containing the rational approximation
      ! of each element.

      type(mfArray), pointer :: N, D
      integer :: nrow, ncol, i, j, status
      real(kind=MF_DOUBLE) :: xn, xd, tol1, norm1, val

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      ! 2 out-args must be specified
      if( out%n /= 2 ) then
         call PrintMessage( "msRat", "E",                               &
                            "two output args required!",                &
                            "syntax is: call msRat ( mfOut(N,D), A )" )
         go to 99
      end if

      ! internal check for all mfOut args
      if( .not. args_mfout_ok( out, A ) ) then
         call PrintMessage( "msRat", "E",                               &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      N => out%ptr1
      D => out%ptr2
      call msSilentRelease( N, D )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "msRat", "E",                               &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( A%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "msRat", "E",                               &
                            "'A' must be a dense, real matrix!" )
         go to 99
      end if

      nrow = A%shape(1)
      ncol = A%shape(2)

      if( present(tol) ) then
         if( tol <= 0.0d0 ) then
            call PrintMessage( "msRat", "E",                            &
                               "'tol' cannot have a negative value!" )
            go to 99
         else if( tol < 1.0d-12 ) then
            call PrintMessage( "msRat", "W",                            &
                               "'tol' was too small. It will be reset to 1e-12" )
            ! The reset will be done in the 'real_to_rat' routine.
         end if
         tol1 = tol
      else
         ! 'mfNorm(.,1)' is not available here
         norm1 = mfMax(mfAbs(mfGet(A,mfIsFinite(A))))
         tol1 = 1.0d-06 * norm1
      end if

      call msAssign( N, mfZeros(nrow,ncol) ) ! numerator
      call msAssign( D, mfZeros(nrow,ncol) ) ! denominator

      do j = 1, ncol
         do i = 1, nrow
            call real_to_rat( xn, xd, A%double(i,j), tol1 )
            N%double(i,j) = xn
            D%double(i,j) = xd
         end do
      end do

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "msRat", "E",                            &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end subroutine msRat
!_______________________________________________________________________
!
end module mod_specfun
