#ifdef _HAS_LOC
#define _MF_LOC_ANY_OBJ_ loc
#else
#define _MF_LOC_ANY_OBJ_ mf_loc_any_obj
#endif

module mod_sparse ! Sparse matrices

   ! Part of MUESLI Graphic Library
   ! Copyright É. Canot 2003-2025 -- IPR/CNRS

!-----------------------------------------------------------------------
!                             used modules
!-----------------------------------------------------------------------

   use mod_core

#ifndef _DEVLP
   use mod_mfarray ! required for the 2nd pass of the double compilation
#endif

   implicit none

#ifndef _DEVLP
   private
#endif

!-----------------------------------------------------------------------
!                             interfaces
!-----------------------------------------------------------------------

! "misc/suitesparse/umf4_f90wrapper.inc" already included via modules

   interface mfSparse
      module procedure mfSparse_mfArray
   end interface mfSparse
   !------ API end ------

   interface mfSpImport
      module procedure mfSpImport_real
      module procedure mfSpImport_cmplx
   end interface mfSpImport
   !------ API end ------

   interface msSpExport
      module procedure msSpExport_real
      module procedure msSpExport_cmplx
   end interface msSpExport
   !------ API end ------

   interface msSpReAlloc
      module procedure msSpReAlloc_int_int
      module procedure msSpReAlloc_minimal
   end interface msSpReAlloc
   !------ API end ------

   interface mfSpDiags
      module procedure mfSpDiags_vec
      module procedure mfSpDiags_vec_one_arg
      module procedure mfSpDiags_mat
   end interface mfSpDiags
   !------ API end ------

   interface msRelease
      module procedure msFreeMatFactor
   end interface msRelease
   !------ API end ------

   public :: mfSparse, &
             mfSpImport, &
             msSpExport, &
             mfSpAlloc, &
             msSpReAlloc, &
             mfNzmax, &
             mfNcolmax, &
             mfSpEye, &
             mfSpOnes, &
             mfSpDiags, &
             mfIsRowSorted, &
             msGetAutoRowSorted, &
             msSetAutoRowSorted, &
             msFreeMatFactor, &
             mfSpCut

   ! rem:    actually in the mf_core module :
   !         mfNnz, msRowSort

   logical :: MF_SP_AUTO_ROW_SORTED = .true.

   ! ordering in SuiteSparse/CHOLMOD
   ! (must match const. defined in
   !      misc/suitesparse/CHOLMOD/Include/cholmod_core.h)
   integer, parameter :: MF_CHOLMOD_NATURAL = 0, &
                         MF_CHOLMOD_GIVEN   = 1, &
                         MF_CHOLMOD_AMD     = 2, &
                         MF_CHOLMOD_METIS   = 3, &
                         MF_CHOLMOD_NESDIS  = 4, &
                         MF_CHOLMOD_COLAMD  = 5

   private :: mfSparse_mfArray, &
              mfSpImport_real, &
              mfSpImport_cmplx, &
              msSpExport_real, &
              msSpExport_cmplx, &
              msSpReAlloc_int_int, &
              msSpReAlloc_minimal, &
              mfSpDiags_vec, &
              mfSpDiags_vec_one_arg, &
              mfSpDiags_mat, &
              process_dupl_entries_struct_only

contains
!_______________________________________________________________________
!
#include "fml_sparse/Sparse.inc"
!_______________________________________________________________________
!
#include "fml_sparse/Import.inc"
!_______________________________________________________________________
!
#include "fml_sparse/Export.inc"
!_______________________________________________________________________
!
#include "fml_sparse/SpEye.inc"
!_______________________________________________________________________
!
#include "fml_sparse/SpOnes.inc"
!_______________________________________________________________________
!
#include "fml_sparse/SpDiags.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/prtmt.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/readmt.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/amib.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/amib1.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/amux.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/aplb.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/aplb1.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/aplib.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/aplib1.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/apply_bin_ops_to_sp_dble.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/nnzaplb.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/coo2csc.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/coo2csc_struct_only.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/dns2csc.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/transp.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/xmua.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/spcut.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/trilsol.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/triusol.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/normcols.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/sqbanded2csc.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/sp_del_0_and_compact.inc"
!_______________________________________________________________________
!
   subroutine msGetAutoRowSorted( auto_row_sorted )

      logical, intent(out) :: auto_row_sorted
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      auto_row_sorted = MF_SP_AUTO_ROW_SORTED

#endif
   end subroutine msGetAutoRowSorted
!_______________________________________________________________________
!
   subroutine msSetAutoRowSorted( auto_row_sorted )

      logical, intent(in) :: auto_row_sorted
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      MF_SP_AUTO_ROW_SORTED = auto_row_sorted

#endif
   end subroutine msSetAutoRowSorted
!_______________________________________________________________________
!
   function mfSpAlloc( m, n, nzmax, ncolmax, kind ) result( out )

      integer, intent(in) :: m, n
      integer, intent(in), optional :: nzmax, ncolmax
      character(len=*), intent(in), optional :: kind
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: nnz, nrow, ncol, njmax
      character(len=7) :: kind0

   !------ end of declarations -- execution starts hereafter  ------

      nrow = m
      ncol = n

      if( present(nzmax) ) then
         nnz = nzmax
      else
         nnz = 0
      end if

      if( present( ncolmax ) ) then
         njmax = ncolmax+1
      else
         njmax = ncol+1
      end if

      if( present( kind ) ) then
         kind0 = to_lower(kind)
      else
         kind0 = "real"
      end if

      out%shape = [ nrow, ncol ]

      if( kind0 == "real" ) then
         out%data_type = MF_DT_SP_DBLE
         allocate( out%a(nnz) )

      else if( kind0 == "complex" ) then
         out%data_type = MF_DT_SP_CMPLX
         allocate( out%z(nnz) )

      else
         call PrintMessage( "mfSpAlloc", "E",                           &
                            "optional arg 'kind' must be equal",        &
                            "to 'real' or 'complex'!" )
         out%data_type = MF_DT_EMPTY
      end if
      allocate( out%i(nnz) )

      allocate( out%j(njmax) )

      out%j(1:ncol+1) = 1
      out%row_sorted = UNKNOWN
      out%status_temporary = .true.

#endif
   end function mfSpAlloc
!_______________________________________________________________________
!
   subroutine msSpReAlloc_int_int( A, nzmax, ncolmax )

      type(mfArray) :: A
      integer, intent(in), optional :: nzmax
      integer, intent(in), optional :: ncolmax
      !------ API end ------

#ifdef _DEVLP
      integer :: ncol, nnz, nzmax_new, ncolmax_new
      integer :: nzmax_e

   !------ end of declarations -- execution starts hereafter  ------

      if( (.not.present(nzmax)) .and. (.not.present(ncolmax)) ) then
         call PrintMessage( "mfSpReAlloc", "E",                         &
                            "at least one of the optional arg 'nzmax'", &
                            "and 'ncolmax' must be present!" )
         return
      end if

      if( A%status_temporary ) then
         call PrintMessage( "mfSpReAlloc", "E",                         &
                            "cannot realloc a tempo sparse matrix!" )
         return
      end if

      if( .not. mfIsSparse(A) ) then
         call PrintMessage( "mfSpReAlloc", "E",                         &
                            "first arg is not a sparse matrix!" )
         return
      end if

      if( A%parameter ) then
         call PrintMessage( "msSpReAlloc", "E",                         &
                            "'A' is a protected array (pseudo-parameter)." )
         return
      end if

      ncol = A%shape(2)
      nnz = A%j(ncol+1) - 1

      if( present(nzmax) ) then
         nzmax_e = nzmax
      else
         nzmax_e = nnz
      end if

      if( nzmax_e < nnz ) then
         call PrintMessage( "mfSpReAlloc", "E",                         &
                            "nzmax too small!" )
         return
      end if
      nzmax_new = nzmax_e

      if( present(ncolmax) ) then
         if( ncolmax < ncol ) then
            call PrintMessage( "mfSpReAlloc", "E",                      &
                               "ncolmax too small!" )
            return
         end if
         ncolmax_new = ncolmax+1
         call realloc_array( A%j, ncolmax_new, ncol+1 )
      end if

      if( A%data_type == MF_DT_SP_DBLE ) then
         call realloc_array( A%a, nzmax_new, nnz )
      else if( A%data_type == MF_DT_SP_CMPLX ) then
         call realloc_array( A%z, nzmax_new, nnz )
      else
         call PrintMessage( "mfSpReAlloc", "E",                         &
                            "unknown data type for A!" )
         return
      end if
      call realloc_array( A%i, nzmax_new, nnz )

#endif
   end subroutine msSpReAlloc_int_int
!_______________________________________________________________________
!
   subroutine msSpReAlloc_minimal( A, string )

      type(mfArray) :: A
      character(len=*), intent(in) :: string
      !------ API end ------

#ifdef _DEVLP
      integer :: ncol, nnz, nb_zeros

   !------ end of declarations -- execution starts hereafter  ------

      if( A%status_temporary ) then
         call PrintMessage( "mfSpReAlloc", "E",                         &
                            "cannot realloc a tempo sparse matrix!" )
         return
      end if

      call msInitArgs( A )

      if( .not. mfIsSparse(A) ) then
         call PrintMessage( "mfSpReAlloc", "E",                         &
                            "first arg is not a sparse matrix!" )
         go to 99
      end if

      if( A%parameter ) then
         call PrintMessage( "msSpReAlloc", "E",                         &
                            "'A' is a protected array (pseudo-parameter)." )
         go to 99
      end if

      if( to_lower(trim(string)) /= "minimal" ) then
         call PrintMessage( "mfSpReAlloc", "E",                         &
                            "second arg must be equal to 'minimal'!" )
         go to 99
      end if

      ncol = A%shape(2) ! the logical size
      nnz = A%j(ncol+1) - 1

      ! Check the presence of unwanted zeros
      if( A%data_type == MF_DT_SP_DBLE ) then
         nb_zeros = count( A%a(1:nnz) == 0.0d0 )
         if( nb_zeros /= 0 ) then
            call sp_del_0_and_compact_r8( ncol, A%a, A%i, A%j, nb_zeros )
            go to 99
         end if
      else if( A%data_type == MF_DT_SP_CMPLX ) then
         nb_zeros = count( A%z(1:nnz) == (0.0d0,0.0d0) )
         if( nb_zeros /= 0 ) then
            call sp_del_0_and_compact_z8( ncol, A%z, A%i, A%j, nb_zeros )
            go to 99
         end if
      end if

      if( size(A%j) > ncol+1 ) then
         call realloc_array( A%j, ncol+1, ncol+1 )
      end if

      if( size(A%i) > nnz ) then
         call realloc_array( A%i, nnz, nnz )
      else
         ! actually nothing else to do
         go to 99
      end if

      if( A%data_type == MF_DT_SP_DBLE ) then
         call realloc_array( A%a, nnz, nnz )
      else if( A%data_type == MF_DT_SP_CMPLX ) then
         call realloc_array( A%z, nnz, nnz )
      end if

 99   continue

      call msFreeArgs( A )

#endif
   end subroutine msSpReAlloc_minimal
!_______________________________________________________________________
!
   function mfNzmax( A ) result( out )

      type(mfArray) :: A
      integer       :: out
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      out = 0

      if( A%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "mfNzmax", "W",                             &
                            "arg. is empty!" )
         return
      end if

      call msInitArgs( A )

      if( mfIsSparse(A) ) then
         out = size(A%i)
      else
         call PrintMessage( "mfNzmax", "E",                             &
                            "arg must be a sparse matrix!" )
      end if

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfNzmax
!_______________________________________________________________________
!
   function mfNcolmax( A ) result( out )

      type(mfArray) :: A
      integer       :: out
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      out = 0

      if( A%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "mfNcolmax", "W",                           &
                            "arg. is empty!" )
         return
      end if

      call msInitArgs( A )

      if( mfIsSparse(A) ) then
         out = size(A%j) - 1
      else
         call PrintMessage( "mfNcolmax", "E",                           &
                            "arg must be a sparse matrix!" )
      end if

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfNcolmax
!_______________________________________________________________________
!
   function mfIsRowSorted( A ) result( bool )

      type(mfArray) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      bool = .false.

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfIsRowSorted", "W",                       &
                            "mfArray is empty!" )
         go to 99
      end if

      if( .not. mfIsSparse(A) ) then
         call PrintMessage( "mfIsRowSorted", "E",                       &
                            "mfArray must be sparse!" )
         go to 99
      end if

      if( A%row_sorted == TRUE ) then
         bool = .true.
      else if( A%row_sorted == UNKNOWN ) then
         bool = mfCheckRowSorted(A)
      end if

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfIsRowSorted
!_______________________________________________________________________
!
   function mfCheckRowSorted( A ) result( bool )

      type(mfArray) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      ! internal routine

      integer :: ncol

   !------ end of declarations -- execution starts hereafter  ------

      ncol = A%shape(2)

      ! on met à jour le flag 'row_sorted' de A
      if( row_sorted(ncol,A%i,A%j) ) then
         A%row_sorted = TRUE
         bool = .true.
      else
         if( A%row_sorted == TRUE ) then
            write(STDERR,*)
            write(STDERR,*) "(MUESLI mfCheckRowSorted:) internal error:"
            write(STDERR,*) "                           sparse mfArray is actually not row sorted,"
            write(STDERR,*) "                           but was tagged as sorted!"
            write(STDERR,*) "                           Please report this bug to: Edouard.Canot@univ-rennes.fr"
            mf_message_displayed = .true.
            call muesli_trace( pause ="yes" )
            stop
         end if
         A%row_sorted = FALSE
         bool = .false.
      end if

#endif
   end function mfCheckRowSorted
!_______________________________________________________________________
!
   subroutine msFreeMatFactor( x )

      type(mfMatFactor) :: x
      !------ API end ------

#ifdef _DEVLP
#include "misc/suitesparse/spqr_f90wrapper.inc"

   !------ end of declarations -- execution starts hereafter  ------

      if( associated(x%mf_ptr_1) ) then
         call msSilentRelease( x%mf_ptr_1 ) ! L
         deallocate(x%mf_ptr_1)

         call msSilentRelease( x%mf_ptr_2 ) ! U
         deallocate(x%mf_ptr_2)

         call msSilentRelease( x%mf_ptr_3 ) ! P
         deallocate(x%mf_ptr_3)

      else if( associated(x%ptr_1) ) then
         if( x%package == 0 ) then ! 0 for empty
            write(STDERR,*)
            write(STDERR,*) "(MUESLI msFreeMatFactor:) internal error:"
            write(STDERR,*) "                          mfMatFactor 'x' is empty, but x%ptr_1 is associated!"
            write(STDERR,*) "                          Please report this bug to: Edouard.Canot@univ-rennes.fr"
            mf_message_displayed = .true.
            call muesli_trace( pause ="yes" )
            stop
         else if( x%package == 1 ) then ! UMFPACK
            call umf4fnum_d( x%ptr_1 )
            deallocate( x%ptr_1 )

         else if( x%package == 2 ) then ! CHOLMOD
            call cholmod_free_factor_f90( x%ptr_1, x%ptr_2 )
            deallocate( x%ptr_1 )

            deallocate( x%ptr_2 )

         else if( x%package == 3 ) then ! SPQR
            call spqr_free_sparse( x%ptr_1, x%ptr_2 ) ! H
            deallocate( x%ptr_2 )

            call spqr_free_dense( x%ptr_1, x%ptr_3 ) ! HTau
            deallocate( x%ptr_3 )

            call spqr_free_long_vec( x%ptr_4 ) ! HPinv
            deallocate( x%ptr_4 )

            call spqr_free_common( x%ptr_1 ) ! Common
            deallocate( x%ptr_1 )

         else if( x%package == 4 ) then ! CHOLMOD special
            call cholmod_free_factor_2( x%ptr_1, x%ptr_2, x%ptr_3 )
            deallocate( x%ptr_1 )

            deallocate( x%ptr_2 )

            deallocate( x%ptr_3 )

         else
            write(STDERR,*)
            write(STDERR,*) "(MUESLI msFreeMatFactor:) internal error:"
            write(STDERR,*) "                          unknown value for package field!"
            write(STDERR,*) "                          Please report this bug to: Edouard.Canot@univ-rennes.fr"
            mf_message_displayed = .true.
            call muesli_trace( pause ="yes" )
            stop
         end if
      end if

      x%package = 0
      x%data_type = MF_DT_EMPTY
      x%order = 0

#endif
   end subroutine msFreeMatFactor
!_______________________________________________________________________
!
   function mfSpCut( A, threshold ) result( out )

      type(mfArray) :: A
      real(kind=MF_DOUBLE) :: threshold
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: nrow, ncol, nnz

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( .not. mfIsSparse(A) ) then
         call PrintMessage( "mfSpCut", "E",                             &
                            "arg 'A' must be a sparse matrix!" )
         go to 99
      end if

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfSpCut", "E",                             &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( threshold <= 0.0d0 ) then
         call PrintMessage( "mfSpCut", "W",                             &
                            "threshold <= 0",                           &
                            "[-> the output will be equal to A]" )
         out = A
         out%status_temporary = .true.
         go to 99
      end if

      nrow = A%shape(1)
      ncol = A%shape(2)

      out%data_type = A%data_type
      out%shape = [ nrow, ncol ]
      if( A%data_type == MF_DT_SP_DBLE ) then
         call spcut_nnz( ncol, A%a, A%i, A%j, threshold, nnz )
         allocate( out%a(nnz) )

      else ! A%data_type == MF_DT_SP_CMPLX
         call spcut_nnz_cmplx( ncol, A%z, A%i, A%j, threshold, nnz )
         allocate( out%z(nnz) )

      end if

      allocate( out%i(nnz) )

      allocate( out%j(ncol+1) )

      if( A%data_type == MF_DT_SP_DBLE ) then
         call spcut( ncol, A%a, A%i, A%j, threshold,                    &
                     out%a, out%i, out%j )
      else ! A%data_type == MF_DT_SP_CMPLX
         call spcut_cmplx( ncol, A%z, A%i, A%j, threshold,              &
                     out%z, out%i, out%j )
      end if

      if( A%prop%tril == TRUE ) then
         out%prop%tril = TRUE
      end if
      if( A%prop%triu == TRUE ) then
         out%prop%triu = TRUE
      end if

      out%row_sorted = A%row_sorted
      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )

#endif
   end function mfSpCut
!_______________________________________________________________________
!
   subroutine process_dupl_entries_struct_only( A, routine_name )

      type(mfArray)                :: A
      character(len=*), intent(in) :: routine_name
      !------ API end ------

#ifdef _DEVLP
      ! A is supposed to be sparse
      !
      ! Matrix A is always sorted and checked for duplicated entries;
      ! Processing of duplicated entries is then made.
      !
      ! At the end of this routine, A is "row sorted".

      integer :: nrow, ncol, nb
      character(len=15) :: nb_char
      logical :: unique

   !------ end of declarations -- execution starts hereafter  ------

      nrow = A%shape(1)
      ncol = A%shape(2)

      if( .not. mfIsRowSorted(A) ) then
         ! new mfArray must be sorted
         call row_sort_struct_only( nrow, ncol, A%i, A%j )
         A%row_sorted = TRUE
      end if
      unique = unique_entries( ncol, A%i, A%j )
      if( .not. unique ) then
         ! duplicated entries are discarded
         call remove_dupl_entries_struct_only( ncol, A%i, A%j, nb )
         write(nb_char,"(1X,I0)") nb
         call PrintMessage( routine_name, "I",                          &
                            trim(adjustl(nb_char)) // " duplicated entries (structure only)", &
                            "have been discarded." )
      end if

#endif
   end subroutine process_dupl_entries_struct_only
!_______________________________________________________________________
!
   subroutine process_dupl_entries( A, work_type, routine_name )

      type(mfArray)                :: A
      character(len=*), intent(in) :: work_type, routine_name
      !------ API end ------

#ifdef _DEVLP
      ! A is supposed to be sparse
      !
      ! Matrix A is always sorted and checked for duplicated entries;
      ! Processing of duplicated entries is then made, according the
      ! 'work_type' argument.
      !
      ! At the end of this routine, A is "row sorted".

      integer :: nrow, ncol, nb
      character(len=15) :: nb_char
      logical :: unique

   !------ end of declarations -- execution starts hereafter  ------

      nrow = A%shape(1)
      ncol = A%shape(2)

      if( .not. mfIsRowSorted(A) ) then
         ! new mfArray must be sorted
         if( A%data_type == MF_DT_SP_DBLE ) then
            call row_sort( nrow, ncol, A%a, A%i, A%j )
         else ! complex
            call row_sort_cmplx( nrow, ncol, A%z, A%i, A%j )
         end if
         A%row_sorted = TRUE
      end if
      unique = unique_entries( ncol, A%i, A%j )
      if( .not. unique ) then
         if( work_type == "ignored" ) then
            ! duplicated entries are discarded
            if( A%data_type == MF_DT_SP_DBLE ) then
               call remove_dupl_entries(ncol,A%a,A%i,A%j,nb)
            else ! complex
               call remove_dupl_entries_cmplx(ncol,A%z,A%i,A%j,nb)
            end if
            write(nb_char,"(1X,I0)") nb
            call PrintMessage( routine_name, "I",                       &
                               trim(adjustl(nb_char)) // " duplicated entries have been removed." )
         else if( work_type == "replaced" ) then
            ! duplicated entries are replaced
            if( A%data_type == MF_DT_SP_DBLE ) then
               call replace_dupl_entries(ncol,A%a,A%i,A%j,nb)
            else ! complex
               call replace_dupl_entries_cmplx(ncol,A%z,A%i,A%j,nb)
            end if
            write(nb_char,"(1X,I0)") nb
            call PrintMessage( routine_name, "I",                       &
                               trim(adjustl(nb_char)) // " duplicated entries have replaced previous ones." )
         else ! "added" [default]
            ! duplicated entries are added
            if( A%data_type == MF_DT_SP_DBLE ) then
               call add_dupl_entries(ncol,A%a,A%i,A%j,nb)
            else ! complex
               call add_dupl_entries_cmplx(ncol,A%z,A%i,A%j,nb)
            end if
            write(nb_char,"(1X,I0)") nb
            call PrintMessage( routine_name, "I",                       &
                               trim(adjustl(nb_char)) // " duplicated entries have been added." )
         end if
      end if

#endif
   end subroutine process_dupl_entries
!_______________________________________________________________________
!
end module mod_sparse
