#if defined _GNU_GFC
! convert (3,2,0) to 30200
#define GCC_VERSION (__GNUC__*10000+__GNUC_MINOR__*100+__GNUC_PATCHLEVEL__)
#endif

module mod_mfaux ! routines auxiliaires

   ! Part of MUESLI Numerical Library
   ! Copyright É. Canot 2003-2025 -- IPR/CNRS

   use mod_mem_debug

   use mod_mfdebug, only: STDERR ! STDERR has been excluded in mod_mem_debug

   use convexhull_mod

#if GCC_VERSION >= 40900
   use iso_C_binding, only : C_int, C_ptr, C_loc, C_sizeof
#else
   ! While Gfortran 4.8 support the iso_C_binding intrinsic module, it is
   ! not mature enough to be employed in this case...
#endif

   implicit none

#ifndef _DEVLP
   private
#endif

   interface quick_sort
      module procedure quick_sort_1      ! ( real A(:) )
      module procedure quick_sort_2      ! ( real A(:), real ind(:) )
      module procedure quick_sort_3      ! ( real A(:), integer ind(:) )
      module procedure quick_sort_1_int  ! ( integer A(:) )
#ifdef _64_BITS
      module procedure quick_sort_3_int8 ! ( integer A(:), integer ind(:) )
#else
      module procedure quick_sort_3_int  ! ( integer A(:), integer ind(:) )
#endif
   end interface quick_sort
   !------ API end ------

   private :: quick_sort_2,              &
              quick_sort_3_int,          &
              quick_sort_3_int8,         &
              partition_sort_1_asc,      &
              partition_sort_1_des,      &
              partition_sort_2_asc,      &
              partition_sort_2_des,      &
              partition_sort_3_asc,      &
              partition_sort_3_des,      &
              partition_sort_1_int_asc,  &
              partition_sort_1_int_des,  &
              partition_sort_3_int_asc,  &
              partition_sort_3_int_des,  &
              partition_sort_3_int8_asc, &
              partition_sort_3_int8_des

   public :: quick_sort_1_int ! used in [FGL]mf/msTriMesh
   public :: quick_sort_1     ! used in mfUnique
   public :: quick_sort_3     ! used in mfUnique and [FGL]pg2dproj_maty

   interface no_duplicates
#ifdef _64_BITS
      module procedure no_duplicates_int8
#else
      module procedure no_duplicates_int
#endif
   end interface no_duplicates

   private :: no_duplicates_int, &
              no_duplicates_int8

   ! index sequence definition
   !
   ! provides :  [.from.] start .to. end            (step=1)
   !             [.from.] start .to. end .by. step
   !             [.from.] start .by. step .to. end
   !                      seq   .but. i
   !                      seq_1 .and. seq_2
   !                      seq_1 .and. i
   !                          i .and. seq_2

   type :: seq_def
#ifndef _DEVLP
      private
#endif
      integer :: start_1 = 1
      integer :: end_1   = 0
      integer :: step_1  = 1
      logical :: start_1_EndIndex = .false.
      logical :: end_1_EndIndex = .false.

      integer :: start_2 = 1
      integer :: end_2   = 0
      integer :: step_2  = 1
      logical :: start_2_EndIndex = .false.
      logical :: end_2_EndIndex = .false.

      integer :: but = 0
      logical :: but_present = .false.
      logical :: but_EndIndex = .false.

   end type seq_def

   interface operator(.from.)
      module procedure seq_from
   end interface
   !------ API end ------

   interface operator(.to.)
      module procedure seq_from_to
      module procedure seq_range_to
   end interface
   !------ API end ------

   ! this operator name is kept for compatibility. New prefered name is .by.
   interface operator(.step.)
      module procedure seq_range_by
      module procedure seq_from_by
   end interface
   !------ API end ------

   interface operator(.by.)
      module procedure seq_range_by
      module procedure seq_from_by
   end interface
   !------ API end ------

   interface operator(.but.)
      module procedure seq_range_but
   end interface
   !------ API end ------

   interface operator(.and.)
      module procedure seq_range_and_range
      module procedure seq_range_and_int
      module procedure seq_int_and_range
   end interface
   !------ API end ------

   private :: seq_from, &
              seq_from_to, &
              seq_range_to, &
              seq_range_by, &
              seq_from_by, &
              seq_range_but, &
              seq_range_and_range, &
              seq_range_and_int, &
              seq_int_and_range

   public :: seq_def,                                                   &
             operator(.from.),                                          &
             operator(.to.),                                            &
             operator(.step.),                                          &
             operator(.by.),                                            &
             operator(.but.),                                           &
             operator(.and.)

   public :: STDERR

   public :: mf_save_and_disable_fpe,                                   &
             mf_restore_fpe
   ! default: full trapping is disabled (i.e. FP exceptions are
   ! detected in the user part of the program, not in the MUESLI
   ! library)
   logical, save :: mf_full_trapping = .false.

   integer, save :: fpu_flags_saved
   ! call to 'mf_save_and_disable_fpe()' can be nested: the following
   ! variable allows us to do one disabling only!
   integer, save :: fpe_nested_call = 0

#if GCC_VERSION >= 40900
   interface
      function icrc32( addr, siz ) bind(C, name="icrc32_")
         import C_int, C_ptr
         type(C_ptr)         :: addr
         integer(kind=C_int) :: icrc32
         integer(kind=C_int) :: siz
      end function
   end interface
#else
   ! -> use the legacy declaration, without interface checking
   integer, external :: icrc32 ! 32 bits CRC routine
#endif

   interface crc_32_bits
      module procedure crc32bits_vec_int
      module procedure crc32bits_vec_dble
      module procedure crc32bits_mat_dble
      module procedure crc32bits_vec_cmplx
      module procedure crc32bits_mat_cmplx
   end interface
   !------ API end ------

   private :: crc32bits_vec_int, &
              crc32bits_vec_dble, &
              crc32bits_mat_dble, &
              crc32bits_vec_cmplx, &
              crc32bits_mat_cmplx

   interface sec_2_hms
      module procedure sec_int_2_hms
      module procedure sec_dp_2_hms
   end interface
   !------ API end ------

   private :: sec_int_2_hms, &
              sec_dp_2_hms

   interface mfToLower
      module procedure to_lower
   end interface

   interface mfToUpper
      module procedure to_upper
   end interface

   interface msFindIOUnit
      module procedure find_unit
   end interface

   public :: mfToLower, mfToUpper, msFindIOUnit

   type, public :: mf_Int_List
      integer, allocatable :: list(:)
   end type mf_Int_List

   type, public :: mf_Real_List
      real(kind=MF_DOUBLE), allocatable :: list(:)
   end type mf_Real_List

   ! definition of a derived type which embbed a function pointer
   type :: func_ptr
      procedure(param_transf), pointer, nopass :: ptr => NULL()
   end type func_ptr

   interface
      function param_transf( x ) result( y )
         import :: MF_DOUBLE
         real(kind=MF_DOUBLE) :: x, y
      end function param_transf
   end interface

   interface mf_max_omitNaN
      module procedure mf_max_omitNaN_scal_scal
      module procedure mf_max_omitNaN_mat_scal
      module procedure mf_max_omitNaN_mat_mat
   end interface

   interface mf_max_includeNaN
      module procedure mf_max_includeNaN_scal_scal
      module procedure mf_max_includeNaN_mat_scal
      module procedure mf_max_includeNaN_mat_mat
   end interface

   interface mf_min_omitNaN
      module procedure mf_min_omitNaN_scal_scal
      module procedure mf_min_omitNaN_mat_scal
      module procedure mf_min_omitNaN_mat_mat
   end interface

   interface mf_min_includeNaN
      module procedure mf_min_includeNaN_scal_scal
      module procedure mf_min_includeNaN_mat_scal
      module procedure mf_min_includeNaN_mat_mat
   end interface

   interface ms_maxval_loc_omitNaN
      module procedure ms_maxval_loc_int_omitNaN
      module procedure ms_maxval_loc_r8_omitNaN
   end interface

   interface ms_maxval_loc_includeNaN
      module procedure ms_maxval_loc_int_includeNaN
      module procedure ms_maxval_loc_r8_includeNaN
   end interface

   interface ms_minval_loc_omitNaN
      module procedure ms_minval_loc_int_omitNaN
      module procedure ms_minval_loc_r8_omitNaN
   end interface

   interface ms_minval_loc_includeNaN
      module procedure ms_minval_loc_int_includeNaN
      module procedure ms_minval_loc_r8_includeNaN
   end interface

   public :: mf_max_omitNaN, mf_max_includeNaN,                         &
             mf_min_omitNaN, mf_min_includeNaN,                         &
             mf_maxval_omitNaN, mf_maxval_includeNaN,                   &
             mf_minval_omitNaN, mf_minval_includeNaN,                   &
             ms_maxval_loc_omitNaN, ms_maxval_loc_includeNaN,           &
             ms_minval_loc_omitNaN, ms_minval_loc_includeNaN,           &
             full_chk_der_ode, full_chk_der_dae

   ! For quick_sortrows
   interface quick_sortrows
      module procedure quick_sortrows_1     ! ( real A(:,:) )
      module procedure quick_sortrows_2     ! ( real A(:,:), real ind(:) )
      module procedure quick_sortrows_3     ! ( real A(:,:), integer ind(:) )
      module procedure quick_sortrows_1_int ! ( integer A(:,:) )
   end interface quick_sortrows
   !------ API end ------

   ! the following symbols must be available in the 'datafun' module.
   integer, public :: size_col_ind, sortrows_nrows
   integer, public, allocatable :: col_ind(:)

   private :: is_EQ_rows, is_EQ_rows_int, is_GT_rows, is_GT_rows_int,   &
              is_GE_rows, is_GE_rows_int,                               &
              is_GT_rows_work, is_GT_rows_int_work,                     &
              is_GE_rows_work, is_GE_rows_int_work

   public :: check_and_remove_trailing_zero

contains
!_______________________________________________________________________
!
#include "fml_core/min_max_nan.inc"
!_______________________________________________________________________
!
#include "fml_core/quick_sort.inc"
!_______________________________________________________________________
!
#include "fml_core/tri.inc"
!_______________________________________________________________________
!
#include "fml_core/quick_sortrows.inc"
!_______________________________________________________________________
!
#include "fml_core/tetra.inc"
!_______________________________________________________________________
!
#include "fml_core/seq_def.inc"
!_______________________________________________________________________
!
   function rank_1_to_2_int( tab, dim_out_1, dim_out_2 ) result( out )

      integer :: dim_out_1, dim_out_2
      integer, target :: tab(dim_out_1,dim_out_2)
      integer, pointer :: out(:,:)
      !------ API end ------

#ifdef _DEVLP
      ! implementation of:
      !
      !                out(:,:) => tab(:)
      !
      ! actual argument 'tab' is a rank-1 array in the calling routine
      ! (this rank-remapping should be obtained in F2003)

   !------ end of declarations -- execution starts hereafter  ------

      out => tab

#endif
   end function rank_1_to_2_int
!_______________________________________________________________________
!
   function rank_1_to_2_real8( tab, dim_out_1, dim_out_2 ) result( out )

      integer :: dim_out_1, dim_out_2
      real(kind=MF_DOUBLE), target :: tab(dim_out_1,dim_out_2)
      real(kind=MF_DOUBLE), pointer :: out(:,:)
      !------ API end ------

#ifdef _DEVLP
      ! implementation of:
      !
      !                out(:,:) => tab(:)
      !
      ! actual argument 'tab' is a rank-1 array in the calling routine
      ! (this rank-remapping should be obtained in F2003)

   !------ end of declarations -- execution starts hereafter  ------

      out => tab

#endif
   end function rank_1_to_2_real8
!_______________________________________________________________________
!
   function rank_1_to_2_cmplx( tab, dim_out_1, dim_out_2 ) result( out )

      integer :: dim_out_1, dim_out_2
      complex(kind=MF_DOUBLE), target :: tab(dim_out_1,dim_out_2)
      complex(kind=MF_DOUBLE), pointer :: out(:,:)
      !------ API end ------

#ifdef _DEVLP
      ! implementation of:
      !
      !                out(:,:) => tab(:)
      !
      ! actual argument 'tab' is a rank-1 array in the calling routine
      ! (this rank-remapping should be obtained in F2003)

   !------ end of declarations -- execution starts hereafter  ------

      out => tab

#endif
   end function rank_1_to_2_cmplx
!_______________________________________________________________________
!
   function rank_2_to_1_int( tab, dim_out ) result( out )

      integer :: dim_out
      integer, target :: tab(dim_out)
      integer, pointer :: out(:)
      !------ API end ------

#ifdef _DEVLP
      ! implementation of:
      !
      !                out(:) => tab(:,:)
      !
      ! actual argument 'tab' is a rank-2 array in the calling routine
      ! (this special rank-remapping cannot be obtained in F2003)

   !------ end of declarations -- execution starts hereafter  ------

      out => tab

#endif
   end function rank_2_to_1_int
!_______________________________________________________________________
!
   function rank_2_to_1_real8( tab, dim_out ) result( out )

      integer :: dim_out
      real(kind=MF_DOUBLE), target :: tab(dim_out)
      real(kind=MF_DOUBLE), pointer :: out(:)
      !------ API end ------

#ifdef _DEVLP
      ! implementation of:
      !
      !                out(:) => tab(:,:)
      !
      ! actual argument 'tab' is a rank-2 array in the calling routine
      ! (this special rank-remapping cannot be obtained in F2003)

   !------ end of declarations -- execution starts hereafter  ------

      out => tab

#endif
   end function rank_2_to_1_real8
!_______________________________________________________________________
!
   function rank_2_to_1_cmplx( tab, dim_out ) result( out )

      integer :: dim_out
      complex(kind=MF_DOUBLE), target :: tab(dim_out)
      complex(kind=MF_DOUBLE), pointer :: out(:)
      !------ API end ------

#ifdef _DEVLP
      ! implementation of:
      !
      !                out(:) => tab(:,:)
      !
      ! actual argument 'tab' is a rank-2 array in the calling routine
      ! (this special rank-remapping cannot be obtained in F2003)

   !------ end of declarations -- execution starts hereafter  ------

      out => tab

#endif
   end function rank_2_to_1_cmplx
!_______________________________________________________________________
!
   function index_in_vector( ind, v ) result( bool )

      integer, intent(in) :: ind
      integer, intent(in) :: v(:)
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( any( ind == v(:) ) ) then
         bool = .true.
      else
         bool = .false.
      end if

#endif
   end function index_in_vector
!_______________________________________________________________________
!
   subroutine find_unit( unit )

      integer, intent(out) :: unit
      !------ API end ------

#ifdef _DEVLP
      integer :: err, i
      logical :: tf1, tf2

   !------ end of declarations -- execution starts hereafter  ------

      do i = 99, 1, -1
         unit = i
         inquire( UNIT=unit, OPENED=tf1, EXIST=tf2, IOSTAT=err )
         if( .not. tf1 .and. tf2 .and. err==0 ) then
           return
         end if
      end do
      unit = -1

#endif
   end subroutine find_unit
!_______________________________________________________________________
!
   function to_lower( keyword )

      character(len=*), intent(in) :: keyword
      character(len=len_trim(keyword)) :: to_lower
      !------ API end ------

#ifdef _DEVLP
      ! just for ASCII-7 bits
      integer :: ishift
      integer :: i, ipos
      character :: c

   !------ end of declarations -- execution starts hereafter  ------

      ishift = ichar("a")-ichar("A")
      to_lower = ""

      do i = 1, len_trim(keyword)
         c = keyword(i:i)
         ipos = ichar(c) - ichar("A") + 1
         if( 1 <= ipos .and. ipos <= 26 ) then
            ! 'c' is in upper case
            to_lower(i:i) = char( ichar(c) + ishift )
         else
            to_lower(i:i) = c
         end if
      end do

#endif
   end function to_lower
!_______________________________________________________________________
!
   function to_upper( keyword )

      character(len=*), intent(in) :: keyword
      character(len=len_trim(keyword)) :: to_upper
      !------ API end ------

#ifdef _DEVLP
      ! just for ASCII-7 bits
      integer :: ishift
      integer :: i, ipos
      character :: c

   !------ end of declarations -- execution starts hereafter  ------

      ishift = ichar("A")-ichar("a")
      to_upper = ""

      do i = 1, len_trim(keyword)
         c = keyword(i:i)
         ipos = ichar(c) - ichar("a") + 1
         if( 1 <= ipos .and. ipos <= 26 ) then
            ! 'c' is in lower case
            to_upper(i:i) = char( ichar(c) + ishift )
         else
            to_upper(i:i) = c
         end if
      end do

#endif
   end function to_upper
!_______________________________________________________________________
!
   subroutine term_col( out, status )

      integer, intent(out) :: out, status
      !------ API end ------

#ifdef _DEVLP
      ! retrieves the nb of columns (i.e. character width) of the
      ! terminal, either from the environment variable MF_COLUMN or
      ! from the 'ioctl' unix command.
      ! (the minimum value is 40)
      !
      ! status = 0 only if successful, /= 0 otherwise

      ! internal routine; for a public routine, use 'mfGetTermWidth()'
      ! which call 'term_col' only once time.

      integer, external :: get_term_col
      character(len=80) :: COLUMNS

   !------ end of declarations -- execution starts hereafter  ------

      ! default value (in case of error)
      out = 80

#ifdef _WINDOWS
      return
#endif

      ! if $MF_COLUMNS exists, trying to get it
      COLUMNS = ""
      call getenv("MF_COLUMNS",COLUMNS)
      if( len_trim(COLUMNS) /= 0 ) then
         read(COLUMNS,*) out
         ! minimum value for which mfDisplay works !
         out = max( out, 40 )
         status = 0
         return
      end if

      ! getting the number of columns via 'ioctl' (C call)
      out = get_term_col()
      if( out == -1 ) then ! 'ioctl' failed in 'get_term_col'
         status = -1
         out = 80
         return
      end if
      out = max( out, 40 )
      status = 0

#endif
   end subroutine term_col
!_______________________________________________________________________
!
   function crc32bits_vec_int( a ) result( res )

      integer, target :: a(:)
      integer :: res
      !------ API end ------

#ifdef _DEVLP
#if GCC_VERSION >= 40900
      integer :: model
      integer, parameter :: model_size = C_sizeof(model)
#endif

   !------ end of declarations -- execution starts hereafter  ------

      ! computes 32 bits CRC
#if GCC_VERSION >= 40900
      res = icrc32( C_loc(a(1)), model_size*size(a) )
#else
      res = icrc32( a(1), 4*size(a) )
#endif

#endif
   end function crc32bits_vec_int
!_______________________________________________________________________
!
   function crc32bits_vec_dble( a ) result( res )

      real(kind=MF_DOUBLE), target :: a(:)
      integer :: res
      !------ API end ------

#ifdef _DEVLP
#if GCC_VERSION >= 40900
      real(kind=MF_DOUBLE) :: model
      integer, parameter :: model_size = C_sizeof(model)
#endif

   !------ end of declarations -- execution starts hereafter  ------

      ! computes 32 bits CRC
#if GCC_VERSION >= 40900
      res = icrc32( C_loc(a(1)), model_size*size(a) )
#else
      res = icrc32( a(1), 8*size(a) )
#endif

#endif
   end function crc32bits_vec_dble
!_______________________________________________________________________
!
   function crc32bits_mat_dble( a ) result( res )

      real(kind=MF_DOUBLE), target :: a(:,:)
      integer :: res
      !------ API end ------

#ifdef _DEVLP
#if GCC_VERSION >= 40900
      real(kind=MF_DOUBLE) :: model
      integer, parameter :: model_size = C_sizeof(model)
#endif

   !------ end of declarations -- execution starts hereafter  ------

      ! computes 32 bits CRC
#if GCC_VERSION >= 40900
      res = icrc32( C_loc(a(1,1)), model_size*size(a) )
#else
      res = icrc32( a(1,1), 8*size(a) )
#endif

#endif
   end function crc32bits_mat_dble
!_______________________________________________________________________
!
   function crc32bits_vec_cmplx( a ) result( res )

      complex(kind=MF_DOUBLE), target :: a(:)
      integer :: res
      !------ API end ------

#ifdef _DEVLP
#if GCC_VERSION >= 40900
      complex(kind=MF_DOUBLE) :: model
      integer, parameter :: model_size = C_sizeof(model)
#endif

   !------ end of declarations -- execution starts hereafter  ------

      ! computes 32 bits CRC
#if GCC_VERSION >= 40900
      res = icrc32( C_loc(a(1)), model_size*size(a) )
#else
      res = icrc32( a(1), 16*size(a) )
#endif

#endif
   end function crc32bits_vec_cmplx
!_______________________________________________________________________
!
   function crc32bits_mat_cmplx( a ) result( res )

      complex(kind=MF_DOUBLE), target :: a(:,:)
      integer :: res
      !------ API end ------

#ifdef _DEVLP
#if GCC_VERSION >= 40900
      complex(kind=MF_DOUBLE) :: model
      integer, parameter :: model_size = C_sizeof(model)
#endif

   !------ end of declarations -- execution starts hereafter  ------

      ! computes 32 bits CRC
#if GCC_VERSION >= 40900
      res = icrc32( C_loc(a(1,1)), model_size*size(a) )
#else
      res = icrc32( a(1,1), 16*size(a) )
#endif

#endif
   end function crc32bits_mat_cmplx
!_______________________________________________________________________
!
   function diag_length( nrow, ncol, diag ) result( res )

      integer, intent(in) :: nrow, ncol, diag
      integer :: res
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      ! computes length of the diagonal number 'diag'
      if( nrow == ncol ) then
         res = nrow - abs(diag)
      else if( nrow < ncol ) then
         if( diag < 0 ) then
            res = nrow - abs(diag)
         else if( 0 <= diag .and. diag <= ncol-nrow ) then
            res = nrow
         else
            res = nrow - (diag-ncol+nrow)
         end if
      else ! ncol < nrow
         if( diag > 0 ) then
            res = ncol - abs(diag)
         else if( ncol-nrow <= diag .and. diag <= 0 ) then
            res = ncol
         else
            res = ncol - (abs(diag)-nrow+ncol)
         end if
      end if

#endif
   end function diag_length
!_______________________________________________________________________
!
   subroutine digit_round( vec, tol )

      real(kind=MF_DOUBLE) :: vec(:)
      real(kind=MF_DOUBLE), intent(in) :: tol
      !------ API end ------

#ifdef _DEVLP
      integer :: n, k
      real(kind=MF_DOUBLE) :: norme, scaling

   !------ end of declarations -- execution starts hereafter  ------

      if( tol < epsilon(tol) ) then
         return
      end if

      n = size(vec)
      norme = maxval( abs( vec(:) ) )

      if( norme == 0.0d0 ) then
         return
      end if

      k = int( log10(norme) )

      scaling = 10.0d0**k * tol

      vec(:) = vec(:) / scaling
      vec(:) = nint( vec(:) )
      vec(:) = vec(:) * scaling

#endif
   end subroutine digit_round
!_______________________________________________________________________
!
   subroutine mf_save_and_disable_fpe( )

#ifdef _DEVLP
#if !defined(_WINDOWS) && !defined(_DARWIN)
      integer, external :: get_fpu_flags

   !------ end of declarations -- execution starts hereafter  ------

      if( mf_full_trapping ) return

      if( fpe_nested_call == 0 ) then

         fpu_flags_saved = get_fpu_flags()
         call disable_all_FPE()
!!print *, "mf_save_and_disable_fpe: FPE disabled."
!!pause "for debugging purpose..."

      end if

      fpe_nested_call = fpe_nested_call + 1
#endif

#endif
   end subroutine mf_save_and_disable_fpe
!_______________________________________________________________________
!
   subroutine mf_restore_fpe( )

#ifdef _DEVLP
#if !defined(_WINDOWS) && !defined(_DARWIN)
   !------ end of declarations -- execution starts hereafter  ------

      if( mf_full_trapping ) return

      fpe_nested_call = fpe_nested_call - 1

      if( fpe_nested_call < 0 ) then
         write(STDERR,*)
         write(STDERR,*) "(MUESLI FPE handling:) internal error:"
         write(STDERR,*) "                       'mf_save_and_disable_fpe' and 'mf_restore_fpe' routines"
         write(STDERR,*) "                       must be called by pair!"
         write(STDERR,*) "                       Please report this bug."
         mf_message_displayed = .true.
         call muesli_trace( pause ="yes" )
         stop
      end if

      if( fpe_nested_call == 0 ) then

         call clear_all_FPE()
         call enable_fpu_flags( fpu_flags_saved )
!!print *, "mf_restore_fpe: FPE re-enabled."
!!pause "for debugging purpose..."

      end if
#endif

#endif
   end subroutine mf_restore_fpe
!_______________________________________________________________________
!
   subroutine sec_int_2_hms( secondes, hrs, min, sec )

      integer, intent(in) :: secondes
      integer             :: hrs, min, sec
      !------ API end ------

#ifdef _DEVLP
      integer :: tmp

   !------ end of declarations -- execution starts hereafter  ------

      hrs = secondes/3600
      tmp = secondes - hrs*3600
      min = tmp/60
      sec = tmp - min*60

#endif
   end subroutine sec_int_2_hms
!_______________________________________________________________________
!
   subroutine sec_dp_2_hms( secondes, hrs, min, sec )

      double precision, intent(in) :: secondes
      integer                      :: hrs, min, sec
      !------ API end ------

#ifdef _DEVLP
      integer :: tmp

   !------ end of declarations -- execution starts hereafter  ------

      hrs = int( secondes/3600 )
      tmp = int( secondes - hrs*3600 )
      min = tmp/60
      sec = tmp - min*60

#endif
   end subroutine sec_dp_2_hms
!_______________________________________________________________________
!
   subroutine no_duplicates_int( tab, status, i1, i2 )

      integer, intent(in) :: tab(:)
      integer :: status, i1, i2
      !------ API end ------

#ifdef _DEVLP
      ! Takes an integer array containing n elements, and verify
      ! that all non-zero entries are unique
      !
      ! status = 0 : no duplicated entries
      !          1 : duplicated entries found,
      !              first entries are located at position: i1 and i2

      ! The integer array 'tab' must be sorted!

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = size(tab)

      status = 0
      do i1 = 1, n-1
         i2 = i1 + 1
         if( tab(i1) /= 0 .and. tab(i1) == tab(i2) ) then
            status = 1
            return
         end if
      end do

#endif
   end subroutine no_duplicates_int
!_______________________________________________________________________
!
   subroutine no_duplicates_int8( tab, status, i1, i2 )

      integer*8, intent(in) :: tab(:)
      integer :: status, i1, i2
      !------ API end ------

#ifdef _DEVLP
      ! Takes an integer array containing n elements, and verify
      ! that all non-zero entries are unique
      !
      ! status = 0 : no duplicated entries
      !          1 : duplicated entries found,
      !              first entries are located at position: i1 and i2

      ! The integer array 'tab' must be sorted!

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = size(tab)

      status = 0
      do i1 = 1, n-1
         i2 = i1 + 1
         if( tab(i1) /= 0 .and. tab(i1) == tab(i2) ) then
            status = 1
            return
         end if
      end do

#endif
   end subroutine no_duplicates_int8
!_______________________________________________________________________
!
   function is_sorted_ascend_real8( tab ) result( status )

      real(kind=MF_DOUBLE), intent(in) :: tab(:)
      integer :: status
      !------ API end ------

#ifdef _DEVLP
      ! Checks if the vector 'tab' is sorted in ascending mode.
      ! Returns 'status' which contains:
      !   0 if the vector is sorted, and NaNs all located at the end;
      !   1 if finite values are not sorted;
      !   2 if NaNs are not all located at the end of the vector

      ! The vector tab may contain NaN values; on such a case, all NaNs
      ! must be located at the end of the vector.

      integer :: i, j, n

   !------ end of declarations -- execution starts hereafter  ------

      n = size(tab)

      status = 0

      if( isnan(tab(1) ) ) then
         i = 1
         go to 10
      end if

      do i = 2, n
         if( isnan(tab(i) ) ) then
            go to 10
         end if
         if( tab(i) < tab(i-1) ) then
            status = 1
            return
         end if
      end do

      return

 10   continue
      ! one NaN value has been found at position 'i'; all next elements
      ! must be NaNs.
      do j = i+1, n
         if( .not. isnan(tab(j) ) ) then
            status = 2
            return
         end if
      end do

#endif
   end function is_sorted_ascend_real8
!_______________________________________________________________________
!
   function is_sorted_descend_real8( tab ) result( status )

      real(kind=MF_DOUBLE), intent(in) :: tab(:)
      integer :: status
      !------ API end ------

#ifdef _DEVLP
      ! Checks if the vector 'tab' is sorted in descending mode.
      ! Returns 'status' which contains:
      !   0 if the vector is sorted, and NaNs all located at the end;
      !   1 if finite values are not sorted;
      !   2 if NaNs are not all located at the end of the vector

      ! the vector tab may contain NaN values; on such a case, all NaNs
      ! must be located at the end of the vector.

      integer :: i, j, n

   !------ end of declarations -- execution starts hereafter  ------

      n = size(tab)

      status = 0

      if( isnan(tab(1) ) ) then
         i = 1
         go to 10
      end if

      do i = 2, n
         if( isnan(tab(i) ) ) then
            go to 10
         end if
         if( tab(i) > tab(i-1) ) then
            status = 1
            return
         end if
      end do

      return

 10   continue
      ! one NaN value has been found at position 'i'; all next elements
      ! must be NaNs.
      do j = i+1, n
         if( .not. isnan(tab(j) ) ) then
            status = 2
            return
         end if
      end do

#endif
   end function is_sorted_descend_real8
!_______________________________________________________________________
!
   function is_sorted_ascend_cmplx8( tab ) result( status )

      complex(kind=MF_DOUBLE), intent(in) :: tab(:)
      integer :: status
      !------ API end ------

#ifdef _DEVLP
      ! Checks if the vector 'tab' is sorted in ascending mode (i.e.
      ! sorted first by module, then by phase-angle).
      ! Returns 'status' which contains:
      !   0 if the vector is sorted, and NaNs all located at the end;
      !   1 if finite values are not sorted;
      !   2 if NaNs are not all located at the end of the vector

      ! The vector tab may contain NaN values; on such a case, all NaNs
      ! must be located at the end of the vector.

      integer :: i, j, n, first, last, n_sub
      logical :: sorted

   !------ end of declarations -- execution starts hereafter  ------

      n = size(tab)

      status = 0

      if( isnan(real(tab(1))) .or. isnan(aimag(tab(1))) ) then
         i = 1
         go to 10
      end if

      first = 1
      do i = 2, n
         if( isnan(real(tab(i))) .or. isnan(aimag(tab(i))) ) then
            go to 10
         end if
         if( abs(tab(i)) == abs(tab(i-1)) ) then
         else if( abs(tab(i)) < abs(tab(i-1)) ) then
            status = 1
            return
         else
            last = i-1
            ! if needed, check the sublist (defined by the first:last indices)
            ! according to the phase-angle
            n_sub = last - first + 1
            if( n_sub >= 2 ) then
               sorted = issorted_cmplx_by_phase_angle( tab(first:last), "asc" )
               if( .not. sorted ) then
                  status = 1
                  return
               end if
            end if
            first = i
         end if
      end do

      return

 10   continue
      ! one NaN value has been found at position 'i'; all next elements
      ! must be NaNs.
      do j = i+1, n
         if( .not.(isnan(real(tab(j))) .or. isnan(aimag(tab(j)))) ) then
            status = 2
            return
         end if
      end do

#endif
   end function is_sorted_ascend_cmplx8
!_______________________________________________________________________
!
   function is_sorted_descend_cmplx8( tab ) result( status )

      complex(kind=MF_DOUBLE), intent(in) :: tab(:)
      integer :: status
      !------ API end ------

#ifdef _DEVLP
      ! Checks if the vector 'tab' is sorted in descending mode (i.e.
      ! sorted first by module, then by phase-angle).
      ! Returns 'status' which contains:
      !   0 if the vector is sorted, and NaNs all located at the end;
      !   1 if finite values are not sorted;
      !   2 if NaNs are not all located at the end of the vector

      ! The vector tab may contain NaN values; on such a case, all NaNs
      ! must be located at the end of the vector.

      integer :: i, j, n, first, last, n_sub
      logical :: sorted

   !------ end of declarations -- execution starts hereafter  ------

      n = size(tab)

      status = 0

      if( isnan(real(tab(1))) .or. isnan(aimag(tab(1))) ) then
         i = 1
         go to 10
      end if

      first = 1
      do i = 2, n
         if( isnan(real(tab(i))) .or. isnan(aimag(tab(i))) ) then
            go to 10
         end if
         if( abs(tab(i)) == abs(tab(i-1)) ) then
         else if( abs(tab(i)) > abs(tab(i-1)) ) then
            status = 1
            return
         else
            last = i-1
            ! if needed, check the sublist (defined by the first:last indices)
            ! according to the phase-angle
            n_sub = last - first + 1
            if( n_sub >= 2 ) then
               sorted = issorted_cmplx_by_phase_angle( tab(first:last), "des" )
               if( .not. sorted ) then
                  status = 1
                  return
               end if
            end if
            first = i
         end if
      end do

      return

 10   continue
      ! one NaN value has been found at position 'i'; all next elements
      ! must be NaNs.
      do j = i+1, n
         if( .not.(isnan(real(tab(j))) .or. isnan(aimag(tab(j)))) ) then
            status = 2
            return
         end if
      end do

#endif
   end function is_sorted_descend_cmplx8
!_______________________________________________________________________
!
   function issorted_cmplx_by_phase_angle( tab, mode ) result( bool )

      complex(kind=MF_DOUBLE), intent(in) :: tab(:)
      character(len=3),        intent(in) :: mode
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      integer :: i, n
      real(kind=MF_DOUBLE) :: tmp_prev, tmp

   !------ end of declarations -- execution starts hereafter  ------

      bool = .true.

      n = size(tab)

      tmp_prev = atan2( aimag(tab(1)), real(tab(1)) )

      if( mode == "asc" ) then
         do i = 2, n
            tmp = atan2( aimag(tab(i)), real(tab(i)) )
            if( tmp < tmp_prev ) then
               bool = .false.
               return
            end if
            tmp_prev = tmp
         end do
      else ! mode = "descend"
         do i = 2, n
            tmp = atan2( aimag(tab(i)), real(tab(i)) )
            if( tmp > tmp_prev ) then
               bool = .false.
               return
            end if
            tmp_prev = tmp
         end do
      end if

#endif
   end function issorted_cmplx_by_phase_angle
!_______________________________________________________________________
!
   subroutine sort_cmplx_by_phase_angle( mode, A_mod, A )

      character(len=3),        intent(in)     :: mode
      real(kind=MF_DOUBLE),    intent(in)     :: A_mod(:)
      complex(kind=MF_DOUBLE), intent(in out) :: A(:)
      !------ API end ------

#ifdef _DEVLP
      ! A_mod must contains the module (abs) of the complex vector A

      integer :: i, j, n, first, last, n_sub
      real(kind=MF_DOUBLE), allocatable :: tmp(:)
      integer, allocatable :: ind(:)

   !------ end of declarations -- execution starts hereafter  ------

      n = size(A_mod)

      first = 1
      do i = 2, n
         ! find sublists of equal modules in A_mod
         if( A_mod(i) /= A_mod(i-1) ) then
            last = i-1
            ! if needed, sort the sublist (defined by by the first:last indices)
            ! according to the phase-angle
            n_sub = last - first + 1
            if( n_sub >= 2 ) then
               allocate( tmp(n_sub) )
               ! atan2 never give NaN, even for (0,0)
               tmp(:) = atan2( aimag(A(first:last)), real(A(first:last)) )
               allocate( ind(n_sub) )
               ind(:) = [ (j,j=first,last) ]
               call quick_sort( mode, tmp, ind )
               A(first:last) = A( ind )
               deallocate( tmp, ind )
            end if
            first = i
         end if
      end do

#endif
   end subroutine sort_cmplx_by_phase_angle
!_______________________________________________________________________
!
   subroutine sort_cmplx_by_phase_angle_2( mode, A_mod, A, ind )

      character(len=3),        intent(in)     :: mode
      real(kind=MF_DOUBLE),    intent(in)     :: A_mod(:)
      complex(kind=MF_DOUBLE), intent(in out) :: A(:)
      integer,                 intent(in out) :: ind(:)
      !------ API end ------

#ifdef _DEVLP
      ! A_mod must contains the module (abs) of the complex vector A

      integer :: i, j, n, first, last, n_sub
      real(kind=MF_DOUBLE), allocatable :: tmp(:)
      integer, allocatable :: ind_2(:)

   !------ end of declarations -- execution starts hereafter  ------

      n = size(A_mod)

      first = 1
      do i = 2, n
         ! find sublists of equal modules in A_mod
         if( A_mod(i) /= A_mod(i-1) ) then
            last = i-1
            ! if needed, sort the sublist (defined by by the first:last indices)
            ! according to the phase-angle
            n_sub = last - first + 1
            if( n_sub >= 2 ) then
               allocate( tmp(n_sub) )
               ! atan2 never give NaN, even for (0,0)
               tmp(:) = atan2( aimag(A(first:last)), real(A(first:last)) )
               allocate( ind_2(n_sub) )
               ind_2(:) = [ (j,j=first,last) ]
               call quick_sort( mode, tmp, ind_2 )
               A(first:last) = A( ind_2 )
               ind(first:last) = ind( ind_2 )
               deallocate( tmp, ind_2 )
            end if
            first = i
         end if
      end do

#endif
   end subroutine sort_cmplx_by_phase_angle_2
!_______________________________________________________________________
!
   subroutine go_home_on_term()

#ifdef _DEVLP
      ! used in 'ms*Progress', 'msOdeSolve' and 'msDaeSolve'

      character :: CR = char(13)

   !------ end of declarations -- execution starts hereafter  ------

      ! go home on a terminal
      write(STDOUT,"(A4)",advance="no") CR

#endif
   end subroutine go_home_on_term
!_______________________________________________________________________
!
   subroutine full_chk_der_ode( n, t, x, fcn, fvec, fjac, err, iflag )

      ! FULL_CHK_DER_ODE (from FULL_CHK_DER of Minpack, adapted for OdeSolve)
      !
      ! modified for MUESLI Fortran library
      ! (C) É. Canot -- IPR/CNRS -- 20 feb 2015

      implicit none

      integer,              intent(in)  :: n
      real(kind=MF_DOUBLE), intent(in)  :: t, x(n), fvec(n), fjac(n,n)
      real,                 intent(out) :: err(n,n) ! single precision is sufficient
      integer,              intent(out) :: iflag

      interface
         subroutine fcn( t, y, yprime, flag ) ! named 'deriv' in OdeSolve
            import MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in)     :: t, y(*)
            real(kind=MF_DOUBLE), intent(out)    :: yprime(*)
            integer,              intent(in out) :: flag
         end subroutine
      end interface
      !------ API end ------

#ifdef _DEVLP
!     ******************************************************************
! EC 2014-06-05:
!
!     1) By using the current routine, the user can locate exactly
!        which component(s) of the jacobian matrix is (are) wrong.
!        This routine is more expensive than 'chkder' (see in Minpack),
!        as it requires n additional calls of the 'fcn' subroutine.
!        It is based on the approximation:
!
!          Fi( p1, ..., pj+dj, ..., pn ) - Fi( p1, ..., pn )
!         --------------------------------------------------- ~ jac(i,j)
!                                   dj
!
!         repeated for j = 1, n
!
!     2) The Minpack routine 'chkder' doesn't make a complete check of
!        the jacobian matrix, but it is cheaper. Indeed, it merely makes
!        a global check by requiring only one additional call of the
!        'fcn' subroutine.
!
!     ******************************************************************
!
!     subroutine full_chk_der_ode
!
!     This subroutine checks all components of the jacobian matrix fjac
!     evaluated at a point x.
!
!     The subroutine does not perform reliably if cancellation or
!     rounding errors cause a severe loss of significance in the
!     evaluation of a function.
!
!     The subroutine statement is
!
!       subroutine full_chk_der_ode( n, t, x, fcn, fvec, fjac, err, iflag )
!
!     where
!
!       n is a positive integer input variable set to the number
!         of equations.
!
!       t is an input double precision real, which is the independant
!         variable.
!
!       x is an input array of length n.
!
!       fcn is the name of the user-supplied subroutine which
!         calculates the functions Fi ('deriv' in OdeSolve)
!
!       fvec is an input array of length n. It contains the functions
!         Fi evaluated at x.
!
!       fjac is an n by n array (and declared exactly of size (n,n)).
!         On input, the rows of fjac must contain the gradients (wrt x)
!         of the respective functions evaluated at x.
!
!       err is an array of size (n,n) which contains the relative
!       error of each component fjac(i,j). The single precision used
!       is sufficient.
!
!       iflag is an integer flag which is usually equal to zero. On the
!       contrary, it indicates an error during the 'fcn' evaluation.
!       Therefore, the jacobian check is done only if iflag is zero.
!
!     subprograms called
!
!       fortran supplied ... abs, log10, sqrt, epsilon
!
!     Muesli numerical library. Feb 2015.
!     Édouard Canot.
!
!     **********
      integer :: i, j
      real(kind=MF_DOUBLE) :: epsmch, eps, dx, xp(n), fvecp(n)
      real :: abs_fjac

!--------- end of declarations -- execution starts hereafter  ------

!     epsmch is the machine precision.
      epsmch = epsilon(1.0d0)

      eps = sqrt(epsmch)

      iflag = 0

      ! it is preferable to initialize to zero; indeed, a return after
      ! the 'fcn' call may lead to NaN in the array err(:,:)
      err(:,:) = 0.0

      do j = 1, n
         dx = eps*abs(x(j))
         if( dx == 0.0d0 ) dx = eps
         xp(:) = x(:)
         xp(j) = xp(j) + dx

         call mf_restore_fpe( )
         call fcn( t, xp, fvecp, iflag )
         call mf_save_and_disable_fpe( )

         if( iflag /= 0 ) then
            return
         end if

         err(:,j) = real( abs( fjac(:,j) - (fvecp(:)-fvec(:)) / dx ) )
         do i = 1, n
            abs_fjac = real( abs(fjac(i,j)) )
            if( abs_fjac > eps ) then
               err(i,j) = err(i,j) / abs_fjac
            end if
         end do
      end do

#endif
   end subroutine full_chk_der_ode
!_______________________________________________________________________
!
   subroutine full_chk_der_dae( n, t, x, xp, fcn, fvec, fjac, err, iflag )

      ! FULL_CHK_DER_DAE (from FULL_CHK_DER of Minpack, adapted for DaeSolve)
      !
      ! modified for MUESLI Fortran library
      ! (C) É. Canot -- IPR/CNRS -- 20 feb 2015

      implicit none

      integer,          intent(in)  :: n
      real(kind=MF_DOUBLE), intent(in)  :: t, x(n), xp(n), fvec(n), fjac(n,n)
      real,             intent(out) :: err(n,n) ! single precision is sufficient
      integer,          intent(out) :: iflag

      interface
         subroutine fcn( t, y, yprime, delta, flag ) ! named 'resid' in DaeSolve
            import MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in)     :: t, y(*), yprime(*)
            real(kind=MF_DOUBLE), intent(out)    :: delta(*)
            integer,              intent(in out) :: flag
         end subroutine
      end interface
      !------ API end ------

#ifdef _DEVLP
!     ******************************************************************
! EC 2014-06-05:
!
!     1) By using the current routine, the user can locate exactly
!        which component(s) of the jacobian matrix is (are) wrong.
!        This routine is more expensive than 'chkder' (see in Minpack),
!        as it requires n additional calls of the 'fcn' subroutine.
!        It is based on the approximation:
!
!          Fi( p1, ..., pj+dj, ..., pn ) - Fi( p1, ..., pn )
!         --------------------------------------------------- ~ jac(i,j)
!                                   dj
!
!         repeated for j = 1, n
!
!     2) The Minpack routine 'chkder' doesn't make a complete check of
!        the jacobian matrix, but it is cheaper. Indeed, it merely makes
!        a global check by requiring only one additional call of the
!        'fcn' subroutine.
!
!     ******************************************************************
!
!     subroutine full_chk_der_dae
!
!     This subroutine checks all components of the jacobian matrix fjac
!     evaluated at a point x.
!
!     The subroutine does not perform reliably if cancellation or
!     rounding errors cause a severe loss of significance in the
!     evaluation of a function.
!
!     The subroutine statement is
!
!       subroutine full_chk_der_dae( n, t, x, fcn, fvec, fjac, err, iflag )
!
!     where
!
!       n is a positive integer input variable set to the number
!         of equations.
!
!       t is an input double precision real, which is the independant
!         variable.
!
!       x is an input array of length n.
!
!       xp is an input array of length n.
!
!       fcn is the name of the user-supplied subroutine which
!         calculates the functions Fi ('deriv' in OdeSolve)
!
!       fvec is an input array of length n. It contains the functions
!         Fi evaluated at x.
!
!       fjac is an n by n array (and declared exactly of size (n,n)).
!         On input, the rows of fjac must contain the gradients (wrt x)
!         of the respective functions evaluated at x.
!
!       err is an array of size (n,n) which contains the relative
!       error of each component fjac(i,j). The single precision used
!       is sufficient.
!
!       iflag is an integer flag which is usually equal to zero. On the
!       contrary, it indicates an error during the 'fcn' evaluation.
!       Therefore, the jacobian check is done only if iflag is zero.
!
!     subprograms called
!
!       fortran supplied ... abs, log10, sqrt, epsilon
!
!     Muesli numerical library. Feb 2015.
!     Édouard Canot.
!
!     **********
      integer :: j
      real(kind=MF_DOUBLE) :: epsmch, eps, dx, xnew(n), fvecp(n)
!NEW: see 'full_chk_der_ode' which compute 'err' in a different way!
real :: max_fjac

!--------- end of declarations -- execution starts hereafter  ------

!     epsmch is the machine precision.
      epsmch = epsilon(1.0d0)

      eps = sqrt(epsmch)

      iflag = 0

      max_fjac = real( maxval( abs(fjac) ) )
      dx = eps*maxval( abs(x) )

      ! it is preferable to initialize to zero; indeed, a return after
      ! the 'fcn' call may lead to NaN in the array err(:,:)
      err(:,:) = 0.0

      do j = 1, n
         if( dx < eps ) dx = eps
         xnew(:) = x(:)
         xnew(j) = xnew(j) + dx

         call mf_restore_fpe( )
         call fcn( t, xnew, xp, fvecp, iflag )
         call mf_save_and_disable_fpe( )

         if( iflag /= 0 ) then
            return
         end if

         err(:,j) = real( abs( fjac(:,j) - (fvecp(:)-fvec(:)) / dx ) )
      end do

      err(:,:) = err(:,:) / max_fjac

#endif
   end subroutine full_chk_der_dae
!_______________________________________________________________________
!
   subroutine check_and_remove_trailing_zero( str )

      character(len=*), intent(in out) :: str

      ! Ne fontionne que pour un réel en format fixe et contenant le
      ! point décimal.
      !  "127.000   " -> "127       "
      !  "127.100   " -> "127.1     "

      integer :: i

!--------- end of declarations -- execution starts hereafter  ------

      ! the string must not contain stars (bad Fortran write)
      i = index( str, "*" )
      if( i /= 0 ) then
         print "(/,A)", "(Muesli) check_and_remove_trailing_zero: ERROR"
         print *,       "   bad write of string: '", str, "'"
      end if

      i = len_trim(str)
      do
         if( str(i:i) == "0" ) then
            str(i:i) = " "
            i = i - 1
            if( i == 0 ) exit
         else if( str(i:i) == "." ) then
            str(i:i) = " "
            exit
         else
            exit
         end if
      end do

   end subroutine check_and_remove_trailing_zero
!_______________________________________________________________________
!
   function clean_exp_format(s) result(out)
      ! Nettoie une écriture scientifique :
      ! - mantisse : enlève les zéros inutiles et le point final,
      ! - exposant : lettre e minuscule, enlève '+' et zéros de tête.
      character(len=*), intent(in) :: s
      character(len=:), allocatable :: out

      integer :: epos, i
      character(len=:), allocatable :: mant, expo, sign, digits

      ! Trouver la position de 'e' ou 'E'
      epos = index(s, 'e')
      if(epos == 0) epos = index(s, 'E')
      if(epos == 0) then
         out = s
         return
      end if

      mant = s(1:epos-1)
      expo = s(epos:)    ! inclut la lettre 'e' ou 'E'

      ! Exposant toujours en minuscule
      expo(1:1) = 'e'

      ! --- Nettoyage mantisse ---
      ! Enlever les zéros de fin
      do i = len(mant), 1, -1
         if(mant(i:i) /= '0') exit
      end do
      mant = mant(1:i)

      ! Si ça finit par ".", l'enlever
      if(mant(len(mant):len(mant)) == '.') then
         mant = mant(1:len(mant)-1)
      end if

      ! --- Nettoyage exposant ---
      ! expo = "e+002" ou "e-03" ou "e5", etc.
      sign = "+"
      digits = ""

      ! Extraire le signe si présent
      if(expo(2:2) == '+' .or. expo(2:2) == '-') then
         sign = expo(2:2)
         digits = expo(3:)
      else
         digits = expo(2:)
      end if

      ! Enlever espaces éventuels
      digits = trim(digits)

      ! Enlever les zéros en tête
      do i = 1, len_trim(digits)
         if(digits(i:i) /= '0') exit
      end do
      if(i > len_trim(digits)) then
         ! exposant = 0
         expo = "e0"
         out = mant // expo
         return
      else
         digits = digits(i:)
      end if

      ! Reconstruire l’exposant :
      ! - signe '-' conservé
      ! - signe '+' supprimé
      if(sign == '-') then
         expo = "e-" // digits
      else
         expo = "e" // digits
      end if

      ! Chaîne finale
      out = mant // expo

   end function clean_exp_format
!_______________________________________________________________________
!
end module mod_mfaux
