#ifdef _HAS_LOC
#define _MF_LOC_ANY_OBJ_ loc
#else
#define _MF_LOC_ANY_OBJ_ mf_loc_any_obj
#endif

module mod_mem_debug

   ! Part of MUESLI Numerical Library
   ! Copyright É. Canot 2003-2025 -- IPR/CNRS

!=======================================================================
!    utilitaries for user-debugging
!    (traces for error/warning)
!=======================================================================

   use mod_mfdebug, stderr_old => STDERR

   implicit none

#ifndef _DEVLP
   private
#endif

#ifndef _HAS_LOC
#ifdef _64_BITS
   ! 64bit integer
   k_address = selected_int_kind(18)
#else
   ! 32bit integer
   k_address = selected_int_kind(8)
#endif
   integer(kind=k_address), external :: mf_loc_any_obj
#endif

   integer, parameter :: STDERR = 0
   private :: STDERR

   interface mf_allocate
      module procedure mf_allocate_integer
      module procedure mf_allocate_integer_2D
      module procedure mf_allocate_character
      module procedure mf_allocate_real
      module procedure mf_allocate_real_2D
      module procedure mf_allocate_double
      module procedure mf_allocate_double_2D
      module procedure mf_allocate_cmplx
      module procedure mf_allocate_cmplx_2D
   end interface mf_allocate
   !------ API end ------

   interface mf_deallocate
      module procedure mf_deallocate_integer
      module procedure mf_deallocate_integer_2D
      module procedure mf_deallocate_character
      module procedure mf_deallocate_real
      module procedure mf_deallocate_real_2D
      module procedure mf_deallocate_double
      module procedure mf_deallocate_double_2D
      module procedure mf_deallocate_cmplx
      module procedure mf_deallocate_cmplx_2D
   end interface mf_deallocate
   !------ API end ------

   interface mfdebug_loc
      integer function mfdebug_loc_char_d1( tab )
         character, pointer :: tab(:)
      end function
      integer function mfdebug_loc_int4_d1( tab )
         integer, pointer :: tab(:)
      end function
      integer function mfdebug_loc_real4_d1( tab )
         real, pointer :: tab(:)
      end function
      integer function mfdebug_loc_real4_d2( tab )
         real, pointer :: tab(:,:)
      end function
      integer function mfdebug_loc_real8_d1( tab )
         import :: MF_DOUBLE
         real(kind=MF_DOUBLE), pointer :: tab(:)
      end function
      integer function mfdebug_loc_real8_d2( tab )
         import :: MF_DOUBLE
         real(kind=MF_DOUBLE), pointer :: tab(:,:)
      end function
      integer function mfdebug_loc_cmplx_d1( tab )
         import :: MF_DOUBLE
         complex(kind=MF_DOUBLE), pointer :: tab(:)
      end function
      integer function mfdebug_loc_cmplx_d2( tab )
         import :: MF_DOUBLE
         complex(kind=MF_DOUBLE), pointer :: tab(:,:)
      end function
      ! another interface, for process 'mfArray' has been
      ! added in the 'mod_mfarray' module
   end interface mfdebug_loc
   !------ API end ------

   interface mfdebug_loc_point_to
      integer function mfdebug_loc_char_d1_point_to( tab )
         character :: tab(:)
      end function
      integer function mfdebug_loc_int4_d1_point_to( tab )
         integer :: tab(:)
      end function
      integer function mfdebug_loc_int4_d2_point_to( tab )
         integer :: tab(:,:)
      end function
      integer function mfdebug_loc_real4_d1_point_to( tab )
         real :: tab(:)
      end function
      integer function mfdebug_loc_real4_d2_point_to( tab )
         real :: tab(:,:)
      end function
      integer function mfdebug_loc_real8_d1_point_to( tab )
         import :: MF_DOUBLE
         real(kind=MF_DOUBLE) :: tab(:)
      end function
      integer function mfdebug_loc_real8_d2_point_to( tab )
         import :: MF_DOUBLE
         real(kind=MF_DOUBLE) :: tab(:,:)
      end function
      integer function mfdebug_loc_cmplx_d1_point_to( tab )
         import :: MF_DOUBLE
         complex(kind=MF_DOUBLE) :: tab(:)
      end function
      integer function mfdebug_loc_cmplx_d2_point_to( tab )
         import :: MF_DOUBLE
         complex(kind=MF_DOUBLE) :: tab(:,:)
      end function
   end interface mfdebug_loc_point_to
   !------ API end ------

   private :: mf_allocate_integer, &
              mf_allocate_integer_2D, &
              mf_allocate_character, &
              mf_allocate_real, &
              mf_allocate_real_2D, &
              mf_allocate_double, &
              mf_allocate_double_2D, &
              mf_allocate_cmplx, &
              mf_allocate_cmplx_2D

   private :: mf_deallocate_integer, &
              mf_deallocate_integer_2D, &
              mf_deallocate_character, &
              mf_deallocate_real, &
              mf_deallocate_real_2D, &
              mf_deallocate_double, &
              mf_deallocate_double_2D, &
              mf_deallocate_cmplx, &
              mf_deallocate_cmplx_2D

   private :: mfdebug_loc_char_d1, &
              mfdebug_loc_int4_d1, &
              mfdebug_loc_real4_d1, &
              mfdebug_loc_real4_d2, &
              mfdebug_loc_real8_d1, &
              mfdebug_loc_cmplx_d1, &
              mfdebug_loc_cmplx_d2

   private :: mfdebug_loc_char_d1_point_to, &
              mfdebug_loc_int4_d1_point_to, &
              mfdebug_loc_int4_d2_point_to, &
              mfdebug_loc_real4_d1_point_to, &
              mfdebug_loc_real4_d2_point_to, &
              mfdebug_loc_real8_d1_point_to, &
              mfdebug_loc_real8_d2_point_to, &
              mfdebug_loc_cmplx_d1_point_to, &
              mfdebug_loc_cmplx_d2_point_to

   private :: mf_traceback

contains
!_______________________________________________________________________
!
   subroutine mf_traceback( )

#ifdef _DEVLP
#if defined _GNU_GFC
#ifndef _OPTIM
#ifdef _HAVE_SHOW_BACKTRACE
      external :: show_backtrace
#endif
#endif
#endif

   !------ end of declarations -- execution starts hereafter  ------

#if defined _INTEL_IFC
      call tracebackqq( user_exit_code=-1 )
#elif defined _GNU_GFC
#ifdef _HAVE_SHOW_BACKTRACE
      call flush(STDERR)
      call show_backtrace()
#else
#ifdef _STD_BACKTRACE
      call backtrace()
#endif
#endif
#endif

#endif
   end subroutine mf_traceback
!_______________________________________________________________________
!
   subroutine mf_allocate_integer( array, n, file, line,                &
                                   symb, unit )

      integer, pointer :: array(:)
      integer :: n
      character(len=*) :: file, line
      character(len=*), optional :: symb, unit
      !------ API end ------

#ifdef _DEVLP
      character(len=80) :: symb2, unit2
      integer(kind=MF_ADDRESS) :: address

   !------ end of declarations -- execution starts hereafter  ------

      allocate( array(n) )
      ! no blank line here

      if( present(symb) ) then
         symb2 = symb
      else
         symb2 = ""
      end if

      if( present(unit) ) then
         unit2 = unit
      else
         unit2 = ""
      end if

      ! printing some infos on STDERR
      address = _MF_LOC_ANY_OBJ_(array)
      ! printing some infos on STDERR
#ifdef _64_BITS
      write(STDERR,"(A,Z16.16,A,I0,A,A,A)")                             &
#else
      write(STDERR,"(A,Z8.8,A,I0,A,A,A)")                               &
#endif
            "[+ addr: ", address,                                       &
            ", size: ", n*4,                                            &
            ", symb: ", trim(symb2),                                    &
            "]"
      write(STDERR,"(A,A,A,A,A,A,A)")                                   &
            "  (in unit: ", trim(unit2),                                &
            ", file: ", trim(file),                                     &
            ", line: ", trim(line), ")"

      call mf_traceback()

      call flush(STDERR)

#endif
   end subroutine mf_allocate_integer
!_______________________________________________________________________
!
   subroutine mf_allocate_integer_2D( array, m, n, file, line,          &
                                      symb, unit )

      integer, pointer :: array(:,:)
      integer :: m, n
      character(len=*) :: file, line
      character(len=*), optional :: symb, unit
      !------ API end ------

#ifdef _DEVLP
      character(len=80) :: symb2, unit2
      integer(kind=MF_ADDRESS) :: address

   !------ end of declarations -- execution starts hereafter  ------

      allocate( array(m,n) )
      ! no blank line here

      if( present(symb) ) then
         symb2 = symb
      else
         symb2 = ""
      end if

      if( present(unit) ) then
         unit2 = unit
      else
         unit2 = ""
      end if

      ! printing some infos on STDERR
      address = _MF_LOC_ANY_OBJ_(array)
      ! printing some infos on STDERR
#ifdef _64_BITS
      write(STDERR,"(A,Z16.16,A,I0,A,A,A)")                             &
#else
      write(STDERR,"(A,Z8.8,A,I0,A,A,A)")                               &
#endif
            "[+ addr: ", address,                                       &
            ", size: ", m*n*4,                                          &
            ", symb: ", trim(symb2),                                    &
            "]"
      write(STDERR,"(A,A,A,A,A,A,A)")                                   &
            "  (in unit: ", trim(unit2),                                &
            ", file: ", trim(file),                                     &
            ", line: ", trim(line), ")"

      call mf_traceback()

      call flush(STDERR)

#endif
   end subroutine mf_allocate_integer_2D
!_______________________________________________________________________
!
   subroutine mf_allocate_character( array, n, file, line,              &
                                     symb, unit )

      character, pointer :: array(:)
      integer :: n
      character(len=*) :: file, line
      character(len=*), optional :: symb, unit
      !------ API end ------

#ifdef _DEVLP
      character(len=80) :: symb2, unit2
      integer(kind=MF_ADDRESS) :: address

   !------ end of declarations -- execution starts hereafter  ------

      allocate( array(n) )
      ! no blank line here

      if( present(symb) ) then
         symb2 = symb
      else
         symb2 = ""
      end if

      if( present(unit) ) then
         unit2 = unit
      else
         unit2 = ""
      end if

      ! printing some infos on STDERR
      address = _MF_LOC_ANY_OBJ_(array)
      ! printing some infos on STDERR
#ifdef _64_BITS
      write(STDERR,"(A,Z16.16,A,I0,A,A,A)")                             &
#else
      write(STDERR,"(A,Z8.8,A,I0,A,A,A)")                               &
#endif
            "[+ addr: ", address,                                       &
            ", size: ", n*4,                                            &
            ", symb: ", trim(symb2),                                    &
            "]"
      write(STDERR,"(A,A,A,A,A,A,A)")                                   &
            "  (in unit: ", trim(unit2),                                &
            ", file: ", trim(file),                                     &
            ", line: ", trim(line), ")"

      call mf_traceback()

      call flush(STDERR)

#endif
   end subroutine mf_allocate_character
!_______________________________________________________________________
!
   subroutine mf_allocate_real( array, n, file, line,                   &
                                symb, unit )

      real, pointer :: array(:)
      integer :: n
      character(len=*) :: file, line
      character(len=*), optional :: symb, unit
      !------ API end ------

#ifdef _DEVLP
      character(len=80) :: symb2, unit2
      integer(kind=MF_ADDRESS) :: address

   !------ end of declarations -- execution starts hereafter  ------

      allocate( array(n) )
      ! no blank line here

      if( present(symb) ) then
         symb2 = symb
      else
         symb2 = ""
      end if

      if( present(unit) ) then
         unit2 = unit
      else
         unit2 = ""
      end if

      ! printing some infos on STDERR
      address = _MF_LOC_ANY_OBJ_(array)
      ! printing some infos on STDERR
#ifdef _64_BITS
      write(STDERR,"(A,Z16.16,A,I0,A,A,A)")                             &
#else
      write(STDERR,"(A,Z8.8,A,I0,A,A,A)")                               &
#endif
            "[+ addr: ", address,                                       &
            ", size: ", n*4,                                            &
            ", symb: ", trim(symb2),                                    &
            "]"
      write(STDERR,"(A,A,A,A,A,A,A)")                                   &
            "  (in unit: ", trim(unit2),                                &
            ", file: ", trim(file),                                     &
            ", line: ", trim(line), ")"

      call mf_traceback()

      call flush(STDERR)

#endif
   end subroutine mf_allocate_real
!_______________________________________________________________________
!
   subroutine mf_allocate_real_2D( array, m, n, file, line,             &
                                   symb, unit )

      real, pointer :: array(:,:)
      integer :: m, n
      character(len=*) :: file, line
      character(len=*), optional :: symb, unit
      !------ API end ------

#ifdef _DEVLP
      character(len=80) :: symb2, unit2
      integer(kind=MF_ADDRESS) :: address

   !------ end of declarations -- execution starts hereafter  ------

      allocate( array(m,n) )
      ! no blank line here

      if( present(symb) ) then
         symb2 = symb
      else
         symb2 = ""
      end if

      if( present(unit) ) then
         unit2 = unit
      else
         unit2 = ""
      end if

      ! printing some infos on STDERR
      address = _MF_LOC_ANY_OBJ_(array)
      ! printing some infos on STDERR
#ifdef _64_BITS
      write(STDERR,"(A,Z16.16,A,I0,A,A,A)")                             &
#else
      write(STDERR,"(A,Z8.8,A,I0,A,A,A)")                               &
#endif
            "[+ addr: ", address,                                       &
            ", size: ", m*n*4,                                          &
            ", symb: ", trim(symb2),                                    &
            "]"
      write(STDERR,"(A,A,A,A,A,A,A)")                                   &
            "  (in unit: ", trim(unit2),                                &
            ", file: ", trim(file),                                     &
            ", line: ", trim(line), ")"

      call mf_traceback()

      call flush(STDERR)

#endif
   end subroutine mf_allocate_real_2D
!_______________________________________________________________________
!
   subroutine mf_allocate_double( array, n, file, line,                 &
                                  symb, unit )

      real(kind=MF_DOUBLE), pointer :: array(:)
      integer :: n
      character(len=*) :: file, line
      character(len=*), optional :: symb, unit
      !------ API end ------

#ifdef _DEVLP
      character(len=80) :: symb2, unit2
      integer(kind=MF_ADDRESS) :: address

   !------ end of declarations -- execution starts hereafter  ------

      allocate( array(n) )
      ! no blank line here

      if( present(symb) ) then
         symb2 = symb
      else
         symb2 = ""
      end if

      if( present(unit) ) then
         unit2 = unit
      else
         unit2 = ""
      end if

      ! printing some infos on STDERR
      address = _MF_LOC_ANY_OBJ_(array)
      ! printing some infos on STDERR
#ifdef _64_BITS
      write(STDERR,"(A,Z16.16,A,I0,A,A,A)")                             &
#else
      write(STDERR,"(A,Z8.8,A,I0,A,A,A)")                               &
#endif
            "[+ addr: ", address,                                       &
            ", size: ", n*8,                                            &
            ", symb: ", trim(symb2),                                    &
            "]"
      write(STDERR,"(A,A,A,A,A,A,A)")                                   &
            "  (in unit: ", trim(unit2),                                &
            ", file: ", trim(file),                                     &
            ", line: ", trim(line), ")"

      call mf_traceback()

      call flush(STDERR)

#endif
   end subroutine mf_allocate_double
!_______________________________________________________________________
!
   subroutine mf_allocate_double_2D( array, m, n, file, line,           &
                                     symb, unit )

      real(kind=MF_DOUBLE), pointer :: array(:,:)
      integer :: m, n
      character(len=*) :: file, line
      character(len=*), optional :: symb, unit
      !------ API end ------

#ifdef _DEVLP
      character(len=80) :: symb2, unit2
      integer(kind=MF_ADDRESS) :: address

   !------ end of declarations -- execution starts hereafter  ------

      allocate( array(m,n) )
      ! no blank line here

      if( present(symb) ) then
         symb2 = symb
      else
         symb2 = ""
      end if

      if( present(unit) ) then
         unit2 = unit
      else
         unit2 = ""
      end if

      address = _MF_LOC_ANY_OBJ_(array)
      ! printing some infos on STDERR
#ifdef _64_BITS
      write(STDERR,"(A,Z16.16,A,I0,A,A,A)")                             &
#else
      write(STDERR,"(A,Z8.8,A,I0,A,A,A)")                               &
#endif
            "[+ addr: ", address,                                       &
            ", size: ", m*n*8,                                          &
            ", symb: ", trim(symb2),                                    &
            "]"

      write(STDERR,"(A,A,A,A,A,A,A)")                                   &
            "  (in unit: ", trim(unit2),                                &
            ", file: ", trim(file),                                     &
            ", line: ", trim(line), ")"

      call mf_traceback()

      call flush(STDERR)

#endif
   end subroutine mf_allocate_double_2D
!_______________________________________________________________________
!
   subroutine mf_allocate_cmplx( array, n, file, line,                  &
                                 symb, unit )

      complex(kind=MF_DOUBLE), pointer :: array(:)
      integer :: n
      character(len=*) :: file, line
      character(len=*), optional :: symb, unit
      !------ API end ------

#ifdef _DEVLP
      character(len=80) :: symb2, unit2
      integer(kind=MF_ADDRESS) :: address

   !------ end of declarations -- execution starts hereafter  ------

      allocate( array(n) )
      ! no blank line here

      if( present(symb) ) then
         symb2 = symb
      else
         symb2 = ""
      end if

      if( present(unit) ) then
         unit2 = unit
      else
         unit2 = ""
      end if

      ! printing some infos on STDERR
      address = _MF_LOC_ANY_OBJ_(array)
      ! printing some infos on STDERR
#ifdef _64_BITS
      write(STDERR,"(A,Z16.16,A,I0,A,A,A)")                             &
#else
      write(STDERR,"(A,Z8.8,A,I0,A,A,A)")                               &
#endif
            "[+ addr: ", address,                                       &
            ", size: ", n*16,                                           &
            ", symb: ", trim(symb2),                                    &
            "]"
      write(STDERR,"(A,A,A,A,A,A,A)")                                   &
            "  (in unit: ", trim(unit2),                                &
            ", file: ", trim(file),                                     &
            ", line: ", trim(line), ")"

      call mf_traceback()

      call flush(STDERR)

#endif
   end subroutine mf_allocate_cmplx
!_______________________________________________________________________
!
   subroutine mf_allocate_cmplx_2D( array, m, n, file, line,            &
                                    symb, unit )

      complex(kind=MF_DOUBLE), pointer :: array(:,:)
      integer :: m, n
      character(len=*) :: file, line
      character(len=*), optional :: symb, unit
      !------ API end ------

#ifdef _DEVLP
      character(len=80) :: symb2, unit2
      integer(kind=MF_ADDRESS) :: address

   !------ end of declarations -- execution starts hereafter  ------

      allocate( array(m,n) )
      ! no blank line here

      if( present(symb) ) then
         symb2 = symb
      else
         symb2 = ""
      end if

      if( present(unit) ) then
         unit2 = unit
      else
         unit2 = ""
      end if

      ! printing some infos on STDERR
      address = _MF_LOC_ANY_OBJ_(array)
      ! printing some infos on STDERR
#ifdef _64_BITS
      write(STDERR,"(A,Z16.16,A,I0,A,A,A)")                             &
#else
      write(STDERR,"(A,Z8.8,A,I0,A,A,A)")                               &
#endif
            "[+ addr: ", address,                                       &
            ", size: ", m*n*16,                                         &
            ", symb: ", trim(symb2),                                    &
            "]"
      write(STDERR,"(A,A,A,A,A,A,A)")                                   &
            "  (in unit: ", trim(unit2),                                &
            ", file: ", trim(file),                                     &
            ", line: ", trim(line), ")"

      call mf_traceback()

      call flush(STDERR)

#endif
   end subroutine mf_allocate_cmplx_2D
!_______________________________________________________________________
!
   subroutine mf_deallocate_integer( array, file, line,                 &
                                     symb, unit )

      integer, pointer :: array(:)
      character(len=*) :: file, line
      character(len=*), optional :: symb, unit
      !------ API end ------

#ifdef _DEVLP
      character(len=80) :: symb2, unit2
      integer(kind=MF_ADDRESS) :: address

   !------ end of declarations -- execution starts hereafter  ------

      if( present(symb) ) then
         symb2 = symb
      else
         symb2 = ""
      end if

      if( present(unit) ) then
         unit2 = unit
      else
         unit2 = ""
      end if

      ! printing some infos on STDERR
      address = _MF_LOC_ANY_OBJ_(array)
      ! printing some infos on STDERR
#ifdef _64_BITS
      write(STDERR,"(A,Z16.16,A,A,A)")                                  &
#else
      write(STDERR,"(A,Z8.8,A,A,A)")                                    &
#endif
            "[- addr: ", address,                                       &
            ", symb: ", trim(symb2),                                    &
            "]"
      write(STDERR,"(A,A,A,A,A,A,A)")                                   &
            "  (in unit: ", trim(unit2),                                &
            ", file: ", trim(file),                                     &
            ", line: ", trim(line), ")"

      deallocate( array )
      ! no blank line here

      call mf_traceback()

      call flush(STDERR)

#endif
   end subroutine mf_deallocate_integer
!_______________________________________________________________________
!
   subroutine mf_deallocate_integer_2D( array, file, line,              &
                                        symb, unit )

      integer, pointer :: array(:,:)
      character(len=*) :: file, line
      character(len=*), optional :: symb, unit
      !------ API end ------

#ifdef _DEVLP
      character(len=80) :: symb2, unit2
      integer(kind=MF_ADDRESS) :: address

   !------ end of declarations -- execution starts hereafter  ------

      if( present(symb) ) then
         symb2 = symb
      else
         symb2 = ""
      end if

      if( present(unit) ) then
         unit2 = unit
      else
         unit2 = ""
      end if

      ! printing some infos on STDERR
      address = _MF_LOC_ANY_OBJ_(array)
      ! printing some infos on STDERR
#ifdef _64_BITS
      write(STDERR,"(A,Z16.16,A,A,A)")                                  &
#else
      write(STDERR,"(A,Z8.8,A,A,A)")                                    &
#endif
            "[- addr: ", address,                                       &
            ", symb: ", trim(symb2),                                    &
            "]"
      write(STDERR,"(A,A,A,A,A,A,A)")                                   &
            "  (in unit: ", trim(unit2),                                &
            ", file: ", trim(file),                                     &
            ", line: ", trim(line), ")"

      deallocate( array )
      ! no blank line here

      call mf_traceback()

      call flush(STDERR)

#endif
   end subroutine mf_deallocate_integer_2D
!_______________________________________________________________________
!
   subroutine mf_deallocate_character( array, file, line,               &
                                       symb, unit )

      character, pointer :: array(:)
      character(len=*) :: file, line
      character(len=*), optional :: symb, unit
      !------ API end ------

#ifdef _DEVLP
      character(len=80) :: symb2, unit2
      integer(kind=MF_ADDRESS) :: address

   !------ end of declarations -- execution starts hereafter  ------

      if( present(symb) ) then
         symb2 = symb
      else
         symb2 = ""
      end if

      if( present(unit) ) then
         unit2 = unit
      else
         unit2 = ""
      end if

      ! printing some infos on STDERR
      address = _MF_LOC_ANY_OBJ_(array)
      ! printing some infos on STDERR
#ifdef _64_BITS
      write(STDERR,"(A,Z16.16,A,A,A)")                                  &
#else
      write(STDERR,"(A,Z8.8,A,A,A)")                                    &
#endif
            "[- addr: ", address,                                       &
            ", symb: ", trim(symb2),                                    &
            "]"
      write(STDERR,"(A,A,A,A,A,A,A)")                                   &
            "  (in unit: ", trim(unit2),                                &
            ", file: ", trim(file),                                     &
            ", line: ", trim(line), ")"

      deallocate( array )
      ! no blank line here

      call mf_traceback()

      call flush(STDERR)

#endif
   end subroutine mf_deallocate_character
!_______________________________________________________________________
!
   subroutine mf_deallocate_real( array, file, line,                    &
                                  symb, unit )

      real, pointer :: array(:)
      character(len=*) :: file, line
      character(len=*), optional :: symb, unit
      !------ API end ------

#ifdef _DEVLP
      character(len=80) :: symb2, unit2
      integer(kind=MF_ADDRESS) :: address

   !------ end of declarations -- execution starts hereafter  ------

      if( present(symb) ) then
         symb2 = symb
      else
         symb2 = ""
      end if

      if( present(unit) ) then
         unit2 = unit
      else
         unit2 = ""
      end if

      ! printing some infos on STDERR
      address = _MF_LOC_ANY_OBJ_(array)
      ! printing some infos on STDERR
#ifdef _64_BITS
      write(STDERR,"(A,Z16.16,A,A,A)")                                  &
#else
      write(STDERR,"(A,Z8.8,A,A,A)")                                    &
#endif
            "[- addr: ", address,                                       &
            ", symb: ", trim(symb2),                                    &
            "]"
      write(STDERR,"(A,A,A,A,A,A,A)")                                   &
            "  (in unit: ", trim(unit2),                                &
            ", file: ", trim(file),                                     &
            ", line: ", trim(line), ")"

      deallocate( array )
      ! no blank line here

      call mf_traceback()

      call flush(STDERR)

#endif
   end subroutine mf_deallocate_real
!_______________________________________________________________________
!
   subroutine mf_deallocate_real_2D( array, file, line,                 &
                                     symb, unit )

      real, pointer :: array(:,:)
      character(len=*) :: file, line
      character(len=*), optional :: symb, unit
      !------ API end ------

#ifdef _DEVLP
      character(len=80) :: symb2, unit2
      integer(kind=MF_ADDRESS) :: address

   !------ end of declarations -- execution starts hereafter  ------

      if( present(symb) ) then
         symb2 = symb
      else
         symb2 = ""
      end if

      if( present(unit) ) then
         unit2 = unit
      else
         unit2 = ""
      end if

      ! printing some infos on STDERR
      address = _MF_LOC_ANY_OBJ_(array)
      ! printing some infos on STDERR
#ifdef _64_BITS
      write(STDERR,"(A,Z16.16,A,A,A)")                                  &
#else
      write(STDERR,"(A,Z8.8,A,A,A)")                                    &
#endif
            "[- addr: ", address,                                       &
            ", symb: ", trim(symb2),                                    &
            "]"
      write(STDERR,"(A,A,A,A,A,A,A)")                                   &
            "  (in unit: ", trim(unit2),                                &
            ", file: ", trim(file),                                     &
            ", line: ", trim(line), ")"

      deallocate( array )
      ! no blank line here

      call mf_traceback()

      call flush(STDERR)

#endif
   end subroutine mf_deallocate_real_2D
!_______________________________________________________________________
!
   subroutine mf_deallocate_double( array, file, line,                  &
                                    symb, unit )

      real(kind=MF_DOUBLE), pointer :: array(:)
      character(len=*) :: file, line
      character(len=*), optional :: symb, unit
      !------ API end ------

#ifdef _DEVLP
      character(len=80) :: symb2, unit2
      integer(kind=MF_ADDRESS) :: address

   !------ end of declarations -- execution starts hereafter  ------

      if( present(symb) ) then
         symb2 = symb
      else
         symb2 = ""
      end if

      if( present(unit) ) then
         unit2 = unit
      else
         unit2 = ""
      end if

      ! printing some infos on STDERR
      address = _MF_LOC_ANY_OBJ_(array)
      ! printing some infos on STDERR
#ifdef _64_BITS
      write(STDERR,"(A,Z16.16,A,A,A)")                                  &
#else
      write(STDERR,"(A,Z8.8,A,A,A)")                                    &
#endif
            "[- addr: ", address,                                       &
            ", symb: ", trim(symb2),                                    &
            "]"
      write(STDERR,"(A,A,A,A,A,A,A)")                                   &
            "  (in unit: ", trim(unit2),                                &
            ", file: ", trim(file),                                     &
            ", line: ", trim(line), ")"

      deallocate( array )
      ! no blank line here

      call mf_traceback()

      call flush(STDERR)

#endif
   end subroutine mf_deallocate_double
!_______________________________________________________________________
!
   subroutine mf_deallocate_double_2D( array, file, line,               &
                                       symb, unit )

      real(kind=MF_DOUBLE), pointer :: array(:,:)
      character(len=*) :: file, line
      character(len=*), optional :: symb, unit
      !------ API end ------

#ifdef _DEVLP
      character(len=80) :: symb2, unit2
      integer(kind=MF_ADDRESS) :: address

   !------ end of declarations -- execution starts hereafter  ------

      if( present(symb) ) then
         symb2 = symb
      else
         symb2 = ""
      end if

      if( present(unit) ) then
         unit2 = unit
      else
         unit2 = ""
      end if

      address = _MF_LOC_ANY_OBJ_(array)
      ! printing some infos on STDERR
#ifdef _64_BITS
      write(STDERR,"(A,Z16.16,A,A,A)")                                  &
#else
      write(STDERR,"(A,Z8.8,A,A,A)")                                    &
#endif
            "[- addr: ", address,                                       &
            ", symb: ", trim(symb2),                                    &
            "]"

      write(STDERR,"(A,A,A,A,A,A,A)")                                   &
            "  (in unit: ", trim(unit2),                                &
            ", file: ", trim(file),                                     &
            ", line: ", trim(line), ")"

      deallocate( array )
      ! no blank line here

      call mf_traceback()

      call flush(STDERR)

#endif
   end subroutine mf_deallocate_double_2D
!_______________________________________________________________________
!
   subroutine mf_deallocate_cmplx( array, file, line,                   &
                                   symb, unit )

      complex(kind=MF_DOUBLE), pointer :: array(:)
      character(len=*) :: file, line
      character(len=*), optional :: symb, unit
      !------ API end ------

#ifdef _DEVLP
      character(len=80) :: symb2, unit2
      integer(kind=MF_ADDRESS) :: address

   !------ end of declarations -- execution starts hereafter  ------

      if( present(symb) ) then
         symb2 = symb
      else
         symb2 = ""
      end if

      if( present(unit) ) then
         unit2 = unit
      else
         unit2 = ""
      end if

      ! printing some infos on STDERR
      address = _MF_LOC_ANY_OBJ_(array)
      ! printing some infos on STDERR
#ifdef _64_BITS
      write(STDERR,"(A,Z16.16,A,A,A)")                                  &
#else
      write(STDERR,"(A,Z8.8,A,A,A)")                                    &
#endif
            "[- addr: ", address,                                       &
            ", symb: ", trim(symb2),                                    &
            "]"
      write(STDERR,"(A,A,A,A,A,A,A)")                                   &
            "  (in unit: ", trim(unit2),                                &
            ", file: ", trim(file),                                     &
            ", line: ", trim(line), ")"

      deallocate( array )
      ! no blank line here

      call mf_traceback()

      call flush(STDERR)

#endif
   end subroutine mf_deallocate_cmplx
!_______________________________________________________________________
!
   subroutine mf_deallocate_cmplx_2D( array, file, line,                &
                                      symb, unit )

      complex(kind=MF_DOUBLE), pointer :: array(:,:)
      character(len=*) :: file, line
      character(len=*), optional :: symb, unit
      !------ API end ------

#ifdef _DEVLP
      character(len=80) :: symb2, unit2
      integer(kind=MF_ADDRESS) :: address

   !------ end of declarations -- execution starts hereafter  ------

      if( present(symb) ) then
         symb2 = symb
      else
         symb2 = ""
      end if

      if( present(unit) ) then
         unit2 = unit
      else
         unit2 = ""
      end if

      ! printing some infos on STDERR
      address = _MF_LOC_ANY_OBJ_(array)
      ! printing some infos on STDERR
#ifdef _64_BITS
      write(STDERR,"(A,Z16.16,A,A,A)")                                  &
#else
      write(STDERR,"(A,Z8.8,A,A,A)")                                    &
#endif
            "[- addr: ", address,                                       &
            ", symb: ", trim(symb2),                                    &
            "]"
      write(STDERR,"(A,A,A,A,A,A,A)")                                   &
            "  (in unit: ", trim(unit2),                                &
            ", file: ", trim(file),                                     &
            ", line: ", trim(line), ")"

      deallocate( array )
      ! no blank line here

      call mf_traceback()

      call flush(STDERR)

#endif
   end subroutine mf_deallocate_cmplx_2D
!_______________________________________________________________________
!
end module mod_mem_debug
