module mod_elmat ! Elementary Matrix Manipulation Functions

   ! Part of MUESLI Numerical Library
   ! Copyright É. Canot 2003-2025 -- IPR/CNRS

!-----------------------------------------------------------------------
!                             used modules
!-----------------------------------------------------------------------

   use mod_ops

#ifndef _DEVLP
   use mod_mfarray ! required for the 2nd pass of the double compilation
   use mod_mfdebug ! required for the 2nd pass of the double compilation
   use mod_core ! required for the 2nd pass of the double compilation
#endif

   implicit none

#ifndef _DEVLP
   private
#endif

!-----------------------------------------------------------------------
!                             interfaces
!-----------------------------------------------------------------------

#include "misc/RngStreams/RngStream_f90wrapper.inc"

   interface mfEye
      module procedure mfEye_one_arg
      module procedure mfEye_two_arg
   end interface mfEye
   !------ API end ------

   interface mfRand
      module procedure mfRand_one_arg
      module procedure mfRand_two_arg
      module procedure mfRand_seed
   end interface mfRand
   !------ API end ------

   interface mfRandN
      module procedure mfRandN_one_arg
      module procedure mfRandN_two_arg
   end interface mfRandN
   !------ API end ------

   interface mfSpRand
      module procedure mfSpRand_S
      module procedure mfSpRand_m_n
   end interface mfSpRand
   !------ API end ------

   interface mfSpRandN
      module procedure mfSpRandN_S
      module procedure mfSpRandN_m_n
   end interface mfSpRandN
   !------ API end ------

   interface mfRandPoiss
      module procedure mfRandPoiss_one_arg
      module procedure mfRandPoiss_two_arg
   end interface mfRandPoiss
   !------ API end ------

   interface mfZeros
      module procedure mfZeros_one_arg
      module procedure mfZeros_two_arg
   end interface mfZeros
   !------ API end ------

   interface mfOnes
      module procedure mfOnes_one_arg
      module procedure mfOnes_two_arg
   end interface mfOnes
   !------ API end ------

   interface mfReshape
      module procedure mfReshape_vec_int
      module procedure mfReshape_vec_real8
      module procedure mfReshape_array
      module procedure mfReshape_array_arg1_empty
      module procedure mfReshape_array_arg2_empty
   end interface mfReshape
   !------ API end ------

   interface operator(.vc.)
      module procedure vc_vec_real8_vec_real8
      module procedure vc_mfArray_vec_real8
      module procedure vc_vec_real8_mfArray
      module procedure vc_vec_cmplx8_vec_cmplx8
      module procedure vc_mfArray_vec_cmplx8
      module procedure vc_vec_cmplx8_mfArray
      module procedure vc_mfArray_mfArray
   end interface operator(.vc.)
   !------ API end ------

   interface operator(.hc.)
      module procedure hc_vec_real8_vec_real8
      module procedure hc_mfArray_vec_real8
      module procedure hc_vec_real8_mfArray
      module procedure hc_vec_cmplx8_vec_cmplx8
      module procedure hc_mfArray_vec_cmplx8
      module procedure hc_vec_cmplx8_mfArray
      module procedure hc_mfArray_mfArray
   end interface operator(.hc.)
   !------ API end ------

   interface msHorizConcat
      module procedure msHorizConcat_mfArray_mfArray
   end interface msHorizConcat
   !------ API end ------

   interface mfBlkDiag
      module procedure mfBlkDiag_mfArray_mfArray
      module procedure mfBlkDiag_mfArray_int
   end interface mfBlkDiag
   !------ API end ------

   interface mfPerm
      module procedure mfPerm_mfArray
      module procedure mfPerm_vec_int
   end interface mfPerm
   !------ API end ------

   interface mfLinSpace
      module procedure mfLinSpace_real4
      module procedure mfLinSpace_real8
      module procedure mfLinSpace_step_real4
      module procedure mfLinSpace_step_real8
   end interface mfLinSpace
   !------ API end ------

   interface mfLogSpace
      module procedure mfLogSpace_real4
      module procedure mfLogSpace_real8
      module procedure mfLogSpace_step_real4
      module procedure mfLogSpace_step_real8
   end interface mfLogSpace
   !------ API end ------

   !--------------------------------------------------------------------

   public :: mfEye, &
             mfLinSpace, &
             mfLogSpace, &
             mfMagic, &
             msMeshGrid, &
             mfOnes, &
             mfRand, msRand, &
             mfSpRand, &
             mfRandN, &
             mfSpRandN, &
             mfRandPoiss, &
             mfHilb, &
             mfInvHilb, &
             mfVander, &
             operator(.vc.), &
             operator(.hc.), &
             msHorizConcat, &
             mfRepMat, &
             mfZeros, &
             mfDiag, msDiag, &
             mfBlkDiag, &
             mfFind, msFind, &
             mfNonZeros, &
             mfReshape, msReshape, &
             mfTril, &
             mfTriu, &
             mfIsNaN, &
             mfIsFinite, &
             mfIsInf, &
             mfFlipLR, &
             mfFlipUD, &
             mfRot90, &
             mfCompan, &
             mfHankel, &
             mfToeplitz, &
             mfMerge, &
             mfPack, &
             mfUnpack, &
             mfCshift, &
             mfEoshift

   public :: mfKron ! should be in 'mod_ops', but it uses 'meshgrid'

   public :: mfPerm, &
             mfRandPerm, &
             mfCheckPerm

   ! usually, all routines are declared as 'private',
   ! excepted those from imported modules!
   private :: mfEye_one_arg, &
              mfEye_two_arg, &
              mfRand_one_arg, &
              mfRand_two_arg, &
              mfRand_seed, &
              mfRandN_one_arg, &
              mfRandN_two_arg, &
              mfSpRand_S, &
              mfSpRand_m_n, &
              mfSpRandN_S, &
              mfSpRandN_m_n, &
              mfRandPoiss_one_arg, &
              mfRandPoiss_two_arg, &
              mfZeros_one_arg, &
              mfZeros_two_arg, &
              mfOnes_one_arg, &
              mfOnes_two_arg, &
              mfReshape_vec_int, &
              mfReshape_vec_real8, &
              mfReshape_array

   private :: vc_vec_real8_vec_real8, &
              vc_mfArray_vec_real8, &
              vc_vec_real8_mfArray, &
              vc_vec_cmplx8_vec_cmplx8, &
              vc_mfArray_vec_cmplx8, &
              vc_vec_cmplx8_mfArray, &
              vc_mfArray_mfArray, &
              hc_vec_real8_vec_real8, &
              hc_mfArray_vec_real8, &
              hc_vec_real8_mfArray, &
              hc_vec_cmplx8_vec_cmplx8, &
              hc_mfArray_vec_cmplx8, &
              hc_vec_cmplx8_mfArray, &
              hc_mfArray_mfArray

   private :: msHorizConcat_mfArray_mfArray, &
              mfBlkDiag_mfArray_mfArray, &
              mfBlkDiag_mfArray_int, &
              mfPerm_mfArray, &
              mfPerm_vec_int

   logical                  :: RngStream_initialized = .false.
   integer(kind=MF_ADDRESS) :: Rng_Stream_g_addr = 0
   real(kind=MF_DOUBLE)     :: Rng_Stream_seed(6) = [0.0d0, 0.0d0, 0.0d0, &
                                                     0.0d0, 0.0d0, 0.0d0 ]
   ! non official, but public
   public :: RngStream_end

contains
!_______________________________________________________________________
!
#include "fml_ops/vert_concat.inc"
!_______________________________________________________________________
!
#include "fml_ops/horiz_concat.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/colconcat.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/colconcat_ip.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/rowconcat.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/getdia.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/getl.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/getu.inc"
!_______________________________________________________________________
!
#include "fml_sparse/util/reshape.inc"
!_______________________________________________________________________
!
   function mfEye_one_arg( n ) result( out )

      integer, intent(in) :: n
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i

   !------ end of declarations -- execution starts hereafter  ------

      if( n < 1 ) then
         call PrintMessage( "mfEye", "E",                               &
                            "bad dimension arg." )
         return
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ n, n ]
      allocate( out%double(n,n) )

      out%double(:,:) = 0.0d0
      do i = 1, n
         out%double(i,i) = 1.0d0
      end do

      out%prop%tril = TRUE
      out%prop%triu = TRUE
      out%prop%symm = TRUE
      out%prop%posd = TRUE

      out%status_temporary = .true.

#endif
   end function mfEye_one_arg
!_______________________________________________________________________
!
   function mfEye_two_arg( n1, n2 ) result( out )

      integer, intent(in) :: n1, n2
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i

   !------ end of declarations -- execution starts hereafter  ------

      if( n1 < 1 .or. n2 < 1 ) then
         call PrintMessage( "mfEye", "E",                               &
                            "bad dimension arg." )
         return
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ n1, n2 ]
      allocate( out%double(n1,n2) )

      out%double(:,:) = 0.0d0
      do i = 1, min(n1,n2)
         out%double(i,i) = 1.0d0
      end do

      out%prop%tril = TRUE
      out%prop%triu = TRUE
      if( n1 == n2 ) then
         out%prop%symm = TRUE
         out%prop%posd = TRUE
      else
         out%prop%symm = FALSE
         out%prop%posd = UNKNOWN
      end if

      out%status_temporary = .true.

#endif
   end function mfEye_two_arg
!_______________________________________________________________________
!
   function mfLinSpace_real4( start, end, nval ) result( out )

      real,    intent(in)           :: start, end
      integer, intent(in), optional :: nval
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: h_size, i, nval_ef
      real(kind=MF_DOUBLE) :: step

   !------ end of declarations -- execution starts hereafter  ------

      if( present(nval) ) then
         if( nval < 2 ) then
            call PrintMessage( "mfLinSpace", "E",                       &
                               "bad dimension arg. nval!" )
            return
         end if
         nval_ef = nval
      else
         nval_ef = 100
      end if

      ! size of returned array
      h_size = nval_ef
      step = dble(end - start)/(nval_ef-1)

      out%data_type = MF_DT_DBLE
      out%shape = [ 1, h_size ]
      allocate( out%double(1,h_size) )

      out%double(1,1) = start
      do i = 2, h_size-1
         out%double(1,i) = start + step*(i-1)
      end do
      out%double(1,h_size) = end

      out%prop%symm = FALSE
      out%prop%posd = FALSE

      out%status_temporary = .true.

#endif
   end function mfLinSpace_real4
!_______________________________________________________________________
!
   function mfLinSpace_real8( start, end, nval ) result( out )

      real(kind=MF_DOUBLE), intent(in)           :: start, end
      integer,              intent(in), optional :: nval
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: h_size, i, nval_ef
      real(kind=MF_DOUBLE) :: step

   !------ end of declarations -- execution starts hereafter  ------

      if( present(nval) ) then
         if( nval < 2 ) then
            call PrintMessage( "mfLinSpace", "E",                       &
                               "bad dimension arg. nval!" )
            return
         end if
         nval_ef = nval
      else
         nval_ef = 100
      end if

      ! size of returned array
      h_size = nval_ef
      step = (end - start)/(nval_ef-1)

      out%data_type = MF_DT_DBLE
      out%shape = [ 1, h_size ]
      allocate( out%double(1,h_size) )

      out%double(1,1) = start
      do i = 2, h_size-1
         out%double(1,i) = start + step*(i-1)
      end do
      out%double(1,h_size) = end

      out%prop%symm = FALSE
      out%prop%posd = FALSE

      out%status_temporary = .true.

#endif
   end function mfLinSpace_real8
!_______________________________________________________________________
!
   function mfLinSpace_step_real4( start, end, nval, step ) result( out )

      real,    intent(in), optional :: start, end
      integer, intent(in)           :: nval
      real,    intent(in)           :: step
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, nb_opt_arg

   !------ end of declarations -- execution starts hereafter  ------

      nb_opt_arg = 0
      if( present(start) ) then
         nb_opt_arg = nb_opt_arg + 1
      end if
      if( present(end) ) then
         nb_opt_arg = nb_opt_arg + 1
      end if

      if( nb_opt_arg /= 1 ) then
         call PrintMessage( "mfLinSpace", "E",                          &
                            "when 'nval' and 'step' are present, only one arg", &
                            "among 'start' and 'end' must be present!" )
         return
      end if

      if( step == 0.0d0 ) then
         call PrintMessage( "mfLinSpace", "E",                          &
                            "'step' must have a non-zero value!" )
         return
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ 1, nval ]
      allocate( out%double(1,nval) )

      if( present(start) ) then

         out%double(1,1) = start
         do i = 2, nval
            out%double(1,i) = dble(start) + dble(step)*(i-1)
         end do

      else ! present(end)

         out%double(1,nval) = end
         do i = nval-1, 1, -1
            out%double(1,i) = dble(end) - dble(step)*(nval-i)
         end do

      end if

      out%prop%symm = FALSE
      out%prop%posd = FALSE

      out%status_temporary = .true.

#endif
   end function mfLinSpace_step_real4
!_______________________________________________________________________
!
   function mfLinSpace_step_real8( start, end, nval, step ) result( out )

      real(kind=MF_DOUBLE), intent(in), optional :: start, end
      integer,              intent(in)           :: nval
      real(kind=MF_DOUBLE), intent(in)           :: step
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, nb_opt_arg

   !------ end of declarations -- execution starts hereafter  ------

      nb_opt_arg = 0
      if( present(start) ) then
         nb_opt_arg = nb_opt_arg + 1
      end if
      if( present(end) ) then
         nb_opt_arg = nb_opt_arg + 1
      end if

      if( nb_opt_arg /= 1 ) then
         call PrintMessage( "mfLinSpace", "E",                          &
                            "when 'nval' and 'step' are present, only one arg", &
                            "among 'start' and 'end' must be present!" )
         return
      end if

      if( step == 0.0d0 ) then
         call PrintMessage( "mfLinSpace", "E",                          &
                            "'step' must have a non-zero value!" )
         return
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ 1, nval ]
      allocate( out%double(1,nval) )

      if( present(start) ) then

         out%double(1,1) = start
         do i = 2, nval
            out%double(1,i) = start + step*(i-1)
         end do

      else ! present(end)

         out%double(1,nval) = end
         do i = nval-1, 1, -1
            out%double(1,i) = end - step*(nval-i)
         end do

      end if

      out%prop%symm = FALSE
      out%prop%posd = FALSE

      out%status_temporary = .true.

#endif
   end function mfLinSpace_step_real8
!_______________________________________________________________________
!
   function mfLogSpace_real4( start, end, nval ) result( out )

      real,    intent(in)           :: start, end
      integer, intent(in), optional :: nval
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: h_size, i, nval_ef
      real(kind=MF_DOUBLE) :: step

   !------ end of declarations -- execution starts hereafter  ------

      if( present(nval) ) then
         if( nval < 2 ) then
            call PrintMessage( "mfLogSpace", "E",                       &
                               "bad dimension arg. nval!" )
            return
         end if
         nval_ef = nval
      else
         nval_ef = 20
      end if

      ! size of returned array
      h_size = nval_ef
      step = dble(end - start)/(nval_ef-1)

      out%data_type = MF_DT_DBLE
      out%shape = [ 1, h_size ]
      allocate( out%double(1,h_size) )

      do i = 1, h_size-1
         out%double(1,i) = 10.0d0**( start + step*(i-1) )
      end do
      out%double(1,h_size) = 10.0d0**end

      out%prop%symm = FALSE
      out%prop%posd = FALSE

      out%status_temporary = .true.

#endif
   end function mfLogSpace_real4
!_______________________________________________________________________
!
   function mfLogSpace_real8( start, end, nval ) result( out )

      real(kind=MF_DOUBLE), intent(in)           :: start, end
      integer,              intent(in), optional :: nval
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: h_size, i, nval_ef
      real(kind=MF_DOUBLE) :: step

   !------ end of declarations -- execution starts hereafter  ------

      if( present(nval) ) then
         if( nval < 2 ) then
            call PrintMessage( "mfLogSpace", "E",                       &
                               "bad dimension arg. nval!" )
            return
         end if
         nval_ef = nval
      else
         nval_ef = 20
      end if

      ! size of returned array
      h_size = nval_ef
      step = (end - start)/(nval_ef-1)

      out%data_type = MF_DT_DBLE
      out%shape = [ 1, h_size ]
      allocate( out%double(1,h_size) )

      do i = 1, h_size-1
         out%double(1,i) = 10.0d0**( start + step*(i-1) )
      end do
      out%double(1,h_size) = 10.0d0**end

      out%prop%symm = FALSE
      out%prop%posd = FALSE

      out%status_temporary = .true.

#endif
   end function mfLogSpace_real8
!_______________________________________________________________________
!
   function mfLogSpace_step_real4( start, end, nval, step ) result( out )

      real,    intent(in), optional :: start, end
      integer, intent(in)           :: nval
      real,    intent(in)           :: step
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, nb_opt_arg

   !------ end of declarations -- execution starts hereafter  ------

      nb_opt_arg = 0
      if( present(start) ) then
         nb_opt_arg = nb_opt_arg + 1
      end if
      if( present(end) ) then
         nb_opt_arg = nb_opt_arg + 1
      end if

      if( nb_opt_arg /= 1 ) then
         call PrintMessage( "mfLogSpace", "E",                          &
                            "when 'nval' and 'step' are present, only one arg", &
                            "among 'start' and 'end' must be present!" )
         return
      end if

      if( step == 0.0d0 ) then
         call PrintMessage( "mfLogSpace", "E",                          &
                            "'step' must have a non-zero value!" )
         return
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ 1, nval ]
      allocate( out%double(1,nval) )

      if( present(start) ) then

         out%double(1,1) = 10.0d0**dble(start)
         do i = 2, nval
            out%double(1,i) = 10.0d0**( dble(start) + dble(step)*(i-1) )
         end do

      else ! present(end)

         out%double(1,nval) = 10.0d0**dble(end)
         do i = nval-1, 1, -1
            out%double(1,i) = 10.0d0**( dble(end) - dble(step)*(nval-i) )
         end do

      end if

      out%prop%symm = FALSE
      out%prop%posd = FALSE

      out%status_temporary = .true.

#endif
   end function mfLogSpace_step_real4
!_______________________________________________________________________
!
   function mfLogSpace_step_real8( start, end, nval, step ) result( out )

      real(kind=MF_DOUBLE), intent(in), optional :: start, end
      integer,              intent(in)           :: nval
      real(kind=MF_DOUBLE), intent(in)           :: step
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, nb_opt_arg

   !------ end of declarations -- execution starts hereafter  ------

      nb_opt_arg = 0
      if( present(start) ) then
         nb_opt_arg = nb_opt_arg + 1
      end if
      if( present(end) ) then
         nb_opt_arg = nb_opt_arg + 1
      end if

      if( nb_opt_arg /= 1 ) then
         call PrintMessage( "mfLogSpace", "E",                          &
                            "when 'nval' and 'step' are present, only one arg", &
                            "among 'start' and 'end' must be present!" )
         return
      end if

      if( step == 0.0d0 ) then
         call PrintMessage( "mfLogSpace", "E",                          &
                            "'step' must have a non-zero value!" )
         return
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ 1, nval ]
      allocate( out%double(1,nval) )

      if( present(start) ) then

         out%double(1,1) = 10.0d0**start
         do i = 2, nval
            out%double(1,i) = 10.0d0**( start + step*(i-1) )
         end do

      else ! present(end)

         out%double(1,nval) = 10.0d0**end
         do i = nval-1, 1, -1
            out%double(1,i) = 10.0d0**( end - step*(nval-i) )
         end do

      end if

      out%prop%symm = FALSE
      out%prop%posd = FALSE

      out%status_temporary = .true.

#endif
   end function mfLogSpace_step_real8
!_______________________________________________________________________
!
   function mfMagic( n ) result( out )

      integer, intent(in) :: n
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( n < 3 ) then
         call PrintMessage( "mfMagic", "E",                             &
                            "n must be >= 3!" )
         return
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ n, n ]
      allocate( out%double(n,n) )

      if( mod(n,2) == 0 ) then
         call even_num( out%double )
      else
         call odd_num( out%double )
      end if

      out%prop%tril = FALSE
      out%prop%triu = FALSE
      out%prop%symm = FALSE
      out%prop%posd = FALSE

      out%status_temporary = .true.

   contains
!--------------------------------
      subroutine odd_num( mat )

         real(kind=MF_DOUBLE), intent(out) :: mat(0:,0:)

         ! Algorithm from: Shin, Kwon Young

         integer :: i, j, num, nn

         num = 1
         nn = (n*3)/2

         do i = 0, n-1
            do j = 0, n-1
               mat( mod(j-i+nn,n), mod(i*2-j+n,n) ) = num
               num = num + 1
            end do
         end do

      end subroutine odd_num
!--------------------------------
      subroutine even_num( mat )

         real(kind=MF_DOUBLE), intent(out) :: mat(0:,0:)

         ! Algorithm from: Shin, Kwon Young

         integer :: i, j, num
         integer :: nminus, nmiddle, nn
         integer :: osl
         integer :: switch_row(0:1)
         integer :: last_switch_column
         integer :: first_block, second_block
         integer :: first_inside, second_inside

         num = 1
         nminus = n - 1
         nmiddle = n/2
         nn = n*n+1
         osl = 0
         first_block = (n-2)/4
         second_block = nminus-first_block
         first_inside = n/4
         second_inside = nminus-first_inside

         do j = 0, n-1
            do i = 0, n-1
               if( i >= first_inside .and. i <= second_inside .and.     &
                   j >= first_inside .and. j <= second_inside ) then
                  mat(i,j) = num
               else if( (i > first_block .and. i < second_block) .or.   &
                        (j > first_block .and. j < second_block) ) then
                  mat(i,j) = nn-num
               else
                  mat(i,j) = num
               end if
               num = num + 1
            end do
         end do
         if( mod(n,4) == 0 ) return

         switch_row(0) = nmiddle
         switch_row(1) = 0
         last_switch_column = 0

         do i = 0, nmiddle-1
            if( i == first_block .or. i == second_block ) then
               osl = 1 - osl
               cycle
            end if
            call swap( second_block, i, second_block, nminus-i, mat )
            call swap( i, first_block, nminus-i, first_block, mat )
            call swap( i, second_block, nminus-i, second_block, mat )
            call swap( i, switch_row(osl), nminus-i, switch_row(osl), mat )
         end do
         do i = first_block+1, second_block-1
            call swap( first_block, i, second_block, i, mat )
            call swap( i, first_block, i, second_block, mat )
         end do
         call swap( first_block, nmiddle, second_block, nmiddle, mat )
         call swap( last_switch_column, first_block,                    &
                    last_switch_column, second_block, mat )

      end subroutine even_num
!--------------------------------
      subroutine swap( i1, j1, i2, j2, mat )
         integer, intent(in) :: i1, j1, i2, j2
         real(kind=MF_DOUBLE) :: mat(0:,0:)
         real(kind=MF_DOUBLE) :: k
         k = mat(i1,j1)
         mat(i1,j1) = mat(i2,j2)
         mat(i2,j2) = k
      end subroutine swap
!--------------------------------
#endif
   end function mfMagic
!_______________________________________________________________________
!
   subroutine msMeshGrid( out, v_x, v_y )

      type(mfArray) :: v_x, v_y
      type(mf_Out) :: out
      !------ API end ------

#ifdef _DEVLP
      type(mfArray), pointer :: X, Y
      integer :: m, n, dim_x, dim_y

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( v_x, v_y )

      ! we must have two output arguments
      if( out%n /= 2 ) then
         call PrintMessage( "msMeshGrid", "E",                          &
                            "two output args required!",                &
                            "syntax is : call msMeshGrid ( mfOut(X,Y), v_x, v_y )" )
         go to 99
      end if

      ! internal check for all mfOut args
      if( .not. args_mfout_ok( out, v_x, v_y ) ) then
         call PrintMessage( "msMeshGrid", "E",                          &
                            "output arguments cannot be tempo, or cannot share", &
                            "same memory as another input argument." )
         go to 99
      end if

      if( mfIsSparse(v_x) .or. mfIsSparse(v_y) ) then
         call PrintMessage( "msMeshGrid", "E",                          &
                            "args cannot be sparse!" )
         go to 99
      end if

      if( .not. ( mfIsReal(v_x) .and. mfIsReal(v_y) ) ) then
         call PrintMessage( "msMeshGrid", "E",                          &
                            "args must be real!" )
         go to 99
      end if

      X => out%ptr1
      Y => out%ptr2
      call msSilentRelease( X, Y )

      if( mfIsEmpty(v_x) .or. mfIsEmpty(v_y) ) then
         go to 99
      end if

      if( v_x%shape(1) /= 1 ) then
         if( v_x%shape(2) /= 1 ) then
            call PrintMessage( "msMeshGrid", "E",                       &
                               "mfArray 'v_x' must be a vector!" )
            go to 99
         else
            dim_x = 1
         end if
      else
         dim_x = 2
      end if

      if( v_y%shape(1) /= 1 ) then
         if( v_y%shape(2) /= 1 ) then
            call PrintMessage( "msMeshGrid", "E",                       &
                               "mfArray 'v_y' must be a vector!" )
            go to 99
         else
            dim_y = 1
         end if
      else
         dim_y = 2
      end if

      if( dim_x == dim_y ) then
         call PrintMessage( "msMeshGrid", "E",                          &
                            "bad args shape: you must provide one row vector", &
                            "and one column vector!" )
         go to 99
      end if

      if( dim_x == 1 ) then
         m = v_x%shape(1)
         n = v_y%shape(2)
         call msAssign( X, mfRepMat( v_x, 1, n ))
         call msAssign( Y, mfRepMat( v_y, m, 1 ))
      else ! dim_x = 2
         m = v_y%shape(1)
         n = v_x%shape(2)
         call msAssign( X, mfRepMat( v_x, m, 1 ))
         call msAssign( Y, mfRepMat( v_y, 1, n ))
      end if

      X%prop%posd = FALSE
      Y%prop%posd = FALSE

      if( mf_phys_units ) then
         X%units(:) = v_x%units(:)
         Y%units(:) = v_y%units(:)
      end if

 99   continue

      call msFreeArgs( v_x, v_y )

      call msAutoRelease( v_x, v_y )

#endif
   end subroutine msMeshGrid
!_______________________________________________________________________
!
   subroutine RngStream_init()

#ifdef _DEVLP
      call rng_create_stream( Rng_Stream_g_addr )

      ! saving the seed
      call rng_get_state( Rng_Stream_g_addr, Rng_Stream_seed )

      RngStream_initialized = .true.

#endif
   end subroutine RngStream_init
!_______________________________________________________________________
!
   subroutine RngStream_end()

#ifdef _DEVLP
      ! not used by any official Muesli routine...
      !
      ! introduced just for freeing memory when check programs with Valgrind...

      call rng_delete_stream( Rng_Stream_g_addr )

      ! deleting the seed
      Rng_Stream_seed(:) = 0.0d0

      RngStream_initialized = .false.

#endif
   end subroutine RngStream_end
!_______________________________________________________________________
!
   function mfRand_one_arg( n ) result( out )

      integer, intent(in) :: n
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      if( n < 1 ) then
         call PrintMessage( "mfRand", "E",                              &
                            "bad arg value!" )
         return
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ n, n ]
      allocate( out%double(n,n) )

      if( .not. RngStream_initialized ) then
         call RngStream_init()
      end if

      do j = 1, n
         do i = 1, n
            call rng_rand_u01( Rng_Stream_g_addr, out%double(i,j) )
         end do
      end do

      out%prop%tril = FALSE
      out%prop%triu = FALSE
      out%prop%symm = FALSE

      out%status_temporary = .true.

#endif
   end function mfRand_one_arg
!_______________________________________________________________________
!
   function mfRand_two_arg( n1, n2 ) result( out )

      integer, intent(in) :: n1, n2
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      if( n1 < 1 .or. n2 < 1 ) then
         call PrintMessage( "mfRand", "E",                              &
                            "bad arg value for at least one arg.!" )
         return
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ n1, n2 ]
      allocate( out%double(n1,n2) )

      if( .not. RngStream_initialized ) then
         call RngStream_init()
      end if

      do j = 1, n2
         do i = 1, n1
            call rng_rand_u01( Rng_Stream_g_addr, out%double(i,j) )
         end do
      end do

      out%prop%tril = FALSE
      out%prop%triu = FALSE
      out%prop%symm = FALSE

      out%status_temporary = .true.

#endif
   end function mfRand_two_arg
!_______________________________________________________________________
!
   function mfRand_seed( string ) result( out )

      character(len=*), intent(in) :: string
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! return an mfArray containing the seed

   !------ end of declarations -- execution starts hereafter  ------

      if( string /=  "seed" ) then
         call PrintMessage( "mfRand", "E",                              &
                            "bad arg value: if character, must be 'seed'!" )
         return
      end if

      if( .not. RngStream_initialized ) then
         call RngStream_init()
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ 1, 6 ]
      allocate( out%double(1,6) )

      out%double(1,:) = Rng_Stream_seed(:)

      out%status_temporary = .true.

#endif
   end function mfRand_seed
!_______________________________________________________________________
!
   subroutine msRand( seed )

      type(mfArray), intent(in), optional :: seed
      !------ API end ------

#ifdef _DEVLP
      ! set the seed either from given values, either from the clock

      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      if( .not. RngStream_initialized ) then
         call RngStream_init()
      end if

      if( present(seed) ) then ! set the seed from given values

         if( seed%data_type /= MF_DT_DBLE ) then
            call PrintMessage( "msRand", "E",                              &
                               "'seed' have bad datatype!" )
            return
         end if
         if( ( seed%shape(1) /= 1 .or. seed%shape(2) /= 6 ) .and.          &
             ( seed%shape(1) /= 6 .or. seed%shape(2) /= 1 )  ) then
            call PrintMessage( "msRand", "E",                              &
                               "'seed' have bad dimensions!",              &
                               "(it must be a vector of length equal to 6)" )
            return
         end if

         Rng_Stream_seed(:) = seed%double(1,:)
         if( any(Rng_Stream_seed(1:3) >= 4294967087.0d0) ) then
            call PrintMessage( "msRand", "E",                              &
                               "'seed(1:3)' must be less than 4294967087!" )
            return
         end if
         if( any(Rng_Stream_seed(4:6) >= 4294944443.0d0) ) then
            call PrintMessage( "msRand", "E",                              &
                               "'seed(4:6)' must be less than 4294944443!" )
            return
         end if
         call rng_set_seed( Rng_Stream_g_addr, Rng_Stream_seed )

      else ! set the seed from the clock

         call system_clock( count=j )
         ! as j is declared integer(kind=4), the output value is always
         ! positive and less than huge(j) ~ 2.e9, so this value is less
         ! than the two maximum values above...
         Rng_Stream_seed(:) = j + 50000*[ (i-1,i=1,6) ]
         call rng_set_seed( Rng_Stream_g_addr, Rng_Stream_seed )

      end if

#endif
   end subroutine msRand
!_______________________________________________________________________
!
   function randn( ) result( out )

      double precision :: out
      !------ API end ------

#ifdef _DEVLP
      ! Returns a normally distributed deviate with zero mean and unit
      ! variance. The routine uses the Box-Muller transformation of
      ! uniform deviates.
      ! See Press et al., Numerical Recipes, 1992, Sec. 7.2.

      double precision :: x, y, r

      integer, save :: i_save = 0
      double precision, save :: out_save

   !------ end of declarations -- execution starts hereafter  ------

      if( i_save == 0 ) then

         do

            call rng_rand_u01( Rng_Stream_g_addr, x )
            x = 2.0d0*x - 1.0d0
            call rng_rand_u01( Rng_Stream_g_addr, y )
            y = 2.0d0*y - 1.0d0
            r = x**2 + y**2
            if( r >= 1.0d0 .or. r == 0.0d0 ) then
               cycle
            else
               exit
            end if

         end do

         out      = x*sqrt(-2.0d0*log(r)/r)
         out_save = y*sqrt(-2.0d0*log(r)/r)

         i_save = 1

      else

         out = out_save
         i_save = 0

      end if

#endif
   end function randn
!_______________________________________________________________________
!
   function mfRandN_one_arg( n ) result( out )

      integer, intent(in) :: n
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      if( n < 1 ) then
         call PrintMessage( "mfRandN", "E",                             &
                            "bad arg value!" )
         return
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ n, n ]
      allocate( out%double(n,n) )

      if( .not. RngStream_initialized ) then
         call RngStream_init()
      end if

      do j = 1, n
         do i = 1, n
            out%double(i,j) = randn()
        end do
      end do

      out%prop%tril = FALSE
      out%prop%triu = FALSE
      out%prop%symm = FALSE

      out%status_temporary = .true.

#endif
   end function mfRandN_one_arg
!_______________________________________________________________________
!
   function mfRandN_two_arg( n1, n2 ) result( out )

      integer, intent(in) :: n1, n2
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      if( n1 < 1 .or. n2 < 1 ) then
         call PrintMessage( "mfRandN", "E",                             &
                            "bad arg value for at least one arg.!" )
         return
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ n1, n2 ]
      allocate( out%double(n1,n2) )

      if( .not. RngStream_initialized ) then
         call RngStream_init()
      end if

      do j = 1, n2
         do i = 1, n1
            out%double(i,j) = randn()
        end do
      end do

      out%prop%tril = FALSE
      out%prop%triu = FALSE
      out%prop%symm = FALSE

      out%status_temporary = .true.

#endif
   end function mfRandN_two_arg
!_______________________________________________________________________
!
   function mfHilb( n ) result( out )

      integer, intent(in) :: n
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      if( n < 1 ) then
         call PrintMessage( "mfHilb", "E",                              &
                            "bad arg value!" )
         return
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ n, n ]
      allocate( out%double(n,n) )

      do j = 1, n
         do i = 1, n
            out%double(i,j) = 1.0d0/(i+j-1.0d0)
         end do
      end do

      out%prop%tril = FALSE
      out%prop%triu = FALSE
      out%prop%symm = TRUE
      out%prop%posd = TRUE

      out%status_temporary = .true.

#endif
   end function mfHilb
!_______________________________________________________________________
!
   function mfInvHilb( n ) result( out )

      integer, intent(in) :: n
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j
      real(kind=MF_DOUBLE) :: p, r

   !------ end of declarations -- execution starts hereafter  ------

      if( n < 1 ) then
         call PrintMessage( "mfInvHilb", "E",                           &
                            "bad arg value!" )
         return
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ n, n ]
      allocate( out%double(n,n) )

      p = n

      i = 1
      r = p**2
      out%double(i,i) = r/(2.0d0*i-1.0d0)
      do j = i+1, n
         r = -((n-j+1.0d0)*r*(n+j-1.0d0))/(j-1.0d0)**2
         out%double(i,j) = r/(i+j-1.0d0)
         out%double(j,i) = out%double(i,j)
      end do

      do i = 2, n
         p = ((n-i+1.0d0)*p*(n+i-1.0d0))/(i-1.0d0)**2
         r = p**2
         out%double(i,i) = r/(2.0d0*i-1.0d0)
         do j = i+1, n
            r = -((n-j+1.0d0)*r*(n+j-1.0d0))/(j-1.0d0)**2
            out%double(i,j) = r/(i+j-1.0d0)
            out%double(j,i) = out%double(i,j)
         end do
      end do

      out%prop%tril = FALSE
      out%prop%triu = FALSE
      out%prop%symm = TRUE
      out%prop%posd = TRUE

      out%status_temporary = .true.

#endif
   end function mfInvHilb
!_______________________________________________________________________
!
   function mfRepmat( A, m, n ) result( out )

      type(mfArray) :: A
      integer, intent(in) :: m, n
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: mm, nn, i, j, i_deb, i_fin, j_deb, j_fin
      type(mfArray) :: C

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfRepmat", "E",                            &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( m < 1 .or. n < 1 ) then
         call PrintMessage( "mfRepmat", "E",                            &
                            "bad arg. value for m or n!" )
         go to 99
      end if

      if( mfIsSparse(A) ) then

         ! first, using operator(.vc.)
         C = A
         do i = 1, m-1
            C = C .vc. A
         end do

         do j = 1, n
            call msHorizConcat( out, C )
         end do

      else

         mm = m * A%shape(1)
         nn = n * A%shape(2)

         out%data_type = A%data_type
         out%shape = [ mm, nn ]
         if( A%data_type == MF_DT_DBLE .or. A%data_type == MF_DT_BOOL ) then
            allocate( out%double(mm,nn) )

            do j = 1, n
               j_deb = (j-1)*A%shape(2)+1
               j_fin =     j*A%shape(2)
               do i = 1, m
                  i_deb = (i-1)*A%shape(1)+1
                  i_fin =     i*A%shape(1)
                  out%double(i_deb:i_fin,j_deb:j_fin) = A%double(:,:)
               end do
            end do
         else
            allocate( out%cmplx(mm,nn) )

            do j = 1, n
               j_deb = (j-1)*A%shape(2)+1
               j_fin =     j*A%shape(2)
               do i = 1, m
                  i_deb = (i-1)*A%shape(1)+1
                  i_fin =     i*A%shape(1)
                  out%cmplx(i_deb:i_fin,j_deb:j_fin) = A%cmplx(:,:)
               end do
            end do
         end if

      end if

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfRepmat
!_______________________________________________________________________
!
   function mfReshape_vec_int( a, m, n ) result( out )

      integer, intent(in) :: a(:)
      integer, intent(in) :: m, n
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( m < 0 .or. m*n /= size(a) ) then
         call PrintMessage( "mfReshape", "E",                           &
                            "bad arg. value for m or n!" )
         return
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ m, n ]
      allocate( out%double(m,n) )

      out%double = reshape( a, [ m, n ] )

      out%status_temporary = .true.

#endif
   end function mfReshape_vec_int
!_______________________________________________________________________
!
   function mfReshape_vec_real8( a, m, n ) result( out )

      real(kind=MF_DOUBLE), intent(in) :: a(:)
      integer, intent(in) :: m, n
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( m < 0 .or. m*n /= size(a) ) then
         call PrintMessage( "mfReshape", "E",                           &
                            "bad arg. value for m or n!" )
         return
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ m, n ]
      allocate( out%double(m,n) )

      out%double = reshape( a, [ m, n ] )

      out%status_temporary = .true.

#endif
   end function mfReshape_vec_real8
!_______________________________________________________________________
!
   function mfReshape_array( A, m, n ) result( out )

      type(mfArray) :: A
      integer, intent(in) :: m, n
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: nrow, ncol, nnz

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfReshape", "E",                           &
                            "mfArray not allocated!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfReshape", "E",                           &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( m < 0 .or. m*n /= A%shape(1)*A%shape(2) ) then
         call PrintMessage( "mfReshape", "E",                           &
                            "bad arg. value for m or n!" )
         go to 99
      end if

      out%data_type = A%data_type
      out%shape = [ m, n ]
      if( mfIsSparse(A) ) then

         nrow = A%shape(1)
         ncol = A%shape(2)
         nnz = A%j(ncol+1) - 1
         if( .not. mfIsRowSorted(A) ) then
            call msRowSort(A)
         end if
            allocate( out%i(nnz) )

            allocate( out%j(n+1) )

         if( A%data_type == MF_DT_SP_DBLE ) then
            allocate( out%a(nnz) )

            call reshape_sparse( nrow, ncol, A%a, A%i, A%j,             &
                                 m, n, out%a, out%i, out%j )
         else
            allocate( out%z(nnz) )

            call reshape_sparse_cmplx( nrow, ncol, A%z, A%i, A%j,       &
                                       m, n, out%z, out%i, out%j )
         end if
      else

         if( A%data_type == MF_DT_DBLE .or. A%data_type == MF_DT_BOOL ) then
            allocate( out%double(m,n) )

            out%double = reshape( A%double, [ m, n ] )
         else if( A%data_type == MF_DT_CMPLX ) then
            allocate( out%cmplx(m,n) )

            out%cmplx = reshape( A%cmplx, [ m, n ] )
         end if

      end if

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfReshape_array
!_______________________________________________________________________
!
   function mfReshape_array_arg1_empty( a, E, n ) result( out )

      type(mfArray) :: a, E
      integer, intent(in) :: n
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! idem reshape(A,[],n) of Matlab
      ! (first dimension is such that all elements of A fill
      !  exactly the new mfArray)

      integer :: m

      integer :: nrow, ncol, nnz

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfReshape", "E",                           &
                            "mfArray not allocated!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfReshape", "E",                           &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( .not. ( mfIsEmpty(E) .or.                                     &
                  (E%shape(1)==-3 .and. E%shape(2)==-3) ) ) then
         call PrintMessage( "mfReshape", "E",                           &
                            "arg. 'E' must be an empty mfArray, or MF_NO_ARG!" )
         go to 99
      end if

      if( n <= 0 ) then
         call PrintMessage( "mfReshape", "E",                           &
                            "arg. 'n' cannot be negative or zero!" )
         go to 99
      end if

      nrow = A%shape(1)
      ncol = A%shape(2)

      if( mod(nrow*ncol,n) /= 0 ) then
         call PrintMessage( "mfReshape", "E",                           &
                            "bad value for arg. 'n' : cannot employ all elements!", &
                            "('n' must be a divisor of the total number of elements)" )
         go to 99
      end if

      m = nrow*ncol / n

      out%data_type = A%data_type
      out%shape = [ m, n ]
      if( mfIsSparse(A) ) then

         nnz = A%j(ncol+1) - 1
         if( .not. mfIsRowSorted(A) ) then
            call msRowSort(A)
         end if
            allocate( out%i(nnz) )

            allocate( out%j(n+1) )

         if( A%data_type == MF_DT_SP_DBLE ) then
            allocate( out%a(nnz) )

            call reshape_sparse( nrow, ncol, A%a, A%i, A%j,             &
                                 m, n, out%a, out%i, out%j )
         else
            allocate( out%z(nnz) )

            call reshape_sparse_cmplx( nrow, ncol, A%z, A%i, A%j,       &
                                       m, n, out%z, out%i, out%j )
         end if
      else

         if( A%data_type == MF_DT_DBLE .or. A%data_type == MF_DT_BOOL ) then
            allocate( out%double(m,n) )

            out%double = reshape( A%double, [ m, n ] )
         else if( A%data_type == MF_DT_CMPLX ) then
            allocate( out%cmplx(m,n) )

            out%cmplx = reshape( A%cmplx, [ m, n ] )
         end if

      end if

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfReshape_array_arg1_empty
!_______________________________________________________________________
!
   function mfReshape_array_arg2_empty( a, m, E ) result( out )

      type(mfArray) :: a, E
      integer, intent(in) :: m
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! idem reshape(A,[],n) of Matlab
      ! (first dimension is such that all elements of A fill
      !  exactly the new mfArray)

      integer :: n

      integer :: nrow, ncol, nnz

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfReshape", "E",                           &
                            "mfArray not allocated!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfReshape", "E",                           &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( .not. ( E%data_type == MF_DT_EMPTY .and. all( E%shape == 0 ) .or. &
                  (E%shape(1)==-3 .and. E%shape(2)==-3) ) ) then
         call PrintMessage( "mfReshape", "E",                           &
                            "arg. 'E' must be an empty mfArray, or MF_NO_ARG!" )
         go to 99
      end if

      if( m <= 0 ) then
         call PrintMessage( "mfReshape", "E",                           &
                            "arg. 'm' cannot be negative or zero!" )
         go to 99
      end if

      nrow = A%shape(1)
      ncol = A%shape(2)

      if( mod(nrow*ncol,m) /= 0 ) then
         call PrintMessage( "mfReshape", "E",                           &
                            "bad value for arg. 'm' : cannot employ all elements!", &
                            "('m' must be a divisor of the total number of elements)" )
         go to 99
      end if

      n = nrow*ncol / m

      out%data_type = A%data_type
      out%shape = [ m, n ]
      if( mfIsSparse(A) ) then

         nnz = A%j(ncol+1) - 1
         if( .not. mfIsRowSorted(A) ) then
            call msRowSort(A)
         end if
            allocate( out%i(nnz) )

            allocate( out%j(n+1) )

         if( A%data_type == MF_DT_SP_DBLE ) then
            allocate( out%a(nnz) )

            call reshape_sparse( nrow, ncol, A%a, A%i, A%j,             &
                                 m, n, out%a, out%i, out%j )
         else
            allocate( out%z(nnz) )

            call reshape_sparse_cmplx( nrow, ncol, A%z, A%i, A%j,       &
                                       m, n, out%z, out%i, out%j )
         end if
      else

         if( A%data_type == MF_DT_DBLE .or. A%data_type == MF_DT_BOOL ) then
            allocate( out%double(m,n) )

            out%double = reshape( A%double, [ m, n ] )
         else if( A%data_type == MF_DT_CMPLX ) then
            allocate( out%cmplx(m,n) )

            out%cmplx = reshape( A%cmplx, [ m, n ] )
         end if

      end if

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfReshape_array_arg2_empty
!_______________________________________________________________________
!
   subroutine msReshape( A, m, n )

      type(mfArray) :: A
      integer, intent(in) :: m, n
      !------ API end ------

#ifdef _DEVLP
      ! 'in place' version of 'mfReshape_array'
      !
      ! Note that we cannot write:
      !      A%double = reshape( A%double, [m,n] )
      ! because 'reshape' acts like a function and doesn't modify its arg.
      ! In the case where A%double would be an allocatable array instead a
      ! pointer, it should be possible.

      real(kind=MF_DOUBLE), allocatable :: tmp(:,:)
      complex(kind=MF_DOUBLE), allocatable :: ztmp(:,:)
      integer, allocatable :: jptr(:)
      integer :: nrow, ncol

   !------ end of declarations -- execution starts hereafter  ------

      if( A%parameter ) then
         call PrintMessage( "msReshape", "E",                           &
                            "'A' cannot be a protected array (pseudo-parameter)." )
         return
      end if

      if( A%status_restricted ) then
         call PrintMessage( "msReshape", "E",                           &
                            "mfArray 'A' is restricted!",              &
                            "'A' cannot be reshaped." )
         return
      end if

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "msReshape", "E",                           &
                            "mfArray not allocated!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfReshape", "E",                           &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( m < 0 .or. m*n /= A%shape(1)*A%shape(2) ) then
         call PrintMessage( "msReshape", "E",                           &
                            "bad arg. value for m or n!" )
         go to 99
      end if

      if( m /= A%shape(1) ) then
         if( A%data_type == MF_DT_DBLE ) then
            allocate( tmp(A%shape(1),A%shape(2)) )

            tmp = A%double
            A%shape = [ m, n ]
            deallocate( A%double )

            allocate( A%double(m,n) )

            A%double = reshape( tmp, [ m, n ] )
            deallocate( tmp )

         else if( A%data_type == MF_DT_CMPLX ) then
            allocate( ztmp(A%shape(1),A%shape(2)) )

            ztmp = A%cmplx
            A%shape = [ m, n ]
            deallocate( A%cmplx )

            allocate( A%cmplx(m,n) )

            A%cmplx = reshape( ztmp, [ m, n ] )
            deallocate( ztmp )

         else if( mfIsSparse(A) ) then
            ! both for real and complex cases
            nrow = A%shape(1)
            ncol = A%shape(2)
            allocate( jptr(n+1) )
            call reshape_sp_inplace( nrow, ncol, A%i, A%j,              &
                                     m, n, jptr )
            deallocate( A%j )
            allocate( A%j(n+1) )
            A%j(:) = jptr(:)
            A%shape = [ m, n ]
         else
            call PrintMessage( "msReshape", "E",                        &
                               "unknown data type!" )
            go to 99
         end if

         A%prop%tril = UNKNOWN
         A%prop%triu = UNKNOWN
         if( m == n ) then
            A%prop%symm = UNKNOWN
         else
            A%prop%symm = FALSE
         end if
         A%prop%posd = UNKNOWN

      end if

 99   continue

      call msFreeArgs( A )

#endif
   end subroutine msReshape
!_______________________________________________________________________
!
   function mfZeros_one_arg( n ) result( out )

      integer, intent(in) :: n
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP

   !------ end of declarations -- execution starts hereafter  ------

      if( n < 1 ) then
         call PrintMessage( "mfZeros", "E",                             &
                            "bad arg. value!" )
         return
      end if

      call mf_save_and_disable_fpe( )

      out%data_type = MF_DT_DBLE
      out%shape = [ n, n ]
      allocate( out%double(n,n) )

      out%double(:,:) = 0.0d0

      out%prop%tril = TRUE
      out%prop%triu = TRUE
      out%prop%symm = TRUE
      out%prop%posd = FALSE

      out%status_temporary = .true.

      call mf_restore_fpe( )

#endif
   end function mfZeros_one_arg
!_______________________________________________________________________
!
   function mfZeros_two_arg( n1, n2 ) result( out )

      integer, intent(in) :: n1, n2
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP

   !------ end of declarations -- execution starts hereafter  ------

      if( n1 < 0 .or. n2 < 0 ) then
         ! one arg. can be null if we want to build the matrix by
         ! concatenating elements
         call PrintMessage( "mfZeros", "E",                             &
                            "bad arg. value for at least one arg.!" )
         return
      end if

      call mf_save_and_disable_fpe( )

      out%data_type = MF_DT_DBLE
      out%shape = [ n1, n2 ]
      allocate( out%double(n1,n2) )

      out%double(:,:) = 0.0d0

      out%prop%tril = TRUE
      out%prop%triu = TRUE
      if( n1 == n2 ) then
         out%prop%symm = TRUE
      else
         out%prop%symm = FALSE
      end if
      out%prop%posd = FALSE

      out%status_temporary = .true.

      call mf_restore_fpe( )

#endif
   end function mfZeros_two_arg
!_______________________________________________________________________
!
   function mfDiag( A, d ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      integer, intent(in), optional :: d
      !------ API end ------

#ifdef _DEVLP
      ! if A is a matrix-like mfArray : extracts
      ! the main (or the d-) diagonal
      ! (the ouput is dense, whatever the storage type of A)
      !
      ! if A is a vector-like mfArray (dense only): builds
      ! a dense square matrix which have 'A' as main (or as d-) diagonal
      ! (for building a sparse matrix, use 'mfSpDiags')

      integer :: i, n, n2
      integer :: nrow, ncol, dd, ld

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfDiag", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( present(d) ) then
         dd = d
      else
         dd = 0
      end if

      if( A%shape(1) == 1 .or. A%shape(2) == 1 ) then ! A is a vector

         if( mfIsSparse(A) ) then
            call PrintMessage( "mfDiag", "E",                           &
                               "vector arg. cannot be sparse!",        &
                               "(for building a sparse matrix, use 'mfSpDiags')" )
            go to 99
         end if

         if( A%data_type == MF_DT_BOOL ) then
            call PrintMessage( "mfDiag", "E",                           &
                               "vector arg. cannot be boolean!" )
            go to 99
         end if

         out%data_type = A%data_type

         ! construct a square matrix from a vector
         n = max(a%shape(1),a%shape(2))
         n2 = n + abs(dd)
         out%shape = [ n2, n2 ]
         if( A%data_type == MF_DT_DBLE ) then
            allocate( out%double(n2,n2) )

            out%double(:,:) = 0.0d0
            if( a%shape(1) > a%shape(2) ) then
               if( dd >= 0 ) then
                  do i = 1, n
                     out%double(i,i+dd) = a%double(i,1)
                  end do
               else
                  do i = 1, n
                     out%double(i-dd,i) = a%double(i,1)
                  end do
               end if
            else
               if( dd >= 0 ) then
                  do i = 1, n
                     out%double(i,i+dd) = a%double(1,i)
                  end do
               else
                  do i = 1, n
                     out%double(i-dd,i) = a%double(1,i)
                  end do
               end if
            end if
            if( dd == 0 ) then
               out%prop%symm = TRUE
               if( minval( [ (abs(out%double(i,i)),i=1,n) ] ) > 0.0d0 ) then
                  out%prop%posd = TRUE
               end if
            end if
         else if( A%data_type == MF_DT_CMPLX ) then
            allocate( out%cmplx(n2,n2) )

            out%cmplx(:,:) = (0.0d0,0.0d0)
            if( a%shape(1) > a%shape(2) ) then
               if( dd >= 0 ) then
                  do i = 1, n
                     out%cmplx(i,i+dd) = a%cmplx(i,1)
                  end do
               else
                  do i = 1, n
                     out%cmplx(i-dd,i) = a%cmplx(i,1)
                  end do
               end if
            else
               if( dd >= 0 ) then
                  do i = 1, n
                     out%cmplx(i,i+dd) = a%cmplx(1,i)
                  end do
               else
                  do i = 1, n
                     out%cmplx(i-dd,i) = a%cmplx(1,i)
                  end do
               end if
            end if
            if( dd == 0 ) then
               ! Warning: for a complex matrix, the 'symm' tag means hermitian !
               if( sum([ (abs(aimag(out%cmplx(i,i))),i=1,n) ]) == 0.0d0 ) then
                  out%prop%symm = TRUE
                  if( minval( [ (abs(real(out%cmplx(i,i))),i=1,n) ] ) > 0.0d0 ) then
                     out%prop%posd = TRUE
                  end if
               end if
            end if
         end if

         if( dd == 0 ) then
            out%prop%tril = TRUE
            out%prop%triu = TRUE
         else if( dd > 0 ) then
            out%prop%tril = FALSE
            out%prop%triu = TRUE
         else ! dd < 0
            out%prop%tril = TRUE
            out%prop%triu = FALSE
         end if

      else ! A is a matrix

         nrow = A%shape(1)
         ncol = A%shape(2)

         if( dd < -(nrow-1) .or. ncol-1 < dd ) then
            call PrintMessage( "mfDiag", "W",                           &
                               "'d' out of range!" )
            go to 99
         end if

         ! computes 'ld' : length of diag. number 'dd'
         ld = diag_length( nrow, ncol, dd )

         ! extract a vector from a matrix
         out%prop%symm = FALSE

         n = ld
         out%shape = [ 1, n ]
         if( a%data_type == MF_DT_DBLE .or. a%data_type == MF_DT_BOOL ) then
            out%data_type = a%data_type
            allocate( out%double(1,n) )

            if( dd >= 0 ) then
               do i = 1, n
                  out%double(1,i) = a%double(i,i+dd)
               end do
            else
               do i = 1, n
                  out%double(1,i) = a%double(i-dd,i)
               end do
            end if
         else if( a%data_type == MF_DT_CMPLX ) then
            out%data_type = a%data_type
            allocate( out%cmplx(1,n) )

            if( dd >= 0 ) then
               do i = 1, n
                  out%cmplx(1,i) = a%cmplx(i,i+dd)
               end do
            else
               do i = 1, n
                  out%cmplx(1,i) = a%cmplx(i-dd,i)
               end do
            end if
         else if( a%data_type == MF_DT_SP_DBLE ) then
            out%data_type = MF_DT_DBLE
            allocate( out%double(1,n) )

            call getdia( nrow, ncol, a%a, a%i, a%j,                     &
                         out%double(1,:), dd )
         else if( a%data_type == MF_DT_SP_CMPLX ) then
            out%data_type = MF_DT_CMPLX
            allocate( out%cmplx(1,n) )

            call getdia_cmplx( nrow, ncol, a%z, a%i, a%j,               &
                               out%cmplx(1,:), dd )
         end if
      end if

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfDiag
!_______________________________________________________________________
!
   subroutine msDiag( A, v, d )

      type(mfArray) :: A
      type(mfArray) :: v
      integer, intent(in), optional :: d
      !------ API end ------

#ifdef _DEVLP
      ! A must be a matrix-like mfArray
      ! v must be a vector-like mfArray
      !
      ! v replaces the main (or the d-) diagonal of A
      ! v is not modified
      !
      ! A may be dense or sparse
      !
      ! in the sparse case:
      ! (i) a new vector vv is used : vv = v - v1,
      !     where v1 is the original d-diagonal of A
      ! (ii) a sparse matrix is build using vv (via mfSpDiags), then
      ! (iii) this latter matrix is added to A

      integer :: dim, n, i, dd
      integer :: nrow, ncol, ld

   !------ end of declarations -- execution starts hereafter  ------

      if( A%parameter ) then
         call PrintMessage( "msDiag", "E",                              &
                            "'A' cannot be a protected array (pseudo-parameter)." )
         return
      end if

      call msInitArgs( A, v )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsEmpty(v) ) then
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "msDiag", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( mfIsPerm(v) ) then
         call PrintMessage( "msDiag", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( present(d) ) then
         dd = d
      else
         dd = 0
      end if

      if( A%shape(1) == 1 .or. A%shape(2) == 1 ) then
         call PrintMessage( "msDiag", "E",                              &
                            "first arg. must be a matrix!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "msDiag", "E",                              &
                            "first arg. (matrix) cannot be boolean!" )
         go to 99
      end if

      if( v%shape(1) == 1 ) then
         dim = 2
      else if( v%shape(2) == 1 ) then
         dim = 1
      else
         call PrintMessage( "msDiag", "E",                              &
                            "second arg. must be a vector!" )
         go to 99
      end if

      if( v%data_type == MF_DT_BOOL ) then
         call PrintMessage( "msDiag", "E",                              &
                            "second arg. (vector) cannot be boolean!" )
         go to 99
      end if

      ! dimensions must match
      n = v%shape(dim)
      nrow = A%shape(1)
      ncol = A%shape(2)

      if( dd < -(nrow-1) .or. ncol-1 < dd ) then
         call PrintMessage( "msDiag", "W",                              &
                            "'d' out of range!" )
         go to 99
      end if

      ! computes 'ld' : length of diag. number 'dd'
      ld = diag_length( nrow, ncol, dd )

      if( ld /= n ) then
         call PrintMessage( "msDiag", "E",                              &
                            "dimensions of 'A' and 'v' must match!" )
         go to 99
      end if

      if( A%data_type == MF_DT_DBLE ) then
         if( v%data_type == MF_DT_DBLE ) then
            if( dim == 1 ) then
               if( dd >= 0 ) then
                  do i = 1, n
                     A%double(i,i+dd) = v%double(i,1)
                  end do
               else
                  do i = 1, n
                     A%double(i-dd,i) = v%double(i,1)
                  end do
               end if
            else ! dim == 2
               if( dd >= 0 ) then
                  do i = 1, n
                     A%double(i,i+dd) = v%double(1,i)
                  end do
               else
                  do i = 1, n
                     A%double(i-dd,i) = v%double(1,i)
                  end do
               end if
            end if
         else if( v%data_type == MF_DT_CMPLX ) then
            A%data_type = MF_DT_CMPLX
            allocate( A%cmplx(A%shape(1),A%shape(2)) )

            A%cmplx(:,:) = A%double(:,:)
            deallocate( A%double )

            if( dim == 1 ) then
               if( dd >= 0 ) then
                  do i = 1, n
                     A%cmplx(i,i+dd) = v%cmplx(i,1)
                  end do
               else
                  do i = 1, n
                     A%cmplx(i-dd,i) = v%cmplx(i,1)
                  end do
               end if
            else ! dim == 2
               if( dd >= 0 ) then
                  do i = 1, n
                     A%cmplx(i,i+dd) = v%cmplx(1,i)
                  end do
               else
                  do i = 1, n
                     A%cmplx(i-dd,i) = v%cmplx(1,i)
                  end do
               end if
            end if
         end if
      else if( A%data_type == MF_DT_CMPLX ) then
         if( v%data_type == MF_DT_DBLE ) then
            if( dim == 1 ) then
               if( dd >= 0 ) then
                  do i = 1, n
                     A%cmplx(i,i) = v%double(i,1)
                  end do
               else
                  do i = 1, n
                     A%cmplx(i,i) = v%double(i,1)
                  end do
               end if
            else ! dim == 2
               if( dd >= 0 ) then
                  do i = 1, n
                     A%cmplx(i,i) = v%double(1,i)
                  end do
               else
                  do i = 1, n
                     A%cmplx(i,i) = v%double(1,i)
                  end do
               end if
            end if
         else if( v%data_type == MF_DT_CMPLX ) then
            if( dim == 1 ) then
               if( dd >= 0 ) then
                  do i = 1, n
                     A%cmplx(i,i) = v%cmplx(i,1)
                  end do
               else
                  do i = 1, n
                     A%cmplx(i,i) = v%cmplx(i,1)
                  end do
               end if
            else ! dim == 2
               if( dd >= 0 ) then
                  do i = 1, n
                     A%cmplx(i,i) = v%cmplx(1,i)
                  end do
               else
                  do i = 1, n
                     A%cmplx(i,i) = v%cmplx(1,i)
                  end do
               end if
            end if
         end if
      else if( mfIsSparse(A) ) then
         call msAssign( A, A + mfSpDiags(nrow,ncol,v-mfDiag(A,dd),dd) )
      end if

 99   continue

      ! as the matrix A has been modified, some of its properties
      ! may be lost, according to the position of the modified diagonal
      if( dd == 0 ) then
         A%prop%posd = UNKNOWN
      else ! dd /= 0
         A%prop%symm = UNKNOWN
         if( dd > 0 ) then
            A%prop%tril = UNKNOWN
         else ! dd < 0
            A%prop%triu = UNKNOWN
         end if
      end if

      call msFreeArgs( A, v )

      call msAutoRelease( A, v )

#endif
   end subroutine msDiag
!_______________________________________________________________________
!
   function mfBlkDiag_mfArray_mfArray( A, B ) result( out )

      type(mfArray) :: A, B
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      type(mfArray) :: tmp
      integer :: m, n

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A, B )

      if( mfIsPerm(A) ) then
         call PrintMessage( "mfBlkDiag", "E",                           &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( mfIsPerm(B) ) then
         call PrintMessage( "mfBlkDiag", "E",                           &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( mfIsEmpty(A) ) then
         if( B%level_protected == 1 ) then
            call msAssign( out, B )
         else
            out = B
         end if
         go to 89
      end if

      if( mfIsEmpty(B) ) then
         if( A%level_protected == 1 ) then
            call msAssign( out, A )
         else
            out = A
         end if
         go to 89
      end if

      m = A%shape(1)
      if( A%shape(2) /= m ) then
         call PrintMessage( "mfBlkDiag", "E",                           &
                            "matrix A must be square!" )
         go to 99
      end if
      n = B%shape(2)
      if( B%shape(1) /= n ) then
         call PrintMessage( "mfBlkDiag", "E",                           &
                            "matrix B must be square!" )
         go to 99
      end if

      ! any type can be used; check is done in .vc. and .hc. operator
      if( mfIsSparse(A) .and. mfIsSparse(B) ) then
         ! result will be sparse also
         ! first adding n empty columns to A -> out
         if( A%level_protected == 1 ) then
            call msAssign( out, A )
         else
            out = A
         end if
         call msHorizConcat( out, mfSpAlloc(m,n) )
         call msAssign( tmp, mfSpAlloc(n,m) )
         call msHorizConcat( tmp, B )
         call msAssign( out, out .vc. tmp )
         call msSilentRelease( tmp )
      else if( .not. mfIsSparse(A) .and. .not. mfIsSparse(B) ) then
         ! result will be dense also
         call msAssign( out, A .hc. mfZeros(m,n) )
         call msAssign( tmp, mfZeros(n,m) .hc. B )
         call msAssign( out, out .vc. tmp )
         call msSilentRelease( tmp )
      else
         ! mixed storage not yet allowed
         call PrintMessage( "mfBlkDiag", "E",                           &
                            "mixed storage not allowed!" )
         go to 99
      end if

      if( A%prop%triu == TRUE .and. B%prop%triu == TRUE ) then
         out%prop%triu = TRUE
      else
         out%prop%triu = UNKNOWN
      end if

      if( A%prop%tril == TRUE .and. B%prop%tril == TRUE ) then
         out%prop%tril = TRUE
      else
         out%prop%tril = UNKNOWN
      end if

      if( A%prop%symm == TRUE .and. B%prop%symm == TRUE ) then
         out%prop%symm = TRUE
      else
         out%prop%symm = UNKNOWN
      end if

      if( A%prop%posd == TRUE .and. B%prop%posd == TRUE ) then
         out%prop%posd = TRUE
      else
         out%prop%posd = UNKNOWN
      end if

 89   out%status_temporary = .true.

 99   continue

      call msFreeArgs( A, B )

      call msAutoRelease( A, B )

#endif
   end function mfBlkDiag_mfArray_mfArray
!_______________________________________________________________________
!
   function mfBlkDiag_mfArray_int( A, n ) result( out )

      type(mfArray) :: A
      integer       :: n
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: k

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfBlkDiag", "W",                           &
                            "A is empty!" )
         go to 99
      end if

      if( mfIsPerm(A) ) then
         call PrintMessage( "mfBlkDiag", "E",                           &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( n == 0 ) then
         call PrintMessage( "mfBlkDiag", "W",                           &
                            "n is null: result will be empty!" )
         go to 99
      end if

      if( n < 0 ) then
         call PrintMessage( "mfBlkDiag", "E",                           &
                            "n cannot be negative!" )
         go to 99
      end if

      if( A%level_protected == 1 ) then
         call msAssign( out, A )
      else
         out = A
      end if
      do k = 2, n
         call msAssign( out, mfBlkDiag_mfArray_mfArray( out, A ) )
      end do

      ! matrix properties are automatically set via the use of
      ! 'mfBlkDiag_mfArray_mfArray'

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfBlkDiag_mfArray_int
!_______________________________________________________________________
!
   function mfVander( v, n ) result( out )

      type(mfArray)                 :: v
      integer, intent(in), optional :: n
      type(mfArray)                 :: out
      !------ API end ------

#ifdef _DEVLP
      ! Vandermonde matrix

      integer :: ncol, idim, m, j

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( v )

      if( mfIsEmpty(v) ) then
         go to 99
      end if

      ! 'v' cannot be sparse
      if( mfIsSparse(v) ) then
         call PrintMessage( "mfVander", "E",                            &
                            "first arg. cannot be sparse!" )
         go to 99
      end if

      ! 'v' must be numeric
      if( .not. mfIsNumeric(v) ) then
         call PrintMessage( "mfVander", "E",                            &
                            "first arg. must be numeric!" )
         go to 99
      end if

      ! 'v' must be a vector
      if( v%shape(1) == 1 ) then
         idim = 2
      else if( v%shape(2) == 1 ) then
         idim = 1
      else
         call PrintMessage( "mfVander", "E",                            &
                            "first arg. must be a vector mfArray!" )
         go to 99
      end if
      m = size(v)

      if( present(n) ) then
         if( n <= 0 ) then
            call PrintMessage( "mfVander", "E",                         &
                               "nb of columns must be >= 1!" )
            go to 99
         end if
         ncol = n
      else
         ncol = m
      end if

      out%data_type = v%data_type
      out%shape = [ m, ncol ]
      if( out%data_type == MF_DT_DBLE ) then
         allocate( out%double(m,ncol) )

         if( idim == 1 ) then
            do j = 1, ncol
               out%double(:,j) = v%double(:,1)**(ncol-j)
            end do
         else
            do j = 1, ncol
               out%double(:,j) = v%double(1,:)**(ncol-j)
            end do
         end if
      else if( out%data_type == MF_DT_CMPLX ) then
         allocate( out%cmplx(m,ncol) )

         if( idim == 1 ) then
            do j = 1, ncol
               out%cmplx(:,j) = v%cmplx(:,1)**(ncol-j)
            end do
         else
            do j = 1, ncol
               out%cmplx(:,j) = v%cmplx(1,:)**(ncol-j)
            end do
         end if
      end if

      out%prop%tril = FALSE
      out%prop%triu = FALSE
      if( mf_phys_units ) then
         out%units(:) = v%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( v )

      call msAutoRelease( v )

#endif
   end function mfVander
!_______________________________________________________________________
!
   function mfFind( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j, k, nnz, ncol

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsPerm(A) ) then
         call PrintMessage( "mfFind", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      ncol = A%shape(2)
      k = 0

      if( mfIsSparse(A) ) then

         nnz = mfNnz( A )
         if( nnz == 0 ) go to 99
         out%data_type = MF_DT_DBLE
         out%shape = [ 1, nnz ]
         allocate( out%double(1,nnz) )

         if( mfIsComplex(A) ) then
            do j = 1, ncol
               do i = A%j(j), A%j(j+1)-1
                  if( A%z(i) /= (0.0d0,0.0d0) ) then
                     k = k + 1
                     out%double(1,k) = A%i(i) + (j-1)*A%shape(1)
                  end if
               end do
            end do
         else
            do j = 1, ncol
               do i = A%j(j), A%j(j+1)-1
                  if( A%a(i) /= 0.0d0 ) then
                     k = k + 1
                     ! long column indexes
                     out%double(1,k) = A%i(i) + (j-1)*A%shape(1)
                  end if
               end do
            end do
         end if

      else

         if( mfIsComplex(A) ) then
            nnz = count( A%cmplx /= (0.0d0,0.0d0) )
            if( nnz == 0 ) go to 99
            out%data_type = MF_DT_DBLE
            out%shape = [ 1, nnz ]
            allocate( out%double(1,nnz) )

            do j = 1, ncol
               do i = 1, A%shape(1)
                  if( A%cmplx(i,j) /= (0.0d0,0.0d0) ) then
                     k = k + 1
                     ! long column indexes
                     out%double(1,k) = i + (j-1)*A%shape(1)
                  end if
               end do
            end do
         else
            nnz = count( A%double /= 0.0d0 )
            if( nnz == 0 ) go to 99
            out%data_type = MF_DT_DBLE
            out%shape = [ 1, nnz ]
            allocate( out%double(1,nnz) )

            do j = 1, ncol
               do i = 1, A%shape(1)
                  if( A%double(i,j) /= 0.0d0 ) then
                     k = k + 1
                     ! long column indexes
                     out%double(1,k) = i + (j-1)*A%shape(1)
                  end if
               end do
            end do
         end if

      end if

      out%prop%symm = FALSE
      out%prop%posd = FALSE

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfFind
!_______________________________________________________________________
!
   subroutine msFind( out, A )

      type(mfArray) :: A
      type(mf_Out) :: out
      !------ API end ------

#ifdef _DEVLP
      type(mfArray), pointer :: i, j, v
      integer :: ii, jj, kk, nnz, ncol

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsPerm(A) ) then
         call PrintMessage( "msFind", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      ! we must have two or three output arguments
      if( out%n /= 2 .and. out%n /= 3 ) then
         call PrintMessage( "msFind", "E",                              &
                            "two or three output args required!",       &
                            "syntax is : call msFind ( mfOut(i,j), A )", &
                            "       or : call msFind ( mfOut(i,j,v), A )" )
         go to 99
      end if

      ! internal check for all mfOut args
      if( .not. args_mfout_ok( out, A ) ) then
         call PrintMessage( "msFind", "E",                              &
                            "output arguments cannot be tempo, or cannot share", &
                            "same memory as another input argument." )
         go to 99
      end if

      i => out%ptr1
      j => out%ptr2
      call msSilentRelease( i, j )
      if( out%n == 3 ) then
         v => out%ptr3
         call msSilentRelease( v )
      end if

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         nnz = mfNnz( A )
      else
         if( mfIsComplex(A) ) then
            nnz = count( A%cmplx /= (0.0d0,0.0d0) )
         else
            nnz = count( A%double /= 0.0d0 )
         end if
      end if
      if( nnz == 0 ) go to 99

      i%data_type = MF_DT_DBLE
      i%shape = [ 1, nnz ]
      i%prop%symm = FALSE
      i%prop%posd = FALSE
      allocate( i%double(1,nnz) )

      j%data_type = MF_DT_DBLE
      j%shape = [ 1, nnz ]
      j%prop%symm = FALSE
      j%prop%posd = FALSE
      allocate( j%double(1,nnz) )

      if( out%n == 3 ) then
         v%shape = [ 1, nnz ]
         v%prop%symm = FALSE
         v%prop%posd = FALSE
         if( mfIsComplex(A) ) then
            v%data_type = MF_DT_CMPLX
            allocate( v%cmplx(1,nnz) )

         else
            v%data_type = MF_DT_DBLE
            allocate( v%double(1,nnz) )

         end if
      end if

      kk = 0
      ncol = A%shape(2)

      if( mfIsSparse(A) ) then

         if( mfIsComplex(A) ) then
#if defined _INTEL_IFC & defined _OPTIM
! long-time waiting bug for INTEL-ifort, version number >= 11.1 when
! using optimization!
            if( out%n == 3 ) then
               do jj = 1, ncol
                  do ii = A%j(jj), A%j(jj+1)-1
                     if( A%z(ii) /= (0.0d0,0.0d0) ) then
                        kk = kk + 1
                        i%double(1,kk) = A%i(ii)
                        j%double(1,kk) = jj
                        v%cmplx(1,kk) = A%z(ii)
                     end if
                  end do
               end do
            else
               do jj = 1, ncol
                  do ii = A%j(jj), A%j(jj+1)-1
                     if( A%z(ii) /= (0.0d0,0.0d0) ) then
                        kk = kk + 1
                        i%double(1,kk) = A%i(ii)
                        j%double(1,kk) = jj
                     end if
                  end do
               end do
            end if
#else
            do jj = 1, ncol
               do ii = A%j(jj), A%j(jj+1)-1
                  if( A%z(ii) /= (0.0d0,0.0d0) ) then
                     kk = kk + 1
                     i%double(1,kk) = A%i(ii)
                     j%double(1,kk) = jj
                     if( out%n == 3 ) then
                        v%cmplx(1,kk) = A%z(ii)
                     end if
                  end if
               end do
            end do
#endif
         else ! real matrix A
#if defined _INTEL_IFC & defined _OPTIM
! long-time waiting bug for INTEL-ifort, version number >= 11.1 when
! using optimization!
            if( out%n == 3 ) then
               do jj = 1, ncol
                  do ii = A%j(jj), A%j(jj+1)-1
                     if( A%a(ii) /= 0.0d0 ) then
                        kk = kk + 1
                        i%double(1,kk) = A%i(ii)
                        j%double(1,kk) = jj
                        v%double(1,kk) = A%a(ii)
                     end if
                  end do
               end do
            else
               do jj = 1, ncol
                  do ii = A%j(jj), A%j(jj+1)-1
                     if( A%a(ii) /= 0.0d0 ) then
                        kk = kk + 1
                        i%double(1,kk) = A%i(ii)
                        j%double(1,kk) = jj
                     end if
                  end do
               end do
            end if
#else
            do jj = 1, ncol
               do ii = A%j(jj), A%j(jj+1)-1
                  if( A%a(ii) /= 0.0d0 ) then
                     kk = kk + 1
                     i%double(1,kk) = A%i(ii)
                     j%double(1,kk) = jj
                     if( out%n == 3 ) then
                        v%double(1,kk) = A%a(ii)
                     end if
                  end if
               end do
            end do
#endif
         end if

         ! Some zero values may be left in the sparse structure!
         ! (however, this case should be rare)
         if( kk < nnz ) then
            i%shape(2) = kk
            j%shape(2) = kk
            v%shape(2) = kk
         end if

      else ! dense matrix A

         if( mfIsComplex(A) ) then
            do jj = 1, ncol
               do ii = 1, a%shape(1)
                  if( a%cmplx(ii,jj) /= (0.0d0,0.0d0) ) then
                     kk = kk + 1
                     i%double(1,kk) = ii
                     j%double(1,kk) = jj
                     if( out%n == 3 ) then
                        v%cmplx(1,kk) = a%cmplx(ii,jj)
                     end if
                  end if
               end do
            end do
         else ! real matrix A
            do jj = 1, ncol
               do ii = 1, a%shape(1)
                  if( a%double(ii,jj) /= 0.0d0 ) then
                     kk = kk + 1
                     i%double(1,kk) = ii
                     j%double(1,kk) = jj
                     if( out%n == 3 ) then
                        v%double(1,kk) = a%double(ii,jj)
                     end if
                  end if
               end do
            end do
         end if

      end if

      if( mf_phys_units ) then
         v%units(:) = a%units(:)
      end if

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end subroutine msFind
!_______________________________________________________________________
!
   function mfNonZeros( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j, k, nnz, ncol

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsPerm(A) ) then
         call PrintMessage( "mfNonZeros", "E",                          &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      ncol = A%shape(2)
      k = 0

      if( mfIsSparse(A) ) then

         nnz = mfNnz( A )
         out%shape = [ 1, nnz ]
         if( mfIsComplex(A) ) then
            out%data_type = MF_DT_CMPLX
            allocate( out%cmplx(1,nnz) )

            do j = 1, ncol
               do i = A%j(j), A%j(j+1)-1
                  if( A%z(i) /= (0.0d0,0.0d0) ) then
                     k = k + 1
                     out%cmplx(1,k) = A%z(i)
                  end if
               end do
            end do
         else
            out%data_type = MF_DT_DBLE
            allocate( out%double(1,nnz) )

            do j = 1, ncol
               do i = A%j(j), A%j(j+1)-1
                  if( A%a(i) /= 0.0d0 ) then
                     k = k + 1
                     out%double(1,k) = A%a(i)
                  end if
               end do
            end do
         end if

      else

         out%data_type = A%data_type
         if( mfIsComplex(A) ) then
            nnz = count( A%cmplx /= (0.0d0,0.0d0) )
            out%shape = [ 1, nnz ]
            allocate( out%cmplx(1,nnz) )

            do j = 1, ncol
               do i = 1, A%shape(1)
                  if( A%cmplx(i,j) /= 0.0d0 ) then
                     k = k + 1
                     out%cmplx(1,k) = A%cmplx(i,j)
                  end if
               end do
            end do
         else
            nnz = count( A%double /= 0.0d0 )
            out%shape = [ 1, nnz ]
            allocate( out%double(1,nnz) )

            do j = 1, ncol
               do i = 1, A%shape(1)
                  if( A%double(i,j) /= 0.0d0 ) then
                     k = k + 1
                     out%double(1,k) = A%double(i,j)
                  end if
               end do
            end do
         end if

      end if

      out%prop%symm = FALSE
      out%prop%posd = FALSE

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfNonZeros
!_______________________________________________________________________
!
   function mfOnes_one_arg( n ) result( out )

      integer, intent(in) :: n
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP

   !------ end of declarations -- execution starts hereafter  ------

      if( n < 1 ) then
         call PrintMessage( "mfOnes", "W",                              &
                            "bad dimension arg." )
         return
      end if

      call mf_save_and_disable_fpe( )

      out%data_type = MF_DT_DBLE
      out%shape = [ n, n ]
      allocate( out%double(n,n) )

      out%double(:,:) = 1.0d0

      out%prop%tril = FALSE
      out%prop%triu = FALSE
      out%prop%symm = TRUE
      out%prop%posd = FALSE

      out%status_temporary = .true.

      call mf_restore_fpe( )

#endif
   end function mfOnes_one_arg
!_______________________________________________________________________
!
   function mfOnes_two_arg( n1, n2 ) result( out )

      integer, intent(in) :: n1, n2
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( n1 < 1 .or. n2 < 1 ) then
         call PrintMessage( "mfOnes", "W",                              &
                            "bad dimension arg." )
         return
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ n1, n2 ]
      allocate( out%double(n1,n2) )

      out%double(:,:) = 1.0d0

      out%prop%tril = FALSE
      out%prop%triu = FALSE
      if( n1 == n2 ) then
         out%prop%symm = TRUE
      else
         out%prop%symm = FALSE
      end if
      out%prop%posd = FALSE

      out%status_temporary = .true.

#endif
   end function mfOnes_two_arg
!_______________________________________________________________________
!
   function mfTril( A, k ) result( out )

      type(mfArray) :: A
      integer, optional :: k
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j, m, n, d
      integer :: nnz

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfTril", "E",                              &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( mfIsPerm(A) ) then
         call PrintMessage( "mfTril", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      m = A%shape(1)
      n = A%shape(2)

      if( present(k) ) then
         d = k
      else
         d = 0
      end if

      if( d >= n-1 ) then
         if( A%level_protected == 1 ) then
            call msAssign( out, A )
         else
            out = A
         end if
         out%status_temporary = .true.
         go to 99
      else if( d <= -m ) then
         if( mfIsSparse(A) ) then
            if( A%data_type == MF_DT_SP_DBLE ) then
               call msAssign( out, mfSpAlloc(m,n) )
            else
               call msAssign( out, mfSpAlloc(m,n,kind="complex") )
            end if
         else
            call msAssign( out, mfZeros(m,n) )
         end if
         out%status_temporary = .true.
         go to 99
      end if

      if( mfIsSparse(A) ) then

         out%data_type = A%data_type
         out%shape = [ m, n ]
         call getl_nnz( n, a%i, a%j, d, nnz )

         if( a%data_type == MF_DT_SP_DBLE ) then
            allocate( out%a(nnz) )

            allocate( out%i(nnz) )

            allocate( out%j(n+1) )

            call getl( n, a%a, a%i, a%j, d, out%a, out%i, out%j )
         else if( a%data_type == MF_DT_SP_CMPLX ) then
            allocate( out%z(nnz) )

            allocate( out%i(nnz) )

            allocate( out%j(n+1) )

            call getl_cmplx( n, a%z, a%i, a%j, d, out%z, out%i, out%j )
         end if

         if( mf_phys_units ) then
            out%units(:) = a%units(:)
         end if

      else

         ! warning : all properties are also copied !
         if( A%level_protected == 1 ) then
            call msAssign( out, A )
         else
            out = A
         end if

         if( a%data_type == MF_DT_DBLE .or. a%data_type == MF_DT_BOOL ) then
            do j = 1, n
               do i = 1, min(j-1-d,m)
                  out%double(i,j) = 0.0d0
               end do
            end do
         else if( a%data_type == MF_DT_CMPLX ) then
            do j = 1, n
               do i = 1, min(j-1-d,m)
                  out%cmplx(i,j) = (0.0d0,0.0d0)
               end do
            end do
         end if

         out%prop%symm = UNKNOWN
         out%prop%posd = UNKNOWN

      end if

      if( d <= 0 ) then
         out%prop%tril = TRUE
      else
         if( A%prop%tril == TRUE ) then
            out%prop%tril = TRUE
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfTril
!_______________________________________________________________________
!
   function mfTriu( a, k ) result( out )

      type(mfArray) :: a
      integer, optional :: k
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j, m, n, d
      integer :: nnz

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( a )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfTriu", "E",                              &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( mfIsPerm(A) ) then
         call PrintMessage( "mfTriu", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      m = A%shape(1)
      n = A%shape(2)

      if( present(k) ) then
         d = k
      else
         d = 0
      end if

      if( d >= n ) then
         if( mfIsSparse(A) ) then
            if( a%data_type == MF_DT_SP_DBLE ) then
               call msAssign( out, mfSpAlloc(m,n) )
            else
               call msAssign( out, mfSpAlloc(m,n,kind="complex") )
            end if
         else
            call msAssign( out, mfZeros(m,n) )
         end if
         out%status_temporary = .true.
         go to 99
      else if( d <= -(m-1) ) then
         if( A%level_protected == 1 ) then
            call msAssign( out, A )
         else
            out = A
         end if
         out%status_temporary = .true.
         go to 99
      end if

      if( mfIsSparse(A) ) then

         out%data_type = a%data_type
         out%shape = [ m, n ]
         call getu_nnz( n, a%i, a%j, d, nnz )

         if( a%data_type == MF_DT_SP_DBLE ) then
            allocate( out%a(nnz) )

            allocate( out%i(nnz) )

            allocate( out%j(n+1) )

            call getu( n, a%a, a%i, a%j, d, out%a, out%i, out%j )
         else if( a%data_type == MF_DT_SP_CMPLX ) then
            allocate( out%z(nnz) )

            allocate( out%i(nnz) )

            allocate( out%j(n+1) )

            call getu_cmplx( n, a%z, a%i, a%j, d, out%z, out%i, out%j )
         end if

         if( mf_phys_units ) then
            out%units(:) = a%units(:)
         end if

      else

         ! warning : all properties are also copied !
         if( A%level_protected == 1 ) then
            call msAssign( out, A )
         else
            out = A
         end if

         if( a%data_type == MF_DT_DBLE .or. a%data_type == MF_DT_BOOL ) then
            do j = 1, n
               do i = max(1,j+1-d), m
                  out%double(i,j) = 0.0d0
               end do
            end do
         else if( a%data_type == MF_DT_CMPLX ) then
            do j = 1, n
               do i = max(1,j+1-d), m
                  out%cmplx(i,j) = (0.0d0,0.0d0)
               end do
            end do
         end if

         out%prop%symm = UNKNOWN
         out%prop%posd = UNKNOWN

      end if

      if( d >= 0 ) then
         out%prop%triu = TRUE
      else
         if( A%prop%triu == TRUE ) then
            out%prop%triu = TRUE
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfTriu
!_______________________________________________________________________
!
   function mfIsNaN( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j, m, n

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfIsNaN", "W",                             &
                            "arg is an empty mfArray!" )
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfIsNaN", "W",                             &
                            "arg cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfIsNaN", "E",                             &
                            "arg cannot be a logical mfArray!" )
         go to 99
      end if

      if( mfIsPerm(A) ) then
         call PrintMessage( "mfIsNaN", "E",                             &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      m = a%shape(1)
      n = a%shape(2)
      call msAssign( out, mfZeros(m,n))
      out%data_type = MF_DT_BOOL                  ! result is a boolean

      if( a%data_type == MF_DT_DBLE ) then
         ! real case
         do j = 1, n
            do i = 1, m
               if( mf_isnan(a%double(i,j)) ) then
                  out%double(i,j) = 1.0d0   ! A(i,j) is NaN
               end if
            end do
         end do
      else if( a%data_type == MF_DT_CMPLX ) then
         ! complex case
         do j = 1, n
            do i = 1, m
               if( mf_isnan(a%cmplx(i,j)) ) then
                  out%double(i,j) = 1.0d0   ! A(i,j) is NaN
               end if
            end do
         end do
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfIsNaN
!_______________________________________________________________________
!
   function mfIsFinite( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j, m, n

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfIsFinite", "W",                          &
                            "arg is an empty mfArray!" )
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfIsFinite", "W",                          &
                            "arg cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfIsFinite", "E",                          &
                            "arg cannot be a logical mfArray!" )
         go to 99
      end if

      if( mfIsPerm(A) ) then
         call PrintMessage( "mfIsFinite", "E",                          &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      m = a%shape(1)
      n = a%shape(2)
      call msAssign(out, mfZeros(m,n))
      out%data_type = MF_DT_BOOL                  ! result is a boolean

      if( a%data_type == MF_DT_DBLE ) then
         ! real case
         do j = 1, n
            do i = 1, m
               if( mf_isfinite(a%double(i,j)) ) then
                  out%double(i,j) = 1.0d0   ! A(i,j) is finite
               end if
            end do
         end do
      else if( a%data_type == MF_DT_CMPLX ) then
         ! complex case
         do j = 1, n
            do i = 1, m
               if( mf_isfinite(a%cmplx(i,j)) ) then
                  out%double(i,j) = 1.0d0   ! A(i,j) is finite
               end if
            end do
         end do
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfIsFinite
!_______________________________________________________________________
!
   function mfIsInf( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! For real numbers: any number 'x' is either 'Finite', or 'Inf',
      ! or 'NaN'
      !
      ! For complex numbers, the previous rule doesn't apply:
      !   (NaN + Inf i) is detected both as 'Inf' and 'NaN'

      integer :: i, j, m, n

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfIsInf", "W",                             &
                            "arg is an empty mfArray!" )
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfIsInf", "W",                             &
                            "arg cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfIsInf", "E",                             &
                            "arg cannot be a logical mfArray!" )
         go to 99
      end if

      if( mfIsPerm(A) ) then
         call PrintMessage( "mfIsInf", "E",                             &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      m = a%shape(1)
      n = a%shape(2)
      call msAssign( out, mfZeros(m,n))
      out%data_type = MF_DT_BOOL                  ! result is a boolean

      if( a%data_type == MF_DT_DBLE ) then
         ! real case
         do j = 1, n
            do i = 1, m
               if( mf_isinf(a%double(i,j)) ) then
                  out%double(i,j) = 1.0d0   ! A(i,j) is inf
               end if
            end do
         end do
      else if( a%data_type == MF_DT_CMPLX ) then
         ! complex case
         do j = 1, n
            do i = 1, m
               if( mf_isinf(a%cmplx(i,j)) ) then
                  out%double(i,j) = 1.0d0   ! A(i,j) is inf
               end if
            end do
         end do
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfIsInf
!_______________________________________________________________________
!
   function mfFlipLR( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! flip Left-Right a matrix

      integer :: i, m, n
      integer, allocatable :: ind(:) ! indexes vector

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfFlipLR", "W",                            &
                            "arg is an empty mfArray!" )
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfFlipLR", "W",                            &
                            "arg cannot (yet) be sparse!" )
         go to 99
      end if

      if( mfIsPerm(A) ) then
         call PrintMessage( "mfFlipLR", "E",                            &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      m = a%shape(1)
      n = a%shape(2)
      allocate( ind(n) )

      ind(:) = [ (i, i = n, 1, -1) ]
      call msAssign( out, mfZeros(m,n))
      out%data_type = A%data_type

      if( a%data_type == MF_DT_DBLE .or. a%data_type == MF_DT_BOOL ) then
         ! real case
         do i = 1, m ! working by row
            out%double(i,:) = a%double(i,ind)
         end do
      else if( a%data_type == MF_DT_CMPLX ) then
         ! complex case
         do i = 1, m
            out%cmplx(i,:) = a%cmplx(i,ind)
         end do
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfFlipLR
!_______________________________________________________________________
!
   function mfFlipUD( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! flip Up-Down a matrix

      integer :: j, m, n
      integer, allocatable :: ind(:) ! indexes vector

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfFlipUD", "W",                            &
                            "arg is an empty mfArray!" )
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfFlipUD", "W",                            &
                            "arg cannot (yet) be sparse!" )
         go to 99
      end if

      if( mfIsPerm(A) ) then
         call PrintMessage( "mfFlipUD", "E",                            &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      m = a%shape(1)
      n = a%shape(2)
      allocate( ind(m) )

      ind(:) = [ (j, j = m, 1, -1) ]
      call msAssign( out, mfZeros(m,n))
      out%data_type = A%data_type

      if( a%data_type == MF_DT_DBLE .or. a%data_type == MF_DT_BOOL ) then
         ! real case
         do j = 1, n ! working by column
            out%double(:,j) = a%double(ind,j)
         end do
      else if( a%data_type == MF_DT_CMPLX ) then
         ! complex case
         do j = 1, n
            out%cmplx(:,j) = a%cmplx(ind,j)
         end do
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfFlipUD
!_______________________________________________________________________
!
   function mfRot90( A, k ) result( out )

      type(mfArray)                 :: A
      integer, intent(in), optional :: k
      type(mfArray)                 :: out
      !------ API end ------

#ifdef _DEVLP
      ! rotate a matrix in the trigonometric direction
      ! (if angle is 90 then last column becomes first row)
      !
      ! the angle is given in term of multiple of 90

      integer :: i, m, n, angle, nnz, itmp
      integer, allocatable :: ir(:), jc(:)
      real(kind=MF_DOUBLE), allocatable :: val(:)
      complex(kind=MF_DOUBLE), allocatable :: zval(:)

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfRot90", "W",                             &
                            "arg is an empty mfArray!" )
         go to 99
      end if

      if( mfIsPerm(A) ) then
         call PrintMessage( "mfRot90", "E",                             &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      ! intrinsic 'mod' is different of 'modulo': the former gives
      ! a result which have the sign of the first operand, whereas the
      ! result of the latter have the sign of the second one.
      if( present(k) ) then
         angle = 90*mod( k, 4 )
      else
         angle = 90
      end if

      m = A%shape(1)
      n = A%shape(2)

      out%data_type = A%data_type
      if( A%data_type == MF_DT_DBLE .or. A%data_type == MF_DT_BOOL ) then
         ! real case
         select case( angle )
            case( 0 )
               out%shape = [ m, n ]
               allocate( out%double(m,n) )

               out%double(:,:) = A%double(:,:)
            case( 90 )
               out%shape = [ n, m ]
               allocate( out%double(n,m) )

               do i = 1, n
                  out%double(i,:) = A%double(:,n-i+1)
               end do
            case( 180 )
               out%shape = [ m, n ]
               allocate( out%double(m,n) )

               do i = 1, m
                  out%double(i,:) = A%double(m-i+1,n:1:-1)
               end do
            case( 270 )
               out%shape = [ n, m ]
               allocate( out%double(n,m) )

               do i = 1, m
                  out%double(:,i) = A%double(m-i+1,:)
               end do
         end select
      else if( A%data_type == MF_DT_CMPLX ) then
         ! complex case
         select case( angle )
            case( 0 )
               out%shape = [ m, n ]
               allocate( out%cmplx(m,n) )

               out%cmplx(:,:) = A%cmplx(:,:)
            case( 90 )
               out%shape = [ n, m ]
               allocate( out%cmplx(n,m) )

               do i = 1, n
                  out%cmplx(i,:) = A%cmplx(:,n-i+1)
               end do
            case( 180 )
               out%shape = [ m, n ]
               allocate( out%cmplx(m,n) )

               do i = 1, m
                  out%cmplx(i,:) = A%cmplx(m-i+1,n:1:-1)
               end do
            case( 270 )
               out%shape = [ n, m ]
               allocate( out%cmplx(n,m) )

               do i = 1, m
                  out%cmplx(:,i) = A%cmplx(m-i+1,:)
               end do
         end select
      else if( A%data_type == MF_DT_SP_DBLE ) then
         nnz = A%j(n+1) - 1
         ! sparse double case
         select case( angle )
            case( 0 )
               out%shape = [ m, n ]
               allocate( out%a(nnz) )

               out%a(:) = A%a(1:nnz)
               allocate( out%i(nnz) )

               out%i(:) = A%i(1:nnz)
               allocate( out%j(n+1) )

               out%j(:) = A%j(1:n+1)
            case( 90 )
               allocate( ir(nnz), jc(nnz), val(nnz) )
               call msSpExport( A, ir, jc, val )
               do i = 1, nnz
                  itmp = ir(i)
                  ir(i) = n - jc(i) + 1
                  jc(i) = itmp
               end do
               call msAssign( out, mfSpImport(ir,jc,val,m=n,n=m) )
            case( 180 )
               allocate( ir(nnz), jc(nnz), val(nnz) )
               call msSpExport( A, ir, jc, val )
               do i = 1, nnz
                  ir(i) = m - ir(i) + 1
                  jc(i) = n - jc(i) + 1
               end do
               call msAssign( out, mfSpImport(ir,jc,val,m=m,n=n) )
            case( 270 )
               allocate( ir(nnz), jc(nnz), val(nnz) )
               call msSpExport( A, ir, jc, val )
               do i = 1, nnz
                  itmp = ir(i)
                  ir(i) = jc(i)
                  jc(i) = m - itmp + 1
               end do
               call msAssign( out, mfSpImport(ir,jc,val,m=n,n=m) )
         end select
      else if( A%data_type == MF_DT_SP_CMPLX ) then
         nnz = A%j(n+1) - 1
         ! sparse complex case
         select case( angle )
            case( 0 )
               out%shape = [ m, n ]
               allocate( out%z(nnz) )

               out%z(:) = A%z(1:nnz)
               allocate( out%i(nnz) )

               out%i(:) = A%i(1:nnz)
               allocate( out%j(n+1) )

               out%j(:) = A%j(1:n+1)
            case( 90 )
               allocate( ir(nnz), jc(nnz), zval(nnz) )
               call msSpExport( A, ir, jc, zval )
               do i = 1, nnz
                  itmp = ir(i)
                  ir(i) = n - jc(i) + 1
                  jc(i) = itmp
               end do
               call msAssign( out, mfSpImport(ir,jc,zval,m=n,n=m) )
            case( 180 )
               allocate( ir(nnz), jc(nnz), zval(nnz) )
               call msSpExport( A, ir, jc, zval )
               do i = 1, nnz
                  ir(i) = m - ir(i) + 1
                  jc(i) = n - jc(i) + 1
               end do
               call msAssign( out, mfSpImport(ir,jc,zval,m=m,n=n) )
            case( 270 )
               allocate( ir(nnz), jc(nnz), zval(nnz) )
               call msSpExport( A, ir, jc, zval )
               do i = 1, nnz
                  itmp = ir(i)
                  ir(i) = jc(i)
                  jc(i) = m - itmp + 1
               end do
               call msAssign( out, mfSpImport(ir,jc,zval,m=n,n=m) )
         end select
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfRot90
!_______________________________________________________________________
!
   function mfCompan( v ) result( out )

      type(mfArray) :: v
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! Companion matrix of a polynomial

      integer :: idim, n, j

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( v )

      if( mfIsEmpty(v) ) then
         go to 99
      end if

      ! 'v' cannot be sparse
      if( mfIsSparse(v) ) then
         call PrintMessage( "mfCompan", "E",                            &
                            "arg. cannot be sparse!" )
         go to 99
      end if

      ! 'v' must be real
      if( v%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfCompan", "E",                            &
                            "arg. must be real!" )
         go to 99
      end if

      ! 'v' must be a vector
      if( v%shape(1) == 1 ) then
         idim = 2
      else if( v%shape(2) == 1 ) then
         idim = 1
      else
         call PrintMessage( "mfCompan", "E",                            &
                            "arg. must be a vector mfArray!" )
         go to 99
      end if
      n = size(v)

      ! 'v' cannot be a scalar
      if( n == 1 ) then
         call PrintMessage( "mfCompan", "E",                            &
                            "arg. cannot be a scalar!" )
         go to 99
      end if

      out%data_type = v%data_type
      out%shape = [ n-1, n-1 ]
      allocate( out%double(n-1,n-1) )

      if( idim == 1 ) then ! column vector
         if( n == 2 ) then
            out%double(1,1) = -v%double(2,1)/v%double(1,1)
         else
            out%double(:,:) = 0.0d0
            do j = 1, n-2
               out%double(j+1,j) = 1.0d0
            end do
            out%double(1,:) = -v%double(2:n,1)/v%double(1,1)
         end if
      else ! row vector
         if( n == 2 ) then
            out%double(1,1) = -v%double(1,2)/v%double(1,1)
         else
            out%double(:,:) = 0.0d0
            do j = 1, n-2
               out%double(j+1,j) = 1.0d0
            end do
            out%double(1,:) = -v%double(1,2:n)/v%double(1,1)
         end if
      end if

      if( mf_phys_units ) then
         out%units(:) = v%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( v )

      call msAutoRelease( v )

#endif
   end function mfCompan
!_______________________________________________________________________
!
   function mfHankel( C, R ) result( out )

      type(mfArray)           :: C
      type(mfArray), optional :: R
      type(mfArray)           :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j, m, n, dim_c, dim_r
      real(kind=MF_DOUBLE), allocatable :: p(:)
      real(kind=MF_DOUBLE) :: last

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( C )

      if( mfIsEmpty(C) ) then
         go to 99
      end if

      ! 'C' cannot be sparse
      if( mfIsSparse(C) ) then
         call PrintMessage( "mfHankel", "E",                            &
                            "first arg. cannot be sparse!" )
         go to 99
      end if

      ! 'C' must be real
      if( .not. mfIsReal(C) ) then
         call PrintMessage( "mfHankel", "E",                            &
                            "first arg. must be real!" )
         go to 99
      end if

      ! 'C' must be a vector
      if( C%shape(1) == 1 ) then
         dim_c = 2
      else if( C%shape(2) == 1 ) then
         dim_c = 1
      else
         call PrintMessage( "mfHankel", "E",                            &
                            "first arg. must be a vector!" )
         go to 99
      end if
      m = C%shape(dim_c)

      if( present(R) ) then

         call msInitArgs( R )

         if( mfIsEmpty(R) ) then
            go to 99
         end if

         ! 'R' cannot be sparse
         if( mfIsSparse(R) ) then
            call PrintMessage( "mfHankel", "E",                         &
                               "second arg. cannot be sparse!" )
            go to 99
         end if

         ! 'R' must be real
         if( .not. mfIsReal(R) ) then
            call PrintMessage( "mfHankel", "E",                         &
                               "second arg. must be real!" )
            go to 99
         end if

         ! 'R' must be a vector
         if( R%shape(1) == 1 ) then
            dim_r = 2
         else if( R%shape(2) == 1 ) then
            dim_r = 1
         else
            call PrintMessage( "mfHankel", "E",                         &
                               "second arg. must be a vector!" )
            go to 99
         end if
         n = R%shape(dim_r)
         allocate( p(m+n-1) )
         if( dim_c == 1 ) then
            p(1:m) = C%double(:,1)
            last = C%double(m,1)
         else
            p(1:m) = C%double(1,:)
            last = C%double(1,m)
         end if
         if( dim_r == 1 ) then
            p(m+1:) = R%double(2:n,1)
         else
            p(m+1:) = R%double(1,2:n)
         end if
         if( R%double(1,1) /= last ) then
            call PrintMessage( "mfHankel", "W",                         &
                               "first elem. of R should be equal to",   &
                               "last elem. of C!" )
         end if

      else
         n = m
         allocate( p(m+n-1) )
         if( dim_c == 1 ) then
            p(1:m) = C%double(:,1)
         else
            p(1:m) = C%double(1,:)
         end if
         p(m+1:) = 0.0d0
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ m, n ]
      allocate( out%double(m,n) )

      do j = 1, n
         do i = 1, m
            out%double(i,j) = p(i+j-1)
         end do
      end do

      if( m == n ) then
         out%prop%symm = TRUE
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( C )
      call msAutoRelease( C )

      if( present(R) ) then
         call msFreeArgs( R )
         call msAutoRelease( R )
      end if

#endif
   end function mfHankel
!_______________________________________________________________________
!
   function mfToeplitz( C, R ) result( out )

      type(mfArray)           :: C
      type(mfArray), optional :: R
      type(mfArray)           :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j, m, n, dim_c, dim_r
      real(kind=MF_DOUBLE), allocatable :: p(:)
      complex(kind=MF_DOUBLE), allocatable :: z(:)
      logical :: real_result

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( C )

      if( mfIsEmpty(C) ) then
         go to 99
      end if

      ! 'C' cannot be sparse
      if( mfIsSparse(C) ) then
         call PrintMessage( "mfToeplitz", "E",                          &
                            "first arg. cannot be sparse!" )
         go to 99
      end if

      if( mfIsPerm(C) ) then
         call PrintMessage( "mfToeplitz", "E",                          &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      ! 'C' must be a vector
      if( C%shape(1) == 1 ) then
         dim_c = 2
      else if( C%shape(2) == 1 ) then
         dim_c = 1
      else
         call PrintMessage( "mfToeplitz", "E",                          &
                            "first arg. must be a vector!" )
         go to 99
      end if
      m = C%shape(dim_c)

      real_result = .false.

      if( present(R) ) then

         ! two args: the first is the row, the second is the column

         call msInitArgs( R )

         if( mfIsEmpty(R) ) then
            go to 99
         end if

         ! 'R' cannot be sparse
         if( mfIsSparse(R) ) then
            call PrintMessage( "mfToeplitz", "E",                       &
                               "second arg. cannot be sparse!" )
            go to 99
         end if

         ! 'R' must be a vector
         if( R%shape(1) == 1 ) then
            dim_r = 2
         else if( R%shape(2) == 1 ) then
            dim_r = 1
         else
            call PrintMessage( "mfToeplitz", "E",                       &
                               "second arg. must be a vector!" )
            go to 99
         end if
         n = R%shape(dim_r)

         if( mfIsComplex(C) ) then
            if( mfIsComplex(R) ) then
               allocate( z(m+n-1) )
               if( dim_c == 1 ) then
                  do i = 2, m
                     z(m+1-i) = C%cmplx(i,1)
                  end do
               else
                  do i = 2, m
                     z(m+1-i) = C%cmplx(1,i)
                  end do
               end if
               if( dim_r == 1 ) then
                  z(m:) = R%cmplx(:,1)
               else
                  z(m:) = R%cmplx(1,:)
               end if
               if( R%cmplx(1,1) /= C%cmplx(1,1) ) then
                  call PrintMessage( "mfToeplitz", "W",                     &
                                     "first elem. of R should be equal to", &
                                     "first elem. of C!" )
               end if
            else ! R is real
               allocate( z(m+n-1) )
               if( dim_c == 1 ) then
                  do i = 2, m
                     z(m+1-i) = C%cmplx(i,1)
                  end do
               else
                  do i = 2, m
                     z(m+1-i) = C%cmplx(1,i)
                  end do
               end if
               if( dim_r == 1 ) then
                  z(m:) = R%double(:,1)
               else
                  z(m:) = R%double(1,:)
               end if
               if( cmplx(R%double(1,1),kind=MF_DOUBLE) /= C%cmplx(1,1) ) then
                  call PrintMessage( "mfToeplitz", "W",                     &
                                     "first elem. of R should be equal to", &
                                     "first elem. of C!" )
               end if
            end if
         else ! C is real
            if( mfIsComplex(R) ) then
               allocate( z(m+n-1) )
               if( dim_c == 1 ) then
                  do i = 2, m
                     z(m+1-i) = C%double(i,1)
                  end do
               else
                  do i = 2, m
                     z(m+1-i) = C%double(1,i)
                  end do
               end if
               if( dim_r == 1 ) then
                  z(m:) = R%cmplx(:,1)
               else
                  z(m:) = R%cmplx(1,:)
               end if
               if( R%cmplx(1,1) /= cmplx(C%double(1,1),kind=MF_DOUBLE) ) then
                  call PrintMessage( "mfToeplitz", "W",                     &
                                     "first elem. of R should be equal to", &
                                     "first elem. of C!" )
               end if
            else ! R is real
               allocate( p(m+n-1) )
               if( dim_c == 1 ) then
                  do i = 2, m
                     p(m+1-i) = C%double(i,1)
                  end do
               else
                  do i = 2, m
                     p(m+1-i) = C%double(1,i)
                  end do
               end if
               if( dim_r == 1 ) then
                  p(m:) = R%double(:,1)
               else
                  p(m:) = R%double(1,:)
               end if
               if( R%double(1,1) /= C%double(1,1) ) then
                  call PrintMessage( "mfToeplitz", "W",                     &
                                     "first elem. of R should be equal to", &
                                     "first elem. of C!" )
               end if

               real_result = .true.
            end if
         end if

      else ! .not. present(R)

         ! one arg: the row (stored in C)

         n = m
         if( mfIsReal(C) ) then
            allocate( p(m+n-1) )
            if( dim_c == 1 ) then
               do i = 2, m
                  p(m+1-i) = C%double(i,1)
               end do
               p(m:) = C%double(:,1)
            else
               do i = 2, m
                  p(m+1-i) = C%double(1,i)
               end do
               p(m:) = C%double(1,:)
            end if
            out%prop%symm = TRUE
            real_result = .true.
         else
            allocate( z(m+n-1) )
            if( dim_c == 1 ) then
               do i = 2, m
                  z(m+1-i) = conjg(C%cmplx(i,1))
               end do
               z(m:) = C%cmplx(:,1)
            else
               do i = 2, m
                  z(m+1-i) = conjg(C%cmplx(1,i))
               end do
               z(m:) = C%cmplx(1,:)
            end if
            if( aimag(C%cmplx(1,1)) == 0.0d0 ) then
               out%prop%symm = TRUE
            end if
         end if

      end if

      out%shape = [ m, n ]
      if( real_result ) then
         out%data_type = MF_DT_DBLE
         allocate( out%double(m,n) )

         do j = 1, n
            do i = 1, m
               out%double(i,j) = p(j-i+m)
            end do
         end do
      else
         out%data_type = MF_DT_CMPLX
         allocate( out%cmplx(m,n) )

         do j = 1, n
            do i = 1, m
               out%cmplx(i,j) = z(j-i+m)
            end do
         end do
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( C )
      call msAutoRelease( C )

      if( present(R) ) then
         call msFreeArgs( R )
         call msAutoRelease( R )
      end if

#endif
   end function mfToeplitz
!_______________________________________________________________________
!
   function mfMerge( A, B, mask ) result( out )

      type(mfArray) :: A, B, mask
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: m, n

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A, B, mask )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsEmpty(B) ) then
         go to 99
      end if

      if( mfIsEmpty(mask) ) then
         go to 99
      end if

      ! 'A' cannot be sparse
      if( mfIsSparse(A) ) then
         call PrintMessage( "mfMerge", "E",                             &
                            "A cannot be sparse!" )
         go to 99
      end if

      ! 'B' cannot be sparse
      if( mfIsSparse(B) ) then
         call PrintMessage( "mfMerge", "E",                             &
                            "B cannot be sparse!" )
         go to 99
      end if

      if( mfIsPerm(A) .or. mfIsPerm(B) ) then
         call PrintMessage( "mfIsFinite", "E",                          &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      ! 'mask' cannot be sparse
      if( mfIsSparse(mask) ) then
         call PrintMessage( "mfMerge", "E",                             &
                            "mask cannot be sparse!" )
         go to 99
      end if

      if( any( B%shape /= A%shape ) .or. any( mask%shape /= A%shape ) ) then
         call PrintMessage( "mfMerge", "E",                             &
                            "A, B and mask must have the same shape!" )
         go to 99
      end if

      ! 'A' must be numeric
      if( .not. mfIsNumeric(A) ) then
         call PrintMessage( "mfMerge", "E",                             &
                            "A must be numeric!" )
         go to 99
      end if

      ! 'B' must have the same type as 'A'
      if( B%data_type /= A%data_type ) then
         call PrintMessage( "mfMerge", "E",                             &
                            "A and B must have the same type!" )
         go to 99
      end if

      ! 'mask' must be boolean
      if( .not. mfIsLogical(mask) ) then
         call PrintMessage( "mfMerge", "E",                             &
                            "mask must be boolean!" )
         go to 99
      end if

      out%data_type = A%data_type
      out%shape = A%shape
      m = A%shape(1)
      n = A%shape(2)

      if( mfIsReal(A) ) then ! real case
         allocate( out%double(m,n) )

         out%double = merge( A%double, B%double, mask%double/=0.0d0 )
      else ! complex case
         allocate( out%cmplx(m,n) )

         out%cmplx = merge( A%cmplx, B%cmplx, mask%double/=0.0d0 )
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A, B, mask )
      call msAutoRelease( A, B, mask )

#endif
   end function mfMerge
!_______________________________________________________________________
!
   function mfPack( A, mask ) result( out )

      type(mfArray) :: A, mask
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! similar to the 'pack' f90 intrinsic function.

      integer :: j, m, n, np
      type(mfArray) :: tmp

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A, mask )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsEmpty(mask) ) then
         go to 99
      end if

      ! 'A' cannot be sparse
      if( mfIsSparse(A) ) then
         call PrintMessage( "mfPack", "E",                              &
                            "A cannot be sparse!" )
         go to 99
      end if

      ! 'mask' cannot be sparse
      if( mfIsSparse(mask) ) then
         call PrintMessage( "mfPack", "E",                              &
                            "mask cannot be sparse!" )
         go to 99
      end if

      if( any( mask%shape /= A%shape ) ) then
         call PrintMessage( "mfPack", "E",                              &
                            "A and mask must have the same shape!" )
         go to 99
      end if

      ! 'A' must be numeric
      if( .not. mfIsNumeric(A) ) then
         call PrintMessage( "mfPack", "E",                              &
                            "A must be numeric!" )
         go to 99
      end if

      ! 'mask' must be boolean
      if( .not. mfIsLogical(mask) ) then
         call PrintMessage( "mfPack", "E",                              &
                            "mask must be boolean!" )
         go to 99
      end if

      ! counting the number np of true elements in mask
      ! 'mfSum' is not available, because defined in mod_datafun.
      call msAssign( tmp, mfCount(mask) )
      n = Size(tmp)
      np = 0
      do j = 1, n
         np = np + int( tmp%double(1,j) )
      end do
      call msRelease(tmp)

      m = A%shape(1)
      n = A%shape(2)
      out%data_type = A%data_type
      out%shape = [ 1, np ]
      if( mfIsReal(A) ) then
         allocate( out%double(1,np) )

         out%double(1,:) = pack( A%double, mask%double/=0.0d0 )
      else
         allocate( out%cmplx(1,np) )

         out%cmplx(1,:) = pack( A%cmplx, mask%double/=0.0d0 )
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A, mask )
      call msAutoRelease( A, mask )

#endif
   end function mfPack
!_______________________________________________________________________
!
   function mfUnpack( A, mask, field ) result( out )

      type(mfArray) :: A, mask, field
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! similar to the 'unpack' f90 intrinsic function.

      integer :: m, n, np

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A, mask, field )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsEmpty(mask) ) then
         go to 99
      end if

      if( mfIsEmpty(field) ) then
         go to 99
      end if

      ! 'A' cannot be sparse
      if( mfIsSparse(A) ) then
         call PrintMessage( "mfUnpack", "E",                            &
                            "A cannot be sparse!" )
         go to 99
      end if

      ! 'mask' cannot be sparse
      if( mfIsSparse(mask) ) then
         call PrintMessage( "mfUnpack", "E",                            &
                            "mask cannot be sparse!" )
         go to 99
      end if

      ! 'field' cannot be sparse
      if( mfIsSparse(field) ) then
         call PrintMessage( "mfUnpack", "E",                            &
                            "field cannot be sparse!" )
         go to 99
      end if

      ! 'A' must be numeric
      if( .not. mfIsNumeric(A) ) then
         call PrintMessage( "mfUnpack", "E",                            &
                            "A must be numeric!" )
         go to 99
      end if

      ! 'field' must be numeric
      if( .not. mfIsNumeric(field) ) then
         call PrintMessage( "mfUnpack", "E",                            &
                            "field must be numeric!" )
         go to 99
      end if

      ! 'A' must be a row vector
      if( A%shape(1) /= 1 ) then
         call PrintMessage( "mfUnpack", "E",                            &
                            "A must be a row vector!" )
         go to 99
      end if

      if( A%data_type /= field%data_type ) then
         call PrintMessage( "mfUnpack", "E",                            &
                            "field and A must have the same data type!" )
         go to 99
      end if

      ! 'mask' must be boolean
      if( .not. mfIsLogical(mask) ) then
         call PrintMessage( "mfUnpack", "E",                            &
                            "mask must be boolean!" )
         go to 99
      end if

      m = mask%shape(1)
      n = mask%shape(2)
      out%data_type = A%data_type
      out%shape = [ m, n ]
      np = Size(A)
      if( mfIsReal(A) ) then
         allocate( out%double(m,n) )

         out%double = unpack( A%double(1,:), mask%double/=0.0d0, field%double )
      else
         allocate( out%cmplx(m,n) )

         out%cmplx = unpack( A%cmplx(1,:), mask%double/=0.0d0, field%cmplx )
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A, mask, field )
      call msAutoRelease( A, mask, field )

#endif
   end function mfUnpack
!_______________________________________________________________________
!
   function mfCshift( A, shift ) result( out )

      type(mfArray) :: A
      integer, intent(in) :: shift
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! vector A must be dense, but of any type : boolean or numeric

      integer :: dim

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      ! 'A' cannot be sparse
      if( mfIsSparse(A) ) then
         call PrintMessage( "mfCshift", "E",                            &
                            "A cannot be sparse!" )
         go to 99
      end if

      if( mfIsPerm(A) ) then
         call PrintMessage( "mfIsCshift", "E",                          &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      ! 'A' must be a vector
      if( A%shape(1) == 1 ) then
         dim = 2
      else if( A%shape(2) == 1 ) then
         dim = 1
      else
         call PrintMessage( "mfCshift", "E",                            &
                            "A must be a vector!" )
         go to 99
      end if

      out%data_type = A%data_type
      out%shape = A%shape

      if( A%data_type == MF_DT_DBLE .or.                                &
          A%data_type == MF_DT_BOOL ) then ! real or bool
         allocate( out%double(out%shape(1),out%shape(2)) )

         out%double = cshift( A%double, shift, dim=dim )
      else if( A%data_type == MF_DT_CMPLX ) then
         allocate( out%cmplx(out%shape(1),out%shape(2)) )

         out%cmplx = cshift( A%cmplx, shift, dim=dim )
      else
         call PrintMessage( "mfCshift", "E",                            &
                            "unknown data type!" )
         go to 99
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfCshift
!_______________________________________________________________________
!
   function mfEoshift( A, shift, boundary ) result( out )

      type(mfArray) :: A
      integer, intent(in) :: shift
      type(mfArray), optional :: boundary
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! vector A must be dense, but of any type : boolean or numeric
      !
      ! if present, boundary must be scalar

      integer :: dim

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      ! 'A' cannot be sparse
      if( mfIsSparse(A) ) then
         call PrintMessage( "mfEoshift", "E",                           &
                            "A cannot be sparse!" )
         go to 99
      end if

      if( mfIsPerm(A) ) then
         call PrintMessage( "mfEoshift", "E",                           &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      ! 'A' must be a vector
      if( A%shape(1) == 1 ) then
         dim = 2
      else if( A%shape(2) == 1 ) then
         dim = 1
      else
         call PrintMessage( "mfEoshift", "E",                           &
                            "A must be a vector!" )
         go to 99
      end if

      if( present(boundary) ) then

         call msInitArgs( boundary )

         if( mfIsEmpty(boundary) ) then
            call PrintMessage( "mfEoshift", "E",                        &
                               "boundary cannot be empty!" )
            go to 99
         end if

         if( boundary%data_type /= A%data_type ) then
            call PrintMessage( "mfEoshift", "E",                        &
                               "boundary must have the same type as A!" )
            go to 99
         end if

         if( .not. mfIsScalar(boundary) ) then
            call PrintMessage( "mfEoshift", "E",                        &
                               "boundary cannot be empty!" )
            go to 99
         end if

      end if

      out%data_type = A%data_type
      out%shape = A%shape

      if( A%data_type == MF_DT_DBLE .or.                                &
          A%data_type == MF_DT_BOOL ) then ! real or bool
         allocate( out%double(out%shape(1),out%shape(2)) )

         if( present(boundary) ) then
            out%double = eoshift( A%double, shift,                      &
                                  boundary%double(1,1), dim=dim )
         else
            out%double = eoshift( A%double, shift, dim=dim )
         end if
      else if( A%data_type == MF_DT_CMPLX ) then
         allocate( out%cmplx(out%shape(1),out%shape(2)) )

         if( present(boundary) ) then
            out%cmplx = eoshift( A%cmplx, shift,                        &
                                 boundary%cmplx(1,1), dim=dim )
         else
            out%cmplx = eoshift( A%cmplx, shift, dim=dim )
         end if
      else
         call PrintMessage( "mfEoshift", "E",                           &
                            "unknown data type!" )
         go to 99
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )
      if( present(boundary) ) then
         call msFreeArgs( boundary )
         call msAutoRelease( boundary )
      end if

#endif
   end function mfEoshift
!_______________________________________________________________________
!
   function mfSpRand_S( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! like the MATLAB-7 function: sprand(A)

      ! MUESLI implementation: very close to 'mfSpOnes' in SPARSE module

      integer :: ncol, nnz, k

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfSpRand", "W",                            &
                            "arg. is empty." )
         go to 99
      end if

      if( mfIsLogical(A) ) then
         call PrintMessage( "mfSpRand", "E",                            &
                            "cannot be applied to a boolean mfArray!" )
         go to 99
      end if

      if( mfIsPerm(A) ) then
         call PrintMessage( "mfSpRand", "E",                            &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( .not. RngStream_initialized ) then
         call RngStream_init()
      end if

      if( mfIsSparse(A) ) then

         out%data_type = MF_DT_SP_DBLE
         out%shape = A%shape

         ncol = A%shape(2)
         nnz = A%j(ncol+1) - 1

         if( A%data_type == MF_DT_SP_DBLE .and.                         &
             A%status_temporary .and. .not. A%status_restricted ) then
            out%i => A%i
            out%j => A%j
            out%a => A%a
            do k = 1, nnz
               call rng_rand_u01( Rng_Stream_g_addr, out%a(k) )
            end do
            A%status_temporary = .false.
         else
            allocate( out%i(nnz) )

            out%i(1:nnz) = A%i(1:nnz)
            allocate( out%j(ncol+1) )

            out%j(:) = A%j(:)
            allocate( out%a(nnz) )

            do k = 1, nnz
               call rng_rand_u01( Rng_Stream_g_addr, out%a(k) )
            end do
         end if

      else ! A is dense

         out = mfSparse( A )

         if( mfIsComplex(A) ) then
            ! convert 'out' to real
            out%data_type = MF_DT_SP_DBLE
            deallocate( out%z )

            ncol = out%shape(2)
            nnz = out%j(ncol+1) - 1
            allocate( out%a(nnz) )

         end if
         do k = 1, nnz
            call rng_rand_u01( Rng_Stream_g_addr, out%a(k) )
         end do

      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfSpRand_S
!_______________________________________________________________________
!
   function mfSpRand_m_n( m, n, density ) result( out )

      integer, intent(in) :: m, n
      real(kind=MF_DOUBLE), intent(in) :: density
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! like the MATLAB-7 function: sprand(A)

      integer :: nnz, k
      integer, allocatable :: i(:), j(:)
      real(kind=MF_DOUBLE), allocatable :: val(:)

   !------ end of declarations -- execution starts hereafter  ------

      if( m < 1 .or. n < 1 ) then
         call PrintMessage( "mfSpRand", "W",                            &
                            "size (m,n) must be consistent!" )
         return
      end if

      if( density < 0.0d0 .or. 1.0d0 < density ) then
         call PrintMessage( "mfSpRand", "W",                            &
                            "'density' must be in [0,1]!" )
         return
      end if

      if( .not. RngStream_initialized ) then
         call RngStream_init()
      end if

      nnz = nint( m*n*density )

      allocate( i(nnz), j(nnz), val(nnz) )

      do k = 1, nnz
         call rng_rand_u01( Rng_Stream_g_addr, val(k) )
      end do
      i(:) = floor( val(:)*m ) + 1
      do k = 1, nnz
         call rng_rand_u01( Rng_Stream_g_addr, val(k) )
      end do
      j(:) = floor( val(:)*n ) + 1
      do k = 1, nnz
         call rng_rand_u01( Rng_Stream_g_addr, val(k) )
      end do

      call msAssign( out, mfSpImport( i, j, val, m, n,                  &
                                      duplicated_entries="ignored" ) )

      out%status_temporary = .true.

#endif
   end function mfSpRand_m_n
!_______________________________________________________________________
!
   function mfSpRandN_S( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! like the MATLAB-7 function: sprandn(A)

      ! MUESLI implementation: very close to 'mfSpOnes' in SPARSE module

      integer :: ncol, nnz, k

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfSpRandN", "W",                           &
                            "arg. is empty." )
         go to 99
      end if

      if( mfIsLogical(A) ) then
         call PrintMessage( "mfSpRandN", "E",                           &
                            "cannot be applied to a boolean mfArray!" )
         go to 99
      end if

      if( mfIsPerm(A) ) then
         call PrintMessage( "mfSpRandN", "E",                           &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( .not. RngStream_initialized ) then
         call RngStream_init()
      end if

      if( mfIsSparse(A) ) then

         out%data_type = MF_DT_SP_DBLE
         out%shape = A%shape

         ncol = A%shape(2)
         nnz = A%j(ncol+1) - 1

         if( A%data_type == MF_DT_SP_DBLE .and.                         &
             A%status_temporary .and. .not. A%status_restricted ) then
            out%i => A%i
            out%j => A%j
            out%a => A%a
            do k = 1, nnz
               out%a(k) = randn()
            end do
            A%status_temporary = .false.
         else
            allocate( out%i(nnz) )

            out%i(1:nnz) = A%i(1:nnz)
            allocate( out%j(ncol+1) )

            out%j(:) = A%j(:)
            allocate( out%a(nnz) )

            do k = 1, nnz
               out%a(k) = randn()
            end do
         end if

      else ! A is dense

         out = mfSparse( A )

         if( mfIsComplex(A) ) then
            ! convert 'out' to real
            out%data_type = MF_DT_SP_DBLE
            deallocate( out%z )

            ncol = out%shape(2)
            nnz = out%j(ncol+1) - 1
            allocate( out%a(nnz) )

         end if
         do k = 1, nnz
            out%a(k) = randn()
         end do

      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfSpRandN_S
!_______________________________________________________________________
!
   function mfSpRandN_m_n( m, n, density ) result( out )

      integer, intent(in) :: m, n
      real(kind=MF_DOUBLE), intent(in) :: density
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! like the MATLAB-7 function: sprand(A)

      integer :: nnz, k
      integer, allocatable :: i(:), j(:)
      real(kind=MF_DOUBLE), allocatable :: val(:)

   !------ end of declarations -- execution starts hereafter  ------

      if( m < 1 .or. n < 1 ) then
         call PrintMessage( "mfSpRandN", "W",                           &
                            "size (m,n) must be consistent!" )
         return
      end if

      if( density < 0.0d0 .or. 1.0d0 < density ) then
         call PrintMessage( "mfSpRandN", "W",                           &
                            "'density' must be in [0,1]!" )
         return
      end if

      if( .not. RngStream_initialized ) then
         call RngStream_init()
      end if

      nnz = nint( m*n*density )

      allocate( i(nnz), j(nnz), val(nnz) )

      do k = 1, nnz
         call rng_rand_u01( Rng_Stream_g_addr, val(k) )
      end do
      i(:) = floor( val(:)*m ) + 1
      do k = 1, nnz
         call rng_rand_u01( Rng_Stream_g_addr, val(k) )
      end do
      j(:) = floor( val(:)*n ) + 1
      do k = 1, nnz
         val(k) = randn()
      end do

      call msAssign( out, mfSpImport( i, j, val, m, n,                  &
                                      duplicated_entries="ignored" ) )

      out%status_temporary = .true.

#endif
   end function mfSpRandN_m_n
!_______________________________________________________________________
!
   function mfKron( A, B ) result( out )

      type(mfArray) :: A, B
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! like the MATLAB-7 function: kron(A,B)

      ! only the real case is implemented

      integer :: ma, na, mb, nb
      type(mfArray) :: ia, ib, ja, jb
      type(mfArray) :: sa, sb, ka, kb, t, ik, jk
      type(mfArray) :: tmp

      real(kind=MF_DOUBLE), pointer :: ik_ptr(:), jk_ptr(:), tmp_ptr(:)

      integer,              allocatable :: ik_vec(:), jk_vec(:)
      real(kind=MF_DOUBLE), allocatable :: tmp_vec(:)

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A, B )

      if( mfIsLogical(A) .or. mfIsLogical(B) ) then
         call PrintMessage( "mfKron", "E",                              &
                            "cannot be applied to a boolean mfArray!" )
         go to 99
      end if

      if( mfIsPerm(A) .or. mfIsPerm(B) ) then
         call PrintMessage( "mfKron", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( mfIsComplex(A) .or. mfIsComplex(B) ) then
         call PrintMessage( "mfKron", "E",                              &
                            "'A' and 'B' must be real!",                &
                            "(complex case not implemented yet)" )
         go to 99
      end if

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfKron", "E",                              &
                            "arg. #1 is empty." )
         go to 99
      end if

      ma = size(A,1)
      na = size(A,2)

      if( mfIsEmpty(B) ) then
         call PrintMessage( "mfKron", "E",                              &
                            "arg. #2 is empty." )
         go to 99
      end if

      mb = size(B,1)
      nb = size(B,2)

      if( mfIsSparse(A) .or. mfIsSparse(B) ) then

         ! At least one input is sparse, result is sparse.
         call msFind( mfOut(ia,ja,sa), A )
         call msFind( mfOut(ib,jb,sb), B )
         call msAssign( ka, mfOnes( size(sa), 1 ) ) ! ka and kb must
         call msAssign( kb, mfOnes( size(sb), 1 ) ) ! be columns vectors
         call msAssign( t, dble(mb)*(ia-1.0d0) ) ! t is a row vector
         call msAssign( ik, mfGet(t,kb,MF_COLON) + mfGet(.t.ib,MF_COLON,ka) )
         call msAssign( t, dble(nb)*(ja-1.0d0) )
         call msAssign( jk, mfGet(t,kb,MF_COLON) + mfGet(.t.jb,MF_COLON,ka) )
         tmp = mfMul( sb, sa, transp=1 )

         ik_ptr => rank_2_to_1_real8( ik%double, size(ik%double) )
         allocate( ik_vec(size(ik)) )
         ik_vec = int( ik_ptr )
         jk_ptr => rank_2_to_1_real8( jk%double, size(jk%double) )
         allocate( jk_vec(size(jk)) )
         jk_vec = int( jk_ptr )
         tmp_ptr => rank_2_to_1_real8( tmp%double, size(tmp%double) )
         allocate( tmp_vec(size(tmp)) )
         tmp_vec = tmp_ptr
         out = mfSpImport( ik_vec, jk_vec, tmp_vec, ma*mb, na*nb )

      else

         ! Both inputs full, result is full.
         call msMeshGrid( mfOut(ia,ib), mfColon(1,ma), .t. mfColon(1,mb) )
         call msMeshGrid( mfOut(ja,jb), mfColon(1,na), .t. mfColon(1,nb) )
         call msAssign( out, mfGet(A,ia,ja) * mfGet(B,ib,jb) )

      end if

      out%status_temporary = .true.

 99   continue

      call msSilentRelease( ia, ib, ja, jb )
      call msSilentRelease( sa, sb, ka, kb, t, ik, jk )
      call msSilentRelease( tmp )

      call msFreeArgs( A, B )
      call msAutoRelease( A, B )

#endif
   end function mfKron
!_______________________________________________________________________
!
   function mfPerm_mfArray( v ) result( out )

      type(mfArray) :: v
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! creation of a integer permutation vector (PERM_VEC data type),
      ! from a dense real mfArray (DBLE data type).

      ! The 'v' mfArray must be a vector and should contain only integer
      ! values.

      integer :: n, idim

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( v )

      if( mfIsEmpty(v) ) then
         call PrintMessage( "mfPerm", "E",                              &
                            "arg. 'v' is empty." )
         go to 99
      end if

      if( v%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfPerm", "E",                              &
                            "arg. 'v' must be a real, dense vector!" )
         go to 99
      end if

      if( v%shape(1) == 1 ) then
         idim = 2
      else  if( v%shape(2) == 1 ) then
         idim = 1
      else
         call PrintMessage( "mfPerm", "E",                              &
                            "arg. 'v' must be a real, dense vector!" )
         go to 99
      end if

      n = v%shape(idim)

      out%data_type = MF_DT_PERM_VEC
      out%shape = [ n, 1 ]
      allocate( out%i(n) )

      if( idim == 1 ) then
         ! col. input vector
         out%i(:) = nint(v%double(:,1))
         if( any( dble(out%i(:)) /= v%double(:,1) ) ) then
            call PrintMessage( "mfPerm", "E",                           &
                               "values for argument 'v' are not integer!", &
                               "(at least one value detected as such)" )
         end if
      else ! idim = 2
         ! row. input vector
         out%i(:) = nint(v%double(1,:))
         if( any( dble(out%i(:)) /= v%double(1,:) ) ) then
            call PrintMessage( "mfPerm", "E",                           &
                               "values for argument 'v' are not integer!", &
                               "(at least one value detected as such)" )
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( v )
      call msAutoRelease( v )

#endif
   end function mfPerm_mfArray
!_______________________________________________________________________
!
   function mfPerm_vec_int( v ) result( out )

      integer, intent(in) :: v(:)
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! creation of a integer permutation vector (PERM_VEC data type),
      ! from an integer vector (Fortran 90).

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = size(v)

      if( n == 0 ) then
         call PrintMessage( "mfPerm", "E",                              &
                            "arg. 'v' is empty." )
         return
      end if

      out%data_type = MF_DT_PERM_VEC
      out%shape = [ n, 1 ]
      allocate( out%i(n) )

      out%i(:) = v(:)

      out%status_temporary = .true.

#endif
   end function mfPerm_vec_int
!_______________________________________________________________________
!
   function mfCheckPerm( A ) result( bool )

      type(mfArray) :: A
      logical :: bool
      !------ API end ------

#ifdef _DEVLP
      integer, allocatable :: p_sorted(:)
      integer :: n, i

   !------ end of declarations -- execution starts hereafter  ------

      if( A%data_type /= MF_DT_PERM_VEC )   then
         call PrintMessage( "mfCheckPerm", "E",                         &
                            "mfArray arg. must be a permutation.",      &
                            "(use mfPerm to convert it)" )
         return
      end if

      n = A%shape(1)
      allocate( p_sorted(n) )
      p_sorted(:) = A%i(:)
      call quick_sort( "asc", p_sorted )

      bool = .true.
      do i = 1, n
         if( p_sorted(i) /= i ) then
            bool = .false.
            exit
         end if
      end do

      call msAutoRelease( A )

#endif
   end function mfCheckPerm
!_______________________________________________________________________
!
   function mfRandPerm( n, k ) result( out )

      integer, intent(in)           :: n
      integer, intent(in), optional :: k
      type(mfArray)                 :: out
      !------ API end ------

#ifdef _DEVLP
      ! = randperm of Matlab

      ! k must be less than or equal to n

      ! creation of a random integer permutation vector (PERM_VEC data type).
      ! (Fisher-Yates algorithm)

      ! here, RandPerm(n) is slightly more efficient than RandPerm(n,n)

      integer :: i, j, s
      real(kind=MF_DOUBLE) :: r
      integer, allocatable :: tmp(:)

   !------ end of declarations -- execution starts hereafter  ------

      if( n < 0 ) then
         call PrintMessage( "mfRandPerm", "E",                          &
                            "arg. 'n' must be greater or equal 1." )
         return
      end if

      if( .not. RngStream_initialized ) then
         call RngStream_init()
      end if

      out%data_type = MF_DT_PERM_VEC

      if( present(k) ) then

         out%shape = [ k, 1 ]
         allocate( out%i(k) )

         allocate( tmp(n) )
         tmp(:) = [ (i, i = 1, n) ]

         do i = n, max(n-k+1,2), -1
            ! swap tmp(i) with any previous element
            call rng_rand_u01( Rng_Stream_g_addr, r )
            j = int(r*i) + 1 ! j should be in [ 1, i ]
            ! swap i-th and j-th elements
            if( j /= i ) then
               s = tmp(i)
               tmp(i) = tmp(j)
               tmp(j) = s
            end if
         end do
         out%i(:) = tmp(n-k+1:n)

      else

         out%shape = [ n, 1 ]
         allocate( out%i(n) )

         out%i(:) = [ (i, i = 1, n) ]

         do i = n, 2, -1
            ! swap out%i(i) with any previous element
            call rng_rand_u01( Rng_Stream_g_addr, r )
            j = int(r*i) + 1 ! j should be in [ 1, i ]
            ! swap i-th and j-th elements
            if( j /= i ) then
               s = out%i(i)
               out%i(i) = out%i(j)
               out%i(j) = s
            end if
         end do

      end if

      out%status_temporary = .true.

#endif
   end function mfRandPerm
!_______________________________________________________________________
!
   function rand_exp() result( r )

      double precision :: r
      !------ API end ------

#ifdef _DEVLP
      ! Generates a random variate in [0,infinity).

   !------ end of declarations -- execution starts hereafter  ------

      do
         call rng_rand_u01( Rng_Stream_g_addr, r )
         if( r > 0.0d0 ) exit
      end do

      r = -log(r)

#endif
   end function rand_exp
!_______________________________________________________________________
!
#include "misc/randlib/ignpoi.inc"
!_______________________________________________________________________
!
   function mfRandPoiss_one_arg( mu, n ) result( out )

      double precision, intent(in) :: mu
      integer, intent(in) :: n
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      if( mu < 0.0d0 ) then
         call PrintMessage( "mfRandPoiss", "E",                         &
                            "bad arg value: mean 'mu' must be positive!" )
         return
      end if

      if( n < 1 ) then
         call PrintMessage( "mfRandPoiss", "E",                         &
                            "bad arg value: first size must be greater than 1!" )
         return
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ n, n ]
      allocate( out%double(n,n) )

      if( .not. RngStream_initialized ) then
         call RngStream_init()
      end if

      do j = 1, n
         do i = 1, n
            out%double(i,j) = ignpoi( mu )
        end do
      end do

      out%prop%tril = FALSE
      out%prop%triu = FALSE
      out%prop%symm = FALSE

      out%status_temporary = .true.

#endif
   end function mfRandPoiss_one_arg
!_______________________________________________________________________
!
   function mfRandPoiss_two_arg( mu, n1, n2 ) result( out )

      double precision, intent(in) :: mu
      integer, intent(in) :: n1, n2
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      if( mu < 0.0d0 ) then
         call PrintMessage( "mfRandPoiss", "E",                         &
                            "bad arg value: mean 'mu' must be positive!" )
         return
      end if

      if( n1 < 1 .or. n2 < 1 ) then
         call PrintMessage( "mfRandPoiss", "E",                         &
                            "bad arg value for at least one size!",     &
                            "(they must be both greater than 1)")
         return
      end if

      out%data_type = MF_DT_DBLE
      out%shape = [ n1, n2 ]
      allocate( out%double(n1,n2) )

      if( .not. RngStream_initialized ) then
         call RngStream_init()
      end if

      do j = 1, n2
         do i = 1, n1
            out%double(i,j) = ignpoi( mu )
        end do
      end do

      out%prop%tril = FALSE
      out%prop%triu = FALSE
      out%prop%symm = FALSE

      out%status_temporary = .true.

#endif
   end function mfRandPoiss_two_arg
!_______________________________________________________________________
!
end module mod_elmat
