module mod_elfun ! Elementary Math Functions

   ! Part of MUESLI Numerical Library
   ! Copyright É. Canot 2003-2025 -- IPR/CNRS

!-----------------------------------------------------------------------
!                             used modules
!-----------------------------------------------------------------------

   use mod_mf_gsl
   use mod_slatec_fun

   use mod_sparse

#ifndef _DEVLP
   use mod_mfarray ! required for the 2nd pass of the double compilation
   use mod_core ! required for the 2nd pass of the double compilation
#endif

   implicit none

#ifndef _DEVLP
   private
#endif

   interface mfComplex
      module procedure mfComplex_one_arg
      module procedure mfComplex_two_arg
   end interface mfComplex
   !------ API end ------

   interface mfPow2
      module procedure mfPow2_1
      module procedure mfPow2_2
   end interface mfPow2
   !------ API end ------

   interface mfFun
      module procedure mfFun_apply_real
      module procedure mfFun_apply_cmplx
   end interface mfFun
   !------ API end ------

   interface mfFun2
      module procedure mfFun2_apply_real
      module procedure mfFun2_apply_cmplx
   end interface mfFun2
   !------ API end ------

   interface mfGridFun
      module procedure mfGridFun_intr
      module procedure mfGridFun_user
   end interface mfGridFun
   !------ API end ------

   public :: mfCos, &
             mfSin, &
             mfATan2, &
             mfExp, &
             mfExpm1, &
             mfLog, &
             mfLog1p, &
             mfLog10, &
             mfLog2, msLog2, &
             mfPow2, &
             mfPow10, &
             mfFun, mfFun2, &
             mfGridFun, &
             mfSqrt, &
             mfAbs, &
             mfSign, &
             mfCSign, &
             mfAngle, &
             mfComplex, &
             mfConj, &
             mfImag, &
             mfReal, &
             mfHypot, &
             mfCeil, &
             mfFix, &
             mfFloor, &
             mfMod, &
             mfRem, &
             mfRound

   public :: mfACos, &
             mfACosh, &
             mfASin, &
             mfASinh, &
             mfATan, &
             mfATanh, &
             mfCosh, &
             mfSinh, &
             mfTan, &
             mfTanh

   public :: mfACot, &
             mfACoth, &
             mfACsc, &
             mfACsch, &
             mfASec, &
             mfASech, &
             mfCot, &
             mfCoth, &
             mfCsc, &
             mfCsch, &
             mfSec, &
             mfSech

#if defined _INTEL_IFC | defined _GNU_GFC
   ! these two compilers both support the 'hypot' Fortran 2008 intrinsic
#else
   ! hypot() will become an intrinsic function only with Fortran 2008
   ! for internal usage (other modules, also)
   interface hypot
      module procedure hypot_real
      module procedure hypot_cmplx
   end interface hypot

   public :: hypot
#endif

   private :: mfComplex_one_arg, mfComplex_two_arg, mfPow2_1, mfPow2_2, &
              mfFun_apply_real, mfFun_apply_cmplx, mfFun2_apply_real, &
              mfFun2_apply_cmplx, hypot_real, hypot_cmplx

contains
!_______________________________________________________________________
!
   ! first, trigonometric routines not in standard Fortran
!_______________________________________________________________________
!
   function mfACos( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      if( mf_auto_complex ) then
         call mf_save_and_disable_fpe( )
      end if

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfACos", "E",                              &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfACos", "E",                              &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfACos", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         if( mf_auto_complex .and.                                      &
             any(abs(A%double(:,:)) > 1.0d0 ) ) then
            out%data_type = MF_DT_CMPLX
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

            ! in Fortran, 'acos' doesn't compute complex val.
            ! out%cmplx(:,:) = acos( cmplx(A%double(:,:)) )
            do j = 1, A%shape(2)
               do i = 1, A%shape(1)
                  out%cmplx(i,j) = mf_gsl_complex_arccos_real(A%double(i,j))
               end do
            end do
         else
            out%data_type = MF_DT_DBLE

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%double => a%double
               call set_status_tempo_to_false( a )
            else
               allocate( out%double(A%shape(1),A%shape(2)) )

            end if
            out%double(:,:) = acos( A%double(:,:) )
            out%prop%symm = A%prop%symm
         end if
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => a%cmplx
            call set_status_tempo_to_false( a )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         ! in Fortran, 'acos' is not defined for complex arg.
         ! out%cmplx(:,:) = acos( A%cmplx(:,:) )
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%cmplx(i,j) = mf_gsl_complex_arccos(A%cmplx(i,j))
            end do
         end do
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfACos", "E",                           &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      if( mf_auto_complex ) then
         call mf_restore_fpe( )
      end if

#endif
   end function mfACos
!_______________________________________________________________________
!
   function mfACosh( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      if( mf_auto_complex ) then
         call mf_save_and_disable_fpe( )
      end if

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfACosh", "E",                             &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfACosh", "E",                             &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfACosh", "E",                             &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         if( mf_auto_complex .and.                                      &
             any( A%double(:,:) < 1.0d0 ) ) then
            out%data_type = MF_DT_CMPLX
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

            do j = 1, A%shape(2)
               do i = 1, A%shape(1)
                  out%cmplx(i,j) = mf_gsl_complex_arccosh_real(A%double(i,j))
               end do
            end do
         else
            out%data_type = MF_DT_DBLE

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%double => a%double
               call set_status_tempo_to_false( a )
            else
               allocate( out%double(A%shape(1),A%shape(2)) )

            end if
            do j = 1, A%shape(2)
               do i = 1, A%shape(1)
                  out%double(i,j) = acosh(A%double(i,j))
               end do
            end do
            out%prop%symm = A%prop%symm
         end if
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => a%cmplx
            call set_status_tempo_to_false( a )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%cmplx(i,j) = mf_gsl_complex_arccosh(A%cmplx(i,j))
            end do
         end do
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfACosh", "E",                          &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      if( mf_auto_complex ) then
         call mf_restore_fpe( )
      end if

#endif
   end function mfACosh
!_______________________________________________________________________
!
   function mfASin( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      if( mf_auto_complex ) then
         call mf_save_and_disable_fpe( )
      end if

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfASin", "E",                              &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfASin", "E",                              &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfASin", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         if( mf_auto_complex .and.                                      &
             any(abs(A%double(:,:)) > 1.0d0 ) ) then
            out%data_type = MF_DT_CMPLX
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

            ! in Fortran, 'asin' doesn't compute complex val.
            ! out%cmplx(:,:) = asin( cmplx(A%double(:,:)) )
            do j = 1, A%shape(2)
               do i = 1, A%shape(1)
                  out%cmplx(i,j) = mf_gsl_complex_arcsin_real(A%double(i,j))
               end do
            end do
         else
            out%data_type = MF_DT_DBLE

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%double => a%double
               call set_status_tempo_to_false( a )
            else
               allocate( out%double(A%shape(1),A%shape(2)) )

            end if
            out%double(:,:) = asin( A%double(:,:) )
            out%prop%symm = A%prop%symm
         end if
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => a%cmplx
            call set_status_tempo_to_false( a )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         ! in Fortran, 'asin' is not defined for complex arg.
         ! out%cmplx(:,:) = asin( A%cmplx(:,:) )
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%cmplx(i,j) = mf_gsl_complex_arcsin(A%cmplx(i,j))
            end do
         end do
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfASin", "E",                           &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      if( mf_auto_complex ) then
         call mf_restore_fpe( )
      end if

#endif
   end function mfASin
!_______________________________________________________________________
!
   function mfASinh( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfASinh", "E",                             &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfASinh", "E",                             &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfASinh", "E",                             &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_DBLE

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%double => a%double
            call set_status_tempo_to_false( a )
         else
            allocate( out%double(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%double(i,j) = asinh(A%double(i,j))
            end do
         end do
         out%prop%symm = A%prop%symm
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => a%cmplx
            call set_status_tempo_to_false( a )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%cmplx(i,j) = mf_gsl_complex_arcsinh(A%cmplx(i,j))
            end do
         end do
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfASinh", "E",                          &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfASinh
!_______________________________________________________________________
!
   function mfATan( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfATan", "E",                              &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfATan", "E",                              &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfATan", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_DBLE

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%double => a%double
               call set_status_tempo_to_false( a )
            else
               allocate( out%double(A%shape(1),A%shape(2)) )

            end if
         out%double(:,:) = atan( A%double(:,:) )
         out%prop%symm = A%prop%symm
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => a%cmplx
            call set_status_tempo_to_false( a )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         ! in Fortran, 'atan' is not defined for complex arg.
         ! out%cmplx(:,:) = atan( A%cmplx(:,:) )
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%cmplx(i,j) = mf_gsl_complex_arctan(A%cmplx(i,j))
            end do
         end do
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfATan", "E",                           &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfATan
!_______________________________________________________________________
!
   function mfATanh( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      if( mf_auto_complex ) then
         call mf_save_and_disable_fpe( )
      end if

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfATanh", "E",                             &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfATanh", "E",                             &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfATanh", "E",                             &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         if( mf_auto_complex .and.                                      &
             any( A%double(:,:) < 1.0d0 ) ) then
            out%data_type = MF_DT_CMPLX
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

            do j = 1, A%shape(2)
               do i = 1, A%shape(1)
                  out%cmplx(i,j) = mf_gsl_complex_arctanh_real(A%double(i,j))
               end do
            end do
         else
            out%data_type = MF_DT_DBLE

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%double => a%double
               call set_status_tempo_to_false( a )
            else
               allocate( out%double(A%shape(1),A%shape(2)) )

            end if
            do j = 1, A%shape(2)
               do i = 1, A%shape(1)
                  out%double(i,j) = atanh(A%double(i,j))
               end do
            end do
            out%prop%symm = A%prop%symm
         end if
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => a%cmplx
            call set_status_tempo_to_false( a )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%cmplx(i,j) = mf_gsl_complex_arctanh(A%cmplx(i,j))
            end do
         end do
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfATanh", "E",                          &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      if( mf_auto_complex ) then
         call mf_restore_fpe( )
      end if

#endif
   end function mfATanh
!_______________________________________________________________________
!
   function mfCosh( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfCosh", "E",                              &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfCosh", "E",                              &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfCosh", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_DBLE

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%double => a%double
            call set_status_tempo_to_false( a )
         else
            allocate( out%double(A%shape(1),A%shape(2)) )

         end if
         out%double(:,:) = cosh( A%double(:,:) )
         out%prop%symm = A%prop%symm
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => a%cmplx
            call set_status_tempo_to_false( a )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         ! in Fortran, 'cosh' is not defined for complex arg.
         ! out%cmplx(:,:) = cosh( A%cmplx(:,:) )
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%cmplx(i,j) = mf_gsl_complex_cosh(A%cmplx(i,j))
            end do
         end do
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfCosh", "E",                           &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfCosh
!_______________________________________________________________________
!
   function mfSinh( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfSinh", "E",                              &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfSinh", "E",                              &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfSinh", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_DBLE

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%double => a%double
            call set_status_tempo_to_false( a )
         else
            allocate( out%double(A%shape(1),A%shape(2)) )

         end if
         out%double(:,:) = sinh( A%double(:,:) )
         out%prop%symm = A%prop%symm
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => a%cmplx
            call set_status_tempo_to_false( a )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         ! in Fortran, 'sinh' is not defined for complex arg.
         ! out%cmplx(:,:) = sinh( A%cmplx(:,:) )
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%cmplx(i,j) = mf_gsl_complex_sinh(A%cmplx(i,j))
            end do
         end do
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfSinh", "E",                           &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfSinh
!_______________________________________________________________________
!
   function mfTan( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfTan", "E",                               &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfTan", "E",                               &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfTan", "E",                               &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_DBLE

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%double => a%double
            call set_status_tempo_to_false( a )
         else
            allocate( out%double(A%shape(1),A%shape(2)) )

         end if
         out%double(:,:) = tan( A%double(:,:) )
         out%prop%symm = A%prop%symm
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => a%cmplx
            call set_status_tempo_to_false( a )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         ! in Fortran, 'tan' is not defined for complex arg.
         ! out%cmplx(:,:) = tan( A%cmplx(:,:) )
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%cmplx(i,j) = mf_gsl_complex_tan(A%cmplx(i,j))
            end do
         end do
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfTan", "E",                            &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfTan
!_______________________________________________________________________
!
   function mfTanh( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfTanh", "E",                              &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfTanh", "E",                              &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfTanh", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_DBLE

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%double => a%double
            call set_status_tempo_to_false( a )
         else
            allocate( out%double(A%shape(1),A%shape(2)) )

         end if
         out%double(:,:) = tanh( A%double(:,:) )
         out%prop%symm = A%prop%symm
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => a%cmplx
            call set_status_tempo_to_false( a )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         ! in Fortran, 'tanh' is not defined for complex arg.
         ! out%cmplx(:,:) = tanh( A%cmplx(:,:) )
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%cmplx(i,j) = mf_gsl_complex_tanh(A%cmplx(i,j))
            end do
         end do
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfTanh", "E",                           &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfTanh
!_______________________________________________________________________
!
   ! then, those using only standard Fortran math routines
!_______________________________________________________________________
!
   function mfSign( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! same behavior as Matlab
      ! supports real, complex and IEEE special values

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfSign", "E",                              &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( .not. mfIsNumeric(A) ) then
         call PrintMessage( "mfSign", "E",                              &
                            "numeric array required!" )
         go to 99
      end if

      out%data_type = MF_DT_DBLE
      out%shape = A%shape

      ! if one of the mfArrays is tempo and not protected,
      ! processing is done 'in place' (avoid an allocation).
      if( can_use_memory(A) ) then
         out%double => a%double
         call set_status_tempo_to_false( a )
      else
         allocate( out%double(A%shape(1),A%shape(2)) )

      end if

      if( A%data_type == MF_DT_DBLE ) then
         where( A%double > 0.0d0 )
            out%double = 1.0d0
         elsewhere( A%double < 0.0d0 )
            out%double = -1.0d0
         elsewhere( A%double == 0.0d0 )
            out%double = 0.0d0
         elsewhere ! NaN value
            out%double = MF_NAN
         end where
      else if( A%data_type == MF_DT_CMPLX ) then
         where( abs(A%cmplx) > 0.0d0 )
            out%double = A%cmplx / abs( A%cmplx )
         elsewhere( A%cmplx == (0.0d0,0.0d0) )
            out%double = 0.0d0
         elsewhere ! NaN value
            out%double = MF_NAN
         end where
      end if

      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfSign
!_______________________________________________________________________
!
   function mfCSign( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! Complex signum (csgn def. from Maple), extended for mfArray

      complex(kind=MF_DOUBLE) :: aa
      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfCSign", "E",                             &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfCSign", "E",                             &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfCSign", "E",                             &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( A%data_type == MF_DT_DBLE ) then
         ! real case : mfCSign = mfSign
         call msAssign( out, mfSign(A) )
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_DBLE
         out%shape = A%shape
         allocate( out%double(A%shape(1),A%shape(2)) )

         ! complex case :
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               aa = A%cmplx(i,j)
               if( aa == (0.0d0,0.0d0) ) then
                  out%double(i,j) =  0.0d0
               else if( real(aa) > 0.0d0 .or.                           &
                        (real(aa)==0.0d0 .and. aimag(aa)>0.0d0) ) then
                  out%double(i,j) =  1.0d0
               else if( real(aa) < 0.0d0 .or.                           &
                        (real(aa)==0.0d0 .and. aimag(aa)<0.0d0) ) then
                  out%double(i,j) = -1.0d0
               end if
            end do
         end do
      end if

      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfCSign
!_______________________________________________________________________
!
   function mfCos( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfCos", "E",                               &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfCos", "E",                               &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfCos", "E",                               &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_DBLE

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%double => a%double
            call set_status_tempo_to_false( a )
         else
            allocate( out%double(A%shape(1),A%shape(2)) )

         end if
         out%double(:,:) = cos( A%double(:,:) )
         out%prop%symm = A%prop%symm
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => a%cmplx
            call set_status_tempo_to_false( a )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         out%cmplx(:,:) = cos( A%cmplx(:,:) )
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfCos", "E",                            &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfCos
!_______________________________________________________________________
!
   function mfSin( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfSin", "E",                               &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfSin", "E",                               &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfSin", "E",                               &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_DBLE

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%double => a%double
            call set_status_tempo_to_false( a )
         else
            allocate( out%double(A%shape(1),A%shape(2)) )

         end if
         out%double(:,:) = sin( A%double(:,:) )
         out%prop%symm = A%prop%symm
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => a%cmplx
            call set_status_tempo_to_false( a )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         out%cmplx(:,:) = sin( A%cmplx(:,:) )
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfSin", "E",                            &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfSin
!_______________________________________________________________________
!
   function mfATan2( Y, X ) result( out )

      type(mfArray) :: X, Y
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! computes atan(Y/X) with a result in (-Pi,Pi].

      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( X, Y )

      if( mfIsEmpty(X) .or. mfIsEmpty(Y) ) then
         go to 99
      end if

      if( X%data_type /= MF_DT_DBLE .or. Y%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfATan2", "E",                             &
                            "X and Y must be real array" )
         go to 99
      end if

      if( any(X%shape/=Y%shape) ) then
         call PrintMessage( "mfATan2", "E",                             &
                            "X and Y must have the same shape" )
         go to 99
      end if

      out%shape = X%shape
      if( X%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_DBLE

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(X) ) then
            out%double => X%double
            call set_status_tempo_to_false( X )
         else if( can_use_memory(Y) ) then
            out%double => Y%double
            call set_status_tempo_to_false( Y )
         else
            allocate( out%double(X%shape(1),X%shape(2)) )

         end if
         out%double(:,:) = atan2( Y%double(:,:), X%double(:,:) )
      else
         call PrintMessage( "mfATan2", "E",                             &
                            "not available for complex args!" )
         go to 99
      end if

      if( Y%prop%symm == TRUE .and. Y%prop%symm == TRUE ) then
         out%prop%symm = TRUE
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( X%units, Y%units, status )
         if( status /= 0 ) then
            call PrintMessage( "mfATan2", "E",                          &
                               "the physical dimensions of the two mfArray's",&
                               "are not consistent!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( X, Y )

      call msAutoRelease(X,Y)

#endif
   end function mfATan2
!_______________________________________________________________________
!
   function mfExp( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfExp", "E",                               &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfExp", "E",                               &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfExp", "E",                               &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_DBLE

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%double => a%double
            call set_status_tempo_to_false( a )
         else
            allocate( out%double(A%shape(1),A%shape(2)) )

         end if
         out%double(:,:) = exp( A%double(:,:) )
         out%prop%symm = A%prop%symm
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => a%cmplx
            call set_status_tempo_to_false( a )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         out%cmplx(:,:) = exp( A%cmplx(:,:) )
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfExp", "E",                            &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfExp
!_______________________________________________________________________
!
   function mfExpm1( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! computes exp(x)-1, accurately for small values of real x
      ! calls DEXPRL() from SLATEC, then multiply by x

      ! if A is complex, exp(x)-1 formula is used

      ! A may be sparse, because expm1(0)=0

      integer :: status
      integer :: i, j, nrow, ncol, nnz, k

   !------ end of declarations -- execution starts hereafter  ------

      if( mf_auto_complex ) then
         call mf_save_and_disable_fpe( )
      end if

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfExpm1", "E",                             &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfExpm1", "E",                             &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( .not. mfIsSparse(A) ) then ! dense

         out%shape = A%shape
         if( A%data_type == MF_DT_DBLE ) then
            out%data_type = MF_DT_DBLE

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%double => a%double
               call set_status_tempo_to_false( a )
            else
               allocate( out%double(A%shape(1),A%shape(2)) )

            end if
            do j = 1, out%shape(2)
               do i = 1, out%shape(1)
                  out%double(i,j) = A%double(i,j) * DEXPRL( A%double(i,j) )
               end do
            end do
            out%prop%symm = A%prop%symm
         else if( A%data_type == MF_DT_CMPLX ) then
            out%data_type = MF_DT_CMPLX

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%cmplx => a%cmplx
               call set_status_tempo_to_false( a )
            else
               allocate( out%cmplx(A%shape(1),A%shape(2)) )

            end if
            out%cmplx(:,:) = exp( A%cmplx(:,:) ) - 1.0d0
         end if

      else ! sparse

         out%shape = A%shape
         nrow = A%shape(1)
         ncol = A%shape(2)
         nnz = A%j(ncol+1) - 1
         ! keeping extra room previously allocated
         k = size(A%i)
         allocate( out%i(k) )

         allocate( out%j(ncol+1) )

         if( A%data_type == MF_DT_SP_DBLE ) then
            out%data_type = MF_DT_SP_DBLE

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%a => A%a
               call set_status_tempo_to_false( A )
            else
               allocate( out%a(k) )

            end if
            do k = 1, nnz
               out%a(k) = A%a(k) * DEXPRL( A%a(k) )
            end do
            out%prop%symm = A%prop%symm
         else if( A%data_type == MF_DT_SP_CMPLX ) then
            out%data_type = MF_DT_SP_CMPLX

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%z => A%z
               call set_status_tempo_to_false( A )
            else
               allocate( out%z(k) )

            end if
            out%z(1:nnz) = exp( A%z(1:nnz) ) - 1.0d0
         end if
         out%i(1:nnz) = A%i(1:nnz)
         out%j(:) = A%j(1:ncol+1)

      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfExpm1", "E",                          &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      if( mf_auto_complex ) then
         call mf_restore_fpe( )
      end if

#endif
   end function mfExpm1
!_______________________________________________________________________
!
   function mfLog( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      if( mf_auto_complex ) then
         call mf_save_and_disable_fpe( )
      end if

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfLog", "E",                               &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfLog", "E",                               &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfLog", "E",                               &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         if( mf_auto_complex .and.                                      &
             any( A%double(:,:) < 0.0d0 ) ) then
            out%data_type = MF_DT_CMPLX
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

            out%cmplx(:,:) = log( cmplx(A%double(:,:),kind=MF_DOUBLE) )
         else
            out%data_type = MF_DT_DBLE

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%double => a%double
               call set_status_tempo_to_false( a )
            else
               allocate( out%double(A%shape(1),A%shape(2)) )

            end if
            out%double(:,:) = log( A%double(:,:) )
            out%prop%symm = A%prop%symm
         end if
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => a%cmplx
            call set_status_tempo_to_false( a )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         out%cmplx(:,:) = log( A%cmplx(:,:) )
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfLog", "E",                            &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      if( mf_auto_complex ) then
         call mf_restore_fpe( )
      end if

#endif
   end function mfLog
!_______________________________________________________________________
!
   function mfLog1p( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! computes ln( 1 + x ), accurately for small values of real x
      ! calls DLNREL() from SLATEC

      ! if A is complex, ln(1+x) formula is used

      ! A may be sparse, because log1p(0)=0

      integer :: status
      integer :: i, j, nrow, ncol, nnz, k

   !------ end of declarations -- execution starts hereafter  ------

      if( mf_auto_complex ) then
         call mf_save_and_disable_fpe( )
      end if

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfLog1p", "E",                             &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfLog1p", "E",                             &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( .not. mfIsSparse(A) ) then ! dense

         out%shape = A%shape
         if( A%data_type == MF_DT_DBLE ) then
            if( mf_auto_complex .and.                                   &
                any( A%double(:,:) < -1.0d0 ) ) then
               out%data_type = MF_DT_CMPLX
               allocate( out%cmplx(A%shape(1),A%shape(2)) )

               out%cmplx(:,:) = log( cmplx(1.0d0+A%double(:,:),kind=MF_DOUBLE) )
            else
               out%data_type = MF_DT_DBLE

               ! if one of the mfArrays is tempo and not protected,
               ! processing is done 'in place' (avoid an allocation).
               if( can_use_memory(A) ) then
                  out%double => a%double
                  call set_status_tempo_to_false( a )
               else
                  allocate( out%double(A%shape(1),A%shape(2)) )

               end if
               do j = 1, out%shape(2)
                  do i = 1, out%shape(1)
                     out%double(i,j) = DLNREL( A%double(i,j) )
                  end do
               end do
               out%prop%symm = A%prop%symm
            end if
         else if( A%data_type == MF_DT_CMPLX ) then
            out%data_type = MF_DT_CMPLX

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%cmplx => a%cmplx
               call set_status_tempo_to_false( a )
            else
               allocate( out%cmplx(A%shape(1),A%shape(2)) )

            end if
            out%cmplx(:,:) = log( 1.0d0 + A%cmplx(:,:) )
         end if

      else ! sparse

         out%shape = A%shape
         nrow = A%shape(1)
         ncol = A%shape(2)
         nnz = A%j(ncol+1) - 1
         ! keeping extra room previously allocated
         k = size(A%i)
         allocate( out%i(k) )

         allocate( out%j(ncol+1) )

         if( A%data_type == MF_DT_SP_DBLE ) then
            if( mf_auto_complex .and.                                   &
                any( A%a(:) < -1.0d0 ) ) then
               out%data_type = MF_DT_SP_CMPLX
               allocate( out%z(k) )

               out%z(1:nnz) = log( cmplx(1.0d0+A%a(1:nnz),kind=MF_DOUBLE) )
            else
               out%data_type = MF_DT_SP_DBLE

               ! if one of the mfArrays is tempo and not protected,
               ! processing is done 'in place' (avoid an allocation).
               if( can_use_memory(A) ) then
                  out%a => A%a
                  call set_status_tempo_to_false( A )
               else
                  allocate( out%a(k) )

               end if
               do k = 1, nnz
                  out%a(k) = DLNREL( A%a(k) )
               end do
               out%prop%symm = A%prop%symm
            end if
         else if( A%data_type == MF_DT_SP_CMPLX ) then
            out%data_type = MF_DT_SP_CMPLX

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%z => A%z
               call set_status_tempo_to_false( A )
            else
               allocate( out%z(k) )

            end if
            out%z(1:nnz) = log( 1.0d0 + A%z(1:nnz) )
         end if
         out%i(1:nnz) = A%i(1:nnz)
         out%j(:) = A%j(1:ncol+1)

      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfLog1p", "E",                          &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      if( mf_auto_complex ) then
         call mf_restore_fpe( )
      end if

#endif
   end function mfLog1p
!_______________________________________________________________________
!
   function mfLog10( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      if( mf_auto_complex ) then
         call mf_save_and_disable_fpe( )
      end if

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfLog10", "E",                             &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfLog10", "E",                             &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfLog10", "E",                             &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         if( mf_auto_complex .and.                                      &
             any( A%double(:,:) < 0.0d0 ) ) then
            out%data_type = MF_DT_CMPLX
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

            ! in Fortran, 'log10' is not defined for complex arg.
            out%cmplx(:,:) = log( cmplx(A%double(:,:),kind=MF_DOUBLE) ) &
                             / MF_LN_10
         else
            out%data_type = MF_DT_DBLE

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%double => a%double
               call set_status_tempo_to_false( a )
            else
               allocate( out%double(A%shape(1),A%shape(2)) )

            end if
            out%double(:,:) = log10( A%double(:,:) )
            out%prop%symm = A%prop%symm
         end if
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => a%cmplx
            call set_status_tempo_to_false( a )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         ! in Fortran, 'log10' is not defined for complex arg.
         out%cmplx(:,:) = log( A%cmplx(:,:) ) / MF_LN_10
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfLog10", "E",                          &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      if( mf_auto_complex ) then
         call mf_restore_fpe( )
      end if

#endif
   end function mfLog10
!_______________________________________________________________________
!
   function mfLog2( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      if( mf_auto_complex ) then
         call mf_save_and_disable_fpe( )
      end if

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfLog2", "E",                              &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfLog2", "E",                              &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfLog2", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         if( mf_auto_complex .and.                                      &
             any( A%double(:,:) < 0.0d0 ) ) then
            out%data_type = MF_DT_CMPLX
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

            out%cmplx(:,:) = log( cmplx(A%double(:,:),kind=MF_DOUBLE) ) &
                             / log(2.0d0)
         else
            out%data_type = MF_DT_DBLE

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%double => a%double
               call set_status_tempo_to_false( a )
            else
               allocate( out%double(A%shape(1),A%shape(2)) )

            end if
            out%double(:,:) = log( A%double(:,:) ) / log(2.0d0)
            out%prop%symm = A%prop%symm
         end if
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => a%cmplx
            call set_status_tempo_to_false( a )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         out%cmplx(:,:) = log( A%cmplx(:,:) ) / log(2.0d0)
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfLog2", "E",                           &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      if( mf_auto_complex ) then
         call mf_restore_fpe( )
      end if

#endif
   end function mfLog2
!_______________________________________________________________________
!
   subroutine msLog2( out, A )

      type(mfArray) :: A
      type(mf_Out) :: out
      !------ API end ------

#ifdef _DEVLP
      type(mfArray), pointer :: m, e
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      ! we must have two output arguments
      if( out%n /= 2 ) then
         call PrintMessage( "msLog2", "E",                              &
                            "two output args required!",                &
                            "syntax is : call msLog2 ( mfOut(m,e), a )" )
         go to 99
      end if

      ! internal check for all mfOut args
      if( .not. args_mfout_ok( out, A ) ) then
         call PrintMessage( "msLog2", "E",                              &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      m => out%ptr1
      e => out%ptr2
      call msSilentRelease( m, e )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( A%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "msLog2", "E",                              &
                            "real array required!" )
         go to 99
      end if

      if( radix(1.0d0) /= 2 ) then
         call PrintMessage( "msLog2", "E",                              &
                            "radix must be equal to 2!" )
         go to 99
      end if

      m%data_type = MF_DT_DBLE
      e%data_type = MF_DT_DBLE
      m%shape = A%shape
      e%shape = A%shape
      allocate( m%double(A%shape(1),A%shape(2)) )

      allocate( e%double(A%shape(1),A%shape(2)) )

      m%double(:,:) = fraction(A%double(:,:))
      e%double(:,:) = exponent(A%double(:,:))

      m%prop%symm = A%prop%symm
      e%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "msLog2", "E",                           &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end subroutine msLog2
!_______________________________________________________________________
!
   function mfPow10( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( A%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfPow10", "E",                             &
                            "real array required!" )
         go to 99
      end if

      out%data_type = MF_DT_DBLE
      out%shape = A%shape

      ! if one of the mfArrays is tempo and not protected,
      ! processing is done 'in place' (avoid an allocation).
      if( can_use_memory(A) ) then
         out%double => a%double
         call set_status_tempo_to_false( a )
      else
         allocate( out%double(A%shape(1),A%shape(2)) )

      end if
      out%double(:,:) = 10.0d0 ** A%double(:,:)

      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfPow10", "E",                          &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfPow10
!_______________________________________________________________________
!
   function mfPow2_1( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( A%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfPow2", "E",                              &
                            "real array required!" )
         go to 99
      end if

      out%data_type = MF_DT_DBLE
      out%shape = A%shape

      ! if one of the mfArrays is tempo and not protected,
      ! processing is done 'in place' (avoid an allocation).
      if( can_use_memory(A) ) then
         out%double => a%double
         call set_status_tempo_to_false( a )
      else
         allocate( out%double(A%shape(1),A%shape(2)) )

      end if
      out%double(:,:) = 2.0d0 ** A%double(:,:)

      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfPow2", "E",                           &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfPow2_1
!_______________________________________________________________________
!
   function mfPow2_2( m, e ) result( out )

      type(mfArray) :: m, e
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( m, e )

      if( m%data_type /= MF_DT_DBLE .or. e%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfPow2", "E",                              &
                            "'m' and 'e' must be real arrays!" )
         go to 99
      end if

      if( any(m%shape/=e%shape) ) then
         call PrintMessage( "mfPow2", "E",                              &
                            "a and b must have the same shape" )
         go to 99
      end if

      if( mfIsEmpty(m) .or. mfIsEmpty(e) ) then
         go to 99
      end if

      if( radix(1.0d0) /= 2 ) then
         call PrintMessage( "mfPow2", "E",                              &
                            "radix must be equal to 2!" )
         go to 99
      end if

      if( any( dble(int(e%double(:,:))) /= e%double(:,:) ) ) then
         call PrintMessage( "mfPow2", "E",                              &
                            "'e' must contain integer numbers!" )
         go to 99
      end if

      out%data_type = MF_DT_DBLE
      out%shape = m%shape
      allocate( out%double(m%shape(1),m%shape(2)) )

      out%double(:,:) = set_exponent(m%double(:,:),int(e%double(:,:)))

      if( m%prop%symm == TRUE .and. e%prop%symm == TRUE ) then
         out%prop%symm = TRUE
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( m%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfPow2", "E",                           &
                               "the physical unit of the mfArray 'm'",  &
                               "must be dimensionless!" )
            go to 99
         end if
         call verif_adim( e%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfPow2", "E",                           &
                               "the physical unit of the mfArray 'e'",  &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( m, e )

      call msAutoRelease( m, e )

#endif
   end function mfPow2_2
!_______________________________________________________________________
!
   function mfSqrt( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: i, status, nrow, ncol, nnz, k

   !------ end of declarations -- execution starts hereafter  ------

      if( mf_auto_complex ) then
         call mf_save_and_disable_fpe( )
      end if

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfSqrt", "E",                              &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfSqrt", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( .not. mfIsSparse(A) ) then ! dense

         out%shape = A%shape
         if( A%data_type == MF_DT_DBLE ) then
            if( mf_auto_complex .and.                                   &
                any( A%double(:,:) < 0.0d0 ) ) then
               out%data_type = MF_DT_CMPLX
               allocate( out%cmplx(A%shape(1),A%shape(2)) )

               out%cmplx(:,:) = sqrt( cmplx(A%double(:,:),kind=MF_DOUBLE) )
            else
               out%data_type = MF_DT_DBLE

               ! if one of the mfArrays is tempo and not protected,
               ! processing is done 'in place' (avoid an allocation).
               if( can_use_memory(A) ) then
                  out%double => A%double
                  call set_status_tempo_to_false( A )
               else
                  allocate( out%double(A%shape(1),A%shape(2)) )

               end if
               out%double(:,:) = sqrt( A%double(:,:) )
               out%prop%symm = A%prop%symm
            end if
         else if( A%data_type == MF_DT_CMPLX ) then
            out%data_type = MF_DT_CMPLX

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%cmplx => A%cmplx
               call set_status_tempo_to_false( A )
            else
               allocate( out%cmplx(A%shape(1),A%shape(2)) )

            end if
            out%cmplx(:,:) = sqrt( A%cmplx(:,:) )
         end if

      else ! sparse

         out%shape = A%shape
         nrow = A%shape(1)
         ncol = A%shape(2)
         nnz = A%j(ncol+1) - 1
         ! keeping extra room previously allocated
         k = size(A%i)
         allocate( out%i(k) )

         allocate( out%j(ncol+1) )

         if( A%data_type == MF_DT_SP_DBLE ) then
            if( mf_auto_complex .and.                                   &
                any( A%a(:) < 0.0d0 ) ) then
               out%data_type = MF_DT_SP_CMPLX
               allocate( out%z(k) )

               out%z(1:nnz) = sqrt( cmplx(A%a(1:nnz),kind=MF_DOUBLE) )
            else
               out%data_type = MF_DT_SP_DBLE

               ! if one of the mfArrays is tempo and not protected,
               ! processing is done 'in place' (avoid an allocation).
               if( can_use_memory(A) ) then
                  out%a => A%a
                  call set_status_tempo_to_false( A )
               else
                  allocate( out%a(k) )

               end if
               out%a(1:nnz) = sqrt( A%a(1:nnz) )
               out%prop%symm = A%prop%symm
            end if
         else if( A%data_type == MF_DT_SP_CMPLX ) then
            out%data_type = MF_DT_SP_CMPLX

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%z => A%z
               call set_status_tempo_to_false( A )
            else
               allocate( out%z(k) )

            end if
            out%z(1:nnz) = sqrt( A%z(1:nnz) )
         end if
         out%i(1:nnz) = A%i(1:nnz)
         out%j(:) = A%j(1:ncol+1)

      end if

      if( mf_phys_units ) then
         do i = 1, num_base_units
            call rational_div( A%units(i), 2,                           &
                               out%units(i), status )
            if( status == 1 ) then
               call PrintMessage( "mfSqrt", "E",                        &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               go to 99
            else if( status == -1 ) then
               call PrintMessage( "mfSqrt", "E",                        &
                                  "in processing physical units:",      &
                                  "Please report this bug to: Edouard.Canot@univ-rennes.fr" )
               go to 99
            end if
         end do
      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      if( mf_auto_complex ) then
         call mf_restore_fpe( )
      end if

#endif
   end function mfSqrt
!_______________________________________________________________________
!
   function mfAbs( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: nrow, ncol, nnz, k

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfAbs", "E",                               &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfAbs", "E",                               &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( .not. mfIsSparse(A) ) then ! dense

         out%data_type = MF_DT_DBLE
         out%shape = A%shape
         if( A%data_type == MF_DT_DBLE ) then

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%double => A%double
               call set_status_tempo_to_false( A )
            else
               allocate( out%double(A%shape(1),A%shape(2)) )

            end if
            out%double(:,:) = abs( A%double(:,:) )
         else if( A%data_type == MF_DT_CMPLX ) then
            allocate( out%double(A%shape(1),A%shape(2)) )

            out%double(:,:) = abs( A%cmplx(:,:) )
         end if

      else ! sparse

         out%data_type = MF_DT_SP_DBLE
         out%shape = A%shape
         nrow = A%shape(1)
         ncol = A%shape(2)
         nnz = A%j(ncol+1) - 1
         ! keeping extra room previously allocated
         k = size(A%i)
         allocate( out%i(k) )

         allocate( out%j(ncol+1) )

         if( A%data_type == MF_DT_SP_DBLE ) then

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%a => A%a
               call set_status_tempo_to_false( A )
            else
               allocate( out%a(k) )

            end if
            out%a(1:nnz) = abs( A%a(1:nnz) )
         else if( A%data_type == MF_DT_SP_CMPLX ) then
            allocate( out%a(k) )

            out%a(1:nnz) = abs( A%z(1:nnz) )
         end if
         out%i(1:nnz) = A%i(1:nnz)
         out%j(:) = A%j(1:ncol+1)

      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu
      if( A%prop%symm == TRUE ) then
         out%prop%symm = TRUE
      end if

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfAbs
!_______________________________________________________________________
!
   function mfAngle( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: status, nrow, ncol, nnz, k

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfAngle", "E",                             &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfAngle", "E",                             &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( .not. mfIsSparse(A) ) then ! dense

         out%data_type = MF_DT_DBLE
         out%shape = A%shape
         if( A%data_type == MF_DT_DBLE ) then

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%double => A%double
               call set_status_tempo_to_false( A )
            else
               allocate( out%double(A%shape(1),A%shape(2)) )

            end if
            out%double(:,:) = atan2( 0.0d0,                             &
                                     A%double(:,:) )
         else if( A%data_type == MF_DT_CMPLX ) then
            allocate( out%double(A%shape(1),A%shape(2)) )

            out%double(:,:) = atan2( aimag(A%cmplx(:,:)),               &
                                     real(A%cmplx(:,:)) )
         end if

      else ! sparse

         out%data_type = MF_DT_SP_DBLE
         out%shape = A%shape
         nrow = A%shape(1)
         ncol = A%shape(2)
         nnz = A%j(ncol+1) - 1
         ! keeping extra room previously allocated
         k = size(A%i)
         allocate( out%i(k) )

         allocate( out%j(ncol+1) )

         if( A%data_type == MF_DT_SP_DBLE ) then

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%a => A%a
               call set_status_tempo_to_false( A )
            else
               allocate( out%a(k) )

            end if
            out%a(1:nnz) = atan2( 0.0d0,                                &
                                  A%a(1:nnz) )
         else if( A%data_type == MF_DT_SP_CMPLX ) then
            allocate( out%a(k) )

            out%a(1:nnz) = atan2( aimag(A%z(1:nnz)),                    &
                                  real(A%z(1:nnz)) )
         end if
         out%i(1:nnz) = A%i(1:nnz)
         out%j(:) = A%j(1:ncol+1)

      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu
      if( A%prop%symm == TRUE ) then
         out%prop%symm = TRUE
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfAngle", "E",                          &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfAngle
!_______________________________________________________________________
!
   function mfConj( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: nrow, ncol, nnz, k

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfConj", "E",                              &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfConj", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( .not. mfIsSparse(A) ) then ! dense

         out%data_type = MF_DT_CMPLX
         out%shape = A%shape
         if( A%data_type == MF_DT_DBLE ) then
            allocate( out%cmplx(out%shape(1),out%shape(2)) )

            out%cmplx(:,:) = A%double(:,:)
         else if( A%data_type == MF_DT_CMPLX ) then

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%cmplx => A%cmplx
               call set_status_tempo_to_false( A )
            else
               allocate( out%cmplx(A%shape(1),A%shape(2)) )

            end if
            out%cmplx(:,:) = conjg( A%cmplx(:,:) )
         end if

      else ! sparse

         out%data_type = MF_DT_SP_CMPLX
         out%shape = A%shape
         nrow = A%shape(1)
         ncol = A%shape(2)
         nnz = A%j(ncol+1) - 1
         ! keeping extra room previously allocated
         k = size(A%i)
         allocate( out%i(k) )

         allocate( out%j(ncol+1) )

         if( A%data_type == MF_DT_SP_DBLE ) then
            allocate( out%z(k) )

            out%z(1:nnz) = A%a(1:nnz)
         else if( A%data_type == MF_DT_SP_CMPLX ) then

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%z => A%z
               call set_status_tempo_to_false( A )
            else
               allocate( out%z(k) )

            end if
            out%z(1:nnz) = conjg( A%z(1:nnz) )
         end if
         out%i(1:nnz) = A%i(1:nnz)
         out%j(:) = A%j(1:ncol+1)

      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu
      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfConj
!_______________________________________________________________________
!
   function mfReal( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: nrow, ncol, nnz, k

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfReal", "E",                              &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfReal", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( .not. mfIsSparse(A) ) then ! dense

         if( A%data_type == MF_DT_DBLE ) then
            if( A%level_protected == 1 ) then
               call msAssign( out, A )
            else
               out = A
            end if
            out%status_temporary = .true.
            go to 99
         else if( A%data_type == MF_DT_CMPLX ) then
            out%data_type = MF_DT_DBLE
            out%shape = A%shape
            allocate( out%double(out%shape(1),out%shape(2)) )

            out%double(:,:) = real( A%cmplx(:,:) )
         end if

      else ! sparse

         if( A%data_type == MF_DT_SP_DBLE ) then
            if( A%level_protected == 1 ) then
               call msAssign( out, A )
            else
               out = A
            end if
            out%status_temporary = .true.
            go to 99
         else if( A%data_type == MF_DT_SP_CMPLX ) then
            out%data_type = MF_DT_SP_DBLE
            out%shape = A%shape
            nrow = A%shape(1)
            ncol = A%shape(2)
            nnz = A%j(ncol+1) - 1
            ! keeping extra room previously allocated
            k = size(A%i)
            allocate( out%a(k) )

            allocate( out%i(k) )

            allocate( out%j(ncol+1) )

            out%a(1:nnz) = real( A%z(1:nnz) )
            out%i(1:nnz) = A%i(1:nnz)
            out%j(:) = A%j(1:ncol+1)

         end if

      end if

      out%prop%tril = UNKNOWN
      out%prop%triu = UNKNOWN
      if( A%prop%symm == TRUE ) then
         out%prop%symm = TRUE
      end if

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfReal
!_______________________________________________________________________
!
   function mfImag( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: nrow, ncol, nnz, k

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfImag", "E",                              &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfImag", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( .not. mfIsSparse(A) ) then ! dense

         out%data_type = MF_DT_DBLE
         out%shape = A%shape
         if( A%data_type == MF_DT_DBLE ) then

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%double => A%double
               call set_status_tempo_to_false( A )
            else
               allocate( out%double(A%shape(1),A%shape(2)) )

            end if
            out%double(:,:) = 0.0d0
            out%prop%symm = TRUE
         else if( A%data_type == MF_DT_CMPLX ) then
            allocate( out%double(out%shape(1),out%shape(2)) )

            out%double(:,:) = aimag( A%cmplx(:,:) )
            if( A%prop%symm == TRUE ) then
               out%prop%symm = TRUE
            end if
         end if

      else ! sparse

         out%data_type = MF_DT_SP_DBLE
         out%shape = A%shape
         nrow = A%shape(1)
         ncol = A%shape(2)
         nnz = A%j(ncol+1) - 1
         ! keeping extra room previously allocated
         k = size(A%i)
         allocate( out%i(k) )

         allocate( out%j(ncol+1) )

         if( A%data_type == MF_DT_SP_DBLE ) then

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%a => A%a
               call set_status_tempo_to_false( A )
            else
               allocate( out%a(k) )

            end if
            out%a(1:nnz) = 0.0d0
            out%prop%symm = TRUE
         else if( A%data_type == MF_DT_SP_CMPLX ) then
            allocate( out%a(k) )

            out%a(1:nnz) = aimag( A%z(1:nnz) )
            if( A%prop%symm == TRUE ) then
               out%prop%symm = TRUE
            end if
         end if
         out%i(1:nnz) = A%i(1:nnz)
         out%j(:) = A%j(1:ncol+1)

      end if

      out%prop%tril = UNKNOWN
      out%prop%triu = UNKNOWN
      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfImag
!_______________________________________________________________________
!
   function mfComplex_one_arg( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: nrow, ncol, nnz, k

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfComplex", "E",                           &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfComplex", "E",                           &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( .not. mfIsSparse(A) ) then ! dense

         out%data_type = MF_DT_CMPLX
         out%shape = A%shape
         if( A%data_type == MF_DT_DBLE ) then
            allocate( out%cmplx(out%shape(1),out%shape(2)) )

            out%cmplx(:,:) = A%double(:,:)
         else if( A%data_type == MF_DT_CMPLX ) then

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%cmplx => A%cmplx
               call set_status_tempo_to_false( A )
            else
               allocate( out%cmplx(A%shape(1),A%shape(2)) )

            end if
            out%cmplx(:,:) = A%cmplx(:,:)
         end if

      else ! sparse

         out%data_type = MF_DT_SP_CMPLX
         out%shape = A%shape
         nrow = A%shape(1)
         ncol = A%shape(2)
         nnz = A%j(ncol+1) - 1
         ! keeping extra room previously allocated
         k = size(A%i)
         allocate( out%i(k) )

         allocate( out%j(ncol+1) )

         if( A%data_type == MF_DT_SP_DBLE ) then
            allocate( out%z(k) )

            out%z(1:nnz) = A%a(1:nnz)
         else if( A%data_type == MF_DT_SP_CMPLX ) then

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%z => A%z
               call set_status_tempo_to_false( A )
            else
               allocate( out%z(k) )

            end if
            out%z(1:nnz) = A%z(1:nnz)
         end if
         out%i(1:nnz) = A%i(1:nnz)
         out%j(:) = A%j(1:ncol+1)

      end if

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfComplex_one_arg
!_______________________________________________________________________
!
   function mfComplex_two_arg( A, B ) result( out )

      type(mfArray) :: A, B
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: status, nrow, ncol, nnz, k, nzmax, nzaplb, jerr, nnz2
      logical :: sorted_version

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A, B )

      if( .not. all( A%shape == B%shape ) ) then
         call PrintMessage( "mfComplex", "E",                           &
                            "args must have the same shape!" )
         go to 99
      end if

      if( mfIsEmpty(A) .and. mfIsEmpty(B) ) then
         go to 99
      end if

      if( .not. mfIsSparse(A) ) then ! dense

         if( A%data_type /= MF_DT_DBLE .or. B%data_type /= MF_DT_DBLE ) then
            call PrintMessage( "mfComplex", "E",                           &
                               "arg 'a' and 'b' must be both of real type!" )
            go to 99
         end if

         out%data_type = MF_DT_CMPLX
         out%shape = A%shape
         allocate( out%cmplx(out%shape(1),out%shape(2)) )

         out%cmplx(:,:) = A%double(:,:) + (0.0d0,1.0d0)*B%double(:,:)

      else ! sparse

         if( A%data_type /= MF_DT_SP_DBLE .or. B%data_type /= MF_DT_SP_DBLE ) then
            call PrintMessage( "mfComplex", "E",                           &
                               "arg 'a' and 'b' must be both of real type!" )
            go to 99
         end if

         out%data_type = MF_DT_SP_CMPLX
         out%shape = A%shape
         nrow = A%shape(1)
         ncol = A%shape(2)
         ! Do A and B have the same structure ?
         nnz = A%j(ncol+1) - 1
         nnz2 = B%j(ncol+1) - 1
         if( nnz2 == nnz ) then

            if( all(A%i(1:nnz) == B%i(1:nnz2)) .or.                        &
                all(A%j(1:ncol+1) == B%j(1:ncol+1)) ) then

               ! keeping extra room previously allocated
               k = size(A%i)
               allocate( out%z(k) )

               allocate( out%i(k) )

               allocate( out%j(ncol+1) )

               out%z(1:nnz) = A%a(1:nnz) + (0.0d0,1.0d0)*B%a(1:nnz)
               out%i(1:nnz) = A%i(1:nnz)
               out%j(:) = A%j(1:ncol+1)

               go to 20

            else

               go to 10

            end if

         else

            go to 10

         end if

 10      continue

         ! cannot use the high level operators of mod_ops

         ! nzmax choice: at least nnzaplb, but preserve extra rooms
         ! from initial sparse arrays.
         call nnzaplb( nrow, ncol, A%i, A%j, B%i, B%j, nzaplb )
         nzmax = max( max(mfNzmax(a),mfNzmax(b)), nzaplb )
         call msAssign( out, mfSpAlloc(nrow,ncol,nzmax,kind="complex") )
         ! check row_sorted strategy
         if( mfIsRowSorted( A ) .and. mfIsRowSorted( B ) ) then
            sorted_version = .true.
         else
            if( MF_SP_AUTO_ROW_SORTED ) then
               if( .not. A%status_temporary ) then
                  if( .not. mfIsRowSorted( A ) ) then
                     call msRowSort( A )
                     call PrintMessage( "mfComplex", "I",               &
                                        "sparse matrix 'A' have been sorted!",&
                                        "(If you don't want that MUESLI sort itself,",&
                                        "you can use the 'msSetAutoRowSorted' routine to",&
                                        "change this behavior.)" )
                  end if
               end if
               if( .not. B%status_temporary ) then
                  if( .not. mfIsRowSorted( B ) ) then
                     call msRowSort( B )
                     call PrintMessage( "mfComplex", "I",               &
                                        "sparse matrix 'B' have been sorted!",&
                                        "(If you don't want that MUESLI sort itself,",&
                                        "you can use the 'msSetAutoRowSorted' routine to",&
                                        "change this behavior.)" )
                  end if
               end if
               if( mfIsRowSorted( A ) .and. mfIsRowSorted( B ) ) then
                  sorted_version = .true.
               else
                  sorted_version = .false.
               end if
            else ! row sorted under the user control
               sorted_version = .false.
               call PrintMessage( "mfComplex", "W",                     &
                                  "at least one sparse matrix is not row sorted!",&
                                  "(sorting it permits to use a better algo.)" )
            end if
         end if
         if( sorted_version ) then
            call aplib1( nrow, ncol, A%a, A%i, A%j, B%a, B%i, B%j,      &
                         out%z, out%i, out%j, nzmax, jerr )
         else
            call aplib( nrow, ncol, A%a, A%i, A%j, B%a, B%i, B%j,       &
                        out%z, out%i, out%j, nzmax, jerr )
         end if
         if( jerr /= 0 ) then
            write(STDERR,*) "(MUESLI mfComplex:) internal error: aplib fails!"
            mf_message_displayed = .true.
            call muesli_trace( pause="yes" )
            stop
         end if
         if( sorted_version ) then
            out%row_sorted = TRUE
         else
            out%row_sorted = UNKNOWN
         end if

 20      continue

      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, B%units, status )
         if( status /= 0 ) then
            call PrintMessage( "mfComplex", "E",                        &
                               "the physical dimensions of the two mfArray's",&
                               "are not consistent!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A, B )

      call msAutoRelease( A, B )

#endif
   end function mfComplex_two_arg
!_______________________________________________________________________
!
   function mfCeil( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: ncol, nnz

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( .not. mfIsReal(A) ) then
         call PrintMessage( "mfCeil", "E",                              &
                            "real array required!" )
         go to 99
      end if

      out%data_type = A%data_type
      out%shape = A%shape

      if( A%data_type == MF_DT_DBLE ) then
         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%double => A%double
            call set_status_tempo_to_false( A )
         else
            allocate( out%double(A%shape(1),A%shape(2)) )

         end if
         out%double(:,:) = ceiling( A%double(:,:) )
      else if( A%data_type == MF_DT_SP_DBLE ) then
         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%i => A%i
            out%j => A%j
            out%a => A%a
            call set_status_tempo_to_false( A )
         else
            ncol = A%shape(2)
            nnz = A%j(ncol+1) - 1
            allocate( out%a(nnz) )

         end if
         out%a(:) = ceiling( A%a(:) )
      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu
      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfCeil
!_______________________________________________________________________
!
   function mfHypot( A, B ) result( out )

      type(mfArray) :: A, B
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! Computes hypotenuse without overflow or underflow

      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A, B )

      if( mfIsEmpty(A) .or. mfIsEmpty(B) ) then
         go to 99
      end if

      if( A%data_type /= MF_DT_DBLE .or. B%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfHypot", "E",                             &
                            "real arrays required!" )
         go to 99
      end if

      if( any(A%shape/=B%shape) ) then
         call PrintMessage( "mfHypot", "E",                             &
                            "a and b must have the same shape" )
         go to 99
      end if

      out%data_type = MF_DT_DBLE
      out%shape = A%shape

      ! if one of the mfArrays is tempo and not protected,
      ! processing is done 'in place' (avoid an allocation).
      if( can_use_memory(A) ) then
         out%double => A%double
         call set_status_tempo_to_false( A )
      else
         allocate( out%double(A%shape(1),A%shape(2)) )

      end if
      do j = 1, A%shape(2)
         do i = 1, A%shape(1)
            out%double(i,j) = hypot( A%double(i,j), B%double(i,j) )
         end do
      end do

      if( A%prop%symm == TRUE .and. B%prop%symm == TRUE ) then
         out%prop%symm = TRUE
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, b%units, status )
         if( status /= 0 ) then
            call PrintMessage( "mfHypot", "E",                          &
                               "the physical dimensions of the two mfArray's",&
                               "are not consistent!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfHypot
#if defined _INTEL_IFC | defined _GNU_GFC
   ! hypot() becomes an intrinsic function only with Fortran 2008
!_______________________________________________________________________
!
   function hypot_real( x, y ) result( out )

      ! Computes hypotenuse without overflow or underflow
      ! (after GSL)

      real(kind=MF_DOUBLE) :: x, y
      real(kind=MF_DOUBLE) :: out
      !------ API end ------

#ifdef _DEVLP
      real(kind=MF_DOUBLE) :: xabs, yabs, valmin, valmax, u2

   !------ end of declarations -- execution starts hereafter  ------

      xabs = abs(x)
      yabs = abs(y)
      if( xabs < yabs ) then
         valmin = xabs
         valmax = yabs
      else
         valmin = yabs
         valmax = xabs
      end if
      if( valmin == 0.0d0 ) then
         out = valmax
      else
         ! here, valmax cannot be null.
         u2 = (valmin/valmax)**2
         if( u2 < MF_EPS ) then
            out = valmax
         else
            out = valmax*sqrt( 1.0d0 + u2 )
         end if
      end if

#endif
   end function hypot_real
!_______________________________________________________________________
!
   function hypot_cmplx( x, y ) result( out )

      ! Computes hypotenuse without overflow or underflow
      !
      ! complex version -- here, we follow the Matlab convention:
      !   for complex numbers: hypot(x,y) = sqrt(|A|^2+|B|^2)

      complex(kind=MF_DOUBLE) :: x, y
      real(kind=MF_DOUBLE) :: out
      !------ API end ------

#ifdef _DEVLP
      real(kind=MF_DOUBLE) :: xabs, yabs, valmin, valmax, u2

   !------ end of declarations -- execution starts hereafter  ------

      xabs = abs(x)
      yabs = abs(y)
      if( xabs < yabs ) then
         valmin = xabs
         valmax = yabs
      else
         valmin = yabs
         valmax = xabs
      end if
      if( valmin == 0.0d0 ) then
         out = valmax
      else
         ! here, valmax cannot be null.
         u2 = (valmin/valmax)**2
         if( u2 < MF_EPS ) then
            out = valmax
         else
            out = valmax*sqrt( 1.0d0 + u2 )
         end if
      end if

#endif
   end function hypot_cmplx
#endif
!_______________________________________________________________________
!
   function mfFix( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: ncol, nnz

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( .not. mfIsReal(A) ) then
         call PrintMessage( "mfFix", "E",                               &
                            "real array required!" )
         go to 99
      end if

      out%data_type = A%data_type
      out%shape = A%shape

      if( A%data_type == MF_DT_DBLE ) then
         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%double => A%double
            call set_status_tempo_to_false( A )
         else
            allocate( out%double(A%shape(1),A%shape(2)) )

         end if
         out%double(:,:) = 1.0d0
         out%double(:,:) = sign( out%double(:,:), A%double(:,:) )
         out%double(:,:) = out%double(:,:)*int(abs(A%double(:,:)))
      else if( A%data_type == MF_DT_SP_DBLE ) then
         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%i => A%i
            out%j => A%j
            out%a => A%a
            call set_status_tempo_to_false( A )
         else
            ncol = A%shape(2)
            nnz = A%j(ncol+1) - 1
            allocate( out%a(nnz) )

         end if
         out%a(:) = 1.0d0
         out%a(:) = sign( out%a(:), A%a(:) )
         out%a(:) = out%a(:)*int(abs(A%a(:)))
      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu
      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfFix
!_______________________________________________________________________
!
   function mfFloor( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: ncol, nnz

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( .not. mfIsReal(A) ) then
         call PrintMessage( "mfFloor", "E",                             &
                            "real array required!" )
         go to 99
      end if

      out%data_type = A%data_type
      out%shape = A%shape

      if( A%data_type == MF_DT_DBLE ) then
         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%double => A%double
            call set_status_tempo_to_false( A )
         else
            allocate( out%double(A%shape(1),A%shape(2)) )

         end if
         out%double(:,:) = floor( A%double(:,:) )
      else if( A%data_type == MF_DT_SP_DBLE ) then
         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%i => A%i
            out%j => A%j
            out%a => A%a
            call set_status_tempo_to_false( A )
         else
            ncol = A%shape(2)
            nnz = A%j(ncol+1) - 1
            allocate( out%a(nnz) )

         end if
         out%a(:) = floor( A%a(:) )
      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu
      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfFloor
!_______________________________________________________________________
!
   function mfMod( A, B ) result( out )

      type(mfArray) :: A, B
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A, B )

      if( any(A%shape/=B%shape) ) then
         call PrintMessage( "mfMod", "E",                               &
                            "A and B must have the same shape" )
         go to 99
      end if

      if( A%data_type /= MF_DT_DBLE .or. B%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfMod", "E",                               &
                            "A and B must be real array!" )
         go to 99
      end if

      if( mfIsEmpty(A) .or. mfIsEmpty(B) ) then
         go to 99
      end if

      out%data_type = MF_DT_DBLE
      out%shape = A%shape

      ! if one of the mfArrays is tempo and not protected,
      ! processing is done 'in place' (avoid an allocation).
      if( can_use_memory(A) ) then
         out%double => A%double
         call set_status_tempo_to_false( A )
      else if( can_use_memory(B) ) then
         out%double => B%double
         call set_status_tempo_to_false( B )
      else
         allocate( out%double(A%shape(1),A%shape(2)) )

      end if
      out%double(:,:) = modulo( A%double(:,:), B%double(:,:) )

      if( A%prop%symm == TRUE .and. B%prop%symm == TRUE ) then
         out%prop%symm = TRUE
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, B%units, status )
         if( status /= 0 ) then
            call PrintMessage( "mfMod", "E",                            &
                               "the physical dimensions of the two mfArray's",&
                               "are not consistent!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A, B )

      call msAutoRelease( A, B )

#endif
   end function mfMod
!_______________________________________________________________________
!
   function mfRem( A, B ) result( out )

      type(mfArray) :: A, B
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A, B )

      if( any(A%shape/=B%shape) ) then
         call PrintMessage( "mfRem", "E",                               &
                            "A and B must have the same shape" )
         go to 99
      end if

      if( A%data_type /= MF_DT_DBLE .or. B%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfRem", "E",                               &
                            "A and B must be real array!" )
         go to 99
      end if

      if( mfIsEmpty(A) .or. mfIsEmpty(B) ) then
         go to 99
      end if

      out%data_type = MF_DT_DBLE
      out%shape = A%shape

      ! if one of the mfArrays is tempo and not protected,
      ! processing is done 'in place' (avoid an allocation).
      if( can_use_memory(A) ) then
         out%double => A%double
         call set_status_tempo_to_false( A )
      else if( can_use_memory(B) ) then
         out%double => B%double
         call set_status_tempo_to_false( B )
      else
         allocate( out%double(A%shape(1),A%shape(2)) )

      end if
      out%double(:,:) = mod( A%double(:,:), B%double(:,:) )

      if( A%prop%symm == TRUE .and. B%prop%symm == TRUE ) then
         out%prop%symm = TRUE
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, b%units, status )
         if( status /= 0 ) then
            call PrintMessage( "mfRem", "E",                            &
                               "the physical dimensions of the two mfArray's",&
                               "are not consistent!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A, B )

      call msAutoRelease( A, B )

#endif
   end function mfRem
!_______________________________________________________________________
!
   function mfRound( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      integer :: ncol, nnz

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( .not. mfIsReal(A) ) then
         call PrintMessage( "mfRound", "E",                             &
                            "real array required!" )
         go to 99
      end if

      out%data_type = A%data_type
      out%shape = A%shape

      if( A%data_type == MF_DT_DBLE ) then
         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%double => A%double
            call set_status_tempo_to_false( A )
         else
            allocate( out%double(A%shape(1),A%shape(2)) )

         end if
         out%double(:,:) = anint( A%double(:,:) )
      else if( A%data_type == MF_DT_SP_DBLE ) then
         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%i => A%i
            out%j => A%j
            out%a => A%a
            call set_status_tempo_to_false( A )
         else
            ncol = A%shape(2)
            nnz = A%j(ncol+1) - 1
            allocate( out%a(nnz) )

         end if
         out%a(:) = anint( A%a(:) )
      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu
      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfRound
!_______________________________________________________________________
!
   ! additional trigonometric functions (not very usual)
!_______________________________________________________________________
!
   function mfCot( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! cotangent

      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfCot", "E",                               &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfCot", "E",                               &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfCot", "E",                               &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_DBLE

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%double => A%double
            call set_status_tempo_to_false( A )
         else
            allocate( out%double(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%double(i,j) = 1.0d0 / tan( A%double(i,j) )
            end do
         end do
         out%prop%symm = A%prop%symm
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => A%cmplx
            call set_status_tempo_to_false( A )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%cmplx(i,j) = (1.0d0,0.0d0) /                         &
                                mf_gsl_complex_tan( A%cmplx(i,j) )
            end do
         end do
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfCot", "E",                            &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfCot
!_______________________________________________________________________
!
   function mfACot( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! arccotangent

      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfACot", "E",                              &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfACot", "E",                              &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfACot", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_DBLE

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%double => A%double
            call set_status_tempo_to_false( A )
         else
            allocate( out%double(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%double(i,j) = atan( 1.0d0 / A%double(i,j) )
            end do
         end do
         out%prop%symm = A%prop%symm
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => A%cmplx
            call set_status_tempo_to_false( A )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%cmplx(i,j) = mf_gsl_complex_arctan(                  &
                                  (1.0d0,0.0d0) / A%cmplx(i,j) )
            end do
         end do
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfACot", "E",                           &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfACot
!_______________________________________________________________________
!
   function mfSec( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! secant

      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfSec", "E",                               &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfSec", "E",                               &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfSec", "E",                               &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_DBLE

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%double => A%double
            call set_status_tempo_to_false( A )
         else
            allocate( out%double(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%double(i,j) = 1.0d0 / cos( A%double(i,j) )
            end do
         end do
         out%prop%symm = A%prop%symm
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => A%cmplx
            call set_status_tempo_to_false( A )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%cmplx(i,j) = (1.0d0,0.0d0) / cos( A%cmplx(i,j) )
            end do
         end do
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfSec", "E",                            &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfSec
!_______________________________________________________________________
!
   function mfASec( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! arcsecant

      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      if( mf_auto_complex ) then
         call mf_save_and_disable_fpe( )
      end if

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfASec", "E",                              &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfASec", "E",                              &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfASec", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         if( mf_auto_complex .and.                                      &
             any(abs(A%double(:,:)) < 1.0d0 ) ) then
            out%data_type = MF_DT_CMPLX
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

            do j = 1, A%shape(2)
               do i = 1, A%shape(1)
                  out%cmplx(i,j) = mf_gsl_complex_arccos_real( 1.0d0 / A%double(i,j) )
               end do
            end do
         else
            out%data_type = MF_DT_DBLE

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%double => A%double
               call set_status_tempo_to_false( A )
            else
               allocate( out%double(A%shape(1),A%shape(2)) )

            end if
            do j = 1, A%shape(2)
               do i = 1, A%shape(1)
                  out%double(i,j) = acos( 1.0d0 / A%double(i,j) )
               end do
            end do
            out%prop%symm = A%prop%symm
         end if
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => A%cmplx
            call set_status_tempo_to_false( A )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%cmplx(i,j) = mf_gsl_complex_arccos(                  &
                                  (1.0d0,0.0d0) / A%cmplx(i,j) )
            end do
         end do
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfASec", "E",                           &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      if( mf_auto_complex ) then
         call mf_restore_fpe( )
      end if

#endif
   end function mfASec
!_______________________________________________________________________
!
   function mfCsc( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! cosecant

      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfCsc", "E",                               &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfCsc", "E",                               &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfCsc", "E",                               &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_DBLE

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%double => A%double
            call set_status_tempo_to_false( A )
         else
            allocate( out%double(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%double(i,j) = 1.0d0 / sin( A%double(i,j) )
            end do
         end do
         out%prop%symm = A%prop%symm
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => A%cmplx
            call set_status_tempo_to_false( A )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%cmplx(i,j) = (1.0d0,0.0d0) / sin( A%cmplx(i,j) )
            end do
         end do
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfCsc", "E",                            &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfCsc
!_______________________________________________________________________
!
   function mfACsc( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! arccosecant

      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      if( mf_auto_complex ) then
         call mf_save_and_disable_fpe( )
      end if

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfACsc", "E",                              &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfACsc", "E",                              &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfACsc", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         if( mf_auto_complex .and.                                      &
             any(abs(A%double(:,:)) < 1.0d0 ) ) then
            out%data_type = MF_DT_CMPLX
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

            do j = 1, A%shape(2)
               do i = 1, A%shape(1)
                  out%cmplx(i,j) = mf_gsl_complex_arcsin_real( 1.0d0 / A%double(i,j) )
               end do
            end do
         else
            out%data_type = MF_DT_DBLE

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%double => A%double
               call set_status_tempo_to_false( A )
            else
               allocate( out%double(A%shape(1),A%shape(2)) )

            end if
            do j = 1, A%shape(2)
               do i = 1, A%shape(1)
                  out%double(i,j) = asin( 1.0d0 / A%double(i,j) )
               end do
            end do
            out%prop%symm = A%prop%symm
         end if
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => A%cmplx
            call set_status_tempo_to_false( A )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%cmplx(i,j) = mf_gsl_complex_arcsin(                  &
                                  (1.0d0,0.0d0) / A%cmplx(i,j) )
            end do
         end do
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfACsc", "E",                           &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      if( mf_auto_complex ) then
         call mf_restore_fpe( )
      end if

#endif
   end function mfACsc
!_______________________________________________________________________
!
   function mfCoth( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! hyperbolic cotangent

      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfCoth", "E",                              &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfCoth", "E",                              &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfCoth", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_DBLE

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%double => A%double
            call set_status_tempo_to_false( A )
         else
            allocate( out%double(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%double(i,j) = 1.0d0 / tanh( A%double(i,j) )
            end do
         end do
         out%prop%symm = A%prop%symm
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => A%cmplx
            call set_status_tempo_to_false( A )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%cmplx(i,j) = (1.0d0,0.0d0) /                         &
                                mf_gsl_complex_tanh( A%cmplx(i,j) )
            end do
         end do
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfCoth", "E",                           &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfCoth
!_______________________________________________________________________
!
   function mfACoth( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! hyperbolic arccotangent

      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      if( mf_auto_complex ) then
         call mf_save_and_disable_fpe( )
      end if

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfACoth", "E",                             &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfACoth", "E",                             &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfACoth", "E",                             &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         if( mf_auto_complex .and.                                      &
             any( A%double(:,:) > 1.0d0 ) ) then
            out%data_type = MF_DT_CMPLX
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

            do j = 1, A%shape(2)
               do i = 1, A%shape(1)
                  out%cmplx(i,j) = mf_gsl_complex_arctanh_real( 1.0d0 / A%double(i,j) )
               end do
            end do
         else
            out%data_type = MF_DT_DBLE

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%double => A%double
               call set_status_tempo_to_false( A )
            else
               allocate( out%double(A%shape(1),A%shape(2)) )

            end if
            do j = 1, A%shape(2)
               do i = 1, A%shape(1)
                  out%double(i,j) = atanh( 1.0d0 / A%double(i,j) )
               end do
            end do
            out%prop%symm = A%prop%symm
         end if
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => A%cmplx
            call set_status_tempo_to_false( A )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%cmplx(i,j) = mf_gsl_complex_arctanh(                 &
                                 (1.0d0,0.0d0) / A%cmplx(i,j) )
            end do
         end do
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfACoth", "E",                          &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      if( mf_auto_complex ) then
         call mf_restore_fpe( )
      end if

#endif
   end function mfACoth
!_______________________________________________________________________
!
   function mfSech( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! hyperbolic secant

      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfSech", "E",                              &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfSech", "E",                              &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfSech", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_DBLE

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%double => A%double
            call set_status_tempo_to_false( A )
         else
            allocate( out%double(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%double(i,j) = 1.0d0 / cosh( A%double(i,j) )
            end do
         end do
         out%prop%symm = A%prop%symm
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => A%cmplx
            call set_status_tempo_to_false( A )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%cmplx(i,j) = (1.0d0,0.0d0) /                         &
                                mf_gsl_complex_cosh( A%cmplx(i,j) )
            end do
         end do
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfSech", "E",                           &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfSech
!_______________________________________________________________________
!
   function mfASech( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! hyperbolic arcsecant

      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      if( mf_auto_complex ) then
         call mf_save_and_disable_fpe( )
      end if

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfASech", "E",                             &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfASech", "E",                             &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfASech", "E",                             &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         if( mf_auto_complex .and.                                      &
             any( A%double(:,:) > 1.0d0 ) ) then
            out%data_type = MF_DT_CMPLX
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

            do j = 1, A%shape(2)
               do i = 1, A%shape(1)
                  out%cmplx(i,j) = mf_gsl_complex_arccosh_real( 1.0d0 / A%double(i,j) )
               end do
            end do
         else
            out%data_type = MF_DT_DBLE

            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%double => A%double
               call set_status_tempo_to_false( A )
            else
               allocate( out%double(A%shape(1),A%shape(2)) )

            end if
            do j = 1, A%shape(2)
               do i = 1, A%shape(1)
                  out%double(i,j) = acosh( 1.0d0 / A%double(i,j) )
               end do
            end do
            out%prop%symm = A%prop%symm
         end if
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => A%cmplx
            call set_status_tempo_to_false( A )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%cmplx(i,j) = mf_gsl_complex_arccosh(                 &
                                 (1.0d0,0.0d0) / A%cmplx(i,j) )
            end do
         end do
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfASech", "E",                          &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      if( mf_auto_complex ) then
         call mf_restore_fpe( )
      end if

#endif
   end function mfASech
!_______________________________________________________________________
!
   function mfCsch( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! hyperbolic cosecant

      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfCsch", "E",                              &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfCsch", "E",                              &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfCsch", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_DBLE

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%double => A%double
            call set_status_tempo_to_false( A )
         else
            allocate( out%double(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%double(i,j) = 1.0d0 / sinh( A%double(i,j) )
            end do
         end do
         out%prop%symm = A%prop%symm
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => A%cmplx
            call set_status_tempo_to_false( A )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%cmplx(i,j) = (1.0d0,0.0d0) /                         &
                                mf_gsl_complex_sinh( A%cmplx(i,j) )
            end do
         end do
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfCsch", "E",                           &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfCsch
!_______________________________________________________________________
!
   function mfACsch( A ) result( out )

      type(mfArray) :: A
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! hyperbolic arccosecant

      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfACsch", "E",                             &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfACsch", "E",                             &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfACsch", "E",                             &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_DBLE

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%double => A%double
            call set_status_tempo_to_false( A )
         else
            allocate( out%double(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%double(i,j) = asinh( 1.0d0 / A%double(i,j) )
            end do
         end do
         out%prop%symm = A%prop%symm
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => A%cmplx
            call set_status_tempo_to_false( A )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%cmplx(i,j) = mf_gsl_complex_arcsinh(                 &
                                 (1.0d0,0.0d0) / A%cmplx(i,j) )
            end do
         end do
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfACsch", "E",                          &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfACsch
!_______________________________________________________________________
!
   function mfFun_apply_real( A, fun ) result( out )

      type(mfArray) :: A

      interface
         function fun(r1) result(r2)
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in) :: r1
            real(kind=MF_DOUBLE)             :: r2
         end function fun
      end interface

      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! element-wise user-defined function

      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfFun", "E",                               &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfFun", "E",                               &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfFun", "E",                               &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( mfIsComplex(a) ) then
         call PrintMessage( "mfFun", "E",                               &
                            "complex mfArray detected...",              &
                            "please provide a complex function instead a real one!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_DBLE

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%double => A%double
            call set_status_tempo_to_false( A )
         else
            allocate( out%double(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%double(i,j) = fun( A%double(i,j) )
            end do
         end do
         out%prop%symm = A%prop%symm
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfFun", "E",                            &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfFun_apply_real
!_______________________________________________________________________
!
   function mfFun_apply_cmplx( A, fun ) result( out )

      type(mfArray) :: A

      interface
         function fun(z1) result(z2)
            import :: MF_DOUBLE
            complex(kind=MF_DOUBLE), intent(in) :: z1
            complex(kind=MF_DOUBLE)             :: z2
         end function fun
      end interface

      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! element-wise user-defined function

      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfFun", "E",                               &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfFun", "E",                               &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfFun", "E",                               &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         call PrintMessage( "mfFun", "I",                               &
                            "complex function applied to a real mfArray...", &
                            "-> converting the mfArray into a complex one!" )
         out%data_type = MF_DT_CMPLX
         allocate( out%cmplx(A%shape(1),A%shape(2)) )

         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%cmplx(i,j) = fun( cmplx(A%double(i,j),kind=MF_DOUBLE) )
            end do
         end do
         out%prop%symm = A%prop%symm
      else if( A%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => A%cmplx
            call set_status_tempo_to_false( A )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%cmplx(i,j) = fun( A%cmplx(i,j) )
            end do
         end do
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfFun", "E",                            &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfFun_apply_cmplx
!_______________________________________________________________________
!
   function mfFun2_apply_real( A, B, fun ) result( out )

      type(mfArray) :: A, B

      interface
         function fun( x1, x2 ) result(res)
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in) :: x1, x2
            real(kind=MF_DOUBLE)             :: res
         end function fun
      end interface

      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! element-wise user-defined function

      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A, B )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsEmpty(B) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfFun2", "E",                              &
                            "mfArray A cannot be sparse!" )
         go to 99
      end if

      if( mfIsSparse(B) ) then
         call PrintMessage( "mfFun2", "E",                              &
                            "mfArray B cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfFun2", "E",                              &
                            "mfArray A cannot be a logical!" )
         go to 99
      end if

      if( B%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfFun2", "E",                              &
                            "mfArray B cannot be a logical!" )
         go to 99
      end if

      if( mfIsPerm(A) ) then
         call PrintMessage( "mfFun2", "E",                              &
                            "mfArray A cannot be a permutation vector!" )
         go to 99
      end if

      if( mfIsPerm(B) ) then
         call PrintMessage( "mfFun2", "E",                              &
                            "mfArray B cannot be a permutation vector!" )
         go to 99
      end if

      if( mfIsComplex(A) ) then
         call PrintMessage( "mfFun2", "E",                              &
                            "complex mfArray A detected...",            &
                            "please provide a complex function instead a real one!" )
         go to 99
      end if

      if( mfIsComplex(B) ) then
         call PrintMessage( "mfFun2", "E",                              &
                            "complex mfArray B detected...",            &
                            "please provide a complex function instead a real one!" )
         go to 99
      end if

      if( any(A%shape /= B%shape) ) then
         call PrintMessage( "mfFun2", "E",                              &
                            "mfArray A and B must have the same shape!" )
         go to 99
      end if

      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE .and.                               &
          B%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_DBLE

         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%double => A%double
            call set_status_tempo_to_false( A )
         else if( can_use_memory(B) ) then
            out%double => B%double
            call set_status_tempo_to_false( B )
         else
            allocate( out%double(A%shape(1),A%shape(2)) )

         end if
         do j = 1, A%shape(2)
            do i = 1, A%shape(1)
               out%double(i,j) = fun( A%double(i,j), B%double(i,j) )
            end do
         end do
         if( A%prop%symm == TRUE .and. B%prop%symm == TRUE ) then
            out%prop%symm = TRUE
         end if
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfFun2", "E",                           &
                               "the physical unit of the mfArray A",    &
                               "must be dimensionless!" )
            go to 99
         end if
         call verif_adim( B%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfFun2", "E",                           &
                               "the physical unit of the mfArray B",    &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A, B )
      call msAutoRelease( A, B )

      call mf_restore_fpe( )

#endif
   end function mfFun2_apply_real
!_______________________________________________________________________
!
   function mfFun2_apply_cmplx( A, B, fun ) result( out )

      type(mfArray) :: A, B

      interface
         function fun( z1, z2 ) result(res)
            import :: MF_DOUBLE
            complex(kind=MF_DOUBLE), intent(in) :: z1, z2
            complex(kind=MF_DOUBLE)             :: res
         end function fun
      end interface

      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! element-wise user-defined function

      integer :: i, j
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A, B )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsEmpty(B) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfFun2", "E",                              &
                            "mfArray A cannot be sparse!" )
         go to 99
      end if

      if( mfIsSparse(B) ) then
         call PrintMessage( "mfFun2", "E",                              &
                            "mfArray B cannot be sparse!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfFun2", "E",                              &
                            "mfArray A cannot be a logical!" )
         go to 99
      end if

      if( B%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfFun2", "E",                              &
                            "mfArray B cannot be a logical!" )
         go to 99
      end if

      if( mfIsPerm(A) ) then
         call PrintMessage( "mfFun2", "E",                              &
                            "mfArray A cannot be a permutation vector!" )
         go to 99
      end if

      if( mfIsPerm(B) ) then
         call PrintMessage( "mfFun2", "E",                              &
                            "mfArray B cannot be a permutation vector!" )
         go to 99
      end if

      if( any(A%shape /= B%shape) ) then
         call PrintMessage( "mfFun2", "E",                              &
                            "mfArray A and B must have the same shape!" )
         go to 99
      end if

      out%shape = A%shape
      out%data_type = MF_DT_CMPLX
      if( A%data_type == MF_DT_DBLE ) then
         call PrintMessage( "mfFun", "I",                               &
                            "complex function applied to a real mfArray...", &
                            "-> converting the mfArray into a complex one!" )
         if( B%data_type == MF_DT_DBLE ) then
            allocate( out%cmplx(A%shape(1),A%shape(2)) )
            do j = 1, A%shape(2)
               do i = 1, A%shape(1)
                  out%cmplx(i,j) = fun( cmplx(A%double(i,j),kind=MF_DOUBLE), &
                                        cmplx(B%double(i,j),kind=MF_DOUBLE) )
               end do
            end do
         else if( B%data_type == MF_DT_CMPLX ) then
            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(B) ) then
               out%cmplx => B%cmplx
               call set_status_tempo_to_false( B )
            else
               allocate( out%cmplx(A%shape(1),A%shape(2)) )
            end if
            do j = 1, A%shape(2)
               do i = 1, A%shape(1)
                  out%cmplx(i,j) = fun( cmplx(A%double(i,j),kind=MF_DOUBLE), &
                                        B%cmplx(i,j) )
               end do
            end do
         end if
      else if( A%data_type == MF_DT_CMPLX ) then
         if( B%data_type == MF_DT_DBLE ) then
            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%cmplx => A%cmplx
               call set_status_tempo_to_false( A )
            else
               allocate( out%cmplx(A%shape(1),A%shape(2)) )
            end if
            do j = 1, A%shape(2)
               do i = 1, A%shape(1)
                  out%cmplx(i,j) = fun( A%cmplx(i,j),                    &
                                        cmplx(B%double(i,j),kind=MF_DOUBLE) )
               end do
            end do
         else if( B%data_type == MF_DT_CMPLX ) then
            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( can_use_memory(A) ) then
               out%cmplx => A%cmplx
               call set_status_tempo_to_false( A )
            else if( can_use_memory(B) ) then
               out%cmplx => B%cmplx
               call set_status_tempo_to_false( B )
            else
               allocate( out%cmplx(A%shape(1),A%shape(2)) )
            end if
            do j = 1, A%shape(2)
               do i = 1, A%shape(1)
                  out%cmplx(i,j) = fun( A%cmplx(i,j), B%cmplx(i,j) )
               end do
            end do
         end if
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( A%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfFun2", "E",                           &
                               "the physical unit of the mfArray A",    &
                               "must be dimensionless!" )
            go to 99
         end if
         call verif_adim( B%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfFun2", "E",                           &
                               "the physical unit of the mfArray B",    &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A, B )
      call msAutoRelease( A, B )

      call mf_restore_fpe( )

#endif
   end function mfFun2_apply_cmplx
!_______________________________________________________________________
!
   function mfGridFun_intr( v_x, v_y, fun ) result( out )

      type(mfArray) :: v_x, v_y
      character(len=*) :: fun

      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! for intrinsic functions

      integer :: i, j, m, n, dim_x, dim_y
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( v_x, v_y )

      if( mfIsEmpty(v_x) .or. mfIsEmpty(v_y) ) then
         go to 99
      end if

      if( mfIsSparse(v_x) .or. mfIsSparse(v_y) ) then
         call PrintMessage( "mfGridFun", "E",                           &
                            "vectors v_x and v_y cannot be sparse!" )
         go to 99
      end if

      if( .not. ( mfIsReal(v_x) .and. mfIsReal(v_y) ) ) then
         call PrintMessage( "mfGridFun", "E",                           &
                            "args must be real!" )
         go to 99
      end if

      if( mfIsEmpty(v_x) .or. mfIsEmpty(v_y) ) then
         go to 99
      end if

      if( v_x%shape(1) /= 1 ) then
         if( v_x%shape(2) /= 1 ) then
            call PrintMessage( "mfGridFun", "E",                        &
                               "mfArray 'v_x' must be a vector!" )
            go to 99
         else
            dim_x = 1
         end if
      else
         dim_x = 2
      end if

      if( v_y%shape(1) /= 1 ) then
         if( v_y%shape(2) /= 1 ) then
            call PrintMessage( "mfGridFun", "E",                        &
                               "mfArray 'v_y' must be a vector!" )
            go to 99
         else
            dim_y = 1
         end if
      else
         dim_y = 2
      end if

      if( dim_x == dim_y ) then
         call PrintMessage( "mfGridFun", "E",                           &
                            "bad args shape: you must provide one row vector", &
                            "and one column vector!" )
         go to 99
      end if

      if( dim_x == 1 ) then
         m = v_x%shape(1)
         n = v_y%shape(2)
      else ! dim_x = 2
         m = v_y%shape(1)
         n = v_x%shape(2)
      end if

      out%shape = [ m, n ]

      select case( trim(fun) )
         case( "cmplx" )
            out%data_type = MF_DT_CMPLX
            allocate( out%cmplx(out%shape(1),out%shape(2)) )
            if( dim_x == 1 ) then
               ! v_x: col, v_y: row
               do j = 1, n
                  do i = 1, m
                     out%cmplx(i,j) = cmplx( v_x%double(i,1),           &
                                             v_y%double(1,j), kind=MF_DOUBLE )
                  end do
               end do
            else
               ! v_x: row, v_y: col
               do j = 1, n
                  do i = 1, m
                     out%cmplx(i,j) = cmplx( v_x%double(1,j),           &
                                             v_y%double(i,1), kind=MF_DOUBLE )
                  end do
               end do
            end if
         case( "hypot" )
            out%data_type = MF_DT_DBLE
            allocate( out%double(out%shape(1),out%shape(2)) )
            if( dim_x == 1 ) then
               ! v_x: col, v_y: row
               do j = 1, n
                  do i = 1, m
                     out%double(i,j) = hypot_real( v_x%double(i,1),     &
                                                   v_y%double(1,j) )
                  end do
               end do
            else
               ! v_x: row, v_y: col
               do j = 1, n
                  do i = 1, m
                     out%double(i,j) = hypot_real( v_x%double(1,j),     &
                                                   v_y%double(i,1) )
                  end do
               end do
            end if
         case( "atan2" )
            out%data_type = MF_DT_DBLE
            allocate( out%double(out%shape(1),out%shape(2)) )
            if( dim_x == 1 ) then
               ! v_x: col, v_y: row
               do j = 1, n
                  do i = 1, m
                     out%double(i,j) = atan2( v_y%double(1,j),          &
                                              v_x%double(i,1) )
                  end do
               end do
            else
               ! v_x: row, v_y: col
               do j = 1, n
                  do i = 1, m
                     out%double(i,j) = atan2( v_y%double(i,1),          &
                                              v_x%double(1,j) )
                  end do
               end do
            end if
         case default
            call PrintMessage( "mfGridFun", "E",                        &
                               "function:" // trim(fun) // " must be one of", &
                               '"cmplx", "hypot", "atan2"' )
            go to 99
      end select

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( v_x%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfGridFun", "E",                        &
                               "the mfArray v_x must be dimensionless!" )
            go to 99
         end if
         call verif_adim( v_y%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfGridFun", "E",                        &
                               "the mfArray v_y must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( v_x, v_y )
      call msAutoRelease( v_x, v_y )

      call mf_restore_fpe( )

#endif
   end function mfGridFun_intr
!_______________________________________________________________________
!
   function mfGridFun_user( v_x, v_y, fun ) result( out )

      type(mfArray) :: v_x, v_y

      interface
         function fun( x1, x2 ) result(res)
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in) :: x1, x2
            real(kind=MF_DOUBLE)             :: res
         end function fun
      end interface

      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! element-wise user-defined function

      integer :: i, j, m, n, dim_x, dim_y
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( v_x, v_y )

      if( mfIsEmpty(v_x) .or. mfIsEmpty(v_y) ) then
         go to 99
      end if

      if( mfIsSparse(v_x) .or. mfIsSparse(v_y) ) then
         call PrintMessage( "mfGridFun", "E",                           &
                            "vectors v_x and v_y cannot be sparse!" )
         go to 99
      end if

      if( .not. ( mfIsReal(v_x) .and. mfIsReal(v_y) ) ) then
         call PrintMessage( "mfGridFun", "E",                           &
                            "args must be real!" )
         go to 99
      end if

      if( mfIsEmpty(v_x) .or. mfIsEmpty(v_y) ) then
         go to 99
      end if

      if( v_x%shape(1) /= 1 ) then
         if( v_x%shape(2) /= 1 ) then
            call PrintMessage( "mfGridFun", "E",                        &
                               "mfArray 'v_x' must be a vector!" )
            go to 99
         else
            dim_x = 1
         end if
      else
         dim_x = 2
      end if

      if( v_y%shape(1) /= 1 ) then
         if( v_y%shape(2) /= 1 ) then
            call PrintMessage( "mfGridFun", "E",                        &
                               "mfArray 'v_y' must be a vector!" )
            go to 99
         else
            dim_y = 1
         end if
      else
         dim_y = 2
      end if

      if( dim_x == dim_y ) then
         call PrintMessage( "mfGridFun", "E",                           &
                            "bad args shape: you must provide one row vector", &
                            "and one column vector!" )
         go to 99
      end if

      if( dim_x == 1 ) then
         m = v_x%shape(1)
         n = v_y%shape(2)
      else ! dim_x = 2
         m = v_y%shape(1)
         n = v_x%shape(2)
      end if

      out%shape = [ m, n ]
      out%data_type = MF_DT_DBLE

      allocate( out%double(out%shape(1),out%shape(2)) )

      if( dim_x == 1 ) then
         ! v_x: col, v_y: row
         do j = 1, n
            do i = 1, m
               out%double(i,j) = fun( v_x%double(i,1), v_y%double(1,j) )
            end do
         end do
      else
         ! v_x: row, v_y: col
         do j = 1, n
            do i = 1, m
               out%double(i,j) = fun( v_x%double(1,j), v_y%double(i,1) )
            end do
         end do
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( v_x%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfGridFun", "E",                        &
                               "the physical unit of the mfArray v_x",  &
                               "must be dimensionless!" )
            go to 99
         end if
         call verif_adim( v_y%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfGridFun", "E",                        &
                               "the physical unit of the mfArray v_y",  &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( v_x, v_y )
      call msAutoRelease( v_x, v_y )

      call mf_restore_fpe( )

#endif
   end function mfGridFun_user
!_______________________________________________________________________
!
end module mod_elfun
