! f90 include file
!
! list of Fortran 90 interfaces for some of the SPQR wrapper routines,
! stored in the file: 'spqr_f90wrapper.c'.
!
! Copyright É. Canot 2012 -- IPR/CNRS

interface
!-----------------------------------------------------------------------
!                      R E A L     V A L U E S
!-----------------------------------------------------------------------

!_______________________________________________________________________
!
! Solve A x = b        [least-square pb]  over-determined system (m>n)
subroutine spqr_solve_lsq( m, n, nnz, Ap, Ai, Ax, b,                    &
                           out, status )
   import :: MF_DOUBLE
   ! shape of the square matrix A (sparse)
   integer, intent(in) :: m, n, nnz
   ! indexes of A (CSC storage with C indexes, i.e. 0-based)
   integer, intent(in) :: Ap(*), Ai(*)
   ! numerical entries of A (CSC)
   real(kind=MF_DOUBLE), intent(in) :: Ax(*), b(*)
   ! address of a specific structure storing information
   real(kind=MF_DOUBLE), intent(out) :: out(*)
   ! returned status
   integer, intent(out) :: status
end subroutine
!_______________________________________________________________________
!
! Solve A' x = b       [least-square pb]  over-determined system (n>m)
subroutine spqr_tsolve_lsq( m, n, nnz, Ap, Ai, Ax, b,                   &
                            out, status )
   import :: MF_DOUBLE
   ! shape of the square matrix A (sparse)
   integer, intent(in) :: m, n, nnz
   ! indexes of A (CSC storage with C indexes, i.e. 0-based)
   integer, intent(in) :: Ap(*), Ai(*)
   ! numerical entries of A (CSC)
   real(kind=MF_DOUBLE), intent(in) :: Ax(*), b(*)
   ! address of a specific structure storing information
   real(kind=MF_DOUBLE), intent(out) :: out(*)
   ! returned status
   integer, intent(out) :: status
end subroutine
!_______________________________________________________________________
!
! Solve A x = b       [min. 2-norm sol]  under-determined system (m<n)
subroutine spqr_solve_min2norm( m, n, nnz, Ap, Ai, Ax, b,               &
                                out, status )
   import :: MF_DOUBLE
   ! shape of the square matrix A (sparse)
   integer, intent(in) :: m, n, nnz
   ! indexes of A (CSC storage with C indexes, i.e. 0-based)
   integer, intent(in) :: Ap(*), Ai(*)
   ! numerical entries of A (CSC)
   real(kind=MF_DOUBLE), intent(in) :: Ax(*), b(*)
   ! address of a specific structure storing information
   real(kind=MF_DOUBLE), intent(out) :: out(*)
   ! returned status
   integer, intent(out) :: status
end subroutine
!_______________________________________________________________________
!
! Solve A' x = b      [min. 2-norm sol]  under-determined system (n<m)
subroutine spqr_tsolve_min2norm( m, n, nnz, Ap, Ai, Ax, b,              &
                                 out, status )
   import :: MF_DOUBLE
   ! shape of the square matrix A (sparse)
   integer, intent(in) :: m, n, nnz
   ! indexes of A (CSC storage with C indexes, i.e. 0-based)
   integer, intent(in) :: Ap(*), Ai(*)
   ! numerical entries of A (CSC)
   real(kind=MF_DOUBLE), intent(in) :: Ax(*), b(*)
   ! address of a specific structure storing information
   real(kind=MF_DOUBLE), intent(out) :: out(*)
   ! returned status
   integer, intent(out) :: status
end subroutine
!_______________________________________________________________________
!
! Do a Q-less factorization of A
subroutine spqr_q_less_prep( m, n, nnz, Ap, Ai, Ax,                     &
                             c_addr, R_addr, rnz, status )
   import :: MF_ADDRESS, MF_DOUBLE
   integer, intent(in) :: m, n ! size of the matrix A (sparse)
   integer, intent(in) :: nnz ! nb of non-zero element in A
   integer, intent(in) :: Ap(*), Ai(*) ! indexes and elements of A
   real(kind=MF_DOUBLE), intent(in) :: Ax(*)   ! (CSC storage with Fortran
                                       !  indexes, i.e. 1-based)
   integer(kind=MF_ADDRESS), intent(out) :: c_addr ! address of the cholmod_common
   integer(kind=MF_ADDRESS), intent(out) :: R_addr ! address of the R factor
   integer, intent(out) :: rnz ! non zeros of R
   integer, intent(out) :: status ! 0: A is positive definite,
                                  ! 1: otherwise
end subroutine
!_______________________________________________________________________
!
! Get the entries of the R factor
subroutine spqr_get_r( c_addr, R_addr, ncol_R, nz_R,                         &
                       Rp, Ri, Rx )
   import :: MF_ADDRESS, MF_DOUBLE
   integer(kind=MF_ADDRESS), intent(in) :: c_addr ! address of the cholmod_common
                                 ! specific to the factorisation
   integer(kind=MF_ADDRESS), intent(in) :: R_addr ! address of the R factor
                                  ! (special internal format)
   integer, intent(in) :: ncol_R
   integer, intent(in) :: nz_R
   integer, intent(out) :: Rp(*), Ri(*) ! indexes and elements of L
   real(kind=MF_DOUBLE), intent(out) :: Rx(*) ! (CSC storage with Fortran
                                              ! indexes, i.e. 1-based)
end subroutine
!_______________________________________________________________________
!
! Do a QR factorization of A
subroutine spqr_qr_prep( m, n, nnz, Ap, Ai, Ax, ordering, tol, econ,    &
                         c_addr, Q_addr, qnz, R_addr, rnz, P_addr,      &
                         rank, status )
   import :: MF_ADDRESS, MF_DOUBLE
   integer, intent(in) :: m, n ! size of the matrix A (sparse)
   integer, intent(in) :: nnz ! nb of non-zero element in A
   integer, intent(in) :: Ap(*), Ai(*) ! indexes and elements of A
   real(kind=MF_DOUBLE), intent(in) :: Ax(*)   ! (CSC storage with Fortran
                                       !  indexes, i.e. 1-based)
   integer, intent(in) :: ordering ! ???
   real(kind=MF_DOUBLE), intent(in) :: tol   ! tolerance for rank computing
   integer, intent(in) :: econ ! economize size of output
   integer(kind=MF_ADDRESS), intent(out) :: c_addr ! address of the cholmod_common
   integer(kind=MF_ADDRESS), intent(out) :: Q_addr ! address of the Q factor
   integer, intent(out) :: qnz ! non zeros of Q
   integer(kind=MF_ADDRESS), intent(out) :: R_addr ! address of the R factor
   integer, intent(out) :: rnz ! non zeros of R
   integer(kind=MF_ADDRESS), intent(out) :: P_addr ! address of the permutation
   integer, intent(out) :: rank ! rank estimation
   integer, intent(out) :: status ! 0: A is positive definite,
                                  ! 1: otherwise
end subroutine
!_______________________________________________________________________
!
! Get the entries of the factors Q and R
subroutine spqr_get_q_r( c_addr, Q_addr, qn, qnz, R_addr, nn, rnz,      &
                         Qp, Qi, Qx, Rp, Ri, Rx )
   import :: MF_ADDRESS, MF_DOUBLE
   integer(kind=MF_ADDRESS), intent(in) :: c_addr ! address of the cholmod_common
                                 ! specific to the factorisation
   integer(kind=MF_ADDRESS), intent(in) :: Q_addr ! address of the Q factor
                                  ! (special internal format)
   integer, intent(in) :: qn ! nb of col of Q
   integer, intent(in) :: qnz !  nnz(R)
   integer(kind=MF_ADDRESS), intent(in) :: R_addr ! address of the R factor
                                  ! (special internal format)
   integer, intent(in) :: nn ! nb of col of R
   integer, intent(in) :: rnz !  nnz(R)
   integer, intent(out) :: Qp(*), Qi(*) ! indexes and elements of L
   real(kind=MF_DOUBLE), intent(out) :: Qx(*)   ! (CSC storage with Fortran
                                       !  indexes, i.e. 1-based)
   integer, intent(out) :: Rp(*), Ri(*) ! indexes and elements of L
   real(kind=MF_DOUBLE), intent(out) :: Rx(*)   ! (CSC storage with Fortran
                                       !  indexes, i.e. 1-based)
end subroutine
!_______________________________________________________________________
!
! Get the entries of the factors Q and R and permutation vector p
subroutine spqr_get_q_r_p( c_addr, Q_addr, qn, qnz, R_addr, nn, rnz,    &
                           p_addr, pn,                                  &
                           Qp, Qi, Qx, Rp, Ri, Rx, p )
   import :: MF_ADDRESS, MF_DOUBLE
   integer(kind=MF_ADDRESS), intent(in) :: c_addr ! address of the cholmod_common
                                 ! specific to the factorisation
   integer(kind=MF_ADDRESS), intent(in) :: Q_addr ! address of the Q factor
                                  ! (special internal format)
   integer, intent(in) :: qn ! nb of col of Q
   integer, intent(in) :: qnz !  nnz(R)
   integer(kind=MF_ADDRESS), intent(in) :: R_addr ! address of the R factor
                                  ! (special internal format)
   integer, intent(in) :: nn ! nb of col of R
   integer, intent(in) :: rnz !  nnz(R)
   integer(kind=MF_ADDRESS), intent(in) :: P_addr ! address of the permutation
   integer, intent(in) :: pn ! size of P
   integer, intent(out) :: Qp(*), Qi(*) ! indexes and elements of L
   real(kind=MF_DOUBLE), intent(out) :: Qx(*)   ! (CSC storage with Fortran
                                       !  indexes, i.e. 1-based)
   integer, intent(out) :: Rp(*), Ri(*) ! indexes and elements of L
   real(kind=MF_DOUBLE), intent(out) :: Rx(*)   ! (CSC storage with Fortran
                                       !  indexes, i.e. 1-based)
   integer, intent(out) :: P(*)   ! vector permutation
end subroutine
!_______________________________________________________________________
!
! Do a QR factorization of A (Q in Householder form)
subroutine spqhr_qr_prep( m, n, nnz, Ap, Ai, Ax, ordering, tol, econ,   &
                          c_addr, H_addr, HTau_addr, HPinv_addr,        &
                          R_addr, rnz, P_addr, rank, status )
   import :: MF_ADDRESS, MF_DOUBLE
   integer, intent(in) :: m, n ! size of the matrix A (sparse)
   integer, intent(in) :: nnz ! nb of non-zero element in A
   integer, intent(in) :: Ap(*), Ai(*) ! indexes and elements of A
   real(kind=MF_DOUBLE), intent(in) :: Ax(*)   ! (CSC storage with Fortran
                                       !  indexes, i.e. 1-based)
   integer, intent(in) :: ordering ! ???
   real(kind=MF_DOUBLE), intent(in) :: tol   ! tolerance for rank computing
   integer, intent(in) :: econ ! economize size of output
   integer(kind=MF_ADDRESS), intent(out) :: c_addr ! address of the cholmod_common
   integer(kind=MF_ADDRESS), intent(out) :: H_addr ! address of H
   integer(kind=MF_ADDRESS), intent(out) :: HTau_addr ! address of HTau_addr
   integer(kind=MF_ADDRESS), intent(out) :: HPinv_addr ! address of HPinv_addr
   integer(kind=MF_ADDRESS), intent(out) :: R_addr ! address of the R factor
   integer, intent(out) :: rnz ! non zeros of R
   integer(kind=MF_ADDRESS), intent(out) :: P_addr ! address of the permutation
   integer, intent(out) :: rank ! rank estimation
   integer, intent(out) :: status ! 0: A is positive definite,
                                  ! 1: otherwise
end subroutine
!_______________________________________________________________________
!
! Get the entries of a cholmod_sparse object
subroutine spqr_get_sparse_entries( c_addr, A_addr, ncol_A, nz_A,       &
                                    Ap, Ai, Ax )
   import :: MF_ADDRESS, MF_DOUBLE
   integer(kind=MF_ADDRESS), intent(in) :: c_addr ! address of the cholmod_common
                                 ! specific to the factorisation
   integer(kind=MF_ADDRESS), intent(in) :: A_addr ! address of the matrix A
   integer, intent(in) :: ncol_A
   integer, intent(in) :: nz_A
   integer, intent(out) :: Ap(*), Ai(*) ! indexes and elements of A
   real(kind=MF_DOUBLE), intent(out) :: Ax(*) ! (CSC storage with Fortran
                                             !  indexes, i.e. 1-based)
end subroutine
!_______________________________________________________________________
!
! Get the entries of the factor R and permutation P
subroutine spqhr_get_r_p( c_addr, R_addr, rn, rnz, P_addr, pn,          &
                          Rp, Ri, Rx, p )
   import :: MF_ADDRESS, MF_DOUBLE
   integer(kind=MF_ADDRESS), intent(in) :: c_addr ! address of the cholmod_common
                                 ! specific to the factorisation
   integer(kind=MF_ADDRESS), intent(in) :: R_addr ! address of the maR factor
   integer, intent(in) :: rn ! nb of col of R
   integer, intent(in) :: rnz ! nnz(R)
   integer(kind=MF_ADDRESS), intent(in) :: P_addr ! address of the permutation
   integer, intent(in) :: pn ! size(P)
   integer, intent(out) :: Rp(*), Ri(*) ! indexes and elements of L
   real(kind=MF_DOUBLE), intent(out) :: Rx(*)   ! (CSC storage with Fortran
                                       !  indexes, i.e. 1-based)
   integer, intent(out) :: p(*)   ! vector permutation
end subroutine
!_______________________________________________________________________
!
! Apply Householder vector on the left, i.e. Q'*A     (A dense)
subroutine spqhr_mleft_dense( c_addr, H_addr, HTau_addr, HPinv_addr,    &
                              A, nrow_A, ncol_A, ld_A,                  &
                              out, status )
   import :: MF_ADDRESS, MF_DOUBLE
   integer(kind=MF_ADDRESS), intent(in) :: c_addr ! address of the cholmod_common
   integer(kind=MF_ADDRESS), intent(in) :: H_addr ! address of H
   integer(kind=MF_ADDRESS), intent(in) :: HTau_addr ! address of HTau_addr
   integer(kind=MF_ADDRESS), intent(in) :: HPinv_addr ! address of HPinv_addr
   real(kind=MF_DOUBLE), intent(in) :: A(*) ! dense matrix
   integer, intent(in) :: nrow_A, ncol_A ! size of the matrix A
   integer, intent(in) :: ld_A ! ???
   real(kind=MF_DOUBLE), intent(out) :: out(*) ! matrix output
   integer, intent(out) :: status ! 0: A is positive definite,
                                  ! 1: otherwise
end subroutine
!_______________________________________________________________________
!
! Apply Householder vector on the right, i.e. A*Q     (A dense)
subroutine spqhr_mright_dense( c_addr, H_addr, HTau_addr, HPinv_addr,   &
                               A, nrow_A, ncol_A, ld_A,                 &
                               out, status )
   import :: MF_ADDRESS, MF_DOUBLE
   integer(kind=MF_ADDRESS), intent(in) :: c_addr ! address of the cholmod_common
   integer(kind=MF_ADDRESS), intent(in) :: H_addr ! address of H
   integer(kind=MF_ADDRESS), intent(in) :: HTau_addr ! address of HTau_addr
   integer(kind=MF_ADDRESS), intent(in) :: HPinv_addr ! address of HPinv_addr
   real(kind=MF_DOUBLE), intent(in) :: A(*) ! dense matrix
   integer, intent(in) :: nrow_A, ncol_A ! size of the matrix A
   integer, intent(in) :: ld_A ! ???
   real(kind=MF_DOUBLE), intent(out) :: out(*) ! matrix output
   integer, intent(out) :: status ! 0: A is positive definite,
                                  ! 1: otherwise
end subroutine
!_______________________________________________________________________
!
! Apply Householder vector on the left, i.e. Q'*A     (A sparse)
subroutine spqhr_mleft_sparse_prep( c_addr, H_addr, HTau_addr, HPinv_addr, &
                                    nrow_A, ncol_A, nnz_A, Ap, Ai, Ax,  &
                                    out_addr, out_nz, status )
   import :: MF_ADDRESS, MF_DOUBLE
   integer(kind=MF_ADDRESS), intent(in) :: c_addr ! address of the cholmod_common
   integer(kind=MF_ADDRESS), intent(in) :: H_addr ! address of H
   integer(kind=MF_ADDRESS), intent(in) :: HTau_addr ! address of HTau_addr
   integer(kind=MF_ADDRESS), intent(in) :: HPinv_addr ! address of HPinv_addr
   integer, intent(in) :: nrow_A, ncol_A ! size of the matrix A
   integer, intent(in) :: nnz_A
   integer, intent(out) :: Ap(*), Ai(*) ! indexes and elements of A
   real(kind=MF_DOUBLE), intent(out) :: Ax(*) ! (CSC storage with Fortran
                                             !  indexes, i.e. 1-based)
   integer(kind=MF_ADDRESS), intent(out) :: out_addr ! address of out
   integer, intent(out) :: out_nz ! nnz(out)
   integer, intent(out) :: status ! 0: A is positive definite,
                                  ! 1: otherwise
end subroutine
!_______________________________________________________________________
!
! Apply Householder vector on the right, i.e. A*Q     (A sparse)
subroutine spqhr_mright_sparse_prep( c_addr, H_addr, HTau_addr, HPinv_addr, &
                                     nrow_A, ncol_A, nnz_A, Ap, Ai, Ax, &
                                     out_addr, out_nz, status )
   import :: MF_ADDRESS, MF_DOUBLE
   integer(kind=MF_ADDRESS), intent(in) :: c_addr ! address of the cholmod_common
   integer(kind=MF_ADDRESS), intent(in) :: H_addr ! address of H
   integer(kind=MF_ADDRESS), intent(in) :: HTau_addr ! address of HTau_addr
   integer(kind=MF_ADDRESS), intent(in) :: HPinv_addr ! address of HPinv_addr
   integer, intent(in) :: nrow_A, ncol_A ! size of the matrix A
   integer, intent(in) :: nnz_A
   integer, intent(out) :: Ap(*), Ai(*) ! indexes and elements of A
   real(kind=MF_DOUBLE), intent(out) :: Ax(*) ! (CSC storage with Fortran
                                             !  indexes, i.e. 1-based)
   integer(kind=MF_ADDRESS), intent(out) :: out_addr ! address of out
   integer, intent(out) :: out_nz ! nnz(out)
   integer, intent(out) :: status ! 0: A is positive definite,
                                  ! 1: otherwise
end subroutine
!_______________________________________________________________________
!
! Free a (long int) vector
subroutine spqr_free_long_vec( A_addr )
   import :: MF_ADDRESS
   integer(kind=MF_ADDRESS), intent(in) :: A_addr ! address of vector
end subroutine
!_______________________________________________________________________
!
! Free a cholmod_common object
subroutine spqr_free_common( c_addr )
   import :: MF_ADDRESS
   integer(kind=MF_ADDRESS), intent(in) :: c_addr ! address of the cholmod_common
end subroutine
!_______________________________________________________________________
!
! Free a cholmod_dense object
subroutine spqr_free_dense( c_addr, A_addr )
   import :: MF_ADDRESS
   integer(kind=MF_ADDRESS), intent(in) :: c_addr ! address of the cholmod_common
   integer(kind=MF_ADDRESS), intent(in) :: A_addr ! address of dense A
end subroutine
!_______________________________________________________________________
!
! Free a cholmod_sparse object
subroutine spqr_free_sparse( c_addr, A_addr )
   import :: MF_ADDRESS
   integer(kind=MF_ADDRESS), intent(in) :: c_addr ! address of the cholmod_common
   integer(kind=MF_ADDRESS), intent(in) :: A_addr ! address of sparse A
end subroutine
!_______________________________________________________________________
!
end interface
