#include <fintrf.h>
! mfload_gateway.f90 - MATLAB gateway function for 'mbfread'
!
! This subroutine is the main gateway to MATLAB.  When a MEX function
! is executed MATLAB calls the MEXFUNCTION subroutine in the
! corresponding MEX file.
!
! Ref: apiext.pdf, apiref.pdf
!
! MATLAB 7.13 (R2011b) (unknown compatibility for previous versions)
! For both 32- and 64-bit machines.
!
! version: 2013-05-16
!
! Copyright É. Canot 2003-2025 -- IPR/CNRS

   subroutine mexfunction( nlhs, plhs, nrhs, prhs )

      implicit none

      mwPointer :: plhs(*), prhs(*)
      integer :: nlhs, nrhs

      mwPointer :: mxCreateDoubleMatrix, mxCreateSparse,                &
                   mxGetPr, mxGetPi, mxGetIr, mxGetJc
      mwPointer :: pr, pi, ir, jc

      mwSize :: mxGetM, mxGetN, mxGetNumberOfElements
      mwSize :: m, n, nnz, str_len
      mwSize, pointer :: ia(:), ja(:)

      integer :: mxREAL = 0, mxCOMPLEX = 1

      logical :: mxIsChar

      ! must match the main declaration in $MUESLI/src/mod_mfdebug.F90
      integer, parameter :: MF_DOUBLE = kind(1.0d0)

      integer, parameter :: len_max = 256
      character(len=len_max) :: filename, message
      real(kind=MF_DOUBLE), pointer :: double(:,:)
      complex(kind=MF_DOUBLE), pointer :: complx(:,:)
      integer :: m4, n4, nnz4
      integer, pointer :: ia4(:), ja4(:)
      real(kind=MF_DOUBLE), pointer :: ra(:)
      complex(kind=MF_DOUBLE), pointer :: ca(:)

      interface
         subroutine mbfread( filename, nrow, ncol, nnz, data_type,      &
                             double, complx, ia, ja, ra, ca )
            character(len=*), intent(in) :: filename
            integer, intent(out) :: nrow, ncol, nnz, data_type
            ! must match the main declaration in $MUESLI/src/mod_mfdebug.F90
            integer, parameter :: MF_DOUBLE = kind(1.0d0)
            real(kind=MF_DOUBLE), pointer :: double(:,:)
            complex(kind=MF_DOUBLE), pointer :: complx(:,:)
            integer, pointer :: ia(:), ja(:)
            real(kind=MF_DOUBLE), pointer :: ra(:)
            complex(kind=MF_DOUBLE), pointer :: ca(:)
         end subroutine mbfread
      end interface
!-----------------------------------------------------------------------

      ! must match the main declaration in $MUESLI/src/mod_mfarray.F90
      integer, parameter :: MF_DT_EMPTY    = 0,                         &
                            MF_DT_DBLE     = 1,                         &
                            MF_DT_CMPLX    = 2,                         &
                            MF_DT_BOOL     = 3,                         &
                            MF_DT_SP_DBLE  = 4,                         &
                            MF_DT_SP_CMPLX = 5,                         &
                            MF_DT_PERM_VEC = 6

      integer :: data_type
      logical :: exist

      ! Check for proper number of arguments
      if( nrhs /= 1 ) then
         call mexerrmsgtxt("  -> one input argument required!")
      else if( nlhs > 1 ) then
         call mexerrmsgtxt("  -> one output argument required!")
      end if

      ! Check argument
      if( .not. mxIsChar(prhs(1)) ) then
         call mexerrmsgtxt("  -> argument must be a string!")
      end if

      m = mxGetM( prhs(1) )
      n = mxGetN( prhs(1) )

      if( m /= 1 .or. n == 0 ) then
         call mexerrmsgtxt("  -> empty string!")
      end if

      str_len = mxGetNumberOfElements( prhs(1) )
      if( str_len > len_max ) then
         write(message,"(A,I0,A)") "  -> argument is too long! (max. is currently: ", &
                                   len_max, " chars)\n"
         message = trim(message)                                        &
                   // "  -> if you have to work with such a long string," &
                   // " please consider to recompile 'mssave_gateway.F90'" &
                   // " in MUESLI library."
         call mexerrmsgtxt(message)
      end if

      ! Copy the string data and place it into 'filename'
      filename = ""
      call mxGetString( prhs(1), filename, str_len )

      inquire( file=trim(filename), exist=exist )
      if( exist ) then
         ! be sure that it is not the name of a folder
         inquire( file="./"//trim(filename)//"/", exist=exist )
         if( exist ) then
            call mexerrmsgtxt("  -> name seems to be a folder!")
         end if
      else
         call mexerrmsgtxt("  -> file doesn't exist!")
      end if

      call mbfread( filename, m4, n4, nnz4, data_type,                  &
                    double, complx, ia4, ja4, ra, ca )
      m = m4
      n = n4

      if( data_type == MF_DT_DBLE ) then ! real case

         ! Create a real matrix for return argument
         plhs(1) = mxCreateDoubleMatrix( m, n, mxREAL )

         ! Assign pointer to the real part of LHS
         pr = mxGetPr( plhs(1) )
         ! Copy output to matrix output
         call mxCopyReal8ToPtr( double, pr, m*n )

         deallocate( double )

      else if( data_type == MF_DT_CMPLX ) then ! complex case

         ! Create a complex matrix for return argument
         plhs(1) = mxCreateDoubleMatrix( m, n, mxCOMPLEX )

         ! Assign pointers to the real and imag part of LHS
         pr = mxGetPr( plhs(1) )
         pi = mxGetPi( plhs(1) )
         ! Copy output to matrix output
         call mxCopyComplex16ToPtr( complx, pr, pi, m*n )

         deallocate( complx )

      else if( data_type == MF_DT_SP_DBLE ) then ! sparse real case

         nnz = nnz4

         ! Create a sparse real matrix for return argument
         plhs(1) = mxCreateSparse( m, n, nnz, mxREAL )

         ! Assign pointers to each part of LHS
         jc = mxGetJc( plhs(1) )
         ir = mxGetIr( plhs(1) )
         pr = mxGetPr( plhs(1) )
         ! Copy output to matrix output
         ! Be aware to Fortran-C offset for indexes!
         allocate( ja(n+1) )
         ja(:) = ja4(:) - 1
         deallocate( ja4 )
#ifdef _64_BITS
         call mxCopyInteger8ToPtr( ja, jc, n+1 )
#else
         call mxCopyInteger4ToPtr( ja, jc, n+1 )
#endif
         allocate( ia(nnz) )
         ia(:) = ia4(:) - 1
         deallocate( ia4 )
#ifdef _64_BITS
         call mxCopyInteger8ToPtr( ia, ir, nnz )
#else
         call mxCopyInteger4ToPtr( ia, ir, nnz )
#endif
         call mxCopyReal8ToPtr( ra, pr, nnz )

         deallocate( ia, ja, ra )

      else if( data_type == MF_DT_SP_CMPLX ) then ! sparse complex case

         nnz = nnz4

         ! Create a sparse complex matrix for return argument
         plhs(1) = mxCreateSparse( m, n, nnz, mxCOMPLEX )

         ! Assign pointers to each part of LHS
         jc = mxGetJc( plhs(1) )
         ir = mxGetIr( plhs(1) )
         pr = mxGetPr( plhs(1) )
         pi = mxGetPi( plhs(1) )
         ! Copy output to matrix output
         ! Be aware to Fortran-C offset for indexes!
         allocate( ja(n+1) )
         ja(:) = ja4(:) - 1
         deallocate( ja4 )
#ifdef _64_BITS
         call mxCopyInteger8ToPtr( ja, jc, n+1 )
#else
         call mxCopyInteger4ToPtr( ja, jc, n+1 )
#endif
         allocate( ia(nnz) )
         ia(:) = ia4(:) - 1
         deallocate( ia4 )
#ifdef _64_BITS
         call mxCopyInteger8ToPtr( ia, ir, nnz )
#else
         call mxCopyInteger4ToPtr( ia, ir, nnz )
#endif
         call mxCopyComplex16ToPtr( ca, pr, pi, nnz )

         deallocate( ia, ja, ca )

      else if( data_type == MF_DT_PERM_VEC ) then ! integer permutation vector

         ! Copy integer indices to double
         allocate( ra(m) )
         ra(:) = ia4(:)

         ! Create a row real vector for return argument
         plhs(1) = mxCreateDoubleMatrix( n, m, mxREAL )

         ! Assign pointer to the real part of LHS
         pr = mxGetPr( plhs(1) )
         ! Copy output to matrix output
         call mxCopyReal8ToPtr( ra, pr, m )

         deallocate( ra )

      end if

   end subroutine mexfunction
