! f90 include file

!_______________________________________________________________________
!
   subroutine build_tri_conn( n_xy, tri_n,                              &
                              face_n, face_tri, tri_f, n_tri,           &
                              check_tri_orientation, tri_modified,      &
                              nb_disconn_tri, tri_renumbering,          &
                              equil_face_orientation )

      double precision, intent(in) :: n_xy(:,:)
      integer, pointer :: tri_n(:,:)
      integer, pointer :: face_n(:,:)
      integer, pointer :: face_tri(:,:)
      integer, pointer :: tri_f(:,:)
      integer, pointer :: n_tri(:)
      integer          :: nb_disconn_tri
      logical, intent(in)  :: check_tri_orientation
      logical, intent(out) :: tri_modified
      logical              :: tri_renumbering, equil_face_orientation
      !------ API end ------

#ifdef _DEVLP
      ! Build a Triangle Connectivity structure (and more, actually)
      ! for searching in a 2D triangulation.
      !
      ! Input arrays are n_xy (nodes' coordinates) and tri_n (nodes of
      ! each triangle). According to other boolean flags (see below),
      ! the table tri_n may be modified.
      !
      ! Output arrays are:
      !   face_n   (couple of nodes for each face),
      !   face_tri (one or two triangles attached to a face),
      !   tri_f    (faces of each triangle),
      !   n_tri    (one triangle connected to a given node).
      !
      ! If tri_renumbering is TRUE then it mades also the renumbering of
      ! the triangles, in such a way that they are, if possible, connected
      ! together for consecutive indices.
      !
      ! If equil_face_orientation is TRUE then the two nodes describing
      ! each face may be swapped, in order to have, if possible, equilibrated
      ! arrival of faces at each nodes. nb_disconn_tri is then the number
      ! of disconnected triangles. In practice, this option is not very
      ! useful...
      !--------------------------------------------------------------------

      integer :: nn, ne, i, j, k, nf, nf_max, i1, i2, iface, inode
      integer, allocatable :: face_n_0(:,:)
      integer, allocatable :: face_tri_0(:,:)

      logical, allocatable :: face_checked(:)
      integer :: f1, f2, f3, i_start, itmp, i_prev, num_1, num_2, face
      integer :: i_tri, j_face, k_tri, nb_voisins, i_tri_new, i_tri_new_new
      integer :: tab(4), i_perm
      integer :: jj
      logical :: found, zero_found
      integer, allocatable :: tri_tri(:,:), perm(:), inv_perm(:)

      type(mf_Int_List), allocatable :: nn_conn_vec(:)
      integer, allocatable :: nn_conn_nb(:)

   !------ end of declarations -- execution starts hereafter  ------

      nn = size(n_xy,1)
      ne = size(tri_n,1)

      tri_modified = .false.
      call check_triangle_orientation()

      ! Secure evaluation for the total number of faces.
      ! (worst case: all triangles are disconnected from each other)
      nf_max = 3*ne
      allocate( face_n_0(nf_max,2) )
      ! No need to initialize face_n to zero.

      allocate( face_tri_0(nf_max,2) )
      ! Very important (0 means that face is at a boundary)
      face_tri_0(:,:) = 0

      allocate( tri_f(ne,3) )

      allocate( nn_conn_vec(nn), nn_conn_nb(nn) )
      nn_conn_nb(:) = 0
      ! Estimate node-node numbers (number slightly overestimated)
      do i = 1, ne
         do j = 1, 3
            i1 = tri_n(i,j)
            if( nn_conn_nb(i1) == 0 ) then
               nn_conn_nb(i1) = 2
            else
               nn_conn_nb(i1) = nn_conn_nb(i1) + 1
            end if
         end do
      end do
      do i = 1, nn
         ! The 2 factor is needed because, below, we will store in it
         ! pairs of integers (one node, one face).
         j = 2*nn_conn_nb(i)
         allocate( nn_conn_vec(i)%list(j) )
      end do
      nn_conn_nb(:) = 0

      nf = 0
      do i = 1, ne ! loop over all triangles
         ! face 1-2
         i1 = tri_n(i,1)
         i2 = tri_n(i,2)
         iface = find_in_nn_conn_vec( nn_conn_vec, nn_conn_nb, i1, i2 )
         if( iface == 0 ) then
            ! This is an anonymous face, so we have to record it.
            nf = nf + 1
            face_n_0(nf,1) = i1
            face_n_0(nf,2) = i2
            face_tri_0(nf,1) = i
            tri_f(i,1) = nf
            nn_conn_nb(i1) = nn_conn_nb(i1) + 1
            nn_conn_vec(i1)%list(2*nn_conn_nb(i1)-1) = i2
            nn_conn_vec(i1)%list(2*nn_conn_nb(i1)) = nf
         else
            ! This is a known face.
            face_tri_0(iface,2) = i
            tri_f(i,1) = iface
         end if
         ! face 2-3
         i1 = tri_n(i,2)
         i2 = tri_n(i,3)
         iface = find_in_nn_conn_vec( nn_conn_vec, nn_conn_nb, i1, i2 )
         if( iface == 0 ) then
            nf = nf + 1
            face_n_0(nf,1) = i1
            face_n_0(nf,2) = i2
            face_tri_0(nf,1) = i
            tri_f(i,2) = nf
            nn_conn_nb(i1) = nn_conn_nb(i1) + 1
            nn_conn_vec(i1)%list(2*nn_conn_nb(i1)-1) = i2
            nn_conn_vec(i1)%list(2*nn_conn_nb(i1)) = nf
         else
            face_tri_0(iface,2) = i
            tri_f(i,2) = iface
         end if
         ! face 3-1
         i1 = tri_n(i,3)
         i2 = tri_n(i,1)
         iface = find_in_nn_conn_vec( nn_conn_vec, nn_conn_nb, i1, i2 )
         if( iface == 0 ) then
            nf = nf + 1
            face_n_0(nf,1) = i1
            face_n_0(nf,2) = i2
            face_tri_0(nf,1) = i
            tri_f(i,3) = nf
            nn_conn_nb(i1) = nn_conn_nb(i1) + 1
            nn_conn_vec(i1)%list(2*nn_conn_nb(i1)-1) = i2
            nn_conn_vec(i1)%list(2*nn_conn_nb(i1)) = nf
         else
            face_tri_0(iface,2) = i
            tri_f(i,3) = iface
         end if
      end do

      ! Now that face_n_0 and face_tri_0 are filled, we must copy them
      ! in smaller arrays, to economize memory...

      allocate( face_n(nf,2) )
      face_n(:,:) = face_n_0(1:nf,:)
      deallocate( face_n_0 )
      allocate( face_tri(nf,2) )
      face_tri(:,:) = face_tri_0(1:nf,:)
      deallocate( face_tri_0 )

      if( tri_renumbering ) then

         ! Si requis lors de l'appel (tri_numbering est optionnel dans
         ! 'msBuildTriConnect'), on effectue une renumérotation des triangles,
         ! de manière à ce qu'il soient le plus possible contigus quand on
         ! parcourt la liste.

         ! L'algorithme suivant a été mis au point par moi-même (EC), en
         ! m'inspirant du parcours à main droite d'un labyrinthe... ici,
         ! on parcours les triangles en gardant une frontière à main droite.

         ! Pour un accès facilité aux triangles voisins d'un triangle donné,
         ! le tableau tri_tri(ne,5) est temporaire; pour chaque ligne
         ! (un triangle donné), les trois premiers indices indiquent les
         ! triangles voisins, suivant une orientation directe. Le 4e indice
         ! donne le nb total de voisins. Le 5e indice indique si le triangle
         ! est libre (1) ou déjà pris (0)

         allocate( tri_tri(ne,5) )
         do i = 1, ne
            nb_voisins = 0
            k = 0
            do j = 1, 3
               face = tri_f(i,j)
               if( face_tri(face,1) == i ) then
                  iface = face_tri(face,2)
                  if( iface == 0 ) then
                     k = k + 1
                     tri_tri(i,k) = 0
                  else ! iface /= 0
                     k = k + 1
                     tri_tri(i,k) = iface
                     nb_voisins = nb_voisins + 1
                  end if
               else ! face_tri(face,1) /= i
                  ! pas de test à faire : face_tri(face,1) n'est jamais égal à 0
                  k = k + 1
                  tri_tri(i,k) = face_tri(face,1)
                  nb_voisins = nb_voisins + 1
               end if
            end do
            tri_tri(i,4) = nb_voisins
            tri_tri(i,5) = 1
         end do

         ! Algorithme de renumérotation
         !   i_tri  : numéro du triangle courant
         ! Départ : on suppose qu'il est au bord; sinon, il faudrait chercher
         ! un triangle de bord (ou plutôt, le triangle ayant le nb minimum de
         ! voisins).

         i_perm = 0
         allocate( perm(ne) )
         perm(:) = 0

         ! Recherche d'un triangle ayant un faible nb de voisins...
         i_tri = 1
         do i = 2, ne
            if( tri_tri(i,4) < tri_tri(i_tri,4) ) i_tri = i
         end do
         i_perm = i_perm + 1
         perm(i_perm) = i_tri

         do ! Boucle sur les chemins...

            do
               ! Comme le triangle est sélectionné, j'efface la référence i_tri
               ! de tri_tri(?,1:3), et je mets à jour le nb de voisins
               do j = 1, 3
                  k_tri = tri_tri(i_tri,j)
                  if( k_tri == 0 ) cycle
                  do k = 1, 3
                     if( tri_tri(k_tri,k) == i_tri ) then
                        tri_tri(k_tri,k) = 0
                        tri_tri(k_tri,4) = tri_tri(k_tri,4) - 1
                        exit
                     end if
                  end do
               end do
               ! On retire le triangle
               tri_tri(i_tri,5) = 0

               ! Triangle suivant : premier voisin non nul (s'il existe)
               ! qui suit un 0
               zero_found = .false.
               i_tri_new = 0
               tab = [ 1, 2, 3, 1 ]
               do jj = 1, 4
                  j = tab(jj)
                  if( tri_tri(i_tri,j) == 0 ) then
                     zero_found = .true.
                  else
                     if( zero_found ) then
                        i_tri_new = tri_tri(i_tri,j)
                        ! est-ce un cul-de-sac ?
                        if( tri_tri(i_tri_new,4) == 0 ) then
                           ! y a-t-il un autre voisin à i_tri ?
                           j = j + 1; if( j > 3 ) j = j - 3
                           i_tri_new_new = tri_tri(i_tri,j)
                           ! est-il valide ?
                           if( i_tri_new_new /= 0 ) then
                              if( tri_tri(i_tri_new_new,5) == 1 ) then
                                 i_tri_new = i_tri_new_new
                              end if
                           end if
                        end if
                        exit
                     end if
                  end if
               end do
               if( i_tri_new == 0 ) then
                  exit
               else
                  i_tri = i_tri_new
               end if
               i_perm = i_perm + 1
               perm(i_perm) = i_tri

            end do

            ! Recherche d'un triangle (nouveau départ) ayant un faible nb
            ! de voisins...
            found = .false.
            do i = 1, ne
               if( tri_tri(i,5) == 1 ) then
                  i_tri = i
                  found = .true.
                  exit
               end if
            end do
            if( found ) then
               do i = i_tri+1, ne
                  if( tri_tri(i,5) == 1 ) then
                     if( tri_tri(i,4) < tri_tri(i_tri,4) ) i_tri = i
                  end if
               end do
               i_perm = i_perm + 1
               perm(i_perm) = i_tri
            else
               exit
            end if

         end do

         if( i_perm /= ne ) then
            write(STDERR,*)
            write(STDERR,*) "(MUESLI build_tri_conn:) internal error:"
            write(STDERR,*) "        Severe error: i_perm /= ne"
            write(STDERR,*) "        Please report this bug to: Edouard.Canot@univ-rennes.fr"
            mf_message_displayed = .true.
            call muesli_trace( pause ="yes" )
            stop
         end if

         ! On applique maintenant la permutation aux tableaux tri_?
         tri_f(:,:) = tri_f(perm,:)
         tri_n(:,:) = tri_n(perm,:)
         ! On inverse la permutation
         allocate( inv_perm(ne) )
         inv_perm(perm) = [ ( i, i = 1, ne ) ]
         do j = 1, nf
            k = face_tri(j,1)
            face_tri(j,1) = inv_perm(k)
            k = face_tri(j,2)
            if( k > 0 ) then
               face_tri(j,2) = inv_perm(k)
            end if
         end do

      end if ! tri_renumbering

      ! Building 'n_tri' array : one triangle (any) for each node
      !                          (useful for 'tsearch', see below)
      allocate( n_tri(nn) )
      do k = 1, 3
         do i = 1, ne
            inode = tri_n(i,k)
            n_tri(inode) = i
         end do
      end do

      if( equil_face_orientation ) then

         ! In order to check each face only once
         allocate( face_checked(nf) )
         face_checked(:) = .false.

         nb_disconn_tri = 0

         do i = 1, ne

            ! Get faces of triangle i
            f1 = tri_f(i,1)
            f2 = tri_f(i,2)
            f3 = tri_f(i,3)

            ! First, find a checked face.
            if( face_checked(f1) ) then
               i_start = 1
            else if( face_checked(f2) ) then
               i_start = 2
            else if( face_checked(f3) ) then
               i_start = 3
            else
               if( i /= 1 ) then
                  ! Disconnected triangle
                  nb_disconn_tri = nb_disconn_tri + 1
               end if
               i_start = 1
               face_checked(f1) = .true.
            end if

            ! Now, the next face which is unchecked.
            found = .false.
            do k = 1, 2
               j = i_start + k
               if( j > 3 ) j = j - 3
               if( .not. face_checked(tri_f(i,j)) ) then
                  found = .true.
                  exit
               end if
            end do

            if( .not. found ) cycle ! all faces are checked

            ! Check now face j which must have same orientation as
            ! the previous one.
            i_prev = j - 1
            if( i_prev < 1 ) i_prev = i_prev + 3
            num_1 = tri_f(i,i_prev)
            num_2 = tri_f(i,j)
            if( face_n(num_1,2) == face_n(num_2,1) .or.                 &
               face_n(num_1,1) == face_n(num_2,2) ) then
               ! Nothing to do
            else
               ! Change face orientation
               itmp = face_n(num_2,1)
               face_n(num_2,1) = face_n(num_2,2)
               face_n(num_2,2) = itmp
            end if
            face_checked(num_2) = .true.

            ! Next face?
            j = j + 1
            if( j > 3 ) j = j - 3
            i_prev = j - 1
            if( i_prev < 1 ) i_prev = i_prev + 3
            num_1 = tri_f(i,i_prev)
            num_2 = tri_f(i,j)
            if( .not. face_checked(num_2) ) then
               if( face_n(num_1,2) == face_n(num_2,1) .or.              &
                  face_n(num_1,1) == face_n(num_2,2) ) then
                  ! Nothing to do
               else
                  ! Change face orientation
                  itmp = face_n(num_2,1)
                  face_n(num_2,1) = face_n(num_2,2)
                  face_n(num_2,2) = itmp
               end if
               face_checked(num_2) = .true.
            end if

         end do

      end if ! equil_face_orientation

   contains

      subroutine check_triangle_orientation()

         integer :: i, n2, n3
         double precision :: x1, y1, x2, y2, x3, y3, u1, v1, u2, v2,    &
                             cross_product

         do i = 1, ne ! loop over triangles

            x1 = n_xy(tri_n(i,1),1)
            y1 = n_xy(tri_n(i,1),2)
            x2 = n_xy(tri_n(i,2),1)
            y2 = n_xy(tri_n(i,2),2)
            x3 = n_xy(tri_n(i,3),1)
            y3 = n_xy(tri_n(i,3),2)
            u1 = x2 - x1
            v1 = y2 - y1
            u2 = x3 - x2
            v2 = y3 - y2
            cross_product = u1*v2 - u2*v1
            if( cross_product < 0.0d0 ) then
               n2 = tri_n(i,2)
               n3 = tri_n(i,3)
               tri_n(i,2) = n3
               tri_n(i,3) = n2
               tri_modified = .true.
            end if

         end do

      end subroutine check_triangle_orientation
#endif
   end subroutine build_tri_conn
!_______________________________________________________________________
!
   subroutine build_tri_conn_min( nn, tri_n, face_tri, tri_f )

      integer, intent(in)  :: nn, tri_n(:,:)
      integer, allocatable :: face_tri(:,:), tri_f(:,:)
      !------ API end ------

#ifdef _DEVLP
      ! Build a partial Triangle Connectivity.
      !
      ! Input array is tri_n (nodes of each triangle).
      !
      ! The following arrays are built:
      !   face_n   (couple of nodes for each face), for internal use
      !   face_tri (one or two triangles attached to a face),
      !   tri_f    (faces of each triangle),
      ! but only the two latters are returned.
      !--------------------------------------------------------------------

      integer :: ne, i, j, nf, nf_max, i1, i2, iface
      integer, allocatable :: face_n(:,:)
      integer, allocatable :: face_tri_0(:,:) ! tempo for face_tri(:,:)

      type(mf_Int_List), allocatable :: nn_conn_vec(:)
      integer, allocatable :: nn_conn_nb(:)

   !------ end of declarations -- execution starts hereafter  ------

      ne = size(tri_n,1)

      ! Secure evaluation for the total number of faces.
      ! (worst case: all triangles are disconnected from each other)
      nf_max = 3*ne
      allocate( face_n(nf_max,2) )
      ! No need to initialize face_n to zero.

      allocate( face_tri_0(nf_max,2) )
      ! Very important (0 means that face is at a boundary)
      face_tri_0(:,:) = 0

      allocate( tri_f(ne,3) )

      allocate( nn_conn_vec(nn), nn_conn_nb(nn) )
      nn_conn_nb(:) = 0
      ! Estimate node-node numbers (number slightly overestimated)
      do i = 1, ne
         do j = 1, 3
            i1 = tri_n(i,j)
            if( nn_conn_nb(i1) == 0 ) then
               nn_conn_nb(i1) = 2
            else
               nn_conn_nb(i1) = nn_conn_nb(i1) + 1
            end if
         end do
      end do
      do i = 1, nn
         ! The 2 factor is needed because, below, we will store in it
         ! pairs of integers (one node, one face).
         j = 2*nn_conn_nb(i)
         allocate( nn_conn_vec(i)%list(j) )
      end do
      nn_conn_nb(:) = 0

      nf = 0
      do i = 1, ne ! loop over all triangles
         ! face 1-2
         i1 = tri_n(i,1)
         i2 = tri_n(i,2)
         iface = find_in_nn_conn_vec( nn_conn_vec, nn_conn_nb, i1, i2 )
         if( iface == 0 ) then
            ! This is an anonymous face, so we have to record it.
            nf = nf + 1
            face_n(nf,1) = i1
            face_n(nf,2) = i2
            face_tri_0(nf,1) = i
            tri_f(i,1) = nf
            nn_conn_nb(i1) = nn_conn_nb(i1) + 1
            nn_conn_vec(i1)%list(2*nn_conn_nb(i1)-1) = i2
            nn_conn_vec(i1)%list(2*nn_conn_nb(i1)) = nf
         else
            ! This is a known face.
            face_tri_0(iface,2) = i
            tri_f(i,1) = iface
         end if
         ! face 2-3
         i1 = tri_n(i,2)
         i2 = tri_n(i,3)
         iface = find_in_nn_conn_vec( nn_conn_vec, nn_conn_nb, i1, i2 )
         if( iface == 0 ) then
            nf = nf + 1
            face_n(nf,1) = i1
            face_n(nf,2) = i2
            face_tri_0(nf,1) = i
            tri_f(i,2) = nf
            nn_conn_nb(i1) = nn_conn_nb(i1) + 1
            nn_conn_vec(i1)%list(2*nn_conn_nb(i1)-1) = i2
            nn_conn_vec(i1)%list(2*nn_conn_nb(i1)) = nf
         else
            face_tri_0(iface,2) = i
            tri_f(i,2) = iface
         end if
         ! face 3-1
         i1 = tri_n(i,3)
         i2 = tri_n(i,1)
         iface = find_in_nn_conn_vec( nn_conn_vec, nn_conn_nb, i1, i2 )
         if( iface == 0 ) then
            nf = nf + 1
            face_n(nf,1) = i1
            face_n(nf,2) = i2
            face_tri_0(nf,1) = i
            tri_f(i,3) = nf
            nn_conn_nb(i1) = nn_conn_nb(i1) + 1
            nn_conn_vec(i1)%list(2*nn_conn_nb(i1)-1) = i2
            nn_conn_vec(i1)%list(2*nn_conn_nb(i1)) = nf
         else
            face_tri_0(iface,2) = i
            tri_f(i,3) = iface
         end if
      end do

      ! Now that face_tri_0 is filled, we can copy it in a smaller
      ! array, to economize memory...
      allocate( face_tri(nf,2) )
      face_tri(:,:) = face_tri_0(1:nf,:)
      deallocate( face_tri_0 )

#endif
   end subroutine build_tri_conn_min
!_______________________________________________________________________
!
   subroutine partition_in_disjoint_zones( tri_f, face_tri,             &
                                           new_tri, new_tri_ptr )

      integer, intent(in) :: tri_f(:,:), face_tri(:,:)
      integer, allocatable :: new_tri(:), new_tri_ptr(:,:)
      !------ API end ------

#ifdef _DEVLP

      integer, allocatable :: new_tri_ptr_0(:,:)
      integer, allocatable :: from(:), ind(:)

      integer :: nt, nf, it, it_next, it_start, k, zone,                &
                 ind_1, ind_2, last
      logical :: found

   !------ end of declarations -- execution starts hereafter  ------

      nt = size(tri_f,1)
      nf = size(face_tri,1)

      allocate( from(nt), ind(nt) )
      from(:) = 0 ; ind(:) = 0

      ! Size of new_tri_ptr_0 is, of course, overestimated!
      ! (we will copy it in a smaller array at the end of the routine)
      allocate( new_tri(nt), new_tri_ptr_0(nt,2) )
      new_tri(:) = 0

      ! The relation between neighboring triangles is considered as a tree,
      ! in which each node has three sons.
      ! A starting point is arbitrarily chosen and we have to walk into the
      ! whole tree. Each triangle visited is tagged.
      it_start = 1
      zone = 1
      do

         from(it_start) = 0

         if( zone == 1 ) then
            last = 0
         else
            last = new_tri_ptr_0(zone-1,2)
         end if
         k = 1
         new_tri(last+k) = it_start

         it = it_start
         do while( ind(it_start) /= -1 )
            call increment_ind()
            if( ind(it) == -1 ) then
               it = from(it)
            else
               it_next = next_neighbor()
               if( it_next /= 0 ) then
                  if( ind(it_next) == 0 ) then
                     from(it_next) = it
                     it = it_next
                     k = k + 1
                     new_tri(last+k) = it
                  end if
               end if
            end if
         end do

         ind_1 = last + 1
         ind_2 = last + k
         new_tri_ptr_0(zone,:) = [ ind_1, ind_2 ]

         where( ind == -1 )
            ind = -2
         end where

         ! Next zone (if it exists) begins at the first triangle not yet
         ! visited...
         found = .false.
         do it = 1, nt
            if( ind(it) == 0 ) then
               found = .true.
               exit
            end if
         end do
         if( .not. found ) then
            exit
         end if

         zone = zone + 1
         it_start = it

      end do

      allocate( new_tri_ptr(zone,2) )
      new_tri_ptr(:,:) = new_tri_ptr_0(1:zone,:)

   contains
      !____________________________________________________________________
      !
      subroutine increment_ind( )
         ind(it) = ind(it) + 1
         if( ind(it) > 3 ) ind(it) = -1
      end subroutine increment_ind
      !____________________________________________________________________
      !
      function next_neighbor( ) result( res )
         integer :: res
         integer :: face
         face = tri_f(it,ind(it))
         res = face_tri(face,1)
         if( res == it ) then
            res = face_tri(face,2)
         end if
      end function next_neighbor
      !____________________________________________________________________
      !
#endif
   end subroutine partition_in_disjoint_zones
!_______________________________________________________________________
!
   function find_in_face_n( face_n, nf, i1, i2 ) result( iface )

      integer, intent(in) :: face_n(:,:), nf, i1, i2
      integer :: iface
      !------ API end ------

#ifdef _DEVLP
      ! slow version

      integer :: i

   !------ end of declarations -- execution starts hereafter  ------

      do i = 1, nf
         if( face_n(i,1) == i1 ) then
            if( face_n(i,2) == i2 ) then
               iface = i ! found
               return
            else
               cycle
            end if
         else if( face_n(i,1) == i2 ) then
            if( face_n(i,2) == i1 ) then
               iface = i ! found
               return
            else
               cycle
            end if
         else
            cycle
         end if
      end do

      iface = 0 ! not found

#endif
   end function find_in_face_n
!_______________________________________________________________________
!
   function find_in_nn_conn_vec( nn_conn_vec, nn_conn_nb, i1, i2 )      &
   result( iface )

      type(mf_Int_List), intent(in) :: nn_conn_vec(:)
      integer,           intent(in) :: nn_conn_nb(:)
      integer,           intent(in) :: i1, i2
      integer                       :: iface
      !------ API end ------

#ifdef _DEVLP
      ! fast version

      integer :: ip

   !------ end of declarations -- execution starts hereafter  ------

      ! try to find (i1,i2)
      do ip = 1, nn_conn_nb(i1)
         if( nn_conn_vec(i1)%list(2*ip-1) == i2 ) then
            iface = nn_conn_vec(i1)%list(2*ip)
            return
         end if
      end do

      ! then, try to find (i2,i1)
      do ip = 1, nn_conn_nb(i2)
         if( nn_conn_vec(i2)%list(2*ip-1) == i1 ) then
            iface = nn_conn_vec(i2)%list(2*ip)
            return
         end if
      end do

      iface = 0 ! not found

#endif
   end function find_in_nn_conn_vec
!_______________________________________________________________________
!
   function tsearch( convex_domain, n_xy, tri_n, face_tri, tri_f,       &
                     n_tri, face_n, faces_boundary, faces_boundary_ptr, &
                     xi, yi, force_inside )                                     &
   result( num )

      integer,          intent(in out) :: convex_domain
      double precision, intent(in)     :: n_xy(:,:)
      integer,          intent(in)     :: tri_n(:,:)
      integer,          intent(in out) :: face_tri(:,:)
      integer,          intent(in)     :: tri_f(:,:)
      integer,          intent(in)     :: n_tri(:)
      integer,          intent(in)     :: face_n(:,:)
      double precision, intent(in)     :: xi, yi
      integer, pointer :: faces_boundary(:), faces_boundary_ptr(:,:)
      logical,          intent(in), optional :: force_inside
      integer :: num
      !------ API end ------

#ifdef _DEVLP
      ! 'force_inside' is an optional argument; when it is set to TRUE,
      ! it is supposed that the searched point is inside the domain, and
      ! that perhaps roundoff errors may put it slightly outside. In this
      ! latter case, the search is always successful and return the best
      ! triangle. (for example, it is used by the 'compute_ep_abs_curv_2'
      ! routine in TriContourF_aux.f90)

      integer :: k, k1, k2, face, face_glob, f2, kk
      integer :: num_save
      logical :: force_inside_0

      integer, save :: num_previous = 1

   !------ end of declarations -- execution starts hereafter  ------

      if( convex_domain == -1 ) then
         ! First search for the current mesh, therefore we must check
         ! its convexity...
         call check_tri_convexity( convex_domain, n_xy, tri_n, face_tri, &
                                   tri_f, n_tri, face_n,                &
                                   faces_boundary, faces_boundary_ptr )
         num_previous = 1
      end if
      num = num_previous

      if( present(force_inside) ) then
         force_inside_0 = force_inside
      else
         force_inside_0 = .false.
      end if

      if( convex_domain == 1 ) then

         do
            ! Does the current triangle contain the final point?
            face = is_in_triangle( n_xy, tri_n, num, xi, yi )
            if( face == 0 ) then
               ! Triangle found
               num_previous = num
               return
            else
               num_save = num
               num = adj_triangle( face_tri, tri_f, num, face )
               if( num <= 0 ) then
                  ! Triangle not found
if( force_inside_0 ) then
!### TODO: en fait toute face d'un bord rectiligne sera ok si le point
!          est très légèrement en dehors. Il faut alors faire des tests
!          supplémentaires, en employant la liste des faces de la frontière !
!          (voir comment j'ai fait dans 'compute_ep_abs_curv_2')
print *, "tsearch: internal error"
print *, "         tri.inc, line 867: to be improved! not finished!"
pause "for debugging purpose only..."
stop
   num_previous = num_save
   num = num_save
end if
                  return
               end if
            end if
         end do

      else ! Domain is not convex

         do
            ! Does the current triangle contain the final point?
            face = is_in_triangle( n_xy, tri_n, num, xi, yi )
            if( face == 0 ) then
               ! Triangle found
               num_previous = num
               return
            else
               face_glob = tri_f(num,face)
               num_save = num
               num = adj_triangle( face_tri, tri_f, num, face )
               if( num <= 0 ) then
                  ! Search for the boundary num
                  call boundary_number_tri_mesh( face_glob, k, k1, k2 )
                  ! Local concavity: find another face of the same boundary,
                  ! closer to the target point.
                  kk = boundary_closer_face( k, k1, k2 )
                  if( kk == k ) then
                     ! Triangle not found
if( force_inside_0 ) then
   num_previous = num_save
   num = num_save
else
                     ! By construction during convexity check, the first
                     ! concavity on the exterior boundary is set to -1
                     ! (i.e. other exterior faces remain to 0), and other
                     ! concavities found lead to negative numbers (-2 for
                     ! the first hole, -3 for the second hole, etc.)
                     num = num + 1
end if
                     return
                  end if
                  ! From 'kk', find the valid triangle
                  face = faces_boundary(kk)
                  num = face_tri(face,1)
               end if
            end if
         end do

      end if

   contains
   !____________________________________________________________________
   !
      subroutine boundary_number_tri_mesh( face_glob, k, k1, k2 )

         integer, intent(in) :: face_glob
         integer :: k, k1, k2
         !------ API end ------

         ! Locate the face num 'face_glob' in the 'faces_boundary' array,
         ! 'k' is the position in this array, while 'k1' and 'k2' are the
         ! extreme position for the faces of the same boundary.

         integer :: i
         logical :: found

         found = .false.
         do i = 1, size(faces_boundary)
            if( faces_boundary(i) == face_glob ) then
               found = .true.
               exit
            end if
         end do
         k = i
         do i = 1, size(faces_boundary_ptr)
            k1 = faces_boundary_ptr(i,1)
            k2 = faces_boundary_ptr(i,2)
            if( k1 <= k .and. k <= k2 ) return
         end do

         write(STDERR,*)
         write(STDERR,*) "(MUESLI [tsearch]/boundary_number_tri_mesh:) internal error:"
         write(STDERR,"(A,I0,A)") "        Severe error: didn't find the global face ", face_glob, " in the array 'faces_boundary'."
         write(STDERR,*) "        Please report this bug to: Edouard.Canot@univ-rennes.fr"
         mf_message_displayed = .true.
         call muesli_trace( pause ="yes" )
         stop

      end subroutine boundary_number_tri_mesh
   !____________________________________________________________________
   !
      function boundary_closer_face( k, k1, k2 ) result( res )

         integer, intent(in) :: k, k1, k2
         integer             :: res
         !------ API end ------

         ! Locate, in the global list of all boundary faces, the closer
         ! face to (xi,yi), for the specific boundary k1-k2.
         ! If the same value k is returned, this means that no other faces
         ! are closer.

         integer :: i, f, n1, n2, i_min
         double precision :: d2, d2_min, mx, my

         ! current distance to the target point.
         f = faces_boundary(k)
         n1 = face_n(f,1)
         n2 = face_n(f,2)
         mx = 0.5d0*(n_xy(n1,1)+n_xy(n2,1))
         my = 0.5d0*(n_xy(n1,2)+n_xy(n2,2))
         d2_min = (xi-mx)**2 + (yi-my)**2
         i_min = k

         do i = k1, k2
            f = faces_boundary(i)
            ! coordinates of the middle of the face
            n1 = face_n(f,1)
            n2 = face_n(f,2)
            mx = 0.5d0*(n_xy(n1,1)+n_xy(n2,1))
            my = 0.5d0*(n_xy(n1,2)+n_xy(n2,2))
            d2 = (xi-mx)**2 + (yi-my)**2
            if( d2 < d2_min ) then
               d2_min = d2
               i_min = i
            end if
         end do
         res = i_min

      end function boundary_closer_face
   !____________________________________________________________________
   !
#endif
   end function tsearch
!_______________________________________________________________________
!
   subroutine check_tri_convexity( convex_domain, n_xy, tri_n, face_tri, &
                                   tri_f, n_tri, face_n,                &
                                   faces_boundary, faces_boundary_ptr )

      integer,          intent(in out) :: convex_domain
      double precision, intent(in)     :: n_xy(:,:)
      integer,          intent(in)     :: tri_n(:,:), n_tri(:)
      integer,          intent(in out) :: face_tri(:,:)
      integer,          intent(in)     :: tri_f(:,:), face_n(:,:)
      integer, pointer :: faces_boundary(:), faces_boundary_ptr(:,:)
      !------ API end ------

#ifdef _DEVLP
      integer, allocatable :: ind(:), node_hull(:), bound_list(:,:)

      ! These arrays are tempo, before copying permanently in the
      ! 'faces_boundary' and 'faces_boundary_ptr' arrays.
      integer, allocatable :: fb(:), fb_ptr(:,:)

      integer:: i, j, m, n, nh, nbl, j_prev, k, nb, node_1, node_2
      integer :: node_start, node, next_node, face_start, face, next_face
      double precision :: aligned

      integer, allocatable :: smallest_node(:)
      integer :: j1, j2
      logical :: incr_order

   !------ end of declarations -- execution starts hereafter  ------

      n = size(n_xy,1)

      ! Construct node list of the Convex Hull

      !   1 - get the most-left node
      j = 1
      do i = 2, n
         if( CH_isLeftOf( n_xy(i,1), n_xy(i,2),                         &
                          n_xy(j,1), n_xy(j,2) ) ) then
            j = i
         end if
      end do
      node_start = j

      !   2 - compute the convex hull (ind is a tempo array)
      allocate( ind(n) )
      call convex_hull_from_connectivity( n_xy, tri_n, tri_f, face_tri, &
                                          face_n, n_tri, node_start,    &
                                          ind, nh )

      !   3 - copy the convex hull nodes in a shorter list
      allocate( node_hull(nh) )
      node_hull(1:nh) = ind(1:nh)
      deallocate( ind )

      ! Prepare the permanent storage of faces of all boundaries, i.e.
      ! the exterior boundary and all holes inside it.
      !   'fb' contains all the faces' indices, grouped by independant
      !   boundaries (first the exterior boundary, then the holes);
      !   'fb_ptr' contains the indices to delimitate the different holes
      !   in the previous array, the first index is the beginning of each
      !   hole, the second one is its end.

      ! Estimation of the number of all boundary faces... let's take
      ! the total number of faces (no better estimation).
      m = size(face_tri,1)
      allocate( fb(m), fb_ptr(m/3,2) )
      fb_ptr(:,:) = 0

      ! Build the node and face list for the exterior boundary
      !   bound_list(:,1) : list of nodes
      !   bound_list(:,2) : list of faces
      ! Orientation: Counter-Clockwise

      allocate( bound_list(m,2) )
      bound_list(:,:) = 0
      bound_list(1,1) = node_start
      node = node_start
      ! Let's also determine the starting face
      call next_face_on_boundary( tri_n, tri_f, face_tri, n_tri,        &
                                  node, face )
      face_start = face
      bound_list(1,2) = face
      nbl = 1
      do
         call next_node_and_face_on_boundary( tri_n, tri_f, face_tri, face_n, &
                                              node, face, next_node, next_face )
         if( next_node == node_start .and. next_face == face_start ) exit
         nbl = nbl + 1
         bound_list(nbl,1) = next_node
         bound_list(nbl,2) = next_face
         node = next_node
         face = next_face
      end do

      fb(1:nbl) = bound_list(1:nbl,2)
      fb_ptr(1,1) = 1
      fb_ptr(1,2) = nbl

      ! Set all faces (no neighbor triangle) to a very large negative
      ! value instead of 0.
      where( face_tri(:,2) == 0 )
         face_tri(:,2) = -huge(0)
      end where
      ! All faces of exterior boundary are set to zero.
      face_tri(bound_list(1:nbl,2),2) = 0

      ! Simultaneous travel of nodes of the convex hull and faces of the
      ! boundary.
      j = 1
      do i = 1, nh-1
         node_1 = node_hull(i)
         node_2 = node_hull(i+1)
         do
            node = bound_list(j,1)
            if( node == node_2 ) then
               exit
            else
               if( node /= node_1 ) then
                  aligned = CH_ccw( n_xy(node_1,1), n_xy(node_1,2),     &
                                    n_xy(node,1), n_xy(node,2),         &
                                    n_xy(node_2,1), n_xy(node_2,2) )
                  if( aligned > 0.0d0 ) then
                     face_tri(bound_list(j-1,2),2) = -1
                     face_tri(bound_list(j,2),2)   = -1
                  end if
               end if
               j = j + 1
            end if
         end do
      end do
      node_1 = node_hull(nh)
      node_2 = node_hull(1)
      do
         node = bound_list(j,1)
         if( node == node_2 ) then
            exit
         else
            if( node /= node_1 ) then
                  aligned = CH_ccw( n_xy(node_1,1), n_xy(node_1,2),     &
                                    n_xy(node,1), n_xy(node,2),         &
                                    n_xy(node_2,1), n_xy(node_2,2) )
               if( aligned > 0.0d0 ) then
                  face_tri(bound_list(j-1,2),2) = -1
                  face_tri(bound_list(j,2),2)   = -1
               end if
            end if
            if( j == nbl ) then
               j = 1
            else
               j = j + 1
            end if
         end if
      end do

      if( any(face_tri(:,2) < 0) ) then
         convex_domain = 0
      else
         convex_domain = 1
      end if

      ! Processing holes: the faces of each hole have a different negative
      ! value.

      ! Examining all faces, each time we found face_tri(:,2) equal to
      ! -huge(0) indicates a new hole.
      k = 0 ! hole number
      do i = 1, m ! loop over all faces
         if( face_tri(i,2) == -huge(0) ) then
            k = k + 1
            ! here, we process the hole #k
            fb_ptr(k+1,1) = fb_ptr(k,2) + 1
            fb_ptr(k+1,2) = fb_ptr(k,2)
            node_start = face_n(i,1)
            ! Let's also determine the starting face
            call next_face_on_boundary( tri_n, tri_f, face_tri, n_tri,  &
                                       node_start, face_start )
            node = node_start
            face = face_start
            do
               call next_node_and_face_on_boundary( tri_n, tri_f, face_tri, &
                                                    face_n, node, face, &
                                                    next_node, next_face )
               face_tri(next_face,2) = -(k+1)
               fb_ptr(k+1,2) = fb_ptr(k+1,2) + 1
               fb(fb_ptr(k+1,2)) = next_face
               if( next_node == node_start .and. next_face == face_start ) exit
               node = next_node
               face = next_face
            end do
         end if
      end do

      nb = k + 1

      if( associated(faces_boundary) ) deallocate(faces_boundary)
      allocate( faces_boundary(fb_ptr(nb,2)) )
      faces_boundary(:) = fb(1:fb_ptr(nb,2))

      incr_order = .true.

      if( nb > 1 ) then
         ! Search the smallest node value for each internal boundary
         allocate( smallest_node(nb) )
         smallest_node(1) = 0 ! to force the exterior boundary to be the first
         ! Quick check of ordering
         do i = 2, nb
            smallest_node(i) = huge(1)
            ! examine all faces of this boundary
            j1 = fb_ptr(i,1)
            j2 = fb_ptr(i,2)
            do j = j1, j2
               face = faces_boundary(j)
               node = face_n(face,1)
               if( node < smallest_node(i) ) smallest_node(i) = node
               node = face_n(face,2)
               if( node < smallest_node(i) ) smallest_node(i) = node
            end do
            if( smallest_node(i) < smallest_node(i-1) ) incr_order = .false.
         end do
      end if

      if( associated(faces_boundary_ptr) ) deallocate(faces_boundary_ptr)
      allocate( faces_boundary_ptr(nb,2) )

      if( incr_order ) then
         faces_boundary_ptr(:,:) = fb_ptr(1:nb,:)
      else
         ! Sort the smallest_node array...
         allocate( ind(nb) )
         ind(:) = [ ( i, i = 1, nb ) ]
         call quick_sort_3_int( "asc", smallest_node, ind )
         do i = 1, nb
            faces_boundary_ptr(i,:) = fb_ptr(ind(i),:)
         end do
      end if

#endif
   end subroutine check_tri_convexity
!_______________________________________________________________________
!
   subroutine convex_hull_from_connectivity( n_xy, tri_n, tri_f,        &
                                             face_tri, face_n, n_tri,   &
                                             node_0, ind, ind_size )

      double precision, intent(in)  :: n_xy(:,:)
      integer,          intent(in)  :: tri_n(:,:), tri_f(:,:), face_tri(:,:)
      integer,          intent(in)  :: n_tri(:), face_n(:,:), node_0
      integer,          intent(out) :: ind(:), ind_size
      !------ API end ------

#ifdef _DEVLP
      ! Build the Convex Hull from the connectivity of a meshed domain.
      ! 'node' is the starting point, on the exterior boundary.
      ! The 'ind(:)' array must be allocated before calling this routine.

      integer :: node, next_node, face, next_face, k
      double precision :: aligned

   !------ end of declarations -- execution starts hereafter  ------

      ind(1) = node_0

      node = node_0
      call next_face_on_boundary( tri_n, tri_f, face_tri, n_tri,        &
                                  node, face )
      ! Add the second node found on the exterior boundary (CCW orientation)
      call next_node_and_face_on_boundary( tri_n, tri_f, face_tri, face_n, &
                                           node, face, next_node, next_face )
      ind(2) = next_node
      face = next_face
      k = 2

      do
         ! Getting one further node of the exterior boundary and test it
         ! against the previous two.
         node = ind(k)
         call next_node_and_face_on_boundary( tri_n, tri_f, face_tri, face_n, &
                                              node, face, next_node, next_face )
         if( next_node == node_0 ) exit
         aligned = CH_ccw( n_xy(ind(k-1),1),  n_xy(ind(k-1),2),         &
                           n_xy(ind(k),1),    n_xy(ind(k),2),           &
                           n_xy(next_node,1), n_xy(next_node,2) )
         if( aligned < 0.0d0 ) then
            ! Add the new node, keeping the previous one
            k = k + 1
            ind(k) = next_node
         else ! aligned >= 0.0d0
            ! Add the new node, deleting the previous one
            ind(k) = next_node
         end if
         if( ind(k) == ind(k-1) ) k = k - 1
         face = next_face
      end do

      ! Lastly, check the first point
      aligned = CH_ccw( n_xy(ind(k-1),1), n_xy(ind(k-1),2),             &
                        n_xy(ind(k),1),   n_xy(ind(k),2),               &
                        n_xy(node_0,1),   n_xy(node_0,2) )
      if( aligned >= 0.0d0 ) then
         k = k - 1
      end if

      ind_size = k

#endif
   end subroutine convex_hull_from_connectivity
!_______________________________________________________________________
!
   subroutine next_face_on_boundary( tri_n, tri_f, face_tri, n_tri,     &
                                     node, next_face )
      integer, intent(in) :: tri_n(:,:), tri_f(:,:), face_tri(:,:)
      integer, intent(in) :: n_tri(:), node
      integer             :: next_face
      !------ API end ------

      ! Starting from a node located on a boundary (exterior or internal),
      ! find the next face, located on the same boundary.
      ! The boundary is travelled with the left hand outside the mesh,
      ! therefore the exterior boundary is travelled in the anti-clockwise
      ! direction, whereas an internal boundary is travelled in the
      ! opposite direction (i.e. clockwise direction).

#ifdef _DEVLP
      integer :: tri, face, tri_new

   !------ end of declarations -- execution starts hereafter  ------

      tri = n_tri(node) ! arbitrary triangle using this node
      do
         face = face_before_node( tri_n, tri_f, tri, node )
         tri_new = face_tri(face,1)
         if( tri_new == tri ) tri_new = face_tri(face,2)
         if( tri_new <= 0 ) then
            exit
         end if
         tri = tri_new
      end do

      next_face = face

#endif
   end subroutine next_face_on_boundary
!_______________________________________________________________________
!
   subroutine next_node_and_face_on_boundary( tri_n, tri_f, face_tri,   &
                                              face_n, node, face,       &
                                              next_node, next_face )
      integer, intent(in) :: tri_n(:,:), tri_f(:,:), face_tri(:,:),     &
                             face_n(:,:), node, face
      integer             :: next_node, next_face
      !------ API end ------

      ! Starting from a node and a face located on a boundary (exterior
      ! or internal), find the next node and the associated next face,
      ! located on the same boundary.
      ! The boundary is travelled with the left hand outside the mesh,
      ! therefore the exterior boundary is travelled in the anti-clockwise
      ! direction, whereas an internal boundary is travelled in the
      ! opposite direction (i.e. clockwise direction).

#ifdef _DEVLP
      integer :: tri, next_tri

   !------ end of declarations -- execution starts hereafter  ------

      ! Next node
      next_node = face_n(face,1)
      if( next_node == node ) next_node = face_n(face,2)

      ! Next face
      tri = face_tri(face,1)
      do
         next_face = face_before_node( tri_n, tri_f, tri, next_node )
         next_tri = face_tri(next_face,1)
         if( next_tri == tri ) next_tri = face_tri(next_face,2)
         if( next_tri <= 0 ) then
            exit
         end if
         tri = next_tri
      end do

#endif
   end subroutine next_node_and_face_on_boundary
!_______________________________________________________________________
!
   function face_before_node( tri_n, tri_f, tri, node ) result( res )

      integer, intent(in) :: tri_n(:,:), tri_f(:,:)
      integer, intent(in) :: tri, node
      integer             :: res
      !------ API end ------

      ! The relative position 'before' is, of course, related to how
      ! nodes and faces are defined in my connectivity tables.
      ! Recall: nodes and faces are stored in the anti-clockwise
      ! direction around each triangle.

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( tri_n(tri,1) == node ) then
         res = tri_f(tri,3)
      else if( tri_n(tri,2) == node ) then
         res = tri_f(tri,1)
      else if( tri_n(tri,3) == node ) then
         res = tri_f(tri,2)
      else
         write(STDERR,*)
         write(STDERR,*) "(MUESLI face_before_node:) internal error:"
         write(STDERR,*) "        line 1416 in 'tri.inc'"
         write(STDERR,"(A,I0,A,I0,A)") "         Severe error: node ", node,        &
                         " doesn't belong to triangle ", tri, "!"
         write(STDERR,*) "        Please report this bug to: Edouard.Canot@univ-rennes.fr"
         mf_message_displayed = .true.
         call muesli_trace( pause ="yes" )
         stop
      end if

#endif
   end function face_before_node
!_______________________________________________________________________
!
   function node_before_node( tri_n, tri, node ) result( res )

      integer, intent(in) :: tri_n(:,:)
      integer, intent(in) :: tri, node
      integer             :: res
      !------ API end ------

      ! The relative position 'before' is, of course, related to how
      ! nodes are defined in my connectivity tables.
      ! Recall: nodes are stored in the anti-clockwise direction
      ! around each triangle.

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( tri_n(tri,1) == node ) then
         res = tri_n(tri,3)
      else if( tri_n(tri,2) == node ) then
         res = tri_n(tri,1)
      else if( tri_n(tri,3) == node ) then
         res = tri_n(tri,2)
      else
         write(STDERR,*)
         write(STDERR,*) "(MUESLI node_before_node:) internal error:"
         write(STDERR,*) "        line 1453 in 'tri.inc'"
         write(STDERR,"(A,I0,A,I0,A)") "         Severe error: node ", node,        &
                         " doesn't belong to triangle ", tri, "!"
         write(STDERR,*) "        Please report this bug to: Edouard.Canot@univ-rennes.fr"
         mf_message_displayed = .true.
         call muesli_trace( pause ="yes" )
         stop
      end if

#endif
   end function node_before_node
!_______________________________________________________________________
!
   function is_in_triangle( n_xy, tri_n, num, x, y ) result( res )

      double precision, intent(in)           :: n_xy(:,:)
      integer,          intent(in)           :: tri_n(:,:)
      integer,          intent(in)           :: num
      double precision, intent(in)           :: x, y
      integer :: res
      !------ API end ------

#ifdef _DEVLP
      ! Returns 0 if the point (x,y) is inside the triangle 'num';
      ! else, returns the first face for which the node is exterior.
      !
      ! Requires that the numbering of the nodes for each triangle is
      ! oriented (direct orientation).
      !
      ! Requires also that, in the local numbering of nodes and faces,
      ! if the nodes describing the triangle are n1, n2, n3, then the
      ! faces must be n1-n2, n2-n3, n3-n1 (in this order).

      double precision :: x1, y1, x2, y2, x3, y3

   !------ end of declarations -- execution starts hereafter  ------

      x1 = n_xy(tri_n(num,1),1)
      y1 = n_xy(tri_n(num,1),2)
      x2 = n_xy(tri_n(num,2),1)
      y2 = n_xy(tri_n(num,2),2)
      x3 = n_xy(tri_n(num,3),1)
      y3 = n_xy(tri_n(num,3),2)

      if( (x2-x1)*(y-y1)-(y2-y1)*(x-x1) < 0.0d0 ) then
         res = 1
         return
      end if

      if( (x3-x2)*(y-y2)-(y3-y2)*(x-x2) < 0.0d0 ) then
         res = 2
         return
      end if

      if( (x1-x3)*(y-y3)-(y1-y3)*(x-x3) < 0.0d0 ) then
         res = 3
         return
      end if

      res = 0

#endif
   end function is_in_triangle
!_______________________________________________________________________
!
   function adj_triangle( face_tri, tri_f, num, face ) result( num_new )

      integer, intent(in) :: face_tri(:,:)
      integer, intent(in) :: tri_f(:,:)
      integer, intent(in) :: num, face
      integer :: num_new
      !------ API end ------

#ifdef _DEVLP
      ! Returns the number of the triangle adjacent to the side
      ! mentionned by 'face' (local index: 1 to 3)

      integer :: iface

   !------ end of declarations -- execution starts hereafter  ------

      if( face == 0 ) then
         write(STDERR,*)
         write(STDERR,*) "(MUESLI adj_triangle:) internal error:"
         write(STDERR,*) "        Severe error: face == 0"
         write(STDERR,*) "        Please report this bug to: Edouard.Canot@univ-rennes.fr"
         mf_message_displayed = .true.
         call muesli_trace( pause ="yes" )
         stop
      end if

      iface = tri_f(num,face)
      num_new = face_tri(iface,1)
      if( num_new == num ) then
         num_new = face_tri(iface,2)
      end if

#endif
   end function adj_triangle
!_______________________________________________________________________
!
   subroutine build_connected_nodes( face_n, face_tri, tri_f, n_tri, tri_n, &
                                     connected_nodes, oriented )

      integer,           intent(in)  :: face_n(:,:), face_tri(:,:),     &
                                        tri_f(:,:), n_tri(:), tri_n(:,:)
      type(mf_Int_List), intent(out) :: connected_nodes(:)
      logical,           intent(in), optional :: oriented
      !------ API end ------

#ifdef _DEVLP
      integer :: nn, nf, i, j, n1, n2, nb, n_start
      integer :: f1, f2, f_start, t1, t2, t_last
      logical :: f_start_stored

      integer, allocatable :: tmp(:)

   !------ end of declarations -- execution starts hereafter  ------

      ! two methods, by filling the output list directly.
      ! oriented = FALSE -- faces are sorted w.r.t. their index (very quick)
      !            TRUE  -- add a post-processing to sort the faces
      !                     w.r.t. their orientation (slower)

      ! connected_nodes(:) must be allocated in the calling unit

      nn = size(connected_nodes,1)
      nf = size(face_n,1)

      ! first, we must count the size of each list (in tmp)
      allocate( tmp(nn) )
      tmp(:) = 0

      ! compute the size of each list
      do i = 1, nf

         n1 = face_n(i,1)
         n2 = face_n(i,2)

         ! processing n1
         tmp(n1) = tmp(n1) + 1

         ! processing n2
         tmp(n2) = tmp(n2) + 1

      end do

      ! now, we are able to allocate the lists
      do i = 1, nn

         nb = tmp(i)
         allocate( connected_nodes(i)%list(nb) )
         connected_nodes(i)%list(:) = 0

      end do

      ! now using tmp(:) to store the current index in each list
      ! (in both methods)
      tmp(:) = 0

      if( oriented ) then

         ! On veut que les nœuds connectées au nœud considéré soient
         ! orientés spatialement de manière directe : on remplit donc
         ! la liste en les mettant dans cet ordre.

         ! Select first all nodes on the boundary

         ! Looking for the first face which is on the boundary
         do i = 1, nf
            if( face_tri(i,2) <= 0 ) then
               f1 = i
               exit
            end if
         end do
         n1 = face_n(f1,1)
         n2 = face_n(f1,2)
         t1 = face_tri(f1,1)

         n_start = n1

         do ! until all boundary nodes are processed

            ! for a given node on the boundary, fill the list of faces,
            ! until the boundary is found again

            ! n2, f1, t1 are known: we can iterate
            do
               tmp(n1) = tmp(n1) + 1
               j = tmp(n1)

               connected_nodes(n1)%list(j) = n2

               if( t1 <= 0 ) exit
               f2 = face_predecessor( tri_f, f1, t1 )
               ! check orientation of f2 to find next triangle
               n2 = face_n(f2,1)
               if( n2 == n1 ) then
                  t2 = face_tri(f2,1)
                  n2 = face_n(f2,2)
               else
                  t2 = face_tri(f2,2)
               end if
               f1 = f2
               t_last = t1
               t1 = t2
            end do
            t1 = t_last
            n2 = n1
            if( face_n(f1,1) == n1 ) then
               n1 = face_n(f1,2)
            else
               n1 = face_n(f1,1)
            end if
            if( n1 == n_start ) exit

         end do

         ! Finally, process the internal nodes
         do i = 1, nn

            if( connected_nodes(i)%list(1) == 0 ) then
               ! the node 'i' is not yet processed

               n1 = i
               t1 = n_tri(n1)
               ! find face f1 after n1 in triangle t1
               do j = 1, 3
                  if( tri_n(t1,j) == n1 ) exit
               end do
               f1 = tri_f(t1,j)
               n2 = face_n(f1,1)
               if( n2 == n1 ) then
                  n2 = face_n(f1,2)
               end if

               f_start = f1
               f_start_stored = .false.

               ! n2, f1, t1 are known: we can iterate
               do
                  if( f_start_stored ) then
                     if( f1 == f_start ) exit
                  end if
                  tmp(n1) = tmp(n1) + 1
                  j = tmp(n1)
                  connected_nodes(n1)%list(j) = n2
                  f_start_stored = .true.

                  f2 = face_predecessor( tri_f, f1, t1 )
                  ! check orientation of f2 to find next triangle
                  n2 = face_n(f2,1)
                  if( n2 == n1 ) then
                     t2 = face_tri(f2,1)
                     n2 = face_n(f2,2)
                  else
                     t2 = face_tri(f2,2)
                  end if
                  f1 = f2
                  t_last = t1
                  t1 = t2
               end do

            end if

         end do

      else

         ! fill the lists
         do i = 1, nf

            n1 = face_n(i,1)
            n2 = face_n(i,2)

            ! processing n1
            tmp(n1) = tmp(n1) + 1
            j = tmp(n1)
            connected_nodes(n1)%list(j) = n2

            ! processing n2
            tmp(n2) = tmp(n2) + 1
            j = tmp(n2)
            connected_nodes(n2)%list(j) = n1

         end do

      end if

      deallocate( tmp )

#endif
   end subroutine build_connected_nodes
!_______________________________________________________________________
!
   subroutine build_connected_faces( face_n, face_tri, tri_f, n_tri, tri_n, &
                                     connected_faces, oriented )

      integer,           intent(in)  :: face_n(:,:), face_tri(:,:),     &
                                        tri_f(:,:), n_tri(:), tri_n(:,:)
      type(mf_Int_List), intent(out) :: connected_faces(:)
      logical,           intent(in)  :: oriented
      !------ API end ------

#ifdef _DEVLP
      integer :: nn, nf, i, j, n1, n2, nb, n_start
      integer :: f1, f2, f_start, t1, t2, t_last
      logical :: f_start_stored

      integer, allocatable :: tmp(:)

   !------ end of declarations -- execution starts hereafter  ------

      ! two methods, by filling the output list directly.
      ! oriented = FALSE -- faces are sorted w.r.t. their index (very quick)
      !            TRUE  -- add a post-processing to sort the faces
      !                     w.r.t. their orientation (slower)

      ! connected_faces(:) must be allocated in the calling unit

      nn = size(connected_faces,1)
      nf = size(face_n,1)

      ! first, we must count the size of each list (in tmp)
      allocate( tmp(nn) )
      tmp(:) = 0

      ! compute the size of each list
      do i = 1, nf

         n1 = face_n(i,1)
         n2 = face_n(i,2)

         ! processing n1
         tmp(n1) = tmp(n1) + 1

         ! processing n2
         tmp(n2) = tmp(n2) + 1

      end do

      ! now, we are able to allocate the lists
      do i = 1, nn

         nb = tmp(i)
         allocate( connected_faces(i)%list(nb) )
         connected_faces(i)%list(:) = 0

      end do

      ! now using tmp(:) to store the current index in each list
      ! (in both methods)
      tmp(:) = 0

      if( oriented ) then

         ! On veut que les faces connectées au nœud considéré soient
         ! orientées spatialement de manière directe : on remplit donc
         ! la liste en les mettant dans cet ordre.

         ! Select first all nodes on the boundary

         ! Looking for the first face which is on the boundary
         do i = 1, nf
            if( face_tri(i,2) <= 0 ) then
               f1 = i
               exit
            end if
         end do
         n1 = face_n(f1,1)
         t1 = face_tri(f1,1)

         n_start = n1

         do ! until all boundary nodes are processed

            ! for a given node on the boundary, fill the list of faces,
            ! until the boundary is found again

            ! n1, f1, t1 are known: we can iterate
            do
               tmp(n1) = tmp(n1) + 1
               j = tmp(n1)
               connected_faces(n1)%list(j) = f1

               if( t1 <= 0 ) exit
               f2 = face_predecessor( tri_f, f1, t1 )
               ! check orientation of f2 to find next triangle
               if( face_n(f2,1) == n1 ) then
                  t2 = face_tri(f2,1)
               else
                  t2 = face_tri(f2,2)
               end if
               f1 = f2
               t_last = t1
               t1 = t2
            end do
            t1 = t_last
            if( face_n(f1,1) == n1 ) then
               n1 = face_n(f1,2)
            else
               n1 = face_n(f1,1)
            end if
            if( n1 == n_start ) exit

         end do

         ! Finally, process the internal nodes
         do i = 1, nn

            if( connected_faces(i)%list(1) == 0 ) then
               ! the node 'i' is not yet processed

               n1 = i
               t1 = n_tri(n1)
               ! find face f1 after n1 in triangle t1
               do j = 1, 3
                  if( tri_n(t1,j) == n1 ) exit
               end do
               f1 = tri_f(t1,j)

               f_start = f1
               f_start_stored = .false.

               ! n1, f1, t1 are known: we can iterate
               do
                  if( f_start_stored ) then
                     if( f1 == f_start ) exit
                  end if
                  tmp(n1) = tmp(n1) + 1
                  j = tmp(n1)
                  connected_faces(n1)%list(j) = f1
                  f_start_stored = .true.

                  f2 = face_predecessor( tri_f, f1, t1 )
                  ! check orientation of f2 to find next triangle
                  if( face_n(f2,1) == n1 ) then
                     t2 = face_tri(f2,1)
                  else
                     t2 = face_tri(f2,2)
                  end if
                  f1 = f2
                  t_last = t1
                  t1 = t2
               end do

            end if

         end do

      else

         ! fill the lists
         do i = 1, nf

            n1 = face_n(i,1)
            n2 = face_n(i,2)

            ! processing n1
            tmp(n1) = tmp(n1) + 1
            j = tmp(n1)
            connected_faces(n1)%list(j) = i

            ! processing n2
            tmp(n2) = tmp(n2) + 1
            j = tmp(n2)
            connected_faces(n2)%list(j) = i

         end do

      end if

      deallocate( tmp )

#endif
   end subroutine build_connected_faces
!_______________________________________________________________________
!
   function face_predecessor( tri_f, f1, tri ) result( f2 )

      integer, intent(in) :: tri_f(:,:)
      integer, intent(in) :: f1, tri
      integer :: f2
      !------ API end ------

#ifdef _DEVLP
      ! In a triangulation given by the array tri_f (definition of triangles
      ! by its faces -- they are oriented), and given a triangle 'tri' and
      ! a face 'f1', compute the predecessor f2 in the same triangle.

      integer :: i

   !------ end of declarations -- execution starts hereafter  ------

      ! locate f1 in tri
      do i = 1, 3
         if( tri_f(tri,i) == f1 ) exit
      end do
      ! get the predecessor
      if( i == 1 ) then
         f2 = tri_f(tri,3)
      else ! i = 2 or 3
         f2 = tri_f(tri,i-1)
      end if

#endif
   end function face_predecessor
