! f90 include file

!_______________________________________________________________________
!
   function mfGet_element( x, i1, i2 ) result( out )

      type(mfArray) :: x
      integer, intent(in) :: i1, i2
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! extracts x(i1,i2) from an mfArray of any shape
      !
      ! returns a scalar-like mfArray

      integer :: row_sorted ! because x%row_sorted is: integer(kind=kind_1)

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x )

      if( mfIsEmpty(x) ) then
         call PrintMessage( "mfGet", "I",                               &
                            "the mfArray x is empty." )
         go to 99
      end if

      if( i1 < 1 .or. x%shape(1) < i1 ) then
         call PrintMessage( "mfGet", "E",                               &
                            "row index is out of range!" )
         go to 99
      end if

      if( i2 < 1 .or. x%shape(2) < i2 ) then
         call PrintMessage( "mfGet", "E",                               &
                            "col index is out of range!" )
         go to 99
      end if

      out%shape = [ 1, 1 ]
      if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
         out%data_type = x%data_type
         allocate( out%double(1,1) )

         out%double(1,1) = x%double(i1,i2)
      else if( x%data_type == MF_DT_CMPLX ) then
         out%data_type = x%data_type
         allocate( out%cmplx(1,1) )

         out%cmplx(1,1) = x%cmplx(i1,i2)
      else if( x%data_type == MF_DT_SP_DBLE ) then
         out%data_type = MF_DT_DBLE
         allocate( out%double(1,1) )

         row_sorted = x%row_sorted
         out%double(1,1) = getelm( i1, i2,                              &
                                   x%a, x%i, x%j, row_sorted )
         out%row_sorted = TRUE
      else if( x%data_type == MF_DT_SP_CMPLX ) then
         out%data_type = MF_DT_CMPLX
         allocate( out%cmplx(1,1) )

         row_sorted = x%row_sorted
         out%cmplx(1,1) = getelm_cmplx( i1, i2,                         &
                                        x%z, x%i, x%j, row_sorted )
         out%row_sorted = TRUE
      end if

      if( mf_phys_units ) then
         out%units(:) = x%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( x )

      call msAutoRelease( x )

#endif
   end function mfGet_element
!_______________________________________________________________________
!
   function mfGet_element_in_vec( x, i ) result( out )

      type(mfArray) :: x
      integer, intent(in) :: i
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! extracts x(i) from a vector-like mfArray
      !
      ! returns a scalar-like mfArray

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x )

      if( mfIsEmpty(x) ) then
         call PrintMessage( "mfGet", "I",                               &
                            "the mfArray x is empty." )
         go to 99
      end if

      if( mfIsSparse(x) ) then
         call PrintMessage( "mfGet", "E",                               &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( x%shape(1) == 1 ) then
         if( i < 1 .or. x%shape(2) < i  ) then
            call PrintMessage( "mfGet", "E",                            &
                               "index is out of range!" )
            go to 99
         end if
         out%data_type = x%data_type
         out%shape = [ 1, 1 ]
         if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
            allocate( out%double(1,1) )

            out%double(1,1) = x%double(1,i)
         else if( x%data_type == MF_DT_CMPLX ) then
            allocate( out%cmplx(1,1) )

            out%cmplx(1,1) = x%cmplx(1,i)
         end if
      else if( x%shape(2) == 1 ) then
         if( i < 1 .or. x%shape(1) < i ) then
            call PrintMessage( "mfGet", "E",                            &
                               "index is out of range!" )
            go to 99
         end if
         out%data_type = x%data_type
         out%shape = [ 1, 1 ]
         if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
            allocate( out%double(1,1) )

            out%double(1,1) = x%double(i,1)
         else if( x%data_type == MF_DT_CMPLX ) then
            allocate( out%cmplx(1,1) )

            out%cmplx(1,1) = x%cmplx(i,1)
         end if
      else
         call PrintMessage( "mfGet", "E",                               &
                            "too few args.!",                           &
                            "(actual mfArray is a matrix)" )
         go to 99
      end if

      if( mf_phys_units ) then
         out%units(:) = x%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( x )

      call msAutoRelease( x )

#endif
   end function mfGet_element_in_vec
!_______________________________________________________________________
!
   function mfGet_section_in_vec( x, i ) result( out )

      type(mfArray) :: x
      integer, intent(in) :: i(:)
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! extracts x(i(:)) from a vector-like mfArray
      ! indexes i(:) doesn't need to be continuous
      !
      ! returns a vector-like mfArray of same size as i(:)

      integer :: dim_out
      real(kind=MF_DOUBLE), pointer :: vec_real(:)
      complex(kind=MF_DOUBLE), pointer :: vec_cmplx(:)

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x )

      if( mfIsEmpty(x) ) then
         call PrintMessage( "mfGet", "I",                               &
                            "the mfArray x is empty." )
         go to 99
      end if

      if( mfIsSparse(x) ) then
         call PrintMessage( "mfGet", "E",                               &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( x%shape(1) == 1 ) then

         if( minval(i) < 1 .or. x%shape(2) < maxval(i)  ) then
            call PrintMessage( "mfGet", "E",                            &
                               "indexes are out of range!" )
            go to 99
         end if
         out%data_type = x%data_type
         out%shape = [ 1, size(i) ]
         if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
            allocate( out%double(1,size(i)) )

            out%double(1,:) = x%double(1,i)
         else if( x%data_type == MF_DT_CMPLX ) then
            allocate( out%cmplx(1,size(i)) )

            out%cmplx(1,:) = x%cmplx(1,i)
         end if

      else if( x%shape(2) == 1 ) then

         if( minval(i) < 1 .or. x%shape(1) < maxval(i) ) then
            call PrintMessage( "mfGet", "E",                            &
                               "indexes are out of range!" )
            go to 99
         end if
         out%data_type = x%data_type
         out%shape = [ size(i), 1 ]
         if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
            allocate( out%double(size(i),1) )
            out%double(:,1) = x%double(i,1)
         else if( x%data_type == MF_DT_CMPLX ) then
            allocate( out%cmplx(size(i),1) )
            out%cmplx(:,1) = x%cmplx(i,1)
         end if

      else ! 2D array

         dim_out = x%shape(1)*x%shape(2)
         if( maxval(i) > dim_out ) then
            call PrintMessage( "mfGet", "E",                            &
                               "indexes are out of range!" )
            go to 99
         end if

         out%data_type = x%data_type
         out%shape = [ 1, size(i) ]
         if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
            allocate( out%double(1,size(i)) )
            vec_real => rank_2_to_1_real8( x%double, dim_out )
            out%double(1,:) = vec_real(i)
            vec_real => null()
         else if( x%data_type == MF_DT_CMPLX ) then
            allocate( out%cmplx(1,size(i)) )
            vec_cmplx => rank_2_to_1_cmplx( x%cmplx, dim_out )
            out%cmplx(1,:) = vec_cmplx(i)
            vec_cmplx => null()
         end if

      end if

      if( mf_phys_units ) then
         out%units(:) = x%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( x )

      call msAutoRelease( x )

#endif
   end function mfGet_section_in_vec
!_______________________________________________________________________
!
   function mfGet_section_in_vseq( x, seq ) result( out )

      type(mfArray) :: x
      type(seq_def) :: seq
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! extracts x(seq) from a vector-like mfArray
      ! argument seq is of the form : i1 .to. i2 .by. i3
      ! which is equivalent to : i1:i2:i3 (f90 section notation)
      !
      ! returns a vector-like mfArray of same size as seq

      integer, pointer :: i(:) => null()
      integer :: nrow, ncol

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x )

      if( mfIsEmpty(x) ) then
         call PrintMessage( "mfGet", "I",                               &
                            "the mfArray x is empty." )
         go to 99
      end if

      if( mfIsSparse(x) ) then
         call PrintMessage( "mfGet", "E",                               &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      nrow = x%shape(1)
      ncol = x%shape(2)
      if( nrow == 1 ) then ! row vector

         if( seq%start_1_EndIndex ) then
            seq%start_1 = ncol + seq%start_1
         end if
         if( seq%end_1_EndIndex ) then
            seq%end_1 = ncol + seq%end_1
         end if
         if( seq%start_2_EndIndex ) then
            seq%start_2 = ncol + seq%start_2
         end if
         if( seq%end_2_EndIndex ) then
            seq%end_2 = ncol + seq%end_2
         end if
         if( seq%but_EndIndex ) then
            seq%but = ncol + seq%but
         end if
         call build_int_seq( i, seq, "mfGet" )

         if( minval(i) < 1 .or. ncol < maxval(i)  ) then
            call PrintMessage( "mfGet", "E",                            &
                               "indexes are out of range!" )
            go to 99
         end if
         out%data_type = x%data_type
         out%shape = [ 1, size(i) ]
         if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
            allocate( out%double(1,size(i)) )

            out%double(1,:) = x%double(1,i)
         else if( x%data_type == MF_DT_CMPLX ) then
            allocate( out%cmplx(1,size(i)) )

            out%cmplx(1,:) = x%cmplx(1,i)
         end if

      else if( ncol == 1 ) then ! column vector

         if( seq%start_1_EndIndex ) then
            seq%start_1 = nrow + seq%start_1
         end if
         if( seq%end_1_EndIndex ) then
            seq%end_1 = nrow + seq%end_1
         end if
         if( seq%start_2_EndIndex ) then
            seq%start_2 = nrow + seq%start_2
         end if
         if( seq%end_2_EndIndex ) then
            seq%end_2 = nrow + seq%end_2
         end if
         if( seq%but_EndIndex ) then
            seq%but = nrow + seq%but
         end if
         call build_int_seq( i, seq, "mfGet" )

         if( minval(i) < 1 .or. nrow < maxval(i) ) then
            call PrintMessage( "mfGet", "E",                            &
                               "indexes are out of range!" )
            go to 99
         end if
         out%data_type = x%data_type
         out%shape = [ size(i), 1 ]
         if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
            allocate( out%double(size(i),1) )

            out%double(:,1) = x%double(i,1)
         else if( x%data_type == MF_DT_CMPLX ) then
            allocate( out%cmplx(size(i),1) )

            out%cmplx(:,1) = x%cmplx(i,1)
         end if

      else

         call PrintMessage( "mfGet", "E",                               &
                            "too few args.!",                           &
                            "(actual mfArray is a matrix)" )
         go to 99

      end if

      if( mf_phys_units ) then
         out%units(:) = x%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      if( associated(i) ) then
#ifndef _TRACE_MEM_ALLOC
         deallocate( i )
#else
         call mf_deallocate( array=i,                                   &
                             file="Get.inc", line="???",                &
                             symb="i", unit="mfGet_section_in_vseq" )
#endif
      end if

      call msFreeArgs( x )

      call msAutoRelease( x )

#endif
   end function mfGet_section_in_vseq
!_______________________________________________________________________
!
   function mfGet_elems_in_array( x, a ) result( out )

      type(mfArray) :: x, a
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! extracts x(section) from a vector-like mfArray
      ! section takes its definition from the vector-like mfArray 'a'
      !
      ! returns a vector-like mfArray of same size as 'a'

      ! mfArray 'a' is usually of type REAL, in which case it must contain
      ! indexes;
      ! if the mfArray 'a' is of type BOOL, then it is a filter (or a mask)
      ! for elements of x.

      ! extended to the case where X is a matrix-like mfArray.
      ! In this latter case, the mfArray A must have the same
      ! shape and must be of type BOOL, and the returned mfArray is a
      ! long column vector.

      ! extended to the case where 'a' contains an EndIndex (which is
      ! either the MF_END parameter itself, or an expression
      ! involving integers and MF_END, like 'MF_END-1'). In this case,
      ! the mfArray 'x' must be a vector and not a matrix!

      integer :: i, j, nnz, ii, nrow, ncol, ndim, n, dim_out
      integer, allocatable :: i1(:)
      real(kind=MF_DOUBLE), pointer :: vec_real(:)
      complex(kind=MF_DOUBLE), pointer :: vec_cmplx(:)

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x, a )

      if( mfIsEmpty(x) .or. mfIsEqual(a,MF_EMPTY) ) then
         go to 99
      end if

      if( mfIsSparse(x) ) then
         call PrintMessage( "mfGet", "E",                               &
                            "mfArray cannot be sparse!" )
         go to 99
      end if

      if( x%shape(1) == 1 ) then ! x is a row vector

         ncol = x%shape(2)
         if( mfIsEqual(a,MF_COLON) ) then

            allocate( i1(ncol) )

            i1 = [ (i, i = 1, ncol) ]

         else if( isEndIndex(a) ) then

            allocate( i1(1) )

            i1(1) = ncol + a%crc
            if( i1(1) < 1 .or. ncol < i1(1) ) then
               call PrintMessage( "mfGet", "E",                         &
                                  "index 'a' is out of range!" )
               go to 99
            end if

         else

            if( a%data_type == MF_DT_DBLE ) then
               if( any( int(a%double) /= a%double) ) then
                  call PrintMessage( "mfGet", "E",                      &
                                     "'a' must contain integer values!" )
                  go to 99
               end if

               if( a%shape(1) /= 1 .and. a%shape(2) /= 1 ) then
                  call PrintMessage( "mfGet", "E",                      &
                                     "mfArray indexes 'a' must be of rank one!" )
                  go to 99
               end if

               if( minval(a%double) < 1 .or. x%shape(2) < maxval(a%double) ) then
                  call PrintMessage( "mfGet", "E",                      &
                                     "indexes are out of range!" )
                  go to 99
               end if

               ! we cannot use pointers because the indexes must be
               ! of type 'integer'.
               allocate( i1(size(a%double)) )

               i1 = reshape( int(a%double), [size(a%double)] )
            else if( a%data_type == MF_DT_BOOL ) then
               if( any(a%shape /= x%shape) ) then
                  call PrintMessage( "mfGet", "E",                      &
                                     "when 'a' is boolean,",            &
                                     "it must have the same shape as 'x'!" )
                  go to 99
               end if

               ! before allocating 'i1', the nomber of 'TRUE' values
               ! must be counted in the vector 'a'
               ! TRUE  : any non-zero value
               ! FALSE : zero value
               nnz = count( .not.(a%double == 0.0d0) )
               allocate( i1(nnz) )

               ! filling indexes in 'i1'
               ii = 0
               do i = 1, a%shape(2)
                  if( .not.(a%double(1,i) == 0.0d0) ) then
                     ii = ii + 1
                     i1(ii) = i
                  end if
               end do

            else
               call PrintMessage( "mfGet", "E",                         &
                                  "'a' must be of type: real, boolean or EndIndex!" )
               go to 99
            end if

         end if

         out%data_type = x%data_type
         out%shape = [ 1, size(i1) ]
         if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
            allocate( out%double(1,size(i1)) )

            out%double(1,:) = x%double(1,i1)
         else if( x%data_type == MF_DT_CMPLX ) then
            allocate( out%cmplx(1,size(i1)) )

            out%cmplx(1,:) = x%cmplx(1,i1)
         end if

      else if( x%shape(2) == 1 ) then ! x is a column vector

         nrow = x%shape(1)
         if( mfIsEqual(a,MF_COLON) ) then

            allocate( i1(nrow) )

            i1 = [ (i, i = 1, nrow) ]

         else if( isEndIndex(a) ) then

            allocate( i1(1) )

            i1(1) = nrow + a%crc
            if( i1(1) < 1 .or. nrow < i1(1) ) then
               call PrintMessage( "mfGet", "E",                         &
                                  "index 'a' is out of range!" )
               go to 99
            end if

         else

            if( a%data_type == MF_DT_DBLE ) then
               if( any( int(a%double) /= a%double) ) then
                  call PrintMessage( "mfGet", "E",                      &
                                     "'a' must contain integer values!" )
                  go to 99
               end if

               if( a%shape(1) /= 1 .and. a%shape(2) /= 1 ) then
                  call PrintMessage( "mfGet", "E",                      &
                                     "mfArray indexes 'a' must be of rank one!" )
                  go to 99
               end if

               if( minval(a%double) < 1 .or. x%shape(1) < maxval(a%double) ) then
                  call PrintMessage( "mfGet", "E",                      &
                                     "indexes are out of range!" )
                  go to 99
               end if

               ! we cannot use pointers because the indexes must be
               ! of type 'integer'.
               allocate( i1(size(a%double)) )

               i1 = reshape( int(a%double), [size(a%double)] )
            else if( a%data_type == MF_DT_BOOL ) then
               if( any(a%shape /= x%shape) ) then
                  call PrintMessage( "mfGet", "E",                      &
                                     "when 'a' is boolean,",            &
                                     "it must have the same shape as 'x'!" )
                  go to 99
               end if

               ! before allocating 'i1', the nomber of 'TRUE' values
               ! must be counted in the vector 'a'
               ! TRUE  : any non-zero value
               ! FALSE : zero value
               nnz = count( .not.(a%double == 0.0d0) )
               allocate( i1(nnz) )

               ! filling indexes in 'i1'
               ii = 0
               do i = 1, a%shape(1)
                  if( .not.(a%double(i,1) == 0.0d0) ) then
                     ii = ii + 1
                     i1(ii) = i
                  end if
               end do

            else
               call PrintMessage( "mfGet", "E",                         &
                                  "'a' must be of type: real, boolean or EndIndex!" )
               go to 99
            end if


         end if

         out%data_type = x%data_type
         out%shape = [ size(i1), 1 ]
         if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
            allocate( out%double(size(i1),1) )

            out%double(:,1) = x%double(i1,1)
         else if( x%data_type == MF_DT_CMPLX ) then
            allocate( out%cmplx(size(i1),1) )

            out%cmplx(:,1) = x%cmplx(i1,1)
         end if

      else ! X is a matrix-like mfArray

         if( all(X%shape == A%shape) ) then
            ! X and A have the same shape

            ! X must be a numeric
            if( .not. mfIsNumeric(X) ) then
               call PrintMessage( "mfGet", "E",                         &
                                  "first arg must be numeric!" )
               go to 99
            end if

            ! A must be a boolean mfArray
            if( A%data_type /= MF_DT_BOOL ) then
               call PrintMessage( "mfGet", "E",                         &
                                  "second arg must be boolean!" )
               go to 99
            end if

            nrow = X%shape(1)
            ncol = X%shape(2)

            out%data_type = X%data_type
            ! before allocating space for 'out', we must count the number
            ! of TRUE values in A
            nnz = count( .not.(A%double == 0.0d0) )
            out%shape = [ nnz, 1 ]
            if( X%data_type == MF_DT_DBLE ) then
               allocate( out%double(nnz,1) )

               ii = 0
               do j = 1, ncol
                  do i = 1, nrow
                     if( A%double(i,j) /= 0.0d0 ) then
                        ii = ii + 1
                        out%double(ii,1) = X%double(i,j)
                     end if
                  end do
               end do
            else if( X%data_type == MF_DT_CMPLX ) then
               allocate( out%cmplx(nnz,1) )

               ii = 0
               do j = 1, ncol
                  do i = 1, nrow
                     if( A%double(i,j) /= 0.0d0 ) then
                        ii = ii + 1
                        out%cmplx(ii,1) = X%cmplx(i,j)
                     end if
                  end do
               end do
            end if

         else ! A is a vector

            allocate( i1(size(a%double)) )
            i1 = a
            n = maxval(i1)

            dim_out = x%shape(1)*x%shape(2)
            if( n > dim_out ) then
               call PrintMessage( "mfGet", "E",                         &
                                  "indexes are out of range!" )
               go to 99
            end if

            out%data_type = x%data_type
            out%shape = [ size(i1), 1 ]
            if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
               allocate( out%double(size(i1),1) )
               vec_real => rank_2_to_1_real8( x%double, dim_out )
               out%double(:,1) = vec_real(i1)
               vec_real => null()
            else if( x%data_type == MF_DT_CMPLX ) then
               allocate( out%cmplx(size(i1),1) )
               vec_cmplx => rank_2_to_1_cmplx( x%cmplx, dim_out )
               out%cmplx(:,1) = vec_cmplx(i1)
               vec_cmplx => null()
            end if

         end if
      end if

      if( mf_phys_units ) then
         out%units(:) = x%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( x, a )

      call msAutoRelease( x, a )

#endif
   end function mfGet_elems_in_array
!_______________________________________________________________________
!
   function mfGet_section_vec_vec( x, i1, i2 ) result( out )

      type(mfArray) :: x
      integer, intent(in) :: i1(:), i2(:)
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! extracts x(i1(:),i2(:)) from an mfArray
      ! indexes i1(:) and i2(:) doesn't need to be continuous
      ! (for sparse storage, they must be contiguous and in
      !  increasing order)
      !
      ! returns an mfArray of shape equal to (size(i1),size(i2))

      integer :: i1_beg, i1_end, n1,                                    &
                 i2_beg, i2_end, n2,                                    &
                 i, nnz
      logical :: contiguous

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x )

      if( mfIsEmpty(x) ) then
         call PrintMessage( "mfGet", "I",                               &
                            "the mfArray x is empty." )
         go to 99
      end if

      if( minval(i1) < 1 .or. x%shape(1) < maxval(i1) ) then
         call PrintMessage( "mfGet", "E",                               &
                            "row indexes are out of range!" )
         go to 99
      end if

      if( minval(i2) < 1 .or. x%shape(2) < maxval(i2) ) then
         call PrintMessage( "mfGet", "E",                               &
                            "col indexes are out of range!" )
         go to 99
      end if

      out%data_type = x%data_type
      out%shape = [ size(i1), size(i2) ]
      if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
         allocate( out%double(size(i1),size(i2)) )

         out%double(:,:) = x%double(i1,i2)
      else if( x%data_type == MF_DT_CMPLX ) then
         allocate( out%cmplx(size(i1),size(i2)) )

         out%cmplx(:,:) = x%cmplx(i1,i2)
      else ! x is sparse
         ! test if i1(:) and i2(:) contain contiguous indexes
         ! in increasing order
         contiguous = .true.
         n1 = size(i1)
         do i = 2, n1
            if( i1(i) /= i1(i-1) + 1 ) then
               contiguous = .false.
               exit
            end if
         end do
         n2 = size(i2)
         do i = 2, n2
            if( i2(i) /= i2(i-1) + 1 ) then
               contiguous = .false.
               exit
            end if
         end do
         if( contiguous ) then
            i1_beg = i1(1)
            i1_end = i1(n1)
            i2_beg = i2(1)
            i2_end = i2(n2)
            call submat_nnz( x%i, x%j, i1_beg, i1_end, i2_beg, i2_end,  &
                             nnz )
            allocate( out%i(nnz) )

            allocate( out%j(n2+1) )

            if( mfIsReal(x) ) then
               out%data_type = MF_DT_SP_DBLE
               allocate( out%a(nnz) )

               call submat( x%a, x%i, x%j, i1_beg, i1_end, i2_beg, i2_end, &
                            out%a, out%i, out%j )
            else ! x is complex
               out%data_type = MF_DT_SP_CMPLX
               allocate( out%z(nnz) )

               call submat_cmplx( x%z, x%i, x%j, i1_beg, i1_end, i2_beg, i2_end, &
                                  out%z, out%i, out%j )
            end if
            out%row_sorted = x%row_sorted
         else
            call PrintMessage( "mfGet", "E",                            &
                               "extracting a sub-matrix from a sparse mfArray:", &
                               "non contiguous indexes : not yet available!" )
            go to 99
         end if
      end if

      if( mf_phys_units ) then
         out%units(:) = x%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( x )

      call msAutoRelease( x )

#endif
   end function mfGet_section_vec_vec
!_______________________________________________________________________
!
   function mfGet_section_vseq_vseq( x, seq1, seq2 ) result( out )

      type(mfArray) :: x
      type(seq_def) :: seq1, seq2
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! extracts x(seq1,seq2) from an mfArray
      ! each argument seq is of the form : i1 .to. i2 .by. i3
      ! which is equivalent to : i1:i2:i3 (f90 section notation)
      ! (for sparse storage, they must be continuous and in increasing
      ! order)
      !
      ! returns an mfArray of shape equal to (size(seq1),size(seq2))

      integer, pointer :: i1(:) => null(), i2(:) => null()
      integer :: i1_beg, i1_end, n1,                                    &
                 i2_beg, i2_end, n2,                                    &
                 i, nnz, nrow, ncol
      logical :: contiguous

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x )

      if( mfIsEmpty(x) ) then
         call PrintMessage( "mfGet", "I",                               &
                            "the mfArray x is empty." )
         go to 99
      end if

      nrow = x%shape(1)
      if( seq1%start_1_EndIndex ) then
         seq1%start_1 = nrow + seq1%start_1
      end if
      if( seq1%end_1_EndIndex ) then
         seq1%end_1 = nrow + seq1%end_1
      end if
      if( seq1%start_2_EndIndex ) then
         seq1%start_2 = nrow + seq1%start_2
      end if
      if( seq1%end_2_EndIndex ) then
         seq1%end_2 = nrow + seq1%end_2
      end if
      if( seq1%but_EndIndex ) then
         seq1%but = nrow + seq1%but
      end if
      call build_int_seq( i1, seq1, "mfGet" )

      ncol = x%shape(2)
      if( seq2%start_1 <= 0 ) then
         if( seq2%start_1_EndIndex ) then
            seq2%start_1 = ncol + seq2%start_1
         else
            call PrintMessage( "mfGet", "E",                            &
                               "cannot use negative indexes!" )
            go to 99
         end if
      end if
      if( seq2%end_1_EndIndex ) then
         seq2%end_1 = ncol + seq2%end_1
      end if
      if( seq2%start_2_EndIndex ) then
         seq2%start_2 = ncol + seq2%start_2
      end if
      if( seq2%end_2_EndIndex ) then
         seq2%end_2 = ncol + seq2%end_2
      end if
      if( seq2%but_EndIndex ) then
         seq2%but = ncol + seq2%but
      end if
      call build_int_seq( i2, seq2, "mfGet" )

      if( minval(i1) < 1 .or. x%shape(1) < maxval(i1) ) then
         call PrintMessage( "mfGet", "E",                               &
                            "row indexes are out of range!" )
         go to 99
      end if

      if( minval(i2) < 1 .or. x%shape(2) < maxval(i2) ) then
         call PrintMessage( "mfGet", "E",                               &
                            "col indexes are out of range!" )
         go to 99
      end if

      out%data_type = x%data_type
      out%shape = [ size(i1), size(i2) ]
      if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
         allocate( out%double(size(i1),size(i2)) )

         out%double(:,:) = x%double(i1,i2)
      else if( x%data_type == MF_DT_CMPLX ) then
         allocate( out%cmplx(size(i1),size(i2)) )

         out%cmplx(:,:) = x%cmplx(i1,i2)
      else ! x is sparse
         ! test if i1(:) and i2(:) contain contiguous indexes
         ! in increasing order
         contiguous = .true.
         n1 = size(i1)
         do i = 2, n1
            if( i1(i) /= i1(i-1) + 1 ) then
               contiguous = .false.
               exit
            end if
         end do
         n2 = size(i2)
         do i = 2, n2
            if( i2(i) /= i2(i-1) + 1 ) then
               contiguous = .false.
               exit
            end if
         end do
         if( contiguous ) then
            i1_beg = i1(1)
            i1_end = i1(n1)
            i2_beg = i2(1)
            i2_end = i2(n2)
            call submat_nnz( x%i, x%j, i1_beg, i1_end, i2_beg, i2_end,  &
                             nnz )
            allocate( out%i(nnz) )

            allocate( out%j(n2+1) )

            if( mfIsReal(x) ) then
               out%data_type = MF_DT_SP_DBLE
               allocate( out%a(nnz) )

               call submat( x%a, x%i, x%j, i1_beg, i1_end, i2_beg, i2_end, &
                            out%a, out%i, out%j )
            else ! x is complex
               out%data_type = MF_DT_SP_CMPLX
               allocate( out%z(nnz) )

               call submat_cmplx( x%z, x%i, x%j, i1_beg, i1_end, i2_beg, i2_end, &
                                  out%z, out%i, out%j )
            end if
            out%row_sorted = x%row_sorted
         else
            call PrintMessage( "mfGet", "E",                            &
                               "extracting a sub-matrix from a sparse mfArray:", &
                               "non contiguous indexes : not yet available!" )
            go to 99
         end if
      end if

      if( mf_phys_units ) then
         out%units(:) = x%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      if( associated(i1) ) then
#ifndef _TRACE_MEM_ALLOC
         deallocate( i1 )
#else
         call mf_deallocate( array=i1,                                  &
                             file="Get.inc", line="???",                &
                             symb="i1", unit="mfGet_section_vseq_vseq" )
#endif
      end if

      if( associated(i2) ) then
#ifndef _TRACE_MEM_ALLOC
         deallocate( i2 )
#else
         call mf_deallocate( array=i2,                                  &
                             file="Get.inc", line="???",                &
                             symb="i2", unit="mfGet_section_vseq_vseq" )
#endif
      end if

      call msFreeArgs( x )
      call msAutoRelease( x )

#endif
   end function mfGet_section_vseq_vseq
!_______________________________________________________________________
!
   function mfGet_section_int_vseq( x, i1, seq2 ) result( out )

      type(mfArray) :: x
      integer, intent(in) :: i1
      type(seq_def) :: seq2
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! extracts x(i1,seq2) from an mfArray
      ! seq2 is of the form : i1 .to. i2 .by. i3
      ! which is equivalent to : i1:i2:i3 (f90 section notation)
      !
      ! returns a dense mfArray of shape equal to (1,size(seq2))

      integer, pointer :: i2(:) => null()
      real(kind=MF_DOUBLE), allocatable :: xtmp(:)
      complex(kind=MF_DOUBLE), allocatable :: ztmp(:)
      integer :: nrow, ncol

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x )

      if( mfIsEmpty(x) ) then
         call PrintMessage( "mfGet", "I",                               &
                            "the mfArray x is empty." )
         go to 99
      end if

      nrow = x%shape(1)
      ncol = x%shape(2)
      if( seq2%start_1_EndIndex ) then
         seq2%start_1 = ncol + seq2%start_1
      end if
      if( seq2%end_1_EndIndex ) then
         seq2%end_1 = ncol + seq2%end_1
      end if
      if( seq2%start_2_EndIndex ) then
         seq2%start_2 = ncol + seq2%start_2
      end if
      if( seq2%end_2_EndIndex ) then
         seq2%end_2 = ncol + seq2%end_2
      end if
      if( seq2%but_EndIndex ) then
         seq2%but = ncol + seq2%but
      end if
      call build_int_seq( i2, seq2, "mfGet" )

      if( i1 < 1 .or. nrow < i1 ) then
         call PrintMessage( "mfGet", "E",                               &
                            "row index is out of range!" )
         go to 99
      end if

      if( minval(i2) < 1 .or. ncol < maxval(i2) ) then
         call PrintMessage( "mfGet", "E",                               &
                            "col indexes are out of range!" )
         go to 99
      end if

      out%data_type = x%data_type
      out%shape = [ 1, size(i2) ]
      if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
         allocate( out%double(1,size(i2)) )

         out%double(1,:) = x%double(i1,i2)
      else if( x%data_type == MF_DT_CMPLX ) then
         allocate( out%cmplx(1,size(i2)) )

         out%cmplx(1,:) = x%cmplx(i1,i2)
      else ! x is sparse
!### TODO 2: pour l'instant l'extraction n'est pas optimisée : on extrait
!          les valeurs de toute la ligne dans un vecteur tempo puis
!          on recopie...
         if( x%data_type == MF_DT_SP_DBLE ) then
            allocate( xtmp(x%shape(2)) )
            call xtrrow( x%shape(2), x%a, x%i, x%j, i1, xtmp )
            out%data_type = MF_DT_DBLE
            allocate( out%double(1,size(i2)) )

            out%double(1,:) = xtmp(i2)
         else if( x%data_type == MF_DT_SP_CMPLX ) then
            allocate( ztmp(x%shape(2)) )
            call xtrrow_cmplx( x%shape(2), x%z, x%i, x%j, i1, ztmp )
            out%data_type = MF_DT_CMPLX
            allocate( out%cmplx(1,size(i2)) )

            out%cmplx(1,:) = ztmp(i2)
         end if
      end if

      if( mf_phys_units ) then
         out%units(:) = x%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      if( associated(i2) ) then
#ifndef _TRACE_MEM_ALLOC
         deallocate( i2 )
#else
         call mf_deallocate( array=i2,                                  &
                             file="Get.inc", line="???",                &
                             symb="i2", unit="mfGet_section_vseq_vseq" )
#endif
      end if

      call msFreeArgs( x )
      call msAutoRelease( x )

#endif
   end function mfGet_section_int_vseq
!_______________________________________________________________________
!
   function mfGet_section_vseq_int( x, seq1, i2 ) result( out )

      type(mfArray) :: x
      type(seq_def) :: seq1
      integer, intent(in) :: i2
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! extracts x(seq1,i2) from an mfArray
      ! seq1 is of the form : i1 .to. i2 .by. i3
      ! which is equivalent to : i1:i2:i3 (f90 section notation)
      !
      ! returns a dense mfArray of shape equal to (size(seq1),1)

      integer, pointer :: i1(:) => null()
      real(kind=MF_DOUBLE), allocatable :: xtmp(:)
      complex(kind=MF_DOUBLE), allocatable :: ztmp(:)
      integer :: nrow, ncol

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x )

      if( mfIsEmpty(x) ) then
         call PrintMessage( "mfGet", "I",                               &
                            "the mfArray x is empty." )
         go to 99
      end if

      nrow = x%shape(1)
      ncol = x%shape(2)
      if( seq1%start_1_EndIndex ) then
         seq1%start_1 = nrow + seq1%start_1
      end if
      if( seq1%end_1_EndIndex ) then
         seq1%end_1 = nrow + seq1%end_1
      end if
      if( seq1%start_2_EndIndex ) then
         seq1%start_2 = nrow + seq1%start_2
      end if
      if( seq1%end_2_EndIndex ) then
         seq1%end_2 = nrow + seq1%end_2
      end if
      if( seq1%but_EndIndex ) then
         seq1%but = nrow + seq1%but
      end if
      call build_int_seq( i1, seq1, "mfGet" )

      if( minval(i1) < 1 .or. nrow < maxval(i1) ) then
         call PrintMessage( "mfGet", "E",                               &
                            "row indexes are out of range!" )
         go to 99
      end if

      if( i2 < 1 .or. ncol < i2 ) then
         call PrintMessage( "mfGet", "E",                               &
                            "col index is out of range!" )
         go to 99
      end if

      out%data_type = x%data_type
      out%shape = [ size(i1), 1 ]
      if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
         allocate( out%double(size(i1),1) )

         out%double(:,1) = x%double(i1,i2)
      else if( x%data_type == MF_DT_CMPLX ) then
         allocate( out%cmplx(size(i1),1) )

         out%cmplx(:,1) = x%cmplx(i1,i2)
      else ! x is sparse
!### TODO 2: pour l'instant l'extraction n'est pas optimisée : on extrait
!          les valeurs de toute la colonne dans un vecteur tempo puis
!          on recopie...
         if( x%data_type == MF_DT_SP_DBLE ) then
            allocate( xtmp(x%shape(1)) )
            call xtrcol( x%shape(1), x%a, x%i, x%j, i2, xtmp )
            out%data_type = MF_DT_DBLE
            allocate( out%double(size(i1),1) )

            out%double(:,1) = xtmp(i1)
         else if( x%data_type == MF_DT_SP_CMPLX ) then
            allocate( ztmp(x%shape(1)) )
            call xtrcol_cmplx( x%shape(1), x%z, x%i, x%j, i2, ztmp )
            out%data_type = MF_DT_CMPLX
            allocate( out%cmplx(size(i1),1) )

            out%cmplx(:,1) = ztmp(i1)
         end if
      end if

      if( mf_phys_units ) then
         out%units(:) = x%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      if( associated(i1) ) then
#ifndef _TRACE_MEM_ALLOC
         deallocate( i1 )
#else
         call mf_deallocate( array=i1,                                  &
                             file="Get.inc", line="???",                &
                             symb="i1", unit="mfGet_section_vseq_vseq" )
#endif
      end if

      call msFreeArgs( x )
      call msAutoRelease( x )

#endif
   end function mfGet_section_vseq_int
!_______________________________________________________________________
!
   function mfGet_section_array_array( x, a1, a2 ) result( out )

      type(mfArray) :: x
      type(mfArray) :: a1, a2
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! extracts x(section1,section2) from an mfArray
      ! section1 (resp. section2) takes its definition from the
      ! vector-like mfArray a1 (resp. a2)
      !
      ! returns an mfArray of shape equal to (size(a1),size(a2))

      ! extended to the case where a1 or a2 is equal to MF_COLON
      !
      ! for sparse x, it requires increasing contiguous indexes,
      ! in both dimensions, but only when a1 /= MF_COLON.
      !
! la contrainte suivante est enlevée (voir plus bas)
!!      ! for dense x, a1 and a2 may be rank-2 mfArrays, but both at the
!!      ! same time.

      ! extended to the case where a1 is an EndIndex (which is
      ! either the MF_END parameter itself, or an expression
      ! involving integers and MF_END, like 'MF_END-1').

      integer, allocatable :: i1(:), i2(:)
      integer :: i1_beg, i1_end, nrow,                                  &
                 i2_beg, i2_end, ncol,                                  &
                 i, nnz
      logical :: contiguous
      logical :: all_cols_in_order

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x, a1, a2 )

      if( mfIsEmpty(x) .or. mfIsEqual(a1,MF_EMPTY) .or. mfIsEqual(a2,MF_EMPTY) ) then
         go to 99
      end if

      ! hereafter, distinguishing between MF_COLON and true mfArray

      if( mfIsEqual(a1,MF_COLON) .and. mfIsEqual(a2,MF_COLON) ) then

         ! trivial case !
         call PrintMessage( "mfGet", "I",                               &
                            "a1 and a2 are both equal to MF_COLON." )
         out = x
         out%status_temporary = .true.
         go to 99

      else if( mfIsEqual(a1,MF_COLON) ) then

         if( isEndIndex(a2) ) then

            allocate( i2(1) )

            ncol = x%shape(2)
            i2(1) = ncol + a2%crc
            if( i2(1) < 1 .or. ncol < i2(1) ) then
               call PrintMessage( "mfGet", "E",                         &
                                  "End-index is out of range!" )
               go to 99
            end if

         else

            if( .not. mfIsReal(a2) ) then
               call PrintMessage( "mfGet", "E",                         &
                                  "a2 must be of type real!" )
               go to 99
            end if

            ! usually, a2 is a vector; when a2 is a rank-2 mfArray,
            ! x must be dense
            if( mfIsMatrix(a2) ) then
               if( mfIsSparse(x) ) then
                  call PrintMessage( "mfGet", "E",                      &
                                     "when 'a2' is a rank-2 array,",    &
                                     "'x' must have dense storage!" )
               end if
            end if

            if( any( int(a2%double) /= a2%double) ) then
               call PrintMessage( "mfGet", "E",                         &
                                  "a2 must contain integer values!" )
               go to 99
            end if

            if( minval(a2%double) < 1 .or. x%shape(2) < maxval(a2%double) ) then
               call PrintMessage( "mfGet", "E",                         &
                                  "col indexes are out of range!" )
               go to 99
            end if

            allocate( i2(size(a2%double)) )

            i2 = reshape( int(a2%double), [size(a2%double)] )

         end if

         out%data_type = x%data_type
         out%shape = [ x%shape(1), size(i2) ]

         if( mfIsSparse(x) ) then

            ncol = x%shape(2)
            ! test if i2 == [1,2,...,ncol]
            if( size(i2) == ncol ) then
               if( all( i2 == [ (i,i=1,ncol) ] ) ) then
                  all_cols_in_order = .true.
               else
                  all_cols_in_order = .false.
               end if
            else
               all_cols_in_order = .false.
            end if
            if( all_cols_in_order ) then
               out = x
            else
               call nnzxtrcols( 1, size(i2), x%i, x%j, i2, nnz )
               if( x%data_type == MF_DT_SP_DBLE ) then
                  allocate( out%a(nnz) )

                  allocate( out%i(nnz) )

                  allocate( out%j(ncol+1) )

                  call xtrcols( 1, size(i2), x%a, x%i, x%j,             &
                                out%a, out%i, out%j, i2 )
               else ! x%data_type == MF_DT_SP_CMPLX
                  allocate( out%z(nnz) )

                  allocate( out%i(nnz) )

                  allocate( out%j(ncol+1) )

                  call xtrcols_cmplx( 1, size(i2), x%z, x%i, x%j,       &
                                      out%z, out%i, out%j, i2 )
               end if
               out%row_sorted = x%row_sorted
            end if

         else

            if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
               allocate( out%double(out%shape(1),out%shape(2)) )

               out%double(:,:) = x%double(:,i2)
            else if( x%data_type == MF_DT_CMPLX ) then
               allocate( out%cmplx(out%shape(1),out%shape(2)) )

               out%cmplx(:,:) = x%cmplx(:,i2)
            end if

         end if

      else if( mfIsEqual(a2,MF_COLON) ) then

         if( isEndIndex(a1) ) then

            allocate( i1(1) )

            nrow = x%shape(1)
            i1(1) = nrow + a1%crc
            if( i1(1) < 1 .or. nrow < i1(1) ) then
               call PrintMessage( "mfGet", "E",                         &
                                  "End-index is out of range!" )
               go to 99
            end if

         else

            if( .not. mfIsReal(a1) ) then
               call PrintMessage( "mfGet", "E",                         &
                                  "a1 must be of type real!" )
               go to 99
            end if

            ! usually, a1 is a vector; when a1 is a rank-2 mfArray,
            ! x must be dense
            if( mfIsMatrix(a1) ) then
               if( mfIsSparse(x) ) then
                  call PrintMessage( "mfGet", "E",                      &
                                     "when 'a1' is a rank-2 array,",    &
                                     "'x' must have dense storage!" )
               end if
            end if

            if( any( int(a1%double) /= a1%double) ) then
               call PrintMessage( "mfGet", "E",                         &
                                  "a1 must contain integer values!" )
               go to 99
            end if

            if( minval(a1%double) < 1 .or. x%shape(1) < maxval(a1%double) ) then
               call PrintMessage( "mfGet", "E",                         &
                                  "row indexes are out of range!" )
               go to 99
            end if

            allocate( i1(size(a1%double)) )

            i1 = reshape( int(a1%double), [size(a1%double)] )

         end if

         out%data_type = x%data_type
         out%shape = [ size(i1), x%shape(2) ]
         if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
            allocate( out%double(out%shape(1),out%shape(2)) )

            out%double(:,:) = x%double(i1,:)
         else if( x%data_type == MF_DT_CMPLX ) then
            allocate( out%cmplx(out%shape(1),out%shape(2)) )

            out%cmplx(:,:) = x%cmplx(i1,:)
         else  ! x is sparse

            ncol = x%shape(2)

            ! test if i1(:) contain contiguous indexes in increasing order
            contiguous = .true.
            nrow = size(i1)
            do i = 2, nrow
               if( i1(i) /= i1(i-1) + 1 ) then
                  contiguous = .false.
                  exit
               end if
            end do
            if( contiguous ) then
               i1_beg = i1(1)
               i1_end = i1(nrow)
               i2_beg = 1
               i2_end = ncol
               call submat_nnz( x%i, x%j, i1_beg, i1_end, i2_beg, i2_end, &
                                nnz )
               allocate( out%i(nnz) )

               allocate( out%j(ncol+1) )

               if( mfIsReal(x) ) then
                  out%data_type = MF_DT_SP_DBLE
                  allocate( out%a(nnz) )

                  call submat( x%a, x%i, x%j, i1_beg, i1_end, i2_beg, i2_end, &
                               out%a, out%i, out%j )
               else ! x is complex
                  out%data_type = MF_DT_SP_CMPLX
                  allocate( out%z(nnz) )

                  call submat_cmplx( x%z, x%i, x%j, i1_beg, i1_end, i2_beg, i2_end, &
                                     out%z, out%i, out%j )
               end if
               out%row_sorted = x%row_sorted
            else
               call PrintMessage( "mfGet", "E",                         &
                                  "extracting a section from a sparse mfArray:", &
                                  "non contiguous indexes : not yet available!" )
               go to 99
            end if

         end if

      else ! neither a1 or a2 is MF_COLON

! la contrainte qui suit ne correspond à rien ! ? ? ?
! puisque a1 et a2 sont changés en long-column-vector, quel que
! soit leur rang !

         ! usually, a1 and a2 are vectors; the only exception is when
         ! a1 and a2 are both rank-2 mfArrays and x is dense...
!!         if( mfIsMatrix(a1) ) then
!!            if( mfIsMatrix(a2) ) then
!!               if( mfIsSparse(x) ) then
!!                  call PrintMessage( "mfGet", "E",                      &
!!                                     "when 'a1' and 'a2' are both rank-2 arrays,", &
!!                                     "'x' must have dense storage!" )
!!               end if
!!            else
!!               call PrintMessage( "mfGet", "E",                         &
!!                                  "'a1' and 'a2' must have the same rank!" )
!!               go to 99
!!            end if
!!         else
!!            if( mfIsMatrix(a2) ) then
!!               call PrintMessage( "mfGet", "E",                         &
!!                                  "'a1' and 'a2' must have the same rank!" )
!!               go to 99
!!            end if
!!         end if

         if( isEndIndex(a1) ) then

            allocate( i1(1) )

            nrow = x%shape(1)
            i1(1) = nrow + a1%crc

            if( i1(1) < 1 .or. nrow < i1(1) ) then
               call PrintMessage( "mfGet", "E",                         &
                                  "End-index a1 is out of range!" )
               go to 99
            end if

         else

            if( .not. mfIsReal(a1) ) then
               call PrintMessage( "mfGet", "E",                         &
                                  "a1 must be of type real!" )
               go to 99
            end if

            if( any( int(a1%double) /= a1%double) ) then
               call PrintMessage( "mfGet", "E",                         &
                                  "a1 must contain integer values!" )
               go to 99
            end if

            if( minval(a1%double) < 1 .or. x%shape(1) < maxval(a1%double) ) then
               call PrintMessage( "mfGet", "E",                         &
                                  "row indexes are out of range!" )
               go to 99
            end if

            allocate( i1(size(a1%double)) )

            i1 = reshape( int(a1%double), [size(a1%double)] )

         end if

         if( isEndIndex(a2) ) then

            allocate( i2(1) )

            ncol = x%shape(2)
            i2(1) = ncol + a2%crc

            if( i2(1) < 1 .or. ncol < i2(1) ) then
               call PrintMessage( "mfGet", "E",                         &
                                  "End-index a2 is out of range!" )
               go to 99
            end if

         else

            if( .not. mfIsReal(a2) ) then
               call PrintMessage( "mfGet", "E",                         &
                                  "a2 must be of type real!" )
               go to 99
            end if

            if( any( int(a2%double) /= a2%double) ) then
               call PrintMessage( "mfGet", "E",                         &
                                  "a2 must contain integer values!" )
               go to 99
            end if

            if( minval(a2%double) < 1 .or. x%shape(2) < maxval(a2%double) ) then
               call PrintMessage( "mfGet", "E",                         &
                                  "col indexes are out of range!" )
               go to 99
            end if

            allocate( i2(size(a2%double)) )

            i2 = reshape( int(a2%double), [size(a2%double)] )

         end if

         out%data_type = x%data_type
         out%shape = [ size(i1), size(i2) ]
         if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
            allocate( out%double(size(i1),size(i2)) )

            out%double(:,:) = x%double(i1,i2)
         else if( x%data_type == MF_DT_CMPLX ) then
            allocate( out%cmplx(size(i1),size(i2)) )

            out%cmplx(:,:) = x%cmplx(i1,i2)
         else ! x is sparse
            ! test if i1(:) and i2(:) contain contiguous indexes
            ! in increasing order
            contiguous = .true.
            nrow = size(i1)
            do i = 2, nrow
               if( i1(i) /= i1(i-1) + 1 ) then
                  contiguous = .false.
                  exit
               end if
            end do
            ncol = size(i2)
            do i = 2, ncol
               if( i2(i) /= i2(i-1) + 1 ) then
                  contiguous = .false.
                  exit
               end if
            end do
            if( contiguous ) then
               i1_beg = i1(1)
               i1_end = i1(nrow)
               i2_beg = i2(1)
               i2_end = i2(ncol)
               call submat_nnz( x%i, x%j, i1_beg, i1_end, i2_beg, i2_end, &
                                nnz )
               allocate( out%i(nnz) )

               allocate( out%j(ncol+1) )

               if( mfIsReal(x) ) then
                  out%data_type = MF_DT_SP_DBLE
                  allocate( out%a(nnz) )

                  call submat( x%a, x%i, x%j, i1_beg, i1_end, i2_beg, i2_end, &
                               out%a, out%i, out%j )
               else ! x is complex
                  out%data_type = MF_DT_SP_CMPLX
                  allocate( out%z(nnz) )

                  call submat_cmplx( x%z, x%i, x%j, i1_beg, i1_end, i2_beg, i2_end, &
                                     out%z, out%i, out%j )
               end if
               out%row_sorted = x%row_sorted
            else
               call PrintMessage( "mfGet", "E",                         &
                                  "extracting a section from a sparse mfArray:", &
                                  "non contiguous indexes : not yet available!" )
               go to 99
            end if
         end if

      end if

      if( mf_phys_units ) then
         out%units(:) = x%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( x, a1, a2 )
      call msAutoRelease( x, a1, a2 )

#endif
   end function mfGet_section_array_array
!_______________________________________________________________________
!
   function mfGet_section_array_vec( x, a1, i2 ) result( out )

      type(mfArray) :: x, a1
      integer, intent(in) :: i2(:)
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! if a1==MF_COLON : extracts selected columns from an mfArray;
      !                   columns indexes comes from the vector i2;
      ! else : extracts a sub-matrix via indexes in a1 and i2.
      !
      ! for sparse x, it requires increasing contiguous indexes,
      ! in both dimensions, but only when a1 /= MF_COLON.

      ! extended to the case where 'a1' contains an EndIndex (which is
      ! either the MF_END parameter itself, or an expression
      ! involving integers and MF_END, like 'MF_END-1').

      integer, allocatable :: i1(:)
      integer :: i1_beg, i1_end, nrow,                                  &
                 i2_beg, i2_end, ncol,                                  &
                 i, nnz
      logical :: contiguous
      logical :: all_cols_in_order

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x, a1 )

      if( mfIsEmpty(x) .or. mfIsEqual(a1,MF_EMPTY) ) then
         go to 99
      end if

      if( minval(i2) < 1 .or. x%shape(2) < maxval(i2) ) then
         call PrintMessage( "mfGet", "E",                               &
                            "col indexes are out of range!" )
         go to 99
      end if

      if( mfIsEqual(a1,MF_COLON) ) then

         allocate( i1(x%shape(1)) )

         i1 = [ (i, i = 1, x%shape(1)) ]

         out%data_type = x%data_type
         out%shape = [ size(i1), size(i2) ]
         if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
            allocate( out%double(size(i1),size(i2)) )

            out%double(:,:) = x%double(i1,i2)
         else if( x%data_type == MF_DT_CMPLX ) then
            allocate( out%cmplx(size(i1),size(i2)) )

            out%cmplx(:,:) = x%cmplx(i1,i2)
         else if( mfIsSparse(x) ) then

            ncol = x%shape(2)
            ! test if i2 == [1,2,...,ncol]
            if( size(i2) == ncol ) then
               if( all( i2 == [ (i,i=1,ncol) ] ) ) then
                  all_cols_in_order = .true.
               else
                  all_cols_in_order = .false.
               end if
            else
               all_cols_in_order = .false.
            end if
            if( all_cols_in_order ) then
               out = x
            else
               call nnzxtrcols( 1, size(i2), x%i, x%j, i2, nnz )
               if( x%data_type == MF_DT_SP_DBLE ) then
                  allocate( out%a(nnz) )

                  allocate( out%i(nnz) )

                  allocate( out%j(ncol+1) )

                  call xtrcols( 1, size(i2), x%a, x%i, x%j,             &
                                out%a, out%i, out%j, i2 )
               else ! x%data_type == MF_DT_SP_CMPLX
                  allocate( out%z(nnz) )

                  allocate( out%i(nnz) )

                  allocate( out%j(ncol+1) )

                  call xtrcols_cmplx( 1, size(i2), x%z, x%i, x%j,       &
                                      out%z, out%i, out%j, i2 )
               end if
               out%row_sorted = x%row_sorted
            end if
         end if

      else if( isEndIndex(a1) ) then

         allocate( i1(1) )

         nrow = x%shape(1)
         i1(1) = nrow + a1%crc
         call msAssign( out, mfGet_section_vec_vec(x,i1,i2) )

      else ! a1 is only a subpart of the row's set

         if( a1%data_type /= MF_DT_DBLE ) then
            call PrintMessage( "mfGet", "E",                            &
                               "a1 must be of type real!" )
            go to 99
         end if

         if( any( int(a1%double) /= a1%double) ) then
            call PrintMessage( "mfGet", "E",                            &
                               "a1 must contain integer values!" )
            go to 99
         end if

         if( a1%shape(1) /= 1 .and. a1%shape(2) /= 1 ) then
            call PrintMessage( "mfGet", "E",                            &
                               "mfArray indexes 'a1' must be of rank one!" )
            go to 99
         end if

         if( minval(a1%double) < 1 .or. x%shape(1) < maxval(a1%double) ) then
            call PrintMessage( "mfGet", "E",                            &
                               "row indexes are out of range!" )
            go to 99
         end if

         allocate( i1(size(a1%double)) )

         i1 = reshape( int(a1%double), [size(a1%double)] )

         out%data_type = x%data_type
         out%shape = [ size(i1), size(i2) ]
         if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
            allocate( out%double(size(i1),size(i2)) )

            out%double(:,:) = x%double(i1,i2)
         else if( x%data_type == MF_DT_CMPLX ) then
            allocate( out%cmplx(size(i1),size(i2)) )

            out%cmplx(:,:) = x%cmplx(i1,i2)
         else ! x is sparse
            ! test if i1(:) and i2(:) contain contiguous indexes
            ! in increasing order
            contiguous = .true.
            nrow = size(i1)
            do i = 2, nrow
               if( i1(i) /= i1(i-1) + 1 ) then
                  contiguous = .false.
                  exit
               end if
            end do
            ncol = size(i2)
            do i = 2, ncol
               if( i2(i) /= i2(i-1) + 1 ) then
                  contiguous = .false.
                  exit
               end if
            end do
            if( contiguous ) then
               i1_beg = i1(1)
               i1_end = i1(nrow)
               i2_beg = i2(1)
               i2_end = i2(ncol)
               call submat_nnz( x%i, x%j, i1_beg, i1_end, i2_beg, i2_end, &
                                nnz )
               allocate( out%i(nnz) )

               allocate( out%j(ncol+1) )

               if( mfIsReal(x) ) then
                  out%data_type = MF_DT_SP_DBLE
                  allocate( out%a(nnz) )

                  call submat( x%a, x%i, x%j, i1_beg, i1_end, i2_beg, i2_end, &
                               out%a, out%i, out%j )
               else ! x is complex
                  out%data_type = MF_DT_SP_CMPLX
                  allocate( out%z(nnz) )

                  call submat_cmplx( x%z, x%i, x%j, i1_beg, i1_end, i2_beg, i2_end, &
                                     out%z, out%i, out%j )
               end if
               out%row_sorted = x%row_sorted
            else
               call PrintMessage( "mfGet", "E",                         &
                                  "extracting a section from a sparse mfArray:", &
                                  "non contiguous indexes : not yet available!" )
               go to 99
            end if
         end if

      end if

      if( mf_phys_units ) then
         out%units(:) = x%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( x, a1 )

      call msAutoRelease( x, a1 )

#endif
   end function mfGet_section_array_vec
!_______________________________________________________________________
!
   function mfGet_section_vec_array( x, i1, a2 ) result( out )

      type(mfArray) :: x
      integer, intent(in) :: i1(:)
      type(mfArray) :: a2
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! if a2==MF_COLON : extracts selected rows from an mfArray
      !                   rows indexes comes from the vector i1;
      ! else : extracts a sub-matrix via indexes in i1 and a2.
      !
      ! for sparse x, it requires increasing contiguous indexes,
      ! in both dimensions.

      ! extended to the case where 'a2' contains an EndIndex (which is
      ! either the MF_END parameter itself, or an expression
      ! involving integers and MF_END, like 'MF_END-1').

      integer, allocatable :: i2(:)
      integer :: i1_beg, i1_end, nrow,                                  &
                 i2_beg, i2_end, ncol,                                  &
                 i, nnz
      logical :: contiguous

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x, a2 )

      if( mfIsEmpty(x) .or. mfIsEqual(a2,MF_EMPTY) ) then
         go to 99
      end if

      if( minval(i1) < 1 .or. x%shape(1) < maxval(i1) ) then
         call PrintMessage( "mfGet", "E",                               &
                            "row indexes are out of range!" )
         go to 99
      end if

      if( mfIsEqual(a2,MF_COLON) ) then

         allocate( i2(x%shape(2)) )

         i2 = [ (i, i = 1, x%shape(2)) ]

      else if( isEndIndex(a2) ) then

         allocate( i2(1) )

         ncol = x%shape(2)
         i2(1) = ncol + a2%crc
         call msAssign( out, mfGet_section_vec_vec(x,i1,i2) )
         out%status_temporary = .true.
         go to 99

      else

         if( a2%data_type /= MF_DT_DBLE ) then
            call PrintMessage( "mfGet", "E",                            &
                               "a2 must be of type real!" )
            go to 99
         end if

         if( any( int(a2%double) /= a2%double) ) then
            call PrintMessage( "mfGet", "E",                            &
                               "a2 must contain integer values!" )
            go to 99
         end if

         if( a2%shape(1) /= 1 .and. a2%shape(2) /= 1 ) then
            call PrintMessage( "mfGet", "E",                            &
                               "mfArray 'a2' must be a vector!" )
            go to 99
         end if

         if( minval(a2%double) < 1 .or. x%shape(2) < maxval(a2%double) ) then
            call PrintMessage( "mfGet", "E",                            &
                               "col indexes are out of range!" )
            go to 99
         end if

         allocate( i2(size(a2%double)) )

         i2 = reshape( int(a2%double), [size(a2%double)] )

      end if

      out%data_type = x%data_type
      out%shape = [ size(i1), size(i2) ]
      if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
         allocate( out%double(size(i1),size(i2)) )

         out%double(:,:) = x%double(i1,i2)
      else if( x%data_type == MF_DT_CMPLX ) then
         allocate( out%cmplx(size(i1),size(i2)) )

         out%cmplx(:,:) = x%cmplx(i1,i2)
      else ! x is sparse
         ! test if i1(:) and i2(:) contain contiguous indexes
         ! in increasing order
         contiguous = .true.
         nrow = size(i1)
         do i = 2, nrow
            if( i1(i) /= i1(i-1) + 1 ) then
               contiguous = .false.
               exit
            end if
         end do
         ncol = size(i2)
         do i = 2, ncol
            if( i2(i) /= i2(i-1) + 1 ) then
               contiguous = .false.
               exit
            end if
         end do
         if( contiguous ) then
            i1_beg = i1(1)
            i1_end = i1(nrow)
            i2_beg = i2(1)
            i2_end = i2(ncol)
            call submat_nnz( x%i, x%j, i1_beg, i1_end, i2_beg, i2_end,  &
                             nnz )
            allocate( out%i(nnz) )

            allocate( out%j(ncol+1) )

            if( mfIsReal(x) ) then
               out%data_type = MF_DT_SP_DBLE
               allocate( out%a(nnz) )

               call submat( x%a, x%i, x%j, i1_beg, i1_end, i2_beg, i2_end, &
                            out%a, out%i, out%j )
            else ! x is complex
               out%data_type = MF_DT_SP_CMPLX
               allocate( out%z(nnz) )

               call submat_cmplx( x%z, x%i, x%j, i1_beg, i1_end, i2_beg, i2_end, &
                                  out%z, out%i, out%j )
            end if
            out%row_sorted = x%row_sorted
         else
            call PrintMessage( "mfGet", "E",                            &
                               "extracting a section from a sparse mfArray:", &
                               "non contiguous indexes : not yet available!" )
            go to 99
         end if
      end if

      if( mf_phys_units ) then
         out%units(:) = x%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( x, a2 )

      call msAutoRelease( x, a2 )

#endif
   end function mfGet_section_vec_array
!_______________________________________________________________________
!
   function mfGet_section_array_vseq( x, a1, seq ) result( out )

      type(mfArray) :: x, a1
      type(seq_def) :: seq
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! if a1==MF_COLON : extracts selected columns from an mfArray
      !                   columns indexes comes from the sequence seq;
      ! else : extracts a sub-matrix via indexes in a1 and seq
      !
      ! for sparse x, it requires increasing contiguous indexes,
      ! in both dimensions, but only when a1 /= MF_COLON.

      ! extended to the case where 'a1' contains an EndIndex (which is
      ! either the MF_END parameter itself, or an expression
      ! involving integers and MF_END, like 'MF_END-1').

      integer, allocatable :: i1(:)
      integer, pointer :: i2(:) => null()
      integer :: i1_beg, i1_end, nrow,                                  &
                 i2_beg, i2_end, ncol,                                  &
                 i, nnz
      logical :: contiguous
      logical :: all_cols_in_order

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x, a1 )

      if( mfIsEmpty(x) .or. mfIsEqual(a1,MF_EMPTY) ) then
         go to 99
      end if

      nrow = x%shape(1)
      ncol = x%shape(2)
      if( seq%start_1_EndIndex ) then
         seq%start_1 = ncol + seq%start_1
      end if
      if( seq%end_1_EndIndex ) then
         seq%end_1 = ncol + seq%end_1
      end if
      if( seq%start_2_EndIndex ) then
         seq%start_2 = ncol + seq%start_2
      end if
      if( seq%end_2_EndIndex ) then
         seq%end_2 = ncol + seq%end_2
      end if
      if( seq%but_EndIndex ) then
         seq%but = ncol + seq%but
      end if
      call build_int_seq( i2, seq, "mfGet" )

      if( minval(i2) < 1 .or. ncol < maxval(i2) ) then
         call PrintMessage( "mfGet", "E",                               &
                            "col indexes are out of range!" )
         go to 99
      end if

      if( mfIsEqual(a1,MF_COLON) ) then

         allocate( i1(nrow) )

         i1 = [ (i, i = 1, nrow) ]

         out%data_type = x%data_type
         out%shape = [ size(i1), size(i2) ]
         if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
            allocate( out%double(size(i1),size(i2)) )

            out%double(:,:) = x%double(i1,i2)
         else if( x%data_type == MF_DT_CMPLX ) then
            allocate( out%cmplx(size(i1),size(i2)) )

            out%cmplx(:,:) = x%cmplx(i1,i2)
         else if( mfIsSparse(x) ) then

            ! test if i2 == [1,2,...,ncol]
            if( size(i2) == ncol ) then
               if( all( i2 == [ (i,i=1,ncol) ] ) ) then
                  all_cols_in_order = .true.
               else
                  all_cols_in_order = .false.
               end if
            else
               all_cols_in_order = .false.
            end if
            if( all_cols_in_order ) then
               out = x
            else
               call nnzxtrcols( 1, size(i2), x%i, x%j, i2, nnz )
               if( x%data_type == MF_DT_SP_DBLE ) then
                  allocate( out%a(nnz) )

                  allocate( out%i(nnz) )

                  allocate( out%j(ncol+1) )

                  call xtrcols( 1, size(i2), x%a, x%i, x%j,             &
                                out%a, out%i, out%j, i2 )
               else ! x%data_type == MF_DT_SP_CMPLX
                  allocate( out%z(nnz) )

                  allocate( out%i(nnz) )

                  allocate( out%j(ncol+1) )

                  call xtrcols_cmplx( 1, size(i2), x%z, x%i, x%j,       &
                                      out%z, out%i, out%j, i2 )
               end if
               out%row_sorted = x%row_sorted
            end if
         end if

      else if( isEndIndex(a1) ) then

         allocate( i1(1) )

         i1(1) = nrow + a1%crc
         call msAssign( out, mfGet_section_vec_vec(x,i1,i2) )

      else ! a1 is only a subpart of the row's set

         if( a1%data_type /= MF_DT_DBLE ) then
            call PrintMessage( "mfGet", "E",                            &
                               "a1 must be of type real!" )
            go to 99
         end if

         if( any( int(a1%double) /= a1%double) ) then
            call PrintMessage( "mfGet", "E",                            &
                               "a1 must contain integer values!" )
            go to 99
         end if

         if( a1%shape(1) /= 1 .and. a1%shape(2) /= 1 ) then
            call PrintMessage( "mfGet", "E",                            &
                               "mfArray indexes 'a1' must be of rank one!" )
            go to 99
         end if

         if( minval(a1%double) < 1 .or. nrow < maxval(a1%double) ) then
            call PrintMessage( "mfGet", "E",                            &
                               "row indexes are out of range!" )
            go to 99
         end if

         allocate( i1(size(a1%double)) )

         i1 = reshape( int(a1%double), [size(a1%double)] )

         out%data_type = x%data_type
         out%shape = [ size(i1), size(i2) ]
         if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
            allocate( out%double(size(i1),size(i2)) )

            out%double(:,:) = x%double(i1,i2)
         else if( x%data_type == MF_DT_CMPLX ) then
            allocate( out%cmplx(size(i1),size(i2)) )

            out%cmplx(:,:) = x%cmplx(i1,i2)
         else ! x is sparse
            ! test if i1(:) and i2(:) contain contiguous indexes
            ! in increasing order
            contiguous = .true.
            nrow = size(i1)
            do i = 2, nrow
               if( i1(i) /= i1(i-1) + 1 ) then
                  contiguous = .false.
                  exit
               end if
            end do
            ncol = size(i2)
            do i = 2, ncol
               if( i2(i) /= i2(i-1) + 1 ) then
                  contiguous = .false.
                  exit
               end if
            end do
            if( contiguous ) then
               i1_beg = i1(1)
               i1_end = i1(nrow)
               i2_beg = i2(1)
               i2_end = i2(ncol)
               call submat_nnz( x%i, x%j, i1_beg, i1_end, i2_beg, i2_end, &
                                nnz )
               allocate( out%i(nnz) )

               allocate( out%j(ncol+1) )

               if( mfIsReal(x) ) then
                  out%data_type = MF_DT_SP_DBLE
                  allocate( out%a(nnz) )

                  call submat( x%a, x%i, x%j, i1_beg, i1_end, i2_beg, i2_end, &
                               out%a, out%i, out%j )
               else ! x is complex
                  out%data_type = MF_DT_SP_CMPLX
                  allocate( out%z(nnz) )

                  call submat_cmplx( x%z, x%i, x%j, i1_beg, i1_end, i2_beg, i2_end, &
                                     out%z, out%i, out%j )
               end if
               out%row_sorted = x%row_sorted
            else
!### TODO 2: to be implemented
               call PrintMessage( "mfGet", "E",                         &
                                  "extracting a section from a sparse mfArray:", &
                                  "non contiguous indexes : not yet available!" )
               go to 99
            end if
         end if
      end if

      if( mf_phys_units ) then
         out%units(:) = x%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      if( associated(i2) ) then
#ifndef _TRACE_MEM_ALLOC
         deallocate( i2 )
#else
         call mf_deallocate( array=i2,                                  &
                             file="Get.inc", line="???",                &
                             symb="i2", unit="mfGet_section_array_vseq" )
#endif
      end if

      call msFreeArgs( x, a1 )

      call msAutoRelease( x, a1 )

#endif
   end function mfGet_section_array_vseq
!_______________________________________________________________________
!
   function mfGet_section_vseq_array( x, seq, a2 ) result( out )

      type(mfArray) :: x
      type(seq_def) :: seq
      type(mfArray) :: a2
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! if a2==MF_COLON : extracts selected rows from an mfArray
      !                   rows indexes comes from the sequence seq;
      ! else : extracts a sub-matrix via indexes in seq and a2
      !
      ! for sparse x, it requires increasing contiguous indexes,
      ! in both dimensions, but only when a2 /= MF_COLON.

      ! extended to the case where 'a2' contains an EndIndex (which is
      ! either the MF_END parameter itself, or an expression
      ! involving integers and MF_END, like 'MF_END-1').

      integer, pointer :: i1(:) => null()
      integer, allocatable :: i2(:)
      integer :: i1_beg, i1_end, nrow,                                  &
                 i2_beg, i2_end, ncol,                                  &
                 i, nnz
      logical :: contiguous

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x, a2 )

      if( mfIsEmpty(x) .or. mfIsEqual(a2,MF_EMPTY) ) then
         go to 99
      end if

      nrow = x%shape(1)
      ncol = x%shape(2)
      if( seq%start_1_EndIndex ) then
         seq%start_1 = nrow + seq%start_1
      end if
      if( seq%end_1_EndIndex ) then
         seq%end_1 = nrow + seq%end_1
      end if
      if( seq%start_2_EndIndex ) then
         seq%start_2 = nrow + seq%start_2
      end if
      if( seq%end_2_EndIndex ) then
         seq%end_2 = nrow + seq%end_2
      end if
      if( seq%but_EndIndex ) then
         seq%but = nrow + seq%but
      end if
      call build_int_seq( i1, seq, "mfGet" )

      if( minval(i1) < 1 .or. nrow < maxval(i1) ) then
         call PrintMessage( "mfGet", "E",                               &
                            "row indexes are out of range!" )
         go to 99
      end if

      if( mfIsEqual(a2,MF_COLON) ) then

         allocate( i2(ncol) )

         i2 = [ (i, i = 1, ncol) ]

      else if( isEndIndex(a2) ) then

         allocate( i2(1) )

         i2(1) = ncol + a2%crc
         call msAssign( out, mfGet_section_vec_vec(x,i1,i2) )
         out%status_temporary = .true.
         go to 99

      else

         if( a2%data_type /= MF_DT_DBLE ) then
            call PrintMessage( "mfGet", "E",                            &
                               "a2 must be of type real!" )
            go to 99
         end if

         if( any( int(a2%double) /= a2%double) ) then
            call PrintMessage( "mfGet", "E",                            &
                               "a2 must contain integer values!" )
            go to 99
         end if

         if( a2%shape(1) /= 1 .and. a2%shape(2) /= 1 ) then
            call PrintMessage( "mfGet", "E",                            &
                               "mfArray indexes 'a2' must be of rank one!" )
            go to 99
         end if

         if( minval(a2%double) < 1 .or. ncol < maxval(a2%double) ) then
            call PrintMessage( "mfGet", "E",                            &
                               "col indexes are out of range!" )
            go to 99
         end if

         allocate( i2(size(a2%double)) )

         i2 = reshape( int(a2%double), [size(a2%double)] )

      end if

      out%data_type = x%data_type
      out%shape = [ size(i1), size(i2) ]
      if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
         allocate( out%double(size(i1),size(i2)) )

         out%double(:,:) = x%double(i1,i2)
      else if( x%data_type == MF_DT_CMPLX ) then
         allocate( out%cmplx(size(i1),size(i2)) )

         out%cmplx(:,:) = x%cmplx(i1,i2)
      else ! x is sparse
         ! test if i1(:) and i2(:) contain contiguous indexes
         ! in increasing order
         contiguous = .true.
         nrow = size(i1)
         do i = 2, nrow
            if( i1(i) /= i1(i-1) + 1 ) then
               contiguous = .false.
               exit
            end if
         end do
         ncol = size(i2)
         do i = 2, ncol
            if( i2(i) /= i2(i-1) + 1 ) then
               contiguous = .false.
               exit
            end if
         end do
         if( contiguous ) then
            i1_beg = i1(1)
            i1_end = i1(nrow)
            i2_beg = i2(1)
            i2_end = i2(ncol)
            call submat_nnz( x%i, x%j, i1_beg, i1_end, i2_beg, i2_end,  &
                             nnz )
            allocate( out%i(nnz) )

            allocate( out%j(ncol+1) )

            if( mfIsReal(x) ) then
               out%data_type = MF_DT_SP_DBLE
               allocate( out%a(nnz) )

               call submat( x%a, x%i, x%j, i1_beg, i1_end, i2_beg, i2_end, &
                            out%a, out%i, out%j )
            else ! x is complex
               out%data_type = MF_DT_SP_CMPLX
               allocate( out%z(nnz) )

               call submat_cmplx( x%z, x%i, x%j, i1_beg, i1_end, i2_beg, i2_end, &
                                  out%z, out%i, out%j )
            end if
            out%row_sorted = x%row_sorted
         else
!### TODO 2: to be implemented
            call PrintMessage( "mfGet", "E",                            &
                               "extracting a section from a sparse mfArray:", &
                               "non contiguous indexes : not yet available!" )
            go to 99
         end if
      end if

      if( mf_phys_units ) then
         out%units(:) = x%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      if( associated(i1) ) then
#ifndef _TRACE_MEM_ALLOC
         deallocate( i1 )
#else
         call mf_deallocate( array=i1,                                  &
                             file="Get.inc", line="???",                &
                             symb="i1", unit="mfGet_section_vseq_array" )
#endif
      end if

      call msFreeArgs( x, a2 )

      call msAutoRelease( x, a2 )

#endif
   end function mfGet_section_vseq_array
!_______________________________________________________________________
!
   function mfGet_section_array_int( x, a1, i2 ) result( out )

      type(mfArray) :: x
      type(mfArray) :: a1
      integer, intent(in) :: i2
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! if a1==MF_COLON : extracts the column i2 from an mfArray;
      ! else : extracts a sub-column via indexes in a1 from col i2
      !
      ! returns always a dense vector mfArray

      ! extended to the case where 'a1' contains an EndIndex (which is
      ! either the MF_END parameter itself, or an expression
      ! involving integers and MF_END, like 'MF_END-1').

      integer, allocatable :: i1(:)
      real(kind=MF_DOUBLE), allocatable :: xtmp(:)
      complex(kind=MF_DOUBLE), allocatable :: ztmp(:)
      integer :: nrow, i

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x, a1 )

      if( mfIsEmpty(x) .or. mfIsEqual(a1,MF_EMPTY) ) then
         go to 99
      end if

      if( i2 < 1 .or. x%shape(2) < i2 ) then
         call PrintMessage( "mfGet", "E",                               &
                            "col index is out of range!" )
         go to 99
      end if

      if( mfIsEqual(a1,MF_COLON) ) then

         out%shape = [ x%shape(1), 1 ]
         if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
            out%data_type = x%data_type
            allocate( out%double(x%shape(1),1) )

            out%double(:,:) = x%double(:,i2:i2)
         else if( x%data_type == MF_DT_CMPLX ) then
            out%data_type = x%data_type
            allocate( out%cmplx(x%shape(1),1) )

            out%cmplx(:,:) = x%cmplx(:,i2:i2)
         else if( x%data_type == MF_DT_SP_DBLE ) then
            out%data_type = MF_DT_DBLE
            allocate( out%double(x%shape(1),1) )

            call xtrcol( x%shape(1), x%a, x%i, x%j, i2, out%double(:,1) )
         else if( x%data_type == MF_DT_SP_CMPLX ) then
            out%data_type = MF_DT_CMPLX
            allocate( out%cmplx(x%shape(1),1) )

            call xtrcol_cmplx( x%shape(1), x%z, x%i, x%j, i2, out%cmplx(:,1) )
         end if

      else if( isEndIndex(a1) ) then

         nrow = x%shape(1)
         i = nrow + a1%crc
         call msAssign( out, mfGet_element(x,i,i2) )

      else

         if( a1%data_type /= MF_DT_DBLE ) then
            call PrintMessage( "mfGet", "E",                            &
                               "a1 must be of type real!" )
            go to 99
         end if

         if( any( int(a1%double) /= a1%double) ) then
            call PrintMessage( "mfGet", "E",                            &
                               "a1 must contain integer values!" )
            go to 99
         end if

         if( a1%shape(1) /= 1 .and. a1%shape(2) /= 1 ) then
            call PrintMessage( "mfGet", "E",                            &
                               "mfArray indexes 'a1' should be of rank one!" )
            go to 99
         end if

         if( minval(a1%double) < 1 .or. x%shape(1) < maxval(a1%double) ) then
            call PrintMessage( "mfGet", "E",                            &
                               "row indexes are out of range!" )
            go to 99
         end if

         allocate( i1(size(a1%double)) )

         i1 = reshape( int(a1%double), [size(a1%double)] )

         out%data_type = x%data_type
         out%shape = [ size(i1), 1 ]
         if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
            allocate( out%double(size(i1),1) )

            out%double(:,:) = x%double(i1,i2:i2)
         else if( x%data_type == MF_DT_CMPLX ) then
            allocate( out%cmplx(size(i1),1) )

            out%cmplx(:,:) = x%cmplx(i1,i2:i2)
         else ! x is sparse
!### TODO 2: pour l'instant l'extraction n'est pas optimisée : on extrait
!          les valeurs de toute la colonne dans un vecteur tempo puis
!          on recopie...
            if( x%data_type == MF_DT_SP_DBLE ) then
               allocate( xtmp(x%shape(1)) )
               call xtrcol( x%shape(1), x%a, x%i, x%j, i2, xtmp )
               out%data_type = MF_DT_DBLE
               allocate( out%double(size(i1),1) )

               out%double(:,1) = xtmp(i1)
            else if( x%data_type == MF_DT_SP_CMPLX ) then
               allocate( ztmp(x%shape(1)) )
               call xtrcol_cmplx( x%shape(1), x%z, x%i, x%j, i2, ztmp )
               out%data_type = MF_DT_CMPLX
               allocate( out%cmplx(size(i1),1) )

               out%cmplx(:,1) = ztmp(i1)
            end if
         end if

      end if

      if( mf_phys_units ) then
         out%units(:) = x%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( x, a1 )

      call msAutoRelease( x, a1 )

#endif
   end function mfGet_section_array_int
!_______________________________________________________________________
!
   function mfGet_section_int_array( x, i1, a2 ) result( out )

      type(mfArray) :: x
      integer, intent(in) :: i1
      type(mfArray) :: a2
      type(mfArray) :: out
      !------ API end ------

#ifdef _DEVLP
      ! if a2==MF_COLON : extracts the row i1 from an mfArray;
      ! else : extracts a sub-matrix via indexes in a2 from row i1
      !
      ! returns always a dense vector mfArray

      ! extended to the case where 'a2' contains an EndIndex (which is
      ! either the MF_END parameter itself, or an expression
      ! involving integers and MF_END, like 'MF_END-1').

      integer, allocatable :: i2(:)
      real(kind=MF_DOUBLE), allocatable :: xtmp(:)
      complex(kind=MF_DOUBLE), allocatable :: ztmp(:)
      integer :: ncol, i

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x, a2 )

      if( mfIsEmpty(x) .or. mfIsEqual(a2,MF_EMPTY) ) then
         go to 99
      end if

      if( i1 < 1 .or. x%shape(1) < i1 ) then
         call PrintMessage( "mfGet", "E",                               &
                            "row index is out of range!" )
         go to 99
      end if

      if( mfIsEqual(a2,MF_COLON) ) then

         out%shape = [ 1, x%shape(2) ]
         if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
            out%data_type = x%data_type
            allocate( out%double(1,x%shape(2)) )

            out%double(:,:) = x%double(i1:i1,:)
         else if( x%data_type == MF_DT_CMPLX ) then
            out%data_type = x%data_type
            allocate( out%cmplx(1,x%shape(2)) )

            out%cmplx(:,:) = x%cmplx(i1:i1,:)
         else if( x%data_type == MF_DT_SP_DBLE ) then
            out%data_type = MF_DT_DBLE
            allocate( out%double(1,x%shape(2)) )

            call xtrrow( x%shape(2), x%a, x%i, x%j, i1, out%double(1,:) )
         else if( x%data_type == MF_DT_SP_CMPLX ) then
            out%data_type = MF_DT_CMPLX
            allocate( out%cmplx(1,x%shape(2)) )

            call xtrrow_cmplx( x%shape(2), x%z, x%i, x%j, i1, out%cmplx(1,:) )
         end if

      else if( isEndIndex(a2) ) then

         ncol = x%shape(2)
         i = ncol + a2%crc
         call msAssign( out, mfGet_element(x,i1,i) )

      else

         if( a2%data_type /= MF_DT_DBLE ) then
            call PrintMessage( "mfGet", "E",                            &
                               "a2 must be of type real!" )
            go to 99
         end if

         if( any( int(a2%double) /= a2%double) ) then
            call PrintMessage( "mfGet", "E",                            &
                               "a2 must contain integer values!" )
            go to 99
         end if

         if( a2%shape(1) /= 1 .and. a2%shape(2) /= 1 ) then
            call PrintMessage( "mfGet", "E",                            &
                               "mfArray indexes 'a2' should be of rank one!" )
            go to 99
         end if

         if( minval(a2%double) < 1 .or. x%shape(2) < maxval(a2%double) ) then
            call PrintMessage( "mfGet", "E",                            &
                               "col indexes are out of range!" )
            go to 99
         end if

         allocate( i2(size(a2%double)) )

         i2 = reshape( int(a2%double), [size(a2%double)] )

         out%data_type = x%data_type
         out%shape = [ 1, size(i2) ]
         if( x%data_type == MF_DT_DBLE .or. x%data_type == MF_DT_BOOL ) then
            allocate( out%double(1,size(i2)) )

            out%double(:,:) = x%double(i1:i1,i2)
         else if( x%data_type == MF_DT_CMPLX ) then
            allocate( out%cmplx(1,size(i2)) )

            out%cmplx(:,:) = x%cmplx(i1:i1,i2)
         else ! x is sparse
!### TODO 2: pour l'instant l'extraction n'est pas optimisée : on extrait
!          les valeurs de toute la ligne dans un vecteur tempo puis
!          on recopie...
            if( x%data_type == MF_DT_SP_DBLE ) then
               allocate( xtmp(x%shape(2)) )
               call xtrrow( x%shape(2), x%a, x%i, x%j, i1, xtmp )
               out%data_type = MF_DT_DBLE
               allocate( out%double(1,size(i2)) )

               out%double(1,:) = xtmp(i2)
            else if( x%data_type == MF_DT_SP_CMPLX ) then
               allocate( ztmp(x%shape(2)) )
               call xtrrow_cmplx( x%shape(2), x%z, x%i, x%j, i1, ztmp )
               out%data_type = MF_DT_CMPLX
               allocate( out%cmplx(1,size(i2)) )

               out%cmplx(1,:) = ztmp(i2)
            end if
         end if

      end if

      if( mf_phys_units ) then
         out%units(:) = x%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( x, a2 )

      call msAutoRelease( x, a2 )

#endif
   end function mfGet_section_int_array
