program main

   ! Étant donné un module pré-compilé Fortran-90, et une liste de
   ! symboles correspondant à des champs internes de types dérivés
   ! (chacun d'eux associé à un module qui l'héberge), on change le
   ! nom de ces symboles en ajoutant le signe '-' en tête de ce symbole.
   !
   ! Works only with GFORTRAN module version '12' to '16' (GCC >= 4.9)
   !
   ! Deux passes sont nécessaires :
   !  - la première pour déterminer l'ordre d'apparition des types
   !    dérivés ;
   !  - la deuxième pour effectuer les modifications.
   !
   ! Ci-dessus, 'nb_modules' doit correspondre exactement au nombre de
   ! types dérivés à traiter.
   ! 'nb_symbols' peut être défini à une valeur arbitraire, mais doit
   ! être suffisant pour contenir tous les symboles d'un type dérivé.
   !
   ! Même code pour transformer fml et fgl : il faut simplement donner
   ! le nom du module à traiter en argument.
   !
   ! Version 4.1
   ! É. Canot -- IPR/CNRS -- 2025-06-24
   !--------------------------------------------------------------------

   ! Nota: si on fait des modifications (nombre de modules ou de symboles),
   !       il vaut mieux ne pas utiliser f90cache, ou alors effacer
   !       complètement le cache auparavant.

   implicit none

   integer, parameter :: nb_modules = 4
   character(len=120) :: modules_list(nb_modules)
   integer            :: tab_lines(nb_modules), perm(nb_modules)

   integer, parameter :: nb_symbols = 20
   character(len=120) :: symbols_list(nb_modules,nb_symbols)
   integer            :: eff_nb_symbols(nb_modules)

   character(len=128) :: line_1, line_2
   character(len=256) :: buf
   character(len=3)   :: arg

   integer, parameter :: MOD_IN = 10, MOD_OUT = 12

   integer :: len_1, len_2, im, i, j, k, module_num, symbol_num
   integer :: iostat, k_line
   integer :: n_args, module_version

   logical :: waiting_for_module, waiting_for_symbol

   !--------------------------------------------------------------------

   im = 0

   im = im + 1
   modules_list(im) = "'Mfarray' 'mod_mfarray' '' 1 ((DERIVED"
   eff_nb_symbols(im) = 19
      symbols_list(im,1) = "'data_type'"
      symbols_list(im,2) = "'shape'"
      symbols_list(im,3) = "'double'"
      symbols_list(im,4) = "'cmplx'"
      symbols_list(im,5) = "'a'"
      symbols_list(im,6) = "'z'"
      symbols_list(im,7) = "'i'"
      symbols_list(im,8) = "'j'"
      symbols_list(im,9) = "'row_sorted'"
      symbols_list(im,10) = "'umf4_ptr_numeric'"
      symbols_list(im,11) = "'prop'"
      symbols_list(im,12) = "'level_locked'"
      symbols_list(im,13) = "'crc_stored'"
      symbols_list(im,14) = "'crc'"
      symbols_list(im,15) = "'status_temporary'"
      symbols_list(im,16) = "'level_protected'"
      symbols_list(im,17) = "'status_restricted'"
      symbols_list(im,18) = "'parameter'"
      symbols_list(im,19) = "'units'"

   im = im + 1
   modules_list(im) = "'Mfmatfactor' 'mod_mfarray' '' 1 ((DERIVED"
   eff_nb_symbols(im) = 12
      symbols_list(im,1) = "'data_type'"
      symbols_list(im,2) = "'mf_ptr_1'"
      symbols_list(im,3) = "'mf_ptr_2'"
      symbols_list(im,4) = "'mf_ptr_3'"
      symbols_list(im,5) = "'order'"
      symbols_list(im,6) = "'package'"
      symbols_list(im,7) = "'ptr_1'"
      symbols_list(im,8) = "'ptr_2'"
      symbols_list(im,9) = "'ptr_3'"
      symbols_list(im,10) = "'ptr_4'"
      symbols_list(im,11) = "'shape'"
      symbols_list(im,12) = "'units'"

   im = im + 1
   modules_list(im) = "'Mftriconnect' 'mod_fileio' '' 1 ((DERIVED"
   eff_nb_symbols(im) = 13
      symbols_list(im,1) = "'init'"
      symbols_list(im,2) = "'tri_renumbering'"
      symbols_list(im,3) = "'face_oriented'"
      symbols_list(im,4) = "'convex_domain'"
      symbols_list(im,5) = "'n_xy'"
      symbols_list(im,6) = "'tri_n'"
      symbols_list(im,7) = "'face_n'"
      symbols_list(im,8) = "'face_tri'"
      symbols_list(im,9) = "'tri_f'"
      symbols_list(im,10) = "'n_tri'"
      symbols_list(im,11) = "'faces_boundary'"
      symbols_list(im,12) = "'faces_boundary_ptr'"
      symbols_list(im,13) = "'status_temporary'"

   im = im + 1
   modules_list(im) = "'Mftetraconnect' 'mod_polyfun' '' 1 ((DERIVED"
   eff_nb_symbols(im) = 11
      symbols_list(im,1) = "'init'"
      symbols_list(im,2) = "'n_xyz'"
      symbols_list(im,3) = "'tetra_n'"
      symbols_list(im,4) = "'tetra_neighbors'"
      symbols_list(im,5) = "'nodes_sorted_x'"
      symbols_list(im,6) = "'nodes_sorted_y'"
      symbols_list(im,7) = "'nodes_sorted_z'"
      symbols_list(im,8) = "'face_n'"
      symbols_list(im,9) = "'face_tetra'"
      symbols_list(im,10) = "'tetra_f'"
      symbols_list(im,11) = "'n_tetra'"

   ! On travaille avec une chaîne (buffer) de longueur au moins double
   ! de celle habituellement trouvée dans les fichiers MOD (environ 80
   ! donc 160).
   ! On remplit ce buffer de deux lignes du fichier MOD et on recherche
   ! le premier module (de modules_list) dans ce buffer.
   !   1) si on ne trouve pas, on écrit la première des deux lignes et
   !      on continue en décalant (on ajoute dans le buffer la ligne
   !      suivante)
   !   2) si on trouve, on arme la recherche des symboles. Dès qu'on
   !      trouve un symbole (de symbols_list), on insère un '-' au
   !      début
   !--------------------------------------------------------------------

   ! Read (mandatory) argument...
   n_args = command_argument_count()
   if( n_args /= 1 ) then
      write(0,*)
      write(0,*) "*** ERROR: subs_in_mod: One argument is required!"
      write(0,*)
      go to 199
   end if
   call get_command_argument( 1, arg )
   if( arg /= "fml" .and. arg /= "fgl" ) then
      write(0,*)
      write(0,*) "*** ERROR: subs_in_mod: Argument must be equal to 'fml' or 'fgl'!"
      write(0,*)
      go to 199
   end if

   ! Ouverture du fichier MOD_IN en lecture
   open( MOD_IN, file=arg )

   ! Some checks
   read( MOD_IN, "(A)" ) line_1
   ! Find the module version
   i = index( line_1, "module version '" )
   if( i == 0 ) then
      write(0,*)
      write(0,*) "*** ERROR: subs_in_mod: unexpected first line of the module!"
      write(0,*) "    -> corrupted file?"
      write(0,*)
      go to 199
   end if
   j = index( line_1, "' created from" )
   if( j == 0 ) then
      write(0,*)
      write(0,*) "*** ERROR: subs_in_mod: unexpected first line of the module!"
      write(0,*) "    -> corrupted file?"
      write(0,*)
      go to 199
   end if
   read( line_1(i+16:j-1), *) module_version
   if( module_version < 12 .or. 16 < module_version ) then
      write(0,*)
      write(0,*) "*** ERROR: subs_in_mod: bad module version!"
      write(0,*) "    -> must be in [12,15]."
      write(0,*)
      go to 199
   end if

   ! Ouverture du fichier MOD_OUT en écriture
   open( MOD_OUT, file=arg//"_new" )

   ! Première étape pour établir l'ordre d'apparition des types
   ! dérivés.

   do im = 1, nb_modules

      rewind( MOD_IN )

      ! Lecture d'une première ligne de MOD_IN
      read( MOD_IN, "(A)" ) line_1
      k_line = 1

      do ! boucle sur les lignes du fichier MOD, jusqu'à trouver le
         ! type dérivé défini dans modules_list(im).

         read( MOD_IN, "(A)", iostat=iostat ) line_2
         k_line = k_line + 1

         if( iostat /= 0 ) then
            write(0,*)
            write(0,*) "*** ERROR: subs_in_mod: Derived type not found:"
            write(0,*) "      ", trim(modules_list(im))
            write(0,*)
            go to 199
         end if

         ! It seems to be required to insert a space between the two parts
         buf = trim(line_1) // " " // trim(line_2)

         i = index( buf, trim(modules_list(im)) )
         if( i > 0 ) then
            tab_lines(im) = k_line
            exit
         end if

      end do

   end do

   ! On détermine l'ordre de passage 'perm' pour traiter les types
   ! dérivés lors de la deuxième passe.
   call determine( tab_lines, perm )

   print *
   print *, " -> first step done."
   print *

   rewind( MOD_IN )

   ! Deuxième passe pour effectuer les modifications.

   ! Lecture d'une première ligne de MOD_IN
   read( MOD_IN, "(A)" ) line_1
   len_1 = len_trim(line_1)

   waiting_for_module = .true.
   waiting_for_symbol = .false.
   module_num = 1
   im = perm(module_num)

   do ! boucle sur toutes les lignes du fichier MOD

      read( MOD_IN, "(A)", END=99 ) line_2
      len_2 = len_trim(line_2)

      ! It seems to be required to insert a space between the two parts
      buf = trim(line_1) // " " // trim(line_2)

      ! No loop for the module string: it is impossible that an entire
      ! module can be stored within two lines...
      if( waiting_for_module ) then
         i = index( buf, trim(modules_list(im)) )
         if( i > 0 ) then
            print *, '"' // trim(modules_list(im)) // '" found.'
            waiting_for_module = .false.
            waiting_for_symbol = .true.
            symbol_num = 1
         end if
      end if

      ! However, it is not impossible to have many symbols in two lines!
      if( waiting_for_symbol ) then

         do k = 1, 2

            j = index( buf(i:), trim(symbols_list(im,symbol_num)) )
            if( i > 1 ) i = 1
            if( j > 0 ) then
               print *, '   ' // trim(symbols_list(im,symbol_num)) // ' found.'
               buf(j+2:) = buf(j+1:)
               buf(j+1:j+1) = '-'
               if( j+1 <= len_1 ) then
                  len_1 = len_1 + 1
               else
                  len_2 = len_2 + 1
               end if
               symbol_num = symbol_num + 1
               if( symbol_num > eff_nb_symbols(im) ) then
                  print *, "   -> (all symbols for this module found.)"
                  print *
                  waiting_for_module = .true.
                  waiting_for_symbol = .false.
                  module_num = module_num + 1
                  if( module_num > nb_modules ) then
                     waiting_for_module = .false.
                  else
                     im = perm(module_num)
                  end if
                  exit
               end if
            end if
         end do

      end if

      ! 'buf' has been perhaps modified, therefore we have to extract
      ! again the two (new?) lines...
      line_1 = buf(1:len_1)
      write( MOD_OUT, "(A)" ) trim(line_1)

      line_2 = buf(len_1+1+1:len_1+1+len_2)

      line_1 = line_2
      len_1 = len_2

   end do

99 continue

   write( MOD_OUT, "(A)" ) trim(line_1)

   print *, " -> second step done."
   print *

   close( MOD_IN )
   close( MOD_OUT )

   go to 299

199 continue

   write(0,*) "Waiting 3 seconds..."
   call sleep(3)
   stop 1

299 continue

contains !==============================================================

   subroutine determine( tab_1, tab_2 )

      integer :: tab_1(:), tab_2(:)

      ! Input:  tab_1 contains different positive integers (destroyed at output).
      ! Output: tab_2 contains the lines of sorted values in tab_1.

      ! Note that tab_1 values are modified. At output, they are all equal
      ! to 0.

      integer :: i, j, n, val

      n = size(tab_1)

      do i = 1, n

         ! Looking for the smallest value (other than 0) in tab_1
         val = huge(1)
         do j = 1, n
            if( tab_1(j) > 0 ) then
               if( tab_1(j) < val ) then
                  val = tab_1(j)
                  tab_2(i) = j
               end if
            end if
         end do
         j = tab_2(i)
         tab_1(j) = 0

      end do

   end subroutine

end program
